/* Copyright (C) 2010 and 2011 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file or uses macros or
inline functions (of any length) should by reason only of that
instantiation or use be subject to the restrictions of use in the GNU
Lesser General Public License.  With that in mind, the words "and
macros, inline functions and instantiations of templates (of any
length)" shall be treated as substituted for the words "and small
macros and small inline functions (ten lines or less in length)" in
the fourth paragraph of section 5 of that licence.  This does not
affect any other reason why object code may be subject to the
restrictions in that licence (nor for the avoidance of doubt does it
affect the application of section 2 of that licence to modifications
of the source code in this file).

*/

#ifndef CGU_PARAM_H
#define CGU_PARAM_H

#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

/**
 * @class Param param.h c++-gtk-utils/param.h
 * @brief Struct for automatic typing of parameter arguments for template functions
 *
 * @details This struct uses template partial specialisation in order
 * to provide automatic type mapping for arguments for templated
 * functions.  It is used by callback object constructors, the unbound
 * arguments of callback objects and their related functors and
 * emitter objects, and the DoIf::mem_fun() and DoIf::fun() functions.
 * With DoIf::mem_fun() and DoIf::fun(), its use avoids errors arising
 * from a template function being instantiated for a reference to
 * reference type, and also ensures that value arguments are passed by
 * reference to const to avoid unnecessary copying.
 *
 * Mapping is as follows:
 *
 *  A value argument is mapped to reference to const of the value
 *  type.
 *
 *  A pointer argument is mapped to pointer argument (its original
 *  type).
 *
 *  A non-const reference argument is mapped to non-const reference
 *  (its original type).
 *
 *  A const reference argument is mapped to const reference (its
 *  original type).
 *
 * Since 1.2.10
 */

template<class T>
struct Param {
  typedef const T& ParamType;
};

template<class T>
struct Param<T&> {
  typedef T& ParamType;
};

template<class T>
struct Param<T*> {
  typedef T* ParamType;
};

/**
 * @class TypeTuple param.h c++-gtk-utils/param.h
 * @brief Struct type to provide unbound arguments for
 * Callback::CallbackArg objects.
 *
 * @details This struct is used with template partial specialisation
 * of Callback::CallbackArg, Callback::FunctorArg and EmitterArg
 * objects to enable them to take up to five unbound arguments,
 * although at present these classes are only specialised to take two
 * or three unbound arguments.  This struct has no members and is
 * never instantiated: it is just used as a placeholder for its
 * contained types.
 *
 * As the compiled part of this library does not use this struct (it
 * just makes it available), the advantage of specialising the
 * (single) template type of Callback::CallbackArg,
 * Callback::FunctorArg and EmitterArg via this struct, rather than
 * specialising them directly, is that this struct can be extended (or
 * have Loki-like typelists substituted) without breaking binary
 * compatibility.  The types have default type values of void,
 * indicating no argument, so this struct is capable of representing
 * any number of arguments from 0 to 5.
 *
 * Since 1.2.10
 */
template <class T1 = void, class T2 = void, class T3 = void,
          class T4 = void, class T5 = void> struct TypeTuple {};

} // namespace Cgu

#endif // CGU_PARAM_H
