// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (c) 2025 Rockchip Electronics Co., Ltd.
 * Author: Elaine Zhang <zhangqing@rock-chips.com>
 */

#include <linux/module.h>
#include <linux/of.h>
#include <dt-bindings/reset/rockchip,rv1126b-cru.h>
#include "clk.h"

/* 0x20000000 + 0x0A00 */
#define TOPCRU_RESET_OFFSET(id, reg, bit) [id] = (0x0 * 4 + reg * 16 + bit)
/* 0x20010000 + 0x0A00 */
#define BUSCRU_RESET_OFFSET(id, reg, bit) [id] = (0x10000 * 4 + reg * 16 + bit)
/* 0x20020000 + 0x0A00 */
#define PERICRU_RESET_OFFSET(id, reg, bit) [id] = (0x20000 * 4 + reg * 16 + bit)
/* 0x20030000 + 0x0A00 */
#define CORECRU_RESET_OFFSET(id, reg, bit) [id] = (0x30000 * 4 + reg * 16 + bit)
/* 0x20040000 + 0x0A00 */
#define PMUCRU_RESET_OFFSET(id, reg, bit) [id] = (0x40000 * 4 + reg * 16 + bit)
/* 0x20050000 + 0x0A00 */
#define PMU1CRU_RESET_OFFSET(id, reg, bit) [id] = (0x50000 * 4 + reg * 16 + bit)
/* 0x20060000 + 0x0A00 */
#define DDRCRU_RESET_OFFSET(id, reg, bit) [id] = (0x60000 * 4 + reg * 16 + bit)
/* 0x20068000 + 0x0A00 */
#define SUBDDRCRU_RESET_OFFSET(id, reg, bit) [id] = (0x68000 * 4 + reg * 16 + bit)
/* 0x20070000 + 0x0A00 */
#define VICRU_RESET_OFFSET(id, reg, bit) [id] = (0x70000 * 4 + reg * 16 + bit)
/* 0x20080000 + 0x0A00 */
#define VEPUCRU_RESET_OFFSET(id, reg, bit) [id] = (0x80000 * 4 + reg * 16 + bit)
/* 0x20090000 + 0x0A00 */
#define NPUCRU_RESET_OFFSET(id, reg, bit) [id] = (0x90000 * 4 + reg * 16 + bit)
/* 0x200A0000 + 0x0A00 */
#define VDOCRU_RESET_OFFSET(id, reg, bit) [id] = (0xA0000 * 4 + reg * 16 + bit)
/* 0x200B0000 + 0x0A00 */
#define VCPCRU_RESET_OFFSET(id, reg, bit) [id] = (0xB0000 * 4 + reg * 16 + bit)

/* =================mapping table for reset ID to register offset================== */
static const int rv1126b_register_offset[] = {
	/* TOPCRU-->SOFTRST_CON00 */

	/* TOPCRU-->SOFTRST_CON15 */
	TOPCRU_RESET_OFFSET(SRST_P_CRU, 15, 1),
	TOPCRU_RESET_OFFSET(SRST_P_CRU_BIU, 15, 2),

	/* BUSCRU-->SOFTRST_CON00 */
	BUSCRU_RESET_OFFSET(SRST_A_TOP_BIU, 0, 0),
	BUSCRU_RESET_OFFSET(SRST_A_RKCE_BIU, 0, 1),
	BUSCRU_RESET_OFFSET(SRST_A_BUS_BIU, 0, 2),
	BUSCRU_RESET_OFFSET(SRST_H_BUS_BIU, 0, 3),
	BUSCRU_RESET_OFFSET(SRST_P_BUS_BIU, 0, 4),
	BUSCRU_RESET_OFFSET(SRST_P_CRU_BUS, 0, 5),
	BUSCRU_RESET_OFFSET(SRST_P_SYS_GRF, 0, 6),
	BUSCRU_RESET_OFFSET(SRST_H_BOOTROM, 0, 7),
	BUSCRU_RESET_OFFSET(SRST_A_GIC400, 0, 8),
	BUSCRU_RESET_OFFSET(SRST_A_SPINLOCK, 0, 9),
	BUSCRU_RESET_OFFSET(SRST_P_WDT_NS, 0, 10),
	BUSCRU_RESET_OFFSET(SRST_T_WDT_NS, 0, 11),

	/* BUSCRU-->SOFTRST_CON01 */
	BUSCRU_RESET_OFFSET(SRST_P_WDT_HPMCU, 1, 0),
	BUSCRU_RESET_OFFSET(SRST_T_WDT_HPMCU, 1, 1),
	BUSCRU_RESET_OFFSET(SRST_H_CACHE, 1, 2),
	BUSCRU_RESET_OFFSET(SRST_P_HPMCU_MAILBOX, 1, 3),
	BUSCRU_RESET_OFFSET(SRST_P_HPMCU_INTMUX, 1, 4),
	BUSCRU_RESET_OFFSET(SRST_HPMCU_FULL_CLUSTER, 1, 5),
	BUSCRU_RESET_OFFSET(SRST_HPMCU_PWUP, 1, 6),
	BUSCRU_RESET_OFFSET(SRST_HPMCU_ONLY_CORE, 1, 7),
	BUSCRU_RESET_OFFSET(SRST_T_HPMCU_JTAG, 1, 8),
	BUSCRU_RESET_OFFSET(SRST_P_RKDMA, 1, 11),
	BUSCRU_RESET_OFFSET(SRST_A_RKDMA, 1, 12),

	/* BUSCRU-->SOFTRST_CON02 */
	BUSCRU_RESET_OFFSET(SRST_P_DCF, 2, 0),
	BUSCRU_RESET_OFFSET(SRST_A_DCF, 2, 1),
	BUSCRU_RESET_OFFSET(SRST_H_RGA, 2, 2),
	BUSCRU_RESET_OFFSET(SRST_A_RGA, 2, 3),
	BUSCRU_RESET_OFFSET(SRST_CORE_RGA, 2, 4),
	BUSCRU_RESET_OFFSET(SRST_P_TIMER, 2, 5),
	BUSCRU_RESET_OFFSET(SRST_TIMER0, 2, 6),
	BUSCRU_RESET_OFFSET(SRST_TIMER1, 2, 7),
	BUSCRU_RESET_OFFSET(SRST_TIMER2, 2, 8),
	BUSCRU_RESET_OFFSET(SRST_TIMER3, 2, 9),
	BUSCRU_RESET_OFFSET(SRST_TIMER4, 2, 10),
	BUSCRU_RESET_OFFSET(SRST_TIMER5, 2, 11),
	BUSCRU_RESET_OFFSET(SRST_A_RKCE, 2, 12),
	BUSCRU_RESET_OFFSET(SRST_PKA_RKCE, 2, 13),
	BUSCRU_RESET_OFFSET(SRST_H_RKRNG_S, 2, 14),
	BUSCRU_RESET_OFFSET(SRST_H_RKRNG_NS, 2, 15),

	/* BUSCRU-->SOFTRST_CON03 */
	BUSCRU_RESET_OFFSET(SRST_P_I2C0, 3, 0),
	BUSCRU_RESET_OFFSET(SRST_I2C0, 3, 1),
	BUSCRU_RESET_OFFSET(SRST_P_I2C1, 3, 2),
	BUSCRU_RESET_OFFSET(SRST_I2C1, 3, 3),
	BUSCRU_RESET_OFFSET(SRST_P_I2C3, 3, 4),
	BUSCRU_RESET_OFFSET(SRST_I2C3, 3, 5),
	BUSCRU_RESET_OFFSET(SRST_P_I2C4, 3, 6),
	BUSCRU_RESET_OFFSET(SRST_I2C4, 3, 7),
	BUSCRU_RESET_OFFSET(SRST_P_I2C5, 3, 8),
	BUSCRU_RESET_OFFSET(SRST_I2C5, 3, 9),
	BUSCRU_RESET_OFFSET(SRST_P_SPI0, 3, 10),
	BUSCRU_RESET_OFFSET(SRST_SPI0, 3, 11),
	BUSCRU_RESET_OFFSET(SRST_P_SPI1, 3, 12),
	BUSCRU_RESET_OFFSET(SRST_SPI1, 3, 13),

	/* BUSCRU-->SOFTRST_CON04 */
	BUSCRU_RESET_OFFSET(SRST_P_PWM0, 4, 0),
	BUSCRU_RESET_OFFSET(SRST_PWM0, 4, 1),
	BUSCRU_RESET_OFFSET(SRST_P_PWM2, 4, 4),
	BUSCRU_RESET_OFFSET(SRST_PWM2, 4, 5),
	BUSCRU_RESET_OFFSET(SRST_P_PWM3, 4, 8),
	BUSCRU_RESET_OFFSET(SRST_PWM3, 4, 9),

	/* BUSCRU-->SOFTRST_CON05 */
	BUSCRU_RESET_OFFSET(SRST_P_UART1, 5, 0),
	BUSCRU_RESET_OFFSET(SRST_S_UART1, 5, 1),
	BUSCRU_RESET_OFFSET(SRST_P_UART2, 5, 2),
	BUSCRU_RESET_OFFSET(SRST_S_UART2, 5, 3),
	BUSCRU_RESET_OFFSET(SRST_P_UART3, 5, 4),
	BUSCRU_RESET_OFFSET(SRST_S_UART3, 5, 5),
	BUSCRU_RESET_OFFSET(SRST_P_UART4, 5, 6),
	BUSCRU_RESET_OFFSET(SRST_S_UART4, 5, 7),
	BUSCRU_RESET_OFFSET(SRST_P_UART5, 5, 8),
	BUSCRU_RESET_OFFSET(SRST_S_UART5, 5, 9),
	BUSCRU_RESET_OFFSET(SRST_P_UART6, 5, 10),
	BUSCRU_RESET_OFFSET(SRST_S_UART6, 5, 11),
	BUSCRU_RESET_OFFSET(SRST_P_UART7, 5, 12),
	BUSCRU_RESET_OFFSET(SRST_S_UART7, 5, 13),

	/* BUSCRU-->SOFTRST_CON06 */
	BUSCRU_RESET_OFFSET(SRST_P_TSADC, 6, 0),
	BUSCRU_RESET_OFFSET(SRST_TSADC, 6, 1),
	BUSCRU_RESET_OFFSET(SRST_H_SAI0, 6, 2),
	BUSCRU_RESET_OFFSET(SRST_M_SAI0, 6, 3),
	BUSCRU_RESET_OFFSET(SRST_H_SAI1, 6, 4),
	BUSCRU_RESET_OFFSET(SRST_M_SAI1, 6, 5),
	BUSCRU_RESET_OFFSET(SRST_H_SAI2, 6, 6),
	BUSCRU_RESET_OFFSET(SRST_M_SAI2, 6, 7),
	BUSCRU_RESET_OFFSET(SRST_H_RKDSM, 6, 8),
	BUSCRU_RESET_OFFSET(SRST_M_RKDSM, 6, 9),
	BUSCRU_RESET_OFFSET(SRST_H_PDM, 6, 10),
	BUSCRU_RESET_OFFSET(SRST_M_PDM, 6, 11),
	BUSCRU_RESET_OFFSET(SRST_PDM, 6, 12),

	/* BUSCRU-->SOFTRST_CON07 */
	BUSCRU_RESET_OFFSET(SRST_H_ASRC0, 7, 0),
	BUSCRU_RESET_OFFSET(SRST_ASRC0, 7, 1),
	BUSCRU_RESET_OFFSET(SRST_H_ASRC1, 7, 2),
	BUSCRU_RESET_OFFSET(SRST_ASRC1, 7, 3),
	BUSCRU_RESET_OFFSET(SRST_P_AUDIO_ADC_BUS, 7, 4),
	BUSCRU_RESET_OFFSET(SRST_M_AUDIO_ADC_BUS, 7, 5),
	BUSCRU_RESET_OFFSET(SRST_P_RKCE, 7, 6),
	BUSCRU_RESET_OFFSET(SRST_H_NS_RKCE, 7, 7),
	BUSCRU_RESET_OFFSET(SRST_P_OTPC_NS, 7, 8),
	BUSCRU_RESET_OFFSET(SRST_SBPI_OTPC_NS, 7, 9),
	BUSCRU_RESET_OFFSET(SRST_USER_OTPC_NS, 7, 10),
	BUSCRU_RESET_OFFSET(SRST_OTPC_ARB, 7, 11),
	BUSCRU_RESET_OFFSET(SRST_P_OTP_MASK, 7, 12),

	/* PERICRU-->SOFTRST_CON00 */
	PERICRU_RESET_OFFSET(SRST_A_PERI_BIU, 0, 0),
	PERICRU_RESET_OFFSET(SRST_P_PERI_BIU, 0, 1),
	PERICRU_RESET_OFFSET(SRST_P_RTC_BIU, 0, 2),
	PERICRU_RESET_OFFSET(SRST_P_CRU_PERI, 0, 3),
	PERICRU_RESET_OFFSET(SRST_P_PERI_GRF, 0, 4),
	PERICRU_RESET_OFFSET(SRST_P_GPIO1, 0, 5),
	PERICRU_RESET_OFFSET(SRST_DB_GPIO1, 0, 6),
	PERICRU_RESET_OFFSET(SRST_P_IOC_VCCIO1, 0, 7),
	PERICRU_RESET_OFFSET(SRST_A_USB3OTG, 0, 8),
	PERICRU_RESET_OFFSET(SRST_H_USB2HOST, 0, 11),
	PERICRU_RESET_OFFSET(SRST_H_ARB_USB2HOST, 0, 12),
	PERICRU_RESET_OFFSET(SRST_P_RTC_TEST, 0, 13),

	/* PERICRU-->SOFTRST_CON01 */
	PERICRU_RESET_OFFSET(SRST_H_EMMC, 1, 0),
	PERICRU_RESET_OFFSET(SRST_H_FSPI0, 1, 1),
	PERICRU_RESET_OFFSET(SRST_H_XIP_FSPI0, 1, 2),
	PERICRU_RESET_OFFSET(SRST_S_2X_FSPI0, 1, 3),
	PERICRU_RESET_OFFSET(SRST_UTMI_USB2HOST, 1, 5),
	PERICRU_RESET_OFFSET(SRST_REF_PIPEPHY, 1, 7),
	PERICRU_RESET_OFFSET(SRST_P_PIPEPHY, 1, 8),
	PERICRU_RESET_OFFSET(SRST_P_PIPEPHY_GRF, 1, 9),
	PERICRU_RESET_OFFSET(SRST_P_USB2PHY, 1, 10),
	PERICRU_RESET_OFFSET(SRST_POR_USB2PHY, 1, 11),
	PERICRU_RESET_OFFSET(SRST_OTG_USB2PHY, 1, 12),
	PERICRU_RESET_OFFSET(SRST_HOST_USB2PHY, 1, 13),

	/* CORECRU-->SOFTRST_CON00 */
	CORECRU_RESET_OFFSET(SRST_REF_PVTPLL_CORE, 0, 0),
	CORECRU_RESET_OFFSET(SRST_NCOREPORESET0, 0, 1),
	CORECRU_RESET_OFFSET(SRST_NCORESET0, 0, 2),
	CORECRU_RESET_OFFSET(SRST_NCOREPORESET1, 0, 3),
	CORECRU_RESET_OFFSET(SRST_NCORESET1, 0, 4),
	CORECRU_RESET_OFFSET(SRST_NCOREPORESET2, 0, 5),
	CORECRU_RESET_OFFSET(SRST_NCORESET2, 0, 6),
	CORECRU_RESET_OFFSET(SRST_NCOREPORESET3, 0, 7),
	CORECRU_RESET_OFFSET(SRST_NCORESET3, 0, 8),
	CORECRU_RESET_OFFSET(SRST_NDBGRESET, 0, 9),
	CORECRU_RESET_OFFSET(SRST_NL2RESET, 0, 10),

	/* CORECRU-->SOFTRST_CON01 */
	CORECRU_RESET_OFFSET(SRST_A_CORE_BIU, 1, 0),
	CORECRU_RESET_OFFSET(SRST_P_CORE_BIU, 1, 1),
	CORECRU_RESET_OFFSET(SRST_H_CORE_BIU, 1, 2),
	CORECRU_RESET_OFFSET(SRST_P_DBG, 1, 3),
	CORECRU_RESET_OFFSET(SRST_POT_DBG, 1, 4),
	CORECRU_RESET_OFFSET(SRST_NT_DBG, 1, 5),
	CORECRU_RESET_OFFSET(SRST_P_CORE_PVTPLL, 1, 6),
	CORECRU_RESET_OFFSET(SRST_P_CRU_CORE, 1, 7),
	CORECRU_RESET_OFFSET(SRST_P_CORE_GRF, 1, 8),
	CORECRU_RESET_OFFSET(SRST_P_DFT2APB, 1, 10),

	/* PMUCRU-->SOFTRST_CON00 */
	PMUCRU_RESET_OFFSET(SRST_H_PMU_BIU, 0, 0),
	PMUCRU_RESET_OFFSET(SRST_P_PMU_GPIO0, 0, 7),
	PMUCRU_RESET_OFFSET(SRST_DB_PMU_GPIO0, 0, 8),
	PMUCRU_RESET_OFFSET(SRST_P_PMU_HP_TIMER, 0, 10),
	PMUCRU_RESET_OFFSET(SRST_PMU_HP_TIMER, 0, 11),
	PMUCRU_RESET_OFFSET(SRST_PMU_32K_HP_TIMER, 0, 12),

	/* PMUCRU-->SOFTRST_CON01 */
	PMUCRU_RESET_OFFSET(SRST_P_PWM1, 1, 0),
	PMUCRU_RESET_OFFSET(SRST_PWM1, 1, 1),
	PMUCRU_RESET_OFFSET(SRST_P_I2C2, 1, 2),
	PMUCRU_RESET_OFFSET(SRST_I2C2, 1, 3),
	PMUCRU_RESET_OFFSET(SRST_P_UART0, 1, 4),
	PMUCRU_RESET_OFFSET(SRST_S_UART0, 1, 5),

	/* PMUCRU-->SOFTRST_CON02 */
	PMUCRU_RESET_OFFSET(SRST_P_RCOSC_CTRL, 2, 0),
	PMUCRU_RESET_OFFSET(SRST_REF_RCOSC_CTRL, 2, 2),
	PMUCRU_RESET_OFFSET(SRST_P_IOC_PMUIO0, 2, 3),
	PMUCRU_RESET_OFFSET(SRST_P_CRU_PMU, 2, 4),
	PMUCRU_RESET_OFFSET(SRST_P_PMU_GRF, 2, 5),
	PMUCRU_RESET_OFFSET(SRST_PREROLL, 2, 7),
	PMUCRU_RESET_OFFSET(SRST_PREROLL_32K, 2, 8),
	PMUCRU_RESET_OFFSET(SRST_H_PMU_SRAM, 2, 9),

	/* PMUCRU-->SOFTRST_CON03 */
	PMUCRU_RESET_OFFSET(SRST_P_WDT_LPMCU, 3, 0),
	PMUCRU_RESET_OFFSET(SRST_T_WDT_LPMCU, 3, 1),
	PMUCRU_RESET_OFFSET(SRST_LPMCU_FULL_CLUSTER, 3, 2),
	PMUCRU_RESET_OFFSET(SRST_LPMCU_PWUP, 3, 3),
	PMUCRU_RESET_OFFSET(SRST_LPMCU_ONLY_CORE, 3, 4),
	PMUCRU_RESET_OFFSET(SRST_T_LPMCU_JTAG, 3, 5),
	PMUCRU_RESET_OFFSET(SRST_P_LPMCU_MAILBOX, 3, 6),

	/* PMU1CRU-->SOFTRST_CON00 */
	PMU1CRU_RESET_OFFSET(SRST_P_SPI2AHB, 0, 0),
	PMU1CRU_RESET_OFFSET(SRST_H_SPI2AHB, 0, 1),
	PMU1CRU_RESET_OFFSET(SRST_H_FSPI1, 0, 2),
	PMU1CRU_RESET_OFFSET(SRST_H_XIP_FSPI1, 0, 3),
	PMU1CRU_RESET_OFFSET(SRST_S_1X_FSPI1, 0, 4),
	PMU1CRU_RESET_OFFSET(SRST_P_IOC_PMUIO1, 0, 5),
	PMU1CRU_RESET_OFFSET(SRST_P_CRU_PMU1, 0, 6),
	PMU1CRU_RESET_OFFSET(SRST_P_AUDIO_ADC_PMU, 0, 7),
	PMU1CRU_RESET_OFFSET(SRST_M_AUDIO_ADC_PMU, 0, 8),
	PMU1CRU_RESET_OFFSET(SRST_H_PMU1_BIU, 0, 9),

	/* PMU1CRU-->SOFTRST_CON01 */
	PMU1CRU_RESET_OFFSET(SRST_P_LPDMA, 1, 0),
	PMU1CRU_RESET_OFFSET(SRST_A_LPDMA, 1, 1),
	PMU1CRU_RESET_OFFSET(SRST_H_LPSAI, 1, 2),
	PMU1CRU_RESET_OFFSET(SRST_M_LPSAI, 1, 3),
	PMU1CRU_RESET_OFFSET(SRST_P_AOA_TDD, 1, 4),
	PMU1CRU_RESET_OFFSET(SRST_P_AOA_FE, 1, 5),
	PMU1CRU_RESET_OFFSET(SRST_P_AOA_AAD, 1, 6),
	PMU1CRU_RESET_OFFSET(SRST_P_AOA_APB, 1, 7),
	PMU1CRU_RESET_OFFSET(SRST_P_AOA_SRAM, 1, 8),

	/* DDRCRU-->SOFTRST_CON00 */
	DDRCRU_RESET_OFFSET(SRST_P_DDR_BIU, 0, 1),
	DDRCRU_RESET_OFFSET(SRST_P_DDRC, 0, 2),
	DDRCRU_RESET_OFFSET(SRST_P_DDRMON, 0, 3),
	DDRCRU_RESET_OFFSET(SRST_TIMER_DDRMON, 0, 4),
	DDRCRU_RESET_OFFSET(SRST_P_DFICTRL, 0, 5),
	DDRCRU_RESET_OFFSET(SRST_P_DDR_GRF, 0, 6),
	DDRCRU_RESET_OFFSET(SRST_P_CRU_DDR, 0, 7),
	DDRCRU_RESET_OFFSET(SRST_P_DDRPHY, 0, 8),
	DDRCRU_RESET_OFFSET(SRST_P_DMA2DDR, 0, 9),

	/* SUBDDRCRU-->SOFTRST_CON00 */
	SUBDDRCRU_RESET_OFFSET(SRST_A_SYSMEM_BIU, 0, 0),
	SUBDDRCRU_RESET_OFFSET(SRST_A_SYSMEM, 0, 1),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DDR_BIU, 0, 2),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DDRSCH0_CPU, 0, 3),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DDRSCH1_NPU, 0, 4),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DDRSCH2_POE, 0, 5),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DDRSCH3_VI, 0, 6),
	SUBDDRCRU_RESET_OFFSET(SRST_CORE_DDRC, 0, 7),
	SUBDDRCRU_RESET_OFFSET(SRST_DDRMON, 0, 8),
	SUBDDRCRU_RESET_OFFSET(SRST_DFICTRL, 0, 9),
	SUBDDRCRU_RESET_OFFSET(SRST_RS, 0, 11),
	SUBDDRCRU_RESET_OFFSET(SRST_A_DMA2DDR, 0, 12),
	SUBDDRCRU_RESET_OFFSET(SRST_DDRPHY, 0, 13),

	/* VICRU-->SOFTRST_CON00 */
	VICRU_RESET_OFFSET(SRST_REF_PVTPLL_ISP, 0, 0),
	VICRU_RESET_OFFSET(SRST_A_GMAC_BIU, 0, 1),
	VICRU_RESET_OFFSET(SRST_A_VI_BIU, 0, 2),
	VICRU_RESET_OFFSET(SRST_H_VI_BIU, 0, 3),
	VICRU_RESET_OFFSET(SRST_P_VI_BIU, 0, 4),
	VICRU_RESET_OFFSET(SRST_P_CRU_VI, 0, 5),
	VICRU_RESET_OFFSET(SRST_P_VI_GRF, 0, 6),
	VICRU_RESET_OFFSET(SRST_P_VI_PVTPLL, 0, 7),
	VICRU_RESET_OFFSET(SRST_P_DSMC, 0, 8),
	VICRU_RESET_OFFSET(SRST_A_DSMC, 0, 9),
	VICRU_RESET_OFFSET(SRST_H_CAN0, 0, 10),
	VICRU_RESET_OFFSET(SRST_CAN0, 0, 11),
	VICRU_RESET_OFFSET(SRST_H_CAN1, 0, 12),
	VICRU_RESET_OFFSET(SRST_CAN1, 0, 13),

	/* VICRU-->SOFTRST_CON01 */
	VICRU_RESET_OFFSET(SRST_P_GPIO2, 1, 0),
	VICRU_RESET_OFFSET(SRST_DB_GPIO2, 1, 1),
	VICRU_RESET_OFFSET(SRST_P_GPIO4, 1, 2),
	VICRU_RESET_OFFSET(SRST_DB_GPIO4, 1, 3),
	VICRU_RESET_OFFSET(SRST_P_GPIO5, 1, 4),
	VICRU_RESET_OFFSET(SRST_DB_GPIO5, 1, 5),
	VICRU_RESET_OFFSET(SRST_P_GPIO6, 1, 6),
	VICRU_RESET_OFFSET(SRST_DB_GPIO6, 1, 7),
	VICRU_RESET_OFFSET(SRST_P_GPIO7, 1, 8),
	VICRU_RESET_OFFSET(SRST_DB_GPIO7, 1, 9),
	VICRU_RESET_OFFSET(SRST_P_IOC_VCCIO2, 1, 10),
	VICRU_RESET_OFFSET(SRST_P_IOC_VCCIO4, 1, 11),
	VICRU_RESET_OFFSET(SRST_P_IOC_VCCIO5, 1, 12),
	VICRU_RESET_OFFSET(SRST_P_IOC_VCCIO6, 1, 13),
	VICRU_RESET_OFFSET(SRST_P_IOC_VCCIO7, 1, 14),

	/* VICRU-->SOFTRST_CON02 */
	VICRU_RESET_OFFSET(SRST_CORE_ISP, 2, 0),
	VICRU_RESET_OFFSET(SRST_H_VICAP, 2, 1),
	VICRU_RESET_OFFSET(SRST_A_VICAP, 2, 2),
	VICRU_RESET_OFFSET(SRST_D_VICAP, 2, 3),
	VICRU_RESET_OFFSET(SRST_ISP0_VICAP, 2, 4),
	VICRU_RESET_OFFSET(SRST_CORE_VPSS, 2, 5),
	VICRU_RESET_OFFSET(SRST_CORE_VPSL, 2, 6),
	VICRU_RESET_OFFSET(SRST_P_CSI2HOST0, 2, 7),
	VICRU_RESET_OFFSET(SRST_P_CSI2HOST1, 2, 8),
	VICRU_RESET_OFFSET(SRST_P_CSI2HOST2, 2, 9),
	VICRU_RESET_OFFSET(SRST_P_CSI2HOST3, 2, 10),
	VICRU_RESET_OFFSET(SRST_H_SDMMC0, 2, 11),
	VICRU_RESET_OFFSET(SRST_A_GMAC, 2, 12),
	VICRU_RESET_OFFSET(SRST_P_CSIPHY0, 2, 13),
	VICRU_RESET_OFFSET(SRST_P_CSIPHY1, 2, 14),

	/* VICRU-->SOFTRST_CON03 */
	VICRU_RESET_OFFSET(SRST_P_MACPHY, 3, 0),
	VICRU_RESET_OFFSET(SRST_MACPHY, 3, 1),
	VICRU_RESET_OFFSET(SRST_P_SARADC1, 3, 2),
	VICRU_RESET_OFFSET(SRST_SARADC1, 3, 3),
	VICRU_RESET_OFFSET(SRST_P_SARADC2, 3, 5),
	VICRU_RESET_OFFSET(SRST_SARADC2, 3, 6),

	/* VEPUCRU-->SOFTRST_CON00 */
	VEPUCRU_RESET_OFFSET(SRST_REF_PVTPLL_VEPU, 0, 0),
	VEPUCRU_RESET_OFFSET(SRST_A_VEPU_BIU, 0, 1),
	VEPUCRU_RESET_OFFSET(SRST_H_VEPU_BIU, 0, 2),
	VEPUCRU_RESET_OFFSET(SRST_P_VEPU_BIU, 0, 3),
	VEPUCRU_RESET_OFFSET(SRST_P_CRU_VEPU, 0, 4),
	VEPUCRU_RESET_OFFSET(SRST_P_VEPU_GRF, 0, 5),
	VEPUCRU_RESET_OFFSET(SRST_P_GPIO3, 0, 7),
	VEPUCRU_RESET_OFFSET(SRST_DB_GPIO3, 0, 8),
	VEPUCRU_RESET_OFFSET(SRST_P_IOC_VCCIO3, 0, 9),
	VEPUCRU_RESET_OFFSET(SRST_P_SARADC0, 0, 10),
	VEPUCRU_RESET_OFFSET(SRST_SARADC0, 0, 11),
	VEPUCRU_RESET_OFFSET(SRST_H_SDMMC1, 0, 13),

	/* VEPUCRU-->SOFTRST_CON01 */
	VEPUCRU_RESET_OFFSET(SRST_P_VEPU_PVTPLL, 1, 0),
	VEPUCRU_RESET_OFFSET(SRST_H_VEPU, 1, 1),
	VEPUCRU_RESET_OFFSET(SRST_A_VEPU, 1, 2),
	VEPUCRU_RESET_OFFSET(SRST_CORE_VEPU, 1, 3),

	/* NPUCRU-->SOFTRST_CON00 */
	NPUCRU_RESET_OFFSET(SRST_REF_PVTPLL_NPU, 0, 0),
	NPUCRU_RESET_OFFSET(SRST_A_NPU_BIU, 0, 2),
	NPUCRU_RESET_OFFSET(SRST_H_NPU_BIU, 0, 3),
	NPUCRU_RESET_OFFSET(SRST_P_NPU_BIU, 0, 4),
	NPUCRU_RESET_OFFSET(SRST_P_CRU_NPU, 0, 5),
	NPUCRU_RESET_OFFSET(SRST_P_NPU_GRF, 0, 6),
	NPUCRU_RESET_OFFSET(SRST_P_NPU_PVTPLL, 0, 8),
	NPUCRU_RESET_OFFSET(SRST_H_RKNN, 0, 9),
	NPUCRU_RESET_OFFSET(SRST_A_RKNN, 0, 10),

	/* VDOCRU-->SOFTRST_CON00 */
	VDOCRU_RESET_OFFSET(SRST_A_RKVDEC_BIU, 0, 0),
	VDOCRU_RESET_OFFSET(SRST_A_VDO_BIU, 0, 1),
	VDOCRU_RESET_OFFSET(SRST_H_VDO_BIU, 0, 3),
	VDOCRU_RESET_OFFSET(SRST_P_VDO_BIU, 0, 4),
	VDOCRU_RESET_OFFSET(SRST_P_CRU_VDO, 0, 5),
	VDOCRU_RESET_OFFSET(SRST_P_VDO_GRF, 0, 6),
	VDOCRU_RESET_OFFSET(SRST_A_RKVDEC, 0, 7),
	VDOCRU_RESET_OFFSET(SRST_H_RKVDEC, 0, 8),
	VDOCRU_RESET_OFFSET(SRST_HEVC_CA_RKVDEC, 0, 9),
	VDOCRU_RESET_OFFSET(SRST_A_VOP, 0, 10),
	VDOCRU_RESET_OFFSET(SRST_H_VOP, 0, 11),
	VDOCRU_RESET_OFFSET(SRST_D_VOP, 0, 12),
	VDOCRU_RESET_OFFSET(SRST_A_OOC, 0, 13),
	VDOCRU_RESET_OFFSET(SRST_H_OOC, 0, 14),
	VDOCRU_RESET_OFFSET(SRST_D_OOC, 0, 15),

	/* VDOCRU-->SOFTRST_CON01 */
	VDOCRU_RESET_OFFSET(SRST_H_RKJPEG, 1, 3),
	VDOCRU_RESET_OFFSET(SRST_A_RKJPEG, 1, 4),
	VDOCRU_RESET_OFFSET(SRST_A_RKMMU_DECOM, 1, 5),
	VDOCRU_RESET_OFFSET(SRST_H_RKMMU_DECOM, 1, 6),
	VDOCRU_RESET_OFFSET(SRST_D_DECOM, 1, 8),
	VDOCRU_RESET_OFFSET(SRST_A_DECOM, 1, 9),
	VDOCRU_RESET_OFFSET(SRST_P_DECOM, 1, 10),
	VDOCRU_RESET_OFFSET(SRST_P_MIPI_DSI, 1, 12),
	VDOCRU_RESET_OFFSET(SRST_P_DSIPHY, 1, 13),

	/* VCPCRU-->SOFTRST_CON00 */
	VCPCRU_RESET_OFFSET(SRST_REF_PVTPLL_VCP, 0, 0),
	VCPCRU_RESET_OFFSET(SRST_A_VCP_BIU, 0, 1),
	VCPCRU_RESET_OFFSET(SRST_H_VCP_BIU, 0, 2),
	VCPCRU_RESET_OFFSET(SRST_P_VCP_BIU, 0, 3),
	VCPCRU_RESET_OFFSET(SRST_P_CRU_VCP, 0, 4),
	VCPCRU_RESET_OFFSET(SRST_P_VCP_GRF, 0, 5),
	VCPCRU_RESET_OFFSET(SRST_P_VCP_PVTPLL, 0, 7),
	VCPCRU_RESET_OFFSET(SRST_A_AISP_BIU, 0, 8),
	VCPCRU_RESET_OFFSET(SRST_H_AISP_BIU, 0, 9),
	VCPCRU_RESET_OFFSET(SRST_CORE_AISP, 0, 13),

	/* VCPCRU-->SOFTRST_CON01 */
	VCPCRU_RESET_OFFSET(SRST_H_FEC, 1, 0),
	VCPCRU_RESET_OFFSET(SRST_A_FEC, 1, 1),
	VCPCRU_RESET_OFFSET(SRST_CORE_FEC, 1, 2),
	VCPCRU_RESET_OFFSET(SRST_H_AVSP, 1, 3),
	VCPCRU_RESET_OFFSET(SRST_A_AVSP, 1, 4),
};

void rv1126b_rst_init(struct device_node *np, void __iomem *reg_base)
{
	rockchip_register_softrst_lut(np,
				      rv1126b_register_offset,
				      ARRAY_SIZE(rv1126b_register_offset),
				      reg_base + RV1126B_SOFTRST_CON(0),
				      ROCKCHIP_SOFTRST_HIWORD_MASK);
}
