/* SPDX-License-Identifier: GPL-2.0 */
/* Copyright (c) Meta Platforms, Inc. and affiliates. */

#ifndef _FBNIC_CSR_H_
#define _FBNIC_CSR_H_

#include <linux/bitops.h>

#define CSR_BIT(nr)		(1u << (nr))
#define CSR_GENMASK(h, l)	GENMASK(h, l)

#define DESC_BIT(nr)		BIT_ULL(nr)
#define DESC_GENMASK(h, l)	GENMASK_ULL(h, l)

#define FW_VER_CODE(_major, _minor, _patch, _build) (		      \
		FIELD_PREP(FBNIC_FW_CAP_RESP_VERSION_MAJOR, _major) | \
		FIELD_PREP(FBNIC_FW_CAP_RESP_VERSION_MINOR, _minor) | \
		FIELD_PREP(FBNIC_FW_CAP_RESP_VERSION_PATCH, _patch) | \
		FIELD_PREP(FBNIC_FW_CAP_RESP_VERSION_BUILD, _build))

/* Defines the minimum firmware version required by the driver */
#define MIN_FW_VER_CODE				FW_VER_CODE(0, 10, 6, 0)

/* Defines the minimum firmware version required for firmware logs */
#define MIN_FW_VER_CODE_LOG			FW_VER_CODE(0, 12, 9, 0)

/* Driver can request that firmware sends all cached logs in bulk. This
 * feature was enabled on older firmware however firmware has a bug
 * which attempted to send 30 messages per mbx message which caused an
 * overflow flooding the mailbox. This results in a kernel warning
 * related to corrupt mailbox messages.
 *
 * If firmware is new enough only request sending historical logs when
 * the log buffer is empty to prevent duplicate logs.
 */
#define MIN_FW_VER_CODE_HIST			FW_VER_CODE(25, 5, 7, 0)

#define PCI_DEVICE_ID_META_FBNIC_ASIC		0x0013

#define FBNIC_CLOCK_FREQ	(600 * (1000 * 1000))

/* Transmit Work Descriptor Format */
/* Length, Type, Offset Masks and Shifts */
#define FBNIC_TWD_L2_HLEN_MASK			DESC_GENMASK(5, 0)

#define FBNIC_TWD_L3_TYPE_MASK			DESC_GENMASK(7, 6)
enum {
	FBNIC_TWD_L3_TYPE_OTHER	= 0,
	FBNIC_TWD_L3_TYPE_IPV4	= 1,
	FBNIC_TWD_L3_TYPE_IPV6	= 2,
	FBNIC_TWD_L3_TYPE_V6V6	= 3,
};

#define FBNIC_TWD_L3_OHLEN_MASK			DESC_GENMASK(15, 8)
#define FBNIC_TWD_L3_IHLEN_MASK			DESC_GENMASK(23, 16)

enum {
	FBNIC_TWD_L4_TYPE_OTHER	= 0,
	FBNIC_TWD_L4_TYPE_TCP	= 1,
	FBNIC_TWD_L4_TYPE_UDP	= 2,
};

#define FBNIC_TWD_CSUM_OFFSET_MASK		DESC_GENMASK(27, 24)
#define FBNIC_TWD_L4_HLEN_MASK			DESC_GENMASK(31, 28)

/* Flags and Type */
#define FBNIC_TWD_L4_TYPE_MASK			DESC_GENMASK(33, 32)
#define FBNIC_TWD_FLAG_REQ_TS			DESC_BIT(34)
#define FBNIC_TWD_FLAG_REQ_LSO			DESC_BIT(35)
#define FBNIC_TWD_FLAG_REQ_CSO			DESC_BIT(36)
#define FBNIC_TWD_FLAG_REQ_COMPLETION		DESC_BIT(37)
#define FBNIC_TWD_FLAG_DEST_MAC			DESC_BIT(43)
#define FBNIC_TWD_FLAG_DEST_BMC			DESC_BIT(44)
#define FBNIC_TWD_FLAG_DEST_FW			DESC_BIT(45)
#define FBNIC_TWD_TYPE_MASK			DESC_GENMASK(47, 46)
enum {
	FBNIC_TWD_TYPE_META	= 0,
	FBNIC_TWD_TYPE_OPT_META	= 1,
	FBNIC_TWD_TYPE_AL	= 2,
	FBNIC_TWD_TYPE_LAST_AL	= 3,
};

/* MSS and Completion Req */
#define FBNIC_TWD_MSS_MASK			DESC_GENMASK(61, 48)

#define FBNIC_TWD_TS_MASK			DESC_GENMASK(39, 0)
#define FBNIC_TWD_ADDR_MASK			DESC_GENMASK(45, 0)
#define FBNIC_TWD_LEN_MASK			DESC_GENMASK(63, 48)

/* Tx Completion Descriptor Format */
#define FBNIC_TCD_TYPE0_HEAD0_MASK		DESC_GENMASK(15, 0)
#define FBNIC_TCD_TYPE0_HEAD1_MASK		DESC_GENMASK(31, 16)

#define FBNIC_TCD_TYPE1_TS_MASK			DESC_GENMASK(39, 0)

#define FBNIC_TCD_STATUS_MASK			DESC_GENMASK(59, 48)
#define FBNIC_TCD_STATUS_TS_INVALID		DESC_BIT(48)
#define FBNIC_TCD_STATUS_ILLEGAL_TS_REQ		DESC_BIT(49)
#define FBNIC_TCD_TWQ1				DESC_BIT(60)
#define FBNIC_TCD_TYPE_MASK			DESC_GENMASK(62, 61)
enum {
	FBNIC_TCD_TYPE_0	= 0,
	FBNIC_TCD_TYPE_1	= 1,
};

#define FBNIC_TCD_DONE				DESC_BIT(63)

/* Rx Buffer Descriptor Format
 *
 * The layout of this can vary depending on the page size of the system.
 *
 * If the page size is 4K then the layout will simply consist of ID for
 * the 16 most significant bits, and the lower 46 are essentially the page
 * address with the lowest 12 bits being reserved 0 due to the fact that
 * a page will be aligned.
 *
 * If the page size is larger than 4K then the lower n bits of the ID and
 * page address will be reserved for the fragment ID. This fragment will
 * be 4K in size and will be used to index both the DMA address and the ID
 * by the same amount.
 */
#define FBNIC_BD_DESC_ADDR_MASK			DESC_GENMASK(45, 12)
#define FBNIC_BD_DESC_ID_MASK			DESC_GENMASK(63, 48)
#define FBNIC_BD_FRAG_SIZE \
	(FBNIC_BD_DESC_ADDR_MASK & ~(FBNIC_BD_DESC_ADDR_MASK - 1))
#define FBNIC_BD_FRAG_COUNT \
	(PAGE_SIZE / FBNIC_BD_FRAG_SIZE)
#define FBNIC_BD_FRAG_ADDR_MASK \
	(FBNIC_BD_DESC_ADDR_MASK & \
	 ~(FBNIC_BD_DESC_ADDR_MASK * FBNIC_BD_FRAG_COUNT))
#define FBNIC_BD_FRAG_ID_MASK \
	(FBNIC_BD_DESC_ID_MASK & \
	 ~(FBNIC_BD_DESC_ID_MASK * FBNIC_BD_FRAG_COUNT))
#define FBNIC_BD_PAGE_ADDR_MASK \
	(FBNIC_BD_DESC_ADDR_MASK & ~FBNIC_BD_FRAG_ADDR_MASK)
#define FBNIC_BD_PAGE_ID_MASK \
	(FBNIC_BD_DESC_ID_MASK & ~FBNIC_BD_FRAG_ID_MASK)

/* Rx Completion Queue Descriptors */
#define FBNIC_RCD_TYPE_MASK			DESC_GENMASK(62, 61)
enum {
	FBNIC_RCD_TYPE_HDR_AL	= 0,
	FBNIC_RCD_TYPE_PAY_AL	= 1,
	FBNIC_RCD_TYPE_OPT_META	= 2,
	FBNIC_RCD_TYPE_META	= 3,
};

#define FBNIC_RCD_DONE				DESC_BIT(63)

/* Address/Length Completion Descriptors */
#define FBNIC_RCD_AL_BUFF_ID_MASK		DESC_GENMASK(15, 0)
#define FBNIC_RCD_AL_BUFF_FRAG_MASK		(FBNIC_BD_FRAG_COUNT - 1)
#define FBNIC_RCD_AL_BUFF_PAGE_MASK \
	(FBNIC_RCD_AL_BUFF_ID_MASK & ~FBNIC_RCD_AL_BUFF_FRAG_MASK)
#define FBNIC_RCD_AL_BUFF_LEN_MASK		DESC_GENMASK(28, 16)
#define FBNIC_RCD_AL_BUFF_OFF_MASK		DESC_GENMASK(43, 32)
#define FBNIC_RCD_AL_PAGE_FIN			DESC_BIT(60)

/* Header AL specific values */
#define FBNIC_RCD_HDR_AL_OVERFLOW		DESC_BIT(53)
#define FBNIC_RCD_HDR_AL_DMA_HINT_MASK		DESC_GENMASK(59, 54)
enum {
	FBNIC_RCD_HDR_AL_DMA_HINT_NONE  = 0,
	FBNIC_RCD_HDR_AL_DMA_HINT_L2	= 1,
	FBNIC_RCD_HDR_AL_DMA_HINT_L3	= 2,
	FBNIC_RCD_HDR_AL_DMA_HINT_L4	= 4,
};

/* Optional Metadata Completion Descriptors */
#define FBNIC_RCD_OPT_META_TS_MASK		DESC_GENMASK(39, 0)
#define FBNIC_RCD_OPT_META_ACTION_MASK		DESC_GENMASK(45, 40)
#define FBNIC_RCD_OPT_META_ACTION		DESC_BIT(57)
#define FBNIC_RCD_OPT_META_TS			DESC_BIT(58)
#define FBNIC_RCD_OPT_META_TYPE_MASK		DESC_GENMASK(60, 59)

/* Metadata Completion Descriptors */
#define FBNIC_RCD_META_RSS_HASH_MASK		DESC_GENMASK(31, 0)
#define FBNIC_RCD_META_L2_CSUM_MASK		DESC_GENMASK(47, 32)
#define FBNIC_RCD_META_L3_TYPE_MASK		DESC_GENMASK(49, 48)
enum {
	FBNIC_RCD_META_L3_TYPE_OTHER	= 0,
	FBNIC_RCD_META_L3_TYPE_IPV4	= 1,
	FBNIC_RCD_META_L3_TYPE_IPV6	= 2,
	FBNIC_RCD_META_L3_TYPE_V6V6	= 3,
};

#define FBNIC_RCD_META_L4_TYPE_MASK		DESC_GENMASK(51, 50)
enum {
	FBNIC_RCD_META_L4_TYPE_OTHER	= 0,
	FBNIC_RCD_META_L4_TYPE_TCP	= 1,
	FBNIC_RCD_META_L4_TYPE_UDP	= 2,
};

#define FBNIC_RCD_META_L4_CSUM_UNNECESSARY	DESC_BIT(52)
#define FBNIC_RCD_META_ERR_MAC_EOP		DESC_BIT(53)
#define FBNIC_RCD_META_ERR_TRUNCATED_FRAME	DESC_BIT(54)
#define FBNIC_RCD_META_ERR_PARSER		DESC_BIT(55)
#define FBNIC_RCD_META_UNCORRECTABLE_ERR_MASK	\
	(FBNIC_RCD_META_ERR_MAC_EOP | FBNIC_RCD_META_ERR_TRUNCATED_FRAME)
#define FBNIC_RCD_META_ECN			DESC_BIT(60)

/* Register Definitions
 *
 * The registers are laid as indexes into an le32 array. As such the actual
 * address is 4 times the index value. Below each register is defined as 3
 * fields, name, index, and Address.
 *
 *      Name				Index		Address
 *************************************************************************/
/* Interrupt Registers */
#define FBNIC_CSR_START_INTR		0x00000	/* CSR section delimiter */
#define FBNIC_INTR_STATUS(n)		(0x00000 + (n))	/* 0x00000 + 4*n */
#define FBNIC_INTR_STATUS_CNT			8
#define FBNIC_INTR_MASK(n)		(0x00008 + (n)) /* 0x00020 + 4*n */
#define FBNIC_INTR_MASK_CNT			8
#define FBNIC_INTR_SET(n)		(0x00010 + (n))	/* 0x00040 + 4*n */
#define FBNIC_INTR_SET_CNT			8
#define FBNIC_INTR_CLEAR(n)		(0x00018 + (n))	/* 0x00060 + 4*n */
#define FBNIC_INTR_CLEAR_CNT			8
#define FBNIC_INTR_SW_STATUS(n)		(0x00020 + (n)) /* 0x00080 + 4*n */
#define FBNIC_INTR_SW_STATUS_CNT		8
#define FBNIC_INTR_SW_AC_MODE(n)	(0x00028 + (n)) /* 0x000a0 + 4*n */
#define FBNIC_INTR_SW_AC_MODE_CNT		8
#define FBNIC_INTR_MASK_SET(n)		(0x00030 + (n)) /* 0x000c0 + 4*n */
#define FBNIC_INTR_MASK_SET_CNT			8
#define FBNIC_INTR_MASK_CLEAR(n)	(0x00038 + (n)) /* 0x000e0 + 4*n */
#define FBNIC_INTR_MASK_CLEAR_CNT		8
#define FBNIC_MAX_MSIX_VECS		256U
#define FBNIC_INTR_MSIX_CTRL(n)		(0x00040 + (n)) /* 0x00100 + 4*n */
#define FBNIC_INTR_MSIX_CTRL_VECTOR_MASK	CSR_GENMASK(7, 0)
#define FBNIC_INTR_MSIX_CTRL_ENABLE		CSR_BIT(31)
enum {
	FBNIC_INTR_MSIX_CTRL_PCS_IDX	= 34,
};

#define FBNIC_CSR_END_INTR		0x0005f	/* CSR section delimiter */

/* Interrupt MSIX Registers */
#define FBNIC_CSR_START_INTR_CQ		0x00400	/* CSR section delimiter */
#define FBNIC_INTR_CQ_REARM(n) \
				(0x00400 + 4 * (n))	/* 0x01000 + 16*n */
#define FBNIC_INTR_CQ_REARM_CNT			256
#define FBNIC_INTR_CQ_REARM_RCQ_TIMEOUT		CSR_GENMASK(13, 0)
#define FBNIC_INTR_CQ_REARM_RCQ_TIMEOUT_UPD_EN	CSR_BIT(14)
#define FBNIC_INTR_CQ_REARM_TCQ_TIMEOUT		CSR_GENMASK(28, 15)
#define FBNIC_INTR_CQ_REARM_TCQ_TIMEOUT_UPD_EN	CSR_BIT(29)
#define FBNIC_INTR_CQ_REARM_INTR_RELOAD		CSR_BIT(30)
#define FBNIC_INTR_CQ_REARM_INTR_UNMASK		CSR_BIT(31)

#define FBNIC_INTR_RCQ_TIMEOUT(n) \
				(0x00401 + 4 * (n))	/* 0x01004 + 16*n */
#define FBNIC_INTR_RCQ_TIMEOUT_CNT		256
#define FBNIC_INTR_TCQ_TIMEOUT(n) \
				(0x00402 + 4 * (n))	/* 0x01008 + 16*n */
#define FBNIC_INTR_TCQ_TIMEOUT_CNT		256
#define FBNIC_CSR_END_INTR_CQ		0x007fe	/* CSR section delimiter */

/* Global QM Tx registers */
#define FBNIC_CSR_START_QM_TX		0x00800	/* CSR section delimiter */
#define FBNIC_QM_TWQ_IDLE(n)		(0x00800 + (n)) /* 0x02000 + 4*n */
#define FBNIC_QM_TWQ_IDLE_CNT			8
#define FBNIC_QM_TWQ_DEFAULT_META_L	0x00818		/* 0x02060 */
#define FBNIC_QM_TWQ_DEFAULT_META_H	0x00819		/* 0x02064 */

#define FBNIC_QM_TQS_CTL0		0x0081b		/* 0x0206c */
#define FBNIC_QM_TQS_CTL0_LSO_TS_MASK	CSR_BIT(0)
enum {
	FBNIC_QM_TQS_CTL0_LSO_TS_FIRST	= 0,
	FBNIC_QM_TQS_CTL0_LSO_TS_LAST	= 1,
};

#define FBNIC_QM_TQS_CTL0_PREFETCH_THRESH	CSR_GENMASK(7, 1)
enum {
	FBNIC_QM_TQS_CTL0_PREFETCH_THRESH_MIN	= 16,
};

#define FBNIC_QM_TQS_CTL1		0x0081c		/* 0x02070 */
#define FBNIC_QM_TQS_CTL1_MC_MAX_CREDITS	CSR_GENMASK(7, 0)
#define FBNIC_QM_TQS_CTL1_BULK_MAX_CREDITS	CSR_GENMASK(15, 8)
#define FBNIC_QM_TQS_MTU_CTL0		0x0081d		/* 0x02074 */
#define FBNIC_QM_TQS_MTU_CTL1		0x0081e		/* 0x02078 */
#define FBNIC_QM_TQS_MTU_CTL1_BULK		CSR_GENMASK(13, 0)
#define FBNIC_QM_TCQ_IDLE(n)		(0x00821 + (n)) /* 0x02084 + 4*n */
#define FBNIC_QM_TCQ_IDLE_CNT			4
#define FBNIC_QM_TCQ_CTL0		0x0082d		/* 0x020b4 */
#define FBNIC_QM_TCQ_CTL0_COAL_WAIT		CSR_GENMASK(15, 0)
#define FBNIC_QM_TCQ_CTL0_TICK_CYCLES		CSR_GENMASK(26, 16)
#define FBNIC_QM_TQS_IDLE(n)		(0x00830 + (n)) /* 0x020c0 + 4*n */
#define FBNIC_QM_TQS_IDLE_CNT			8
#define FBNIC_QM_TQS_EDT_TS_RANGE	0x00849		/* 0x2124 */
#define FBNIC_QM_TDE_IDLE(n)		(0x00853 + (n)) /* 0x0214c + 4*n */
#define FBNIC_QM_TDE_IDLE_CNT			8
#define FBNIC_QM_TNI_TDF_CTL		0x0086c		/* 0x021b0 */
#define FBNIC_QM_TNI_TDF_CTL_MRRS		CSR_GENMASK(1, 0)
#define FBNIC_QM_TNI_TDF_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_TNI_TDF_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_TNI_TDF_CTL_MAX_OB		CSR_GENMASK(23, 12)
#define FBNIC_QM_TNI_TDE_CTL		0x0086d		/* 0x021b4 */
#define FBNIC_QM_TNI_TDE_CTL_MRRS		CSR_GENMASK(1, 0)
#define FBNIC_QM_TNI_TDE_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_TNI_TDE_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_TNI_TDE_CTL_MAX_OB		CSR_GENMASK(24, 12)
#define FBNIC_QM_TNI_TDE_CTL_MRRS_1K		CSR_BIT(25)
#define FBNIC_QM_TNI_TCM_CTL		0x0086e		/* 0x021b8 */
#define FBNIC_QM_TNI_TCM_CTL_MPS		CSR_GENMASK(1, 0)
#define FBNIC_QM_TNI_TCM_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_TNI_TCM_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_TNI_TCM_CTL_MAX_OB		CSR_GENMASK(23, 12)
#define FBNIC_CSR_END_QM_TX		0x00873	/* CSR section delimiter */

/* Global QM Rx registers */
#define FBNIC_CSR_START_QM_RX		0x00c00	/* CSR section delimiter */
#define FBNIC_QM_RCQ_IDLE(n)		(0x00c00 + (n)) /* 0x03000 + 4*n */
#define FBNIC_QM_RCQ_IDLE_CNT			4
#define FBNIC_QM_RCQ_CTL0		0x00c0c		/* 0x03030 */
#define FBNIC_QM_RCQ_CTL0_COAL_WAIT		CSR_GENMASK(15, 0)
#define FBNIC_QM_RCQ_CTL0_TICK_CYCLES		CSR_GENMASK(26, 16)
#define FBNIC_QM_HPQ_IDLE(n)		(0x00c0f + (n)) /* 0x0303c + 4*n */
#define FBNIC_QM_HPQ_IDLE_CNT			4
#define FBNIC_QM_PPQ_IDLE(n)		(0x00c13 + (n)) /* 0x0304c + 4*n */
#define FBNIC_QM_PPQ_IDLE_CNT			4
#define FBNIC_QM_RNI_RBP_CTL		0x00c2d		/* 0x030b4 */
#define FBNIC_QM_RNI_RBP_CTL_MRRS		CSR_GENMASK(1, 0)
#define FBNIC_QM_RNI_RBP_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_RNI_RBP_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_RNI_RBP_CTL_MAX_OB		CSR_GENMASK(23, 12)
#define FBNIC_QM_RNI_RDE_CTL		0x00c2e		/* 0x030b8 */
#define FBNIC_QM_RNI_RDE_CTL_MPS		CSR_GENMASK(1, 0)
#define FBNIC_QM_RNI_RDE_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_RNI_RDE_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_RNI_RDE_CTL_MAX_OB		CSR_GENMASK(23, 12)
#define FBNIC_QM_RNI_RCM_CTL		0x00c2f		/* 0x030bc */
#define FBNIC_QM_RNI_RCM_CTL_MPS		CSR_GENMASK(1, 0)
#define FBNIC_QM_RNI_RCM_CTL_CLS		CSR_GENMASK(3, 2)
#define FBNIC_QM_RNI_RCM_CTL_MAX_OT		CSR_GENMASK(11, 4)
#define FBNIC_QM_RNI_RCM_CTL_MAX_OB		CSR_GENMASK(23, 12)
#define FBNIC_CSR_END_QM_RX		0x00c34	/* CSR section delimiter */

/* TCE registers */
#define FBNIC_CSR_START_TCE		0x04000	/* CSR section delimiter */
#define FBNIC_TCE_REG_BASE		0x04000		/* 0x10000 */

#define FBNIC_TCE_LSO_CTRL		0x04000		/* 0x10000 */
#define FBNIC_TCE_LSO_CTRL_TCPF_CLR_1ST		CSR_GENMASK(8, 0)
#define FBNIC_TCE_LSO_CTRL_TCPF_CLR_MID		CSR_GENMASK(17, 9)
#define FBNIC_TCE_LSO_CTRL_TCPF_CLR_END		CSR_GENMASK(26, 18)
#define FBNIC_TCE_LSO_CTRL_IPID_MODE_INC	CSR_BIT(27)

#define FBNIC_TCE_CSO_CTRL		0x04001		/* 0x10004 */
#define FBNIC_TCE_CSO_CTRL_TCP_ZERO_CSUM	CSR_BIT(0)

#define FBNIC_TCE_TXB_CTRL		0x04002		/* 0x10008 */
#define FBNIC_TCE_TXB_CTRL_LOAD			CSR_BIT(0)
#define FBNIC_TCE_TXB_CTRL_TCAM_ENABLE		CSR_BIT(1)
#define FBNIC_TCE_TXB_CTRL_DISABLE		CSR_BIT(2)

#define FBNIC_TCE_TXB_ENQ_WRR_CTRL	0x04003		/* 0x1000c */
#define FBNIC_TCE_TXB_ENQ_WRR_CTRL_WEIGHT0	CSR_GENMASK(7, 0)
#define FBNIC_TCE_TXB_ENQ_WRR_CTRL_WEIGHT1	CSR_GENMASK(15, 8)
#define FBNIC_TCE_TXB_ENQ_WRR_CTRL_WEIGHT2	CSR_GENMASK(23, 16)

#define FBNIC_TCE_TXB_TEI_Q0_CTRL	0x04004		/* 0x10010 */
#define FBNIC_TCE_TXB_TEI_Q1_CTRL	0x04005		/* 0x10014 */
#define FBNIC_TCE_TXB_MC_Q_CTRL		0x04006		/* 0x10018 */
#define FBNIC_TCE_TXB_RX_TEI_Q_CTRL	0x04007		/* 0x1001c */
#define FBNIC_TCE_TXB_RX_BMC_Q_CTRL	0x04008		/* 0x10020 */
#define FBNIC_TCE_TXB_Q_CTRL_START		CSR_GENMASK(10, 0)
#define FBNIC_TCE_TXB_Q_CTRL_SIZE		CSR_GENMASK(22, 11)

#define FBNIC_TCE_TXB_TEI_DWRR_CTRL	0x04009		/* 0x10024 */
#define FBNIC_TCE_TXB_TEI_DWRR_CTRL_QUANTUM0	CSR_GENMASK(7, 0)
#define FBNIC_TCE_TXB_TEI_DWRR_CTRL_QUANTUM1	CSR_GENMASK(15, 8)
#define FBNIC_TCE_TXB_NTWRK_DWRR_CTRL	0x0400a		/* 0x10028 */
#define FBNIC_TCE_TXB_NTWRK_DWRR_CTRL_QUANTUM0	CSR_GENMASK(7, 0)
#define FBNIC_TCE_TXB_NTWRK_DWRR_CTRL_QUANTUM1	CSR_GENMASK(15, 8)
#define FBNIC_TCE_TXB_NTWRK_DWRR_CTRL_QUANTUM2	CSR_GENMASK(23, 16)

#define FBNIC_TCE_TXB_CLDR_CFG		0x0400b		/* 0x1002c */
#define FBNIC_TCE_TXB_CLDR_CFG_NUM_SLOT		CSR_GENMASK(5, 0)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG(n)	(0x0400c + (n))	/* 0x10030 + 4*n */
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_CNT		16
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_0_0	CSR_GENMASK(1, 0)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_0_1	CSR_GENMASK(3, 2)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_0_2	CSR_GENMASK(5, 4)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_0_3	CSR_GENMASK(7, 6)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_1_0	CSR_GENMASK(9, 8)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_1_1	CSR_GENMASK(11, 10)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_1_2	CSR_GENMASK(13, 12)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_1_3	CSR_GENMASK(15, 14)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_2_0	CSR_GENMASK(17, 16)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_2_1	CSR_GENMASK(19, 18)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_2_2	CSR_GENMASK(21, 20)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_2_3	CSR_GENMASK(23, 22)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_3_0	CSR_GENMASK(25, 24)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_3_1	CSR_GENMASK(27, 26)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_3_2	CSR_GENMASK(29, 28)
#define FBNIC_TCE_TXB_CLDR_SLOT_CFG_DEST_ID_3_3	CSR_GENMASK(31, 30)

#define FBNIC_TCE_BMC_MAX_PKTSZ		0x0403a		/* 0x100e8 */
#define FBNIC_TCE_BMC_MAX_PKTSZ_TX		CSR_GENMASK(13, 0)
#define FBNIC_TCE_BMC_MAX_PKTSZ_RX		CSR_GENMASK(27, 14)
#define FBNIC_TCE_MC_MAX_PKTSZ		0x0403b		/* 0x100ec */
#define FBNIC_TCE_MC_MAX_PKTSZ_TMI		CSR_GENMASK(13, 0)

#define FBNIC_TCE_SOP_PROT_CTRL		0x0403c		/* 0x100f0 */
#define FBNIC_TCE_SOP_PROT_CTRL_TBI		CSR_GENMASK(7, 0)
#define FBNIC_TCE_SOP_PROT_CTRL_TTI_FRM		CSR_GENMASK(14, 8)
#define FBNIC_TCE_SOP_PROT_CTRL_TTI_CM		CSR_GENMASK(18, 15)

#define FBNIC_TCE_DROP_CTRL		0x0403d		/* 0x100f4 */
#define FBNIC_TCE_DROP_CTRL_TTI_CM_DROP_EN	CSR_BIT(0)
#define FBNIC_TCE_DROP_CTRL_TTI_FRM_DROP_EN	CSR_BIT(1)
#define FBNIC_TCE_DROP_CTRL_TTI_TBI_DROP_EN	CSR_BIT(2)

#define FBNIC_TCE_TTI_CM_DROP_PKTS	0x0403e		/* 0x100f8 */
#define FBNIC_TCE_TTI_CM_DROP_BYTE_L	0x0403f		/* 0x100fc */
#define FBNIC_TCE_TTI_CM_DROP_BYTE_H	0x04040		/* 0x10100 */
#define FBNIC_TCE_TTI_FRAME_DROP_PKTS	0x04041		/* 0x10104 */
#define FBNIC_TCE_TTI_FRAME_DROP_BYTE_L	0x04042		/* 0x10108 */
#define FBNIC_TCE_TTI_FRAME_DROP_BYTE_H	0x04043		/* 0x1010c */
#define FBNIC_TCE_TBI_DROP_PKTS		0x04044		/* 0x10110 */
#define FBNIC_TCE_TBI_DROP_BYTE_L	0x04045		/* 0x10114 */

#define FBNIC_TCE_TCAM_IDX2DEST_MAP	0x0404A		/* 0x10128 */
#define FBNIC_TCE_TCAM_IDX2DEST_MAP_DEST_ID_0	CSR_GENMASK(3, 0)
enum {
	FBNIC_TCE_TCAM_DEST_MAC		= 1,
	FBNIC_TCE_TCAM_DEST_BMC		= 2,
	FBNIC_TCE_TCAM_DEST_FW		= 4,
};

#define FBNIC_TCE_TXB_TX_BMC_Q_CTRL	0x0404B		/* 0x1012c */
#define FBNIC_TCE_TXB_BMC_DWRR_CTRL	0x0404C		/* 0x10130 */
#define FBNIC_TCE_TXB_BMC_DWRR_CTRL_QUANTUM0	CSR_GENMASK(7, 0)
#define FBNIC_TCE_TXB_BMC_DWRR_CTRL_QUANTUM1	CSR_GENMASK(15, 8)
#define FBNIC_TCE_TXB_TEI_DWRR_CTRL_EXT	0x0404D		/* 0x10134 */
#define FBNIC_TCE_TXB_NTWRK_DWRR_CTRL_EXT \
					0x0404E		/* 0x10138 */
#define FBNIC_TCE_TXB_BMC_DWRR_CTRL_EXT	0x0404F		/* 0x1013c */
#define FBNIC_CSR_END_TCE		0x04050	/* CSR section delimiter */

/* TCE RAM registers */
#define FBNIC_CSR_START_TCE_RAM		0x04200	/* CSR section delimiter */
#define FBNIC_TCE_RAM_TCAM(m, n) \
	(0x04200 + 0x8 * (n) + (m))		/* 0x10800 + 32*n + 4*m */
#define FBNIC_TCE_RAM_TCAM_MASK			CSR_GENMASK(15, 0)
#define FBNIC_TCE_RAM_TCAM_VALUE		CSR_GENMASK(31, 16)
#define FBNIC_TCE_RAM_TCAM3(n)		(0x04218 + (n))	/* 0x010860 + 4*n */
#define FBNIC_TCE_RAM_TCAM3_DEST_MASK		CSR_GENMASK(5, 3)
#define FBNIC_TCE_RAM_TCAM3_MCQ_MASK		CSR_BIT(7)
#define FBNIC_TCE_RAM_TCAM3_VALIDATE		CSR_BIT(31)
#define FBNIC_CSR_END_TCE_RAM		0x0421F	/* CSR section delimiter */

/* TMI registers */
#define FBNIC_CSR_START_TMI		0x04400	/* CSR section delimiter */
#define FBNIC_TMI_SOP_PROT_CTRL		0x04400		/* 0x11000 */
#define FBNIC_TMI_DROP_CTRL		0x04401		/* 0x11004 */
#define FBNIC_TMI_DROP_CTRL_EN			CSR_BIT(0)
#define FBNIC_TMI_DROP_PKTS		0x04402		/* 0x11008 */
#define FBNIC_TMI_DROP_BYTE_L		0x04403		/* 0x1100c */
#define FBNIC_TMI_ILLEGAL_PTP_REQS	0x04409		/* 0x11024 */
#define FBNIC_TMI_GOOD_PTP_TS		0x0440a		/* 0x11028 */
#define FBNIC_TMI_BAD_PTP_TS		0x0440b		/* 0x1102c */
#define FBNIC_TMI_STAT_TX_PACKET_1519_2047_BYTES_L \
					0x04433		/* 0x110cc */
#define FBNIC_TMI_STAT_TX_PACKET_1519_2047_BYTES_H \
					0x04434		/* 0x110d0 */
#define FBNIC_TMI_STAT_TX_PACKET_2048_4095_BYTES_L \
					0x04435		/* 0x110d4 */
#define FBNIC_TMI_STAT_TX_PACKET_2048_4095_BYTES_H \
					0x04436		/* 0x110d8 */
#define FBNIC_TMI_STAT_TX_PACKET_4096_8191_BYTES_L \
					0x04437		/* 0x110dc */
#define FBNIC_TMI_STAT_TX_PACKET_4096_8191_BYTES_H \
					0x04438		/* 0x110e0 */
#define FBNIC_TMI_STAT_TX_PACKET_8192_9216_BYTES_L \
					0x04439		/* 0x110e4 */
#define FBNIC_TMI_STAT_TX_PACKET_8192_9216_BYTES_H \
					0x0443a		/* 0x110e8 */
#define FBNIC_TMI_STAT_TX_PACKET_9217_MAX_BYTES_L \
					0x0443b		/* 0x110ec */
#define FBNIC_TMI_STAT_TX_PACKET_9217_MAX_BYTES_H \
					0x0443c		/* 0x110f0 */
#define FBNIC_CSR_END_TMI		0x0443f	/* CSR section delimiter */

/* Precision Time Protocol Registers */
#define FBNIC_CSR_START_PTP		0x04800 /* CSR section delimiter */
#define FBNIC_PTP_REG_BASE		0x04800		/* 0x12000 */

#define FBNIC_PTP_CTRL			0x04800		/* 0x12000 */
#define FBNIC_PTP_CTRL_EN			CSR_BIT(0)
#define FBNIC_PTP_CTRL_MONO_EN			CSR_BIT(4)
#define FBNIC_PTP_CTRL_TQS_OUT_EN		CSR_BIT(8)
#define FBNIC_PTP_CTRL_MAC_OUT_IVAL		CSR_GENMASK(16, 12)
#define FBNIC_PTP_CTRL_TICK_IVAL		CSR_GENMASK(23, 20)

#define FBNIC_PTP_ADJUST		0x04801		/* 0x12004 */
#define FBNIC_PTP_ADJUST_INIT			CSR_BIT(0)
#define FBNIC_PTP_ADJUST_SUB_NUDGE		CSR_BIT(8)
#define FBNIC_PTP_ADJUST_ADD_NUDGE		CSR_BIT(16)
#define FBNIC_PTP_ADJUST_ADDEND_SET		CSR_BIT(24)

#define FBNIC_PTP_INIT_HI		0x04802		/* 0x12008 */
#define FBNIC_PTP_INIT_LO		0x04803		/* 0x1200c */

#define FBNIC_PTP_NUDGE_NS		0x04804		/* 0x12010 */
#define FBNIC_PTP_NUDGE_SUBNS		0x04805		/* 0x12014 */

#define FBNIC_PTP_ADD_VAL_NS		0x04806		/* 0x12018 */
#define FBNIC_PTP_ADD_VAL_NS_MASK		CSR_GENMASK(15, 0)
#define FBNIC_PTP_ADD_VAL_SUBNS		0x04807		/* 0x1201c */

#define FBNIC_PTP_CTR_VAL_HI		0x04808		/* 0x12020 */
#define FBNIC_PTP_CTR_VAL_LO		0x04809		/* 0x12024 */

#define FBNIC_PTP_MONO_PTP_CTR_HI	0x0480a		/* 0x12028 */
#define FBNIC_PTP_MONO_PTP_CTR_LO	0x0480b		/* 0x1202c */

#define FBNIC_PTP_CDC_FIFO_STATUS	0x0480c		/* 0x12030 */
#define FBNIC_PTP_SPARE			0x0480d		/* 0x12034 */
#define FBNIC_CSR_END_PTP		0x0480d /* CSR section delimiter */

/* Rx Buffer Registers */
#define FBNIC_CSR_START_RXB		0x08000	/* CSR section delimiter */
enum {
	FBNIC_RXB_FIFO_MC		= 0,
	/* Unused */
	/* Unused */
	FBNIC_RXB_FIFO_NET_TO_BMC	= 3,
	FBNIC_RXB_FIFO_HOST		= 4,
	/* Unused */
	FBNIC_RXB_FIFO_BMC_TO_HOST	= 6,
	/* Unused */
	FBNIC_RXB_FIFO_INDICES		= 8
};

enum {
	FBNIC_RXB_INTF_NET = 0,
	FBNIC_RXB_INTF_RBT = 1,
	/* Unused */
	/* Unused */
	FBNIC_RXB_INTF_INDICES	= 4
};

#define FBNIC_RXB_CT_SIZE(n)		(0x08000 + (n))	/* 0x20000 + 4*n */
#define FBNIC_RXB_CT_SIZE_CNT			8
#define FBNIC_RXB_CT_SIZE_HEADER		CSR_GENMASK(5, 0)
#define FBNIC_RXB_CT_SIZE_PAYLOAD		CSR_GENMASK(11, 6)
#define FBNIC_RXB_CT_SIZE_ENABLE		CSR_BIT(12)
#define FBNIC_RXB_PAUSE_DROP_CTRL	0x08008		/* 0x20020 */
#define FBNIC_RXB_PAUSE_DROP_CTRL_DROP_ENABLE	CSR_GENMASK(7, 0)
#define FBNIC_RXB_PAUSE_DROP_CTRL_PAUSE_ENABLE	CSR_GENMASK(15, 8)
#define FBNIC_RXB_PAUSE_DROP_CTRL_ECN_ENABLE	CSR_GENMASK(23, 16)
#define FBNIC_RXB_PAUSE_DROP_CTRL_PS_ENABLE	CSR_GENMASK(27, 24)
#define FBNIC_RXB_PAUSE_THLD(n)		(0x08009 + (n)) /* 0x20024 + 4*n */
#define FBNIC_RXB_PAUSE_THLD_CNT		8
#define FBNIC_RXB_PAUSE_THLD_ON			CSR_GENMASK(12, 0)
#define FBNIC_RXB_PAUSE_THLD_OFF		CSR_GENMASK(25, 13)
#define FBNIC_RXB_DROP_THLD(n)		(0x08011 + (n)) /* 0x20044 + 4*n */
#define FBNIC_RXB_DROP_THLD_CNT			8
#define FBNIC_RXB_DROP_THLD_ON			CSR_GENMASK(12, 0)
#define FBNIC_RXB_DROP_THLD_OFF			CSR_GENMASK(25, 13)
#define FBNIC_RXB_ECN_THLD(n)		(0x0801e + (n)) /* 0x20078 + 4*n */
#define FBNIC_RXB_ECN_THLD_CNT			8
#define FBNIC_RXB_ECN_THLD_ON			CSR_GENMASK(12, 0)
#define FBNIC_RXB_ECN_THLD_OFF			CSR_GENMASK(25, 13)
#define FBNIC_RXB_PBUF_CFG(n)		(0x08027 + (n))	/* 0x2009c + 4*n */
#define FBNIC_RXB_PBUF_CFG_CNT			8
#define FBNIC_RXB_PBUF_BASE_ADDR		CSR_GENMASK(12, 0)
#define FBNIC_RXB_PBUF_SIZE			CSR_GENMASK(21, 13)
#define FBNIC_RXB_DWRR_RDE_WEIGHT0	0x0802f		/* 0x200bc */
#define FBNIC_RXB_DWRR_RDE_WEIGHT0_QUANTUM0	CSR_GENMASK(7, 0)
#define FBNIC_RXB_DWRR_RDE_WEIGHT0_QUANTUM1	CSR_GENMASK(15, 8)
#define FBNIC_RXB_DWRR_RDE_WEIGHT0_QUANTUM2	CSR_GENMASK(23, 16)
#define FBNIC_RXB_DWRR_RDE_WEIGHT0_QUANTUM3	CSR_GENMASK(31, 24)
#define FBNIC_RXB_DWRR_RDE_WEIGHT1	0x08030		/* 0x200c0 */
#define FBNIC_RXB_DWRR_RDE_WEIGHT1_QUANTUM4	CSR_GENMASK(7, 0)
#define FBNIC_RXB_DWRR_BMC_WEIGHT	0x08031		/* 0x200c4 */
#define FBNIC_RXB_CLDR_PRIO_CFG(n)	(0x8034 + (n))	/* 0x200d0 + 4*n */
#define FBNIC_RXB_CLDR_PRIO_CFG_CNT		16
#define FBNIC_RXB_ENDIAN_FCS		0x08044		/* 0x20110 */
enum {
	/* Unused */
	/* Unused */
	FBNIC_RXB_DEQUEUE_BMC		= 2,
	FBNIC_RXB_DEQUEUE_HOST		= 3,
	FBNIC_RXB_DEQUEUE_INDICES	= 4
};

#define FBNIC_RXB_PBUF_CREDIT(n)	(0x08047 + (n))	/* 0x2011C + 4*n */
#define FBNIC_RXB_PBUF_CREDIT_CNT		8
#define FBNIC_RXB_PBUF_CREDIT_MASK		CSR_GENMASK(13, 0)
#define FBNIC_RXB_INTF_CREDIT		0x0804f		/* 0x2013C */
#define FBNIC_RXB_INTF_CREDIT_MASK0		CSR_GENMASK(3, 0)
#define FBNIC_RXB_INTF_CREDIT_MASK1		CSR_GENMASK(7, 4)
#define FBNIC_RXB_INTF_CREDIT_MASK2		CSR_GENMASK(11, 8)
#define FBNIC_RXB_INTF_CREDIT_MASK3		CSR_GENMASK(15, 12)

#define FBNIC_RXB_PAUSE_EVENT_CNT(n)	(0x08053 + (n))	/* 0x2014c + 4*n */
#define FBNIC_RXB_DROP_FRMS_STS(n)	(0x08057 + (n))	/* 0x2015c + 4*n */
#define FBNIC_RXB_DROP_BYTES_STS_L(n) \
				(0x08080 + 2 * (n))	/* 0x20200 + 8*n */
#define FBNIC_RXB_DROP_BYTES_STS_H(n) \
				(0x08081 + 2 * (n))	/* 0x20204 + 8*n */
#define FBNIC_RXB_TRUN_FRMS_STS(n)	(0x08091 + (n))	/* 0x20244 + 4*n */
#define FBNIC_RXB_TRUN_BYTES_STS_L(n) \
				(0x080c0 + 2 * (n))	/* 0x20300 + 8*n */
#define FBNIC_RXB_TRUN_BYTES_STS_H(n) \
				(0x080c1 + 2 * (n))	/* 0x20304 + 8*n */
#define FBNIC_RXB_TRANS_PAUSE_STS(n)	(0x080d1 + (n))	/* 0x20344 + 4*n */
#define FBNIC_RXB_TRANS_DROP_STS(n)	(0x080d9 + (n))	/* 0x20364 + 4*n */
#define FBNIC_RXB_TRANS_ECN_STS(n)	(0x080e1 + (n))	/* 0x20384 + 4*n */
enum {
	FBNIC_RXB_ENQUEUE_NET		= 0,
	FBNIC_RXB_ENQUEUE_BMC		= 1,
	/* Unused */
	/* Unused */
	FBNIC_RXB_ENQUEUE_INDICES	= 4
};

#define FBNIC_RXB_DRBO_FRM_CNT_SRC(n)	(0x080f9 + (n))	/* 0x203e4 + 4*n */
#define FBNIC_RXB_DRBO_BYTE_CNT_SRC_L(n) \
					(0x080fd + (n))	/* 0x203f4 + 4*n */
#define FBNIC_RXB_DRBO_BYTE_CNT_SRC_H(n) \
					(0x08101 + (n))	/* 0x20404 + 4*n */
#define FBNIC_RXB_INTF_FRM_CNT_DST(n)	(0x08105 + (n))	/* 0x20414 + 4*n */
#define FBNIC_RXB_INTF_BYTE_CNT_DST_L(n) \
					(0x08109 + (n))	/* 0x20424 + 4*n */
#define FBNIC_RXB_INTF_BYTE_CNT_DST_H(n) \
					(0x0810d + (n))	/* 0x20434 + 4*n */
#define FBNIC_RXB_PBUF_FRM_CNT_DST(n)	(0x08111 + (n))	/* 0x20444 + 4*n */
#define FBNIC_RXB_PBUF_BYTE_CNT_DST_L(n) \
					(0x08115 + (n))	/* 0x20454 + 4*n */
#define FBNIC_RXB_PBUF_BYTE_CNT_DST_H(n) \
					(0x08119 + (n))	/* 0x20464 + 4*n */

#define FBNIC_RXB_PBUF_FIFO_LEVEL(n)	(0x0811d + (n)) /* 0x20474 + 4*n */

#define FBNIC_RXB_INTEGRITY_ERR(n)	(0x0812f + (n))	/* 0x204bc + 4*n */
#define FBNIC_RXB_MAC_ERR(n)		(0x08133 + (n))	/* 0x204cc + 4*n */
#define FBNIC_RXB_PARSER_ERR(n)		(0x08137 + (n))	/* 0x204dc + 4*n */
#define FBNIC_RXB_FRM_ERR(n)		(0x0813b + (n))	/* 0x204ec + 4*n */

#define FBNIC_RXB_DWRR_RDE_WEIGHT0_EXT	0x08143		/* 0x2050c */
#define FBNIC_RXB_DWRR_RDE_WEIGHT1_EXT	0x08144		/* 0x20510 */
#define FBNIC_CSR_END_RXB		0x081b1	/* CSR section delimiter */

/* Rx Parser and Classifier Registers */
#define FBNIC_CSR_START_RPC		0x08400	/* CSR section delimiter */
#define FBNIC_RPC_RMI_CONFIG		0x08400		/* 0x21000 */
#define FBNIC_RPC_RMI_CONFIG_OH_BYTES		CSR_GENMASK(4, 0)
#define FBNIC_RPC_RMI_CONFIG_FCS_PRESENT	CSR_BIT(8)
#define FBNIC_RPC_RMI_CONFIG_ENABLE		CSR_BIT(12)
#define FBNIC_RPC_RMI_CONFIG_MTU		CSR_GENMASK(31, 16)

#define FBNIC_RPC_ACT_TBL0_DEFAULT	0x0840a		/* 0x21028 */
#define FBNIC_RPC_ACT_TBL0_DROP			CSR_BIT(0)
#define FBNIC_RPC_ACT_TBL0_DEST_MASK		CSR_GENMASK(3, 1)
enum {
	FBNIC_RPC_ACT_TBL0_DEST_HOST	= 1,
	FBNIC_RPC_ACT_TBL0_DEST_BMC	= 2,
	FBNIC_RPC_ACT_TBL0_DEST_EI	= 4,
};

#define FBNIC_RPC_ACT_TBL0_Q_SEL		CSR_BIT(4)
#define FBNIC_RPC_ACT_TBL0_Q_ID			CSR_GENMASK(15, 8)
#define FBNIC_RPC_ACT_TBL0_DMA_HINT		CSR_GENMASK(24, 16)
#define FBNIC_RPC_ACT_TBL0_TS_ENA		CSR_BIT(28)
#define FBNIC_RPC_ACT_TBL0_ACT_TBL_IDX		CSR_BIT(29)
#define FBNIC_RPC_ACT_TBL0_RSS_CTXT_ID		CSR_BIT(30)

#define FBNIC_RPC_ACT_TBL1_DEFAULT	0x0840b		/* 0x2102c */
#define FBNIC_RPC_ACT_TBL1_RSS_ENA_MASK		CSR_GENMASK(15, 0)
enum {
	FBNIC_RPC_ACT_TBL1_RSS_ENA_IP_SRC	= 1,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_IP_DST	= 2,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L4_SRC	= 4,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L4_DST	= 8,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L2_DA	= 16,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L4_RSS_BYTE	= 32,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_IV6_FL_LBL	= 64,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_OV6_FL_LBL	= 128,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_DSCP		= 256,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L3_PROT	= 512,
	FBNIC_RPC_ACT_TBL1_RSS_ENA_L4_PROT	= 1024,
};

#define FBNIC_RPC_RSS_KEY(n)		(0x0840c + (n))	/* 0x21030 + 4*n */
#define FBNIC_RPC_RSS_KEY_BIT_LEN		425
#define FBNIC_RPC_RSS_KEY_BYTE_LEN \
	DIV_ROUND_UP(FBNIC_RPC_RSS_KEY_BIT_LEN, 8)
#define FBNIC_RPC_RSS_KEY_DWORD_LEN \
	DIV_ROUND_UP(FBNIC_RPC_RSS_KEY_BIT_LEN, 32)
#define FBNIC_RPC_RSS_KEY_LAST_IDX \
	(FBNIC_RPC_RSS_KEY_DWORD_LEN - 1)
#define FBNIC_RPC_RSS_KEY_LAST_MASK \
	CSR_GENMASK(31, \
		    FBNIC_RPC_RSS_KEY_DWORD_LEN * 32 - \
		    FBNIC_RPC_RSS_KEY_BIT_LEN)

#define FBNIC_RPC_CNTR_TCP_OPT_ERR	0x0849e		/* 0x21278 */
#define FBNIC_RPC_CNTR_UNKN_ETYPE	0x0849f		/* 0x2127c */
#define FBNIC_RPC_CNTR_IPV4_FRAG	0x084a0		/* 0x21280 */
#define FBNIC_RPC_CNTR_IPV6_FRAG	0x084a1		/* 0x21284 */
#define FBNIC_RPC_CNTR_IPV4_ESP		0x084a2		/* 0x21288 */
#define FBNIC_RPC_CNTR_IPV6_ESP		0x084a3		/* 0x2128c */
#define FBNIC_RPC_CNTR_UNKN_EXT_HDR	0x084a4		/* 0x21290 */
#define FBNIC_RPC_CNTR_OUT_OF_HDR_ERR	0x084a5		/* 0x21294 */
#define FBNIC_RPC_CNTR_OVR_SIZE_ERR	0x084a6		/* 0x21298 */

#define FBNIC_RPC_TCAM_MACDA_VALIDATE	0x0852d		/* 0x214b4 */
#define FBNIC_RPC_STAT_RX_PACKET_1519_2047_BYTES_L \
					0x0855f		/* 0x2157c */
#define FBNIC_RPC_STAT_RX_PACKET_1519_2047_BYTES_H \
					0x08560		/* 0x21580 */
#define FBNIC_RPC_STAT_RX_PACKET_2048_4095_BYTES_L \
					0x08561		/* 0x21584 */
#define FBNIC_RPC_STAT_RX_PACKET_2048_4095_BYTES_H \
					0x08562		/* 0x21588 */
#define FBNIC_RPC_STAT_RX_PACKET_4096_8191_BYTES_L \
					0x08563		/* 0x2158c */
#define FBNIC_RPC_STAT_RX_PACKET_4096_8191_BYTES_H \
					0x08564		/* 0x21590 */
#define FBNIC_RPC_STAT_RX_PACKET_8192_9216_BYTES_L \
					0x08565		/* 0x21594 */
#define FBNIC_RPC_STAT_RX_PACKET_8192_9216_BYTES_H \
					0x08566		/* 0x21598 */
#define FBNIC_RPC_STAT_RX_PACKET_9217_MAX_BYTES_L \
					0x08567		/* 0x2159c */
#define FBNIC_RPC_STAT_RX_PACKET_9217_MAX_BYTES_H \
					0x08568		/* 0x215a0 */
#define FBNIC_CSR_END_RPC		0x0856b	/* CSR section delimiter */

/* RPC RAM Registers */

#define FBNIC_CSR_START_RPC_RAM		0x08800	/* CSR section delimiter */
#define FBNIC_RPC_ACT_TBL0(n)		(0x08800 + (n))	/* 0x22000 + 4*n */
#define FBNIC_RPC_ACT_TBL1(n)		(0x08840 + (n))	/* 0x22100 + 4*n */
#define FBNIC_RPC_ACT_TBL_NUM_ENTRIES		64

/* TCAM Tables */
#define FBNIC_RPC_TCAM_VALIDATE			CSR_BIT(31)

/* 64 Action TCAM Entries, 12 registers
 * 3 mixed, src port, dst port, 6 L4 words, and Validate
 */
#define FBNIC_RPC_TCAM_ACT(m, n) \
	(0x08880 + 0x40 * (n) + (m))		/* 0x22200 + 256*n + 4*m */

#define FBNIC_RPC_TCAM_ACT_VALUE		CSR_GENMASK(15, 0)
#define FBNIC_RPC_TCAM_ACT_MASK			CSR_GENMASK(31, 16)

#define FBNIC_RPC_TCAM_MACDA(m, n) \
	(0x08b80 + 0x20 * (n) + (m))		/* 0x022e00 + 128*n + 4*m */
#define FBNIC_RPC_TCAM_MACDA_VALUE		CSR_GENMASK(15, 0)
#define FBNIC_RPC_TCAM_MACDA_MASK		CSR_GENMASK(31, 16)

#define FBNIC_RPC_TCAM_OUTER_IPSRC(m, n)\
	(0x08c00 + 0x08 * (n) + (m))		/* 0x023000 + 32*n + 4*m */
#define FBNIC_RPC_TCAM_IP_ADDR_VALUE		CSR_GENMASK(15, 0)
#define FBNIC_RPC_TCAM_IP_ADDR_MASK		CSR_GENMASK(31, 16)

#define FBNIC_RPC_TCAM_OUTER_IPDST(m, n)\
	(0x08c48 + 0x08 * (n) + (m))		/* 0x023120 + 32*n + 4*m */
#define FBNIC_RPC_TCAM_IPSRC(m, n)\
	(0x08c90 + 0x08 * (n) + (m))		/* 0x023240 + 32*n + 4*m */
#define FBNIC_RPC_TCAM_IPDST(m, n)\
	(0x08cd8 + 0x08 * (n) + (m))		/* 0x023360 + 32*n + 4*m */

#define FBNIC_RPC_RSS_TBL(n, m) \
	(0x08d20 + 0x100 * (n) + (m))		/* 0x023480 + 1024*n + 4*m */
#define FBNIC_RPC_RSS_TBL_COUNT			2
#define FBNIC_RPC_RSS_TBL_SIZE			256
#define FBNIC_CSR_END_RPC_RAM		0x08f1f	/* CSR section delimiter */

/* Fab Registers */
#define FBNIC_CSR_START_FAB		0x0C000 /* CSR section delimiter */
#define FBNIC_FAB_AXI4_AR_SPACER_2_CFG		0x0C005		/* 0x30014 */
#define FBNIC_FAB_AXI4_AR_SPACER_MASK		CSR_BIT(16)
#define FBNIC_FAB_AXI4_AR_SPACER_THREADSHOLD	CSR_GENMASK(15, 0)
#define FBNIC_CSR_END_FAB		0x0C020	    /* CSR section delimiter */

/* Master Registers */
#define FBNIC_CSR_START_MASTER		0x0C400	/* CSR section delimiter */
#define FBNIC_MASTER_SPARE_0		0x0C41B		/* 0x3106c */
#define FBNIC_CSR_END_MASTER		0x0C452	/* CSR section delimiter */

/* MAC PCS registers */
#define FBNIC_CSR_START_PCS		0x10000 /* CSR section delimiter */
#define FBNIC_PCS_PAGE(n)	(0x10000 + 0x400 * (n))	/* 0x40000 + 1024*n */
#define FBNIC_PCS(reg, n)	((reg) + FBNIC_PCS_PAGE(n))
#define FBNIC_CSR_END_PCS		0x10668 /* CSR section delimiter */

#define FBNIC_CSR_START_RSFEC		0x10800 /* CSR section delimiter */

/* We have 4 RSFEC engines present in our part, however we are only using 1.
 * As such only CCW(0) and NCCW(0) will never be non-zero and the other
 * registers can be ignored.
 */
#define FBNIC_RSFEC_CCW_LO(n)	(0x10802 + 8 * (n))	/* 0x42008 + 32*n */
#define FBNIC_RSFEC_CCW_HI(n)	(0x10803 + 8 * (n))	/* 0x4200c + 32*n */
#define FBNIC_RSFEC_NCCW_LO(n)	(0x10804 + 8 * (n))	/* 0x42010 + 32*n */
#define FBNIC_RSFEC_NCCW_HI(n)	(0x10805 + 8 * (n))	/* 0x42014 + 32*n */

#define FBNIC_PCS_MAX_LANES			4
#define FBNIC_PCS_SYMBLERR_LO(n) \
				(0x10880 + 2 * (n))	/* 0x42200 + 8*n */
#define FBNIC_PCS_SYMBLERR_HI(n) \
				(0x10881 + 2 * (n))	/* 0x42204 + 8*n */
#define FBNIC_CSR_END_RSFEC		0x108c8 /* CSR section delimiter */

/* MAC MAC registers (ASIC only) */
#define FBNIC_CSR_START_MAC_MAC		0x11000 /* CSR section delimiter */
#define FBNIC_MAC_COMMAND_CONFIG	0x11002		/* 0x44008 */
#define FBNIC_MAC_COMMAND_CONFIG_RX_PAUSE_DIS	CSR_BIT(29)
#define FBNIC_MAC_COMMAND_CONFIG_TX_PAUSE_DIS	CSR_BIT(28)
#define FBNIC_MAC_COMMAND_CONFIG_FLT_HDL_DIS	CSR_BIT(27)
#define FBNIC_MAC_COMMAND_CONFIG_TX_PAD_EN	CSR_BIT(11)
#define FBNIC_MAC_COMMAND_CONFIG_LOOPBACK_EN	CSR_BIT(10)
#define FBNIC_MAC_COMMAND_CONFIG_PROMISC_EN	CSR_BIT(4)
#define FBNIC_MAC_COMMAND_CONFIG_RX_ENA		CSR_BIT(1)
#define FBNIC_MAC_COMMAND_CONFIG_TX_ENA		CSR_BIT(0)
#define FBNIC_MAC_CL01_PAUSE_QUANTA	0x11015		/* 0x44054 */
#define FBNIC_MAC_CL01_QUANTA_THRESH	0x11019		/* 0x44064 */
#define FBNIC_CSR_END_MAC_MAC		0x11028 /* CSR section delimiter */

/* Signals from MAC, AN, PCS, and LED CSR registers (ASIC only) */
#define FBNIC_CSR_START_SIG		0x11800 /* CSR section delimiter */
#define FBNIC_SIG_MAC_IN0		0x11800		/* 0x46000 */
#define FBNIC_SIG_MAC_IN0_RESET_FF_TX_CLK	CSR_BIT(14)
#define FBNIC_SIG_MAC_IN0_RESET_FF_RX_CLK	CSR_BIT(13)
#define FBNIC_SIG_MAC_IN0_RESET_TX_CLK		CSR_BIT(12)
#define FBNIC_SIG_MAC_IN0_RESET_RX_CLK		CSR_BIT(11)
#define FBNIC_SIG_MAC_IN0_TX_CRC		CSR_BIT(8)
#define FBNIC_SIG_MAC_IN0_CFG_MODE128		CSR_BIT(10)
#define FBNIC_SIG_PCS_OUT0		0x11808		/* 0x46020 */
#define FBNIC_SIG_PCS_OUT0_LINK			CSR_BIT(27)
#define FBNIC_SIG_PCS_OUT0_BLOCK_LOCK		CSR_GENMASK(24, 5)
#define FBNIC_SIG_PCS_OUT0_AMPS_LOCK		CSR_GENMASK(4, 1)
#define FBNIC_SIG_PCS_OUT1		0x11809		/* 0x46024 */
#define FBNIC_SIG_PCS_OUT1_FCFEC_LOCK		CSR_GENMASK(11, 8)
#define FBNIC_SIG_PCS_INTR_STS		0x11814		/* 0x46050 */
#define FBNIC_SIG_PCS_INTR_LINK_DOWN		CSR_BIT(1)
#define FBNIC_SIG_PCS_INTR_LINK_UP		CSR_BIT(0)
#define FBNIC_SIG_PCS_INTR_MASK		0x11816		/* 0x46058 */
#define FBNIC_CSR_END_SIG		0x1184e /* CSR section delimiter */

#define FBNIC_CSR_START_MAC_STAT	0x11a00
#define FBNIC_MAC_STAT_RX_XOFF_STB_L	0x11a00		/* 0x46800 */
#define FBNIC_MAC_STAT_RX_XOFF_STB_H	0x11a01		/* 0x46804 */
#define FBNIC_MAC_STAT_TX_XOFF_STB_L	0x11a04		/* 0x46810 */
#define FBNIC_MAC_STAT_TX_XOFF_STB_H	0x11a05		/* 0x46814 */
#define FBNIC_MAC_STAT_RX_BYTE_COUNT_L	0x11a08		/* 0x46820 */
#define FBNIC_MAC_STAT_RX_BYTE_COUNT_H	0x11a09		/* 0x46824 */
#define FBNIC_MAC_STAT_RX_ALIGN_ERROR_L	0x11a0a		/* 0x46828 */
#define FBNIC_MAC_STAT_RX_ALIGN_ERROR_H	0x11a0b		/* 0x4682c */
#define FBNIC_MAC_STAT_RX_TOOLONG_L	0x11a0e		/* 0x46838 */
#define FBNIC_MAC_STAT_RX_TOOLONG_H	0x11a0f		/* 0x4683c */
#define FBNIC_MAC_STAT_RX_RECEIVED_OK_L	0x11a12		/* 0x46848 */
#define FBNIC_MAC_STAT_RX_RECEIVED_OK_H	0x11a13		/* 0x4684c */
#define FBNIC_MAC_STAT_RX_PACKET_BAD_FCS_L \
					0x11a14		/* 0x46850 */
#define FBNIC_MAC_STAT_RX_PACKET_BAD_FCS_H \
					0x11a15		/* 0x46854 */
#define FBNIC_MAC_STAT_RX_IFINERRORS_L	0x11a18		/* 0x46860 */
#define FBNIC_MAC_STAT_RX_IFINERRORS_H	0x11a19		/* 0x46864 */
#define FBNIC_MAC_STAT_RX_MULTICAST_L	0x11a1c		/* 0x46870 */
#define FBNIC_MAC_STAT_RX_MULTICAST_H	0x11a1d		/* 0x46874 */
#define FBNIC_MAC_STAT_RX_BROADCAST_L	0x11a1e		/* 0x46878 */
#define FBNIC_MAC_STAT_RX_BROADCAST_H	0x11a1f		/* 0x4687c */
#define FBNIC_MAC_STAT_RX_UNDERSIZE_L	0x11a24		/* 0x46890 */
#define FBNIC_MAC_STAT_RX_UNDERSIZE_H	0x11a25		/* 0x46894 */
#define FBNIC_MAC_STAT_RX_PACKET_64_BYTES_L \
					0x11a26		/* 0x46898 */
#define FBNIC_MAC_STAT_RX_PACKET_64_BYTES_H \
					0x11a27		/* 0x4689c */
#define FBNIC_MAC_STAT_RX_PACKET_65_127_BYTES_L \
					0x11a28		/* 0x468a0 */
#define FBNIC_MAC_STAT_RX_PACKET_65_127_BYTES_H \
					0x11a29		/* 0x468a4 */
#define FBNIC_MAC_STAT_RX_PACKET_128_255_BYTES_L \
					0x11a2a		/* 0x468a8 */
#define FBNIC_MAC_STAT_RX_PACKET_128_255_BYTES_H \
					0x11a2b		/* 0x468ac */
#define FBNIC_MAC_STAT_RX_PACKET_256_511_BYTES_L \
					0x11a2c		/* 0x468b0 */
#define FBNIC_MAC_STAT_RX_PACKET_256_511_BYTES_H \
					0x11a2d		/* 0x468b4 */
#define FBNIC_MAC_STAT_RX_PACKET_512_1023_BYTES_L \
					0x11a2e		/* 0x468b8 */
#define FBNIC_MAC_STAT_RX_PACKET_512_1023_BYTES_H \
					0x11a2f		/* 0x468bc */
#define FBNIC_MAC_STAT_RX_PACKET_1024_1518_BYTES_L \
					0x11a30		/* 0x468c0 */
#define FBNIC_MAC_STAT_RX_PACKET_1024_1518_BYTES_H \
					0x11a31		/* 0x468c4 */
#define FBNIC_MAC_STAT_RX_PACKET_1519_MAX_BYTES_L \
					0x11a32		/* 0x468c8 */
#define FBNIC_MAC_STAT_RX_PACKET_1519_MAX_BYTES_H \
					0x11a33		/* 0x468cc */
#define FBNIC_MAC_STAT_RX_OVERSIZE_L	0x11a34		/* 0x468d0 */
#define FBNIC_MAC_STAT_RX_OVERSSIZE_H	0x11a35		/* 0x468d4 */
#define FBNIC_MAC_STAT_RX_JABBER_L	0x11a36		/* 0x468d8 */
#define FBNIC_MAC_STAT_RX_JABBER_H	0x11a37		/* 0x468dc */
#define FBNIC_MAC_STAT_RX_FRAGMENT_L	0x11a38		/* 0x468e0 */
#define FBNIC_MAC_STAT_RX_FRAGMENT_H	0x11a39		/* 0x468e4 */
#define FBNIC_MAC_STAT_RX_CONTROL_FRAMES_L \
					0x11a3c		/* 0x468f0 */
#define FBNIC_MAC_STAT_RX_CONTROL_FRAMES_H \
					0x11a3d		/* 0x468f4 */
#define FBNIC_MAC_STAT_TX_BYTE_COUNT_L	0x11a3e		/* 0x468f8 */
#define FBNIC_MAC_STAT_TX_BYTE_COUNT_H	0x11a3f		/* 0x468fc */
#define FBNIC_MAC_STAT_TX_TRANSMITTED_OK_L \
					0x11a42		/* 0x46908 */
#define FBNIC_MAC_STAT_TX_TRANSMITTED_OK_H \
					0x11a43		/* 0x4690c */
#define FBNIC_MAC_STAT_TX_IFOUTERRORS_L	0x11a46		/* 0x46918 */
#define FBNIC_MAC_STAT_TX_IFOUTERRORS_H	0x11a47		/* 0x4691c */
#define FBNIC_MAC_STAT_TX_MULTICAST_L	0x11a4a		/* 0x46928 */
#define FBNIC_MAC_STAT_TX_MULTICAST_H	0x11a4b		/* 0x4692c */
#define FBNIC_MAC_STAT_TX_BROADCAST_L	0x11a4c		/* 0x46930 */
#define FBNIC_MAC_STAT_TX_BROADCAST_H	0x11a4d		/* 0x46934 */
#define FBNIC_MAC_STAT_TX_PACKET_64_BYTES_L \
					0x11a4e		/* 0x46938 */
#define FBNIC_MAC_STAT_TX_PACKET_64_BYTES_H \
					0x11a4f		/* 0x4693c */
#define FBNIC_MAC_STAT_TX_PACKET_65_127_BYTES_L \
					0x11a50		/* 0x46940 */
#define FBNIC_MAC_STAT_TX_PACKET_65_127_BYTES_H \
					0x11a51		/* 0x46944 */
#define FBNIC_MAC_STAT_TX_PACKET_128_255_BYTES_L \
					0x11a52		/* 0x46948 */
#define FBNIC_MAC_STAT_TX_PACKET_128_255_BYTES_H \
					0x11a53		/* 0x4694c */
#define FBNIC_MAC_STAT_TX_PACKET_256_511_BYTES_L \
					0x11a54		/* 0x46950 */
#define FBNIC_MAC_STAT_TX_PACKET_256_511_BYTES_H \
					0x11a55		/* 0x46954 */
#define FBNIC_MAC_STAT_TX_PACKET_512_1023_BYTES_L \
					0x11a56		/* 0x46958 */
#define FBNIC_MAC_STAT_TX_PACKET_512_1023_BYTES_H \
					0x11a57		/* 0x4695c */
#define FBNIC_MAC_STAT_TX_PACKET_1024_1518_BYTES_L \
					0x11a58		/* 0x46960 */
#define FBNIC_MAC_STAT_TX_PACKET_1024_1518_BYTES_H \
					0x11a59		/* 0x46964 */
#define FBNIC_MAC_STAT_TX_PACKET_1519_MAX_BYTES_L \
					0x11a5a		/* 0x46968 */
#define FBNIC_MAC_STAT_TX_PACKET_1519_MAX_BYTES_H \
					0x11a5b		/* 0x4696c */
#define FBNIC_MAC_STAT_TX_CONTROL_FRAMES_L \
					0x11a5e		/* 0x46978 */
#define FBNIC_MAC_STAT_TX_CONTROL_FRAMES_H \
					0x11a5f		/* 0x4697c */

/* PCIE Comphy Registers */
#define FBNIC_CSR_START_PCIE_SS_COMPHY	0x2442e /* CSR section delimiter */
#define FBNIC_CSR_END_PCIE_SS_COMPHY	0x279d7	/* CSR section delimiter */

/* PUL User Registers */
#define FBNIC_CSR_START_PUL_USER	0x31000	/* CSR section delimiter */
#define FBNIC_PUL_OB_TLP_HDR_AW_CFG	0x3103d		/* 0xc40f4 */
#define FBNIC_PUL_OB_TLP_HDR_AW_CFG_FLUSH	CSR_BIT(19)
#define FBNIC_PUL_OB_TLP_HDR_AW_CFG_BME		CSR_BIT(18)
#define FBNIC_PUL_OB_TLP_HDR_AR_CFG	0x3103e		/* 0xc40f8 */
#define FBNIC_PUL_OB_TLP_HDR_AR_CFG_FLUSH	CSR_BIT(19)
#define FBNIC_PUL_OB_TLP_HDR_AR_CFG_BME		CSR_BIT(18)
#define FBNIC_PUL_USER_OB_RD_TLP_CNT_31_0 \
					0x3106e		/* 0xc41b8 */
#define FBNIC_PUL_USER_OB_RD_DWORD_CNT_31_0 \
					0x31070		/* 0xc41c0 */
#define FBNIC_PUL_USER_OB_RD_DWORD_CNT_63_32 \
					0x31071		/* 0xc41c4 */
#define FBNIC_PUL_USER_OB_WR_TLP_CNT_31_0 \
					0x31072		/* 0xc41c8 */
#define FBNIC_PUL_USER_OB_WR_TLP_CNT_63_32 \
					0x31073		/* 0xc41cc */
#define FBNIC_PUL_USER_OB_WR_DWORD_CNT_31_0 \
					0x31074		/* 0xc41d0 */
#define FBNIC_PUL_USER_OB_WR_DWORD_CNT_63_32 \
					0x31075		/* 0xc41d4 */
#define FBNIC_PUL_USER_OB_CPL_TLP_CNT_31_0 \
					0x31076		/* 0xc41d8 */
#define FBNIC_PUL_USER_OB_CPL_TLP_CNT_63_32 \
					0x31077		/* 0xc41dc */
#define FBNIC_PUL_USER_OB_CPL_DWORD_CNT_31_0 \
					0x31078		/* 0xc41e0 */
#define FBNIC_PUL_USER_OB_CPL_DWORD_CNT_63_32 \
					0x31079		/* 0xc41e4 */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_CPL_CRED_31_0 \
					0x3107a		/* 0xc41e8 */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_CPL_CRED_63_32 \
					0x3107b		/* 0xc41ec */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_TAG_31_0 \
					0x3107c		/* 0xc41f0 */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_TAG_63_32 \
					0x3107d		/* 0xc41f4 */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_NP_CRED_31_0 \
					0x3107e		/* 0xc41f8 */
#define FBNIC_PUL_USER_OB_RD_DBG_CNT_NP_CRED_63_32 \
					0x3107f		/* 0xc41fc */
#define FBNIC_CSR_END_PUL_USER		0x310ea	/* CSR section delimiter */

/* Queue Registers
 *
 * The queue register offsets are specific for a given queue grouping. So to
 * find the actual register offset it is necessary to combine FBNIC_QUEUE(n)
 * with the register to get the actual register offset like so:
 *   FBNIC_QUEUE_TWQ0_CTL(n) == FBNIC_QUEUE(n) + FBNIC_QUEUE_TWQ0_CTL
 */
#define FBNIC_CSR_START_QUEUE		0x40000	/* CSR section delimiter */
#define FBNIC_QUEUE_STRIDE		0x400		/* 0x1000 */
#define FBNIC_QUEUE(n)\
	(0x40000 + FBNIC_QUEUE_STRIDE * (n))	/* 0x100000 + 4096*n */

#define FBNIC_QUEUE_TWQ0_CTL		0x000		/* 0x000 */
#define FBNIC_QUEUE_TWQ1_CTL		0x001		/* 0x004 */
#define FBNIC_QUEUE_TWQ_CTL_RESET		CSR_BIT(0)
#define FBNIC_QUEUE_TWQ_CTL_ENABLE		CSR_BIT(1)
#define FBNIC_QUEUE_TWQ0_TAIL		0x002		/* 0x008 */
#define FBNIC_QUEUE_TWQ1_TAIL		0x003		/* 0x00c */

#define FBNIC_QUEUE_TWQ0_SIZE		0x00a		/* 0x028 */
#define FBNIC_QUEUE_TWQ1_SIZE		0x00b		/* 0x02c */
#define FBNIC_QUEUE_TWQ_SIZE_MASK		CSR_GENMASK(3, 0)

#define FBNIC_QUEUE_TWQ0_BAL		0x020		/* 0x080 */
#define FBNIC_QUEUE_BAL_MASK			CSR_GENMASK(31, 7)
#define FBNIC_QUEUE_TWQ0_BAH		0x021		/* 0x084 */
#define FBNIC_QUEUE_TWQ1_BAL		0x022		/* 0x088 */
#define FBNIC_QUEUE_TWQ1_BAH		0x023		/* 0x08c */

/* Tx Work Queue Statistics Registers */
#define FBNIC_QUEUE_TWQ0_PKT_CNT	0x062		/* 0x188 */
#define FBNIC_QUEUE_TWQ0_ERR_CNT	0x063		/* 0x18c */
#define FBNIC_QUEUE_TWQ1_PKT_CNT	0x072		/* 0x1c8 */
#define FBNIC_QUEUE_TWQ1_ERR_CNT	0x073		/* 0x1cc */

/* Tx Completion Queue Registers */
#define FBNIC_QUEUE_TCQ_CTL		0x080		/* 0x200 */
#define FBNIC_QUEUE_TCQ_CTL_RESET		CSR_BIT(0)
#define FBNIC_QUEUE_TCQ_CTL_ENABLE		CSR_BIT(1)

#define FBNIC_QUEUE_TCQ_HEAD		0x081		/* 0x204 */

#define FBNIC_QUEUE_TCQ_SIZE		0x084		/* 0x210 */
#define FBNIC_QUEUE_TCQ_SIZE_MASK		CSR_GENMASK(3, 0)

#define FBNIC_QUEUE_TCQ_BAL		0x0a0		/* 0x280 */
#define FBNIC_QUEUE_TCQ_BAH		0x0a1		/* 0x284 */

/* Tx Interrupt Manager Registers */
#define FBNIC_QUEUE_TIM_CTL		0x0c0		/* 0x300 */
#define FBNIC_QUEUE_TIM_CTL_MSIX_MASK		CSR_GENMASK(7, 0)

#define FBNIC_QUEUE_TIM_THRESHOLD	0x0c1		/* 0x304 */
#define FBNIC_QUEUE_TIM_THRESHOLD_TWD_MASK	CSR_GENMASK(14, 0)

#define FBNIC_QUEUE_TIM_CLEAR		0x0c2		/* 0x308 */
#define FBNIC_QUEUE_TIM_CLEAR_MASK		CSR_BIT(0)
#define FBNIC_QUEUE_TIM_SET		0x0c3		/* 0x30c */
#define FBNIC_QUEUE_TIM_SET_MASK		CSR_BIT(0)
#define FBNIC_QUEUE_TIM_MASK		0x0c4		/* 0x310 */
#define FBNIC_QUEUE_TIM_MASK_MASK		CSR_BIT(0)

#define FBNIC_QUEUE_TIM_TIMER		0x0c5		/* 0x314 */

#define FBNIC_QUEUE_TIM_COUNTS		0x0c6		/* 0x318 */
#define FBNIC_QUEUE_TIM_COUNTS_CNT1_MASK	CSR_GENMASK(30, 16)
#define FBNIC_QUEUE_TIM_COUNTS_CNT0_MASK	CSR_GENMASK(14, 0)

/* Rx Completion Queue Registers */
#define FBNIC_QUEUE_RCQ_CTL		0x200		/* 0x800 */
#define FBNIC_QUEUE_RCQ_CTL_RESET		CSR_BIT(0)
#define FBNIC_QUEUE_RCQ_CTL_ENABLE		CSR_BIT(1)

#define FBNIC_QUEUE_RCQ_HEAD		0x201		/* 0x804 */

#define FBNIC_QUEUE_RCQ_SIZE		0x204		/* 0x810 */
#define FBNIC_QUEUE_RCQ_SIZE_MASK		CSR_GENMASK(3, 0)

#define FBNIC_QUEUE_RCQ_BAL		0x220		/* 0x880 */
#define FBNIC_QUEUE_RCQ_BAH		0x221		/* 0x884 */

/* Rx Buffer Descriptor Queue Registers */
#define FBNIC_QUEUE_BDQ_CTL		0x240		/* 0x900 */
#define FBNIC_QUEUE_BDQ_CTL_RESET		CSR_BIT(0)
#define FBNIC_QUEUE_BDQ_CTL_ENABLE		CSR_BIT(1)
#define FBNIC_QUEUE_BDQ_CTL_PPQ_ENABLE		CSR_BIT(30)

#define FBNIC_QUEUE_BDQ_HPQ_TAIL	0x241		/* 0x904 */
#define FBNIC_QUEUE_BDQ_PPQ_TAIL	0x242		/* 0x908 */

#define FBNIC_QUEUE_BDQ_HPQ_SIZE	0x247		/* 0x91c */
#define FBNIC_QUEUE_BDQ_PPQ_SIZE	0x248		/* 0x920 */
#define FBNIC_QUEUE_BDQ_SIZE_MASK		CSR_GENMASK(3, 0)

#define FBNIC_QUEUE_BDQ_HPQ_BAL		0x260		/* 0x980 */
#define FBNIC_QUEUE_BDQ_HPQ_BAH		0x261		/* 0x984 */
#define FBNIC_QUEUE_BDQ_PPQ_BAL		0x262		/* 0x988 */
#define FBNIC_QUEUE_BDQ_PPQ_BAH		0x263		/* 0x98c */

/* Rx DMA Engine Configuration */
#define FBNIC_QUEUE_RDE_CTL0		0x2a0		/* 0xa80 */
#define FBNIC_QUEUE_RDE_CTL0_EN_HDR_SPLIT	CSR_BIT(31)
#define FBNIC_QUEUE_RDE_CTL0_DROP_MODE_MASK	CSR_GENMASK(30, 29)
enum {
	FBNIC_QUEUE_RDE_CTL0_DROP_IMMEDIATE	= 0,
	FBNIC_QUEUE_RDE_CTL0_DROP_WAIT		= 1,
	FBNIC_QUEUE_RDE_CTL0_DROP_NEVER		= 2,
};

#define FBNIC_QUEUE_RDE_CTL0_MIN_HROOM_MASK	CSR_GENMASK(28, 20)
#define FBNIC_QUEUE_RDE_CTL0_MIN_TROOM_MASK	CSR_GENMASK(19, 11)

#define FBNIC_QUEUE_RDE_CTL1		0x2a1		/* 0xa84 */
#define FBNIC_QUEUE_RDE_CTL1_MAX_HDR_MASK	CSR_GENMASK(24, 12)
#define FBNIC_QUEUE_RDE_CTL1_PAYLD_OFF_MASK	CSR_GENMASK(11, 9)
#define FBNIC_QUEUE_RDE_CTL1_PAYLD_PG_CL_MASK	CSR_GENMASK(8, 6)
#define FBNIC_QUEUE_RDE_CTL1_PADLEN_MASK	CSR_GENMASK(5, 2)
#define FBNIC_QUEUE_RDE_CTL1_PAYLD_PACK_MASK	CSR_GENMASK(1, 0)
enum {
	FBNIC_QUEUE_RDE_CTL1_PAYLD_PACK_NONE	= 0,
	FBNIC_QUEUE_RDE_CTL1_PAYLD_PACK_ALL	= 1,
	FBNIC_QUEUE_RDE_CTL1_PAYLD_PACK_RSS	= 2,
};

/* Rx Per CQ Statistics Counters */
#define FBNIC_QUEUE_RDE_PKT_CNT		0x2a2		/* 0xa88 */
#define FBNIC_QUEUE_RDE_PKT_ERR_CNT	0x2a3		/* 0xa8c */
#define FBNIC_QUEUE_RDE_CQ_DROP_CNT	0x2a4		/* 0xa90 */
#define FBNIC_QUEUE_RDE_BDQ_DROP_CNT	0x2a5		/* 0xa94 */

/* Rx Interrupt Manager Registers */
#define FBNIC_QUEUE_RIM_CTL		0x2c0		/* 0xb00 */
#define FBNIC_QUEUE_RIM_CTL_MSIX_MASK		CSR_GENMASK(7, 0)

#define FBNIC_QUEUE_RIM_THRESHOLD	0x2c1		/* 0xb04 */
#define FBNIC_QUEUE_RIM_THRESHOLD_RCD_MASK	CSR_GENMASK(14, 0)

#define FBNIC_QUEUE_RIM_CLEAR		0x2c2		/* 0xb08 */
#define FBNIC_QUEUE_RIM_CLEAR_MASK		CSR_BIT(0)
#define FBNIC_QUEUE_RIM_SET		0x2c3		/* 0xb0c */
#define FBNIC_QUEUE_RIM_SET_MASK		CSR_BIT(0)
#define FBNIC_QUEUE_RIM_MASK		0x2c4		/* 0xb10 */
#define FBNIC_QUEUE_RIM_MASK_MASK		CSR_BIT(0)

#define FBNIC_QUEUE_RIM_COAL_STATUS	0x2c5		/* 0xb14 */
#define FBNIC_QUEUE_RIM_RCD_COUNT_MASK		CSR_GENMASK(30, 16)
#define FBNIC_QUEUE_RIM_TIMER_MASK		CSR_GENMASK(13, 0)
#define FBNIC_MAX_QUEUES		128
#define FBNIC_CSR_END_QUEUE	(0x40000 + 0x400 * FBNIC_MAX_QUEUES - 1)

/* BAR 4 CSRs */

/* The IPC mailbox consists of 32 mailboxes, with each mailbox consisting
 * of 32 4 byte registers. We will use 2 registers per descriptor so the
 * length of the mailbox is reduced to 16.
 *
 * Currently we use an offset of 0x6000 on BAR4 for the mailbox so we just
 * have to do the math and determine the offset based on the mailbox
 * direction and index inside that mailbox.
 */
#define FBNIC_IPC_MBX_DESC_LEN	16
#define FBNIC_IPC_MBX(mbx_idx, desc_idx)	\
	((((mbx_idx) * FBNIC_IPC_MBX_DESC_LEN + (desc_idx)) * 2) + 0x6000)

/* Use first register in mailbox to flush writes */
#define FBNIC_FW_ZERO_REG	FBNIC_IPC_MBX(0, 0)

enum {
	FBNIC_IPC_MBX_RX_IDX,
	FBNIC_IPC_MBX_TX_IDX,
	FBNIC_IPC_MBX_INDICES,
};

#define FBNIC_IPC_MBX_DESC_LEN_MASK	DESC_GENMASK(63, 48)
#define FBNIC_IPC_MBX_DESC_EOM		DESC_BIT(46)
#define FBNIC_IPC_MBX_DESC_ADDR_MASK	DESC_GENMASK(45, 3)
#define FBNIC_IPC_MBX_DESC_FW_CMPL	DESC_BIT(1)
#define FBNIC_IPC_MBX_DESC_HOST_CMPL	DESC_BIT(0)

/* OTP Registers
 * These registers are accessible via bar4 offset and are written by CMRT
 * on boot. For the write status, the register is broken up in half with OTP
 * Write Data Status occupying the top 16 bits and the ECC status occupying the
 * bottom 16 bits.
 */
#define FBNIC_NS_OTP_STATUS		0x0021d
#define FBNIC_NS_OTP_WRITE_STATUS	0x0021e

#define FBNIC_NS_OTP_WRITE_DATA_STATUS_MASK	CSR_GENMASK(31, 16)
#define FBNIC_NS_OTP_WRITE_ECC_STATUS_MASK	CSR_GENMASK(15, 0)

#define FBNIC_REGS_VERSION			CSR_GENMASK(31, 16)
#define FBNIC_REGS_HW_TYPE			CSR_GENMASK(15, 8)
enum{
	FBNIC_CSR_VERSION_V1_0_ASIC = 1,
};

#endif /* _FBNIC_CSR_H_ */
