/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various indepstopent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QDebug>
#include <QStringList>
#include <QRegularExpression>
#include <QRegularExpressionMatch>

/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/Utils.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


/*!
\class MsXpS::libXpertMassCore::IndexRange
\inmodule libXpertMassCore
\ingroup XpertMassCalculations
\inheaderfile IndexRangeCollection.hpp

\brief The IndexRange class provides a range of indices that delimit a
region of interest in a \l{Polymer}'s \l{Sequence} instance.

One major IndexRange use case is when defining for what sequence region of a
Polymer the masses or the elemental composition is to be computed. Another use
case is when performing Sequence cleavages or Oligomer fragmentations: the
IndexRange enables the delimitation of the sequence of interest inside a given
\l{Polymer}'s \l{Sequence} instance.

When instantiating a new IndexRange, the start and stop values are automatically
sorted in ascending order (start <= stop).

\sa IndexRangeCollection, sortAscending(), sortDescending()
*/

/*!
\variable MsXpS::libXpertMassCore::IndexRange::m_start

\brief Index that marks the start of the index range.
*/

/*!
\variable MsXpS::libXpertMassCore::IndexRange::m_stop

\brief Index that marks the stop of the index range.
*/


/*!
\brief Constructs an IndexRange instance using \a parent as the parent object of
this new instance.

Upon construction, the member data m_start and m_stop are initialized with
std::numeric_limits<std::size_t>::max() values as a means to check if the
IndexRange was initialized by the user or not.
*/
IndexRange::IndexRange(QObject *parent): QObject(parent)
{
}

/*!
\ brief Constructs an IndexRange instance initializing the* start and stop
members to the \a index_start and \a index_stop values, respectively.

The values are sorted in ascending order upon construction.
*/
IndexRange::IndexRange(qsizetype index_start,
                       qsizetype index_stop,
                       QObject *parent)
  : QObject(parent), m_start(index_start), m_stop(index_stop)
{
  sortAscending();
}

/*!
\brief Constructs an IndexRange instance using \a other as a template and \a
parent as the parent of this new instance.

The values are sorted in ascending order upon construction.

\sa sortAscending(), sortDescending()
*/
IndexRange::IndexRange(const IndexRange &other, QObject *parent)
  : QObject(parent), m_start(other.m_start), m_stop(other.m_stop)
{
  sortAscending();
}

/*!
\ brief Destructs this IndexRange instance.
*/
IndexRange::~IndexRange()
{
}

/*!
\brief Initializes this instance using \a other. The parent of this instance is
unchanged.

The values are sorted in ascending order upon initialization.

\sa sortAscending(), sortDescending()
*/
void
IndexRange::initialize(const IndexRange &other)
{
  m_start = other.m_start;
  m_stop  = other.m_stop;
  sortAscending();
}

/*!
\brief Allocates an returns a new IndexRange instance, initializing it using
this instance's member values and setting its parent to \a parent.

The values are sorted in ascending order in the clone.

\sa sortAscending(), sortDescending()
 */
IndexRange *
IndexRange::clone(QObject *parent)
{
  IndexRange *copy_p = new IndexRange(parent);
  copy_p->initialize(*this);
  copy_p->sortAscending();
  return copy_p;
}

/*!
 \ br*ief Allocates an returns a new IndexRange instance, initializing it using
 \a other and setting its parent to \a parent.

 The values are sorted in ascending order in the clone.

 \sa sortAscending(), sortDescending()
 */
IndexRange *
IndexRange::clone(const IndexRange &other, QObject *parent)
{
  IndexRange *copy_p = new IndexRange(parent);
  copy_p->initialize(other);
  copy_p->sortAscending();
  return copy_p;
}

/*!
\brief Returns true if this and \a other IndexRange instances are identical.

The parent is not checked.
 */
bool
IndexRange::operator==(const IndexRange &other) const
{
  return m_start == other.m_start && m_stop == other.m_stop;
};

/*!
 \brief Returns true if this and \a other IndexRange instances are different.

 The parent is not checked.

 Return the negation of \l{operator==()}.
 */
bool
IndexRange::operator!=(const IndexRange &other) const
{
  return !operator==(other);
}

/*!
\brief Makes sure that \l m_start is <= to \l m_stop.
*/
void
IndexRange::sortAscending()
{
  if(m_stop < m_start)
    std::swap(m_start, m_stop);
}

/*!
\brief Makes sure that \l m_start is >= to \l m_stop.
*/
void
IndexRange::sortDescending()
{
  if(m_stop > m_start)
    std::swap(m_start, m_stop);
}

/*!
\brief Returns true if this instance is valid.

The IndexRange intance is considered valid if \l m_start and \l m_stop contain
valid values, that is, not the default-assigned values.
*/
bool
IndexRange::isValid() const
{
  return m_start < std::numeric_limits<qsizetype>::max() &&
         m_stop < std::numeric_limits<qsizetype>::max();
}

/*!
 \brief Resets the member data to default values.
 */
void
IndexRange::reset()
{
  m_start = std::numeric_limits<qsizetype>::max();
  m_stop  = std::numeric_limits<qsizetype>::max();
}

/*!
\brief Returns a string representing this IndexRange instance in the form
"[start-stop]".

The start and stop values are indices (not positions).

\sa positionsAsText()
 */
QString
IndexRange::indicesAsText() const
{
  return QString("[%1-%2]").arg(m_start).arg(m_stop);
};

/*!
\brief Returns a string representing this IndexRange instance in the form
"[start-stop]".

\note The start and stop values are positions (not indices). This means that the
returned values correspond to \l m_start and \l m_stop both incremented by one
unit.

\sa indicesAsText()
*/
QString
IndexRange::positionsAsText() const
{
  return QString("[%1-%2]").arg(m_start + 1).arg(m_stop + 1);
};

void
IndexRange::registerJsConstructor(QJSEngine *engine)

{
  if(!engine)
    {
      qDebug() << "Cannot register IndexRange class: engine is null";
      return;
    }

  // Register the meta object as a constructor

  QJSValue jsMetaObject = engine->newQMetaObject(&IndexRange::staticMetaObject);
  engine->globalObject().setProperty("IndexRange", jsMetaObject);
}


} // namespace libXpertMassCore
} // namespace MsXpS
