/**
 * Basis Universal single file library. Generated using:
 * \code
 *	./combine.sh -r ../../transcoder -o basisu_transcoder.cpp basisu_transcoder-in.cpp
 * \endcode
 */

/*
 * Transcoder build options for known platforms (iOS has ETC, ASTC and PVRTC;
 * Emscripten adds DXT to iOS's options; Android adds PVRTC2 to Emscripten's
 * options; other platforms build all except FXT1).
 * 
 * See https://github.com/BinomialLLC/basis_universal#shrinking-the-transcoders-compiled-size
 */
#ifdef __APPLE__
	#include <TargetConditionals.h>
#endif
#if TARGET_OS_IPHONE
	#define BASISD_SUPPORT_DXT1  0
	#define BASISD_SUPPORT_DXT5A 0
#endif
#if TARGET_OS_IPHONE || defined(__EMSCRIPTEN__) || defined(__ANDROID__)
	#define BASISD_SUPPORT_BC7 0
	#define BASISD_SUPPORT_ATC 0
	#ifndef __ANDROID__
		#define BASISD_SUPPORT_PVRTC2 0
	#endif
#endif
#define BASISD_SUPPORT_FXT1 0

/*
 * KTX2 support disabled.
 */
#define BASISD_SUPPORT_KTX2 0

/**** start inlining basisu_transcoder.cpp ****/
// basisu_transcoder.cpp
// Copyright (C) 2019-2021 Binomial LLC. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**** start inlining basisu_transcoder.h ****/
// basisu_transcoder.h
// Copyright (C) 2019-2021 Binomial LLC. All Rights Reserved.
// Important: If compiling with gcc, be sure strict aliasing is disabled: -fno-strict-aliasing
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// By default KTX2 support is enabled to simplify compilation. This implies the need for the Zstandard library (which we distribute as a single source file in the "zstd" directory) by default.
// Set BASISD_SUPPORT_KTX2 to 0 to completely disable KTX2 support as well as Zstd/miniz usage which is only required for UASTC supercompression in KTX2 files.
// Also see BASISD_SUPPORT_KTX2_ZSTD in basisu_transcoder.cpp, which individually disables Zstd usage.
#ifndef BASISD_SUPPORT_KTX2
	#define BASISD_SUPPORT_KTX2 1
#endif

// Set BASISD_SUPPORT_KTX2_ZSTD to 0 to disable Zstd usage and KTX2 UASTC Zstd supercompression support 
#ifndef BASISD_SUPPORT_KTX2_ZSTD
	#define BASISD_SUPPORT_KTX2_ZSTD 1
#endif

// Set BASISU_FORCE_DEVEL_MESSAGES to 1 to enable debug printf()'s whenever an error occurs, for easier debugging during development.
#ifndef BASISU_FORCE_DEVEL_MESSAGES
	#define BASISU_FORCE_DEVEL_MESSAGES 0
#endif

/**** start inlining basisu_transcoder_internal.h ****/
// basisu_transcoder_internal.h - Universal texture format transcoder library.
// Copyright (C) 2019-2021 Binomial LLC. All Rights Reserved.
//
// Important: If compiling with gcc, be sure strict aliasing is disabled: -fno-strict-aliasing
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifdef _MSC_VER
#pragma warning (disable: 4127) //  conditional expression is constant
#endif

#define BASISD_LIB_VERSION 116
#define BASISD_VERSION_STRING "01.16"

#ifdef _DEBUG
#define BASISD_BUILD_DEBUG
#else
#define BASISD_BUILD_RELEASE
#endif

/**** start inlining basisu.h ****/
// basisu.h
// Copyright (C) 2019-2021 Binomial LLC. All Rights Reserved.
// Important: If compiling with gcc, be sure strict aliasing is disabled: -fno-strict-aliasing
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifdef _MSC_VER

	#pragma warning (disable : 4201)
	#pragma warning (disable : 4127) // warning C4127: conditional expression is constant
	#pragma warning (disable : 4530) // C++ exception handler used, but unwind semantics are not enabled.

	// Slamming this off always for v1.16 because we've gotten rid of most std containers.
	#ifndef BASISU_NO_ITERATOR_DEBUG_LEVEL
		#define BASISU_NO_ITERATOR_DEBUG_LEVEL (1)
	#endif

	#ifndef BASISU_NO_ITERATOR_DEBUG_LEVEL
		//#define _HAS_ITERATOR_DEBUGGING 0

		#if defined(_DEBUG) || defined(DEBUG)
			// This is madness, but we need to disable iterator debugging in debug builds or the encoder is unsable because MSVC's iterator debugging implementation is totally broken.
			#ifndef _ITERATOR_DEBUG_LEVEL
			#define _ITERATOR_DEBUG_LEVEL 1
			#endif
			#ifndef _SECURE_SCL
			#define _SECURE_SCL 1
			#endif
		#else // defined(_DEBUG) || defined(DEBUG)
			#ifndef _SECURE_SCL
			#define _SECURE_SCL 0
			#endif
			#ifndef _ITERATOR_DEBUG_LEVEL
			#define _ITERATOR_DEBUG_LEVEL 0
			#endif
		#endif // defined(_DEBUG) || defined(DEBUG)

	#endif // BASISU_NO_ITERATOR_DEBUG_LEVEL

#endif // _MSC_VER

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <stdarg.h>
#include <string.h>
#include <memory.h>
#include <limits.h>
#include <stdint.h>

#include <algorithm>
#include <limits>
#include <functional>
#include <iterator>
#include <type_traits>
#include <assert.h>
#include <random>

/**** start inlining basisu_containers.h ****/
// basisu_containers.h
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <assert.h>
#include <algorithm>

#if defined(__linux__) && !defined(ANDROID)
// Only for malloc_usable_size() in basisu_containers_impl.h
#include <malloc.h>
#define HAS_MALLOC_USABLE_SIZE 1
#endif

// Set to 1 to always check vector operator[], front(), and back() even in release.
#define BASISU_VECTOR_FORCE_CHECKING 0

// If 1, the vector container will not query the CRT to get the size of resized memory blocks.
#define BASISU_VECTOR_DETERMINISTIC 1

#ifdef _MSC_VER
#define BASISU_FORCE_INLINE __forceinline
#else
#define BASISU_FORCE_INLINE inline
#endif

namespace basisu
{
   enum { cInvalidIndex = -1 };

   namespace helpers
   {
      inline bool is_power_of_2(uint32_t x) { return x && ((x & (x - 1U)) == 0U); }
      inline bool is_power_of_2(uint64_t x) { return x && ((x & (x - 1U)) == 0U); }
      template<class T> const T& minimum(const T& a, const T& b) { return (b < a) ? b : a; }
      template<class T> const T& maximum(const T& a, const T& b) { return (a < b) ? b : a; }

      inline uint32_t floor_log2i(uint32_t v)
      {
         uint32_t l = 0;
         while (v > 1U)
         {
            v >>= 1;
            l++;
         }
         return l;
      }

      inline uint32_t next_pow2(uint32_t val)
      {
         val--;
         val |= val >> 16;
         val |= val >> 8;
         val |= val >> 4;
         val |= val >> 2;
         val |= val >> 1;
         return val + 1;
      }

      inline uint64_t next_pow2(uint64_t val)
      {
         val--;
         val |= val >> 32;
         val |= val >> 16;
         val |= val >> 8;
         val |= val >> 4;
         val |= val >> 2;
         val |= val >> 1;
         return val + 1;
      }
   } // namespace helpers

   template <typename T>
   inline T* construct(T* p)
   {
      return new (static_cast<void*>(p)) T;
   }

   template <typename T, typename U>
   inline T* construct(T* p, const U& init)
   {
      return new (static_cast<void*>(p)) T(init);
   }

   template <typename T>
   inline void construct_array(T* p, size_t n)
   {
      T* q = p + n;
      for (; p != q; ++p)
         new (static_cast<void*>(p)) T;
   }

   template <typename T, typename U>
   inline void construct_array(T* p, size_t n, const U& init)
   {
      T* q = p + n;
      for (; p != q; ++p)
         new (static_cast<void*>(p)) T(init);
   }

   template <typename T>
   inline void destruct(T* p)
   {
      (void)p;
      p->~T();
   }

   template <typename T> inline void destruct_array(T* p, size_t n)
   {
      T* q = p + n;
      for (; p != q; ++p)
         p->~T();
   }

   template<typename T> struct int_traits { enum { cMin = INT32_MIN, cMax = INT32_MAX, cSigned = true }; };

   template<> struct int_traits<int8_t> { enum { cMin = INT8_MIN, cMax = INT8_MAX, cSigned = true }; };
   template<> struct int_traits<int16_t> { enum { cMin = INT16_MIN, cMax = INT16_MAX, cSigned = true }; };
   template<> struct int_traits<int32_t> { enum { cMin = INT32_MIN, cMax = INT32_MAX, cSigned = true }; };

   template<> struct int_traits<uint8_t> { enum { cMin = 0, cMax = UINT8_MAX, cSigned = false }; };
   template<> struct int_traits<uint16_t> { enum { cMin = 0, cMax = UINT16_MAX, cSigned = false }; };
   template<> struct int_traits<uint32_t> { enum { cMin = 0, cMax = UINT32_MAX, cSigned = false }; };

   template<typename T>
   struct scalar_type
   {
      enum { cFlag = false };
      static inline void construct(T* p) { basisu::construct(p); }
      static inline void construct(T* p, const T& init) { basisu::construct(p, init); }
      static inline void construct_array(T* p, size_t n) { basisu::construct_array(p, n); }
      static inline void destruct(T* p) { basisu::destruct(p); }
      static inline void destruct_array(T* p, size_t n) { basisu::destruct_array(p, n); }
   };

   template<typename T> struct scalar_type<T*>
   {
      enum { cFlag = true };
      static inline void construct(T** p) { memset(p, 0, sizeof(T*)); }
      static inline void construct(T** p, T* init) { *p = init; }
      static inline void construct_array(T** p, size_t n) { memset(p, 0, sizeof(T*) * n); }
      static inline void destruct(T** p) { p; }
      static inline void destruct_array(T** p, size_t n) { p, n; }
   };

#define BASISU_DEFINE_BUILT_IN_TYPE(X) \
   template<> struct scalar_type<X> { \
   enum { cFlag = true }; \
   static inline void construct(X* p) { memset(p, 0, sizeof(X)); } \
   static inline void construct(X* p, const X& init) { memcpy(p, &init, sizeof(X)); } \
   static inline void construct_array(X* p, size_t n) { memset(p, 0, sizeof(X) * n); } \
   static inline void destruct(X* p) { p; } \
   static inline void destruct_array(X* p, size_t n) { p, n; } };

   BASISU_DEFINE_BUILT_IN_TYPE(bool)
   BASISU_DEFINE_BUILT_IN_TYPE(char)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned char)
   BASISU_DEFINE_BUILT_IN_TYPE(short)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned short)
   BASISU_DEFINE_BUILT_IN_TYPE(int)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned int)
   BASISU_DEFINE_BUILT_IN_TYPE(long)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned long)
#ifdef __GNUC__
   BASISU_DEFINE_BUILT_IN_TYPE(long long)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned long long)
#else
   BASISU_DEFINE_BUILT_IN_TYPE(__int64)
   BASISU_DEFINE_BUILT_IN_TYPE(unsigned __int64)
#endif
   BASISU_DEFINE_BUILT_IN_TYPE(float)
   BASISU_DEFINE_BUILT_IN_TYPE(double)
   BASISU_DEFINE_BUILT_IN_TYPE(long double)

#undef BASISU_DEFINE_BUILT_IN_TYPE

   template<typename T>
   struct bitwise_movable { enum { cFlag = false }; };

#define BASISU_DEFINE_BITWISE_MOVABLE(Q) template<> struct bitwise_movable<Q> { enum { cFlag = true }; };

   template<typename T>
   struct bitwise_copyable { enum { cFlag = false }; };

#define BASISU_DEFINE_BITWISE_COPYABLE(Q) template<> struct bitwise_copyable<Q> { enum { cFlag = true }; };

#define BASISU_IS_POD(T) __is_pod(T)

#define BASISU_IS_SCALAR_TYPE(T) (scalar_type<T>::cFlag)

#if defined(__GNUC__) && __GNUC__<5
   #define BASISU_IS_TRIVIALLY_COPYABLE(...) __has_trivial_copy(__VA_ARGS__)
#else
   #define BASISU_IS_TRIVIALLY_COPYABLE(...) std::is_trivially_copyable<__VA_ARGS__>::value
#endif

// TODO: clean this up
#define BASISU_IS_BITWISE_COPYABLE(T) (BASISU_IS_SCALAR_TYPE(T) || BASISU_IS_POD(T) || BASISU_IS_TRIVIALLY_COPYABLE(T) || (bitwise_copyable<T>::cFlag))

#define BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(T) (BASISU_IS_BITWISE_COPYABLE(T) || (bitwise_movable<T>::cFlag))

#define BASISU_HAS_DESTRUCTOR(T) ((!scalar_type<T>::cFlag) && (!__is_pod(T)))

   typedef char(&yes_t)[1];
   typedef char(&no_t)[2];

   template <class U> yes_t class_test(int U::*);
   template <class U> no_t class_test(...);

   template <class T> struct is_class
   {
      enum { value = (sizeof(class_test<T>(0)) == sizeof(yes_t)) };
   };

   template <typename T> struct is_pointer
   {
      enum { value = false };
   };

   template <typename T> struct is_pointer<T*>
   {
      enum { value = true };
   };

   struct empty_type { };

   BASISU_DEFINE_BITWISE_COPYABLE(empty_type);
   BASISU_DEFINE_BITWISE_MOVABLE(empty_type);

   template<typename T> struct rel_ops
   {
      friend bool operator!=(const T& x, const T& y) { return (!(x == y)); }
      friend bool operator> (const T& x, const T& y) { return (y < x); }
      friend bool operator<=(const T& x, const T& y) { return (!(y < x)); }
      friend bool operator>=(const T& x, const T& y) { return (!(x < y)); }
   };

   struct elemental_vector
   {
      void* m_p;
      uint32_t m_size;
      uint32_t m_capacity;

      typedef void (*object_mover)(void* pDst, void* pSrc, uint32_t num);

      bool increase_capacity(uint32_t min_new_capacity, bool grow_hint, uint32_t element_size, object_mover pRelocate, bool nofail);
   };

   template<typename T>
   class vector : public rel_ops< vector<T> >
   {
   public:
      typedef T* iterator;
      typedef const T* const_iterator;
      typedef T value_type;
      typedef T& reference;
      typedef const T& const_reference;
      typedef T* pointer;
      typedef const T* const_pointer;

      inline vector() :
         m_p(NULL),
         m_size(0),
         m_capacity(0)
      {
      }

      inline vector(uint32_t n, const T& init) :
         m_p(NULL),
         m_size(0),
         m_capacity(0)
      {
         increase_capacity(n, false);
         construct_array(m_p, n, init);
         m_size = n;
      }

      inline vector(const vector& other) :
         m_p(NULL),
         m_size(0),
         m_capacity(0)
      {
         increase_capacity(other.m_size, false);

         m_size = other.m_size;

         if (BASISU_IS_BITWISE_COPYABLE(T))
            memcpy(m_p, other.m_p, m_size * sizeof(T));
         else
         {
            T* pDst = m_p;
            const T* pSrc = other.m_p;
            for (uint32_t i = m_size; i > 0; i--)
               construct(pDst++, *pSrc++);
         }
      }

      inline explicit vector(size_t size) :
         m_p(NULL),
         m_size(0),
         m_capacity(0)
      {
         resize(size);
      }

      inline ~vector()
      {
         if (m_p)
         {
            scalar_type<T>::destruct_array(m_p, m_size);
            free(m_p);
         }
      }

      inline vector& operator= (const vector& other)
      {
         if (this == &other)
            return *this;

         if (m_capacity >= other.m_size)
            resize(0);
         else
         {
            clear();
            increase_capacity(other.m_size, false);
         }

         if (BASISU_IS_BITWISE_COPYABLE(T))
            memcpy(m_p, other.m_p, other.m_size * sizeof(T));
         else
         {
            T* pDst = m_p;
            const T* pSrc = other.m_p;
            for (uint32_t i = other.m_size; i > 0; i--)
               construct(pDst++, *pSrc++);
         }

         m_size = other.m_size;

         return *this;
      }

      BASISU_FORCE_INLINE const T* begin() const { return m_p; }
      BASISU_FORCE_INLINE T* begin() { return m_p; }

      BASISU_FORCE_INLINE const T* end() const { return m_p + m_size; }
      BASISU_FORCE_INLINE T* end() { return m_p + m_size; }

      BASISU_FORCE_INLINE bool empty() const { return !m_size; }
      BASISU_FORCE_INLINE uint32_t size() const { return m_size; }
      BASISU_FORCE_INLINE uint32_t size_in_bytes() const { return m_size * sizeof(T); }
      BASISU_FORCE_INLINE uint32_t capacity() const { return m_capacity; }

      // operator[] will assert on out of range indices, but in final builds there is (and will never be) any range checking on this method.
      //BASISU_FORCE_INLINE const T& operator[] (uint32_t i) const { assert(i < m_size); return m_p[i]; }
      //BASISU_FORCE_INLINE T& operator[] (uint32_t i) { assert(i < m_size); return m_p[i]; }
            
#if !BASISU_VECTOR_FORCE_CHECKING
      BASISU_FORCE_INLINE const T& operator[] (size_t i) const { assert(i < m_size); return m_p[i]; }
      BASISU_FORCE_INLINE T& operator[] (size_t i) { assert(i < m_size); return m_p[i]; }
#else
      BASISU_FORCE_INLINE const T& operator[] (size_t i) const 
      { 
          if (i >= m_size)
          {
              fprintf(stderr, "operator[] invalid index: %u, max entries %u, type size %u\n", (uint32_t)i, m_size, (uint32_t)sizeof(T));
              abort();
          }
          return m_p[i]; 
      }
      BASISU_FORCE_INLINE T& operator[] (size_t i) 
      { 
          if (i >= m_size)
          {
              fprintf(stderr, "operator[] invalid index: %u, max entries %u, type size %u\n", (uint32_t)i, m_size, (uint32_t)sizeof(T));
              abort();
          }
          return m_p[i]; 
      }
#endif

      // at() always includes range checking, even in final builds, unlike operator [].
      // The first element is returned if the index is out of range.
      BASISU_FORCE_INLINE const T& at(size_t i) const { assert(i < m_size); return (i >= m_size) ? m_p[0] : m_p[i]; }
      BASISU_FORCE_INLINE T& at(size_t i) { assert(i < m_size); return (i >= m_size) ? m_p[0] : m_p[i]; }
            
#if !BASISU_VECTOR_FORCE_CHECKING
      BASISU_FORCE_INLINE const T& front() const { assert(m_size); return m_p[0]; }
      BASISU_FORCE_INLINE T& front() { assert(m_size); return m_p[0]; }

      BASISU_FORCE_INLINE const T& back() const { assert(m_size); return m_p[m_size - 1]; }
      BASISU_FORCE_INLINE T& back() { assert(m_size); return m_p[m_size - 1]; }
#else
      BASISU_FORCE_INLINE const T& front() const 
      { 
          if (!m_size)
          {
              fprintf(stderr, "front: vector is empty, type size %u\n", (uint32_t)sizeof(T));
              abort();
          }
          return m_p[0]; 
      }
      BASISU_FORCE_INLINE T& front() 
      { 
          if (!m_size)
          {
              fprintf(stderr, "front: vector is empty, type size %u\n", (uint32_t)sizeof(T));
              abort();
          }
          return m_p[0]; 
      }

      BASISU_FORCE_INLINE const T& back() const 
      { 
          if(!m_size)
          {
              fprintf(stderr, "back: vector is empty, type size %u\n", (uint32_t)sizeof(T));
              abort();
          }
          return m_p[m_size - 1]; 
      }
      BASISU_FORCE_INLINE T& back() 
      { 
          if (!m_size)
          {
              fprintf(stderr, "back: vector is empty, type size %u\n", (uint32_t)sizeof(T));
              abort();
          }
          return m_p[m_size - 1]; 
      }
#endif

      BASISU_FORCE_INLINE const T* get_ptr() const { return m_p; }
      BASISU_FORCE_INLINE T* get_ptr() { return m_p; }

      BASISU_FORCE_INLINE const T* data() const { return m_p; }
      BASISU_FORCE_INLINE T* data() { return m_p; }

      // clear() sets the container to empty, then frees the allocated block.
      inline void clear()
      {
         if (m_p)
         {
            scalar_type<T>::destruct_array(m_p, m_size);
            free(m_p);
            m_p = NULL;
            m_size = 0;
            m_capacity = 0;
         }
      }

      inline void clear_no_destruction()
      {
         if (m_p)
         {
            free(m_p);
            m_p = NULL;
            m_size = 0;
            m_capacity = 0;
         }
      }

      inline void reserve(size_t new_capacity_size_t)
      {
         if (new_capacity_size_t > UINT32_MAX)
         {
            assert(0);
            return;
         }

         uint32_t new_capacity = (uint32_t)new_capacity_size_t;

         if (new_capacity > m_capacity)
            increase_capacity(new_capacity, false);
         else if (new_capacity < m_capacity)
         {
            // Must work around the lack of a "decrease_capacity()" method.
            // This case is rare enough in practice that it's probably not worth implementing an optimized in-place resize.
            vector tmp;
            tmp.increase_capacity(helpers::maximum(m_size, new_capacity), false);
            tmp = *this;
            swap(tmp);
         }
      }

      inline bool try_reserve(size_t new_capacity_size_t)
      {
         if (new_capacity_size_t > UINT32_MAX)
         {
            assert(0);
            return false;
         }

         uint32_t new_capacity = (uint32_t)new_capacity_size_t;

         if (new_capacity > m_capacity)
         {
            if (!increase_capacity(new_capacity, false))
               return false;
         }
         else if (new_capacity < m_capacity)
         {
            // Must work around the lack of a "decrease_capacity()" method.
            // This case is rare enough in practice that it's probably not worth implementing an optimized in-place resize.
            vector tmp;
            tmp.increase_capacity(helpers::maximum(m_size, new_capacity), false);
            tmp = *this;
            swap(tmp);
         }

         return true;
      }

      // resize(0) sets the container to empty, but does not free the allocated block.
      inline void resize(size_t new_size_size_t, bool grow_hint = false)
      {
         if (new_size_size_t > UINT32_MAX)
         {
            assert(0);
            return;
         }

         uint32_t new_size = (uint32_t)new_size_size_t;

         if (m_size != new_size)
         {
            if (new_size < m_size)
               scalar_type<T>::destruct_array(m_p + new_size, m_size - new_size);
            else
            {
               if (new_size > m_capacity)
                  increase_capacity(new_size, (new_size == (m_size + 1)) || grow_hint);

               scalar_type<T>::construct_array(m_p + m_size, new_size - m_size);
            }

            m_size = new_size;
         }
      }

      inline bool try_resize(size_t new_size_size_t, bool grow_hint = false)
      {
         if (new_size_size_t > UINT32_MAX)
         {
            assert(0);
            return false;
         }

         uint32_t new_size = (uint32_t)new_size_size_t;

         if (m_size != new_size)
         {
            if (new_size < m_size)
               scalar_type<T>::destruct_array(m_p + new_size, m_size - new_size);
            else
            {
               if (new_size > m_capacity)
               {
                  if (!increase_capacity(new_size, (new_size == (m_size + 1)) || grow_hint, true))
                     return false;
               }

               scalar_type<T>::construct_array(m_p + m_size, new_size - m_size);
            }

            m_size = new_size;
         }

         return true;
      }

      // If size >= capacity/2, reset() sets the container's size to 0 but doesn't free the allocated block (because the container may be similarly loaded in the future).
      // Otherwise it blows away the allocated block. See http://www.codercorner.com/blog/?p=494
      inline void reset()
      {
         if (m_size >= (m_capacity >> 1))
            resize(0);
         else
            clear();
      }

      inline T* enlarge(uint32_t i)
      {
         uint32_t cur_size = m_size;
         resize(cur_size + i, true);
         return get_ptr() + cur_size;
      }

      inline T* try_enlarge(uint32_t i)
      {
         uint32_t cur_size = m_size;
         if (!try_resize(cur_size + i, true))
            return NULL;
         return get_ptr() + cur_size;
      }

      BASISU_FORCE_INLINE void push_back(const T& obj)
      {
         assert(!m_p || (&obj < m_p) || (&obj >= (m_p + m_size)));

         if (m_size >= m_capacity)
            increase_capacity(m_size + 1, true);

         scalar_type<T>::construct(m_p + m_size, obj);
         m_size++;
      }

      inline bool try_push_back(const T& obj)
      {
         assert(!m_p || (&obj < m_p) || (&obj >= (m_p + m_size)));

         if (m_size >= m_capacity)
         {
            if (!increase_capacity(m_size + 1, true, true))
               return false;
         }

         scalar_type<T>::construct(m_p + m_size, obj);
         m_size++;

         return true;
      }

      inline void push_back_value(T obj)
      {
         if (m_size >= m_capacity)
            increase_capacity(m_size + 1, true);

         scalar_type<T>::construct(m_p + m_size, obj);
         m_size++;
      }

      inline void pop_back()
      {
         assert(m_size);

         if (m_size)
         {
            m_size--;
            scalar_type<T>::destruct(&m_p[m_size]);
         }
      }

      inline void insert(uint32_t index, const T* p, uint32_t n)
      {
         assert(index <= m_size);
         if (!n)
            return;

         const uint32_t orig_size = m_size;
         resize(m_size + n, true);

         const uint32_t num_to_move = orig_size - index;

         if (BASISU_IS_BITWISE_COPYABLE(T))
         {
            // This overwrites the destination object bits, but bitwise copyable means we don't need to worry about destruction.
            memmove(m_p + index + n, m_p + index, sizeof(T) * num_to_move);
         }
         else
         {
            const T* pSrc = m_p + orig_size - 1;
            T* pDst = const_cast<T*>(pSrc) + n;

            for (uint32_t i = 0; i < num_to_move; i++)
            {
               assert((pDst - m_p) < (int)m_size);
               *pDst-- = *pSrc--;
            }
         }

         T* pDst = m_p + index;

         if (BASISU_IS_BITWISE_COPYABLE(T))
         {
            // This copies in the new bits, overwriting the existing objects, which is OK for copyable types that don't need destruction.
            memcpy(pDst, p, sizeof(T) * n);
         }
         else
         {
            for (uint32_t i = 0; i < n; i++)
            {
               assert((pDst - m_p) < (int)m_size);
               *pDst++ = *p++;
            }
         }
      }

      inline void insert(T* p, const T& obj)
      {
         int64_t ofs = p - begin();
         if ((ofs < 0) || (ofs > UINT32_MAX))
         {
            assert(0);
            return;
         }

         insert((uint32_t)ofs, &obj, 1);
      }

      // push_front() isn't going to be very fast - it's only here for usability.
      inline void push_front(const T& obj)
      {
         insert(0, &obj, 1);
      }

      vector& append(const vector& other)
      {
         if (other.m_size)
            insert(m_size, &other[0], other.m_size);
         return *this;
      }

      vector& append(const T* p, uint32_t n)
      {
         if (n)
            insert(m_size, p, n);
         return *this;
      }
            
      inline void erase(uint32_t start, uint32_t n)
      {
         assert((start + n) <= m_size);
         if ((start + n) > m_size)
            return;

         if (!n)
            return;

         const uint32_t num_to_move = m_size - (start + n);

         T* pDst = m_p + start;

         const T* pSrc = m_p + start + n;

         if (BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(T))
         {
            // This test is overly cautious.
            if ((!BASISU_IS_BITWISE_COPYABLE(T)) || (BASISU_HAS_DESTRUCTOR(T)))
            {
               // Type has been marked explictly as bitwise movable, which means we can move them around but they may need to be destructed.
               // First destroy the erased objects.
               scalar_type<T>::destruct_array(pDst, n);
            }

            // Copy "down" the objects to preserve, filling in the empty slots.
            memmove(pDst, pSrc, num_to_move * sizeof(T));
         }
         else
         {
            // Type is not bitwise copyable or movable. 
            // Move them down one at a time by using the equals operator, and destroying anything that's left over at the end.
            T* pDst_end = pDst + num_to_move;
            while (pDst != pDst_end)
               *pDst++ = *pSrc++;

            scalar_type<T>::destruct_array(pDst_end, n);
         }

         m_size -= n;
      }

      inline void erase(uint32_t index)
      {
         erase(index, 1);
      }

      inline void erase(T* p)
      {
         assert((p >= m_p) && (p < (m_p + m_size)));
         erase(static_cast<uint32_t>(p - m_p));
      }

      inline void erase(T *pFirst, T *pEnd)
      {
         assert(pFirst <= pEnd);
         assert(pFirst >= begin() && pFirst <= end());
         assert(pEnd >= begin() && pEnd <= end());

         int64_t ofs = pFirst - begin();
         if ((ofs < 0) || (ofs > UINT32_MAX))
         {
            assert(0);
            return;
         }

         int64_t n = pEnd - pFirst;
         if ((n < 0) || (n > UINT32_MAX))
         {
            assert(0);
            return;
         }

         erase((uint32_t)ofs, (uint32_t)n);
      }

      void erase_unordered(uint32_t index)
      {
         assert(index < m_size);

         if ((index + 1) < m_size)
            (*this)[index] = back();

         pop_back();
      }

      inline bool operator== (const vector& rhs) const
      {
         if (m_size != rhs.m_size)
            return false;
         else if (m_size)
         {
            if (scalar_type<T>::cFlag)
               return memcmp(m_p, rhs.m_p, sizeof(T) * m_size) == 0;
            else
            {
               const T* pSrc = m_p;
               const T* pDst = rhs.m_p;
               for (uint32_t i = m_size; i; i--)
                  if (!(*pSrc++ == *pDst++))
                     return false;
            }
         }

         return true;
      }

      inline bool operator< (const vector& rhs) const
      {
         const uint32_t min_size = helpers::minimum(m_size, rhs.m_size);

         const T* pSrc = m_p;
         const T* pSrc_end = m_p + min_size;
         const T* pDst = rhs.m_p;

         while ((pSrc < pSrc_end) && (*pSrc == *pDst))
         {
            pSrc++;
            pDst++;
         }

         if (pSrc < pSrc_end)
            return *pSrc < *pDst;

         return m_size < rhs.m_size;
      }

      inline void swap(vector& other)
      {
         std::swap(m_p, other.m_p);
         std::swap(m_size, other.m_size);
         std::swap(m_capacity, other.m_capacity);
      }

      inline void sort()
      {
         std::sort(begin(), end());
      }

      inline void unique()
      {
         if (!empty())
         {
            sort();

            resize(std::unique(begin(), end()) - begin());
         }
      }

      inline void reverse()
      {
         uint32_t j = m_size >> 1;
         for (uint32_t i = 0; i < j; i++)
            std::swap(m_p[i], m_p[m_size - 1 - i]);
      }

      inline int find(const T& key) const
      {
         const T* p = m_p;
         const T* p_end = m_p + m_size;

         uint32_t index = 0;

         while (p != p_end)
         {
            if (key == *p)
               return index;

            p++;
            index++;
         }

         return cInvalidIndex;
      }

      inline int find_sorted(const T& key) const
      {
         if (m_size)
         {
            // Uniform binary search - Knuth Algorithm 6.2.1 U, unrolled twice.
            int i = ((m_size + 1) >> 1) - 1;
            int m = m_size;

            for (; ; )
            {
               assert(i >= 0 && i < (int)m_size);
               const T* pKey_i = m_p + i;
               int cmp = key < *pKey_i;
#if defined(_DEBUG) || defined(DEBUG)
               int cmp2 = *pKey_i < key;
               assert((cmp != cmp2) || (key == *pKey_i));
#endif
               if ((!cmp) && (key == *pKey_i)) return i;
               m >>= 1;
               if (!m) break;
               cmp = -cmp;
               i += (((m + 1) >> 1) ^ cmp) - cmp;
               if (i < 0)
                  break;

               assert(i >= 0 && i < (int)m_size);
               pKey_i = m_p + i;
               cmp = key < *pKey_i;
#if defined(_DEBUG) || defined(DEBUG)
               cmp2 = *pKey_i < key;
               assert((cmp != cmp2) || (key == *pKey_i));
#endif
               if ((!cmp) && (key == *pKey_i)) return i;
               m >>= 1;
               if (!m) break;
               cmp = -cmp;
               i += (((m + 1) >> 1) ^ cmp) - cmp;
               if (i < 0)
                  break;
            }
         }

         return cInvalidIndex;
      }

      template<typename Q>
      inline int find_sorted(const T& key, Q less_than) const
      {
         if (m_size)
         {
            // Uniform binary search - Knuth Algorithm 6.2.1 U, unrolled twice.
            int i = ((m_size + 1) >> 1) - 1;
            int m = m_size;

            for (; ; )
            {
               assert(i >= 0 && i < (int)m_size);
               const T* pKey_i = m_p + i;
               int cmp = less_than(key, *pKey_i);
               if ((!cmp) && (!less_than(*pKey_i, key))) return i;
               m >>= 1;
               if (!m) break;
               cmp = -cmp;
               i += (((m + 1) >> 1) ^ cmp) - cmp;
               if (i < 0)
                  break;

               assert(i >= 0 && i < (int)m_size);
               pKey_i = m_p + i;
               cmp = less_than(key, *pKey_i);
               if ((!cmp) && (!less_than(*pKey_i, key))) return i;
               m >>= 1;
               if (!m) break;
               cmp = -cmp;
               i += (((m + 1) >> 1) ^ cmp) - cmp;
               if (i < 0) 
                  break;
            }
         }

         return cInvalidIndex;
      }

      inline uint32_t count_occurences(const T& key) const
      {
         uint32_t c = 0;

         const T* p = m_p;
         const T* p_end = m_p + m_size;

         while (p != p_end)
         {
            if (key == *p)
               c++;

            p++;
         }

         return c;
      }

      inline void set_all(const T& o)
      {
         if ((sizeof(T) == 1) && (scalar_type<T>::cFlag))
            memset(m_p, *reinterpret_cast<const uint8_t*>(&o), m_size);
         else
         {
            T* pDst = m_p;
            T* pDst_end = pDst + m_size;
            while (pDst != pDst_end)
               *pDst++ = o;
         }
      }

      // Caller assumes ownership of the heap block associated with the container. Container is cleared.
      inline void* assume_ownership()
      {
         T* p = m_p;
         m_p = NULL;
         m_size = 0;
         m_capacity = 0;
         return p;
      }

      // Caller is granting ownership of the indicated heap block.
      // Block must have size constructed elements, and have enough room for capacity elements.
      // The block must have been allocated using malloc().
      // Important: This method is used in Basis Universal. If you change how this container allocates memory, you'll need to change any users of this method.
      inline bool grant_ownership(T* p, uint32_t size, uint32_t capacity)
      {
         // To to prevent the caller from obviously shooting themselves in the foot.
         if (((p + capacity) > m_p) && (p < (m_p + m_capacity)))
         {
            // Can grant ownership of a block inside the container itself!
            assert(0);
            return false;
         }

         if (size > capacity)
         {
            assert(0);
            return false;
         }

         if (!p)
         {
            if (capacity)
            {
               assert(0);
               return false;
            }
         }
         else if (!capacity)
         {
            assert(0);
            return false;
         }

         clear();
         m_p = p;
         m_size = size;
         m_capacity = capacity;
         return true;
      }

   private:
      T* m_p;
      uint32_t m_size;
      uint32_t m_capacity;

      template<typename Q> struct is_vector { enum { cFlag = false }; };
      template<typename Q> struct is_vector< vector<Q> > { enum { cFlag = true }; };

      static void object_mover(void* pDst_void, void* pSrc_void, uint32_t num)
      {
         T* pSrc = static_cast<T*>(pSrc_void);
         T* const pSrc_end = pSrc + num;
         T* pDst = static_cast<T*>(pDst_void);

         while (pSrc != pSrc_end)
         {
            // placement new
            new (static_cast<void*>(pDst)) T(*pSrc);
            pSrc->~T();
            ++pSrc;
            ++pDst;
         }
      }

      inline bool increase_capacity(uint32_t min_new_capacity, bool grow_hint, bool nofail = false)
      {
         return reinterpret_cast<elemental_vector*>(this)->increase_capacity(
            min_new_capacity, grow_hint, sizeof(T),
            (BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(T) || (is_vector<T>::cFlag)) ? NULL : object_mover, nofail);
      }
   };

   template<typename T> struct bitwise_movable< vector<T> > { enum { cFlag = true }; };

   // Hash map

   template <typename T>
   struct hasher
   {
      inline size_t operator() (const T& key) const { return static_cast<size_t>(key); }
   };

   template <typename T>
   struct equal_to
   {
      inline bool operator()(const T& a, const T& b) const { return a == b; }
   };

   // Important: The Hasher and Equals objects must be bitwise movable!
   template<typename Key, typename Value = empty_type, typename Hasher = hasher<Key>, typename Equals = equal_to<Key> >
   class hash_map
   {
   public:
      class iterator;
      class const_iterator;
   
   private:
      friend class iterator;
      friend class const_iterator;

      enum state
      {
         cStateInvalid = 0,
         cStateValid = 1
      };

      enum
      {
         cMinHashSize = 4U
      };

   public:
      typedef hash_map<Key, Value, Hasher, Equals> hash_map_type;
      typedef std::pair<Key, Value> value_type;
      typedef Key                   key_type;
      typedef Value                 referent_type;
      typedef Hasher                hasher_type;
      typedef Equals                equals_type;

      hash_map() :
         m_hash_shift(32), m_num_valid(0), m_grow_threshold(0)
      {
      }

      hash_map(const hash_map& other) :
         m_values(other.m_values),
         m_hash_shift(other.m_hash_shift),
         m_hasher(other.m_hasher),
         m_equals(other.m_equals),
         m_num_valid(other.m_num_valid),
         m_grow_threshold(other.m_grow_threshold)
      {
      }

      hash_map& operator= (const hash_map& other)
      {
         if (this == &other)
            return *this;

         clear();

         m_values = other.m_values;
         m_hash_shift = other.m_hash_shift;
         m_num_valid = other.m_num_valid;
         m_grow_threshold = other.m_grow_threshold;
         m_hasher = other.m_hasher;
         m_equals = other.m_equals;

         return *this;
      }

      inline ~hash_map()
      {
         clear();
      }

      const Equals& get_equals() const { return m_equals; }
      Equals& get_equals() { return m_equals; }

      void set_equals(const Equals& equals) { m_equals = equals; }

      const Hasher& get_hasher() const { return m_hasher; }
      Hasher& get_hasher() { return m_hasher; }

      void set_hasher(const Hasher& hasher) { m_hasher = hasher; }

      inline void clear()
      {
         if (!m_values.empty())
         {
            if (BASISU_HAS_DESTRUCTOR(Key) || BASISU_HAS_DESTRUCTOR(Value))
            {
               node* p = &get_node(0);
               node* p_end = p + m_values.size();

               uint32_t num_remaining = m_num_valid;
               while (p != p_end)
               {
                  if (p->state)
                  {
                     destruct_value_type(p);
                     num_remaining--;
                     if (!num_remaining)
                        break;
                  }

                  p++;
               }
            }

            m_values.clear_no_destruction();

            m_hash_shift = 32;
            m_num_valid = 0;
            m_grow_threshold = 0;
         }
      }

      inline void reset()
      {
         if (!m_num_valid)
            return;

         if (BASISU_HAS_DESTRUCTOR(Key) || BASISU_HAS_DESTRUCTOR(Value))
         {
            node* p = &get_node(0);
            node* p_end = p + m_values.size();

            uint32_t num_remaining = m_num_valid;
            while (p != p_end)
            {
               if (p->state)
               {
                  destruct_value_type(p);
                  p->state = cStateInvalid;

                  num_remaining--;
                  if (!num_remaining)
                     break;
               }

               p++;
            }
         }
         else if (sizeof(node) <= 32)
         {
            memset(&m_values[0], 0, m_values.size_in_bytes());
         }
         else
         {
            node* p = &get_node(0);
            node* p_end = p + m_values.size();

            uint32_t num_remaining = m_num_valid;
            while (p != p_end)
            {
               if (p->state)
               {
                  p->state = cStateInvalid;

                  num_remaining--;
                  if (!num_remaining)
                     break;
               }

               p++;
            }
         }

         m_num_valid = 0;
      }

      inline uint32_t size()
      {
         return m_num_valid;
      }

      inline uint32_t get_table_size()
      {
         return m_values.size();
      }

      inline bool empty()
      {
         return !m_num_valid;
      }

      inline void reserve(uint32_t new_capacity)
      {
         uint64_t new_hash_size = helpers::maximum(1U, new_capacity);

         new_hash_size = new_hash_size * 2ULL;

         if (!helpers::is_power_of_2(new_hash_size))
            new_hash_size = helpers::next_pow2(new_hash_size);

         new_hash_size = helpers::maximum<uint64_t>(cMinHashSize, new_hash_size);

         new_hash_size = helpers::minimum<uint64_t>(0x80000000UL, new_hash_size);

         if (new_hash_size > m_values.size())
            rehash((uint32_t)new_hash_size);
      }
            
      class iterator
      {
         friend class hash_map<Key, Value, Hasher, Equals>;
         friend class hash_map<Key, Value, Hasher, Equals>::const_iterator;

      public:
         inline iterator() : m_pTable(NULL), m_index(0) { }
         inline iterator(hash_map_type& table, uint32_t index) : m_pTable(&table), m_index(index) { }
         inline iterator(const iterator& other) : m_pTable(other.m_pTable), m_index(other.m_index) { }

         inline iterator& operator= (const iterator& other)
         {
            m_pTable = other.m_pTable;
            m_index = other.m_index;
            return *this;
         }

         // post-increment
         inline iterator operator++(int)
         {
            iterator result(*this);
            ++*this;
            return result;
         }

         // pre-increment
         inline iterator& operator++()
         {
            probe();
            return *this;
         }

         inline value_type& operator*() const { return *get_cur(); }
         inline value_type* operator->() const { return get_cur(); }

         inline bool operator == (const iterator& b) const { return (m_pTable == b.m_pTable) && (m_index == b.m_index); }
         inline bool operator != (const iterator& b) const { return !(*this == b); }
         inline bool operator == (const const_iterator& b) const { return (m_pTable == b.m_pTable) && (m_index == b.m_index); }
         inline bool operator != (const const_iterator& b) const { return !(*this == b); }

      private:
         hash_map_type* m_pTable;
         uint32_t m_index;

         inline value_type* get_cur() const
         {
            assert(m_pTable && (m_index < m_pTable->m_values.size()));
            assert(m_pTable->get_node_state(m_index) == cStateValid);

            return &m_pTable->get_node(m_index);
         }

         inline void probe()
         {
            assert(m_pTable);
            m_index = m_pTable->find_next(m_index);
         }
      };

      class const_iterator
      {
         friend class hash_map<Key, Value, Hasher, Equals>;
         friend class hash_map<Key, Value, Hasher, Equals>::iterator;

      public:
         inline const_iterator() : m_pTable(NULL), m_index(0) { }
         inline const_iterator(const hash_map_type& table, uint32_t index) : m_pTable(&table), m_index(index) { }
         inline const_iterator(const iterator& other) : m_pTable(other.m_pTable), m_index(other.m_index) { }
         inline const_iterator(const const_iterator& other) : m_pTable(other.m_pTable), m_index(other.m_index) { }

         inline const_iterator& operator= (const const_iterator& other)
         {
            m_pTable = other.m_pTable;
            m_index = other.m_index;
            return *this;
         }

         inline const_iterator& operator= (const iterator& other)
         {
            m_pTable = other.m_pTable;
            m_index = other.m_index;
            return *this;
         }

         // post-increment
         inline const_iterator operator++(int)
         {
            const_iterator result(*this);
            ++*this;
            return result;
         }

         // pre-increment
         inline const_iterator& operator++()
         {
            probe();
            return *this;
         }

         inline const value_type& operator*() const { return *get_cur(); }
         inline const value_type* operator->() const { return get_cur(); }

         inline bool operator == (const const_iterator& b) const { return (m_pTable == b.m_pTable) && (m_index == b.m_index); }
         inline bool operator != (const const_iterator& b) const { return !(*this == b); }
         inline bool operator == (const iterator& b) const { return (m_pTable == b.m_pTable) && (m_index == b.m_index); }
         inline bool operator != (const iterator& b) const { return !(*this == b); }

      private:
         const hash_map_type* m_pTable;
         uint32_t m_index;

         inline const value_type* get_cur() const
         {
            assert(m_pTable && (m_index < m_pTable->m_values.size()));
            assert(m_pTable->get_node_state(m_index) == cStateValid);

            return &m_pTable->get_node(m_index);
         }

         inline void probe()
         {
            assert(m_pTable);
            m_index = m_pTable->find_next(m_index);
         }
      };

      inline const_iterator begin() const
      {
         if (!m_num_valid)
            return end();

         return const_iterator(*this, find_next(UINT32_MAX));
      }

      inline const_iterator end() const
      {
         return const_iterator(*this, m_values.size());
      }

      inline iterator begin()
      {
         if (!m_num_valid)
            return end();

         return iterator(*this, find_next(UINT32_MAX));
      }

      inline iterator end()
      {
         return iterator(*this, m_values.size());
      }

      // insert_result.first will always point to inserted key/value (or the already existing key/value).
      // insert_resutt.second will be true if a new key/value was inserted, or false if the key already existed (in which case first will point to the already existing value).
      typedef std::pair<iterator, bool> insert_result;

      inline insert_result insert(const Key& k, const Value& v = Value())
      {
         insert_result result;
         if (!insert_no_grow(result, k, v))
         {
            grow();

            // This must succeed.
            if (!insert_no_grow(result, k, v))
            {
               fprintf(stderr, "insert() failed");
               abort();
            }
         }

         return result;
      }

      inline insert_result insert(const value_type& v)
      {
         return insert(v.first, v.second);
      }

      inline const_iterator find(const Key& k) const
      {
         return const_iterator(*this, find_index(k));
      }

      inline iterator find(const Key& k)
      {
         return iterator(*this, find_index(k));
      }

      inline bool erase(const Key& k)
      {
         uint32_t i = find_index(k);

         if (i >= m_values.size())
            return false;

         node* pDst = &get_node(i);
         destruct_value_type(pDst);
         pDst->state = cStateInvalid;

         m_num_valid--;

         for (; ; )
         {
            uint32_t r, j = i;

            node* pSrc = pDst;

            do
            {
               if (!i)
               {
                  i = m_values.size() - 1;
                  pSrc = &get_node(i);
               }
               else
               {
                  i--;
                  pSrc--;
               }

               if (!pSrc->state)
                  return true;

               r = hash_key(pSrc->first);

            } while ((i <= r && r < j) || (r < j && j < i) || (j < i && i <= r));

            move_node(pDst, pSrc);

            pDst = pSrc;
         }
      }

      inline void swap(hash_map_type& other)
      {
         m_values.swap(other.m_values);
         std::swap(m_hash_shift, other.m_hash_shift);
         std::swap(m_num_valid, other.m_num_valid);
         std::swap(m_grow_threshold, other.m_grow_threshold);
         std::swap(m_hasher, other.m_hasher);
         std::swap(m_equals, other.m_equals);
      }

   private:
      struct node : public value_type
      {
         uint8_t state;
      };

      static inline void construct_value_type(value_type* pDst, const Key& k, const Value& v)
      {
         if (BASISU_IS_BITWISE_COPYABLE(Key))
            memcpy(&pDst->first, &k, sizeof(Key));
         else
            scalar_type<Key>::construct(&pDst->first, k);

         if (BASISU_IS_BITWISE_COPYABLE(Value))
            memcpy(&pDst->second, &v, sizeof(Value));
         else
            scalar_type<Value>::construct(&pDst->second, v);
      }

      static inline void construct_value_type(value_type* pDst, const value_type* pSrc)
      {
         if ((BASISU_IS_BITWISE_COPYABLE(Key)) && (BASISU_IS_BITWISE_COPYABLE(Value)))
         {
            memcpy(pDst, pSrc, sizeof(value_type));
         }
         else
         {
            if (BASISU_IS_BITWISE_COPYABLE(Key))
               memcpy(&pDst->first, &pSrc->first, sizeof(Key));
            else
               scalar_type<Key>::construct(&pDst->first, pSrc->first);

            if (BASISU_IS_BITWISE_COPYABLE(Value))
               memcpy(&pDst->second, &pSrc->second, sizeof(Value));
            else
               scalar_type<Value>::construct(&pDst->second, pSrc->second);
         }
      }

      static inline void destruct_value_type(value_type* p)
      {
         scalar_type<Key>::destruct(&p->first);
         scalar_type<Value>::destruct(&p->second);
      }

      // Moves *pSrc to *pDst efficiently.
      // pDst should NOT be constructed on entry.
      static inline void move_node(node* pDst, node* pSrc, bool update_src_state = true)
      {
         assert(!pDst->state);

         if (BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(Key) && BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(Value))
         {
            memcpy(pDst, pSrc, sizeof(node));
         }
         else
         {
            if (BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(Key))
               memcpy(&pDst->first, &pSrc->first, sizeof(Key));
            else
            {
               scalar_type<Key>::construct(&pDst->first, pSrc->first);
               scalar_type<Key>::destruct(&pSrc->first);
            }

            if (BASISU_IS_BITWISE_COPYABLE_OR_MOVABLE(Value))
               memcpy(&pDst->second, &pSrc->second, sizeof(Value));
            else
            {
               scalar_type<Value>::construct(&pDst->second, pSrc->second);
               scalar_type<Value>::destruct(&pSrc->second);
            }

            pDst->state = cStateValid;
         }

         if (update_src_state)
            pSrc->state = cStateInvalid;
      }

      struct raw_node
      {
         inline raw_node()
         {
            node* p = reinterpret_cast<node*>(this);
            p->state = cStateInvalid;
         }

         inline ~raw_node()
         {
            node* p = reinterpret_cast<node*>(this);
            if (p->state)
               hash_map_type::destruct_value_type(p);
         }

         inline raw_node(const raw_node& other)
         {
            node* pDst = reinterpret_cast<node*>(this);
            const node* pSrc = reinterpret_cast<const node*>(&other);

            if (pSrc->state)
            {
               hash_map_type::construct_value_type(pDst, pSrc);
               pDst->state = cStateValid;
            }
            else
               pDst->state = cStateInvalid;
         }

         inline raw_node& operator= (const raw_node& rhs)
         {
            if (this == &rhs)
               return *this;

            node* pDst = reinterpret_cast<node*>(this);
            const node* pSrc = reinterpret_cast<const node*>(&rhs);

            if (pSrc->state)
            {
               if (pDst->state)
               {
                  pDst->first = pSrc->first;
                  pDst->second = pSrc->second;
               }
               else
               {
                  hash_map_type::construct_value_type(pDst, pSrc);
                  pDst->state = cStateValid;
               }
            }
            else if (pDst->state)
            {
               hash_map_type::destruct_value_type(pDst);
               pDst->state = cStateInvalid;
            }

            return *this;
         }

         uint8_t m_bits[sizeof(node)];
      };

      typedef basisu::vector<raw_node> node_vector;

      node_vector    m_values;
      uint32_t       m_hash_shift;

      Hasher         m_hasher;
      Equals         m_equals;

      uint32_t       m_num_valid;

      uint32_t       m_grow_threshold;

      inline uint32_t hash_key(const Key& k) const
      {
         assert((1U << (32U - m_hash_shift)) == m_values.size());

         uint32_t hash = static_cast<uint32_t>(m_hasher(k));

         // Fibonacci hashing
         hash = (2654435769U * hash) >> m_hash_shift;

         assert(hash < m_values.size());
         return hash;
      }

      inline const node& get_node(uint32_t index) const
      {
         return *reinterpret_cast<const node*>(&m_values[index]);
      }

      inline node& get_node(uint32_t index)
      {
         return *reinterpret_cast<node*>(&m_values[index]);
      }

      inline state get_node_state(uint32_t index) const
      {
         return static_cast<state>(get_node(index).state);
      }

      inline void set_node_state(uint32_t index, bool valid)
      {
         get_node(index).state = valid;
      }

      inline void grow()
      {
         uint64_t n = m_values.size() * 3ULL; // was * 2
         
         if (!helpers::is_power_of_2(n))
            n = helpers::next_pow2(n);

         if (n > 0x80000000UL)
            n = 0x80000000UL;

         rehash(helpers::maximum<uint32_t>(cMinHashSize, (uint32_t)n));
      }

      inline void rehash(uint32_t new_hash_size)
      {
         assert(new_hash_size >= m_num_valid);
         assert(helpers::is_power_of_2(new_hash_size));

         if ((new_hash_size < m_num_valid) || (new_hash_size == m_values.size()))
            return;

         hash_map new_map;
         new_map.m_values.resize(new_hash_size);
         new_map.m_hash_shift = 32U - helpers::floor_log2i(new_hash_size);
         assert(new_hash_size == (1U << (32U - new_map.m_hash_shift)));
         new_map.m_grow_threshold = UINT_MAX;

         node* pNode = reinterpret_cast<node*>(m_values.begin());
         node* pNode_end = pNode + m_values.size();

         while (pNode != pNode_end)
         {
            if (pNode->state)
            {
               new_map.move_into(pNode);

               if (new_map.m_num_valid == m_num_valid)
                  break;
            }

            pNode++;
         }

         new_map.m_grow_threshold = (new_hash_size + 1U) >> 1U;

         m_values.clear_no_destruction();
         m_hash_shift = 32;

         swap(new_map);
      }

      inline uint32_t find_next(uint32_t index) const
      {
         index++;

         if (index >= m_values.size())
            return index;

         const node* pNode = &get_node(index);

         for (; ; )
         {
            if (pNode->state)
               break;

            if (++index >= m_values.size())
               break;

            pNode++;
         }

         return index;
      }

      inline uint32_t find_index(const Key& k) const
      {
         if (m_num_valid)
         {
            uint32_t index = hash_key(k);
            const node* pNode = &get_node(index);

            if (pNode->state)
            {
               if (m_equals(pNode->first, k))
                  return index;

               const uint32_t orig_index = index;

               for (; ; )
               {
                  if (!index)
                  {
                     index = m_values.size() - 1;
                     pNode = &get_node(index);
                  }
                  else
                  {
                     index--;
                     pNode--;
                  }

                  if (index == orig_index)
                     break;

                  if (!pNode->state)
                     break;

                  if (m_equals(pNode->first, k))
                     return index;
               }
            }
         }

         return m_values.size();
      }

      inline bool insert_no_grow(insert_result& result, const Key& k, const Value& v = Value())
      {
         if (!m_values.size())
            return false;

         uint32_t index = hash_key(k);
         node* pNode = &get_node(index);

         if (pNode->state)
         {
            if (m_equals(pNode->first, k))
            {
               result.first = iterator(*this, index);
               result.second = false;
               return true;
            }

            const uint32_t orig_index = index;

            for (; ; )
            {
               if (!index)
               {
                  index = m_values.size() - 1;
                  pNode = &get_node(index);
               }
               else
               {
                  index--;
                  pNode--;
               }

               if (orig_index == index)
                  return false;

               if (!pNode->state)
                  break;

               if (m_equals(pNode->first, k))
               {
                  result.first = iterator(*this, index);
                  result.second = false;
                  return true;
               }
            }
         }

         if (m_num_valid >= m_grow_threshold)
            return false;

         construct_value_type(pNode, k, v);

         pNode->state = cStateValid;

         m_num_valid++;
         assert(m_num_valid <= m_values.size());

         result.first = iterator(*this, index);
         result.second = true;

         return true;
      }

      inline void move_into(node* pNode)
      {
         uint32_t index = hash_key(pNode->first);
         node* pDst_node = &get_node(index);

         if (pDst_node->state)
         {
            const uint32_t orig_index = index;

            for (; ; )
            {
               if (!index)
               {
                  index = m_values.size() - 1;
                  pDst_node = &get_node(index);
               }
               else
               {
                  index--;
                  pDst_node--;
               }

               if (index == orig_index)
               {
                  assert(false);
                  return;
               }

               if (!pDst_node->state)
                  break;
            }
         }

         move_node(pDst_node, pNode, false);

         m_num_valid++;
      }
   };

   template<typename Key, typename Value, typename Hasher, typename Equals>
   struct bitwise_movable< hash_map<Key, Value, Hasher, Equals> > { enum { cFlag = true }; };
   
#if BASISU_HASHMAP_TEST
   extern void hash_map_test();
#endif
      
} // namespace basisu

namespace std
{
   template<typename T>
   inline void swap(basisu::vector<T>& a, basisu::vector<T>& b)
   {
      a.swap(b);
   }

   template<typename Key, typename Value, typename Hasher, typename Equals>
   inline void swap(basisu::hash_map<Key, Value, Hasher, Equals>& a, basisu::hash_map<Key, Value, Hasher, Equals>& b)
   {
      a.swap(b);
   }

} // namespace std
/**** ended inlining basisu_containers.h ****/

#ifdef max
#undef max
#endif

#ifdef min
#undef min
#endif

#ifdef _WIN32
#define strcasecmp _stricmp
#endif

// Set to one to enable debug printf()'s when any errors occur, for development/debugging. Especially useful for WebGL development.
#ifndef BASISU_FORCE_DEVEL_MESSAGES
#define BASISU_FORCE_DEVEL_MESSAGES 0
#endif

#define BASISU_NOTE_UNUSED(x) (void)(x)
#define BASISU_ARRAY_SIZE(x) (sizeof(x) / sizeof(x[0]))
#define BASISU_NO_EQUALS_OR_COPY_CONSTRUCT(x) x(const x &) = delete; x& operator= (const x &) = delete;
#define BASISU_ASSUME(x) static_assert(x, #x);
#define BASISU_OFFSETOF(s, m) offsetof(s, m)
#define BASISU_STRINGIZE(x) #x
#define BASISU_STRINGIZE2(x) BASISU_STRINGIZE(x)

#if BASISU_FORCE_DEVEL_MESSAGES
	#define BASISU_DEVEL_ERROR(...) do { basisu::debug_printf(__VA_ARGS__); } while(0)
#else
	#define BASISU_DEVEL_ERROR(...)
#endif

namespace basisu
{
	// Types/utilities

#ifdef _WIN32
	const char BASISU_PATH_SEPERATOR_CHAR = '\\';
#else
	const char BASISU_PATH_SEPERATOR_CHAR = '/';
#endif

	typedef basisu::vector<uint8_t> uint8_vec;
	typedef basisu::vector<int16_t> int16_vec;
	typedef basisu::vector<uint16_t> uint16_vec;
	typedef basisu::vector<uint32_t> uint_vec;
	typedef basisu::vector<uint64_t> uint64_vec;
	typedef basisu::vector<int> int_vec;
	typedef basisu::vector<bool> bool_vec;

	void enable_debug_printf(bool enabled);
	void debug_printf(const char *pFmt, ...);
		

	template <typename T> inline void clear_obj(T& obj) { memset(&obj, 0, sizeof(obj)); }

	template <typename T0, typename T1> inline T0 lerp(T0 a, T0 b, T1 c) { return a + (b - a) * c; }

	template <typename S> inline S maximum(S a, S b) { return (a > b) ? a : b; }
	template <typename S> inline S maximum(S a, S b, S c) { return maximum(maximum(a, b), c); }
	template <typename S> inline S maximum(S a, S b, S c, S d) { return maximum(maximum(maximum(a, b), c), d); }
	
	template <typename S> inline S minimum(S a, S b) {	return (a < b) ? a : b; }
	template <typename S> inline S minimum(S a, S b, S c) {	return minimum(minimum(a, b), c); }
	template <typename S> inline S minimum(S a, S b, S c, S d) { return minimum(minimum(minimum(a, b), c), d); }

	inline float clampf(float value, float low, float high) { if (value < low) value = low; else if (value > high) value = high;	return value; }
	inline float saturate(float value) { return clampf(value, 0, 1.0f); }
	inline uint8_t minimumub(uint8_t a, uint8_t b) { return (a < b) ? a : b; }
	inline uint32_t minimumu(uint32_t a, uint32_t b) { return (a < b) ? a : b; }
	inline int32_t minimumi(int32_t a, int32_t b) { return (a < b) ? a : b; }
	inline float minimumf(float a, float b) { return (a < b) ? a : b; }
	inline uint8_t maximumub(uint8_t a, uint8_t b) { return (a > b) ? a : b; }
	inline uint32_t maximumu(uint32_t a, uint32_t b) { return (a > b) ? a : b; }
	inline int32_t maximumi(int32_t a, int32_t b) { return (a > b) ? a : b; }
	inline float maximumf(float a, float b) { return (a > b) ? a : b; }
	inline int squarei(int i) { return i * i; }
	inline float squaref(float i) { return i * i; }
	template<typename T> inline T square(T a) { return a * a; }

	template <typename S> inline S clamp(S value, S low, S high) { return (value < low) ? low : ((value > high) ? high : value); }

	inline uint32_t iabs(int32_t i) { return (i < 0) ? static_cast<uint32_t>(-i) : static_cast<uint32_t>(i);	}
	inline uint64_t iabs64(int64_t i) {	return (i < 0) ? static_cast<uint64_t>(-i) : static_cast<uint64_t>(i); }

	template<typename T> inline void clear_vector(T &vec) { vec.erase(vec.begin(), vec.end()); }		
	template<typename T> inline typename T::value_type *enlarge_vector(T &vec, size_t n) { size_t cs = vec.size(); vec.resize(cs + n); return &vec[cs]; }

	inline bool is_pow2(uint32_t x) { return x && ((x & (x - 1U)) == 0U); }
	inline bool is_pow2(uint64_t x) { return x && ((x & (x - 1U)) == 0U); }

	template<typename T> inline T open_range_check(T v, T minv, T maxv) { assert(v >= minv && v < maxv); BASISU_NOTE_UNUSED(minv); BASISU_NOTE_UNUSED(maxv); return v; }
	template<typename T> inline T open_range_check(T v, T maxv) { assert(v < maxv); BASISU_NOTE_UNUSED(maxv); return v; }

	inline uint32_t total_bits(uint32_t v) { uint32_t l = 0; for ( ; v > 0U; ++l) v >>= 1; return l; }

	template<typename T> inline T saturate(T val) { return clamp(val, 0.0f, 1.0f); }

	template<typename T, typename R> inline void append_vector(T &vec, const R *pObjs, size_t n) 
	{ 
		if (n)
		{
			if (vec.size())
			{
				assert((pObjs + n) <= vec.begin() || (pObjs >= vec.end()));
			}
			const size_t cur_s = vec.size();
			vec.resize(cur_s + n);
			memcpy(&vec[cur_s], pObjs, sizeof(R) * n);
		}
	}

	template<typename T> inline void append_vector(T &vec, const T &other_vec)
	{
		assert(&vec != &other_vec);
		if (other_vec.size())
			append_vector(vec, &other_vec[0], other_vec.size());
	}

	template<typename T> inline void vector_ensure_element_is_valid(T &vec, size_t idx)
	{
		if (idx >= vec.size())
			vec.resize(idx + 1);
	}

	template<typename T> inline void vector_sort(T &vec)
	{
		if (vec.size())
			std::sort(vec.begin(), vec.end());
	}

	template<typename T, typename U> inline bool unordered_set_contains(T& set, const U&obj)
	{
		return set.find(obj) != set.end();
	}

	template<typename T> int vector_find(const T &vec, const typename T::value_type &obj)
	{
		assert(vec.size() <= INT_MAX);
		for (size_t i = 0; i < vec.size(); i++)
			if (vec[i] == obj)
				return static_cast<int>(i);
		return -1;
	}

	template<typename T> void vector_set_all(T &vec, const typename T::value_type &obj)
	{
		for (size_t i = 0; i < vec.size(); i++)
			vec[i] = obj;
	}
		
	inline uint64_t read_be64(const void *p)
	{
		uint64_t val = 0;
		for (uint32_t i = 0; i < 8; i++)
			val |= (static_cast<uint64_t>(static_cast<const uint8_t *>(p)[7 - i]) << (i * 8));
		return val;
	}

	inline void write_be64(void *p, uint64_t x)
	{
		for (uint32_t i = 0; i < 8; i++)
			static_cast<uint8_t *>(p)[7 - i] = static_cast<uint8_t>(x >> (i * 8));
	}

	static inline uint16_t byteswap16(uint16_t x) { return static_cast<uint16_t>((x << 8) | (x >> 8)); }
	static inline uint32_t byteswap32(uint32_t x) { return ((x << 24) | ((x << 8) & 0x00FF0000) | ((x >> 8) & 0x0000FF00) | (x >> 24)); }

	inline uint32_t floor_log2i(uint32_t v)
	{
		uint32_t b = 0;
		for (; v > 1U; ++b)
			v >>= 1;
		return b;
	}

	inline uint32_t ceil_log2i(uint32_t v)
	{
		uint32_t b = floor_log2i(v);
		if ((b != 32) && (v > (1U << b)))
			++b;
		return b;
	}

	inline int posmod(int x, int y)
	{
		if (x >= 0)
			return (x < y) ? x : (x % y);
		int m = (-x) % y;
		return (m != 0) ? (y - m) : m;
	}

	inline bool do_excl_ranges_overlap(int la, int ha, int lb, int hb)
	{
		assert(la < ha && lb < hb);
		if ((ha <= lb) || (la >= hb)) return false;
		return true;
	}

	static inline uint32_t read_le_dword(const uint8_t *pBytes)
	{
		return (pBytes[3] << 24U) | (pBytes[2] << 16U) | (pBytes[1] << 8U) | (pBytes[0]);
	}

	static inline void write_le_dword(uint8_t* pBytes, uint32_t val)
	{
		pBytes[0] = (uint8_t)val;
		pBytes[1] = (uint8_t)(val >> 8U);
		pBytes[2] = (uint8_t)(val >> 16U);
		pBytes[3] = (uint8_t)(val >> 24U);
	}
		
	// Always little endian 1-8 byte unsigned int
	template<uint32_t NumBytes>
	struct packed_uint
	{
		uint8_t m_bytes[NumBytes];

		inline packed_uint() { static_assert(NumBytes <= sizeof(uint64_t), "Invalid NumBytes"); }
		inline packed_uint(uint64_t v) { *this = v; }
		inline packed_uint(const packed_uint& other) { *this = other; }
						
		inline packed_uint& operator= (uint64_t v) 
		{ 
			for (uint32_t i = 0; i < NumBytes; i++) 
				m_bytes[i] = static_cast<uint8_t>(v >> (i * 8)); 
			return *this; 
		}

		inline packed_uint& operator= (const packed_uint& rhs) 
		{ 
			memcpy(m_bytes, rhs.m_bytes, sizeof(m_bytes)); 
			return *this;
		}

		inline operator uint32_t() const
		{
			switch (NumBytes)
			{
				case 1:  
				{
					return  m_bytes[0];
				}
				case 2:  
				{
					return (m_bytes[1] << 8U) | m_bytes[0];
				}
				case 3:  
				{
					return (m_bytes[2] << 16U) | (m_bytes[1] << 8U) | m_bytes[0];
				}
				case 4:  
				{
					return read_le_dword(m_bytes);
				}
				case 5:
				{
					uint32_t l = read_le_dword(m_bytes);
					uint32_t h = m_bytes[4];
					return static_cast<uint64_t>(l) | (static_cast<uint64_t>(h) << 32U);
				}
				case 6:
				{
					uint32_t l = read_le_dword(m_bytes);
					uint32_t h = (m_bytes[5] << 8U) | m_bytes[4];
					return static_cast<uint64_t>(l) | (static_cast<uint64_t>(h) << 32U);
				}
				case 7:
				{
					uint32_t l = read_le_dword(m_bytes);
					uint32_t h = (m_bytes[6] << 16U) | (m_bytes[5] << 8U) | m_bytes[4];
					return static_cast<uint64_t>(l) | (static_cast<uint64_t>(h) << 32U);
				}
				case 8:  
				{
					uint32_t l = read_le_dword(m_bytes);
					uint32_t h = read_le_dword(m_bytes + 4);
					return static_cast<uint64_t>(l) | (static_cast<uint64_t>(h) << 32U);
				}
				default: 
				{
					assert(0);
					return 0;
				}
			}
		}
	};

	enum eZero { cZero };
	enum eNoClamp { cNoClamp };
	
	// Rice/Huffman entropy coding
		
	// This is basically Deflate-style canonical Huffman, except we allow for a lot more symbols.
	enum
	{
		cHuffmanMaxSupportedCodeSize = 16, cHuffmanMaxSupportedInternalCodeSize = 31, 
		cHuffmanFastLookupBits = 10, 
		cHuffmanMaxSymsLog2 = 14, cHuffmanMaxSyms = 1 << cHuffmanMaxSymsLog2,

		// Small zero runs
		cHuffmanSmallZeroRunSizeMin = 3, cHuffmanSmallZeroRunSizeMax = 10, cHuffmanSmallZeroRunExtraBits = 3,

		// Big zero run
		cHuffmanBigZeroRunSizeMin = 11, cHuffmanBigZeroRunSizeMax = 138, cHuffmanBigZeroRunExtraBits = 7,

		// Small non-zero run
		cHuffmanSmallRepeatSizeMin = 3, cHuffmanSmallRepeatSizeMax = 6, cHuffmanSmallRepeatExtraBits = 2,

		// Big non-zero run
		cHuffmanBigRepeatSizeMin = 7, cHuffmanBigRepeatSizeMax = 134, cHuffmanBigRepeatExtraBits = 7,

		cHuffmanTotalCodelengthCodes = 21, cHuffmanSmallZeroRunCode = 17, cHuffmanBigZeroRunCode = 18, cHuffmanSmallRepeatCode = 19, cHuffmanBigRepeatCode = 20
	};

	static const uint8_t g_huffman_sorted_codelength_codes[] = { cHuffmanSmallZeroRunCode, cHuffmanBigZeroRunCode,	cHuffmanSmallRepeatCode, cHuffmanBigRepeatCode, 0, 8, 7, 9, 6, 0xA, 5, 0xB, 4, 0xC, 3, 0xD, 2, 0xE, 1, 0xF, 0x10 };
	const uint32_t cHuffmanTotalSortedCodelengthCodes = sizeof(g_huffman_sorted_codelength_codes) / sizeof(g_huffman_sorted_codelength_codes[0]);

	// GPU texture formats

	enum class texture_format
	{
		cInvalidTextureFormat = -1,
		
		// Block-based formats
		cETC1,			// ETC1
		cETC1S,			// ETC1 (subset: diff colors only, no subblocks)
		cETC2_RGB,		// ETC2 color block (basisu doesn't support ETC2 planar/T/H modes - just basic ETC1)
		cETC2_RGBA,		// ETC2 EAC alpha block followed by ETC2 color block
		cETC2_ALPHA,	// ETC2 EAC alpha block 
		cBC1,				// DXT1
		cBC3,				// DXT5 (BC4/DXT5A block followed by a BC1/DXT1 block)
		cBC4,				// DXT5A
		cBC5,				// 3DC/DXN (two BC4/DXT5A blocks)
		cBC7,
		cASTC4x4,		// LDR only
		cPVRTC1_4_RGB,
		cPVRTC1_4_RGBA,
		cATC_RGB,
		cATC_RGBA_INTERPOLATED_ALPHA,
		cFXT1_RGB,
		cPVRTC2_4_RGBA,
		cETC2_R11_EAC,
		cETC2_RG11_EAC,
		cUASTC4x4,		
		cBC1_NV,
		cBC1_AMD,
		
		// Uncompressed/raw pixels
		cRGBA32,
		cRGB565,
		cBGR565,
		cRGBA4444,
		cABGR4444
	};

	inline uint32_t get_bytes_per_block(texture_format fmt)
	{
		switch (fmt)
		{
		case texture_format::cETC1:
		case texture_format::cETC1S:
		case texture_format::cETC2_RGB:
		case texture_format::cETC2_ALPHA:
		case texture_format::cBC1:
		case texture_format::cBC1_NV:
		case texture_format::cBC1_AMD:
		case texture_format::cBC4:
		case texture_format::cPVRTC1_4_RGB:
		case texture_format::cPVRTC1_4_RGBA:
		case texture_format::cATC_RGB:
		case texture_format::cPVRTC2_4_RGBA:
		case texture_format::cETC2_R11_EAC:
			return 8;
		case texture_format::cRGBA32:
			return sizeof(uint32_t) * 16;
		default:
			break;
		}
		return 16;
	}

	inline uint32_t get_qwords_per_block(texture_format fmt)
	{
		return get_bytes_per_block(fmt) >> 3;
	}

	inline uint32_t get_block_width(texture_format fmt)
	{
		BASISU_NOTE_UNUSED(fmt);
		switch (fmt)
		{
		case texture_format::cFXT1_RGB:
			return 8;
		default:
			break;
		}
		return 4;
	}

	inline uint32_t get_block_height(texture_format fmt)
	{
		BASISU_NOTE_UNUSED(fmt);
		return 4;
	}
							
} // namespace basisu

/**** ended inlining basisu.h ****/

#define BASISD_znew (z = 36969 * (z & 65535) + (z >> 16))

namespace basisu
{
	extern bool g_debug_printf;
}

namespace basist
{
	// Low-level formats directly supported by the transcoder (other supported texture formats are combinations of these low-level block formats).
	// You probably don't care about these enum's unless you are going pretty low-level and calling the transcoder to decode individual slices.
	enum class block_format
	{
		cETC1,								// ETC1S RGB 
		cETC2_RGBA,							// full ETC2 EAC RGBA8 block
		cBC1,									// DXT1 RGB 
		cBC3,									// BC4 block followed by a four color BC1 block
		cBC4,									// DXT5A (alpha block only)
		cBC5,									// two BC4 blocks
		cPVRTC1_4_RGB,						// opaque-only PVRTC1 4bpp
		cPVRTC1_4_RGBA,					// PVRTC1 4bpp RGBA
		cBC7,									// Full BC7 block, any mode
		cBC7_M5_COLOR,						// RGB BC7 mode 5 color (writes an opaque mode 5 block)
		cBC7_M5_ALPHA,						// alpha portion of BC7 mode 5 (cBC7_M5_COLOR output data must have been written to the output buffer first to set the mode/rot fields etc.)
		cETC2_EAC_A8,						// alpha block of ETC2 EAC (first 8 bytes of the 16-bit ETC2 EAC RGBA format)
		cASTC_4x4,							// ASTC 4x4 (either color-only or color+alpha). Note that the transcoder always currently assumes sRGB is not enabled when outputting ASTC 
												// data. If you use a sRGB ASTC format you'll get ~1 LSB of additional error, because of the different way ASTC decoders scale 8-bit endpoints to 16-bits during unpacking.
		
		cATC_RGB,
		cATC_RGBA_INTERPOLATED_ALPHA,
		cFXT1_RGB,							// Opaque-only, has oddball 8x4 pixel block size

		cPVRTC2_4_RGB,
		cPVRTC2_4_RGBA,

		cETC2_EAC_R11,
		cETC2_EAC_RG11,
												
		cIndices,							// Used internally: Write 16-bit endpoint and selector indices directly to output (output block must be at least 32-bits)

		cRGB32,								// Writes RGB components to 32bpp output pixels
		cRGBA32,								// Writes RGB255 components to 32bpp output pixels
		cA32,									// Writes alpha component to 32bpp output pixels
				
		cRGB565,
		cBGR565,
		
		cRGBA4444_COLOR,
		cRGBA4444_ALPHA,
		cRGBA4444_COLOR_OPAQUE,
		cRGBA4444,
						
		cTotalBlockFormats
	};

	const int COLOR5_PAL0_PREV_HI = 9, COLOR5_PAL0_DELTA_LO = -9, COLOR5_PAL0_DELTA_HI = 31;
	const int COLOR5_PAL1_PREV_HI = 21, COLOR5_PAL1_DELTA_LO = -21, COLOR5_PAL1_DELTA_HI = 21;
	const int COLOR5_PAL2_PREV_HI = 31, COLOR5_PAL2_DELTA_LO = -31, COLOR5_PAL2_DELTA_HI = 9;
	const int COLOR5_PAL_MIN_DELTA_B_RUNLEN = 3, COLOR5_PAL_DELTA_5_RUNLEN_VLC_BITS = 3;

	const uint32_t ENDPOINT_PRED_TOTAL_SYMBOLS = (4 * 4 * 4 * 4) + 1;
	const uint32_t ENDPOINT_PRED_REPEAT_LAST_SYMBOL = ENDPOINT_PRED_TOTAL_SYMBOLS - 1;
	const uint32_t ENDPOINT_PRED_MIN_REPEAT_COUNT = 3;
	const uint32_t ENDPOINT_PRED_COUNT_VLC_BITS = 4;

	const uint32_t NUM_ENDPOINT_PREDS = 3;// BASISU_ARRAY_SIZE(g_endpoint_preds);
	const uint32_t CR_ENDPOINT_PRED_INDEX = NUM_ENDPOINT_PREDS - 1;
	const uint32_t NO_ENDPOINT_PRED_INDEX = 3;//NUM_ENDPOINT_PREDS;
	const uint32_t MAX_SELECTOR_HISTORY_BUF_SIZE = 64;
	const uint32_t SELECTOR_HISTORY_BUF_RLE_COUNT_THRESH = 3;
	const uint32_t SELECTOR_HISTORY_BUF_RLE_COUNT_BITS = 6;
	const uint32_t SELECTOR_HISTORY_BUF_RLE_COUNT_TOTAL = (1 << SELECTOR_HISTORY_BUF_RLE_COUNT_BITS);
		
	uint16_t crc16(const void *r, size_t size, uint16_t crc);
		
	class huffman_decoding_table
	{
		friend class bitwise_decoder;

	public:
		huffman_decoding_table()
		{
		}

		void clear()
		{
			basisu::clear_vector(m_code_sizes);
			basisu::clear_vector(m_lookup);
			basisu::clear_vector(m_tree);
		}

		bool init(uint32_t total_syms, const uint8_t *pCode_sizes, uint32_t fast_lookup_bits = basisu::cHuffmanFastLookupBits)
		{
			if (!total_syms)
			{
				clear();
				return true;
			}

			m_code_sizes.resize(total_syms);
			memcpy(&m_code_sizes[0], pCode_sizes, total_syms);

			const uint32_t huffman_fast_lookup_size = 1 << fast_lookup_bits;

			m_lookup.resize(0);
			m_lookup.resize(huffman_fast_lookup_size);

			m_tree.resize(0);
			m_tree.resize(total_syms * 2);

			uint32_t syms_using_codesize[basisu::cHuffmanMaxSupportedInternalCodeSize + 1];
			basisu::clear_obj(syms_using_codesize);
			for (uint32_t i = 0; i < total_syms; i++)
			{
				if (pCode_sizes[i] > basisu::cHuffmanMaxSupportedInternalCodeSize)
					return false;
				syms_using_codesize[pCode_sizes[i]]++;
			}

			uint32_t next_code[basisu::cHuffmanMaxSupportedInternalCodeSize + 1];
			next_code[0] = next_code[1] = 0;

			uint32_t used_syms = 0, total = 0;
			for (uint32_t i = 1; i < basisu::cHuffmanMaxSupportedInternalCodeSize; i++)
			{
				used_syms += syms_using_codesize[i];
				next_code[i + 1] = (total = ((total + syms_using_codesize[i]) << 1));
			}

			if (((1U << basisu::cHuffmanMaxSupportedInternalCodeSize) != total) && (used_syms > 1U))
				return false;

			for (int tree_next = -1, sym_index = 0; sym_index < (int)total_syms; ++sym_index)
			{
				uint32_t rev_code = 0, l, cur_code, code_size = pCode_sizes[sym_index];
				if (!code_size)
					continue;

				cur_code = next_code[code_size]++;

				for (l = code_size; l > 0; l--, cur_code >>= 1)
					rev_code = (rev_code << 1) | (cur_code & 1);

				if (code_size <= fast_lookup_bits)
				{
					uint32_t k = (code_size << 16) | sym_index;
					while (rev_code < huffman_fast_lookup_size)
					{
						if (m_lookup[rev_code] != 0)
						{
							// Supplied codesizes can't create a valid prefix code.
							return false;
						}

						m_lookup[rev_code] = k;
						rev_code += (1 << code_size);
					}
					continue;
				}

				int tree_cur;
				if (0 == (tree_cur = m_lookup[rev_code & (huffman_fast_lookup_size - 1)]))
				{
					const uint32_t idx = rev_code & (huffman_fast_lookup_size - 1);
					if (m_lookup[idx] != 0)
					{
						// Supplied codesizes can't create a valid prefix code.
						return false;
					}

					m_lookup[idx] = tree_next;
					tree_cur = tree_next;
					tree_next -= 2;
				}

				if (tree_cur >= 0)
				{
					// Supplied codesizes can't create a valid prefix code.
					return false;
				}

				rev_code >>= (fast_lookup_bits - 1);

				for (int j = code_size; j > ((int)fast_lookup_bits + 1); j--)
				{
					tree_cur -= ((rev_code >>= 1) & 1);

					const int idx = -tree_cur - 1;
					if (idx < 0)
						return false;
					else if (idx >= (int)m_tree.size())
						m_tree.resize(idx + 1);
										
					if (!m_tree[idx])
					{
						m_tree[idx] = (int16_t)tree_next;
						tree_cur = tree_next;
						tree_next -= 2;
					}
					else
					{
						tree_cur = m_tree[idx];
						if (tree_cur >= 0)
						{
							// Supplied codesizes can't create a valid prefix code.
							return false;
						}
					}
				}

				tree_cur -= ((rev_code >>= 1) & 1);

				const int idx = -tree_cur - 1;
				if (idx < 0)
					return false;
				else if (idx >= (int)m_tree.size())
					m_tree.resize(idx + 1);

				if (m_tree[idx] != 0)
				{
					// Supplied codesizes can't create a valid prefix code.
					return false;
				}

				m_tree[idx] = (int16_t)sym_index;
			}

			return true;
		}

		const basisu::uint8_vec &get_code_sizes() const { return m_code_sizes; }
		const basisu::int_vec get_lookup() const { return m_lookup; }
		const basisu::int16_vec get_tree() const { return m_tree; }

		bool is_valid() const { return m_code_sizes.size() > 0; }

	private:
		basisu::uint8_vec m_code_sizes;
		basisu::int_vec m_lookup;
		basisu::int16_vec m_tree;
	};

	class bitwise_decoder
	{
	public:
		bitwise_decoder() :
			m_buf_size(0),
			m_pBuf(nullptr),
			m_pBuf_start(nullptr),
			m_pBuf_end(nullptr),
			m_bit_buf(0),
			m_bit_buf_size(0)
		{
		}

		void clear()
		{
			m_buf_size = 0;
			m_pBuf = nullptr;
			m_pBuf_start = nullptr;
			m_pBuf_end = nullptr;
			m_bit_buf = 0;
			m_bit_buf_size = 0;
		}

		bool init(const uint8_t *pBuf, uint32_t buf_size)
		{
			if ((!pBuf) && (buf_size))
				return false;

			m_buf_size = buf_size;
			m_pBuf = pBuf;
			m_pBuf_start = pBuf;
			m_pBuf_end = pBuf + buf_size;
			m_bit_buf = 0;
			m_bit_buf_size = 0;
			return true;
		}

		void stop()
		{
		}

		inline uint32_t peek_bits(uint32_t num_bits)
		{
			if (!num_bits)
				return 0;

			assert(num_bits <= 25);

			while (m_bit_buf_size < num_bits)
			{
				uint32_t c = 0;
				if (m_pBuf < m_pBuf_end)
					c = *m_pBuf++;

				m_bit_buf |= (c << m_bit_buf_size);
				m_bit_buf_size += 8;
				assert(m_bit_buf_size <= 32);
			}

			return m_bit_buf & ((1 << num_bits) - 1);
		}

		void remove_bits(uint32_t num_bits)
		{
			assert(m_bit_buf_size >= num_bits);

			m_bit_buf >>= num_bits;
			m_bit_buf_size -= num_bits;
		}

		uint32_t get_bits(uint32_t num_bits)
		{
			if (num_bits > 25)
			{
				assert(num_bits <= 32);

				const uint32_t bits0 = peek_bits(25);
				m_bit_buf >>= 25;
				m_bit_buf_size -= 25;
				num_bits -= 25;

				const uint32_t bits = peek_bits(num_bits);
				m_bit_buf >>= num_bits;
				m_bit_buf_size -= num_bits;

				return bits0 | (bits << 25);
			}

			const uint32_t bits = peek_bits(num_bits);

			m_bit_buf >>= num_bits;
			m_bit_buf_size -= num_bits;

			return bits;
		}

		uint32_t decode_truncated_binary(uint32_t n)
		{
			assert(n >= 2);

			const uint32_t k = basisu::floor_log2i(n);
			const uint32_t u = (1 << (k + 1)) - n;

			uint32_t result = get_bits(k);

			if (result >= u)
				result = ((result << 1) | get_bits(1)) - u;

			return result;
		}

		uint32_t decode_rice(uint32_t m)
		{
			assert(m);

			uint32_t q = 0;
			for (;;)
			{
				uint32_t k = peek_bits(16);
				
				uint32_t l = 0;
				while (k & 1)
				{
					l++;
					k >>= 1;
				}
				
				q += l;

				remove_bits(l);

				if (l < 16)
					break;
			}

			return (q << m) + (get_bits(m + 1) >> 1);
		}

		inline uint32_t decode_vlc(uint32_t chunk_bits)
		{
			assert(chunk_bits);

			const uint32_t chunk_size = 1 << chunk_bits;
			const uint32_t chunk_mask = chunk_size - 1;
					
			uint32_t v = 0;
			uint32_t ofs = 0;

			for ( ; ; )
			{
				uint32_t s = get_bits(chunk_bits + 1);
				v |= ((s & chunk_mask) << ofs);
				ofs += chunk_bits;

				if ((s & chunk_size) == 0)
					break;
				
				if (ofs >= 32)
				{
					assert(0);
					break;
				}
			}

			return v;
		}

		inline uint32_t decode_huffman(const huffman_decoding_table &ct, int fast_lookup_bits = basisu::cHuffmanFastLookupBits)
		{
			assert(ct.m_code_sizes.size());

			const uint32_t huffman_fast_lookup_size = 1 << fast_lookup_bits;
						
			while (m_bit_buf_size < 16)
			{
				uint32_t c = 0;
				if (m_pBuf < m_pBuf_end)
					c = *m_pBuf++;

				m_bit_buf |= (c << m_bit_buf_size);
				m_bit_buf_size += 8;
				assert(m_bit_buf_size <= 32);
			}
						
			int code_len;

			int sym;
			if ((sym = ct.m_lookup[m_bit_buf & (huffman_fast_lookup_size - 1)]) >= 0)
			{
				code_len = sym >> 16;
				sym &= 0xFFFF;
			}
			else
			{
				code_len = fast_lookup_bits;
				do
				{
					sym = ct.m_tree[~sym + ((m_bit_buf >> code_len++) & 1)]; // ~sym = -sym - 1
				} while (sym < 0);
			}

			m_bit_buf >>= code_len;
			m_bit_buf_size -= code_len;

			return sym;
		}

		bool read_huffman_table(huffman_decoding_table &ct)
		{
			ct.clear();

			const uint32_t total_used_syms = get_bits(basisu::cHuffmanMaxSymsLog2);

			if (!total_used_syms)
				return true;
			if (total_used_syms > basisu::cHuffmanMaxSyms)
				return false;

			uint8_t code_length_code_sizes[basisu::cHuffmanTotalCodelengthCodes];
			basisu::clear_obj(code_length_code_sizes);

			const uint32_t num_codelength_codes = get_bits(5);
			if ((num_codelength_codes < 1) || (num_codelength_codes > basisu::cHuffmanTotalCodelengthCodes))
				return false;

			for (uint32_t i = 0; i < num_codelength_codes; i++)
				code_length_code_sizes[basisu::g_huffman_sorted_codelength_codes[i]] = static_cast<uint8_t>(get_bits(3));

			huffman_decoding_table code_length_table;
			if (!code_length_table.init(basisu::cHuffmanTotalCodelengthCodes, code_length_code_sizes))
				return false;

			if (!code_length_table.is_valid())
				return false;

			basisu::uint8_vec code_sizes(total_used_syms);

			uint32_t cur = 0;
			while (cur < total_used_syms)
			{
				int c = decode_huffman(code_length_table);

				if (c <= 16)
					code_sizes[cur++] = static_cast<uint8_t>(c);
				else if (c == basisu::cHuffmanSmallZeroRunCode)
					cur += get_bits(basisu::cHuffmanSmallZeroRunExtraBits) + basisu::cHuffmanSmallZeroRunSizeMin;
				else if (c == basisu::cHuffmanBigZeroRunCode)
					cur += get_bits(basisu::cHuffmanBigZeroRunExtraBits) + basisu::cHuffmanBigZeroRunSizeMin;
				else
				{
					if (!cur)
						return false;

					uint32_t l;
					if (c == basisu::cHuffmanSmallRepeatCode)
						l = get_bits(basisu::cHuffmanSmallRepeatExtraBits) + basisu::cHuffmanSmallRepeatSizeMin;
					else
						l = get_bits(basisu::cHuffmanBigRepeatExtraBits) + basisu::cHuffmanBigRepeatSizeMin;

					const uint8_t prev = code_sizes[cur - 1];
					if (prev == 0)
						return false;
					do
					{
						if (cur >= total_used_syms)
							return false;
						code_sizes[cur++] = prev;
					} while (--l > 0);
				}
			}

			if (cur != total_used_syms)
				return false;

			return ct.init(total_used_syms, &code_sizes[0]);
		}

	private:
		uint32_t m_buf_size;
		const uint8_t *m_pBuf;
		const uint8_t *m_pBuf_start;
		const uint8_t *m_pBuf_end;

		uint32_t m_bit_buf;
		uint32_t m_bit_buf_size;
	};

	inline uint32_t basisd_rand(uint32_t seed)
	{
		if (!seed)
			seed++;
		uint32_t z = seed;
		BASISD_znew;
		return z;
	}

	// Returns random number in [0,limit). Max limit is 0xFFFF.
	inline uint32_t basisd_urand(uint32_t& seed, uint32_t limit)
	{
		seed = basisd_rand(seed);
		return (((seed ^ (seed >> 16)) & 0xFFFF) * limit) >> 16;
	}

	class approx_move_to_front
	{
	public:
		approx_move_to_front(uint32_t n)
		{
			init(n);
		}

		void init(uint32_t n)
		{
			m_values.resize(n);
			m_rover = n / 2;
		}

		const basisu::int_vec& get_values() const { return m_values; }
		basisu::int_vec& get_values() { return m_values; }

		uint32_t size() const { return (uint32_t)m_values.size(); }

		const int& operator[] (uint32_t index) const { return m_values[index]; }
		int operator[] (uint32_t index) { return m_values[index]; }

		void add(int new_value)
		{
			m_values[m_rover++] = new_value;
			if (m_rover == m_values.size())
				m_rover = (uint32_t)m_values.size() / 2;
		}

		void use(uint32_t index)
		{
			if (index)
			{
				//std::swap(m_values[index / 2], m_values[index]);
				int x = m_values[index / 2];
				int y = m_values[index];
				m_values[index / 2] = y;
				m_values[index] = x;
			}
		}

		// returns -1 if not found
		int find(int value) const
		{
			for (uint32_t i = 0; i < m_values.size(); i++)
				if (m_values[i] == value)
					return i;
			return -1;
		}

		void reset()
		{
			const uint32_t n = (uint32_t)m_values.size();

			m_values.clear();

			init(n);
		}

	private:
		basisu::int_vec m_values;
		uint32_t m_rover;
	};

	struct decoder_etc_block;
	
	inline uint8_t clamp255(int32_t i)
	{
		return (uint8_t)((i & 0xFFFFFF00U) ? (~(i >> 31)) : i);
	}

	enum eNoClamp
	{
		cNoClamp = 0
	};

	struct color32
	{
		union
		{
			struct
			{
				uint8_t r;
				uint8_t g;
				uint8_t b;
				uint8_t a;
			};

			uint8_t c[4];
			
			uint32_t m;
		};

		color32() { }

		color32(uint32_t vr, uint32_t vg, uint32_t vb, uint32_t va) { set(vr, vg, vb, va); }
		color32(eNoClamp unused, uint32_t vr, uint32_t vg, uint32_t vb, uint32_t va) { (void)unused; set_noclamp_rgba(vr, vg, vb, va); }

		void set(uint32_t vr, uint32_t vg, uint32_t vb, uint32_t va) { c[0] = static_cast<uint8_t>(vr); c[1] = static_cast<uint8_t>(vg); c[2] = static_cast<uint8_t>(vb); c[3] = static_cast<uint8_t>(va); }

		void set_noclamp_rgb(uint32_t vr, uint32_t vg, uint32_t vb) { c[0] = static_cast<uint8_t>(vr); c[1] = static_cast<uint8_t>(vg); c[2] = static_cast<uint8_t>(vb); }
		void set_noclamp_rgba(uint32_t vr, uint32_t vg, uint32_t vb, uint32_t va) { set(vr, vg, vb, va); }

		void set_clamped(int vr, int vg, int vb, int va) { c[0] = clamp255(vr); c[1] = clamp255(vg);	c[2] = clamp255(vb); c[3] = clamp255(va); }

		uint8_t operator[] (uint32_t idx) const { assert(idx < 4); return c[idx]; }
		uint8_t &operator[] (uint32_t idx) { assert(idx < 4); return c[idx]; }

		bool operator== (const color32&rhs) const { return m == rhs.m; }

		static color32 comp_min(const color32& a, const color32& b) { return color32(cNoClamp, basisu::minimum(a[0], b[0]), basisu::minimum(a[1], b[1]), basisu::minimum(a[2], b[2]), basisu::minimum(a[3], b[3])); }
		static color32 comp_max(const color32& a, const color32& b) { return color32(cNoClamp, basisu::maximum(a[0], b[0]), basisu::maximum(a[1], b[1]), basisu::maximum(a[2], b[2]), basisu::maximum(a[3], b[3])); }
	};

	struct endpoint
	{
		color32 m_color5;
		uint8_t m_inten5;
		bool operator== (const endpoint& rhs) const
		{
			return (m_color5.r == rhs.m_color5.r) && (m_color5.g == rhs.m_color5.g) && (m_color5.b == rhs.m_color5.b) && (m_inten5 == rhs.m_inten5);
		}
		bool operator!= (const endpoint& rhs) const { return !(*this == rhs); }
	};

	struct selector
	{
		// Plain selectors (2-bits per value)
		uint8_t m_selectors[4];

		// ETC1 selectors
		uint8_t m_bytes[4];

		uint8_t m_lo_selector, m_hi_selector;
		uint8_t m_num_unique_selectors;
		bool operator== (const selector& rhs) const
		{
			return (m_selectors[0] == rhs.m_selectors[0]) &&
				(m_selectors[1] == rhs.m_selectors[1]) &&
				(m_selectors[2] == rhs.m_selectors[2]) &&
				(m_selectors[3] == rhs.m_selectors[3]);
		}
		bool operator!= (const selector& rhs) const
		{
			return !(*this == rhs);
		}

		void init_flags()
		{
			uint32_t hist[4] = { 0, 0, 0, 0 };
			for (uint32_t y = 0; y < 4; y++)
			{
				for (uint32_t x = 0; x < 4; x++)
				{
					uint32_t s = get_selector(x, y);
					hist[s]++;
				}
			}

			m_lo_selector = 3;
			m_hi_selector = 0;
			m_num_unique_selectors = 0;

			for (uint32_t i = 0; i < 4; i++)
			{
				if (hist[i])
				{
					m_num_unique_selectors++;
					if (i < m_lo_selector) m_lo_selector = static_cast<uint8_t>(i);
					if (i > m_hi_selector) m_hi_selector = static_cast<uint8_t>(i);
				}
			}
		}

		// Returned selector value ranges from 0-3 and is a direct index into g_etc1_inten_tables.
		inline uint32_t get_selector(uint32_t x, uint32_t y) const
		{
			assert((x < 4) && (y < 4));
			return (m_selectors[y] >> (x * 2)) & 3;
		}

		void set_selector(uint32_t x, uint32_t y, uint32_t val)
		{
			static const uint8_t s_selector_index_to_etc1[4] = { 3, 2, 0, 1 };

			assert((x | y | val) < 4);

			m_selectors[y] &= ~(3 << (x * 2));
			m_selectors[y] |= (val << (x * 2));

			const uint32_t etc1_bit_index = x * 4 + y;

			uint8_t *p = &m_bytes[3 - (etc1_bit_index >> 3)];

			const uint32_t byte_bit_ofs = etc1_bit_index & 7;
			const uint32_t mask = 1 << byte_bit_ofs;

			const uint32_t etc1_val = s_selector_index_to_etc1[val];

			const uint32_t lsb = etc1_val & 1;
			const uint32_t msb = etc1_val >> 1;

			p[0] &= ~mask;
			p[0] |= (lsb << byte_bit_ofs);

			p[-2] &= ~mask;
			p[-2] |= (msb << byte_bit_ofs);
		}
	};

	bool basis_block_format_is_uncompressed(block_format tex_type);
	
} // namespace basist



/**** ended inlining basisu_transcoder_internal.h ****/
/**** start inlining basisu_transcoder_uastc.h ****/
// basisu_transcoder_uastc.h
/**** skipping file: basisu_transcoder_internal.h ****/

namespace basist
{
	struct color_quad_u8
	{ 
		uint8_t m_c[4]; 
	};

	const uint32_t TOTAL_UASTC_MODES = 19;
	const uint32_t UASTC_MODE_INDEX_SOLID_COLOR = 8;

	const uint32_t TOTAL_ASTC_BC7_COMMON_PARTITIONS2 = 30;
	const uint32_t TOTAL_ASTC_BC7_COMMON_PARTITIONS3 = 11;
	const uint32_t TOTAL_BC7_3_ASTC2_COMMON_PARTITIONS = 19;

	extern const uint8_t g_uastc_mode_weight_bits[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_weight_ranges[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_endpoint_ranges[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_subsets[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_planes[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_comps[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_has_etc1_bias[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_has_bc1_hint0[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_has_bc1_hint1[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_has_alpha[TOTAL_UASTC_MODES];
	extern const uint8_t g_uastc_mode_is_la[TOTAL_UASTC_MODES];

	struct astc_bc7_common_partition2_desc
	{
		uint8_t m_bc7;
		uint16_t m_astc;
		bool m_invert;
	};

	extern const astc_bc7_common_partition2_desc g_astc_bc7_common_partitions2[TOTAL_ASTC_BC7_COMMON_PARTITIONS2];

	struct bc73_astc2_common_partition_desc
	{
		uint8_t m_bc73;
		uint16_t m_astc2;
		uint8_t k;		// 0-5 - how to modify the BC7 3-subset pattern to match the ASTC pattern (LSB=invert)
	};

	extern const bc73_astc2_common_partition_desc g_bc7_3_astc2_common_partitions[TOTAL_BC7_3_ASTC2_COMMON_PARTITIONS];

	struct astc_bc7_common_partition3_desc
	{
		uint8_t m_bc7;
		uint16_t m_astc;
		uint8_t m_astc_to_bc7_perm; // converts ASTC to BC7 partition using g_astc_bc7_partition_index_perm_tables[][]
	};

	extern const astc_bc7_common_partition3_desc g_astc_bc7_common_partitions3[TOTAL_ASTC_BC7_COMMON_PARTITIONS3];

	extern const uint8_t g_astc_bc7_patterns2[TOTAL_ASTC_BC7_COMMON_PARTITIONS2][16];
	extern const uint8_t g_astc_bc7_patterns3[TOTAL_ASTC_BC7_COMMON_PARTITIONS3][16];
	extern const uint8_t g_bc7_3_astc2_patterns2[TOTAL_BC7_3_ASTC2_COMMON_PARTITIONS][16];

	extern const uint8_t g_astc_bc7_pattern2_anchors[TOTAL_ASTC_BC7_COMMON_PARTITIONS2][3];
	extern const uint8_t g_astc_bc7_pattern3_anchors[TOTAL_ASTC_BC7_COMMON_PARTITIONS3][3];
	extern const uint8_t g_bc7_3_astc2_patterns2_anchors[TOTAL_BC7_3_ASTC2_COMMON_PARTITIONS][3];

	extern const uint32_t g_uastc_mode_huff_codes[TOTAL_UASTC_MODES + 1][2];

	extern const uint8_t g_astc_to_bc7_partition_index_perm_tables[6][3];
	extern const uint8_t g_bc7_to_astc_partition_index_perm_tables[6][3]; // inverse of g_astc_to_bc7_partition_index_perm_tables

	extern const uint8_t* s_uastc_to_bc1_weights[6];

	uint32_t bc7_convert_partition_index_3_to_2(uint32_t p, uint32_t k);

	inline uint32_t astc_interpolate(uint32_t l, uint32_t h, uint32_t w, bool srgb)
	{
		if (srgb)
		{
			l = (l << 8) | 0x80;
			h = (h << 8) | 0x80;
		}
		else
		{
			l = (l << 8) | l;
			h = (h << 8) | h;
		}

		uint32_t k = (l * (64 - w) + h * w + 32) >> 6;

		return k >> 8;
	}

	struct astc_block_desc
	{
		int m_weight_range;	// weight BISE range

		int m_subsets;			// number of ASTC partitions
		int m_partition_seed;	// partition pattern seed
		int m_cem;				// color endpoint mode used by all subsets

		int m_ccs;				// color component selector (dual plane only)
		bool m_dual_plane;	// true if dual plane

		// Weight and endpoint BISE values. 
		// Note these values are NOT linear, they must be BISE encoded. See Table 97 and Table 107.
		uint8_t m_endpoints[18];	// endpoint values, in RR GG BB etc. order 
		uint8_t m_weights[64];		// weight index values, raster order, in P0 P1, P0 P1, etc. or P0, P0, P0, P0, etc. order
	};

	const uint32_t BC7ENC_TOTAL_ASTC_RANGES = 21;

	// See tables 81, 93, 18.13.Endpoint Unquantization
	const uint32_t TOTAL_ASTC_RANGES = 21;
	extern const int g_astc_bise_range_table[TOTAL_ASTC_RANGES][3];

	struct astc_quant_bin
	{
		uint8_t m_unquant; // unquantized value
		uint8_t m_index; // sorted index
	};

	extern astc_quant_bin g_astc_unquant[BC7ENC_TOTAL_ASTC_RANGES][256]; // [ASTC encoded endpoint index]

	int astc_get_levels(int range);
	bool astc_is_valid_endpoint_range(uint32_t range);
	uint32_t unquant_astc_endpoint(uint32_t packed_bits, uint32_t packed_trits, uint32_t packed_quints, uint32_t range);
	uint32_t unquant_astc_endpoint_val(uint32_t packed_val, uint32_t range);

	const uint8_t* get_anchor_indices(uint32_t subsets, uint32_t mode, uint32_t common_pattern, const uint8_t*& pPartition_pattern);

	// BC7
	const uint32_t BC7ENC_BLOCK_SIZE = 16;

	struct bc7_block
	{
		uint64_t m_qwords[2];
	};

	struct bc7_optimization_results
	{
		uint32_t m_mode;
		uint32_t m_partition;
		uint8_t m_selectors[16];
		uint8_t m_alpha_selectors[16];
		color_quad_u8 m_low[3];
		color_quad_u8 m_high[3];
		uint32_t m_pbits[3][2];
		uint32_t m_index_selector;
		uint32_t m_rotation;
	};

	extern const uint32_t g_bc7_weights1[2];
	extern const uint32_t g_bc7_weights2[4];
	extern const uint32_t g_bc7_weights3[8];
	extern const uint32_t g_bc7_weights4[16];
	extern const uint32_t g_astc_weights4[16];
	extern const uint32_t g_astc_weights5[32];
	extern const uint32_t g_astc_weights_3levels[3];
	extern const uint8_t g_bc7_partition1[16];
	extern const uint8_t g_bc7_partition2[64 * 16];
	extern const uint8_t g_bc7_partition3[64 * 16];
	extern const uint8_t g_bc7_table_anchor_index_second_subset[64];
	extern const uint8_t g_bc7_table_anchor_index_third_subset_1[64];
	extern const uint8_t g_bc7_table_anchor_index_third_subset_2[64];
	extern const uint8_t g_bc7_num_subsets[8];
	extern const uint8_t g_bc7_partition_bits[8];
	extern const uint8_t g_bc7_color_index_bitcount[8];
	extern const uint8_t g_bc7_mode_has_p_bits[8];
	extern const uint8_t g_bc7_mode_has_shared_p_bits[8];
	extern const uint8_t g_bc7_color_precision_table[8];
	extern const int8_t g_bc7_alpha_precision_table[8];
	extern const uint8_t g_bc7_alpha_index_bitcount[8];

	inline bool get_bc7_mode_has_seperate_alpha_selectors(int mode) { return (mode == 4) || (mode == 5); }
	inline int get_bc7_color_index_size(int mode, int index_selection_bit) { return g_bc7_color_index_bitcount[mode] + index_selection_bit; }
	inline int get_bc7_alpha_index_size(int mode, int index_selection_bit) { return g_bc7_alpha_index_bitcount[mode] - index_selection_bit; }

	struct endpoint_err
	{
		uint16_t m_error; uint8_t m_lo; uint8_t m_hi;
	};

	extern endpoint_err g_bc7_mode_6_optimal_endpoints[256][2]; // [c][pbit]
	const uint32_t BC7ENC_MODE_6_OPTIMAL_INDEX = 5;

	extern endpoint_err g_bc7_mode_5_optimal_endpoints[256]; // [c]
	const uint32_t BC7ENC_MODE_5_OPTIMAL_INDEX = 1;

	// Packs a BC7 block from a high-level description. Handles all BC7 modes.
	void encode_bc7_block(void* pBlock, const bc7_optimization_results* pResults);

	// Packs an ASTC block
	// Constraints: Always 4x4, all subset CEM's must be equal, only tested with LDR CEM's.
	bool pack_astc_block(uint32_t* pDst, const astc_block_desc* pBlock, uint32_t mode);

	void pack_astc_solid_block(void* pDst_block, const color32& color);

#ifdef _DEBUG
	int astc_compute_texel_partition(int seed, int x, int y, int z, int partitioncount, bool small_block);
#endif
		
	struct uastc_block
	{
		union
		{
			uint8_t m_bytes[16];
			uint32_t m_dwords[4];

#ifndef __EMSCRIPTEN__
			uint64_t m_qwords[2];
#endif
		};
	};

	struct unpacked_uastc_block
	{
		astc_block_desc m_astc;

		uint32_t m_mode;
		uint32_t m_common_pattern;

		color32 m_solid_color;

		bool m_bc1_hint0;
		bool m_bc1_hint1;

		bool m_etc1_flip;
		bool m_etc1_diff;
		uint32_t m_etc1_inten0;
		uint32_t m_etc1_inten1;

		uint32_t m_etc1_bias;

		uint32_t m_etc2_hints;

		uint32_t m_etc1_selector;
		uint32_t m_etc1_r, m_etc1_g, m_etc1_b;
	};

	color32 apply_etc1_bias(const color32 &block_color, uint32_t bias, uint32_t limit, uint32_t subblock);
	
	struct decoder_etc_block;
	struct eac_block;
		
	bool unpack_uastc(uint32_t mode, uint32_t common_pattern, const color32& solid_color, const astc_block_desc& astc, color32* pPixels, bool srgb);
	bool unpack_uastc(const unpacked_uastc_block& unpacked_blk, color32* pPixels, bool srgb);

	bool unpack_uastc(const uastc_block& blk, color32* pPixels, bool srgb);
	bool unpack_uastc(const uastc_block& blk, unpacked_uastc_block& unpacked, bool undo_blue_contract, bool read_hints = true);

	bool transcode_uastc_to_astc(const uastc_block& src_blk, void* pDst);

	bool transcode_uastc_to_bc7(const unpacked_uastc_block& unpacked_src_blk, bc7_optimization_results& dst_blk);
	bool transcode_uastc_to_bc7(const uastc_block& src_blk, bc7_optimization_results& dst_blk);
	bool transcode_uastc_to_bc7(const uastc_block& src_blk, void* pDst);

	void transcode_uastc_to_etc1(unpacked_uastc_block& unpacked_src_blk, color32 block_pixels[4][4], void* pDst);
	bool transcode_uastc_to_etc1(const uastc_block& src_blk, void* pDst);
	bool transcode_uastc_to_etc1(const uastc_block& src_blk, void* pDst, uint32_t channel);

	void transcode_uastc_to_etc2_eac_a8(unpacked_uastc_block& unpacked_src_blk, color32 block_pixels[4][4], void* pDst);
	bool transcode_uastc_to_etc2_rgba(const uastc_block& src_blk, void* pDst);

	// Packs 16 scalar values to BC4. Same PSNR as stb_dxt's BC4 encoder, around 13% faster.
	void encode_bc4(void* pDst, const uint8_t* pPixels, uint32_t stride);
	
	void encode_bc1_solid_block(void* pDst, uint32_t fr, uint32_t fg, uint32_t fb);

	enum
	{
		cEncodeBC1HighQuality = 1,
		cEncodeBC1HigherQuality = 2,
		cEncodeBC1UseSelectors = 4,
	};
	void encode_bc1(void* pDst, const uint8_t* pPixels, uint32_t flags);
	
	// Alternate PCA-free encoder, around 15% faster, same (or slightly higher) avg. PSNR
	void encode_bc1_alt(void* pDst, const uint8_t* pPixels, uint32_t flags);

	void transcode_uastc_to_bc1_hint0(const unpacked_uastc_block& unpacked_src_blk, void* pDst);
	void transcode_uastc_to_bc1_hint1(const unpacked_uastc_block& unpacked_src_blk, const color32 block_pixels[4][4], void* pDst, bool high_quality);

	bool transcode_uastc_to_bc1(const uastc_block& src_blk, void* pDst, bool high_quality);
	bool transcode_uastc_to_bc3(const uastc_block& src_blk, void* pDst, bool high_quality);
	bool transcode_uastc_to_bc4(const uastc_block& src_blk, void* pDst, bool high_quality, uint32_t chan0);
	bool transcode_uastc_to_bc5(const uastc_block& src_blk, void* pDst, bool high_quality, uint32_t chan0, uint32_t chan1);

	bool transcode_uastc_to_etc2_eac_r11(const uastc_block& src_blk, void* pDst, bool high_quality, uint32_t chan0);
	bool transcode_uastc_to_etc2_eac_rg11(const uastc_block& src_blk, void* pDst, bool high_quality, uint32_t chan0, uint32_t chan1);

	bool transcode_uastc_to_pvrtc1_4_rgb(const uastc_block* pSrc_blocks, void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, bool high_quality, bool from_alpha);
	bool transcode_uastc_to_pvrtc1_4_rgba(const uastc_block* pSrc_blocks, void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, bool high_quality);
		
	// uastc_init() MUST be called before using this module.
	void uastc_init();

} // namespace basist
/**** ended inlining basisu_transcoder_uastc.h ****/
/**** start inlining basisu_file_headers.h ****/
// basis_file_headers.h
// Copyright (C) 2019-2020 Binomial LLC. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
/**** skipping file: basisu_transcoder_internal.h ****/

namespace basist
{
	// Slice desc header flags
	enum basis_slice_desc_flags
	{
		cSliceDescFlagsHasAlpha = 1,
		
		// Video only: Frame doesn't refer to previous frame (no usage of conditional replenishment pred symbols)
		// Currently the first frame is always an I-Frame, all subsequent frames are P-Frames. This will eventually be changed to periodic I-Frames.
		cSliceDescFlagsFrameIsIFrame = 2			
	};

#pragma pack(push)
#pragma pack(1)
	struct basis_slice_desc
	{
		basisu::packed_uint<3> m_image_index;  // The index of the source image provided to the encoder (will always appear in order from first to last, first image index is 0, no skipping allowed)
		basisu::packed_uint<1> m_level_index;	// The mipmap level index (mipmaps will always appear from largest to smallest)
		basisu::packed_uint<1> m_flags;			// enum basis_slice_desc_flags

		basisu::packed_uint<2> m_orig_width;	// The original image width (may not be a multiple of 4 pixels)
		basisu::packed_uint<2> m_orig_height;  // The original image height (may not be a multiple of 4 pixels)

		basisu::packed_uint<2> m_num_blocks_x;	// The slice's block X dimensions. Each block is 4x4 pixels. The slice's pixel resolution may or may not be a power of 2.
		basisu::packed_uint<2> m_num_blocks_y;	// The slice's block Y dimensions. 

		basisu::packed_uint<4> m_file_ofs;		// Offset from the start of the file to the start of the slice's data
		basisu::packed_uint<4> m_file_size;		// The size of the compressed slice data in bytes

		basisu::packed_uint<2> m_slice_data_crc16; // The CRC16 of the compressed slice data, for extra-paranoid use cases
	};

	// File header files
	enum basis_header_flags
	{
		// Always set for ETC1S files. Not set for UASTC files.
		cBASISHeaderFlagETC1S = 1,					 
		
		// Set if the texture had to be Y flipped before encoding. The actual interpretation of this (is Y up or down?) is up to the user.
		cBASISHeaderFlagYFlipped = 2,				 
		
		// Set if any slices contain alpha (for ETC1S, if the odd slices contain alpha data)
		cBASISHeaderFlagHasAlphaSlices = 4,		 
		
		// For ETC1S files, this will be true if the file utilizes a codebook from another .basis file. 
		cBASISHeaderFlagUsesGlobalCodebook = 8, 
		
		// Set if the texture data is sRGB, otherwise it's linear. 
		// In reality, we have no idea if the texture data is actually linear or sRGB. This is the m_perceptual parameter passed to the compressor.
		cBASISHeaderFlagSRGB = 16,					 
	};

	// The image type field attempts to describe how to interpret the image data in a Basis file.
	// The encoder library doesn't really do anything special or different with these texture types, this is mostly here for the benefit of the user. 
	// We do make sure the various constraints are followed (2DArray/cubemap/videoframes/volume implies that each image has the same resolution and # of mipmap levels, etc., cubemap implies that the # of image slices is a multiple of 6)
	enum basis_texture_type
	{
		cBASISTexType2D = 0,					// An arbitrary array of 2D RGB or RGBA images with optional mipmaps, array size = # images, each image may have a different resolution and # of mipmap levels
		cBASISTexType2DArray = 1,			// An array of 2D RGB or RGBA images with optional mipmaps, array size = # images, each image has the same resolution and mipmap levels
		cBASISTexTypeCubemapArray = 2,	// an array of cubemap levels, total # of images must be divisable by 6, in X+, X-, Y+, Y-, Z+, Z- order, with optional mipmaps
		cBASISTexTypeVideoFrames = 3,		// An array of 2D video frames, with optional mipmaps, # frames = # images, each image has the same resolution and # of mipmap levels
		cBASISTexTypeVolume = 4,			// A 3D texture with optional mipmaps, Z dimension = # images, each image has the same resolution and # of mipmap levels

		cBASISTexTypeTotal
	};

	enum
	{
		cBASISMaxUSPerFrame = 0xFFFFFF
	};

	enum class basis_tex_format
	{
		cETC1S = 0,
		cUASTC4x4 = 1
	};

	struct basis_file_header
	{
		enum
		{
			cBASISSigValue = ('B' << 8) | 's',
			cBASISFirstVersion = 0x10
		};

		basisu::packed_uint<2>      m_sig;				// 2 byte file signature
		basisu::packed_uint<2>      m_ver;				// Baseline file version
		basisu::packed_uint<2>      m_header_size;	// Header size in bytes, sizeof(basis_file_header)
		basisu::packed_uint<2>      m_header_crc16;	// CRC16 of the remaining header data

		basisu::packed_uint<4>      m_data_size;		// The total size of all data after the header
		basisu::packed_uint<2>      m_data_crc16;		// The CRC16 of all data after the header

		basisu::packed_uint<3>      m_total_slices;	// The total # of compressed slices (1 slice per image, or 2 for alpha .basis files)

		basisu::packed_uint<3>      m_total_images;	// The total # of images
				
		basisu::packed_uint<1>      m_tex_format;		// enum basis_tex_format
		basisu::packed_uint<2>      m_flags;			// enum basist::header_flags
		basisu::packed_uint<1>      m_tex_type;		// enum basist::basis_texture_type
		basisu::packed_uint<3>      m_us_per_frame;	// Framerate of video, in microseconds per frame

		basisu::packed_uint<4>      m_reserved;		// For future use
		basisu::packed_uint<4>      m_userdata0;		// For client use
		basisu::packed_uint<4>      m_userdata1;		// For client use

		basisu::packed_uint<2>      m_total_endpoints;			// The number of endpoints in the endpoint codebook 
		basisu::packed_uint<4>      m_endpoint_cb_file_ofs;	// The compressed endpoint codebook's file offset relative to the start of the file
		basisu::packed_uint<3>      m_endpoint_cb_file_size;	// The compressed endpoint codebook's size in bytes

		basisu::packed_uint<2>      m_total_selectors;			// The number of selectors in the endpoint codebook 
		basisu::packed_uint<4>      m_selector_cb_file_ofs;	// The compressed selectors codebook's file offset relative to the start of the file
		basisu::packed_uint<3>      m_selector_cb_file_size;	// The compressed selector codebook's size in bytes

		basisu::packed_uint<4>      m_tables_file_ofs;			// The file offset of the compressed Huffman codelength tables, for decompressing slices
		basisu::packed_uint<4>      m_tables_file_size;			// The file size in bytes of the compressed huffman codelength tables

		basisu::packed_uint<4>      m_slice_desc_file_ofs;		// The file offset to the slice description array, usually follows the header
		
		basisu::packed_uint<4>      m_extended_file_ofs;		// The file offset of the "extended" header and compressed data, for future use
		basisu::packed_uint<4>      m_extended_file_size;		// The file size in bytes of the "extended" header and compressed data, for future use
	};
#pragma pack (pop)

} // namespace basist
/**** ended inlining basisu_file_headers.h ****/

namespace basist
{
	// High-level composite texture formats supported by the transcoder.
	// Each of these texture formats directly correspond to OpenGL/D3D/Vulkan etc. texture formats.
	// Notes:
	// - If you specify a texture format that supports alpha, but the .basis file doesn't have alpha, the transcoder will automatically output a 
	// fully opaque (255) alpha channel.
	// - The PVRTC1 texture formats only support power of 2 dimension .basis files, but this may be relaxed in a future version.
	// - The PVRTC1 transcoders are real-time encoders, so don't expect the highest quality. We may add a slower encoder with improved quality.
	// - These enums must be kept in sync with Javascript code that calls the transcoder.
	enum class transcoder_texture_format
	{
		// Compressed formats

		// ETC1-2
		cTFETC1_RGB = 0,							// Opaque only, returns RGB or alpha data if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified
		cTFETC2_RGBA = 1,							// Opaque+alpha, ETC2_EAC_A8 block followed by a ETC1 block, alpha channel will be opaque for opaque .basis files

		// BC1-5, BC7 (desktop, some mobile devices)
		cTFBC1_RGB = 2,							// Opaque only, no punchthrough alpha support yet, transcodes alpha slice if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified
		cTFBC3_RGBA = 3, 							// Opaque+alpha, BC4 followed by a BC1 block, alpha channel will be opaque for opaque .basis files
		cTFBC4_R = 4,								// Red only, alpha slice is transcoded to output if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified
		cTFBC5_RG = 5,								// XY: Two BC4 blocks, X=R and Y=Alpha, .basis file should have alpha data (if not Y will be all 255's)
		cTFBC7_RGBA = 6,							// RGB or RGBA, mode 5 for ETC1S, modes (1,2,3,5,6,7) for UASTC

		// PVRTC1 4bpp (mobile, PowerVR devices)
		cTFPVRTC1_4_RGB = 8,						// Opaque only, RGB or alpha if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified, nearly lowest quality of any texture format.
		cTFPVRTC1_4_RGBA = 9,					// Opaque+alpha, most useful for simple opacity maps. If .basis file doesn't have alpha cTFPVRTC1_4_RGB will be used instead. Lowest quality of any supported texture format.

		// ASTC (mobile, Intel devices, hopefully all desktop GPU's one day)
		cTFASTC_4x4_RGBA = 10,					// Opaque+alpha, ASTC 4x4, alpha channel will be opaque for opaque .basis files. Transcoder uses RGB/RGBA/L/LA modes, void extent, and up to two ([0,47] and [0,255]) endpoint precisions.

		// ATC (mobile, Adreno devices, this is a niche format)
		cTFATC_RGB = 11,							// Opaque, RGB or alpha if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified. ATI ATC (GL_ATC_RGB_AMD)
		cTFATC_RGBA = 12,							// Opaque+alpha, alpha channel will be opaque for opaque .basis files. ATI ATC (GL_ATC_RGBA_INTERPOLATED_ALPHA_AMD) 

		// FXT1 (desktop, Intel devices, this is a super obscure format)
		cTFFXT1_RGB = 17,							// Opaque only, uses exclusively CC_MIXED blocks. Notable for having a 8x4 block size. GL_3DFX_texture_compression_FXT1 is supported on Intel integrated GPU's (such as HD 630).
														// Punch-through alpha is relatively easy to support, but full alpha is harder. This format is only here for completeness so opaque-only is fine for now.
														// See the BASISU_USE_ORIGINAL_3DFX_FXT1_ENCODING macro in basisu_transcoder_internal.h.

		cTFPVRTC2_4_RGB = 18,					// Opaque-only, almost BC1 quality, much faster to transcode and supports arbitrary texture dimensions (unlike PVRTC1 RGB).
		cTFPVRTC2_4_RGBA = 19,					// Opaque+alpha, slower to encode than cTFPVRTC2_4_RGB. Premultiplied alpha is highly recommended, otherwise the color channel can leak into the alpha channel on transparent blocks.

		cTFETC2_EAC_R11 = 20,					// R only (ETC2 EAC R11 unsigned)
		cTFETC2_EAC_RG11 = 21,					// RG only (ETC2 EAC RG11 unsigned), R=opaque.r, G=alpha - for tangent space normal maps

		// Uncompressed (raw pixel) formats
		cTFRGBA32 = 13,							// 32bpp RGBA image stored in raster (not block) order in memory, R is first byte, A is last byte.
		cTFRGB565 = 14,							// 16bpp RGB image stored in raster (not block) order in memory, R at bit position 11
		cTFBGR565 = 15,							// 16bpp RGB image stored in raster (not block) order in memory, R at bit position 0
		cTFRGBA4444 = 16,							// 16bpp RGBA image stored in raster (not block) order in memory, R at bit position 12, A at bit position 0

		cTFTotalTextureFormats = 22,

		// Old enums for compatibility with code compiled against previous versions
		cTFETC1 = cTFETC1_RGB,
		cTFETC2 = cTFETC2_RGBA,
		cTFBC1 = cTFBC1_RGB,
		cTFBC3 = cTFBC3_RGBA,
		cTFBC4 = cTFBC4_R,
		cTFBC5 = cTFBC5_RG,

		// Previously, the caller had some control over which BC7 mode the transcoder output. We've simplified this due to UASTC, which supports numerous modes.
		cTFBC7_M6_RGB = cTFBC7_RGBA,			// Opaque only, RGB or alpha if cDecodeFlagsTranscodeAlphaDataToOpaqueFormats flag is specified. Highest quality of all the non-ETC1 formats.
		cTFBC7_M5_RGBA = cTFBC7_RGBA,			// Opaque+alpha, alpha channel will be opaque for opaque .basis files
		cTFBC7_M6_OPAQUE_ONLY = cTFBC7_RGBA,
		cTFBC7_M5 = cTFBC7_RGBA,
		cTFBC7_ALT = 7,

		cTFASTC_4x4 = cTFASTC_4x4_RGBA,

		cTFATC_RGBA_INTERPOLATED_ALPHA = cTFATC_RGBA,
	};

	// For compressed texture formats, this returns the # of bytes per block. For uncompressed, it returns the # of bytes per pixel.
	// NOTE: Previously, this function was called basis_get_bytes_per_block(), and it always returned 16*bytes_per_pixel for uncompressed formats which was confusing.
	uint32_t basis_get_bytes_per_block_or_pixel(transcoder_texture_format fmt);

	// Returns format's name in ASCII
	const char* basis_get_format_name(transcoder_texture_format fmt);

	// Returns block format name in ASCII
	const char* basis_get_block_format_name(block_format fmt);

	// Returns true if the format supports an alpha channel.
	bool basis_transcoder_format_has_alpha(transcoder_texture_format fmt);

	// Returns the basisu::texture_format corresponding to the specified transcoder_texture_format.
	basisu::texture_format basis_get_basisu_texture_format(transcoder_texture_format fmt);

	// Returns the texture type's name in ASCII.
	const char* basis_get_texture_type_name(basis_texture_type tex_type);

	// Returns true if the transcoder texture type is an uncompressed (raw pixel) format.
	bool basis_transcoder_format_is_uncompressed(transcoder_texture_format tex_type);

	// Returns the # of bytes per pixel for uncompressed formats, or 0 for block texture formats.
	uint32_t basis_get_uncompressed_bytes_per_pixel(transcoder_texture_format fmt);

	// Returns the block width for the specified texture format, which is currently either 4 or 8 for FXT1.
	uint32_t basis_get_block_width(transcoder_texture_format tex_type);

	// Returns the block height for the specified texture format, which is currently always 4.
	uint32_t basis_get_block_height(transcoder_texture_format tex_type);

	// Returns true if the specified format was enabled at compile time.
	bool basis_is_format_supported(transcoder_texture_format tex_type, basis_tex_format fmt = basis_tex_format::cETC1S);

	// Validates that the output buffer is large enough to hold the entire transcoded texture.
	// For uncompressed texture formats, most input parameters are in pixels, not blocks. Blocks are 4x4 pixels.
	bool basis_validate_output_buffer_size(transcoder_texture_format target_format,
		uint32_t output_blocks_buf_size_in_blocks_or_pixels,
		uint32_t orig_width, uint32_t orig_height,
		uint32_t output_row_pitch_in_blocks_or_pixels,
		uint32_t output_rows_in_pixels,
		uint32_t total_slice_blocks);

	class basisu_transcoder;

	// This struct holds all state used during transcoding. For video, it needs to persist between image transcodes (it holds the previous frame).
	// For threading you can use one state per thread.
	struct basisu_transcoder_state
	{
		struct block_preds
		{
			uint16_t m_endpoint_index;
			uint8_t m_pred_bits;
		};

		basisu::vector<block_preds> m_block_endpoint_preds[2];

		enum { cMaxPrevFrameLevels = 16 };
		basisu::vector<uint32_t> m_prev_frame_indices[2][cMaxPrevFrameLevels]; // [alpha_flag][level_index] 

		void clear()
		{
			for (uint32_t i = 0; i < 2; i++)
			{
				m_block_endpoint_preds[i].clear();

				for (uint32_t j = 0; j < cMaxPrevFrameLevels; j++)
					m_prev_frame_indices[i][j].clear();
			}
		}
	};

	// Low-level helper class that does the actual transcoding.
	class basisu_lowlevel_etc1s_transcoder
	{
		friend class basisu_transcoder;

	public:
		basisu_lowlevel_etc1s_transcoder();

		void set_global_codebooks(const basisu_lowlevel_etc1s_transcoder* pGlobal_codebook) { m_pGlobal_codebook = pGlobal_codebook; }
		const basisu_lowlevel_etc1s_transcoder* get_global_codebooks() const { return m_pGlobal_codebook; }

		bool decode_palettes(
			uint32_t num_endpoints, const uint8_t* pEndpoints_data, uint32_t endpoints_data_size,
			uint32_t num_selectors, const uint8_t* pSelectors_data, uint32_t selectors_data_size);

		bool decode_tables(const uint8_t* pTable_data, uint32_t table_data_size);

		bool transcode_slice(void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, const uint8_t* pImage_data, uint32_t image_data_size, block_format fmt,
			uint32_t output_block_or_pixel_stride_in_bytes, bool bc1_allow_threecolor_blocks, const bool is_video, const bool is_alpha_slice, const uint32_t level_index, const uint32_t orig_width, const uint32_t orig_height, uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr, bool astc_transcode_alpha = false, void* pAlpha_blocks = nullptr, uint32_t output_rows_in_pixels = 0);

		bool transcode_slice(void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, const uint8_t* pImage_data, uint32_t image_data_size, block_format fmt,
			uint32_t output_block_or_pixel_stride_in_bytes, bool bc1_allow_threecolor_blocks, const basis_file_header& header, const basis_slice_desc& slice_desc, uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr, bool astc_transcode_alpha = false, void* pAlpha_blocks = nullptr, uint32_t output_rows_in_pixels = 0)
		{
			return transcode_slice(pDst_blocks, num_blocks_x, num_blocks_y, pImage_data, image_data_size, fmt, output_block_or_pixel_stride_in_bytes, bc1_allow_threecolor_blocks,
				header.m_tex_type == cBASISTexTypeVideoFrames, (slice_desc.m_flags & cSliceDescFlagsHasAlpha) != 0, slice_desc.m_level_index,
				slice_desc.m_orig_width, slice_desc.m_orig_height, output_row_pitch_in_blocks_or_pixels, pState,
				astc_transcode_alpha,
				pAlpha_blocks,
				output_rows_in_pixels);
		}

		// Container independent transcoding
		bool transcode_image(
			transcoder_texture_format target_format,
			void* pOutput_blocks, uint32_t output_blocks_buf_size_in_blocks_or_pixels,
			const uint8_t* pCompressed_data, uint32_t compressed_data_length,
			uint32_t num_blocks_x, uint32_t num_blocks_y, uint32_t orig_width, uint32_t orig_height, uint32_t level_index,
			uint32_t rgb_offset, uint32_t rgb_length, uint32_t alpha_offset, uint32_t alpha_length,
			uint32_t decode_flags = 0,
			bool basis_file_has_alpha_slices = false,
			bool is_video = false,
			uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr,
			uint32_t output_rows_in_pixels = 0);

		void clear()
		{
			m_local_endpoints.clear();
			m_local_selectors.clear();
			m_endpoint_pred_model.clear();
			m_delta_endpoint_model.clear();
			m_selector_model.clear();
			m_selector_history_buf_rle_model.clear();
			m_selector_history_buf_size = 0;
		}

		// Low-level methods
		typedef basisu::vector<endpoint> endpoint_vec;
		const endpoint_vec& get_endpoints() const { return m_local_endpoints; }

		typedef basisu::vector<selector> selector_vec;
		const selector_vec& get_selectors() const { return m_local_selectors; }
				
	private:
		const basisu_lowlevel_etc1s_transcoder* m_pGlobal_codebook;

		endpoint_vec m_local_endpoints;
		selector_vec m_local_selectors;
				
		huffman_decoding_table m_endpoint_pred_model, m_delta_endpoint_model, m_selector_model, m_selector_history_buf_rle_model;

		uint32_t m_selector_history_buf_size;

		basisu_transcoder_state m_def_state;
	};

	enum basisu_decode_flags
	{
		// PVRTC1: decode non-pow2 ETC1S texture level to the next larger power of 2 (not implemented yet, but we're going to support it). Ignored if the slice's dimensions are already a power of 2.
		cDecodeFlagsPVRTCDecodeToNextPow2 = 2,

		// When decoding to an opaque texture format, if the basis file has alpha, decode the alpha slice instead of the color slice to the output texture format.
		// This is primarily to allow decoding of textures with alpha to multiple ETC1 textures (one for color, another for alpha).
		cDecodeFlagsTranscodeAlphaDataToOpaqueFormats = 4,

		// Forbid usage of BC1 3 color blocks (we don't support BC1 punchthrough alpha yet).
		// This flag is used internally when decoding to BC3.
		cDecodeFlagsBC1ForbidThreeColorBlocks = 8,

		// The output buffer contains alpha endpoint/selector indices. 
		// Used internally when decoding formats like ASTC that require both color and alpha data to be available when transcoding to the output format.
		cDecodeFlagsOutputHasAlphaIndices = 16,

		cDecodeFlagsHighQuality = 32
	};

	class basisu_lowlevel_uastc_transcoder
	{
		friend class basisu_transcoder;

	public:
		basisu_lowlevel_uastc_transcoder();

		bool transcode_slice(void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, const uint8_t* pImage_data, uint32_t image_data_size, block_format fmt,
			uint32_t output_block_or_pixel_stride_in_bytes, bool bc1_allow_threecolor_blocks, bool has_alpha, const uint32_t orig_width, const uint32_t orig_height, uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr, uint32_t output_rows_in_pixels = 0, int channel0 = -1, int channel1 = -1, uint32_t decode_flags = 0);

		bool transcode_slice(void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, const uint8_t* pImage_data, uint32_t image_data_size, block_format fmt,
			uint32_t output_block_or_pixel_stride_in_bytes, bool bc1_allow_threecolor_blocks, const basis_file_header& header, const basis_slice_desc& slice_desc, uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr, uint32_t output_rows_in_pixels = 0, int channel0 = -1, int channel1 = -1, uint32_t decode_flags = 0)
		{
			return transcode_slice(pDst_blocks, num_blocks_x, num_blocks_y, pImage_data, image_data_size, fmt,
				output_block_or_pixel_stride_in_bytes, bc1_allow_threecolor_blocks, (header.m_flags & cBASISHeaderFlagHasAlphaSlices) != 0, slice_desc.m_orig_width, slice_desc.m_orig_height, output_row_pitch_in_blocks_or_pixels,
				pState, output_rows_in_pixels, channel0, channel1, decode_flags);
		}

		// Container independent transcoding
		bool transcode_image(
			transcoder_texture_format target_format,
			void* pOutput_blocks, uint32_t output_blocks_buf_size_in_blocks_or_pixels,
			const uint8_t* pCompressed_data, uint32_t compressed_data_length,
			uint32_t num_blocks_x, uint32_t num_blocks_y, uint32_t orig_width, uint32_t orig_height, uint32_t level_index,
			uint32_t slice_offset, uint32_t slice_length,
			uint32_t decode_flags = 0,
			bool has_alpha = false,
			bool is_video = false,
			uint32_t output_row_pitch_in_blocks_or_pixels = 0,
			basisu_transcoder_state* pState = nullptr,
			uint32_t output_rows_in_pixels = 0,
			int channel0 = -1, int channel1 = -1);
	};

	struct basisu_slice_info
	{
		uint32_t m_orig_width;
		uint32_t m_orig_height;

		uint32_t m_width;
		uint32_t m_height;

		uint32_t m_num_blocks_x;
		uint32_t m_num_blocks_y;
		uint32_t m_total_blocks;

		uint32_t m_compressed_size;

		uint32_t m_slice_index;	// the slice index in the .basis file
		uint32_t m_image_index;	// the source image index originally provided to the encoder
		uint32_t m_level_index;	// the mipmap level within this image

		uint32_t m_unpacked_slice_crc16;

		bool m_alpha_flag;		// true if the slice has alpha data
		bool m_iframe_flag;		// true if the slice is an I-Frame
	};

	typedef basisu::vector<basisu_slice_info> basisu_slice_info_vec;

	struct basisu_image_info
	{
		uint32_t m_image_index;
		uint32_t m_total_levels;

		uint32_t m_orig_width;
		uint32_t m_orig_height;

		uint32_t m_width;
		uint32_t m_height;

		uint32_t m_num_blocks_x;
		uint32_t m_num_blocks_y;
		uint32_t m_total_blocks;

		uint32_t m_first_slice_index;

		bool m_alpha_flag;		// true if the image has alpha data
		bool m_iframe_flag;		// true if the image is an I-Frame
	};

	struct basisu_image_level_info
	{
		uint32_t m_image_index;
		uint32_t m_level_index;

		uint32_t m_orig_width;
		uint32_t m_orig_height;

		uint32_t m_width;
		uint32_t m_height;

		uint32_t m_num_blocks_x;
		uint32_t m_num_blocks_y;
		uint32_t m_total_blocks;

		uint32_t m_first_slice_index;

		uint32_t m_rgb_file_ofs;
		uint32_t m_rgb_file_len;
		uint32_t m_alpha_file_ofs;
		uint32_t m_alpha_file_len;

		bool m_alpha_flag;		// true if the image has alpha data
		bool m_iframe_flag;		// true if the image is an I-Frame
	};

	struct basisu_file_info
	{
		uint32_t m_version;
		uint32_t m_total_header_size;

		uint32_t m_total_selectors;
		// will be 0 for UASTC or if the file uses global codebooks
		uint32_t m_selector_codebook_ofs;
		uint32_t m_selector_codebook_size;

		uint32_t m_total_endpoints;
		// will be 0 for UASTC or if the file uses global codebooks
		uint32_t m_endpoint_codebook_ofs;
		uint32_t m_endpoint_codebook_size;

		uint32_t m_tables_ofs;
		uint32_t m_tables_size;

		uint32_t m_slices_size;

		basis_texture_type m_tex_type;
		uint32_t m_us_per_frame;

		// Low-level slice information (1 slice per image for color-only basis files, 2 for alpha basis files)
		basisu_slice_info_vec m_slice_info;

		uint32_t m_total_images;	 // total # of images
		basisu::vector<uint32_t> m_image_mipmap_levels; // the # of mipmap levels for each image

		uint32_t m_userdata0;
		uint32_t m_userdata1;

		basis_tex_format m_tex_format; // ETC1S, UASTC, etc.

		bool m_y_flipped;				// true if the image was Y flipped
		bool m_etc1s;					// true if the file is ETC1S
		bool m_has_alpha_slices;	// true if the texture has alpha slices (for ETC1S: even slices RGB, odd slices alpha)
	};

	// High-level transcoder class which accepts .basis file data and allows the caller to query information about the file and transcode image levels to various texture formats.
	// If you're just starting out this is the class you care about.
	class basisu_transcoder
	{
		basisu_transcoder(basisu_transcoder&);
		basisu_transcoder& operator= (const basisu_transcoder&);

	public:
		basisu_transcoder();

		// Validates the .basis file. This computes a crc16 over the entire file, so it's slow.
		bool validate_file_checksums(const void* pData, uint32_t data_size, bool full_validation) const;

		// Quick header validation - no crc16 checks.
		bool validate_header(const void* pData, uint32_t data_size) const;

		basis_texture_type get_texture_type(const void* pData, uint32_t data_size) const;
		bool get_userdata(const void* pData, uint32_t data_size, uint32_t& userdata0, uint32_t& userdata1) const;

		// Returns the total number of images in the basis file (always 1 or more).
		// Note that the number of mipmap levels for each image may differ, and that images may have different resolutions.
		uint32_t get_total_images(const void* pData, uint32_t data_size) const;

		basis_tex_format get_tex_format(const void* pData, uint32_t data_size) const;

		// Returns the number of mipmap levels in an image.
		uint32_t get_total_image_levels(const void* pData, uint32_t data_size, uint32_t image_index) const;

		// Returns basic information about an image. Note that orig_width/orig_height may not be a multiple of 4.
		bool get_image_level_desc(const void* pData, uint32_t data_size, uint32_t image_index, uint32_t level_index, uint32_t& orig_width, uint32_t& orig_height, uint32_t& total_blocks) const;

		// Returns information about the specified image.
		bool get_image_info(const void* pData, uint32_t data_size, basisu_image_info& image_info, uint32_t image_index) const;

		// Returns information about the specified image's mipmap level.
		bool get_image_level_info(const void* pData, uint32_t data_size, basisu_image_level_info& level_info, uint32_t image_index, uint32_t level_index) const;

		// Get a description of the basis file and low-level information about each slice.
		bool get_file_info(const void* pData, uint32_t data_size, basisu_file_info& file_info) const;

		// start_transcoding() must be called before calling transcode_slice() or transcode_image_level().
		// For ETC1S files, this call decompresses the selector/endpoint codebooks, so ideally you would only call this once per .basis file (not each image/mipmap level).
		bool start_transcoding(const void* pData, uint32_t data_size);

		bool stop_transcoding();

		// Returns true if start_transcoding() has been called.
		bool get_ready_to_transcode() const { return m_ready_to_transcode; }

		// transcode_image_level() decodes a single mipmap level from the .basis file to any of the supported output texture formats.
		// It'll first find the slice(s) to transcode, then call transcode_slice() one or two times to decode both the color and alpha texture data (or RG texture data from two slices for BC5).
		// If the .basis file doesn't have alpha slices, the output alpha blocks will be set to fully opaque (all 255's).
		// Currently, to decode to PVRTC1 the basis texture's dimensions in pixels must be a power of 2, due to PVRTC1 format requirements. 
		// output_blocks_buf_size_in_blocks_or_pixels should be at least the image level's total_blocks (num_blocks_x * num_blocks_y), or the total number of output pixels if fmt==cTFRGBA32.
		// output_row_pitch_in_blocks_or_pixels: Number of blocks or pixels per row. If 0, the transcoder uses the slice's num_blocks_x or orig_width (NOT num_blocks_x * 4). Ignored for PVRTC1 (due to texture swizzling).
		// output_rows_in_pixels: Ignored unless fmt is uncompressed (cRGBA32, etc.). The total number of output rows in the output buffer. If 0, the transcoder assumes the slice's orig_height (NOT num_blocks_y * 4).
		// Notes: 
		// - basisu_transcoder_init() must have been called first to initialize the transcoder lookup tables before calling this function.
		// - This method assumes the output texture buffer is readable. In some cases to handle alpha, the transcoder will write temporary data to the output texture in
		// a first pass, which will be read in a second pass.
		bool transcode_image_level(
			const void* pData, uint32_t data_size,
			uint32_t image_index, uint32_t level_index,
			void* pOutput_blocks, uint32_t output_blocks_buf_size_in_blocks_or_pixels,
			transcoder_texture_format fmt,
			uint32_t decode_flags = 0, uint32_t output_row_pitch_in_blocks_or_pixels = 0, basisu_transcoder_state* pState = nullptr, uint32_t output_rows_in_pixels = 0) const;

		// Finds the basis slice corresponding to the specified image/level/alpha params, or -1 if the slice can't be found.
		int find_slice(const void* pData, uint32_t data_size, uint32_t image_index, uint32_t level_index, bool alpha_data) const;

		// transcode_slice() decodes a single slice from the .basis file. It's a low-level API - most likely you want to use transcode_image_level().
		// This is a low-level API, and will be needed to be called multiple times to decode some texture formats (like BC3, BC5, or ETC2).
		// output_blocks_buf_size_in_blocks_or_pixels is just used for verification to make sure the output buffer is large enough.
		// output_blocks_buf_size_in_blocks_or_pixels should be at least the image level's total_blocks (num_blocks_x * num_blocks_y), or the total number of output pixels if fmt==cTFRGBA32.
		// output_block_stride_in_bytes: Number of bytes between each output block.
		// output_row_pitch_in_blocks_or_pixels: Number of blocks or pixels per row. If 0, the transcoder uses the slice's num_blocks_x or orig_width (NOT num_blocks_x * 4). Ignored for PVRTC1 (due to texture swizzling).
		// output_rows_in_pixels: Ignored unless fmt is cRGBA32. The total number of output rows in the output buffer. If 0, the transcoder assumes the slice's orig_height (NOT num_blocks_y * 4).
		// Notes:
		// - basisu_transcoder_init() must have been called first to initialize the transcoder lookup tables before calling this function.
		bool transcode_slice(const void* pData, uint32_t data_size, uint32_t slice_index,
			void* pOutput_blocks, uint32_t output_blocks_buf_size_in_blocks_or_pixels,
			block_format fmt, uint32_t output_block_stride_in_bytes, uint32_t decode_flags = 0, uint32_t output_row_pitch_in_blocks_or_pixels = 0, basisu_transcoder_state* pState = nullptr, void* pAlpha_blocks = nullptr,
			uint32_t output_rows_in_pixels = 0, int channel0 = -1, int channel1 = -1) const;

		static void write_opaque_alpha_blocks(
			uint32_t num_blocks_x, uint32_t num_blocks_y,
			void* pOutput_blocks, block_format fmt,
			uint32_t block_stride_in_bytes, uint32_t output_row_pitch_in_blocks_or_pixels);

		void set_global_codebooks(const basisu_lowlevel_etc1s_transcoder* pGlobal_codebook) { m_lowlevel_etc1s_decoder.set_global_codebooks(pGlobal_codebook); }
		const basisu_lowlevel_etc1s_transcoder* get_global_codebooks() const { return m_lowlevel_etc1s_decoder.get_global_codebooks(); }

		const basisu_lowlevel_etc1s_transcoder& get_lowlevel_etc1s_decoder() const { return m_lowlevel_etc1s_decoder; }
		basisu_lowlevel_etc1s_transcoder& get_lowlevel_etc1s_decoder() { return m_lowlevel_etc1s_decoder; }

		const basisu_lowlevel_uastc_transcoder& get_lowlevel_uastc_decoder() const { return m_lowlevel_uastc_decoder; }
		basisu_lowlevel_uastc_transcoder& get_lowlevel_uastc_decoder() { return m_lowlevel_uastc_decoder; }

	private:
		mutable basisu_lowlevel_etc1s_transcoder m_lowlevel_etc1s_decoder;
		mutable basisu_lowlevel_uastc_transcoder m_lowlevel_uastc_decoder;

		bool m_ready_to_transcode;

		int find_first_slice_index(const void* pData, uint32_t data_size, uint32_t image_index, uint32_t level_index) const;

		bool validate_header_quick(const void* pData, uint32_t data_size) const;
	};

	// basisu_transcoder_init() MUST be called before a .basis file can be transcoded.
	void basisu_transcoder_init();
		
	enum debug_flags_t
	{
		cDebugFlagVisCRs = 1,
		cDebugFlagVisBC1Sels = 2,
		cDebugFlagVisBC1Endpoints = 4
	};
	uint32_t get_debug_flags();
	void set_debug_flags(uint32_t f);

	// ------------------------------------------------------------------------------------------------------ 
	// Optional .KTX2 file format support
	// KTX2 reading optionally requires miniz or Zstd decompressors for supercompressed UASTC files.
	// ------------------------------------------------------------------------------------------------------ 
#if BASISD_SUPPORT_KTX2
#pragma pack(push)
#pragma pack(1)
	struct ktx2_header
	{
		uint8_t m_identifier[12];
		basisu::packed_uint<4> m_vk_format;
		basisu::packed_uint<4> m_type_size;
		basisu::packed_uint<4> m_pixel_width;
		basisu::packed_uint<4> m_pixel_height;
		basisu::packed_uint<4> m_pixel_depth;
		basisu::packed_uint<4> m_layer_count;
		basisu::packed_uint<4> m_face_count;
		basisu::packed_uint<4> m_level_count;
		basisu::packed_uint<4> m_supercompression_scheme;
		basisu::packed_uint<4> m_dfd_byte_offset;
		basisu::packed_uint<4> m_dfd_byte_length;
		basisu::packed_uint<4> m_kvd_byte_offset;
		basisu::packed_uint<4> m_kvd_byte_length;
		basisu::packed_uint<8> m_sgd_byte_offset;
		basisu::packed_uint<8> m_sgd_byte_length;
	};

	struct ktx2_level_index
	{
		basisu::packed_uint<8> m_byte_offset;
		basisu::packed_uint<8> m_byte_length;
		basisu::packed_uint<8> m_uncompressed_byte_length;
	};

	struct ktx2_etc1s_global_data_header
	{
		basisu::packed_uint<2> m_endpoint_count;
		basisu::packed_uint<2> m_selector_count;
		basisu::packed_uint<4> m_endpoints_byte_length;
		basisu::packed_uint<4> m_selectors_byte_length;
		basisu::packed_uint<4> m_tables_byte_length;
		basisu::packed_uint<4> m_extended_byte_length;
	};

	struct ktx2_etc1s_image_desc
	{
		basisu::packed_uint<4> m_image_flags;
		basisu::packed_uint<4> m_rgb_slice_byte_offset;
		basisu::packed_uint<4> m_rgb_slice_byte_length;
		basisu::packed_uint<4> m_alpha_slice_byte_offset;
		basisu::packed_uint<4> m_alpha_slice_byte_length;
	};

	struct ktx2_animdata
	{
		basisu::packed_uint<4> m_duration;
		basisu::packed_uint<4> m_timescale;
		basisu::packed_uint<4> m_loopcount;
	};
#pragma pack(pop)

	const uint32_t KTX2_VK_FORMAT_UNDEFINED = 0;
	const uint32_t KTX2_KDF_DF_MODEL_UASTC = 166;
	const uint32_t KTX2_KDF_DF_MODEL_ETC1S = 163;
	const uint32_t KTX2_IMAGE_IS_P_FRAME = 2;
	const uint32_t KTX2_UASTC_BLOCK_SIZE = 16;
	const uint32_t KTX2_MAX_SUPPORTED_LEVEL_COUNT = 16; // this is an implementation specific constraint and can be increased

	// The KTX2 transfer functions supported by KTX2
	const uint32_t KTX2_KHR_DF_TRANSFER_LINEAR = 1;
	const uint32_t KTX2_KHR_DF_TRANSFER_SRGB = 2;

	enum ktx2_supercompression
	{
		KTX2_SS_NONE = 0,
		KTX2_SS_BASISLZ = 1,
		KTX2_SS_ZSTANDARD = 2
	};

	extern const uint8_t g_ktx2_file_identifier[12];

	enum ktx2_df_channel_id
	{
		KTX2_DF_CHANNEL_ETC1S_RGB = 0U,
		KTX2_DF_CHANNEL_ETC1S_RRR = 3U,
		KTX2_DF_CHANNEL_ETC1S_GGG = 4U,
		KTX2_DF_CHANNEL_ETC1S_AAA = 15U,

		KTX2_DF_CHANNEL_UASTC_DATA = 0U,
		KTX2_DF_CHANNEL_UASTC_RGB = 0U,
		KTX2_DF_CHANNEL_UASTC_RGBA = 3U,
		KTX2_DF_CHANNEL_UASTC_RRR = 4U,
		KTX2_DF_CHANNEL_UASTC_RRRG = 5U,
		KTX2_DF_CHANNEL_UASTC_RG = 6U,
	};

	inline const char* ktx2_get_etc1s_df_channel_id_str(ktx2_df_channel_id id)
	{
		switch (id)
		{
		case KTX2_DF_CHANNEL_ETC1S_RGB: return "RGB";
		case KTX2_DF_CHANNEL_ETC1S_RRR: return "RRR";
		case KTX2_DF_CHANNEL_ETC1S_GGG: return "GGG";
		case KTX2_DF_CHANNEL_ETC1S_AAA: return "AAA";
		default: break;
		}
		return "?";
	}

	inline const char* ktx2_get_uastc_df_channel_id_str(ktx2_df_channel_id id)
	{
		switch (id)
		{
		case KTX2_DF_CHANNEL_UASTC_RGB: return "RGB";
		case KTX2_DF_CHANNEL_UASTC_RGBA: return "RGBA";
		case KTX2_DF_CHANNEL_UASTC_RRR: return "RRR";
		case KTX2_DF_CHANNEL_UASTC_RRRG: return "RRRG";
		case KTX2_DF_CHANNEL_UASTC_RG: return "RG";
		default: break;
		}
		return "?";
	}

	enum ktx2_df_color_primaries
	{
		KTX2_DF_PRIMARIES_UNSPECIFIED = 0,
		KTX2_DF_PRIMARIES_BT709 = 1,
		KTX2_DF_PRIMARIES_SRGB = 1,
		KTX2_DF_PRIMARIES_BT601_EBU = 2,
		KTX2_DF_PRIMARIES_BT601_SMPTE = 3,
		KTX2_DF_PRIMARIES_BT2020 = 4,
		KTX2_DF_PRIMARIES_CIEXYZ = 5,
		KTX2_DF_PRIMARIES_ACES = 6,
		KTX2_DF_PRIMARIES_ACESCC = 7,
		KTX2_DF_PRIMARIES_NTSC1953 = 8,
		KTX2_DF_PRIMARIES_PAL525 = 9,
		KTX2_DF_PRIMARIES_DISPLAYP3 = 10,
		KTX2_DF_PRIMARIES_ADOBERGB = 11
	};

	inline const char* ktx2_get_df_color_primaries_str(ktx2_df_color_primaries p)
	{
		switch (p)
		{
		case KTX2_DF_PRIMARIES_UNSPECIFIED: return "UNSPECIFIED";
		case KTX2_DF_PRIMARIES_BT709: return "BT709";
		case KTX2_DF_PRIMARIES_BT601_EBU: return "EBU"; 
		case KTX2_DF_PRIMARIES_BT601_SMPTE: return "SMPTE";
		case KTX2_DF_PRIMARIES_BT2020: return "BT2020";
		case KTX2_DF_PRIMARIES_CIEXYZ: return "CIEXYZ";
		case KTX2_DF_PRIMARIES_ACES: return "ACES";
		case KTX2_DF_PRIMARIES_ACESCC: return "ACESCC"; 
		case KTX2_DF_PRIMARIES_NTSC1953: return "NTSC1953";
		case KTX2_DF_PRIMARIES_PAL525: return "PAL525";
		case KTX2_DF_PRIMARIES_DISPLAYP3: return "DISPLAYP3";
		case KTX2_DF_PRIMARIES_ADOBERGB: return "ADOBERGB";
		default: break;
		}
		return "?";
	}	

	// Information about a single 2D texture "image" in a KTX2 file.
	struct ktx2_image_level_info
	{
		// The mipmap level index (0=largest), texture array layer index, and cubemap face index of the image.
		uint32_t m_level_index;
		uint32_t m_layer_index;
		uint32_t m_face_index;

		// The image's actual (or the original source image's) width/height in pixels, which may not be divisible by 4 pixels.
		uint32_t m_orig_width;
		uint32_t m_orig_height;

		// The image's physical width/height, which will always be divisible by 4 pixels.
		uint32_t m_width;
		uint32_t m_height;

		// The texture's dimensions in 4x4 texel blocks.
		uint32_t m_num_blocks_x;
		uint32_t m_num_blocks_y;

		// The total number of blocks
		uint32_t m_total_blocks;

		// true if the image has alpha data
		bool m_alpha_flag;

		// true if the image is an I-Frame. Currently, for ETC1S textures, the first frame will always be an I-Frame, and subsequent frames will always be P-Frames.
		bool m_iframe_flag;
	};
		
	// Thread-specific ETC1S/supercompressed UASTC transcoder state. (If you're not doing multithreading transcoding you can ignore this.)
	struct ktx2_transcoder_state
	{
		basist::basisu_transcoder_state m_transcoder_state;
		basisu::uint8_vec m_level_uncomp_data;
		int m_uncomp_data_level_index;

		void clear()
		{
			m_transcoder_state.clear();
			m_level_uncomp_data.clear();
			m_uncomp_data_level_index = -1;
		}
	};

	// This class is quite similar to basisu_transcoder. It treats KTX2 files as a simple container for ETC1S/UASTC texture data.
	// It does not support 1D or 3D textures.
	// It only supports 2D and cubemap textures, with or without mipmaps, texture arrays of 2D/cubemap textures, and texture video files. 
	// It only supports raw non-supercompressed UASTC, ETC1S, UASTC+Zstd, or UASTC+zlib compressed files.
	// DFD (Data Format Descriptor) parsing is purposely as simple as possible. 
	// If you need to know how to interpret the texture channels you'll need to parse the DFD yourself after calling get_dfd().
	class ktx2_transcoder
	{
	public:
		ktx2_transcoder();

		// Frees all allocations, resets object.
		void clear();

		// init() parses the KTX2 header, level index array, DFD, and key values, but nothing else.
		// Importantly, it does not parse or decompress the ETC1S global supercompressed data, so some things (like which frames are I/P-Frames) won't be available until start_transcoding() is called.
		// This method holds a pointer to the file data until clear() is called.
		bool init(const void* pData, uint32_t data_size);

		// Returns the data/size passed to init().
		const uint8_t* get_data() const { return m_pData; }
		uint32_t get_data_size() const { return m_data_size; }

		// Returns the KTX2 header. Valid after init().
		const ktx2_header& get_header() const { return m_header; }

		// Returns the KTX2 level index array. There will be one entry for each mipmap level. Valid after init().
		const basisu::vector<ktx2_level_index>& get_level_index() const { return m_levels; }

		// Returns the texture's width in texels. Always non-zero, might not be divisible by 4. Valid after init().
		uint32_t get_width() const { return m_header.m_pixel_width; }

		// Returns the texture's height in texels. Always non-zero, might not be divisible by 4. Valid after init().
		uint32_t get_height() const { return m_header.m_pixel_height; }

		// Returns the texture's number of mipmap levels. Always returns 1 or higher. Valid after init().
		uint32_t get_levels() const { return m_header.m_level_count; }

		// Returns the number of faces. Returns 1 for 2D textures and or 6 for cubemaps. Valid after init().
		uint32_t get_faces() const { return m_header.m_face_count; }

		// Returns 0 or the number of layers in the texture array or texture video. Valid after init().
		uint32_t get_layers() const { return m_header.m_layer_count; }

		// Returns cETC1S or cUASTC4x4. Valid after init().
		basist::basis_tex_format get_format() const { return m_format; } 

		bool is_etc1s() const { return get_format() == basist::basis_tex_format::cETC1S; }

		bool is_uastc() const { return get_format() == basist::basis_tex_format::cUASTC4x4; }

		// Returns true if the ETC1S file has two planes (typically RGBA, or RRRG), or true if the UASTC file has alpha data. Valid after init().
		uint32_t get_has_alpha() const { return m_has_alpha; }

		// Returns the entire Data Format Descriptor (DFD) from the KTX2 file. Valid after init().
		// See https://www.khronos.org/registry/DataFormat/specs/1.3/dataformat.1.3.html#_the_khronos_data_format_descriptor_overview
		const basisu::uint8_vec& get_dfd() const { return m_dfd; }

		// Some basic DFD accessors. Valid after init().
		uint32_t get_dfd_color_model() const { return m_dfd_color_model; }

		// Returns the DFD color primary.
		// We do not validate the color primaries, so the returned value may not be in the ktx2_df_color_primaries enum.
		ktx2_df_color_primaries get_dfd_color_primaries() const { return m_dfd_color_prims; }
		
		// Returns KTX2_KHR_DF_TRANSFER_LINEAR or KTX2_KHR_DF_TRANSFER_SRGB.
		uint32_t get_dfd_transfer_func() const { return m_dfd_transfer_func; }

		uint32_t get_dfd_flags() const { return m_dfd_flags; }

		// Returns 1 (ETC1S/UASTC) or 2 (ETC1S with an internal alpha channel).
		uint32_t get_dfd_total_samples() const { return m_dfd_samples;	}
		
		// Returns the channel mapping for each DFD "sample". UASTC always has 1 sample, ETC1S can have one or two. 
		// Note the returned value SHOULD be one of the ktx2_df_channel_id enums, but we don't validate that. 
		// It's up to the caller to decide what to do if the value isn't in the enum.
		ktx2_df_channel_id get_dfd_channel_id0() const { return m_dfd_chan0; }
		ktx2_df_channel_id get_dfd_channel_id1() const { return m_dfd_chan1; }

		// Key value field data.
		struct key_value
		{
			// The key field is UTF8 and always zero terminated.
			basisu::uint8_vec m_key;

			// The value may be empty. It consists of raw bytes which may or may not be zero terminated.
			basisu::uint8_vec m_value;

			bool operator< (const key_value& rhs) const { return strcmp((const char*)m_key.data(), (const char *)rhs.m_key.data()) < 0; }
		};
		typedef basisu::vector<key_value> key_value_vec;

		// Returns the array of key-value entries. This may be empty. Valid after init().
		// The order of key values fields in this array exactly matches the order they were stored in the file. The keys are supposed to be sorted by their Unicode code points.
		const key_value_vec& get_key_values() const { return m_key_values; }

		const basisu::uint8_vec *find_key(const std::string& key_name) const;

		// Low-level ETC1S specific accessors

		// Returns the ETC1S global supercompression data header, which is only valid after start_transcoding() is called.
		const ktx2_etc1s_global_data_header& get_etc1s_header() const { return m_etc1s_header; }

		// Returns the array of ETC1S image descriptors, which is only valid after get_etc1s_image_descs() is called.
		const basisu::vector<ktx2_etc1s_image_desc>& get_etc1s_image_descs() const { return m_etc1s_image_descs; }

		// Must have called startTranscoding() first
		uint32_t get_etc1s_image_descs_image_flags(uint32_t level_index, uint32_t layer_index, uint32_t face_index) const;

		// is_video() is only valid after start_transcoding() is called.
		// For ETC1S data, if this returns true you must currently transcode the file from first to last frame, in order, without skipping any frames.
		bool is_video() const { return m_is_video; }
				
		// start_transcoding() MUST be called before calling transcode_image().
		// This method decompresses the ETC1S global endpoint/selector codebooks, which is not free, so try to avoid calling it excessively.
		bool start_transcoding();
								
		// get_image_level_info() be called after init(), but the m_iframe_flag's won't be valid until start_transcoding() is called.
		// You can call this method before calling transcode_image_level() to retrieve basic information about the mipmap level's dimensions, etc.
		bool get_image_level_info(ktx2_image_level_info& level_info, uint32_t level_index, uint32_t layer_index, uint32_t face_index) const;

		// transcode_image_level() transcodes a single 2D texture or cubemap face from the KTX2 file.
		// Internally it uses the same low-level transcode API's as basisu_transcoder::transcode_image_level().
		// If the file is UASTC and is supercompressed with Zstandard, and the file is a texture array or cubemap, it's highly recommended that each mipmap level is 
		// completely transcoded before switching to another level. Every time the mipmap level is changed all supercompressed level data must be decompressed using Zstandard as a single unit.
		// Currently ETC1S videos must always be transcoded from first to last frame (or KTX2 "layer"), in order, with no skipping of frames.
		// By default this method is not thread safe unless you specify a pointer to a user allocated thread-specific transcoder_state struct.
		bool transcode_image_level(
			uint32_t level_index, uint32_t layer_index, uint32_t face_index,
			void* pOutput_blocks, uint32_t output_blocks_buf_size_in_blocks_or_pixels,
			basist::transcoder_texture_format fmt,
			uint32_t decode_flags = 0, uint32_t output_row_pitch_in_blocks_or_pixels = 0, uint32_t output_rows_in_pixels = 0, int channel0 = -1, int channel1 = -1,
			ktx2_transcoder_state *pState = nullptr);
				
	private:
		const uint8_t* m_pData;
		uint32_t m_data_size;

		ktx2_header m_header;
		basisu::vector<ktx2_level_index> m_levels;
		basisu::uint8_vec m_dfd;
		key_value_vec m_key_values;
		
		ktx2_etc1s_global_data_header m_etc1s_header;
		basisu::vector<ktx2_etc1s_image_desc> m_etc1s_image_descs;

		basist::basis_tex_format m_format;
					
		uint32_t m_dfd_color_model;
		ktx2_df_color_primaries m_dfd_color_prims;
		uint32_t m_dfd_transfer_func;
		uint32_t m_dfd_flags;
		uint32_t m_dfd_samples;
		ktx2_df_channel_id m_dfd_chan0, m_dfd_chan1;
								
		basist::basisu_lowlevel_etc1s_transcoder m_etc1s_transcoder;
		basist::basisu_lowlevel_uastc_transcoder m_uastc_transcoder;
				
		ktx2_transcoder_state m_def_transcoder_state;

		bool m_has_alpha;
		bool m_is_video;

		bool decompress_level_data(uint32_t level_index, basisu::uint8_vec& uncomp_data);
		bool decompress_etc1s_global_data();
		bool read_key_values();
	};

#endif // BASISD_SUPPORT_KTX2

	// Returns true if the transcoder was compiled with KTX2 support.
	bool basisu_transcoder_supports_ktx2();

	// Returns true if the transcoder was compiled with Zstandard support.
	bool basisu_transcoder_supports_ktx2_zstd();

} // namespace basisu

/**** ended inlining basisu_transcoder.h ****/
#include <limits.h>
/**** start inlining basisu_containers_impl.h ****/
// basisu_containers_impl.h
// Do not include directly

#ifdef _MSC_VER
#pragma warning (disable:4127) // warning C4127: conditional expression is constant
#endif

namespace basisu
{
   bool elemental_vector::increase_capacity(uint32_t min_new_capacity, bool grow_hint, uint32_t element_size, object_mover pMover, bool nofail)
   {
      assert(m_size <= m_capacity);

      if (sizeof(void *) == sizeof(uint64_t))
         assert(min_new_capacity < (0x400000000ULL / element_size));
      else
         assert(min_new_capacity < (0x7FFF0000U / element_size));

      if (m_capacity >= min_new_capacity)
         return true;

      size_t new_capacity = min_new_capacity;
      if ((grow_hint) && (!helpers::is_power_of_2((uint64_t)new_capacity)))
      {
         new_capacity = (size_t)helpers::next_pow2((uint64_t)new_capacity);

         assert(new_capacity && (new_capacity > m_capacity));

         if (new_capacity < min_new_capacity)
         {
            if (nofail)
               return false;
            fprintf(stderr, "vector too large\n");
            abort();
         }
      }
            
      const size_t desired_size = element_size * new_capacity;
      size_t actual_size = 0;
      if (!pMover)
      {
         void* new_p = realloc(m_p, desired_size);
         if (!new_p)
         {
            if (nofail)
               return false;

            char buf[256];
#ifdef _MSC_VER
            sprintf_s(buf, sizeof(buf), "vector: realloc() failed allocating %u bytes", (uint32_t)desired_size);
#else
            sprintf(buf, "vector: realloc() failed allocating %u bytes", (uint32_t)desired_size);
#endif
            fprintf(stderr, "%s", buf);
            abort();
         }

#if BASISU_VECTOR_DETERMINISTIC
         actual_size = desired_size;
#elif defined(_MSC_VER)
         actual_size = _msize(new_p);
#elif HAS_MALLOC_USABLE_SIZE
         actual_size = malloc_usable_size(new_p);
#else
         actual_size = desired_size;
#endif
         m_p = new_p;
      }
      else
      {
         void* new_p = malloc(desired_size);
         if (!new_p)
         {
            if (nofail)
               return false;

            char buf[256];
#ifdef _MSC_VER
            sprintf_s(buf, sizeof(buf), "vector: malloc() failed allocating %u bytes", (uint32_t)desired_size);
#else
            sprintf(buf, "vector: malloc() failed allocating %u bytes", (uint32_t)desired_size);
#endif
            fprintf(stderr, "%s", buf);
            abort();
         }

#if BASISU_VECTOR_DETERMINISTIC
         actual_size = desired_size;
#elif defined(_MSC_VER)
         actual_size = _msize(new_p);
#elif HAS_MALLOC_USABLE_SIZE
         actual_size = malloc_usable_size(new_p);
#else
         actual_size = desired_size;
#endif

         (*pMover)(new_p, m_p, m_size);

         if (m_p)
            free(m_p);
         
         m_p = new_p;
      }

      if (actual_size > desired_size)
         m_capacity = static_cast<uint32_t>(actual_size / element_size);
      else
         m_capacity = static_cast<uint32_t>(new_capacity);

      return true;
   }

#if BASISU_HASHMAP_TEST

#define HASHMAP_TEST_VERIFY(c) do { if (!(c)) handle_hashmap_test_verify_failure(__LINE__); } while(0)

   static void handle_hashmap_test_verify_failure(int line)
   {
      fprintf(stderr, "HASHMAP_TEST_VERIFY() faild on line %i\n", line);
      abort();
   }

   class counted_obj
   {
   public:
      counted_obj(uint32_t v = 0) :
         m_val(v)
      {
         m_count++;
      }

      counted_obj(const counted_obj& obj) :
         m_val(obj.m_val)
      {
         m_count++;
      }

      ~counted_obj()
      {
         assert(m_count > 0);
         m_count--;
      }

      static uint32_t m_count;

      uint32_t m_val;

      operator size_t() const { return m_val; }

      bool operator== (const counted_obj& rhs) const { return m_val == rhs.m_val; }
      bool operator== (const uint32_t rhs) const { return m_val == rhs; }

   };

   uint32_t counted_obj::m_count;

   static uint32_t urand32()
   {
      uint32_t a = rand();
      uint32_t b = rand() << 15;
      uint32_t c = rand() << (32 - 15);
      return a ^ b ^ c;
   }

   static int irand32(int l, int h)
   {
      assert(l < h);
      if (l >= h)
         return l;

      uint32_t range = static_cast<uint32_t>(h - l);

      uint32_t rnd = urand32();

      uint32_t rnd_range = static_cast<uint32_t>((((uint64_t)range) * ((uint64_t)rnd)) >> 32U);

      int result = l + rnd_range;
      assert((result >= l) && (result < h));
      return result;
   }

   void hash_map_test()
   {
      {
         basisu::hash_map<uint64_t, uint64_t> k;
         basisu::hash_map<uint64_t, uint64_t> l;
         std::swap(k, l);

         k.begin();
         k.end();
         k.clear();
         k.empty();
         k.erase(0);
         k.insert(0, 1);
         k.find(0);
         k.get_equals();
         k.get_hasher();
         k.get_table_size();
         k.reset();
         k.reserve(1);
         k = l;
         k.set_equals(l.get_equals());
         k.set_hasher(l.get_hasher());
         k.get_table_size();
      }

      uint32_t seed = 0;
      for (; ; )
      {
         seed++;

         typedef basisu::hash_map<counted_obj, counted_obj> my_hash_map;
         my_hash_map m;

         const uint32_t n = irand32(0, 100000);

         printf("%u\n", n);

         srand(seed); // r1.seed(seed);

         basisu::vector<int> q;

         uint32_t count = 0;
         for (uint32_t i = 0; i < n; i++)
         {
            uint32_t v = urand32() & 0x7FFFFFFF;
            my_hash_map::insert_result res = m.insert(counted_obj(v), counted_obj(v ^ 0xdeadbeef));
            if (res.second)
            {
               count++;
               q.push_back(v);
            }
         }

         HASHMAP_TEST_VERIFY(m.size() == count);

         srand(seed);

         my_hash_map cm(m);
         m.clear();
         m = cm;
         cm.reset();

         for (uint32_t i = 0; i < n; i++)
         {
            uint32_t v = urand32() & 0x7FFFFFFF;
            my_hash_map::const_iterator it = m.find(counted_obj(v));
            HASHMAP_TEST_VERIFY(it != m.end());
            HASHMAP_TEST_VERIFY(it->first == v);
            HASHMAP_TEST_VERIFY(it->second == (v ^ 0xdeadbeef));
         }

         for (uint32_t t = 0; t < 2; t++)
         {
            const uint32_t nd = irand32(1, q.size() + 1);
            for (uint32_t i = 0; i < nd; i++)
            {
               uint32_t p = irand32(0, q.size());

               int k = q[p];
               if (k >= 0)
               {
                  q[p] = -k - 1;

                  bool s = m.erase(counted_obj(k));
                  HASHMAP_TEST_VERIFY(s);
               }
            }

            typedef basisu::hash_map<uint32_t, empty_type> uint_hash_set;
            uint_hash_set s;

            for (uint32_t i = 0; i < q.size(); i++)
            {
               int v = q[i];

               if (v >= 0)
               {
                  my_hash_map::const_iterator it = m.find(counted_obj(v));
                  HASHMAP_TEST_VERIFY(it != m.end());
                  HASHMAP_TEST_VERIFY(it->first == (uint32_t)v);
                  HASHMAP_TEST_VERIFY(it->second == ((uint32_t)v ^ 0xdeadbeef));

                  s.insert(v);
               }
               else
               {
                  my_hash_map::const_iterator it = m.find(counted_obj(-v - 1));
                  HASHMAP_TEST_VERIFY(it == m.end());
               }
            }

            uint32_t found_count = 0;
            for (my_hash_map::const_iterator it = m.begin(); it != m.end(); ++it)
            {
               HASHMAP_TEST_VERIFY(it->second == ((uint32_t)it->first ^ 0xdeadbeef));

               uint_hash_set::const_iterator fit(s.find((uint32_t)it->first));
               HASHMAP_TEST_VERIFY(fit != s.end());

               HASHMAP_TEST_VERIFY(fit->first == it->first);

               found_count++;
            }

            HASHMAP_TEST_VERIFY(found_count == s.size());
         }

         HASHMAP_TEST_VERIFY(counted_obj::m_count == m.size() * 2);
      }
   }

#endif // BASISU_HASHMAP_TEST

} // namespace basisu
/**** ended inlining basisu_containers_impl.h ****/

#ifndef BASISD_IS_BIG_ENDIAN
// TODO: This doesn't work on OSX. How can this be so difficult?
//#if defined(__BIG_ENDIAN__) || defined(_BIG_ENDIAN) || defined(BIG_ENDIAN)
//	#define BASISD_IS_BIG_ENDIAN (1)
//#else
	#define BASISD_IS_BIG_ENDIAN (0)
//#endif
#endif

#ifndef BASISD_USE_UNALIGNED_WORD_READS
	#ifdef __EMSCRIPTEN__
		// Can't use unaligned loads/stores with WebAssembly.
		#define BASISD_USE_UNALIGNED_WORD_READS (0)
	#elif defined(_M_AMD64) || defined(_M_IX86) || defined(__i386__) || defined(__x86_64__)
		#define BASISD_USE_UNALIGNED_WORD_READS (1)
	#else
		#define BASISD_USE_UNALIGNED_WORD_READS (0)
	#endif
#endif

#define BASISD_SUPPORTED_BASIS_VERSION (0x13)

#ifndef BASISD_SUPPORT_KTX2
	#error Must have defined BASISD_SUPPORT_KTX2
#endif

#ifndef BASISD_SUPPORT_KTX2_ZSTD
#error Must have defined BASISD_SUPPORT_KTX2_ZSTD
#endif

// Set to 1 for fuzz testing. This will disable all CRC16 checks on headers and compressed data.
#ifndef BASISU_NO_HEADER_OR_DATA_CRC16_CHECKS
	#define BASISU_NO_HEADER_OR_DATA_CRC16_CHECKS 0
#endif

#ifndef BASISD_SUPPORT_DXT1
	#define BASISD_SUPPORT_DXT1 1
#endif

#ifndef BASISD_SUPPORT_DXT5A
	#define BASISD_SUPPORT_DXT5A 1
#endif

// Disable all BC7 transcoders if necessary (useful when cross compiling to Javascript)
#if defined(BASISD_SUPPORT_BC7) && !BASISD_SUPPORT_BC7
	#ifndef BASISD_SUPPORT_BC7_MODE5
		#define BASISD_SUPPORT_BC7_MODE5 0
	#endif
#endif // !BASISD_SUPPORT_BC7

// BC7 mode 5 supports both opaque and opaque+alpha textures, and uses less memory BC1.
#ifndef BASISD_SUPPORT_BC7_MODE5
	#define BASISD_SUPPORT_BC7_MODE5 1
#endif

#ifndef BASISD_SUPPORT_PVRTC1
	#define BASISD_SUPPORT_PVRTC1 1
#endif

#ifndef BASISD_SUPPORT_ETC2_EAC_A8
	#define BASISD_SUPPORT_ETC2_EAC_A8 1
#endif

// Set BASISD_SUPPORT_UASTC to 0 to completely disable support for transcoding UASTC files.
#ifndef BASISD_SUPPORT_UASTC
	#define BASISD_SUPPORT_UASTC 1
#endif

#ifndef BASISD_SUPPORT_ASTC
	#define BASISD_SUPPORT_ASTC 1
#endif

// Note that if BASISD_SUPPORT_ATC is enabled, BASISD_SUPPORT_DXT5A should also be enabled for alpha support.
#ifndef BASISD_SUPPORT_ATC
	#define BASISD_SUPPORT_ATC 1
#endif

// Support for ETC2 EAC R11 and ETC2 EAC RG11
#ifndef BASISD_SUPPORT_ETC2_EAC_RG11
	#define BASISD_SUPPORT_ETC2_EAC_RG11 1
#endif

// If BASISD_SUPPORT_ASTC_HIGHER_OPAQUE_QUALITY is 1, opaque blocks will be transcoded to ASTC at slightly higher quality (higher than BC1), but the transcoder tables will be 2x as large.
// This impacts grayscale and grayscale+alpha textures the most.
#ifndef BASISD_SUPPORT_ASTC_HIGHER_OPAQUE_QUALITY
	#ifdef __EMSCRIPTEN__
		// Let's assume size matters more than quality when compiling with emscripten.
		#define BASISD_SUPPORT_ASTC_HIGHER_OPAQUE_QUALITY 0
	#else
		// Compiling native, so an extra 64K lookup table is probably acceptable.
		#define BASISD_SUPPORT_ASTC_HIGHER_OPAQUE_QUALITY 1
	#endif
#endif

#ifndef BASISD_SUPPORT_FXT1
	#define BASISD_SUPPORT_FXT1 1
#endif

#ifndef BASISD_SUPPORT_PVRTC2
	#define BASISD_SUPPORT_PVRTC2 1
#endif

#if BASISD_SUPPORT_PVRTC2
	#if !BASISD_SUPPORT_ATC
		#error BASISD_SUPPORT_ATC must be 1 if BASISD_SUPPORT_PVRTC2 is 1
	#endif
#endif

#if BASISD_SUPPORT_ATC
	#if !BASISD_SUPPORT_DXT5A
		#error BASISD_SUPPORT_DXT5A must be 1 if BASISD_SUPPORT_ATC is 1
	#endif
#endif

#define BASISD_WRITE_NEW_BC7_MODE5_TABLES			0
#define BASISD_WRITE_NEW_DXT1_TABLES				0
#define BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES		0
#define BASISD_WRITE_NEW_ASTC_TABLES				0
#define BASISD_WRITE_NEW_ATC_TABLES					0
#define BASISD_WRITE_NEW_ETC2_EAC_R11_TABLES		0

#ifndef BASISD_ENABLE_DEBUG_FLAGS
	#define BASISD_ENABLE_DEBUG_FLAGS	0
#endif

// If KTX2 support is enabled, we may need Zstd for decompression of supercompressed UASTC files. Include this header.
#if BASISD_SUPPORT_KTX2
   // If BASISD_SUPPORT_KTX2_ZSTD is 0, UASTC files compressed with Zstd cannot be loaded.
	#if BASISD_SUPPORT_KTX2_ZSTD
		// We only use two Zstd API's: ZSTD_decompress() and ZSTD_isError()
/**** start inlining zstd ****/
#error Unable to find "zstd"
/**** ended inlining zstd ****/
	#endif
#endif

namespace basisu
{
	bool g_debug_printf;

	void enable_debug_printf(bool enabled)
	{
		g_debug_printf = enabled;
	}

	void debug_printf(const char* pFmt, ...)
	{
#if BASISU_FORCE_DEVEL_MESSAGES	
		g_debug_printf = true;
#endif
		if (g_debug_printf)
		{
			va_list args;
			va_start(args, pFmt);
			vprintf(pFmt, args);
			va_end(args);
		}
	}
} // namespace basisu

namespace basist
{

#if BASISD_ENABLE_DEBUG_FLAGS
	static uint32_t g_debug_flags = 0;
#endif

	uint32_t get_debug_flags()
	{
#if BASISD_ENABLE_DEBUG_FLAGS
		return g_debug_flags;
#else
		return 0;
#endif
	}

	void set_debug_flags(uint32_t f)
	{
		BASISU_NOTE_UNUSED(f);
#if BASISD_ENABLE_DEBUG_FLAGS
		g_debug_flags = f;
#endif
	}

	inline uint16_t byteswap_uint16(uint16_t v)
	{
		return static_cast<uint16_t>((v >> 8) | (v << 8));
	}

	static inline int32_t clampi(int32_t value, int32_t low, int32_t high) { if (value < low) value = low; else if (value > high) value = high;	return value; }
	static inline float clampf(float value, float low, float high) { if (value < low) value = low; else if (value > high) value = high;	return value; }
	static inline float saturate(float value) { return clampf(value, 0, 1.0f); }

	static inline uint8_t mul_8(uint32_t v, uint32_t q) { v = v * q + 128; return (uint8_t)((v + (v >> 8)) >> 8); }

	uint16_t crc16(const void* r, size_t size, uint16_t crc)
	{
		crc = ~crc;

		const uint8_t* p = static_cast<const uint8_t*>(r);
		for (; size; --size)
		{
			const uint16_t q = *p++ ^ (crc >> 8);
			uint16_t k = (q >> 4) ^ q;
			crc = (((crc << 8) ^ k) ^ (k << 5)) ^ (k << 12);
		}

		return static_cast<uint16_t>(~crc);
	}
		
	enum etc_constants
	{
		cETC1BytesPerBlock = 8U,

		cETC1SelectorBits = 2U,
		cETC1SelectorValues = 1U << cETC1SelectorBits,
		cETC1SelectorMask = cETC1SelectorValues - 1U,

		cETC1BlockShift = 2U,
		cETC1BlockSize = 1U << cETC1BlockShift,

		cETC1LSBSelectorIndicesBitOffset = 0,
		cETC1MSBSelectorIndicesBitOffset = 16,

		cETC1FlipBitOffset = 32,
		cETC1DiffBitOffset = 33,

		cETC1IntenModifierNumBits = 3,
		cETC1IntenModifierValues = 1 << cETC1IntenModifierNumBits,
		cETC1RightIntenModifierTableBitOffset = 34,
		cETC1LeftIntenModifierTableBitOffset = 37,

		// Base+Delta encoding (5 bit bases, 3 bit delta)
		cETC1BaseColorCompNumBits = 5,
		cETC1BaseColorCompMax = 1 << cETC1BaseColorCompNumBits,

		cETC1DeltaColorCompNumBits = 3,
		cETC1DeltaColorComp = 1 << cETC1DeltaColorCompNumBits,
		cETC1DeltaColorCompMax = 1 << cETC1DeltaColorCompNumBits,

		cETC1BaseColor5RBitOffset = 59,
		cETC1BaseColor5GBitOffset = 51,
		cETC1BaseColor5BBitOffset = 43,

		cETC1DeltaColor3RBitOffset = 56,
		cETC1DeltaColor3GBitOffset = 48,
		cETC1DeltaColor3BBitOffset = 40,

		// Absolute (non-delta) encoding (two 4-bit per component bases)
		cETC1AbsColorCompNumBits = 4,
		cETC1AbsColorCompMax = 1 << cETC1AbsColorCompNumBits,

		cETC1AbsColor4R1BitOffset = 60,
		cETC1AbsColor4G1BitOffset = 52,
		cETC1AbsColor4B1BitOffset = 44,

		cETC1AbsColor4R2BitOffset = 56,
		cETC1AbsColor4G2BitOffset = 48,
		cETC1AbsColor4B2BitOffset = 40,

		cETC1ColorDeltaMin = -4,
		cETC1ColorDeltaMax = 3,

		// Delta3:
		// 0   1   2   3   4   5   6   7
		// 000 001 010 011 100 101 110 111
		// 0   1   2   3   -4  -3  -2  -1
	};

#define DECLARE_ETC1_INTEN_TABLE(name, N) \
	static const int name[cETC1IntenModifierValues][cETC1SelectorValues] = \
	{ \
		{ N * -8,  N * -2,   N * 2,   N * 8 },{ N * -17,  N * -5,  N * 5,  N * 17 },{ N * -29,  N * -9,   N * 9,  N * 29 },{ N * -42, N * -13, N * 13,  N * 42 }, \
		{ N * -60, N * -18, N * 18,  N * 60 },{ N * -80, N * -24, N * 24,  N * 80 },{ N * -106, N * -33, N * 33, N * 106 },{ N * -183, N * -47, N * 47, N * 183 } \
	};

	DECLARE_ETC1_INTEN_TABLE(g_etc1_inten_tables, 1);
	DECLARE_ETC1_INTEN_TABLE(g_etc1_inten_tables16, 16);
	DECLARE_ETC1_INTEN_TABLE(g_etc1_inten_tables48, 3 * 16);

	//const uint8_t g_etc1_to_selector_index[cETC1SelectorValues] = { 2, 3, 1, 0 };
	const uint8_t g_selector_index_to_etc1[cETC1SelectorValues] = { 3, 2, 0, 1 };
	
	static const uint8_t g_etc_5_to_8[32] = { 0, 8, 16, 24, 33, 41, 49, 57, 66, 74, 82, 90, 99, 107, 115, 123, 132, 140, 148, 156, 165, 173, 181, 189, 198, 206, 214, 222, 231, 239, 247, 255 };

	struct decoder_etc_block
	{
		// big endian uint64:
		// bit ofs:  56  48  40  32  24  16   8   0
		// byte ofs: b0, b1, b2, b3, b4, b5, b6, b7 
		union
		{
			uint64_t m_uint64;

			uint32_t m_uint32[2];

			uint8_t m_bytes[8];

			struct
			{
				signed m_dred2 : 3;
				uint32_t m_red1 : 5;

				signed m_dgreen2 : 3;
				uint32_t m_green1 : 5;

				signed m_dblue2 : 3;
				uint32_t m_blue1 : 5;

				uint32_t m_flip : 1;
				uint32_t m_diff : 1;
				uint32_t m_cw2 : 3;
				uint32_t m_cw1 : 3;

				uint32_t m_selectors;
			} m_differential;
		};

		inline void clear()
		{
			assert(sizeof(*this) == 8);
			basisu::clear_obj(*this);
		}

		inline void set_byte_bits(uint32_t ofs, uint32_t num, uint32_t bits)
		{
			assert((ofs + num) <= 64U);
			assert(num && (num < 32U));
			assert((ofs >> 3) == ((ofs + num - 1) >> 3));
			assert(bits < (1U << num));
			const uint32_t byte_ofs = 7 - (ofs >> 3);
			const uint32_t byte_bit_ofs = ofs & 7;
			const uint32_t mask = (1 << num) - 1;
			m_bytes[byte_ofs] &= ~(mask << byte_bit_ofs);
			m_bytes[byte_ofs] |= (bits << byte_bit_ofs);
		}

		inline void set_flip_bit(bool flip)
		{
			m_bytes[3] &= ~1;
			m_bytes[3] |= static_cast<uint8_t>(flip);
		}

		inline void set_diff_bit(bool diff)
		{
			m_bytes[3] &= ~2;
			m_bytes[3] |= (static_cast<uint32_t>(diff) << 1);
		}

		// Sets intensity modifier table (0-7) used by subblock subblock_id (0 or 1)
		inline void set_inten_table(uint32_t subblock_id, uint32_t t)
		{
			assert(subblock_id < 2);
			assert(t < 8);
			const uint32_t ofs = subblock_id ? 2 : 5;
			m_bytes[3] &= ~(7 << ofs);
			m_bytes[3] |= (t << ofs);
		}

		// Selector "val" ranges from 0-3 and is a direct index into g_etc1_inten_tables.
		inline void set_selector(uint32_t x, uint32_t y, uint32_t val)
		{
			assert((x | y | val) < 4);
			const uint32_t bit_index = x * 4 + y;

			uint8_t* p = &m_bytes[7 - (bit_index >> 3)];

			const uint32_t byte_bit_ofs = bit_index & 7;
			const uint32_t mask = 1 << byte_bit_ofs;

			static const uint8_t s_selector_index_to_etc1[4] = { 3, 2, 0, 1 };
			const uint32_t etc1_val = s_selector_index_to_etc1[val];

			const uint32_t lsb = etc1_val & 1;
			const uint32_t msb = etc1_val >> 1;

			p[0] &= ~mask;
			p[0] |= (lsb << byte_bit_ofs);

			p[-2] &= ~mask;
			p[-2] |= (msb << byte_bit_ofs);
		}

		// Returned encoded selector value ranges from 0-3 (this is NOT a direct index into g_etc1_inten_tables, see get_selector())
		inline uint32_t get_raw_selector(uint32_t x, uint32_t y) const
		{
			assert((x | y) < 4);

			const uint32_t bit_index = x * 4 + y;
			const uint32_t byte_bit_ofs = bit_index & 7;
			const uint8_t* p = &m_bytes[7 - (bit_index >> 3)];
			const uint32_t lsb = (p[0] >> byte_bit_ofs) & 1;
			const uint32_t msb = (p[-2] >> byte_bit_ofs) & 1;
			const uint32_t val = lsb | (msb << 1);

			return val;
		}

		// Returned selector value ranges from 0-3 and is a direct index into g_etc1_inten_tables.
		inline uint32_t get_selector(uint32_t x, uint32_t y) const
		{
			static const uint8_t s_etc1_to_selector_index[cETC1SelectorValues] = { 2, 3, 1, 0 };
			return s_etc1_to_selector_index[get_raw_selector(x, y)];
		}

		inline void set_raw_selector_bits(uint32_t bits)
		{
			m_bytes[4] = static_cast<uint8_t>(bits);
			m_bytes[5] = static_cast<uint8_t>(bits >> 8);
			m_bytes[6] = static_cast<uint8_t>(bits >> 16);
			m_bytes[7] = static_cast<uint8_t>(bits >> 24);
		}

		inline bool are_all_selectors_the_same() const
		{
			uint32_t v = *reinterpret_cast<const uint32_t*>(&m_bytes[4]);

			if ((v == 0xFFFFFFFF) || (v == 0xFFFF) || (!v) || (v == 0xFFFF0000))
				return true;

			return false;
		}

		inline void set_raw_selector_bits(uint8_t byte0, uint8_t byte1, uint8_t byte2, uint8_t byte3)
		{
			m_bytes[4] = byte0;
			m_bytes[5] = byte1;
			m_bytes[6] = byte2;
			m_bytes[7] = byte3;
		}

		inline uint32_t get_raw_selector_bits() const
		{
			return m_bytes[4] | (m_bytes[5] << 8) | (m_bytes[6] << 16) | (m_bytes[7] << 24);
		}

		inline void set_base4_color(uint32_t idx, uint16_t c)
		{
			if (idx)
			{
				set_byte_bits(cETC1AbsColor4R2BitOffset, 4, (c >> 8) & 15);
				set_byte_bits(cETC1AbsColor4G2BitOffset, 4, (c >> 4) & 15);
				set_byte_bits(cETC1AbsColor4B2BitOffset, 4, c & 15);
			}
			else
			{
				set_byte_bits(cETC1AbsColor4R1BitOffset, 4, (c >> 8) & 15);
				set_byte_bits(cETC1AbsColor4G1BitOffset, 4, (c >> 4) & 15);
				set_byte_bits(cETC1AbsColor4B1BitOffset, 4, c & 15);
			}
		}

		inline void set_base5_color(uint16_t c)
		{
			set_byte_bits(cETC1BaseColor5RBitOffset, 5, (c >> 10) & 31);
			set_byte_bits(cETC1BaseColor5GBitOffset, 5, (c >> 5) & 31);
			set_byte_bits(cETC1BaseColor5BBitOffset, 5, c & 31);
		}

		void set_delta3_color(uint16_t c)
		{
			set_byte_bits(cETC1DeltaColor3RBitOffset, 3, (c >> 6) & 7);
			set_byte_bits(cETC1DeltaColor3GBitOffset, 3, (c >> 3) & 7);
			set_byte_bits(cETC1DeltaColor3BBitOffset, 3, c & 7);
		}

		void set_block_color4(const color32& c0_unscaled, const color32& c1_unscaled)
		{
			set_diff_bit(false);

			set_base4_color(0, pack_color4(c0_unscaled, false));
			set_base4_color(1, pack_color4(c1_unscaled, false));
		}

		void set_block_color5(const color32& c0_unscaled, const color32& c1_unscaled)
		{
			set_diff_bit(true);

			set_base5_color(pack_color5(c0_unscaled, false));

			int dr = c1_unscaled.r - c0_unscaled.r;
			int dg = c1_unscaled.g - c0_unscaled.g;
			int db = c1_unscaled.b - c0_unscaled.b;

			set_delta3_color(pack_delta3(dr, dg, db));
		}

		bool set_block_color5_check(const color32& c0_unscaled, const color32& c1_unscaled)
		{
			set_diff_bit(true);

			set_base5_color(pack_color5(c0_unscaled, false));

			int dr = c1_unscaled.r - c0_unscaled.r;
			int dg = c1_unscaled.g - c0_unscaled.g;
			int db = c1_unscaled.b - c0_unscaled.b;

			if (((dr < cETC1ColorDeltaMin) || (dr > cETC1ColorDeltaMax)) ||
				((dg < cETC1ColorDeltaMin) || (dg > cETC1ColorDeltaMax)) ||
				((db < cETC1ColorDeltaMin) || (db > cETC1ColorDeltaMax)))
				return false;

			set_delta3_color(pack_delta3(dr, dg, db));

			return true;
		}

		inline uint32_t get_byte_bits(uint32_t ofs, uint32_t num) const
		{
			assert((ofs + num) <= 64U);
			assert(num && (num <= 8U));
			assert((ofs >> 3) == ((ofs + num - 1) >> 3));
			const uint32_t byte_ofs = 7 - (ofs >> 3);
			const uint32_t byte_bit_ofs = ofs & 7;
			return (m_bytes[byte_ofs] >> byte_bit_ofs) & ((1 << num) - 1);
		}

		inline uint16_t get_base5_color() const
		{
			const uint32_t r = get_byte_bits(cETC1BaseColor5RBitOffset, 5);
			const uint32_t g = get_byte_bits(cETC1BaseColor5GBitOffset, 5);
			const uint32_t b = get_byte_bits(cETC1BaseColor5BBitOffset, 5);
			return static_cast<uint16_t>(b | (g << 5U) | (r << 10U));
		}

		inline uint16_t get_base4_color(uint32_t idx) const
		{
			uint32_t r, g, b;
			if (idx)
			{
				r = get_byte_bits(cETC1AbsColor4R2BitOffset, 4);
				g = get_byte_bits(cETC1AbsColor4G2BitOffset, 4);
				b = get_byte_bits(cETC1AbsColor4B2BitOffset, 4);
			}
			else
			{
				r = get_byte_bits(cETC1AbsColor4R1BitOffset, 4);
				g = get_byte_bits(cETC1AbsColor4G1BitOffset, 4);
				b = get_byte_bits(cETC1AbsColor4B1BitOffset, 4);
			}
			return static_cast<uint16_t>(b | (g << 4U) | (r << 8U));
		}

		inline color32 get_base5_color_unscaled() const
		{
			return color32(m_differential.m_red1, m_differential.m_green1, m_differential.m_blue1, 255);
		}

		inline bool get_flip_bit() const
		{
			return (m_bytes[3] & 1) != 0;
		}

		inline bool get_diff_bit() const
		{
			return (m_bytes[3] & 2) != 0;
		}
				
		inline uint32_t get_inten_table(uint32_t subblock_id) const
		{
			assert(subblock_id < 2);
			const uint32_t ofs = subblock_id ? 2 : 5;
			return (m_bytes[3] >> ofs) & 7;
		}

		inline uint16_t get_delta3_color() const
		{
			const uint32_t r = get_byte_bits(cETC1DeltaColor3RBitOffset, 3);
			const uint32_t g = get_byte_bits(cETC1DeltaColor3GBitOffset, 3);
			const uint32_t b = get_byte_bits(cETC1DeltaColor3BBitOffset, 3);
			return static_cast<uint16_t>(b | (g << 3U) | (r << 6U));
		}
		
		void get_block_colors(color32* pBlock_colors, uint32_t subblock_index) const
		{
			color32 b;

			if (get_diff_bit())
			{
				if (subblock_index)
					unpack_color5(b, get_base5_color(), get_delta3_color(), true, 255);
				else
					unpack_color5(b, get_base5_color(), true);
			}
			else
			{
				b = unpack_color4(get_base4_color(subblock_index), true, 255);
			}

			const int* pInten_table = g_etc1_inten_tables[get_inten_table(subblock_index)];

			pBlock_colors[0].set_noclamp_rgba(clamp255(b.r + pInten_table[0]), clamp255(b.g + pInten_table[0]), clamp255(b.b + pInten_table[0]), 255);
			pBlock_colors[1].set_noclamp_rgba(clamp255(b.r + pInten_table[1]), clamp255(b.g + pInten_table[1]), clamp255(b.b + pInten_table[1]), 255);
			pBlock_colors[2].set_noclamp_rgba(clamp255(b.r + pInten_table[2]), clamp255(b.g + pInten_table[2]), clamp255(b.b + pInten_table[2]), 255);
			pBlock_colors[3].set_noclamp_rgba(clamp255(b.r + pInten_table[3]), clamp255(b.g + pInten_table[3]), clamp255(b.b + pInten_table[3]), 255);
		}

		static uint16_t pack_color4(const color32& color, bool scaled, uint32_t bias = 127U)
		{
			return pack_color4(color.r, color.g, color.b, scaled, bias);
		}

		static uint16_t pack_color4(uint32_t r, uint32_t g, uint32_t b, bool scaled, uint32_t bias = 127U)
		{
			if (scaled)
			{
				r = (r * 15U + bias) / 255U;
				g = (g * 15U + bias) / 255U;
				b = (b * 15U + bias) / 255U;
			}

			r = basisu::minimum(r, 15U);
			g = basisu::minimum(g, 15U);
			b = basisu::minimum(b, 15U);

			return static_cast<uint16_t>(b | (g << 4U) | (r << 8U));
		}

		static uint16_t pack_color5(const color32& color, bool scaled, uint32_t bias = 127U)
		{
			return pack_color5(color.r, color.g, color.b, scaled, bias);
		}

		static uint16_t pack_color5(uint32_t r, uint32_t g, uint32_t b, bool scaled, uint32_t bias = 127U)
		{
			if (scaled)
			{
				r = (r * 31U + bias) / 255U;
				g = (g * 31U + bias) / 255U;
				b = (b * 31U + bias) / 255U;
			}

			r = basisu::minimum(r, 31U);
			g = basisu::minimum(g, 31U);
			b = basisu::minimum(b, 31U);

			return static_cast<uint16_t>(b | (g << 5U) | (r << 10U));
		}

		uint16_t pack_delta3(const color32& color)
		{
			return pack_delta3(color.r, color.g, color.b);
		}

		uint16_t pack_delta3(int r, int g, int b)
		{
			assert((r >= cETC1ColorDeltaMin) && (r <= cETC1ColorDeltaMax));
			assert((g >= cETC1ColorDeltaMin) && (g <= cETC1ColorDeltaMax));
			assert((b >= cETC1ColorDeltaMin) && (b <= cETC1ColorDeltaMax));
			if (r < 0) r += 8;
			if (g < 0) g += 8;
			if (b < 0) b += 8;
			return static_cast<uint16_t>(b | (g << 3) | (r << 6));
		}

		static void unpack_delta3(int& r, int& g, int& b, uint16_t packed_delta3)
		{
			r = (packed_delta3 >> 6) & 7;
			g = (packed_delta3 >> 3) & 7;
			b = packed_delta3 & 7;
			if (r >= 4) r -= 8;
			if (g >= 4) g -= 8;
			if (b >= 4) b -= 8;
		}

		static color32 unpack_color5(uint16_t packed_color5, bool scaled, uint32_t alpha)
		{
			uint32_t b = packed_color5 & 31U;
			uint32_t g = (packed_color5 >> 5U) & 31U;
			uint32_t r = (packed_color5 >> 10U) & 31U;

			if (scaled)
			{
				b = (b << 3U) | (b >> 2U);
				g = (g << 3U) | (g >> 2U);
				r = (r << 3U) | (r >> 2U);
			}

			assert(alpha <= 255);

			return color32(cNoClamp, r, g, b, alpha);
		}

		static void unpack_color5(uint32_t& r, uint32_t& g, uint32_t& b, uint16_t packed_color5, bool scaled)
		{
			color32 c(unpack_color5(packed_color5, scaled, 0));
			r = c.r;
			g = c.g;
			b = c.b;
		}
				
		static void unpack_color5(color32& result, uint16_t packed_color5, bool scaled)
		{
			result = unpack_color5(packed_color5, scaled, 255);
		}

		static bool unpack_color5(color32& result, uint16_t packed_color5, uint16_t packed_delta3, bool scaled, uint32_t alpha)
		{
			int dr, dg, db;
			unpack_delta3(dr, dg, db, packed_delta3);

			int r = ((packed_color5 >> 10U) & 31U) + dr;
			int g = ((packed_color5 >> 5U) & 31U) + dg;
			int b = (packed_color5 & 31U) + db;

			bool success = true;
			if (static_cast<uint32_t>(r | g | b) > 31U)
			{
				success = false;
				r = basisu::clamp<int>(r, 0, 31);
				g = basisu::clamp<int>(g, 0, 31);
				b = basisu::clamp<int>(b, 0, 31);
			}

			if (scaled)
			{
				b = (b << 3U) | (b >> 2U);
				g = (g << 3U) | (g >> 2U);
				r = (r << 3U) | (r >> 2U);
			}

			result.set_noclamp_rgba(r, g, b, basisu::minimum(alpha, 255U));
			return success;
		}

		static color32 unpack_color4(uint16_t packed_color4, bool scaled, uint32_t alpha)
		{
			uint32_t b = packed_color4 & 15U;
			uint32_t g = (packed_color4 >> 4U) & 15U;
			uint32_t r = (packed_color4 >> 8U) & 15U;

			if (scaled)
			{
				b = (b << 4U) | b;
				g = (g << 4U) | g;
				r = (r << 4U) | r;
			}

			return color32(cNoClamp, r, g, b, basisu::minimum(alpha, 255U));
		}

		static void unpack_color4(uint32_t& r, uint32_t& g, uint32_t& b, uint16_t packed_color4, bool scaled)
		{
			color32 c(unpack_color4(packed_color4, scaled, 0));
			r = c.r;
			g = c.g;
			b = c.b;
		}

		static void get_diff_subblock_colors(color32* pDst, uint16_t packed_color5, uint32_t table_idx)
		{
			assert(table_idx < cETC1IntenModifierValues);
			const int* pInten_modifer_table = &g_etc1_inten_tables[table_idx][0];

			uint32_t r, g, b;
			unpack_color5(r, g, b, packed_color5, true);

			const int ir = static_cast<int>(r), ig = static_cast<int>(g), ib = static_cast<int>(b);

			const int y0 = pInten_modifer_table[0];
			pDst[0].set(clamp255(ir + y0), clamp255(ig + y0), clamp255(ib + y0), 255);

			const int y1 = pInten_modifer_table[1];
			pDst[1].set(clamp255(ir + y1), clamp255(ig + y1), clamp255(ib + y1), 255);

			const int y2 = pInten_modifer_table[2];
			pDst[2].set(clamp255(ir + y2), clamp255(ig + y2), clamp255(ib + y2), 255);

			const int y3 = pInten_modifer_table[3];
			pDst[3].set(clamp255(ir + y3), clamp255(ig + y3), clamp255(ib + y3), 255);
		}

		static int clamp255(int x)
		{
			if (x & 0xFFFFFF00)
			{
				if (x < 0)
					x = 0;
				else if (x > 255)
					x = 255;
			}

			return x;
		}

		static void get_block_colors5(color32* pBlock_colors, const color32& base_color5, uint32_t inten_table)
		{
			color32 b(base_color5);

			b.r = (b.r << 3) | (b.r >> 2);
			b.g = (b.g << 3) | (b.g >> 2);
			b.b = (b.b << 3) | (b.b >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			pBlock_colors[0].set(clamp255(b.r + pInten_table[0]), clamp255(b.g + pInten_table[0]), clamp255(b.b + pInten_table[0]), 255);
			pBlock_colors[1].set(clamp255(b.r + pInten_table[1]), clamp255(b.g + pInten_table[1]), clamp255(b.b + pInten_table[1]), 255);
			pBlock_colors[2].set(clamp255(b.r + pInten_table[2]), clamp255(b.g + pInten_table[2]), clamp255(b.b + pInten_table[2]), 255);
			pBlock_colors[3].set(clamp255(b.r + pInten_table[3]), clamp255(b.g + pInten_table[3]), clamp255(b.b + pInten_table[3]), 255);
		}

		static void get_block_color5(const color32& base_color5, uint32_t inten_table, uint32_t index, uint32_t& r, uint32_t &g, uint32_t &b)
		{
			assert(index < 4);

			uint32_t br = (base_color5.r << 3) | (base_color5.r >> 2);
			uint32_t bg = (base_color5.g << 3) | (base_color5.g >> 2);
			uint32_t bb = (base_color5.b << 3) | (base_color5.b >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			r = clamp255(br + pInten_table[index]);
			g = clamp255(bg + pInten_table[index]);
			b = clamp255(bb + pInten_table[index]);
		}

		static void get_block_color5_r(const color32& base_color5, uint32_t inten_table, uint32_t index, uint32_t &r)
		{
			assert(index < 4);
						
			uint32_t br = (base_color5.r << 3) | (base_color5.r >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			r = clamp255(br + pInten_table[index]);
		}

		static void get_block_colors5_g(int* pBlock_colors, const color32& base_color5, uint32_t inten_table)
		{
			const int g = (base_color5.g << 3) | (base_color5.g >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			pBlock_colors[0] = clamp255(g + pInten_table[0]);
			pBlock_colors[1] = clamp255(g + pInten_table[1]);
			pBlock_colors[2] = clamp255(g + pInten_table[2]);
			pBlock_colors[3] = clamp255(g + pInten_table[3]);
		}

		static void get_block_colors5_bounds(color32* pBlock_colors, const color32& base_color5, uint32_t inten_table, uint32_t l = 0, uint32_t h = 3)
		{
			color32 b(base_color5);

			b.r = (b.r << 3) | (b.r >> 2);
			b.g = (b.g << 3) | (b.g >> 2);
			b.b = (b.b << 3) | (b.b >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			pBlock_colors[0].set(clamp255(b.r + pInten_table[l]), clamp255(b.g + pInten_table[l]), clamp255(b.b + pInten_table[l]), 255);
			pBlock_colors[1].set(clamp255(b.r + pInten_table[h]), clamp255(b.g + pInten_table[h]), clamp255(b.b + pInten_table[h]), 255);
		}

		static void get_block_colors5_bounds_g(uint32_t* pBlock_colors, const color32& base_color5, uint32_t inten_table, uint32_t l = 0, uint32_t h = 3)
		{
			color32 b(base_color5);

			b.g = (b.g << 3) | (b.g >> 2);

			const int* pInten_table = g_etc1_inten_tables[inten_table];

			pBlock_colors[0] = clamp255(b.g + pInten_table[l]);
			pBlock_colors[1] = clamp255(b.g + pInten_table[h]);
		}
	};

	enum dxt_constants
	{
		cDXT1SelectorBits = 2U, cDXT1SelectorValues = 1U << cDXT1SelectorBits, cDXT1SelectorMask = cDXT1SelectorValues - 1U,
		cDXT5SelectorBits = 3U, cDXT5SelectorValues = 1U << cDXT5SelectorBits, cDXT5SelectorMask = cDXT5SelectorValues - 1U,
	};

	static const uint8_t g_etc1_x_selector_unpack[4][256] =
	{
		{
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
			0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3, 2, 3,
		},
		{
			0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1,
			2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3,
			0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1,
			2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3,
			0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1,
			2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3,
			0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1,
			2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3, 2, 2, 3, 3,
		},

		{
			0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1,
			0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1,
			2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3,
			2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3,
			0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1,
			0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1,
			2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3,
			2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 3, 3,
		},

		{
			0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
			0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
			0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
			0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
			2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
			2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
			2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
			2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
		}
	};

	struct dxt1_block
	{
		enum { cTotalEndpointBytes = 2, cTotalSelectorBytes = 4 };

		uint8_t m_low_color[cTotalEndpointBytes];
		uint8_t m_high_color[cTotalEndpointBytes];
		uint8_t m_selectors[cTotalSelectorBytes];

		inline void clear() { basisu::clear_obj(*this); }

		inline uint32_t get_high_color() const { return m_high_color[0] | (m_high_color[1] << 8U); }
		inline uint32_t get_low_color() const { return m_low_color[0] | (m_low_color[1] << 8U); }
		inline void set_low_color(uint16_t c) { m_low_color[0] = static_cast<uint8_t>(c & 0xFF); m_low_color[1] = static_cast<uint8_t>((c >> 8) & 0xFF); }
		inline void set_high_color(uint16_t c) { m_high_color[0] = static_cast<uint8_t>(c & 0xFF); m_high_color[1] = static_cast<uint8_t>((c >> 8) & 0xFF); }
		inline uint32_t get_selector(uint32_t x, uint32_t y) const { assert((x < 4U) && (y < 4U)); return (m_selectors[y] >> (x * cDXT1SelectorBits)) & cDXT1SelectorMask; }
		inline void set_selector(uint32_t x, uint32_t y, uint32_t val) { assert((x < 4U) && (y < 4U) && (val < 4U)); m_selectors[y] &= (~(cDXT1SelectorMask << (x * cDXT1SelectorBits))); m_selectors[y] |= (val << (x * cDXT1SelectorBits)); }

		static uint16_t pack_color(const color32& color, bool scaled, uint32_t bias = 127U)
		{
			uint32_t r = color.r, g = color.g, b = color.b;
			if (scaled)
			{
				r = (r * 31U + bias) / 255U;
				g = (g * 63U + bias) / 255U;
				b = (b * 31U + bias) / 255U;
			}
			return static_cast<uint16_t>(basisu::minimum(b, 31U) | (basisu::minimum(g, 63U) << 5U) | (basisu::minimum(r, 31U) << 11U));
		}

		static uint16_t pack_unscaled_color(uint32_t r, uint32_t g, uint32_t b) { return static_cast<uint16_t>(b | (g << 5U) | (r << 11U)); }
	};

	struct dxt_selector_range
	{
		uint32_t m_low;
		uint32_t m_high;
	};

	struct etc1_to_dxt1_56_solution
	{
		uint8_t m_lo;
		uint8_t m_hi;
		uint16_t m_err;
	};

#if BASISD_SUPPORT_DXT1
	static dxt_selector_range g_etc1_to_dxt1_selector_ranges[] =
	{
		{ 0, 3 },

		{ 1, 3 },
		{ 0, 2 },

		{ 1, 2 },

		{ 2, 3 },
		{ 0, 1 },
	};

	const uint32_t NUM_ETC1_TO_DXT1_SELECTOR_RANGES = sizeof(g_etc1_to_dxt1_selector_ranges) / sizeof(g_etc1_to_dxt1_selector_ranges[0]);

	static uint32_t g_etc1_to_dxt1_selector_range_index[4][4];

	const uint32_t NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS = 10;
	static const uint8_t g_etc1_to_dxt1_selector_mappings[NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS][4] =
	{
		{ 0, 0, 1, 1 },
		{ 0, 0, 1, 2 },
		{ 0, 0, 1, 3 },
		{ 0, 0, 2, 3 },
		{ 0, 1, 1, 1 },
		{ 0, 1, 2, 2 },
		{ 0, 1, 2, 3 },
		{ 0, 2, 3, 3 },
		{ 1, 2, 2, 2 },
		{ 1, 2, 3, 3 },
	};
	
	static uint8_t g_etc1_to_dxt1_selector_mappings_raw_dxt1_256[NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS][256];
	static uint8_t g_etc1_to_dxt1_selector_mappings_raw_dxt1_inv_256[NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS][256];

	static const etc1_to_dxt1_56_solution g_etc1_to_dxt_6[32 * 8 * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS * NUM_ETC1_TO_DXT1_SELECTOR_RANGES] = {
/**** start inlining basisu_transcoder_tables_dxt1_6.inc ****/
// Copyright (C) 2017-2019 Binomial LLC. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
{0,4,18},{0,3,4},{0,2,0},{0,2,9},{0,3,36},{0,2,22},{0,2,13},{0,1,24},{0,1,41},{0,1,25},{0,4,18},{0,3,4},{0,2,0},{0,2,9},{1,1,36},{0,2,22},{0,2,13},{0,1,24},{3,0,36},{0,1,24},{0,2,0},{0,2,0},{0,2,0},{0,1,0},{0,1,2},{0,1,1},{0,1,1},{0,0,4},{0,0,4},{0,0,4},{0,2,0},
{0,2,0},{0,2,0},{0,1,0},{0,1,2},{0,1,1},{0,1,1},{0,0,4},{1,0,2},{0,0,4},{2,0,18},{0,3,4},{0,2,0},{0,2,9},{2,0,18},{4,0,18},{0,2,9},{0,1,20},{4,0,18},{0,1,20},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,8,38},{1,6,21},{1,4,24},
{1,4,24},{0,8,52},{0,5,18},{0,4,1},{0,3,24},{0,4,77},{0,3,40},{2,6,22},{1,6,5},{2,4,4},{1,4,8},{4,0,52},{0,5,18},{0,4,1},{0,3,24},{8,0,52},{0,3,24},{1,6,20},{1,6,20},{1,6,20},{1,3,21},{0,6,8},{0,4,1},{0,4,1},{0,2,5},{0,3,24},{0,2,9},{2,4,4},{2,4,4},{2,4,4},{2,3,4},{3,0,8},
{1,3,1},{1,3,1},{1,2,4},{6,0,8},{1,2,4},{5,0,18},{1,6,1},{2,4,0},{0,4,0},{5,0,18},{10,0,18},{0,4,0},{0,3,20},{10,0,18},{0,3,20},{1,0,20},{1,0,20},{1,0,20},{1,0,20},{0,5,0},{0,5,0},{0,5,0},{0,2,1},{0,2,5},{0,2,5},{3,10,38},{3,8,21},{3,6,24},{3,6,24},{2,10,52},{2,7,18},{2,6,1},
{2,5,24},{0,7,53},{1,5,21},{4,8,22},{3,8,5},{4,6,4},{3,6,8},{7,0,52},{2,7,18},{2,6,1},{1,5,20},{14,0,52},{1,5,20},{3,8,20},{3,8,20},{3,8,20},{3,5,21},{2,8,8},{2,6,1},{2,6,1},{2,4,5},{0,6,8},{1,5,5},{4,6,4},{4,6,4},{4,6,4},{4,5,4},{6,0,8},{3,5,1},{3,5,1},{3,4,4},{12,0,8},
{3,4,4},{8,0,18},{3,8,1},{4,6,0},{2,6,0},{8,0,18},{16,0,18},{2,6,0},{0,5,20},{16,0,18},{0,5,20},{3,0,20},{3,0,20},{3,0,20},{3,0,20},{2,7,0},{2,7,0},{2,7,0},{2,4,1},{1,5,1},{1,5,1},{5,12,38},{5,10,21},{5,8,24},{5,8,24},{4,12,52},{4,9,18},{4,8,1},{4,7,24},{2,9,53},{3,7,21},{6,10,22},
{5,10,5},{6,8,4},{5,8,8},{2,16,51},{4,9,18},{4,8,1},{3,7,20},{20,0,51},{3,7,20},{5,10,20},{5,10,20},{5,10,20},{5,7,21},{4,10,8},{4,8,1},{4,8,1},{4,6,5},{2,8,8},{3,7,5},{6,8,4},{6,8,4},{6,8,4},{6,7,4},{9,0,8},{5,7,1},{5,7,1},{5,6,4},{18,0,8},{5,6,4},{11,0,18},{5,10,1},{6,8,0},
{4,8,0},{11,0,18},{22,0,18},{4,8,0},{0,7,20},{22,0,18},{0,7,20},{5,0,20},{5,0,20},{5,0,20},{5,0,20},{4,9,0},{4,9,0},{4,9,0},{4,6,1},{3,7,1},{3,7,1},{7,15,36},{7,12,19},{7,10,28},{7,10,20},{6,15,52},{6,11,22},{6,10,7},{6,9,28},{3,12,52},{5,9,27},{8,13,19},{8,11,3},{8,10,3},{8,10,6},{13,1,51},
{6,11,21},{7,10,3},{5,9,27},{27,0,51},{5,9,27},{7,12,19},{7,12,19},{7,12,19},{7,10,19},{6,13,9},{6,10,6},{6,10,6},{6,9,3},{5,10,9},{5,9,2},{8,10,2},{8,10,2},{8,10,2},{8,9,2},{12,1,8},{7,10,2},{7,10,2},{5,9,2},{25,0,8},{5,9,2},{14,1,18},{7,12,1},{8,10,2},{6,10,2},{14,1,18},{15,7,18},{6,10,2},
{0,9,26},{15,7,18},{0,9,26},{7,0,18},{7,0,18},{7,0,18},{7,0,18},{6,11,2},{6,11,2},{6,11,2},{6,9,2},{5,9,1},{5,9,1},{9,16,38},{9,14,19},{9,12,28},{9,12,20},{8,17,52},{8,13,22},{8,12,7},{8,11,28},{5,14,52},{7,11,27},{10,15,19},{10,13,3},{10,12,3},{10,12,6},{16,1,51},{8,13,21},{9,12,3},{7,11,27},{33,0,51},
{7,11,27},{9,14,19},{9,14,19},{9,14,19},{9,12,19},{8,15,9},{8,12,6},{8,12,6},{8,11,3},{7,12,9},{7,11,2},{10,12,2},{10,12,2},{10,12,2},{10,11,2},{15,1,8},{9,12,2},{9,12,2},{7,11,2},{31,0,8},{7,11,2},{17,0,18},{9,14,1},{10,12,2},{8,12,2},{17,0,18},{34,0,18},{8,12,2},{0,11,26},{34,0,18},{0,11,26},{9,0,18},
{9,0,18},{9,0,18},{9,0,18},{8,13,2},{8,13,2},{8,13,2},{8,11,2},{7,11,1},{7,11,1},{11,18,38},{11,16,19},{11,14,28},{11,14,20},{10,19,52},{10,15,22},{10,14,7},{10,13,28},{7,16,52},{9,13,27},{12,16,21},{12,15,3},{12,14,3},{12,14,6},{19,1,51},{10,15,21},{11,14,3},{9,13,27},{39,0,51},{9,13,27},{11,16,18},{11,16,18},{11,16,18},
{11,14,19},{10,17,9},{10,14,6},{10,14,6},{10,13,3},{9,14,9},{9,13,2},{12,14,2},{12,14,2},{12,14,2},{12,13,2},{15,7,8},{11,14,2},{11,14,2},{9,13,2},{31,3,8},{9,13,2},{20,0,18},{11,16,1},{12,14,2},{10,14,2},{20,0,18},{40,0,18},{10,14,2},{0,13,26},{40,0,18},{0,13,26},{11,0,18},{11,0,18},{11,0,18},{11,0,18},{10,15,2},
{10,15,2},{10,15,2},{10,13,2},{9,13,1},{9,13,1},{13,20,38},{13,18,19},{13,16,27},{13,16,19},{12,21,52},{12,17,19},{12,16,5},{12,15,28},{10,17,52},{11,15,27},{14,18,21},{14,17,3},{14,16,1},{13,16,10},{22,1,51},{12,17,18},{13,16,2},{11,15,27},{45,0,51},{11,15,27},{13,18,18},{13,18,18},{13,18,18},{13,16,19},{12,19,9},{12,16,5},{12,16,5},
{12,15,3},{10,16,11},{11,15,2},{14,16,1},{14,16,1},{14,16,1},{14,15,2},{15,13,8},{13,16,2},{13,16,2},{11,15,2},{31,6,8},{11,15,2},{23,0,18},{13,18,1},{14,16,0},{12,16,0},{23,0,18},{46,0,18},{12,16,0},{0,15,26},{46,0,18},{0,15,26},{13,0,18},{13,0,18},{13,0,18},{13,0,18},{12,17,1},{12,17,1},{12,17,1},{12,15,2},{11,15,1},
{11,15,1},{15,23,38},{15,20,21},{15,18,37},{15,18,21},{15,22,55},{15,19,23},{15,18,5},{14,17,30},{13,19,56},{13,17,28},{16,21,19},{16,19,3},{16,18,3},{16,18,6},{17,17,51},{15,19,19},{15,18,1},{14,17,26},{51,0,51},{14,17,26},{15,21,20},{15,21,20},{15,21,20},{15,18,20},{15,19,14},{15,18,4},{15,18,4},{14,17,5},{13,18,9},{13,17,3},{16,18,2},
{16,18,2},{16,18,2},{16,17,2},{24,1,8},{15,18,0},{15,18,0},{14,17,1},{49,0,8},{14,17,1},{26,1,18},{15,20,1},{16,18,2},{15,18,1},{26,1,18},{53,0,18},{15,18,1},{0,17,26},{53,0,18},{0,17,26},{15,0,20},{15,0,20},{15,0,20},{15,0,20},{15,18,4},{15,18,4},{15,18,4},{14,17,4},{13,17,2},{13,17,2},{17,25,36},{17,22,19},{17,20,28},
{17,20,20},{16,25,52},{16,21,22},{16,20,7},{16,19,28},{15,21,56},{15,19,28},{18,23,19},{18,21,3},{18,20,3},{18,20,6},{20,17,51},{16,21,21},{17,20,3},{14,20,26},{57,0,51},{14,20,26},{17,22,19},{17,22,19},{17,22,19},{17,20,19},{16,23,9},{16,20,6},{16,20,6},{16,19,3},{15,20,9},{15,19,3},{18,20,2},{18,20,2},{18,20,2},{18,19,2},{27,1,8},
{17,20,2},{17,20,2},{15,19,2},{55,0,8},{15,19,2},{29,1,18},{17,22,1},{18,20,2},{16,20,2},{29,1,18},{59,0,18},{16,20,2},{0,19,26},{59,0,18},{0,19,26},{17,0,18},{17,0,18},{17,0,18},{17,0,18},{16,21,2},{16,21,2},{16,21,2},{16,19,2},{15,19,2},{15,19,2},{19,27,36},{19,24,19},{19,22,28},{19,22,20},{18,27,52},{18,23,22},{18,22,7},
{18,21,28},{15,24,56},{17,21,27},{20,25,19},{20,23,3},{20,22,3},{20,22,6},{23,17,51},{18,23,21},{19,22,3},{17,21,27},{63,0,51},{17,21,27},{19,24,19},{19,24,19},{19,24,19},{19,22,19},{18,25,9},{18,22,6},{18,22,6},{18,21,3},{17,22,9},{17,21,2},{20,22,2},{20,22,2},{20,22,2},{20,21,2},{30,1,8},{19,22,2},{19,22,2},{17,21,2},{61,0,8},
{17,21,2},{31,3,18},{19,24,1},{20,22,2},{18,22,2},{31,3,18},{63,1,18},{18,22,2},{0,21,26},{63,1,18},{0,21,26},{19,0,18},{19,0,18},{19,0,18},{19,0,18},{18,23,2},{18,23,2},{18,23,2},{18,21,2},{17,21,1},{17,21,1},{21,29,36},{21,26,19},{21,24,28},{21,24,20},{20,29,52},{20,25,22},{20,24,7},{20,23,28},{17,26,52},{19,23,27},{22,27,19},
{22,25,3},{22,24,3},{22,24,6},{34,1,51},{20,25,21},{21,24,3},{19,23,27},{63,3,51},{19,23,27},{21,26,19},{21,26,19},{21,26,19},{21,24,19},{20,27,9},{20,24,6},{20,24,6},{20,23,3},{19,24,9},{19,23,2},{22,24,2},{22,24,2},{22,24,2},{22,23,2},{33,1,8},{21,24,2},{21,24,2},{19,23,2},{63,2,8},{19,23,2},{31,9,18},{21,26,1},{22,24,2},
{20,24,2},{31,9,18},{63,4,18},{20,24,2},{0,23,26},{63,4,18},{0,23,26},{21,0,18},{21,0,18},{21,0,18},{21,0,18},{20,25,2},{20,25,2},{20,25,2},{20,23,2},{19,23,1},{19,23,1},{23,31,40},{23,29,24},{23,27,33},{23,26,24},{23,30,55},{22,28,24},{23,26,8},{22,26,28},{20,28,51},{21,26,21},{24,29,20},{24,28,1},{24,27,4},{24,26,5},{38,0,51},
{22,28,20},{23,26,4},{20,26,20},{62,7,51},{20,26,20},{23,29,20},{23,29,20},{23,29,20},{23,26,20},{23,28,12},{23,26,4},{23,26,4},{22,25,4},{20,27,9},{22,25,4},{24,27,0},{24,27,0},{24,27,0},{24,26,1},{31,12,8},{23,26,0},{23,26,0},{22,25,0},{62,6,8},{22,25,0},{38,1,18},{24,28,1},{24,27,4},{23,26,4},{38,1,18},{45,16,18},{23,26,4},
{0,26,20},{45,16,18},{0,26,20},{23,0,20},{23,0,20},{23,0,20},{23,0,20},{23,26,4},{23,26,4},{23,26,4},{22,25,4},{21,26,1},{21,26,1},{25,33,38},{25,31,24},{25,29,33},{25,28,24},{25,32,55},{24,30,24},{25,28,8},{24,28,28},{22,30,51},{23,28,21},{26,31,20},{26,30,1},{26,29,4},{26,28,5},{41,0,51},{24,30,20},{25,28,4},{22,28,20},{62,10,51},
{22,28,20},{25,31,20},{25,31,20},{25,31,20},{25,28,20},{25,30,12},{25,28,4},{25,28,4},{24,27,4},{22,29,9},{24,27,4},{26,29,0},{26,29,0},{26,29,0},{26,28,1},{31,18,8},{25,28,0},{25,28,0},{24,27,0},{62,9,8},{24,27,0},{41,1,18},{26,30,1},{26,29,4},{25,28,4},{41,1,18},{51,16,18},{25,28,4},{0,28,20},{51,16,18},{0,28,20},{25,0,20},
{25,0,20},{25,0,20},{25,0,20},{25,28,4},{25,28,4},{25,28,4},{24,27,4},{23,28,1},{23,28,1},{27,35,38},{27,32,21},{27,31,33},{27,30,24},{27,34,55},{26,32,24},{27,30,8},{26,30,28},{25,31,56},{25,30,21},{28,33,18},{28,32,2},{28,31,4},{28,30,5},{44,0,51},{26,32,20},{27,30,4},{24,30,20},{62,13,51},{24,30,20},{27,33,20},{27,33,20},{27,33,20},
{27,30,20},{27,31,14},{27,30,4},{27,30,4},{26,29,4},{24,31,9},{26,29,4},{28,31,0},{28,31,0},{28,31,0},{28,30,1},{34,17,8},{27,30,0},{27,30,0},{26,29,0},{62,12,8},{26,29,0},{44,1,18},{27,32,1},{28,31,4},{27,30,4},{44,1,18},{57,16,18},{27,30,4},{0,30,20},{57,16,18},{0,30,20},{27,0,20},{27,0,20},{27,0,20},{27,0,20},{27,30,4},
{27,30,4},{27,30,4},{26,29,4},{25,30,1},{25,30,1},{29,37,38},{29,34,21},{29,32,37},{29,32,21},{29,36,55},{29,33,23},{29,32,5},{28,32,39},{27,33,56},{26,32,30},{30,35,18},{30,34,2},{30,32,2},{30,32,5},{47,0,51},{29,33,19},{29,32,1},{26,32,26},{46,24,51},{26,32,26},{29,35,20},{29,35,20},{29,35,20},{29,32,20},{29,33,14},{29,32,4},{29,32,4},
{28,31,4},{27,32,9},{28,31,4},{30,33,0},{30,33,0},{30,33,0},{30,31,4},{37,17,8},{29,32,0},{29,32,0},{28,31,0},{62,15,8},{28,31,0},{47,1,18},{29,34,1},{30,32,2},{29,32,1},{47,1,18},{63,16,18},{29,32,1},{0,32,26},{63,16,18},{0,32,26},{29,0,20},{29,0,20},{29,0,20},{29,0,20},{29,32,4},{29,32,4},{29,32,4},{28,31,4},{28,31,4},
{28,31,4},{31,40,44},{31,37,28},{32,35,40},{31,34,31},{31,38,53},{31,35,21},{31,34,7},{31,34,30},{28,36,51},{29,34,21},{32,37,20},{32,36,1},{32,35,4},{32,34,5},{50,0,51},{30,36,19},{31,34,6},{28,34,21},{62,19,51},{28,34,21},{31,38,26},{31,38,26},{31,38,26},{31,34,27},{31,36,9},{31,34,3},{31,34,3},{31,33,2},{29,34,10},{30,33,2},{32,35,0},
{32,35,0},{32,35,0},{32,34,1},{49,0,8},{31,34,2},{31,34,2},{30,33,1},{62,18,8},{30,33,1},{47,8,18},{32,36,1},{32,35,4},{31,34,5},{47,8,18},{62,20,18},{31,34,5},{0,34,20},{62,20,18},{0,34,20},{31,0,26},{31,0,26},{31,0,26},{31,0,26},{31,35,1},{31,35,1},{31,35,1},{31,33,2},{29,34,1},{29,34,1},{33,41,40},{33,39,24},{33,37,33},
{33,36,24},{33,40,55},{32,38,24},{33,36,8},{32,36,28},{30,38,51},{31,36,21},{34,39,20},{34,38,1},{34,37,4},{34,36,5},{53,0,51},{32,38,20},{33,36,4},{30,36,21},{62,22,51},{30,36,21},{33,39,20},{33,39,20},{33,39,20},{33,36,20},{33,38,12},{33,36,4},{33,36,4},{32,35,4},{31,36,10},{32,35,4},{34,37,0},{34,37,0},{34,37,0},{34,36,1},{52,0,8},
{33,36,0},{33,36,0},{32,35,0},{62,21,8},{32,35,0},{47,14,18},{34,38,1},{34,37,4},{33,36,4},{47,14,18},{62,23,18},{33,36,4},{0,36,20},{62,23,18},{0,36,20},{33,0,20},{33,0,20},{33,0,20},{33,0,20},{33,36,4},{33,36,4},{33,36,4},{32,35,4},{31,36,1},{31,36,1},{35,43,40},{35,41,24},{35,39,33},{35,38,24},{35,42,55},{34,40,24},{35,38,8},
{34,38,28},{32,40,51},{33,38,21},{36,41,20},{36,40,1},{36,39,4},{36,38,5},{56,0,51},{34,40,20},{35,38,4},{32,38,20},{62,25,51},{32,38,20},{35,41,20},{35,41,20},{35,41,20},{35,38,20},{35,40,12},{35,38,4},{35,38,4},{34,37,4},{32,39,9},{34,37,4},{36,39,0},{36,39,0},{36,39,0},{36,38,1},{55,0,8},{35,38,0},{35,38,0},{34,37,0},{62,24,8},
{34,37,0},{48,17,18},{36,40,1},{36,39,4},{35,38,4},{48,17,18},{62,26,18},{35,38,4},{0,38,20},{62,26,18},{0,38,20},{35,0,20},{35,0,20},{35,0,20},{35,0,20},{35,38,4},{35,38,4},{35,38,4},{34,37,4},{33,38,1},{33,38,1},{37,45,40},{37,43,24},{37,41,33},{37,40,24},{37,44,55},{36,42,24},{37,40,8},{36,40,28},{34,42,51},{35,40,21},{38,43,20},
{38,42,1},{38,41,4},{38,40,5},{59,0,51},{36,42,20},{37,40,4},{34,40,20},{62,28,51},{34,40,20},{37,43,20},{37,43,20},{37,43,20},{37,40,20},{37,42,12},{37,40,4},{37,40,4},{36,39,4},{34,41,9},{36,39,4},{38,41,0},{38,41,0},{38,41,0},{38,40,1},{58,0,8},{37,40,0},{37,40,0},{36,39,0},{62,27,8},{36,39,0},{51,17,18},{38,42,1},{38,41,4},
{37,40,4},{51,17,18},{62,29,18},{37,40,4},{0,40,20},{62,29,18},{0,40,20},{37,0,20},{37,0,20},{37,0,20},{37,0,20},{37,40,4},{37,40,4},{37,40,4},{36,39,4},{35,40,1},{35,40,1},{40,46,44},{40,44,27},{40,43,28},{39,43,28},{39,47,52},{39,44,22},{39,43,3},{38,42,28},{36,44,53},{37,42,19},{40,46,19},{40,44,2},{40,43,3},{40,42,10},{62,1,51},
{38,44,19},{39,43,2},{37,42,18},{63,31,51},{37,42,18},{40,44,26},{40,44,26},{40,44,26},{40,42,26},{39,44,11},{39,43,2},{39,43,2},{39,41,2},{37,43,11},{38,41,3},{40,44,1},{40,44,1},{40,44,1},{40,42,1},{53,16,8},{40,42,1},{40,42,1},{39,41,1},{63,30,8},{39,41,1},{63,0,18},{40,44,1},{40,43,2},{38,43,1},{63,0,18},{62,32,18},{38,43,1},
{0,42,18},{62,32,18},{0,42,18},{39,0,26},{39,0,26},{39,0,26},{39,0,26},{39,43,1},{39,43,1},{39,43,1},{39,41,1},{37,42,1},{37,42,1},{42,48,44},{42,46,27},{42,45,28},{41,45,28},{41,48,53},{41,46,22},{41,45,3},{40,44,28},{38,46,53},{39,44,19},{42,48,19},{42,46,2},{42,45,3},{42,44,10},{63,5,51},{40,46,19},{41,45,2},{39,44,18},{47,42,51},
{39,44,18},{42,46,26},{42,46,26},{42,46,26},{42,44,26},{41,46,11},{41,45,2},{41,45,2},{41,43,2},{39,45,11},{40,43,3},{42,46,1},{42,46,1},{42,46,1},{42,44,1},{56,16,8},{42,44,1},{42,44,1},{41,43,1},{62,33,8},{41,43,1},{63,6,18},{42,46,1},{42,45,2},{40,45,1},{63,6,18},{62,35,18},{40,45,1},{0,44,18},{62,35,18},{0,44,18},{41,0,26},
{41,0,26},{41,0,26},{41,0,26},{41,45,1},{41,45,1},{41,45,1},{41,43,1},{39,44,1},{39,44,1},{44,50,44},{44,48,26},{44,47,28},{43,47,28},{43,50,53},{43,47,27},{43,47,3},{42,46,28},{40,48,51},{41,46,19},{44,50,19},{44,48,1},{44,47,3},{44,46,10},{63,11,51},{42,48,19},{43,47,2},{41,46,18},{47,45,51},{41,46,18},{44,48,26},{44,48,26},{44,48,26},
{44,46,26},{43,48,9},{43,47,2},{43,47,2},{43,45,2},{41,47,11},{42,45,3},{44,48,1},{44,48,1},{44,48,1},{44,46,1},{59,16,8},{44,46,1},{44,46,1},{43,45,1},{62,36,8},{43,45,1},{63,12,18},{44,48,0},{44,47,2},{42,47,1},{63,12,18},{62,38,18},{42,47,1},{0,46,18},{62,38,18},{0,46,18},{43,0,26},{43,0,26},{43,0,26},{43,0,26},{43,47,1},
{43,47,1},{43,47,1},{43,45,1},{41,46,1},{41,46,1},{46,52,44},{46,50,26},{46,49,31},{45,48,31},{45,52,53},{45,49,21},{45,48,7},{45,48,30},{42,50,51},{43,48,21},{46,52,19},{46,50,1},{46,49,6},{46,48,6},{55,32,51},{44,50,19},{45,48,6},{42,48,21},{46,48,51},{42,48,21},{46,50,26},{46,50,26},{46,50,26},{45,48,27},{45,50,9},{45,48,3},{45,48,3},
{45,47,2},{43,48,10},{44,47,3},{46,50,1},{46,50,1},{46,50,1},{46,48,2},{62,16,8},{45,48,2},{45,48,2},{45,47,1},{62,39,8},{45,47,1},{63,18,18},{46,50,0},{47,48,4},{45,48,5},{63,18,18},{62,41,18},{45,48,5},{0,48,20},{62,41,18},{0,48,20},{45,0,26},{45,0,26},{45,0,26},{45,0,26},{45,49,1},{45,49,1},{45,49,1},{45,47,1},{43,48,1},
{43,48,1},{48,54,44},{48,52,27},{48,51,28},{48,50,35},{47,55,51},{47,52,21},{47,51,3},{47,50,22},{45,52,52},{45,50,19},{48,54,19},{48,52,2},{48,51,3},{48,50,10},{63,23,51},{47,52,21},{47,51,3},{45,50,18},{63,43,51},{45,50,18},{48,52,26},{48,52,26},{48,52,26},{48,50,26},{47,53,8},{47,51,2},{47,51,2},{47,49,1},{45,51,8},{46,49,5},{48,52,1},
{48,52,1},{48,52,1},{48,50,1},{63,21,8},{48,50,1},{48,50,1},{47,49,1},{63,42,8},{47,49,1},{63,25,18},{48,52,1},{48,51,2},{46,51,1},{63,25,18},{63,44,18},{46,51,1},{0,50,18},{63,44,18},{0,50,18},{48,0,26},{48,0,26},{48,0,26},{48,0,26},{47,51,1},{47,51,1},{47,51,1},{47,49,0},{45,50,1},{45,50,1},{50,56,44},{50,54,27},{50,53,28},
{49,53,28},{49,57,52},{49,54,22},{49,53,3},{48,52,28},{47,54,52},{47,52,19},{50,56,19},{50,54,2},{50,53,3},{50,52,10},{63,29,51},{48,54,19},{49,53,2},{47,52,18},{63,46,51},{47,52,18},{50,54,26},{50,54,26},{50,54,26},{50,52,26},{49,54,11},{49,53,2},{49,53,2},{49,51,2},{47,53,8},{48,51,3},{50,54,1},{50,54,1},{50,54,1},{50,52,1},{63,27,8},
{50,52,1},{50,52,1},{49,51,1},{63,45,8},{49,51,1},{63,31,18},{50,54,1},{50,53,2},{48,53,1},{63,31,18},{63,47,18},{48,53,1},{0,52,18},{63,47,18},{0,52,18},{49,0,26},{49,0,26},{49,0,26},{49,0,26},{49,53,1},{49,53,1},{49,53,1},{49,51,1},{47,52,1},{47,52,1},{52,58,44},{52,56,27},{52,55,28},{51,55,28},{51,59,52},{51,56,22},{51,55,3},
{50,54,28},{48,56,53},{49,54,19},{52,58,19},{52,56,2},{52,55,3},{52,54,10},{63,35,51},{50,56,19},{51,55,2},{49,54,18},{63,49,51},{49,54,18},{52,56,26},{52,56,26},{52,56,26},{52,54,26},{51,56,11},{51,55,2},{51,55,2},{51,53,2},{49,55,11},{50,53,3},{52,56,1},{52,56,1},{52,56,1},{52,54,1},{63,33,8},{52,54,1},{52,54,1},{51,53,1},{47,56,8},
{51,53,1},{57,48,18},{52,56,1},{52,55,2},{50,55,1},{57,48,18},{62,50,18},{50,55,1},{0,54,18},{62,50,18},{0,54,18},{51,0,26},{51,0,26},{51,0,26},{51,0,26},{51,55,1},{51,55,1},{51,55,1},{51,53,1},{49,54,1},{49,54,1},{54,60,44},{54,58,27},{54,57,28},{53,57,28},{53,61,52},{53,58,22},{53,57,3},{52,56,28},{50,58,53},{51,56,19},{54,60,19},
{54,58,2},{54,57,3},{54,56,10},{63,41,51},{52,58,19},{53,57,2},{51,56,18},{63,52,51},{51,56,18},{54,58,26},{54,58,26},{54,58,26},{54,56,26},{53,58,11},{53,57,2},{53,57,2},{53,55,2},{51,57,11},{52,55,3},{54,58,1},{54,58,1},{54,58,1},{54,56,1},{63,39,8},{54,56,1},{54,56,1},{53,55,1},{47,59,8},{53,55,1},{60,48,18},{54,58,1},{54,57,2},
{52,57,1},{60,48,18},{62,53,18},{52,57,1},{0,56,18},{62,53,18},{0,56,18},{53,0,26},{53,0,26},{53,0,26},{53,0,26},{53,57,1},{53,57,1},{53,57,1},{53,55,1},{51,56,1},{51,56,1},{56,63,38},{56,61,21},{56,59,24},{56,59,24},{55,63,52},{55,60,18},{55,59,1},{55,58,24},{53,60,53},{54,58,21},{57,61,22},{56,61,5},{57,59,4},{56,59,8},{63,47,52},
{55,60,18},{55,59,1},{54,58,20},{63,55,52},{54,58,20},{56,61,20},{56,61,20},{56,61,20},{56,58,21},{55,61,8},{55,59,1},{55,59,1},{55,57,5},{53,59,8},{54,58,5},{57,59,4},{57,59,4},{57,59,4},{57,58,4},{61,49,8},{56,58,1},{56,58,1},{56,57,4},{63,54,8},{56,57,4},{63,49,18},{56,61,1},{57,59,0},{55,59,0},{63,49,18},{63,56,18},{55,59,0},
{0,58,20},{63,56,18},{0,58,20},{56,0,20},{56,0,20},{56,0,20},{56,0,20},{55,60,0},{55,60,0},{55,60,0},{55,57,1},{54,58,1},{54,58,1},{58,63,56},{58,63,21},{58,61,24},{58,61,24},{58,63,68},{57,62,18},{57,61,1},{57,60,24},{55,62,53},{56,60,21},{59,63,22},{58,63,5},{59,61,4},{58,61,8},{63,53,52},{57,62,18},{57,61,1},{56,60,20},{63,58,52},
{56,60,20},{58,63,20},{58,63,20},{58,63,20},{58,60,21},{57,63,8},{57,61,1},{57,61,1},{57,59,5},{55,61,8},{56,60,5},{59,61,4},{59,61,4},{59,61,4},{59,60,4},{63,51,8},{58,60,1},{58,60,1},{58,59,4},{63,57,8},{58,59,4},{63,55,18},{58,63,1},{59,61,0},{57,61,0},{63,55,18},{63,59,18},{57,61,0},{0,60,20},{63,59,18},{0,60,20},{58,0,20},
{58,0,20},{58,0,20},{58,0,20},{57,62,0},{57,62,0},{57,62,0},{57,59,1},{56,60,1},{56,60,1},{60,63,88},{60,63,40},{60,63,24},{60,63,24},{60,63,88},{59,63,37},{59,63,1},{59,62,24},{58,63,63},{58,62,21},{61,63,40},{61,63,13},{61,63,4},{60,63,8},{63,59,52},{60,63,24},{59,63,1},{58,62,20},{63,61,52},{58,62,20},{60,63,24},{60,63,24},{60,63,24},
{60,62,21},{60,63,24},{59,63,1},{59,63,1},{59,61,5},{57,63,8},{58,62,5},{61,63,4},{61,63,4},{61,63,4},{61,62,4},{63,57,8},{60,62,1},{60,62,1},{60,61,4},{63,60,8},{60,61,4},{63,61,18},{61,63,9},{61,63,0},{59,63,0},{63,61,18},{63,62,18},{59,63,0},{0,62,20},{63,62,18},{0,62,20},{60,0,20},{60,0,20},{60,0,20},{60,0,20},{59,63,1},
{59,63,1},{59,63,1},{59,61,1},{58,62,1},{58,62,1},{62,63,38},{62,63,33},{62,63,29},{62,63,24},{62,63,35},{62,63,25},{62,63,21},{61,63,1},{61,63,23},{60,63,4},{63,63,4},{63,63,4},{63,63,4},{63,63,4},{63,63,4},{63,63,4},{63,63,4},{62,63,0},{63,63,4},{62,63,0},{62,63,29},{62,63,29},{62,63,29},{62,63,24},{62,63,26},{62,63,21},{62,63,21},
{61,63,1},{61,63,14},{60,63,4},{63,63,4},{63,63,4},{63,63,4},{63,63,4},{63,62,4},{63,63,4},{63,63,4},{62,63,0},{62,63,4},{62,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{62,0,20},{62,0,20},{62,0,20},{62,0,20},{61,63,16},{61,63,16},{61,63,16},{61,63,1},{60,63,4},
{60,63,4},{0,8,74},{0,6,10},{0,4,1},{0,4,26},{0,6,154},{0,4,99},{0,3,50},{0,2,115},{0,3,170},{0,2,119},{0,8,74},{0,6,10},{0,4,1},{0,4,26},{3,0,154},{0,4,99},{0,3,50},{0,2,115},{6,0,154},{0,2,115},{0,4,0},{0,4,0},{0,4,0},{0,2,0},{0,2,13},{0,2,4},{0,2,4},{0,1,5},{0,1,14},{0,1,6},{0,4,0},
{0,4,0},{0,4,0},{0,2,0},{1,0,13},{0,2,4},{0,2,4},{0,1,5},{2,0,13},{0,1,5},{4,0,74},{0,6,10},{0,4,1},{0,4,26},{4,0,74},{8,0,74},{0,4,26},{0,3,74},{8,0,74},{0,3,74},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,14,83},{0,10,10},{1,6,27},
{0,6,19},{0,11,243},{0,7,110},{0,5,34},{0,4,139},{0,5,280},{0,4,164},{1,12,75},{1,9,2},{1,6,11},{1,6,18},{5,1,243},{0,7,110},{0,5,34},{0,4,139},{11,0,243},{0,4,139},{0,10,9},{0,10,9},{0,10,9},{0,5,9},{0,6,50},{0,5,9},{0,5,9},{0,3,26},{0,3,66},{0,2,33},{1,8,1},{1,8,1},{1,8,1},{1,4,2},{3,0,50},
{0,5,9},{0,5,9},{0,3,26},{6,0,50},{0,3,26},{7,0,74},{1,9,1},{2,6,1},{0,6,10},{7,0,74},{14,0,74},{0,6,10},{0,5,74},{14,0,74},{0,5,74},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,2,1},{0,2,1},{0,2,1},{0,1,1},{0,1,2},{0,1,2},{1,18,137},{1,12,74},{2,9,98},{1,8,67},{0,16,244},{0,10,78},{0,8,2},
{0,6,115},{0,8,344},{0,6,179},{3,14,75},{3,11,2},{3,8,11},{3,8,18},{8,1,243},{0,10,78},{0,8,2},{0,6,115},{15,1,243},{0,6,115},{1,14,65},{1,14,65},{1,14,65},{1,7,66},{0,12,50},{0,8,1},{0,8,1},{0,5,5},{0,5,104},{0,5,41},{3,10,1},{3,10,1},{3,10,1},{3,6,2},{6,0,50},{0,8,1},{0,8,1},{0,5,5},{12,0,50},
{0,5,5},{2,16,72},{3,11,1},{4,8,1},{0,8,1},{2,16,72},{20,0,72},{0,8,1},{0,7,74},{20,0,72},{0,7,74},{1,0,65},{1,0,65},{1,0,65},{1,0,65},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{0,3,25},{0,3,25},{3,20,146},{3,14,83},{4,11,115},{3,10,76},{2,18,245},{2,12,79},{2,10,3},{1,8,108},{0,11,293},{0,8,103},{5,16,73},
{5,13,2},{5,10,11},{5,10,18},{11,1,243},{1,13,75},{2,10,2},{0,8,94},{15,4,243},{0,8,94},{3,16,74},{3,16,74},{3,16,74},{3,9,75},{2,14,51},{2,10,2},{2,10,2},{2,7,6},{0,9,75},{0,7,6},{5,12,1},{5,12,1},{5,12,1},{5,8,2},{9,0,50},{2,10,1},{2,10,1},{0,7,2},{18,0,50},{0,7,2},{5,16,72},{5,13,1},{6,10,1},
{2,10,1},{5,16,72},{26,0,72},{2,10,1},{0,9,74},{26,0,72},{0,9,74},{3,0,74},{3,0,74},{3,0,74},{3,0,74},{2,10,2},{2,10,2},{2,10,2},{2,6,2},{0,7,5},{0,7,5},{6,21,152},{6,16,82},{6,13,109},{5,12,84},{4,21,243},{4,15,78},{4,13,8},{4,11,108},{0,14,255},{0,11,77},{7,19,72},{7,15,1},{7,13,8},{7,12,13},{6,17,243},
{3,15,72},{5,12,5},{0,11,73},{29,0,243},{0,11,73},{6,16,81},{6,16,81},{6,16,81},{5,12,80},{4,17,50},{4,13,4},{4,13,4},{4,9,5},{0,12,52},{1,10,4},{7,15,0},{7,15,0},{7,15,0},{7,11,0},{12,1,50},{5,12,1},{5,12,1},{3,9,4},{25,0,50},{3,9,4},{16,1,72},{7,15,1},{8,13,4},{5,12,4},{16,1,72},{33,0,72},{5,12,4},
{0,11,72},{33,0,72},{0,11,72},{5,0,80},{5,0,80},{5,0,80},{5,0,80},{4,13,0},{4,13,0},{4,13,0},{4,9,1},{1,10,0},{1,10,0},{8,23,152},{8,18,82},{8,15,109},{7,14,84},{6,23,243},{6,16,75},{6,15,8},{6,13,108},{1,17,244},{2,13,77},{9,21,72},{9,17,1},{9,15,8},{9,14,13},{17,1,243},{5,17,72},{7,14,5},{1,13,72},{35,0,243},
{1,13,72},{8,18,81},{8,18,81},{8,18,81},{7,14,80},{6,19,50},{6,15,4},{6,15,4},{6,11,5},{2,14,52},{3,12,4},{9,16,1},{9,16,1},{9,16,1},{9,13,0},{15,1,50},{7,14,1},{7,14,1},{5,11,4},{31,0,50},{5,11,4},{19,1,72},{9,17,1},{10,15,4},{7,14,4},{19,1,72},{39,0,72},{7,14,4},{0,13,72},{39,0,72},{0,13,72},{7,0,80},
{7,0,80},{7,0,80},{7,0,80},{6,15,0},{6,15,0},{6,15,0},{6,11,1},{3,12,0},{3,12,0},{10,25,152},{10,20,82},{10,17,114},{9,16,82},{8,25,243},{8,18,75},{8,16,2},{8,15,108},{3,19,244},{4,15,77},{11,23,72},{11,19,1},{11,16,10},{11,16,17},{20,1,243},{7,19,72},{8,16,2},{3,15,72},{41,0,243},{3,15,72},{10,20,81},{10,20,81},{10,20,81},
{10,15,81},{8,21,50},{8,16,1},{8,16,1},{8,13,5},{4,16,52},{5,14,4},{11,18,1},{11,18,1},{11,18,1},{11,15,0},{15,7,50},{8,16,1},{8,16,1},{7,13,4},{31,3,50},{7,13,4},{22,1,72},{11,19,1},{12,16,2},{8,16,1},{22,1,72},{45,0,72},{8,16,1},{0,15,72},{45,0,72},{0,15,72},{9,0,80},{9,0,80},{9,0,80},{9,0,80},{8,17,0},
{8,17,0},{8,17,0},{8,13,1},{5,14,0},{5,14,0},{12,27,152},{12,22,82},{12,19,114},{11,18,82},{10,27,243},{10,20,75},{10,18,2},{10,16,106},{5,21,244},{6,17,79},{13,25,72},{13,21,1},{13,18,10},{13,18,17},{23,1,243},{9,21,72},{10,18,2},{5,17,74},{47,0,243},{5,17,74},{12,22,81},{12,22,81},{12,22,81},{12,17,80},{10,23,50},{10,18,1},{10,18,1},
{10,15,5},{6,18,52},{8,15,9},{13,20,1},{13,20,1},{13,20,1},{13,17,1},{15,13,50},{10,18,1},{10,18,1},{9,15,4},{31,6,50},{9,15,4},{25,1,72},{13,21,1},{14,18,2},{10,18,1},{25,1,72},{47,2,72},{10,18,1},{0,17,74},{47,2,72},{0,17,74},{11,0,80},{11,0,80},{11,0,80},{11,0,80},{10,19,0},{10,19,0},{10,19,0},{10,15,1},{7,16,0},
{7,16,0},{14,30,146},{14,24,78},{14,21,114},{14,20,79},{12,30,244},{12,23,79},{12,21,7},{12,19,109},{7,23,244},{9,19,76},{15,28,73},{15,23,2},{15,21,14},{15,20,14},{27,0,243},{11,23,75},{13,20,4},{7,19,73},{46,4,243},{7,19,73},{14,25,74},{14,25,74},{14,25,74},{14,19,75},{12,25,53},{12,21,3},{12,21,3},{12,17,6},{8,21,52},{9,18,6},{15,23,1},
{15,23,1},{15,23,1},{15,19,1},{24,1,50},{13,20,0},{13,20,0},{11,17,5},{49,0,50},{11,17,5},{20,17,72},{15,23,1},{16,21,4},{13,20,4},{20,17,72},{57,0,72},{13,20,4},{0,19,72},{57,0,72},{0,19,72},{14,0,74},{14,0,74},{14,0,74},{14,0,74},{12,22,1},{12,22,1},{12,22,1},{12,17,2},{9,18,2},{9,18,2},{16,31,152},{16,26,81},{16,23,109},
{16,22,88},{14,32,244},{14,25,79},{14,23,7},{14,21,109},{9,25,244},{11,21,76},{17,29,72},{17,25,1},{17,23,8},{17,22,13},{30,0,243},{13,25,75},{15,22,4},{9,21,73},{46,7,243},{9,21,73},{16,27,80},{16,27,80},{16,27,80},{16,21,81},{14,27,53},{14,23,3},{14,23,3},{14,19,6},{10,23,52},{11,20,6},{17,25,0},{17,25,0},{17,25,0},{17,21,0},{27,1,50},
{15,22,0},{15,22,0},{13,19,5},{55,0,50},{13,19,5},{23,17,72},{17,25,1},{18,23,4},{15,22,4},{23,17,72},{63,0,72},{15,22,4},{0,21,72},{63,0,72},{0,21,72},{16,0,80},{16,0,80},{16,0,80},{16,0,80},{14,24,1},{14,24,1},{14,24,1},{14,19,2},{11,20,2},{11,20,2},{18,33,152},{18,28,81},{18,25,109},{17,24,84},{16,33,243},{16,27,78},{16,25,8},
{16,23,108},{11,27,244},{13,23,76},{19,31,72},{19,27,1},{19,25,8},{19,24,13},{32,1,243},{15,27,75},{17,24,5},{11,23,73},{46,10,243},{11,23,73},{18,29,80},{18,29,80},{18,29,80},{17,24,80},{16,29,50},{16,25,4},{16,25,4},{16,21,5},{12,25,52},{13,22,6},{19,27,0},{19,27,0},{19,27,0},{19,23,0},{30,1,50},{17,24,1},{17,24,1},{15,21,5},{61,0,50},
{15,21,5},{34,1,72},{19,27,1},{20,25,4},{17,24,4},{34,1,72},{63,3,72},{17,24,4},{0,23,72},{63,3,72},{0,23,72},{17,0,80},{17,0,80},{17,0,80},{17,0,80},{16,25,0},{16,25,0},{16,25,0},{16,21,1},{13,22,2},{13,22,2},{20,35,152},{20,30,81},{20,27,109},{19,26,84},{18,35,243},{18,29,78},{18,27,8},{18,25,108},{13,29,244},{15,25,76},{21,33,72},
{21,29,1},{21,27,8},{21,26,13},{35,1,243},{17,29,72},{19,26,5},{13,25,73},{46,13,243},{13,25,73},{20,31,80},{20,31,80},{20,31,80},{19,26,80},{18,31,50},{18,27,4},{18,27,4},{18,23,5},{14,27,52},{15,24,6},{21,29,0},{21,29,0},{21,29,0},{21,25,0},{33,1,50},{19,26,1},{19,26,1},{17,23,4},{63,2,50},{17,23,4},{37,1,72},{21,29,1},{22,27,4},
{19,26,4},{37,1,72},{63,6,72},{19,26,4},{0,25,72},{63,6,72},{0,25,72},{19,0,80},{19,0,80},{19,0,80},{19,0,80},{18,27,0},{18,27,0},{18,27,0},{18,23,1},{15,24,2},{15,24,2},{22,38,146},{22,32,78},{22,29,111},{22,28,84},{20,38,244},{20,31,74},{20,29,4},{20,27,100},{14,32,247},{17,27,75},{23,36,73},{23,32,3},{23,29,11},{23,28,14},{39,0,243},
{20,31,73},{20,29,3},{16,27,74},{62,8,243},{16,27,74},{22,33,74},{22,33,74},{22,33,74},{22,28,75},{20,33,53},{20,29,3},{20,29,3},{20,25,10},{16,29,50},{18,26,2},{23,31,2},{23,31,2},{23,31,2},{23,27,2},{31,12,50},{21,28,1},{21,28,1},{18,26,1},{62,6,50},{18,26,1},{41,0,72},{23,32,2},{24,29,2},{19,29,2},{41,0,72},{62,10,72},{19,29,2},
{0,27,74},{62,10,72},{0,27,74},{22,0,74},{22,0,74},{22,0,74},{22,0,74},{20,30,1},{20,30,1},{20,30,1},{20,25,1},{18,26,1},{18,26,1},{24,40,146},{24,34,78},{24,31,111},{24,30,84},{22,40,244},{22,33,79},{22,31,4},{22,29,100},{17,33,244},{19,29,75},{25,38,73},{25,33,2},{25,31,11},{25,30,14},{42,0,243},{21,33,75},{22,31,3},{18,29,74},{62,11,243},
{18,29,74},{24,35,74},{24,35,74},{24,35,74},{24,30,75},{22,35,53},{22,31,3},{22,31,3},{22,27,10},{18,31,50},{20,28,2},{25,33,1},{25,33,1},{25,33,1},{25,29,2},{31,18,50},{23,30,1},{23,30,1},{20,28,1},{62,9,50},{20,28,1},{44,0,72},{25,33,1},{26,31,2},{21,31,2},{44,0,72},{62,13,72},{21,31,2},{0,29,74},{62,13,72},{0,29,74},{24,0,74},
{24,0,74},{24,0,74},{24,0,74},{22,32,1},{22,32,1},{22,32,1},{22,27,1},{20,28,1},{20,28,1},{26,42,146},{26,36,78},{26,33,114},{26,32,79},{24,42,244},{24,35,79},{24,33,7},{24,31,100},{19,35,244},{21,31,75},{27,40,73},{27,35,2},{28,33,13},{27,32,14},{45,0,243},{23,35,75},{25,32,4},{20,31,74},{62,14,243},{20,31,74},{26,37,74},{26,37,74},{26,37,74},
{26,32,75},{24,37,53},{24,33,3},{24,33,3},{24,29,10},{20,33,52},{22,30,2},{27,35,1},{27,35,1},{27,35,1},{27,31,2},{34,17,50},{25,32,0},{25,32,0},{22,30,1},{62,12,50},{22,30,1},{47,0,72},{27,35,1},{28,33,4},{25,32,4},{47,0,72},{46,24,72},{25,32,4},{0,31,74},{46,24,72},{0,31,74},{26,0,74},{26,0,74},{26,0,74},{26,0,74},{24,34,1},
{24,34,1},{24,34,1},{24,29,1},{22,30,1},{22,30,1},{28,44,146},{28,38,78},{28,35,114},{28,34,79},{26,44,244},{26,37,79},{26,35,7},{26,33,109},{21,37,244},{23,33,76},{29,42,73},{29,37,2},{30,35,13},{29,34,14},{47,2,243},{25,37,75},{27,34,4},{21,33,73},{62,17,243},{21,33,73},{28,39,74},{28,39,74},{28,39,74},{28,33,75},{26,39,53},{26,35,3},{26,35,3},
{26,31,10},{22,35,52},{23,32,6},{29,37,1},{29,37,1},{29,37,1},{29,33,1},{37,17,50},{27,34,0},{27,34,0},{23,32,5},{62,15,50},{23,32,5},{49,1,72},{29,37,1},{30,35,4},{27,34,4},{49,1,72},{46,27,72},{27,34,4},{0,33,72},{46,27,72},{0,33,72},{28,0,74},{28,0,74},{28,0,74},{28,0,74},{26,36,1},{26,36,1},{26,36,1},{26,31,1},{23,32,2},
{23,32,2},{30,46,146},{30,41,77},{31,37,121},{30,36,81},{29,45,247},{28,39,77},{28,37,9},{28,35,103},{24,39,248},{25,35,76},{31,44,78},{31,40,4},{32,37,11},{31,36,17},{51,0,243},{28,39,73},{28,37,5},{24,35,74},{47,28,243},{24,35,74},{30,42,72},{30,42,72},{30,42,72},{30,36,72},{29,40,54},{29,36,6},{29,36,6},{28,34,9},{24,37,51},{26,34,2},{31,40,4},
{31,40,4},{31,40,4},{31,35,5},{49,0,50},{29,36,2},{29,36,2},{26,34,1},{62,18,50},{26,34,1},{53,0,72},{31,40,0},{32,37,2},{28,37,1},{53,0,72},{62,22,72},{28,37,1},{0,35,74},{62,22,72},{0,35,74},{30,0,72},{30,0,72},{30,0,72},{30,0,72},{29,36,5},{29,36,5},{29,36,5},{28,33,4},{26,34,1},{26,34,1},{32,48,146},{32,42,79},{32,39,111},
{32,38,84},{31,47,247},{30,41,77},{30,39,9},{30,37,103},{26,41,248},{27,37,76},{33,46,73},{33,42,3},{33,39,11},{33,38,14},{54,0,243},{30,41,73},{30,39,5},{26,37,74},{47,31,243},{26,37,74},{32,43,75},{32,43,75},{32,43,75},{32,38,75},{31,42,54},{31,38,6},{31,38,6},{30,36,9},{26,39,51},{28,36,2},{33,41,2},{33,41,2},{33,41,2},{33,37,2},{52,0,50},
{31,38,2},{31,38,2},{28,36,1},{62,21,50},{28,36,1},{56,0,72},{33,42,2},{34,39,2},{30,39,1},{56,0,72},{62,25,72},{30,39,1},{0,37,74},{62,25,72},{0,37,74},{32,0,74},{32,0,74},{32,0,74},{32,0,74},{31,38,5},{31,38,5},{31,38,5},{30,35,4},{28,36,1},{28,36,1},{34,50,146},{34,44,79},{34,41,111},{34,40,84},{32,50,244},{32,43,74},{32,41,4},
{32,39,100},{28,43,248},{29,39,76},{35,48,73},{35,44,3},{35,41,11},{35,40,14},{57,0,243},{32,43,73},{32,41,3},{28,39,74},{50,32,243},{28,39,74},{34,45,75},{34,45,75},{34,45,75},{34,40,75},{32,46,51},{32,41,3},{32,41,3},{32,37,10},{28,41,51},{30,38,2},{35,43,2},{35,43,2},{35,43,2},{35,39,2},{55,0,50},{33,40,1},{33,40,1},{30,38,1},{62,24,50},
{30,38,1},{59,0,72},{35,44,2},{36,41,2},{31,41,2},{59,0,72},{62,28,72},{31,41,2},{0,39,74},{62,28,72},{0,39,74},{34,0,74},{34,0,74},{34,0,74},{34,0,74},{32,42,1},{32,42,1},{32,42,1},{32,37,1},{30,38,1},{30,38,1},{36,52,146},{36,46,79},{36,43,111},{36,42,84},{34,52,244},{34,45,74},{34,43,4},{34,41,100},{30,45,248},{31,41,76},{37,50,73},
{37,46,3},{37,43,11},{37,42,14},{60,0,243},{34,45,73},{34,43,3},{30,41,74},{56,32,243},{30,41,74},{36,47,75},{36,47,75},{36,47,75},{36,42,75},{34,47,53},{34,43,3},{34,43,3},{34,39,10},{30,43,51},{32,40,2},{37,45,2},{37,45,2},{37,45,2},{37,41,2},{58,0,50},{35,42,1},{35,42,1},{32,40,1},{62,27,50},{32,40,1},{62,0,72},{37,46,2},{38,43,2},
{33,43,2},{62,0,72},{62,31,72},{33,43,2},{0,41,74},{62,31,72},{0,41,74},{36,0,74},{36,0,74},{36,0,74},{36,0,74},{34,44,1},{34,44,1},{34,44,1},{34,39,1},{32,40,1},{32,40,1},{38,54,146},{38,49,77},{39,45,120},{38,45,76},{37,53,247},{36,47,78},{37,45,5},{36,43,100},{31,48,243},{33,43,81},{40,50,78},{39,48,4},{40,45,8},{39,45,20},{63,1,243},
{36,47,74},{37,45,1},{33,43,80},{63,32,243},{33,43,80},{38,50,72},{38,50,72},{38,50,72},{38,44,72},{37,48,54},{37,45,5},{37,45,5},{36,42,8},{32,45,53},{34,42,1},{40,46,4},{40,46,4},{40,46,4},{40,43,4},{53,16,50},{37,45,1},{37,45,1},{35,42,0},{63,30,50},{35,42,0},{63,5,72},{39,48,0},{41,45,1},{36,45,0},{63,5,72},{47,42,72},{36,45,0},
{0,43,80},{47,42,72},{0,43,80},{38,0,72},{38,0,72},{38,0,72},{38,0,72},{37,45,4},{37,45,4},{37,45,4},{37,41,4},{34,42,1},{34,42,1},{40,56,146},{40,51,77},{41,47,120},{40,47,76},{39,55,247},{38,49,77},{39,47,5},{38,45,100},{34,49,248},{35,45,81},{42,52,78},{41,50,4},{42,47,8},{41,47,20},{63,7,243},{38,49,73},{39,47,1},{35,45,80},{63,35,243},
{35,45,80},{40,52,72},{40,52,72},{40,52,72},{40,46,72},{39,50,54},{39,47,5},{39,47,5},{38,44,8},{34,47,53},{36,44,1},{42,48,4},{42,48,4},{42,48,4},{42,45,4},{56,16,50},{39,47,1},{39,47,1},{37,44,0},{62,33,50},{37,44,0},{63,11,72},{41,50,0},{43,47,1},{38,47,0},{63,11,72},{47,45,72},{38,47,0},{0,45,80},{47,45,72},{0,45,80},{40,0,72},
{40,0,72},{40,0,72},{40,0,72},{39,47,4},{39,47,4},{39,47,4},{39,43,4},{36,44,1},{36,44,1},{42,58,146},{42,53,77},{43,49,121},{42,48,81},{41,57,247},{40,51,77},{40,49,9},{40,47,100},{36,51,248},{37,47,81},{44,54,78},{43,52,4},{44,49,6},{43,48,17},{63,13,243},{40,51,73},{40,49,5},{37,47,80},{63,38,243},{37,47,80},{42,54,72},{42,54,72},{42,54,72},
{42,48,72},{41,52,54},{41,48,6},{41,48,6},{40,46,8},{36,49,51},{38,46,1},{44,50,4},{44,50,4},{44,50,4},{44,47,4},{59,16,50},{41,48,2},{41,48,2},{39,46,0},{62,36,50},{39,46,0},{55,32,72},{43,52,0},{44,49,2},{40,49,1},{55,32,72},{46,48,72},{40,49,1},{0,47,80},{46,48,72},{0,47,80},{42,0,72},{42,0,72},{42,0,72},{42,0,72},{41,48,5},
{41,48,5},{41,48,5},{41,45,4},{38,46,1},{38,46,1},{44,60,146},{44,55,77},{45,51,121},{44,50,81},{43,59,247},{42,53,77},{42,51,9},{42,49,103},{38,53,248},{39,49,76},{46,56,78},{45,54,4},{46,51,6},{45,50,17},{63,19,243},{42,53,73},{42,51,5},{38,49,74},{63,41,243},{38,49,74},{44,56,72},{44,56,72},{44,56,72},{44,50,72},{43,54,54},{43,50,6},{43,50,6},
{42,48,9},{38,51,51},{40,48,2},{46,52,4},{46,52,4},{46,52,4},{46,49,4},{62,16,50},{43,50,2},{43,50,2},{40,48,1},{62,39,50},{40,48,1},{58,32,72},{45,54,0},{46,51,2},{42,51,1},{58,32,72},{52,48,72},{42,51,1},{0,49,74},{52,48,72},{0,49,74},{44,0,72},{44,0,72},{44,0,72},{44,0,72},{43,50,5},{43,50,5},{43,50,5},{43,47,4},{40,48,1},
{40,48,1},{46,63,146},{46,57,79},{47,53,115},{46,53,78},{45,61,245},{45,55,77},{45,53,2},{44,51,105},{39,56,243},{42,51,82},{48,59,76},{48,55,9},{48,53,8},{47,53,18},{59,33,243},{44,55,74},{45,53,1},{41,51,80},{55,48,243},{41,51,80},{46,59,74},{46,59,74},{46,59,74},{46,52,75},{45,57,51},{45,53,2},{45,53,2},{44,50,10},{41,53,51},{42,50,1},{48,54,4},
{48,54,4},{48,54,4},{48,51,4},{63,21,50},{45,53,1},{45,53,1},{43,50,0},{63,42,50},{43,50,0},{63,29,72},{47,56,2},{49,53,1},{44,53,1},{63,29,72},{63,46,72},{44,53,1},{0,51,80},{63,46,72},{0,51,80},{46,0,74},{46,0,74},{46,0,74},{46,0,74},{45,53,1},{45,53,1},{45,53,1},{45,49,1},{42,50,1},{42,50,1},{48,63,152},{48,59,77},{49,55,120},
{48,55,76},{47,63,245},{47,57,77},{47,55,2},{46,53,105},{41,58,243},{44,53,82},{50,61,76},{49,58,4},{50,55,8},{49,55,20},{62,33,243},{46,57,74},{47,55,1},{43,53,80},{61,48,243},{43,53,80},{48,60,72},{48,60,72},{48,60,72},{48,54,72},{47,59,51},{47,55,2},{47,55,2},{46,52,10},{43,55,51},{44,52,1},{50,56,4},{50,56,4},{50,56,4},{50,53,4},{63,27,50},
{47,55,1},{47,55,1},{45,52,0},{63,45,50},{45,52,0},{63,35,72},{49,58,0},{51,55,1},{46,55,1},{63,35,72},{63,49,72},{46,55,1},{0,53,80},{63,49,72},{0,53,80},{48,0,72},{48,0,72},{48,0,72},{48,0,72},{47,55,1},{47,55,1},{47,55,1},{47,51,1},{44,52,1},{44,52,1},{51,63,184},{50,61,77},{51,57,120},{50,57,76},{49,63,268},{48,59,78},{49,57,5},
{48,55,100},{43,60,243},{46,55,82},{52,63,76},{51,60,4},{52,57,8},{51,57,20},{63,37,243},{48,59,74},{49,57,1},{45,55,80},{63,50,243},{45,55,80},{50,62,72},{50,62,72},{50,62,72},{50,56,72},{49,60,56},{49,57,5},{49,57,5},{48,54,8},{45,57,51},{46,54,1},{52,58,4},{52,58,4},{52,58,4},{52,55,4},{63,33,50},{49,57,1},{49,57,1},{47,54,0},{47,56,50},
{47,54,0},{63,41,72},{51,60,0},{53,57,1},{48,57,0},{63,41,72},{63,52,72},{48,57,0},{0,55,80},{63,52,72},{0,55,80},{50,0,72},{50,0,72},{50,0,72},{50,0,72},{49,57,4},{49,57,4},{49,57,4},{49,53,4},{46,54,1},{46,54,1},{53,63,226},{52,63,77},{53,59,120},{52,59,76},{52,63,300},{50,61,78},{51,59,5},{50,57,100},{45,62,243},{47,57,84},{54,63,84},
{53,62,4},{54,59,8},{53,59,20},{63,43,243},{49,62,73},{51,59,1},{47,57,80},{63,53,243},{47,57,80},{52,63,76},{52,63,76},{52,63,76},{52,58,72},{51,62,56},{51,59,5},{51,59,5},{50,56,8},{47,59,51},{48,56,1},{54,60,4},{54,60,4},{54,60,4},{54,57,4},{63,39,50},{51,59,1},{51,59,1},{49,56,0},{47,59,50},{49,56,0},{63,47,72},{53,62,0},{55,59,1},
{50,59,0},{63,47,72},{63,55,72},{50,59,0},{0,57,80},{63,55,72},{0,57,80},{52,0,72},{52,0,72},{52,0,72},{52,0,72},{51,59,4},{51,59,4},{51,59,4},{51,55,4},{48,56,1},{48,56,1},{56,63,314},{55,63,115},{55,62,115},{54,61,76},{54,63,364},{53,63,79},{53,61,3},{52,59,108},{48,63,252},{49,60,79},{57,63,115},{56,63,10},{56,61,11},{56,61,18},{63,50,243},
{53,63,78},{53,61,2},{47,60,75},{62,57,243},{47,60,75},{54,63,99},{54,63,99},{54,63,99},{54,60,75},{53,63,69},{53,61,2},{53,61,2},{53,58,6},{49,61,51},{50,58,6},{56,63,1},{56,63,1},{56,63,1},{56,59,2},{61,49,50},{53,61,1},{53,61,1},{51,58,2},{63,54,50},{51,58,2},{63,53,74},{56,63,9},{57,61,1},{53,61,1},{63,53,74},{63,58,74},{53,61,1},
{0,60,74},{63,58,74},{0,60,74},{54,0,74},{54,0,74},{54,0,74},{54,0,74},{53,61,2},{53,61,2},{53,61,2},{53,57,2},{50,59,2},{50,59,2},{57,63,371},{57,63,179},{57,63,115},{56,63,75},{57,63,387},{55,63,123},{55,63,2},{54,61,91},{52,63,286},{51,62,70},{59,63,146},{58,63,59},{58,63,10},{58,63,17},{63,55,221},{57,63,98},{55,63,1},{49,62,66},{63,59,221},
{49,62,66},{57,63,115},{57,63,115},{57,63,115},{56,62,75},{56,63,93},{55,63,2},{55,63,2},{55,60,6},{51,63,51},{52,60,6},{58,63,10},{58,63,10},{58,63,10},{58,61,2},{63,51,50},{55,63,1},{55,63,1},{53,60,2},{63,57,50},{53,60,2},{63,59,61},{60,63,25},{59,63,0},{55,63,0},{63,59,61},{63,61,61},{55,63,0},{0,62,65},{63,61,61},{0,62,65},{56,0,74},
{56,0,74},{56,0,74},{56,0,74},{55,63,2},{55,63,2},{55,63,2},{55,59,2},{52,61,2},{52,61,2},{60,63,259},{59,63,190},{59,63,154},{58,63,90},{59,63,270},{58,63,91},{57,63,35},{57,62,22},{56,63,194},{54,63,11},{61,63,70},{60,63,42},{60,63,26},{60,63,2},{63,59,94},{60,63,42},{59,63,13},{53,63,10},{63,61,94},{53,63,10},{59,63,154},{59,63,154},{59,63,154},
{58,63,90},{58,63,147},{57,63,35},{57,63,35},{57,62,6},{55,63,77},{54,62,6},{60,63,26},{60,63,26},{60,63,26},{60,63,2},{63,57,50},{59,63,13},{59,63,13},{55,62,2},{63,60,50},{55,62,2},{63,62,5},{62,63,4},{62,63,0},{61,63,0},{63,62,5},{62,63,5},{61,63,0},{0,63,9},{62,63,5},{0,63,9},{58,0,74},{58,0,74},{58,0,74},{58,0,74},{57,63,10},
{57,63,10},{57,63,10},{57,61,2},{54,63,2},{54,63,2},{61,63,162},{61,63,135},{61,63,126},{60,63,90},{61,63,154},{60,63,66},{60,63,50},{59,63,2},{59,63,109},{57,63,10},{62,63,19},{62,63,14},{62,63,10},{62,63,5},{63,62,17},{62,63,12},{62,63,8},{59,63,1},{62,63,17},{59,63,1},{61,63,126},{61,63,126},{61,63,126},{60,63,90},{60,63,98},{60,63,50},{60,63,50},
{59,63,2},{58,63,62},{57,63,10},{62,63,10},{62,63,10},{62,63,10},{62,63,5},{63,61,13},{62,63,8},{62,63,8},{59,63,1},{63,62,13},{59,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{60,0,74},{60,0,74},{60,0,74},{60,0,74},{60,63,34},{60,63,34},{60,63,34},{59,63,2},{57,63,10},
{57,63,10},{0,14,202},{0,10,25},{0,7,1},{0,6,74},{0,10,441},{0,6,282},{0,5,133},{0,4,318},{0,5,477},{0,4,343},{0,14,202},{0,10,25},{0,7,1},{0,6,74},{5,0,441},{0,6,282},{0,5,133},{0,4,318},{10,0,441},{0,4,318},{0,7,0},{0,7,0},{0,7,0},{0,3,1},{0,3,41},{0,3,17},{0,3,17},{0,2,26},{0,2,45},{0,1,30},{0,7,0},
{0,7,0},{0,7,0},{0,3,1},{2,0,41},{0,3,17},{0,3,17},{0,2,26},{3,0,41},{0,2,26},{7,0,202},{0,10,25},{0,7,1},{0,6,74},{7,0,202},{14,0,202},{0,6,74},{0,5,202},{14,0,202},{0,5,202},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,20,200},{0,14,1},{0,10,25},
{0,8,41},{0,14,686},{0,9,362},{0,8,141},{0,5,467},{0,6,762},{0,5,503},{0,20,200},{0,14,1},{0,10,25},{0,8,41},{7,0,686},{0,9,362},{0,8,141},{0,5,467},{14,0,686},{0,5,467},{0,13,0},{0,13,0},{0,13,0},{0,6,1},{0,6,145},{0,5,52},{0,5,52},{0,3,89},{0,3,161},{0,3,105},{0,13,0},{0,13,0},{0,13,0},{0,6,1},{3,0,145},
{0,5,52},{0,5,52},{0,3,89},{6,0,145},{0,3,89},{2,16,200},{0,14,1},{2,9,1},{0,8,41},{2,16,200},{20,0,200},{0,8,41},{0,7,202},{20,0,200},{0,7,202},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,24,225},{1,16,27},{1,12,83},{1,11,51},{0,19,724},{0,12,299},{0,10,62},
{0,8,414},{0,9,875},{0,7,500},{2,22,201},{2,16,6},{2,12,26},{1,11,35},{10,0,723},{0,12,299},{0,10,62},{0,8,414},{14,3,723},{0,8,414},{1,17,25},{1,17,25},{1,17,25},{1,9,25},{0,12,162},{0,8,25},{0,8,25},{0,5,61},{0,5,216},{0,5,97},{2,15,1},{2,15,1},{2,15,1},{2,8,2},{6,0,162},{0,8,25},{0,8,25},{0,5,61},{12,0,162},
{0,5,61},{5,16,200},{1,16,2},{4,11,1},{0,11,17},{5,16,200},{26,0,200},{0,11,17},{0,9,202},{26,0,200},{0,9,202},{1,0,25},{1,0,25},{1,0,25},{1,0,25},{0,5,1},{0,5,1},{0,5,1},{0,3,1},{0,2,8},{0,2,8},{2,28,313},{2,19,118},{2,14,203},{2,13,130},{0,25,724},{0,16,236},{0,13,6},{0,10,339},{0,11,984},{0,10,508},{4,24,201},
{4,17,5},{4,14,26},{3,13,35},{13,0,723},{0,16,236},{0,13,6},{0,10,339},{14,6,723},{0,10,339},{2,21,113},{2,21,113},{2,21,113},{2,11,114},{0,18,162},{0,12,2},{0,12,2},{0,7,34},{0,8,280},{0,7,115},{4,17,1},{4,17,1},{4,17,1},{4,10,2},{9,0,162},{0,12,2},{0,12,2},{0,7,34},{18,0,162},{0,7,34},{16,0,200},{3,18,2},{6,13,1},
{0,13,2},{16,0,200},{32,0,200},{0,13,2},{0,11,202},{32,0,200},{0,11,202},{2,0,113},{2,0,113},{2,0,113},{2,0,113},{0,11,1},{0,11,1},{0,11,1},{0,6,1},{0,5,40},{0,5,40},{4,31,408},{4,22,216},{4,16,317},{3,15,216},{1,30,723},{1,19,216},{1,15,9},{0,12,312},{0,15,1000},{0,12,440},{6,27,200},{6,20,1},{7,16,29},{5,15,36},{16,0,723},
{0,19,203},{2,15,5},{0,12,296},{32,0,723},{0,12,296},{4,23,209},{4,23,209},{4,23,209},{4,14,208},{1,23,162},{1,16,4},{1,16,4},{1,10,20},{0,11,294},{0,10,89},{6,19,1},{6,19,1},{6,19,1},{6,13,0},{12,1,162},{2,15,1},{2,15,1},{0,10,8},{25,0,162},{0,10,8},{19,1,200},{6,20,1},{8,15,5},{2,15,4},{19,1,200},{39,0,200},{2,15,4},
{0,13,200},{39,0,200},{0,13,200},{3,0,208},{3,0,208},{3,0,208},{3,0,208},{1,16,0},{1,16,0},{1,16,0},{1,9,1},{0,8,45},{0,8,45},{6,33,408},{6,24,216},{6,18,317},{5,17,213},{3,32,723},{3,21,216},{3,17,2},{2,14,312},{0,18,888},{0,14,293},{8,29,200},{8,22,1},{9,17,26},{8,17,37},{19,0,723},{1,22,200},{3,17,2},{0,14,257},{38,0,723},
{0,14,257},{6,25,209},{6,25,209},{6,25,209},{5,16,208},{3,25,162},{3,17,1},{3,17,1},{3,12,20},{0,15,228},{0,12,20},{8,21,1},{8,21,1},{8,21,1},{8,15,0},{15,1,162},{3,17,1},{3,17,1},{0,12,4},{31,0,162},{0,12,4},{22,1,200},{8,22,1},{10,17,2},{3,17,1},{22,1,200},{45,0,200},{3,17,1},{0,15,200},{45,0,200},{0,15,200},{5,0,208},
{5,0,208},{5,0,208},{5,0,208},{3,18,0},{3,18,0},{3,18,0},{3,11,1},{0,11,13},{0,11,13},{8,35,408},{8,26,216},{8,20,317},{7,19,213},{5,34,723},{5,23,216},{5,19,2},{4,16,298},{0,22,804},{0,16,228},{10,31,200},{10,24,1},{11,19,26},{10,19,37},{22,0,723},{3,24,200},{5,19,2},{0,16,227},{44,0,723},{0,16,227},{8,27,209},{8,27,209},{8,27,209},
{7,18,208},{5,27,162},{5,19,1},{5,19,1},{5,14,20},{0,18,180},{0,15,4},{10,23,1},{10,23,1},{10,23,1},{10,17,1},{15,7,162},{5,19,1},{5,19,1},{2,14,4},{31,3,162},{2,14,4},{25,1,200},{10,24,1},{12,19,2},{5,19,1},{25,1,200},{47,2,200},{5,19,1},{0,17,202},{47,2,200},{0,17,202},{7,0,208},{7,0,208},{7,0,208},{7,0,208},{5,20,0},
{5,20,0},{5,20,0},{5,13,1},{0,15,0},{0,15,0},{10,37,408},{10,28,216},{10,22,317},{9,21,213},{7,36,723},{7,25,216},{7,21,2},{6,18,298},{0,25,748},{1,19,218},{12,33,200},{12,26,1},{13,21,26},{12,21,37},{25,0,723},{5,26,200},{7,21,2},{0,19,211},{50,0,723},{0,19,211},{10,29,209},{10,29,209},{10,29,209},{9,20,208},{7,29,162},{7,21,1},{7,21,1},
{7,16,17},{0,21,164},{3,16,3},{12,25,1},{12,25,1},{12,25,1},{12,19,1},{15,13,162},{7,21,1},{7,21,1},{4,16,1},{31,6,162},{4,16,1},{28,1,200},{12,26,1},{14,21,2},{7,21,1},{28,1,200},{47,5,200},{7,21,1},{0,19,202},{47,5,200},{0,19,202},{9,0,208},{9,0,208},{9,0,208},{9,0,208},{7,22,0},{7,22,0},{7,22,0},{7,15,1},{2,17,0},
{2,17,0},{12,39,404},{12,30,212},{13,24,318},{12,23,215},{9,39,724},{9,27,212},{9,24,7},{8,20,306},{0,28,724},{3,21,210},{14,35,203},{14,28,2},{15,24,22},{14,23,30},{28,1,723},{7,28,203},{10,23,4},{0,21,201},{47,5,723},{0,21,201},{12,32,202},{12,32,202},{12,32,202},{12,22,202},{9,31,165},{9,24,3},{9,24,3},{9,18,14},{2,24,164},{4,19,6},{14,28,1},
{14,28,1},{14,28,1},{14,21,1},{24,1,162},{10,23,0},{10,23,0},{6,18,5},{49,0,162},{6,18,5},{23,17,200},{14,28,1},{16,24,4},{10,23,4},{23,17,200},{63,0,200},{10,23,4},{0,21,200},{63,0,200},{0,21,200},{12,0,202},{12,0,202},{12,0,202},{12,0,202},{9,25,1},{9,25,1},{9,25,1},{9,17,2},{4,19,2},{4,19,2},{14,41,404},{14,32,215},{15,26,318},
{14,25,215},{11,41,724},{11,29,212},{11,26,7},{10,22,306},{2,30,724},{5,23,210},{16,37,200},{16,30,1},{17,26,24},{16,25,40},{31,1,723},{9,30,203},{12,25,4},{2,23,201},{47,8,723},{2,23,201},{14,34,202},{14,34,202},{14,34,202},{14,24,202},{11,33,163},{11,26,3},{11,26,3},{11,20,14},{4,26,164},{6,21,6},{16,30,0},{16,30,0},{16,30,0},{16,23,0},{27,1,162},
{12,25,0},{12,25,0},{8,20,5},{55,0,162},{8,20,5},{34,1,200},{16,30,1},{18,26,4},{12,25,4},{34,1,200},{63,3,200},{12,25,4},{0,23,200},{63,3,200},{0,23,200},{14,0,202},{14,0,202},{14,0,202},{14,0,202},{11,27,1},{11,27,1},{11,27,1},{11,19,2},{6,21,2},{6,21,2},{16,43,408},{16,34,216},{16,28,312},{16,27,221},{13,43,724},{13,31,212},{13,28,7},
{12,24,306},{4,32,724},{7,25,210},{18,39,200},{18,32,1},{19,28,24},{17,27,36},{34,0,723},{11,32,201},{14,27,4},{4,25,201},{47,11,723},{4,25,201},{16,35,209},{16,35,209},{16,35,209},{16,26,208},{13,35,163},{13,28,3},{13,28,3},{13,22,14},{6,28,164},{8,23,6},{18,31,1},{18,31,1},{18,31,1},{18,25,0},{30,1,162},{14,27,0},{14,27,0},{10,22,5},{61,0,162},
{10,22,5},{37,1,200},{18,32,1},{20,28,4},{14,27,4},{37,1,200},{63,6,200},{14,27,4},{0,25,200},{63,6,200},{0,25,200},{16,0,208},{16,0,208},{16,0,208},{16,0,208},{13,29,1},{13,29,1},{13,29,1},{13,21,2},{8,23,2},{8,23,2},{18,45,408},{18,36,216},{18,30,312},{17,29,216},{15,45,724},{15,33,210},{15,30,7},{14,26,306},{6,34,724},{9,27,210},{20,41,200},
{20,34,1},{21,30,24},{19,29,36},{37,0,723},{13,34,201},{16,29,5},{6,27,201},{47,14,723},{6,27,201},{18,37,209},{18,37,209},{18,37,209},{18,28,208},{15,37,163},{15,30,3},{15,30,3},{15,24,14},{8,30,164},{10,25,6},{20,33,1},{20,33,1},{20,33,1},{20,27,0},{33,1,162},{16,29,1},{16,29,1},{12,24,5},{63,2,162},{12,24,5},{40,1,200},{20,34,1},{22,30,4},
{16,29,4},{40,1,200},{63,9,200},{16,29,4},{0,27,200},{63,9,200},{0,27,200},{17,0,208},{17,0,208},{17,0,208},{17,0,208},{15,31,1},{15,31,1},{15,31,1},{15,23,2},{10,25,2},{10,25,2},{20,47,404},{20,38,212},{21,32,318},{20,31,215},{17,47,724},{17,35,212},{17,32,7},{16,29,308},{8,36,724},{11,29,210},{22,44,201},{22,36,2},{23,32,22},{22,31,35},{40,1,723},
{15,36,206},{17,32,6},{9,29,202},{63,9,723},{9,29,202},{20,40,202},{20,40,202},{20,40,202},{20,30,203},{17,39,165},{17,32,3},{17,32,3},{17,26,19},{10,32,164},{13,27,2},{22,36,1},{22,36,1},{22,36,1},{22,29,2},{31,12,162},{18,31,1},{18,31,1},{13,27,1},{62,6,162},{13,27,1},{44,0,200},{22,36,1},{24,32,4},{16,32,4},{44,0,200},{62,13,200},{16,32,4},
{0,29,202},{62,13,200},{0,29,202},{20,0,202},{20,0,202},{20,0,202},{20,0,202},{17,33,1},{17,33,1},{17,33,1},{17,25,1},{13,27,1},{13,27,1},{22,49,404},{22,40,212},{23,34,318},{22,33,215},{19,49,724},{19,37,212},{19,34,7},{18,31,308},{10,38,724},{13,31,210},{24,46,201},{24,38,2},{25,34,22},{24,33,30},{43,1,723},{17,38,203},{20,33,4},{11,31,202},{63,12,723},
{11,31,202},{22,42,202},{22,42,202},{22,42,202},{22,32,202},{19,41,165},{19,34,3},{19,34,3},{19,28,19},{12,34,164},{15,29,2},{24,38,1},{24,38,1},{24,38,1},{24,31,2},{31,18,162},{20,33,0},{20,33,0},{15,29,1},{62,9,162},{15,29,1},{47,0,200},{24,38,1},{26,34,4},{20,33,4},{47,0,200},{46,24,200},{20,33,4},{0,31,202},{46,24,200},{0,31,202},{22,0,202},
{22,0,202},{22,0,202},{22,0,202},{19,35,1},{19,35,1},{19,35,1},{19,27,1},{15,29,1},{15,29,1},{24,51,404},{24,42,212},{25,36,318},{24,35,215},{21,51,724},{21,39,212},{21,36,7},{20,32,306},{12,40,724},{15,33,213},{26,47,203},{26,40,2},{27,36,22},{26,35,30},{46,1,723},{19,40,203},{22,35,4},{13,33,200},{63,15,723},{13,33,200},{24,44,202},{24,44,202},{24,44,202},
{24,34,202},{21,43,165},{21,36,3},{21,36,3},{21,30,19},{14,36,164},{17,31,2},{26,40,1},{26,40,1},{26,40,1},{26,33,1},{34,17,162},{22,35,0},{22,35,0},{17,31,1},{62,12,162},{17,31,1},{49,1,200},{26,40,1},{28,36,4},{22,35,4},{49,1,200},{46,27,200},{22,35,4},{0,33,200},{46,27,200},{0,33,200},{24,0,202},{24,0,202},{24,0,202},{24,0,202},{21,37,1},
{21,37,1},{21,37,1},{21,29,1},{17,31,1},{17,31,1},{26,53,404},{26,44,212},{27,38,318},{26,37,215},{23,53,724},{23,41,212},{23,38,7},{22,34,306},{14,42,724},{17,35,210},{28,49,203},{28,42,2},{29,38,22},{28,37,30},{47,5,723},{21,42,203},{24,37,4},{15,35,200},{63,18,723},{15,35,200},{26,46,202},{26,46,202},{26,46,202},{26,36,202},{23,45,165},{23,38,3},{23,38,3},
{23,32,14},{16,38,164},{18,33,6},{28,42,1},{28,42,1},{28,42,1},{28,35,1},{37,17,162},{24,37,0},{24,37,0},{20,32,5},{62,15,162},{20,32,5},{52,1,200},{28,42,1},{30,38,4},{24,37,4},{52,1,200},{46,30,200},{24,37,4},{0,35,200},{46,30,200},{0,35,200},{26,0,202},{26,0,202},{26,0,202},{26,0,202},{23,39,1},{23,39,1},{23,39,1},{23,31,1},{18,33,2},
{18,33,2},{28,56,400},{28,46,216},{29,40,314},{28,39,213},{26,53,728},{25,44,215},{25,40,9},{25,37,306},{17,44,728},{19,37,210},{31,50,204},{30,45,4},{31,40,21},{30,39,29},{52,1,723},{24,44,201},{25,40,5},{17,37,202},{46,30,723},{17,37,202},{28,49,200},{28,49,200},{28,49,200},{28,38,201},{26,46,166},{26,39,6},{26,39,6},{25,34,17},{18,40,163},{21,35,2},{31,43,4},
{31,43,4},{31,43,4},{31,37,4},{49,0,162},{26,39,2},{26,39,2},{21,35,1},{62,18,162},{21,35,1},{56,0,200},{30,45,0},{32,40,2},{25,40,1},{56,0,200},{62,25,200},{25,40,1},{0,37,202},{62,25,200},{0,37,202},{28,0,200},{28,0,200},{28,0,200},{28,0,200},{26,39,5},{26,39,5},{26,39,5},{25,33,4},{21,35,1},{21,35,1},{30,58,400},{30,48,217},{31,42,314},
{30,41,213},{28,55,728},{27,46,215},{27,42,9},{27,39,306},{19,46,728},{21,39,210},{32,54,201},{32,47,3},{33,42,19},{32,41,35},{55,1,723},{26,46,201},{27,42,5},{19,39,202},{47,32,723},{19,39,202},{30,51,200},{30,51,200},{30,51,200},{30,40,201},{28,48,166},{28,41,6},{28,41,6},{27,36,17},{20,42,163},{23,37,2},{32,46,2},{32,46,2},{32,46,2},{32,39,2},{52,0,162},
{28,41,2},{28,41,2},{23,37,1},{62,21,162},{23,37,1},{59,0,200},{32,47,2},{34,42,2},{27,42,1},{59,0,200},{62,28,200},{27,42,1},{0,39,202},{62,28,200},{0,39,202},{30,0,200},{30,0,200},{30,0,200},{30,0,200},{28,41,5},{28,41,5},{28,41,5},{27,35,4},{23,37,1},{23,37,1},{32,60,402},{32,50,212},{32,44,324},{32,43,215},{30,57,728},{29,47,216},{29,44,9},
{29,41,306},{20,48,724},{23,41,210},{34,56,201},{34,48,2},{35,44,19},{34,43,35},{58,1,723},{28,48,201},{29,44,5},{21,41,202},{53,32,723},{21,41,202},{32,52,202},{32,52,202},{32,52,202},{32,42,203},{30,50,166},{30,43,6},{30,43,6},{29,38,17},{22,44,163},{25,39,2},{34,48,1},{34,48,1},{34,48,1},{34,41,2},{55,0,162},{30,43,2},{30,43,2},{25,39,1},{62,24,162},
{25,39,1},{62,0,200},{34,48,1},{36,44,2},{29,44,1},{62,0,200},{62,31,200},{29,44,1},{0,41,202},{62,31,200},{0,41,202},{32,0,202},{32,0,202},{32,0,202},{32,0,202},{30,43,5},{30,43,5},{30,43,5},{29,37,4},{25,39,1},{25,39,1},{34,62,402},{34,52,212},{34,46,324},{34,45,215},{31,61,728},{31,49,210},{31,46,9},{31,43,306},{22,50,724},{25,43,210},{36,58,201},
{36,50,2},{37,46,19},{36,45,35},{61,1,723},{30,50,201},{31,46,5},{23,43,202},{59,32,723},{23,43,202},{34,54,202},{34,54,202},{34,54,202},{34,44,203},{31,54,166},{31,46,8},{31,46,8},{31,40,17},{24,46,163},{27,41,2},{36,50,1},{36,50,1},{36,50,1},{36,43,2},{58,0,162},{32,45,1},{32,45,1},{27,41,1},{62,27,162},{27,41,1},{63,4,200},{36,50,1},{38,46,2},
{31,46,1},{63,4,200},{62,34,200},{31,46,1},{0,43,202},{62,34,200},{0,43,202},{34,0,202},{34,0,202},{34,0,202},{34,0,202},{31,47,5},{31,47,5},{31,47,5},{31,39,4},{27,41,1},{27,41,1},{36,63,408},{36,54,216},{37,48,314},{36,47,217},{34,62,727},{33,52,215},{33,48,9},{32,45,308},{24,53,723},{28,45,217},{39,58,206},{38,53,4},{39,48,21},{38,47,36},{63,4,723},
{32,52,201},{33,48,5},{26,45,208},{62,34,723},{26,45,208},{36,57,200},{36,57,200},{36,57,200},{36,47,201},{34,54,166},{33,48,8},{33,48,8},{33,43,20},{26,48,163},{29,43,1},{39,51,4},{39,51,4},{39,51,4},{39,45,4},{53,16,162},{35,47,4},{35,47,4},{30,43,0},{63,30,162},{30,43,0},{63,11,200},{38,53,0},{40,48,2},{33,48,1},{63,11,200},{47,45,200},{33,48,1},
{0,45,208},{47,45,200},{0,45,208},{36,0,200},{36,0,200},{36,0,200},{36,0,200},{34,47,5},{34,47,5},{34,47,5},{34,41,4},{29,43,1},{29,43,1},{39,63,440},{38,56,216},{39,50,314},{38,49,213},{36,63,728},{35,54,215},{35,50,9},{34,47,308},{26,55,723},{30,47,217},{41,60,206},{40,55,4},{41,50,21},{40,49,29},{63,10,723},{34,54,201},{35,50,5},{28,47,208},{62,37,723},
{28,47,208},{38,59,200},{38,59,200},{38,59,200},{38,48,201},{36,56,166},{36,49,6},{36,49,6},{35,45,20},{28,50,163},{31,45,1},{41,53,4},{41,53,4},{41,53,4},{41,47,4},{56,16,162},{36,49,2},{36,49,2},{32,45,0},{62,33,162},{32,45,0},{55,32,200},{40,55,0},{42,50,2},{35,50,1},{55,32,200},{46,48,200},{35,50,1},{0,47,208},{46,48,200},{0,47,208},{38,0,200},
{38,0,200},{38,0,200},{38,0,200},{36,49,5},{36,49,5},{36,49,5},{36,43,4},{31,45,1},{31,45,1},{41,63,482},{40,58,216},{41,52,314},{40,51,213},{39,63,760},{37,56,215},{37,52,9},{37,49,306},{28,57,723},{31,49,215},{43,62,206},{42,57,4},{43,52,21},{42,51,29},{63,16,723},{36,56,201},{37,52,5},{29,49,203},{62,40,723},{29,49,203},{40,61,200},{40,61,200},{40,61,200},
{40,50,201},{38,58,166},{38,51,6},{38,51,6},{37,47,20},{30,52,163},{33,47,1},{43,55,4},{43,55,4},{43,55,4},{43,49,4},{59,16,162},{38,51,2},{38,51,2},{34,47,0},{62,36,162},{34,47,0},{58,32,200},{42,57,0},{44,52,2},{37,52,1},{58,32,200},{52,48,200},{37,52,1},{0,49,202},{52,48,200},{0,49,202},{40,0,200},{40,0,200},{40,0,200},{40,0,200},{38,51,5},
{38,51,5},{38,51,5},{38,45,4},{33,47,1},{33,47,1},{43,63,530},{42,60,216},{43,54,314},{42,53,213},{41,63,799},{39,58,215},{39,54,9},{39,51,306},{30,59,723},{33,51,210},{45,63,212},{44,59,4},{45,54,21},{44,53,29},{63,22,723},{38,58,201},{39,54,5},{31,51,203},{62,43,723},{31,51,203},{42,63,200},{42,63,200},{42,63,200},{42,52,201},{40,60,166},{40,53,6},{40,53,6},
{39,48,17},{32,54,163},{35,49,2},{45,57,4},{45,57,4},{45,57,4},{45,51,4},{62,16,162},{40,53,2},{40,53,2},{35,49,1},{62,39,162},{35,49,1},{61,32,200},{44,59,0},{46,54,2},{39,54,1},{61,32,200},{58,48,200},{39,54,1},{0,51,202},{58,48,200},{0,51,202},{42,0,200},{42,0,200},{42,0,200},{42,0,200},{40,53,5},{40,53,5},{40,53,5},{40,47,4},{35,49,1},
{35,49,1},{46,63,626},{44,63,222},{45,57,315},{44,56,211},{44,63,869},{41,60,213},{42,56,2},{41,53,298},{32,61,723},{36,53,217},{48,63,244},{47,60,5},{47,56,26},{46,55,34},{61,32,723},{40,60,202},{42,56,1},{34,53,208},{58,48,723},{34,53,208},{44,63,218},{44,63,218},{44,63,218},{44,55,203},{42,63,163},{42,56,2},{42,56,2},{41,51,25},{35,56,163},{37,51,1},{47,60,1},
{47,60,1},{47,60,1},{47,53,2},{63,21,162},{42,56,1},{42,56,1},{38,51,0},{63,42,162},{38,51,0},{63,35,200},{46,61,2},{49,56,1},{41,56,1},{63,35,200},{63,49,200},{41,56,1},{0,53,208},{63,49,200},{0,53,208},{44,0,202},{44,0,202},{44,0,202},{44,0,202},{42,56,1},{42,56,1},{42,56,1},{42,49,1},{37,51,1},{37,51,1},{48,63,728},{47,63,238},{47,59,315},
{46,58,211},{46,63,937},{43,62,213},{44,58,2},{43,55,298},{34,63,723},{38,55,217},{50,63,286},{48,63,4},{49,58,20},{48,57,36},{63,34,723},{42,62,202},{44,58,1},{36,55,208},{62,49,723},{36,55,208},{47,63,234},{47,63,234},{47,63,234},{46,57,203},{44,63,181},{44,58,2},{44,58,2},{43,53,25},{37,58,163},{39,53,1},{49,61,4},{49,61,4},{49,61,4},{49,55,4},{63,27,162},
{44,58,1},{44,58,1},{40,53,0},{63,45,162},{40,53,0},{63,41,200},{48,63,0},{51,58,1},{43,58,1},{63,41,200},{63,52,200},{43,58,1},{0,55,208},{63,52,200},{0,55,208},{46,0,202},{46,0,202},{46,0,202},{46,0,202},{44,58,1},{44,58,1},{44,58,1},{44,51,1},{39,53,1},{39,53,1},{50,63,866},{49,63,317},{49,61,312},{48,60,216},{48,63,1048},{46,63,219},{46,60,2},
{45,57,298},{38,63,748},{40,57,217},{53,63,350},{51,63,20},{51,60,20},{50,59,36},{63,40,723},{46,63,218},{46,60,1},{38,57,208},{62,52,723},{38,57,208},{48,63,264},{48,63,264},{48,63,264},{48,59,201},{47,63,205},{46,60,2},{46,60,2},{45,55,25},{39,60,163},{41,55,1},{51,63,4},{51,63,4},{51,63,4},{51,57,4},{63,33,162},{46,60,1},{46,60,1},{42,55,0},{47,56,162},
{42,55,0},{63,47,200},{52,63,13},{53,60,1},{45,60,1},{63,47,200},{63,55,200},{45,60,1},{0,57,208},{63,55,200},{0,57,208},{48,0,200},{48,0,200},{48,0,200},{48,0,200},{46,60,1},{46,60,1},{46,60,1},{46,53,1},{41,55,1},{41,55,1},{53,63,1026},{51,63,440},{51,63,312},{50,62,216},{51,63,1144},{48,63,280},{48,62,5},{47,59,298},{43,63,827},{42,59,217},{55,63,428},
{54,63,100},{53,62,20},{52,61,36},{63,46,723},{49,63,266},{48,62,1},{40,59,208},{62,55,723},{40,59,208},{51,63,296},{51,63,296},{51,63,296},{50,61,201},{49,63,248},{48,62,5},{48,62,5},{47,57,25},{41,62,163},{43,57,1},{53,63,13},{53,63,13},{53,63,13},{53,59,4},{63,39,162},{48,62,1},{48,62,1},{44,57,0},{47,59,162},{44,57,0},{63,53,200},{55,63,53},{55,62,1},
{47,62,1},{63,53,200},{63,58,200},{47,62,1},{0,59,208},{63,58,200},{0,59,208},{50,0,200},{50,0,200},{50,0,200},{50,0,200},{48,62,4},{48,62,4},{48,62,4},{48,55,4},{43,57,1},{43,57,1},{54,63,955},{54,63,523},{53,63,362},{52,63,202},{54,63,1027},{51,63,283},{50,63,14},{49,60,198},{47,63,735},{44,61,118},{57,63,338},{56,63,141},{56,63,41},{55,63,17},{63,51,546},
{53,63,233},{51,63,2},{43,61,113},{63,57,546},{43,61,113},{53,63,362},{53,63,362},{53,63,362},{52,63,202},{51,63,315},{50,63,14},{50,63,14},{49,59,19},{44,63,171},{46,59,3},{56,63,41},{56,63,41},{56,63,41},{55,61,2},{61,49,162},{51,63,2},{51,63,2},{47,59,1},{63,54,162},{47,59,1},{63,57,113},{58,63,50},{57,63,1},{52,63,0},{63,57,113},{63,60,113},{52,63,0},
{0,61,113},{63,60,113},{0,61,113},{52,0,202},{52,0,202},{52,0,202},{52,0,202},{50,63,5},{50,63,5},{50,63,5},{50,57,2},{45,60,0},{45,60,0},{57,63,779},{56,63,542},{56,63,442},{54,63,227},{56,63,830},{54,63,251},{53,63,78},{51,62,70},{51,63,587},{47,62,25},{59,63,218},{58,63,123},{58,63,74},{57,63,2},{63,55,333},{56,63,145},{55,63,25},{47,62,25},{63,59,333},
{47,62,25},{56,63,442},{56,63,442},{56,63,442},{54,63,227},{54,63,371},{53,63,78},{53,63,78},{51,61,19},{48,63,219},{47,61,6},{58,63,74},{58,63,74},{58,63,74},{57,63,2},{63,51,162},{55,63,25},{55,63,25},{48,61,2},{63,57,162},{48,61,2},{63,60,25},{61,63,10},{60,63,1},{58,63,0},{63,60,25},{62,62,25},{58,63,0},{0,62,25},{62,62,25},{0,62,25},{54,0,202},
{54,0,202},{54,0,202},{54,0,202},{52,63,26},{52,63,26},{52,63,26},{52,59,2},{47,62,0},{47,62,0},{59,63,684},{57,63,538},{57,63,474},{57,63,282},{57,63,682},{56,63,285},{55,63,171},{53,63,18},{54,63,482},{49,63,5},{60,63,153},{60,63,105},{60,63,89},{59,63,37},{63,59,193},{58,63,107},{58,63,58},{50,63,1},{63,61,193},{50,63,1},{57,63,474},{57,63,474},{57,63,474},
{57,63,282},{57,63,426},{55,63,171},{55,63,171},{53,63,18},{52,63,278},{49,63,5},{60,63,89},{60,63,89},{60,63,89},{59,63,37},{63,57,145},{58,63,58},{58,63,58},{50,63,1},{63,60,145},{50,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{56,0,202},{56,0,202},{56,0,202},{56,0,202},{54,63,50},
{54,63,50},{54,63,50},{54,61,2},{49,63,5},{49,63,5},{60,63,426},{59,63,375},{59,63,339},{58,63,251},{59,63,415},{57,63,202},{57,63,138},{56,63,2},{55,63,295},{53,63,29},{62,63,43},{62,63,38},{62,63,34},{61,63,10},{63,61,54},{61,63,27},{60,63,17},{56,63,1},{63,62,54},{56,63,1},{59,63,339},{59,63,339},{59,63,339},{58,63,251},{59,63,294},{57,63,138},{57,63,138},
{56,63,2},{55,63,174},{53,63,29},{62,63,34},{62,63,34},{62,63,34},{61,63,10},{63,60,41},{60,63,17},{60,63,17},{56,63,1},{62,62,41},{56,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{58,0,202},{58,0,202},{58,0,202},{58,0,202},{57,63,74},{57,63,74},{57,63,74},{56,63,2},{53,63,29},
{53,63,29},{0,20,421},{0,14,50},{0,10,4},{0,9,157},{0,14,925},{0,9,589},{0,8,264},{0,6,701},{0,6,1005},{0,5,738},{0,20,421},{0,14,50},{0,10,4},{0,9,157},{7,0,925},{0,9,589},{0,8,264},{0,6,701},{14,0,925},{0,6,701},{0,10,0},{0,10,0},{0,10,0},{0,5,0},{0,5,85},{0,4,34},{0,4,34},{0,2,50},{0,2,93},{0,2,54},{0,10,0},
{0,10,0},{0,10,0},{0,5,0},{2,1,85},{0,4,34},{0,4,34},{0,2,50},{5,0,85},{0,2,50},{10,1,421},{0,14,50},{0,10,4},{0,9,157},{10,1,421},{20,0,421},{0,9,157},{0,7,421},{20,0,421},{0,7,421},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,26,421},{0,18,13},{0,13,20},
{0,11,100},{0,18,1261},{0,11,701},{0,10,294},{0,7,857},{0,8,1382},{0,7,938},{0,26,421},{0,18,13},{0,13,20},{0,11,100},{9,0,1261},{0,11,701},{0,10,294},{0,7,857},{18,0,1261},{0,7,857},{0,16,0},{0,16,0},{0,16,0},{0,8,0},{0,8,221},{0,6,89},{0,6,89},{0,4,125},{0,4,246},{0,4,150},{0,16,0},{0,16,0},{0,16,0},{0,8,0},{4,0,221},
{0,6,89},{0,6,89},{0,4,125},{8,0,221},{0,4,125},{13,1,421},{0,18,13},{2,12,4},{0,11,100},{13,1,421},{26,0,421},{0,11,100},{0,9,421},{26,0,421},{0,9,421},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,32,430},{0,22,10},{1,15,70},{0,13,70},{0,22,1517},{0,14,730},{0,13,257},
{0,9,974},{0,11,1713},{0,8,1109},{1,30,422},{1,21,5},{2,15,45},{0,13,70},{11,1,1514},{0,14,730},{0,13,257},{0,9,974},{15,4,1514},{0,9,974},{0,22,9},{0,22,9},{0,22,9},{0,11,9},{0,12,338},{0,9,106},{0,9,106},{0,5,181},{0,5,392},{0,5,217},{1,20,1},{1,20,1},{1,20,1},{1,10,2},{6,0,338},{0,9,106},{0,9,106},{0,5,181},{12,0,338},
{0,5,181},{16,0,421},{0,22,1},{4,14,4},{0,13,61},{16,0,421},{32,0,421},{0,13,61},{0,11,421},{32,0,421},{0,11,421},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,2,1},{0,2,1},{0,2,1},{0,1,1},{0,1,2},{0,1,2},{1,36,486},{1,24,69},{2,17,163},{1,16,115},{0,28,1517},{0,18,614},{0,15,126},{0,11,857},{0,14,1841},{0,11,1053},{3,32,422},
{3,23,5},{3,17,46},{2,15,70},{14,1,1514},{0,18,614},{0,15,126},{0,11,857},{15,7,1514},{0,11,857},{1,26,65},{1,26,65},{1,26,65},{1,13,66},{0,18,338},{0,13,45},{0,13,45},{0,8,125},{0,8,456},{0,7,211},{3,22,1},{3,22,1},{3,22,1},{3,12,2},{9,0,338},{0,13,45},{0,13,45},{0,8,125},{18,0,338},{0,8,125},{19,0,421},{2,24,1},{6,16,1},
{0,16,37},{19,0,421},{38,0,421},{0,16,37},{0,13,421},{38,0,421},{0,13,421},{1,0,65},{1,0,65},{1,0,65},{1,0,65},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{0,3,25},{0,3,25},{3,39,629},{3,27,213},{3,20,365},{2,18,237},{0,35,1514},{0,22,506},{0,18,24},{0,14,750},{0,16,2003},{0,13,1050},{5,35,421},{5,25,2},{6,19,45},{5,17,72},{17,1,1514},
{0,22,506},{0,18,24},{0,14,750},{35,0,1514},{0,14,750},{3,28,209},{3,28,209},{3,28,209},{2,16,208},{0,25,338},{0,17,9},{0,17,9},{0,10,72},{0,11,566},{0,10,241},{5,24,1},{5,24,1},{5,24,1},{5,15,0},{12,1,338},{0,17,9},{0,17,9},{0,10,72},{25,0,338},{0,10,72},{22,1,421},{4,26,1},{8,18,5},{0,18,8},{22,1,421},{45,0,421},{0,18,8},
{0,15,421},{45,0,421},{0,15,421},{2,0,208},{2,0,208},{2,0,208},{2,0,208},{0,15,0},{0,15,0},{0,15,0},{0,8,1},{0,6,80},{0,6,80},{4,43,821},{4,30,418},{5,22,621},{3,20,420},{0,41,1514},{0,25,450},{0,20,5},{0,16,670},{0,19,2187},{0,15,1109},{7,37,421},{7,27,2},{8,21,45},{7,19,72},{20,1,1514},{0,25,450},{0,20,5},{0,16,670},{41,0,1514},
{0,16,670},{4,32,400},{4,32,400},{4,32,400},{4,18,400},{0,31,338},{0,20,1},{0,20,1},{0,13,40},{0,14,694},{0,11,297},{7,26,1},{7,26,1},{7,26,1},{7,17,1},{15,1,338},{0,20,1},{0,20,1},{0,13,40},{31,0,338},{0,13,40},{25,1,421},{6,28,1},{10,20,5},{0,20,4},{25,1,421},{51,0,421},{0,20,4},{0,17,421},{51,0,421},{0,17,421},{3,0,400},
{3,0,400},{3,0,400},{3,0,400},{0,21,0},{0,21,0},{0,21,0},{0,11,1},{0,8,149},{0,8,149},{5,47,846},{6,32,451},{6,23,662},{5,22,445},{2,43,1515},{1,28,446},{2,22,6},{0,18,638},{0,22,2046},{0,18,878},{9,39,421},{9,29,2},{10,23,45},{9,21,72},{23,1,1514},{0,28,426},{2,22,5},{0,18,589},{47,0,1514},{0,18,589},{5,36,425},{5,36,425},{5,36,425},
{5,21,426},{2,32,341},{2,22,2},{2,22,2},{1,15,38},{0,16,606},{0,14,158},{9,28,1},{9,28,1},{9,28,1},{9,19,1},{15,7,338},{2,22,1},{2,22,1},{0,15,20},{31,3,338},{0,15,20},{28,1,421},{8,30,1},{12,22,5},{2,22,4},{28,1,421},{57,0,421},{2,22,4},{0,19,421},{57,0,421},{0,19,421},{5,0,425},{5,0,425},{5,0,425},{5,0,425},{2,23,1},
{2,23,1},{2,23,1},{2,13,2},{0,12,97},{0,12,97},{7,49,846},{8,33,450},{8,25,662},{7,24,445},{4,45,1515},{3,30,446},{4,24,6},{2,20,638},{0,25,1886},{0,19,682},{11,41,421},{11,31,2},{12,25,45},{11,23,72},{26,1,1514},{1,31,421},{4,24,5},{0,20,545},{53,0,1514},{0,20,545},{7,38,425},{7,38,425},{7,38,425},{7,23,426},{4,34,341},{4,24,2},{4,24,2},
{4,16,42},{0,20,500},{0,16,62},{11,30,1},{11,30,1},{11,30,1},{11,21,1},{15,13,338},{4,24,1},{4,24,1},{0,17,5},{31,6,338},{0,17,5},{31,1,421},{10,32,1},{14,24,5},{4,24,4},{31,1,421},{63,0,421},{4,24,4},{0,21,421},{63,0,421},{0,21,421},{7,0,425},{7,0,425},{7,0,425},{7,0,425},{4,25,1},{4,25,1},{4,25,1},{4,15,2},{0,16,37},
{0,16,37},{10,49,846},{10,36,445},{11,28,650},{10,26,450},{6,47,1517},{5,32,446},{6,27,3},{4,22,646},{0,29,1751},{0,22,522},{13,44,422},{13,34,3},{14,27,46},{13,26,74},{30,0,1514},{3,33,422},{6,27,3},{0,22,497},{46,7,1514},{0,22,497},{10,39,425},{10,39,425},{10,39,425},{10,24,426},{6,37,338},{6,27,2},{6,27,2},{6,19,40},{0,24,410},{0,19,9},{13,33,1},
{13,33,1},{13,33,1},{13,23,1},{24,1,338},{7,26,0},{7,26,0},{1,19,5},{49,0,338},{1,19,5},{35,0,421},{12,34,1},{16,27,1},{5,27,1},{35,0,421},{63,3,421},{5,27,1},{0,23,421},{63,3,421},{0,23,421},{10,0,425},{10,0,425},{10,0,425},{10,0,425},{6,28,0},{6,28,0},{6,28,0},{6,17,1},{0,19,5},{0,19,5},{12,51,846},{12,38,445},{13,30,650},
{12,28,450},{8,49,1517},{7,34,446},{8,29,3},{6,24,646},{0,32,1647},{0,25,458},{15,46,422},{15,36,3},{16,29,42},{15,28,74},{32,1,1514},{5,35,422},{8,29,3},{0,25,457},{46,10,1514},{0,25,457},{12,41,425},{12,41,425},{12,41,425},{12,26,426},{8,39,338},{8,29,2},{8,29,2},{8,21,40},{0,27,362},{1,22,5},{15,35,1},{15,35,1},{15,35,1},{15,25,1},{27,1,338},
{9,28,0},{9,28,0},{3,21,5},{55,0,338},{3,21,5},{38,0,421},{14,36,1},{18,29,1},{7,29,1},{38,0,421},{63,6,421},{7,29,1},{0,25,421},{63,6,421},{0,25,421},{12,0,425},{12,0,425},{12,0,425},{12,0,425},{8,30,0},{8,30,0},{8,30,0},{8,19,1},{1,22,1},{1,22,1},{14,53,846},{14,40,445},{15,32,661},{14,30,450},{10,51,1517},{9,36,446},{10,31,3},
{8,26,646},{0,35,1575},{0,27,446},{17,47,421},{17,37,2},{18,31,42},{17,30,73},{35,1,1514},{7,37,422},{10,31,3},{0,27,430},{46,13,1514},{0,27,430},{14,43,425},{14,43,425},{14,43,425},{14,28,426},{10,41,338},{10,31,2},{10,31,2},{10,23,40},{0,31,341},{3,24,5},{17,36,1},{17,36,1},{17,36,1},{17,27,0},{30,1,338},{11,30,0},{11,30,0},{5,23,5},{61,0,338},
{5,23,5},{41,0,421},{16,38,1},{20,31,1},{9,31,1},{41,0,421},{63,9,421},{9,31,1},{0,27,421},{63,9,421},{0,27,421},{14,0,425},{14,0,425},{14,0,425},{14,0,425},{10,31,1},{10,31,1},{10,31,1},{10,21,1},{3,24,1},{3,24,1},{16,55,854},{16,42,451},{16,33,662},{15,32,446},{12,53,1517},{11,38,446},{12,32,9},{10,28,646},{0,39,1533},{2,29,446},{19,49,421},
{19,39,2},{20,33,45},{18,31,81},{38,1,1514},{9,39,422},{13,32,6},{0,29,422},{45,16,1514},{0,29,422},{15,47,433},{15,47,433},{15,47,433},{15,31,433},{12,43,338},{12,32,5},{12,32,5},{12,25,40},{1,33,339},{5,26,5},{19,38,1},{19,38,1},{19,38,1},{19,29,0},{33,1,338},{13,32,2},{13,32,2},{7,25,5},{63,2,338},{7,25,5},{44,0,421},{18,40,1},{22,32,5},
{12,32,5},{44,0,421},{63,12,421},{12,32,5},{0,29,421},{63,12,421},{0,29,421},{15,0,433},{15,0,433},{15,0,433},{15,0,433},{12,33,1},{12,33,1},{12,33,1},{12,23,1},{5,26,1},{5,26,1},{18,58,846},{18,44,445},{19,36,650},{18,34,450},{14,56,1515},{14,40,447},{14,35,7},{13,31,646},{0,42,1515},{5,31,445},{21,52,422},{21,42,2},{22,35,46},{21,34,74},{42,0,1514},
{12,41,422},{14,35,6},{2,31,425},{62,11,1514},{2,31,425},{18,47,425},{18,47,425},{18,47,425},{18,32,426},{14,46,339},{14,35,6},{14,35,6},{14,27,35},{4,35,341},{8,28,5},{21,41,1},{21,41,1},{21,41,1},{21,31,2},{31,12,338},{15,34,0},{15,34,0},{8,28,1},{62,6,338},{8,28,1},{47,0,421},{21,42,1},{24,35,1},{13,35,1},{47,0,421},{62,16,421},{13,35,1},
{0,31,425},{62,16,421},{0,31,425},{18,0,425},{18,0,425},{18,0,425},{18,0,425},{14,36,1},{14,36,1},{14,36,1},{14,25,1},{7,29,1},{7,29,1},{20,60,846},{20,46,445},{21,38,650},{20,36,450},{16,58,1514},{16,41,449},{16,37,3},{15,32,630},{2,44,1515},{7,33,441},{23,54,422},{23,44,2},{24,37,46},{23,36,74},{45,0,1514},{14,43,422},{16,37,3},{4,33,421},{62,14,1514},
{4,33,421},{20,49,425},{20,49,425},{20,49,425},{20,34,426},{16,47,340},{16,37,2},{16,37,2},{16,29,34},{6,37,341},{10,30,5},{23,43,1},{23,43,1},{23,43,1},{23,33,1},{31,18,338},{17,36,0},{17,36,0},{10,30,1},{62,9,338},{10,30,1},{50,0,421},{23,44,1},{26,37,1},{15,37,1},{50,0,421},{62,19,421},{15,37,1},{0,33,421},{62,19,421},{0,33,421},{20,0,425},
{20,0,425},{20,0,425},{20,0,425},{16,38,0},{16,38,0},{16,38,0},{16,27,0},{9,31,1},{9,31,1},{22,62,846},{22,48,445},{23,40,650},{22,38,450},{18,60,1514},{18,43,449},{18,39,3},{16,34,646},{4,46,1515},{9,35,441},{25,56,422},{25,46,2},{26,39,46},{25,38,74},{47,2,1514},{16,45,425},{18,39,3},{6,35,421},{62,17,1514},{6,35,421},{22,51,425},{22,51,425},{22,51,425},
{22,36,426},{18,49,338},{18,39,2},{18,39,2},{18,31,34},{8,39,341},{11,32,6},{25,45,1},{25,45,1},{25,45,1},{25,35,1},{34,17,338},{19,38,0},{19,38,0},{12,32,4},{62,12,338},{12,32,4},{53,0,421},{25,46,1},{28,39,1},{17,39,1},{53,0,421},{62,22,421},{17,39,1},{0,35,421},{62,22,421},{0,35,421},{22,0,425},{22,0,425},{22,0,425},{22,0,425},{18,40,0},
{18,40,0},{18,40,0},{18,29,0},{11,32,2},{11,32,2},{24,63,850},{24,50,445},{25,42,650},{24,40,450},{20,62,1514},{20,45,449},{20,41,3},{18,36,646},{6,48,1517},{11,37,441},{27,58,422},{27,48,3},{28,41,46},{27,40,74},{47,8,1514},{18,47,425},{20,41,3},{8,37,421},{62,20,1514},{8,37,421},{24,53,425},{24,53,425},{24,53,425},{24,38,426},{20,51,338},{20,41,2},{20,41,2},
{20,33,40},{10,41,341},{13,34,6},{27,47,1},{27,47,1},{27,47,1},{27,37,1},{37,17,338},{21,40,0},{21,40,0},{14,34,4},{62,15,338},{14,34,4},{56,0,421},{26,48,1},{30,41,1},{19,41,1},{56,0,421},{62,25,421},{19,41,1},{0,37,421},{62,25,421},{0,37,421},{24,0,425},{24,0,425},{24,0,425},{24,0,425},{20,42,0},{20,42,0},{20,42,0},{20,31,0},{13,34,2},
{13,34,2},{26,63,882},{26,52,447},{27,44,646},{26,42,446},{23,62,1526},{22,48,447},{22,43,5},{21,39,646},{8,50,1515},{13,39,443},{30,58,425},{29,50,4},{30,44,45},{29,42,69},{54,0,1514},{20,49,422},{22,43,4},{10,39,426},{47,31,1514},{10,39,426},{26,56,421},{26,56,421},{26,56,421},{26,41,421},{22,54,339},{22,43,5},{22,43,5},{22,35,35},{12,43,338},{16,36,5},{30,48,4},
{30,48,4},{30,48,4},{30,39,4},{49,0,338},{23,42,2},{23,42,2},{16,36,1},{62,18,338},{16,36,1},{59,0,421},{29,50,0},{32,43,1},{22,43,0},{59,0,421},{62,28,421},{22,43,0},{0,39,425},{62,28,421},{0,39,425},{26,0,421},{26,0,421},{26,0,421},{26,0,421},{22,44,2},{22,44,2},{22,44,2},{22,33,1},{15,37,1},{15,37,1},{29,63,922},{28,54,447},{29,46,646},
{28,44,446},{25,63,1535},{24,50,447},{24,45,5},{23,41,646},{10,52,1515},{15,41,443},{31,62,425},{31,52,4},{32,45,45},{31,44,69},{57,0,1514},{22,51,422},{24,45,4},{12,41,426},{50,32,1514},{12,41,426},{28,58,421},{28,58,421},{28,58,421},{28,43,421},{24,56,339},{24,45,5},{24,45,5},{24,37,35},{14,45,338},{18,38,5},{31,52,4},{31,52,4},{31,52,4},{31,41,5},{52,0,338},
{25,44,2},{25,44,2},{18,38,1},{62,21,338},{18,38,1},{62,0,421},{31,52,0},{34,45,1},{24,45,0},{62,0,421},{62,31,421},{24,45,0},{0,41,425},{62,31,421},{0,41,425},{28,0,421},{28,0,421},{28,0,421},{28,0,421},{24,46,2},{24,46,2},{24,46,2},{24,35,1},{17,39,1},{17,39,1},{31,63,994},{30,56,447},{31,48,655},{30,46,446},{28,63,1575},{26,52,447},{26,47,5},
{25,43,646},{12,54,1515},{17,43,445},{33,63,426},{33,54,2},{34,47,45},{33,46,66},{60,0,1514},{24,53,422},{26,47,4},{14,43,426},{56,32,1514},{14,43,426},{30,60,421},{30,60,421},{30,60,421},{30,45,421},{26,58,339},{26,47,5},{26,47,5},{26,39,35},{16,47,338},{20,40,5},{33,53,1},{33,53,1},{33,53,1},{33,43,2},{55,0,338},{27,46,2},{27,46,2},{20,40,1},{62,24,338},
{20,40,1},{63,4,421},{33,54,1},{36,47,1},{26,47,0},{63,4,421},{62,34,421},{26,47,0},{0,43,425},{62,34,421},{0,43,425},{30,0,421},{30,0,421},{30,0,421},{30,0,421},{26,48,1},{26,48,1},{26,48,1},{26,37,1},{19,41,1},{19,41,1},{33,63,1082},{32,58,445},{33,50,650},{32,48,450},{30,63,1638},{28,54,447},{28,49,7},{27,45,646},{14,56,1515},{19,45,445},{36,63,434},
{35,56,2},{36,49,46},{35,48,74},{63,0,1514},{26,55,422},{28,49,6},{16,45,425},{62,32,1514},{16,45,425},{32,61,425},{32,61,425},{32,61,425},{32,47,426},{28,60,339},{28,49,6},{28,49,6},{28,41,35},{18,49,341},{22,42,5},{35,55,1},{35,55,1},{35,55,1},{35,45,2},{58,0,338},{29,48,0},{29,48,0},{22,42,1},{62,27,338},{22,42,1},{63,10,421},{35,56,1},{38,49,1},
{27,49,1},{63,10,421},{62,37,421},{27,49,1},{0,45,425},{62,37,421},{0,45,425},{32,0,425},{32,0,425},{32,0,425},{32,0,425},{28,50,1},{28,50,1},{28,50,1},{28,39,1},{21,43,1},{21,43,1},{36,63,1206},{34,61,446},{35,52,646},{34,50,446},{33,63,1710},{30,56,445},{31,51,5},{29,47,638},{17,58,1515},{22,47,450},{39,63,469},{37,58,5},{38,52,45},{37,50,69},{63,7,1514},
{27,58,422},{31,51,1},{19,47,433},{63,35,1514},{19,47,433},{34,63,422},{34,63,422},{34,63,422},{34,49,421},{31,61,342},{31,51,5},{31,51,5},{30,43,36},{20,51,338},{24,44,2},{38,56,4},{38,56,4},{38,56,4},{38,47,4},{53,16,338},{31,51,1},{31,51,1},{25,44,0},{63,30,338},{25,44,0},{63,17,421},{37,58,1},{40,51,1},{30,51,0},{63,17,421},{63,40,421},{30,51,0},
{0,47,433},{63,40,421},{0,47,433},{34,0,421},{34,0,421},{34,0,421},{34,0,421},{31,51,4},{31,51,4},{31,51,4},{31,41,4},{23,45,1},{23,45,1},{38,63,1350},{36,63,446},{37,54,646},{36,52,446},{36,63,1814},{32,58,447},{32,53,5},{31,48,638},{19,60,1515},{23,49,443},{41,63,517},{39,60,5},{40,54,45},{39,52,69},{63,13,1514},{29,60,422},{32,53,4},{20,49,426},{63,38,1514},
{20,49,426},{36,63,437},{36,63,437},{36,63,437},{36,51,421},{33,62,347},{32,53,5},{32,53,5},{32,45,33},{22,53,338},{26,46,2},{40,58,4},{40,58,4},{40,58,4},{40,49,4},{56,16,338},{33,52,2},{33,52,2},{27,46,0},{62,33,338},{27,46,0},{63,23,421},{39,60,1},{42,53,1},{32,53,0},{63,23,421},{63,43,421},{32,53,0},{0,49,425},{63,43,421},{0,49,425},{36,0,421},
{36,0,421},{36,0,421},{36,0,421},{32,54,2},{32,54,2},{32,54,2},{32,43,2},{25,47,1},{25,47,1},{40,63,1466},{38,63,474},{39,56,646},{38,54,446},{38,63,1931},{34,60,447},{34,55,5},{33,51,646},{21,62,1515},{25,51,443},{43,63,569},{41,62,5},{42,56,45},{41,54,69},{63,19,1514},{32,61,422},{34,55,4},{22,51,426},{63,41,1514},{22,51,426},{38,63,470},{38,63,470},{38,63,470},
{38,53,421},{35,63,355},{34,55,5},{34,55,5},{34,47,33},{24,55,338},{28,48,3},{42,60,4},{42,60,4},{42,60,4},{42,51,4},{59,16,338},{35,54,2},{35,54,2},{28,48,2},{62,36,338},{28,48,2},{63,29,421},{41,62,1},{44,55,1},{34,55,0},{63,29,421},{63,46,421},{34,55,0},{0,51,425},{63,46,421},{0,51,425},{38,0,421},{38,0,421},{38,0,421},{38,0,421},{34,56,2},
{34,56,2},{34,56,2},{34,45,2},{27,49,1},{27,49,1},{43,63,1634},{41,63,546},{41,58,646},{40,56,446},{40,63,2039},{36,62,447},{36,57,5},{35,53,646},{24,63,1521},{27,53,443},{46,63,633},{44,63,9},{44,58,45},{43,56,69},{63,25,1514},{34,63,422},{36,57,4},{24,53,426},{63,44,1514},{24,53,426},{40,63,502},{40,63,502},{40,63,502},{40,55,421},{37,63,379},{36,57,5},{36,57,5},
{36,49,35},{26,57,338},{30,50,3},{44,62,4},{44,62,4},{44,62,4},{44,53,4},{62,16,338},{37,56,2},{37,56,2},{30,50,2},{62,39,338},{30,50,2},{63,34,421},{44,63,5},{46,57,1},{36,57,0},{63,34,421},{62,49,421},{36,57,0},{0,53,425},{62,49,421},{0,53,425},{40,0,421},{40,0,421},{40,0,421},{40,0,421},{36,58,2},{36,58,2},{36,58,2},{36,47,2},{29,51,1},
{29,51,1},{45,63,1866},{43,63,689},{43,60,650},{42,59,447},{43,63,2201},{38,63,469},{39,59,6},{37,55,638},{29,63,1590},{30,55,450},{48,63,741},{46,63,69},{46,60,46},{45,58,70},{62,33,1514},{38,63,465},{39,59,2},{25,56,425},{61,48,1514},{25,56,425},{43,63,545},{43,63,545},{43,63,545},{42,57,422},{40,63,424},{39,59,5},{39,59,5},{38,51,36},{29,59,339},{32,52,2},{46,63,5},
{46,63,5},{46,63,5},{46,55,2},{63,21,338},{39,59,1},{39,59,1},{33,52,0},{63,42,338},{33,52,0},{63,41,421},{48,63,41},{49,59,2},{38,59,2},{63,41,421},{63,52,421},{38,59,2},{0,56,425},{63,52,421},{0,56,425},{42,0,421},{42,0,421},{42,0,421},{42,0,421},{39,59,4},{39,59,4},{39,59,4},{39,49,4},{31,53,1},{31,53,1},{48,63,2070},{46,63,889},{45,62,650},
{44,61,447},{46,63,2361},{41,63,573},{41,61,6},{39,57,638},{32,63,1710},{32,57,450},{51,63,837},{49,63,184},{48,62,49},{47,60,70},{63,37,1514},{43,63,545},{41,61,2},{27,58,425},{63,50,1514},{27,58,425},{45,63,614},{45,63,614},{45,63,614},{44,59,422},{43,63,488},{41,61,5},{41,61,5},{40,53,36},{31,61,339},{34,54,2},{48,63,20},{48,63,20},{48,63,20},{48,57,4},{63,27,338},
{41,61,1},{41,61,1},{35,54,0},{63,45,338},{35,54,0},{63,47,421},{51,63,97},{51,61,2},{40,61,2},{63,47,421},{63,55,421},{40,61,2},{0,58,425},{63,55,421},{0,58,425},{44,0,421},{44,0,421},{44,0,421},{44,0,421},{41,61,4},{41,61,4},{41,61,4},{41,51,4},{33,55,1},{33,55,1},{50,63,2239},{48,63,1109},{47,63,701},{46,62,441},{48,63,2469},{44,63,720},{43,63,5},
{41,59,605},{37,63,1804},{34,59,417},{54,63,916},{51,63,308},{51,63,52},{49,62,56},{63,43,1459},{46,63,618},{43,63,1},{31,59,400},{63,53,1459},{31,59,400},{47,63,701},{47,63,701},{47,63,701},{46,61,422},{45,63,566},{43,63,5},{43,63,5},{42,55,36},{32,63,341},{36,56,2},{51,63,52},{51,63,52},{51,63,52},{50,59,4},{63,33,338},{43,63,1},{43,63,1},{37,56,0},{47,56,338},
{37,56,0},{63,53,392},{55,63,157},{53,63,1},{42,63,1},{63,53,392},{63,58,392},{42,63,1},{0,59,400},{63,58,392},{0,59,400},{46,0,421},{46,0,421},{46,0,421},{46,0,421},{43,63,4},{43,63,4},{43,63,4},{43,53,4},{35,57,1},{35,57,1},{51,63,1901},{50,63,1114},{49,63,785},{48,63,421},{51,63,2093},{46,63,574},{45,63,38},{44,60,356},{40,63,1476},{36,60,213},{55,63,684},
{54,63,260},{53,63,85},{51,63,20},{63,47,1064},{49,63,426},{47,63,8},{35,60,208},{63,55,1064},{35,60,208},{49,63,785},{49,63,785},{49,63,785},{48,63,421},{48,63,661},{45,63,38},{45,63,38},{44,57,36},{37,63,371},{38,58,2},{53,63,85},{53,63,85},{53,63,85},{52,61,4},{63,39,338},{47,63,8},{47,63,8},{39,58,0},{47,59,338},{39,58,0},{63,56,200},{57,63,80},{56,63,1},
{48,63,0},{63,56,200},{62,60,200},{48,63,0},{0,60,208},{62,60,200},{0,60,208},{48,0,421},{48,0,421},{48,0,421},{48,0,421},{45,63,13},{45,63,13},{45,63,13},{45,55,4},{37,59,1},{37,59,1},{54,63,1646},{52,63,1119},{51,63,886},{50,63,470},{53,63,1761},{48,63,526},{48,63,126},{46,61,146},{44,63,1218},{39,62,69},{57,63,450},{56,63,245},{56,63,145},{54,63,2},{63,51,722},
{52,63,290},{51,63,50},{38,62,65},{63,57,722},{38,62,65},{51,63,886},{51,63,886},{51,63,886},{50,63,470},{50,63,776},{48,63,126},{48,63,126},{47,59,42},{41,63,446},{40,60,5},{56,63,145},{56,63,145},{56,63,145},{54,63,2},{61,49,338},{51,63,50},{51,63,50},{42,60,1},{63,54,338},{42,60,1},{63,59,61},{60,63,25},{59,63,0},{55,63,0},{63,59,61},{63,61,61},{55,63,0},
{0,62,65},{63,61,61},{0,62,65},{50,0,421},{50,0,421},{50,0,421},{50,0,421},{48,63,45},{48,63,45},{48,63,45},{47,57,2},{40,61,1},{40,61,1},{56,63,1518},{54,63,1118},{54,63,974},{52,63,565},{54,63,1526},{51,63,534},{51,63,278},{48,62,57},{47,63,1090},{42,63,10},{59,63,354},{58,63,251},{57,63,194},{56,63,50},{63,55,509},{55,63,234},{54,63,106},{42,63,9},{63,59,509},
{42,63,9},{54,63,974},{54,63,974},{54,63,974},{52,63,565},{53,63,904},{51,63,278},{51,63,278},{48,61,38},{44,63,574},{42,62,5},{57,63,194},{57,63,194},{57,63,194},{56,63,50},{63,51,338},{54,63,106},{54,63,106},{44,62,1},{63,57,338},{44,62,1},{63,62,5},{62,63,4},{62,63,0},{61,63,0},{63,62,5},{62,63,5},{61,63,0},{0,63,9},{62,63,5},{0,63,9},{52,0,421},
{52,0,421},{52,0,421},{52,0,421},{50,63,72},{50,63,72},{50,63,72},{49,59,5},{42,63,1},{42,63,1},{57,63,1197},{56,63,994},{56,63,894},{54,63,565},{56,63,1210},{54,63,493},{53,63,332},{50,63,13},{51,63,861},{45,63,17},{60,63,209},{60,63,161},{60,63,145},{58,63,50},{63,58,294},{58,63,147},{57,63,89},{47,63,1},{62,61,294},{47,63,1},{56,63,894},{56,63,894},{56,63,894},
{54,63,565},{54,63,781},{53,63,332},{53,63,332},{50,63,13},{48,63,501},{45,63,17},{60,63,145},{60,63,145},{60,63,145},{58,63,50},{63,55,221},{57,63,89},{57,63,89},{47,63,1},{63,59,221},{47,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{54,0,421},{54,0,421},{54,0,421},{54,0,421},{52,63,117},
{52,63,117},{52,63,117},{51,61,5},{45,63,17},{45,63,17},{58,63,925},{57,63,765},{57,63,701},{57,63,509},{57,63,845},{55,63,423},{54,63,301},{53,63,5},{52,63,598},{49,63,52},{61,63,97},{61,63,70},{61,63,61},{60,63,17},{63,60,113},{60,63,57},{59,63,40},{53,63,1},{62,62,113},{53,63,1},{57,63,701},{57,63,701},{57,63,701},{57,63,509},{57,63,589},{54,63,301},{54,63,301},
{53,63,5},{51,63,365},{49,63,52},{61,63,61},{61,63,61},{61,63,61},{60,63,17},{63,58,85},{59,63,40},{59,63,40},{53,63,1},{62,61,85},{53,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{56,0,421},{56,0,421},{56,0,421},{56,0,421},{54,63,157},{54,63,157},{54,63,157},{53,63,5},{49,63,52},
{49,63,52},{0,29,882},{0,21,116},{0,15,4},{0,13,320},{0,19,1899},{0,14,1214},{0,11,573},{0,8,1421},{0,9,2052},{0,8,1521},{0,29,882},{0,21,116},{0,15,4},{0,13,320},{10,0,1896},{0,14,1214},{0,11,573},{0,8,1421},{14,3,1896},{0,8,1421},{0,14,0},{0,14,0},{0,14,0},{0,7,0},{0,7,162},{0,5,61},{0,5,61},{0,3,100},{0,3,180},{0,3,116},{0,14,0},
{0,14,0},{0,14,0},{0,7,0},{3,1,162},{0,5,61},{0,5,61},{0,3,100},{7,0,162},{0,3,100},{6,17,882},{0,21,116},{0,15,4},{0,13,320},{6,17,882},{29,0,882},{0,13,320},{0,10,884},{29,0,882},{0,10,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,35,882},{0,25,50},{0,17,10},
{0,15,260},{0,23,2355},{0,16,1355},{0,14,589},{0,10,1656},{0,11,2567},{0,10,1825},{0,35,882},{0,25,50},{0,17,10},{0,15,260},{3,17,2355},{0,16,1355},{0,14,589},{0,10,1656},{23,0,2355},{0,10,1656},{0,19,1},{0,19,1},{0,19,1},{0,10,0},{0,10,338},{0,8,125},{0,8,125},{0,4,200},{0,5,374},{0,4,225},{0,19,1},{0,19,1},{0,19,1},{0,10,0},{5,0,338},
{0,8,125},{0,8,125},{0,4,200},{10,0,338},{0,4,200},{17,1,882},{0,25,50},{1,17,2},{0,15,260},{17,1,882},{35,0,882},{0,15,260},{0,12,884},{35,0,882},{0,12,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,41,882},{0,28,10},{1,19,69},{0,17,193},{0,28,2899},{0,19,1539},{0,16,643},
{0,11,1965},{0,13,3209},{0,11,2161},{0,41,882},{0,28,10},{1,19,53},{0,17,193},{14,0,2899},{0,19,1539},{0,16,643},{0,11,1965},{28,0,2899},{0,11,1965},{0,25,1},{0,25,1},{0,25,1},{0,13,0},{0,13,578},{0,11,221},{0,11,221},{0,6,356},{0,6,644},{0,5,401},{0,25,1},{0,25,1},{0,25,1},{0,13,0},{6,1,578},{0,11,221},{0,11,221},{0,6,356},{13,0,578},
{0,6,356},{20,1,882},{0,28,10},{3,19,2},{0,17,193},{20,1,882},{41,0,882},{0,17,193},{0,14,884},{41,0,882},{0,14,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,45,902},{1,31,24},{1,21,142},{0,19,166},{0,33,3048},{0,22,1443},{0,18,481},{0,13,1908},{0,16,3492},{0,13,2197},{2,43,886},
{1,31,8},{3,21,73},{1,19,153},{16,1,3048},{0,22,1443},{0,18,481},{0,13,1908},{33,0,3048},{0,13,1908},{1,29,21},{1,29,21},{1,29,21},{1,15,21},{0,18,648},{0,14,169},{0,14,169},{0,8,325},{0,8,766},{0,7,421},{2,27,5},{2,27,5},{2,27,5},{2,15,4},{9,0,648},{0,14,169},{0,14,169},{0,8,325},{18,0,648},{0,8,325},{23,1,882},{0,31,2},{5,21,2},
{0,19,130},{23,1,882},{47,0,882},{0,19,130},{0,16,890},{47,0,882},{0,16,890},{1,0,20},{1,0,20},{1,0,20},{1,0,20},{0,5,0},{0,5,0},{0,5,0},{0,2,1},{0,2,5},{0,2,5},{2,49,997},{2,34,114},{3,24,290},{1,22,234},{0,39,3051},{0,25,1256},{0,21,258},{0,16,1764},{0,19,3685},{0,15,2195},{4,46,883},{3,33,10},{5,23,74},{3,21,154},{15,10,3048},
{0,25,1256},{0,21,258},{0,16,1764},{30,5,3048},{0,16,1764},{2,34,113},{2,34,113},{2,34,113},{2,18,113},{0,25,648},{0,17,89},{0,17,89},{0,10,242},{0,11,876},{0,10,411},{4,30,1},{4,30,1},{4,30,1},{4,17,1},{12,1,648},{0,17,89},{0,17,89},{0,10,242},{25,0,648},{0,10,242},{27,0,882},{3,33,1},{8,23,1},{0,22,80},{27,0,882},{46,4,882},{0,22,80},
{0,18,884},{46,4,882},{0,18,884},{2,0,113},{2,0,113},{2,0,113},{2,0,113},{0,11,1},{0,11,1},{0,11,1},{0,6,1},{0,5,40},{0,5,40},{3,53,1149},{3,36,274},{4,27,513},{3,24,362},{0,45,3051},{0,28,1128},{0,24,122},{0,18,1605},{0,22,3901},{0,16,2173},{6,47,885},{5,35,10},{7,25,74},{5,23,154},{15,16,3048},{0,28,1128},{0,24,122},{0,18,1605},{30,8,3048},
{0,18,1605},{3,38,265},{3,38,265},{3,38,265},{3,20,266},{0,31,648},{0,21,34},{0,21,34},{0,13,170},{0,14,1004},{0,11,457},{6,32,1},{6,32,1},{6,32,1},{6,19,1},{15,1,648},{0,21,34},{0,21,34},{0,13,170},{31,0,648},{0,13,170},{30,0,882},{5,35,1},{10,25,1},{0,24,41},{30,0,882},{46,7,882},{0,24,41},{0,20,884},{46,7,882},{0,20,884},{3,0,265},
{3,0,265},{3,0,265},{3,0,265},{0,17,0},{0,17,0},{0,17,0},{0,9,1},{0,8,104},{0,8,104},{4,57,1365},{4,39,498},{5,28,793},{4,26,561},{0,51,3048},{0,31,1032},{0,26,41},{0,20,1509},{0,24,4147},{0,19,2149},{8,49,885},{7,37,10},{9,27,74},{7,25,154},{17,17,3048},{0,31,1032},{0,26,41},{0,20,1509},{51,0,3048},{0,20,1509},{4,42,481},{4,42,481},{4,42,481},
{4,23,481},{0,36,650},{0,25,4},{0,25,4},{0,15,130},{0,16,1161},{0,14,481},{8,34,1},{8,34,1},{8,34,1},{8,21,1},{15,7,648},{0,25,4},{0,25,4},{0,15,130},{31,3,648},{0,15,130},{32,1,882},{7,37,1},{12,27,1},{0,26,25},{32,1,882},{46,10,882},{0,26,25},{0,22,884},{46,10,882},{0,22,884},{4,0,481},{4,0,481},{4,0,481},{4,0,481},{0,23,0},
{0,23,0},{0,23,0},{0,12,1},{0,9,193},{0,9,193},{5,61,1645},{5,42,795},{7,30,1177},{4,28,826},{0,57,3048},{0,35,954},{0,29,9},{0,22,1380},{0,27,4419},{0,21,2197},{10,51,885},{9,39,10},{11,29,74},{9,27,154},{20,17,3048},{0,35,954},{0,29,9},{0,22,1380},{57,0,3048},{0,22,1380},{5,46,761},{5,46,761},{5,46,761},{5,25,762},{0,42,650},{0,28,4},{0,28,4},
{0,17,85},{0,19,1345},{0,16,557},{10,36,1},{10,36,1},{10,36,1},{10,23,1},{15,13,648},{1,27,1},{1,27,1},{0,17,85},{31,6,648},{0,17,85},{35,1,882},{9,39,1},{14,29,1},{0,29,9},{35,1,882},{46,13,882},{0,29,9},{0,24,884},{46,13,882},{0,24,884},{5,0,761},{5,0,761},{5,0,761},{5,0,761},{0,29,0},{0,29,0},{0,29,0},{0,15,1},{0,11,296},
{0,11,296},{7,63,1774},{7,44,924},{8,33,1320},{6,30,941},{2,60,3052},{1,38,924},{2,31,10},{0,24,1302},{0,31,4300},{0,24,1942},{12,54,882},{12,40,8},{13,32,68},{11,29,161},{32,0,3048},{0,39,898},{2,31,6},{0,24,1266},{62,1,3048},{0,24,1266},{7,48,885},{7,48,885},{7,48,885},{7,28,884},{2,45,652},{2,31,9},{2,31,9},{1,20,62},{0,22,1275},{0,19,419},{12,39,0},
{12,39,0},{12,39,0},{12,25,1},{24,1,648},{3,30,0},{3,30,0},{0,20,45},{49,0,648},{0,20,45},{39,0,882},{11,41,1},{16,31,2},{1,31,1},{39,0,882},{62,8,882},{1,31,1},{0,26,882},{62,8,882},{0,26,882},{7,0,884},{7,0,884},{7,0,884},{7,0,884},{2,32,4},{2,32,4},{2,32,4},{2,17,4},{0,16,250},{0,16,250},{9,63,1798},{9,46,924},{10,35,1320},
{9,32,936},{4,62,3052},{3,40,924},{4,33,9},{2,26,1302},{0,33,4023},{0,26,1647},{14,56,882},{14,42,8},{15,34,68},{13,31,161},{35,0,3048},{0,42,882},{4,33,5},{0,27,1170},{62,4,3048},{0,27,1170},{9,50,885},{9,50,885},{9,50,885},{9,30,884},{4,47,652},{4,32,8},{4,32,8},{3,22,62},{0,25,1107},{0,22,243},{14,41,0},{14,41,0},{14,41,0},{14,27,1},{27,1,648},
{5,32,1},{5,32,1},{0,22,18},{55,0,648},{0,22,18},{42,0,882},{13,43,1},{18,33,1},{2,33,0},{42,0,882},{62,11,882},{2,33,0},{0,28,882},{62,11,882},{0,28,882},{9,0,884},{9,0,884},{9,0,884},{9,0,884},{4,34,4},{4,34,4},{4,34,4},{4,19,4},{0,19,146},{0,19,146},{12,63,1846},{11,48,924},{12,37,1320},{11,34,936},{6,63,3055},{5,42,924},{6,35,9},
{4,28,1302},{0,36,3799},{0,28,1356},{16,58,883},{16,44,9},{17,35,74},{15,33,161},{38,0,3048},{2,44,882},{6,35,5},{0,29,1110},{62,7,3048},{0,29,1110},{11,52,885},{11,52,885},{11,52,885},{11,32,885},{6,49,652},{6,34,8},{6,34,8},{5,24,62},{0,29,969},{0,24,109},{16,42,1},{16,42,1},{16,42,1},{16,29,2},{30,1,648},{7,34,1},{7,34,1},{0,24,9},{61,0,648},
{0,24,9},{45,0,882},{15,45,1},{20,35,1},{4,35,0},{45,0,882},{62,14,882},{4,35,0},{0,30,882},{62,14,882},{0,30,882},{11,0,884},{11,0,884},{11,0,884},{11,0,884},{6,36,4},{6,36,4},{6,36,4},{6,21,4},{0,22,74},{0,22,74},{14,63,1912},{13,50,924},{14,39,1320},{13,36,936},{9,63,3087},{7,44,924},{8,37,9},{6,30,1302},{0,39,3607},{0,31,1156},{18,60,883},
{18,46,9},{19,37,74},{17,35,154},{41,0,3048},{4,46,882},{8,37,5},{0,31,1035},{62,10,3048},{0,31,1035},{13,54,885},{13,54,885},{13,54,885},{13,34,885},{8,51,652},{8,36,8},{8,36,8},{7,26,62},{0,33,846},{0,27,37},{18,44,1},{18,44,1},{18,44,1},{18,31,2},{33,1,648},{9,36,1},{9,36,1},{0,27,1},{63,2,648},{0,27,1},{47,2,882},{16,48,1},{22,37,1},
{6,37,0},{47,2,882},{62,17,882},{6,37,0},{0,32,884},{62,17,882},{0,32,884},{13,0,884},{13,0,884},{13,0,884},{13,0,884},{8,38,4},{8,38,4},{8,38,4},{8,23,4},{0,26,29},{0,26,29},{16,63,2014},{15,52,932},{16,40,1341},{15,38,932},{12,63,3141},{9,46,924},{10,39,7},{9,32,1308},{0,43,3436},{0,33,1003},{20,62,884},{20,48,8},{21,40,74},{19,37,161},{36,16,3048},
{6,48,882},{10,39,6},{0,33,978},{63,13,3048},{0,33,978},{15,57,882},{15,57,882},{15,57,882},{15,36,882},{10,54,649},{10,39,6},{10,39,6},{9,28,61},{0,36,737},{1,29,8},{20,47,0},{20,47,0},{20,47,0},{20,33,1},{31,12,648},{11,38,1},{11,38,1},{2,29,1},{62,6,648},{2,29,1},{51,0,882},{19,49,1},{24,39,1},{9,39,1},{51,0,882},{47,28,882},{9,39,1},
{0,34,882},{47,28,882},{0,34,882},{15,0,882},{15,0,882},{15,0,882},{15,0,882},{10,40,2},{10,40,2},{10,40,2},{10,25,2},{0,30,2},{0,30,2},{19,63,2126},{17,54,924},{18,42,1341},{16,40,941},{14,63,3204},{11,48,924},{12,41,7},{11,34,1308},{0,46,3300},{0,35,939},{22,63,890},{22,50,8},{23,42,74},{21,39,161},{39,16,3048},{8,50,882},{12,41,6},{0,35,939},{62,16,3048},
{0,35,939},{17,59,884},{17,59,884},{17,59,884},{17,38,884},{12,56,649},{12,41,6},{12,41,6},{11,30,61},{0,39,681},{3,31,8},{22,49,0},{22,49,0},{22,49,0},{22,35,1},{31,18,648},{13,40,1},{13,40,1},{4,31,1},{62,9,648},{4,31,1},{54,0,882},{21,51,1},{26,41,1},{11,41,1},{54,0,882},{47,31,882},{11,41,1},{0,36,882},{47,31,882},{0,36,882},{17,0,884},
{17,0,884},{17,0,884},{17,0,884},{12,42,2},{12,42,2},{12,42,2},{12,27,2},{2,32,1},{2,32,1},{20,63,2264},{19,56,924},{20,44,1341},{18,42,941},{17,63,3292},{13,50,924},{14,43,7},{13,36,1308},{0,50,3192},{0,38,923},{25,63,906},{24,52,8},{25,44,74},{23,41,161},{50,0,3048},{10,52,882},{14,43,6},{0,38,907},{62,19,3048},{0,38,907},{19,61,884},{19,61,884},{19,61,884},
{19,40,884},{14,58,649},{14,43,6},{14,43,6},{13,32,67},{0,42,657},{4,33,10},{24,51,0},{24,51,0},{24,51,0},{24,37,1},{34,17,648},{15,42,1},{15,42,1},{6,33,1},{62,12,648},{6,33,1},{57,0,882},{23,53,1},{28,43,1},{13,43,1},{57,0,882},{50,32,882},{13,43,1},{0,38,882},{50,32,882},{0,38,882},{19,0,884},{19,0,884},{19,0,884},{19,0,884},{14,44,2},
{14,44,2},{14,44,2},{14,29,2},{4,34,1},{4,34,1},{23,63,2376},{21,58,924},{22,46,1341},{20,44,941},{19,63,3391},{15,52,924},{16,45,10},{15,38,1308},{0,53,3112},{2,40,923},{27,63,948},{26,54,8},{27,46,74},{25,43,161},{53,0,3048},{12,54,882},{16,45,6},{0,40,891},{62,22,3048},{0,40,891},{21,63,884},{21,63,884},{21,63,884},{21,42,884},{16,59,652},{16,45,9},{16,45,9},
{15,34,67},{1,45,650},{6,35,10},{26,53,0},{26,53,0},{26,53,0},{26,39,1},{37,17,648},{17,44,0},{17,44,0},{8,35,1},{62,15,648},{8,35,1},{60,0,882},{25,55,1},{30,45,1},{15,45,1},{60,0,882},{56,32,882},{15,45,1},{0,40,882},{56,32,882},{0,40,882},{21,0,884},{21,0,884},{21,0,884},{21,0,884},{16,46,4},{16,46,4},{16,46,4},{16,31,4},{6,36,1},
{6,36,1},{26,63,2564},{23,61,927},{24,49,1324},{23,47,935},{22,63,3529},{17,54,924},{18,47,10},{17,41,1302},{0,56,3060},{5,42,924},{30,63,1003},{28,57,6},{29,48,67},{28,46,158},{56,1,3048},{15,56,883},{19,47,8},{1,42,884},{63,25,3048},{1,42,884},{23,63,891},{23,63,891},{23,63,891},{23,44,883},{18,62,649},{18,47,6},{18,47,6},{17,36,61},{4,47,652},{9,37,8},{28,55,2},
{28,55,2},{28,55,2},{28,42,2},{49,0,648},{19,46,2},{19,46,2},{11,37,0},{62,18,648},{11,37,0},{63,1,882},{27,58,2},{32,47,4},{17,47,5},{63,1,882},{63,32,882},{17,47,5},{0,42,884},{63,32,882},{0,42,884},{23,0,882},{23,0,882},{23,0,882},{23,0,882},{18,48,2},{18,48,2},{18,48,2},{18,33,2},{8,38,1},{8,38,1},{28,63,2774},{25,63,927},{26,51,1324},
{25,48,932},{25,63,3681},{19,56,924},{20,49,7},{19,43,1302},{0,59,3052},{7,44,924},{32,63,1074},{30,59,6},{31,50,67},{30,48,170},{59,1,3048},{16,58,885},{20,49,6},{3,44,884},{63,28,3048},{3,44,884},{25,63,918},{25,63,918},{25,63,918},{25,46,883},{20,63,651},{20,49,6},{20,49,6},{19,38,61},{5,49,651},{11,39,8},{30,57,2},{30,57,2},{30,57,2},{30,44,2},{52,0,648},
{21,48,1},{21,48,1},{13,39,0},{62,21,648},{13,39,0},{63,7,882},{29,60,2},{34,49,1},{19,49,1},{63,7,882},{63,35,882},{19,49,1},{0,44,884},{63,35,882},{0,44,884},{25,0,882},{25,0,882},{25,0,882},{25,0,882},{20,50,2},{20,50,2},{20,50,2},{20,35,2},{10,40,1},{10,40,1},{31,63,2998},{28,63,951},{28,53,1324},{27,50,932},{26,63,3844},{21,58,924},{22,51,7},
{21,45,1302},{2,61,3052},{9,46,924},{34,63,1146},{32,61,9},{33,52,74},{32,50,169},{62,1,3048},{18,60,885},{22,51,6},{5,46,884},{63,31,3048},{5,46,884},{28,63,950},{28,63,950},{28,63,950},{27,48,882},{23,63,675},{22,51,6},{22,51,6},{21,40,61},{7,51,651},{13,41,8},{32,59,0},{32,59,0},{32,59,0},{32,46,1},{55,0,648},{23,50,1},{23,50,1},{15,41,0},{62,24,648},
{15,41,0},{63,13,882},{31,62,2},{36,51,1},{21,51,1},{63,13,882},{63,38,882},{21,51,1},{0,46,884},{63,38,882},{0,46,884},{27,0,882},{27,0,882},{27,0,882},{27,0,882},{22,52,2},{22,52,2},{22,52,2},{22,37,2},{12,42,1},{12,42,1},{33,63,3224},{30,63,1031},{30,55,1324},{29,52,932},{29,63,3996},{23,60,924},{24,53,7},{23,47,1302},{4,63,3052},{11,48,922},{37,63,1226},
{34,63,9},{35,54,74},{33,51,161},{63,5,3048},{20,62,885},{24,53,6},{6,48,882},{47,42,3048},{6,48,882},{30,63,995},{30,63,995},{30,63,995},{29,50,882},{25,63,705},{24,53,6},{24,53,6},{23,42,61},{9,53,651},{15,43,8},{34,61,0},{34,61,0},{34,61,0},{34,47,4},{58,0,648},{25,52,1},{25,52,1},{16,43,1},{62,27,648},{16,43,1},{63,19,882},{33,63,4},{38,53,1},
{23,53,1},{63,19,882},{63,41,882},{23,53,1},{0,48,882},{63,41,882},{0,48,882},{29,0,882},{29,0,882},{29,0,882},{29,0,882},{24,54,2},{24,54,2},{24,54,2},{24,39,2},{14,44,1},{14,44,1},{34,63,3510},{32,63,1202},{33,57,1335},{31,54,936},{33,63,4252},{25,63,925},{26,55,13},{25,49,1309},{9,63,3091},{13,50,918},{40,63,1349},{37,63,38},{37,56,77},{36,54,158},{60,17,3048},
{24,63,894},{27,55,8},{9,50,884},{63,37,3048},{9,50,884},{32,63,1058},{32,63,1058},{32,63,1058},{31,52,885},{28,63,762},{26,55,9},{26,55,9},{26,44,61},{11,56,650},{17,45,11},{36,63,2},{36,63,2},{36,63,2},{36,50,2},{53,16,648},{28,54,1},{28,54,1},{19,45,1},{63,30,648},{19,45,1},{59,33,882},{37,63,29},{40,56,4},{26,55,4},{59,33,882},{55,48,882},{26,55,4},
{0,50,884},{55,48,882},{0,50,884},{31,0,884},{31,0,884},{31,0,884},{31,0,884},{26,57,0},{26,57,0},{26,57,0},{26,42,1},{16,47,2},{16,47,2},{37,63,3734},{34,63,1399},{35,59,1335},{33,57,935},{34,63,4441},{28,63,957},{28,57,13},{27,51,1309},{13,63,3192},{15,52,918},{43,63,1485},{39,63,131},{39,58,77},{38,56,158},{63,17,3048},{28,63,957},{29,57,8},{11,52,884},{63,40,3048},
{11,52,884},{34,63,1110},{34,63,1110},{34,63,1110},{33,54,883},{30,63,840},{28,57,9},{28,57,9},{28,46,61},{13,58,650},{19,47,11},{39,63,10},{39,63,10},{39,63,10},{38,52,2},{56,16,648},{30,56,1},{30,56,1},{21,47,1},{62,33,648},{21,47,1},{62,33,882},{41,63,80},{42,58,4},{28,57,4},{62,33,882},{61,48,882},{28,57,4},{0,52,884},{61,48,882},{0,52,884},{33,0,882},
{33,0,882},{33,0,882},{33,0,882},{28,59,0},{28,59,0},{28,59,0},{28,44,1},{18,48,1},{18,48,1},{40,63,4022},{37,63,1647},{37,61,1335},{35,59,935},{37,63,4657},{31,63,1085},{30,59,13},{29,53,1309},{18,63,3339},{17,54,924},{45,63,1635},{42,63,275},{41,60,77},{40,58,158},{63,23,3048},{32,63,1044},{31,59,8},{13,54,884},{63,43,3048},{13,54,884},{36,63,1203},{36,63,1203},{36,63,1203},
{35,56,883},{33,63,915},{30,59,9},{30,59,9},{30,48,65},{15,60,650},{21,49,8},{41,63,25},{41,63,25},{41,63,25},{40,54,2},{59,16,648},{32,57,4},{32,57,4},{23,49,0},{62,36,648},{23,49,0},{63,37,882},{44,63,160},{44,60,4},{30,59,4},{63,37,882},{63,50,882},{30,59,4},{0,54,884},{63,50,882},{0,54,884},{35,0,882},{35,0,882},{35,0,882},{35,0,882},{30,61,0},
{30,61,0},{30,61,0},{30,46,1},{20,50,1},{20,50,1},{42,63,4364},{40,63,1991},{39,63,1335},{37,61,935},{40,63,4905},{34,63,1287},{32,61,10},{31,55,1309},{21,63,3555},{19,56,924},{48,63,1796},{44,63,465},{43,62,77},{42,60,158},{63,29,3048},{35,63,1188},{33,61,8},{15,56,884},{63,46,3048},{15,56,884},{39,63,1299},{39,63,1299},{39,63,1299},{37,58,883},{34,63,1017},{32,61,6},{32,61,6},
{32,50,66},{17,62,650},{23,51,8},{43,63,45},{43,63,45},{43,63,45},{42,56,2},{62,16,648},{33,60,2},{33,60,2},{25,51,0},{62,39,648},{25,51,0},{63,43,882},{48,63,260},{46,62,4},{30,62,4},{63,43,882},{63,53,882},{30,62,4},{0,56,884},{63,53,882},{0,56,884},{37,0,882},{37,0,882},{37,0,882},{37,0,882},{32,63,1},{32,63,1},{32,63,1},{32,47,2},{22,52,1},
{22,52,1},{45,63,4441},{43,63,2286},{41,63,1421},{39,62,916},{43,63,4878},{37,63,1410},{34,63,9},{33,57,1141},{26,63,3535},{21,58,795},{50,63,1770},{47,63,609},{46,63,85},{44,62,120},{63,35,2814},{40,63,1194},{35,63,2},{17,58,762},{63,49,2814},{17,58,762},{41,63,1421},{41,63,1421},{41,63,1421},{39,61,885},{37,63,1130},{34,63,9},{34,63,9},{34,52,61},{20,63,651},{25,53,11},{46,63,85},
{46,63,85},{46,63,85},{44,58,4},{63,21,648},{36,62,1},{36,62,1},{27,53,2},{63,42,648},{27,53,2},{63,48,761},{51,63,305},{48,63,1},{34,63,0},{63,48,761},{62,56,761},{34,63,0},{0,58,761},{62,56,761},{0,58,761},{39,0,884},{39,0,884},{39,0,884},{39,0,884},{34,63,9},{34,63,9},{34,63,9},{34,50,1},{24,55,2},{24,55,2},{46,63,3945},{43,63,2238},{43,63,1509},
{41,63,885},{45,63,4345},{38,63,1230},{37,63,41},{34,58,781},{30,63,3066},{24,59,494},{51,63,1386},{48,63,530},{48,63,130},{46,62,45},{63,39,2249},{43,63,914},{39,63,8},{21,59,482},{47,59,2249},{21,59,482},{43,63,1509},{43,63,1509},{43,63,1509},{41,63,885},{40,63,1242},{37,63,41},{37,63,41},{36,54,61},{24,63,689},{27,55,11},{48,63,130},{48,63,130},{48,63,130},{46,60,4},{63,27,648},
{39,63,8},{39,63,8},{29,55,2},{63,45,648},{29,55,2},{63,51,481},{54,63,193},{51,63,1},{40,63,0},{63,51,481},{63,57,481},{40,63,0},{0,59,481},{63,57,481},{0,59,481},{41,0,884},{41,0,884},{41,0,884},{41,0,884},{37,63,25},{37,63,25},{37,63,25},{36,52,1},{26,57,2},{26,57,2},{48,63,3571},{46,63,2182},{46,63,1653},{43,63,900},{46,63,3838},{41,63,1094},{40,63,137},
{37,59,490},{32,63,2703},{27,60,270},{53,63,1106},{51,63,458},{50,63,193},{48,63,10},{63,43,1769},{46,63,698},{42,63,40},{25,60,266},{63,53,1769},{25,60,266},{46,63,1653},{46,63,1653},{46,63,1653},{43,63,900},{43,63,1386},{40,63,137},{40,63,137},{38,56,61},{29,63,779},{29,57,11},{50,63,193},{50,63,193},{50,63,193},{48,62,1},{63,33,648},{42,63,40},{42,63,40},{31,57,2},{47,56,648},
{31,57,2},{63,54,265},{57,63,113},{54,63,1},{46,63,0},{63,54,265},{62,59,265},{46,63,0},{0,60,265},{62,59,265},{0,60,265},{43,0,884},{43,0,884},{43,0,884},{43,0,884},{39,63,52},{39,63,52},{39,63,52},{38,54,1},{28,59,2},{28,59,2},{50,63,3307},{48,63,2195},{48,63,1795},{46,63,964},{48,63,3435},{43,63,1026},{42,63,296},{39,60,269},{37,63,2410},{30,61,115},{54,63,882},
{54,63,450},{53,63,265},{50,63,10},{63,47,1374},{48,63,546},{46,63,89},{29,61,114},{63,55,1374},{29,61,114},{48,63,1795},{48,63,1795},{48,63,1795},{46,63,964},{45,63,1560},{42,63,296},{42,63,296},{40,58,61},{32,63,891},{31,59,11},{53,63,265},{53,63,265},{53,63,265},{50,63,10},{63,39,648},{46,63,89},{46,63,89},{33,59,1},{47,59,648},{33,59,1},{63,57,113},{58,63,50},{57,63,1},
{52,63,0},{63,57,113},{63,60,113},{52,63,0},{0,61,113},{63,60,113},{0,61,113},{45,0,884},{45,0,884},{45,0,884},{45,0,884},{41,63,97},{41,63,97},{41,63,97},{40,56,1},{30,61,2},{30,61,2},{51,63,3032},{51,63,2264},{50,63,1965},{48,63,1096},{51,63,3096},{46,63,1059},{46,63,530},{41,62,126},{40,63,2191},{33,62,25},{57,63,680},{56,63,465},{55,63,356},{53,63,68},{63,51,1032},
{52,63,450},{49,63,185},{34,62,20},{63,57,1032},{34,62,20},{50,63,1965},{50,63,1965},{50,63,1965},{48,63,1096},{48,63,1736},{46,63,530},{46,63,530},{42,60,62},{37,63,1086},{33,62,9},{55,63,356},{55,63,356},{55,63,356},{53,63,68},{61,49,648},{49,63,185},{49,63,185},{36,61,4},{63,54,648},{36,61,4},{63,61,18},{61,63,9},{61,63,0},{59,63,0},{63,61,18},{63,62,18},{59,63,0},
{0,62,20},{63,62,18},{0,62,20},{47,0,890},{47,0,890},{47,0,890},{47,0,890},{44,63,149},{44,63,149},{44,63,149},{42,58,1},{32,63,0},{32,63,0},{54,63,2756},{52,63,2249},{51,63,2004},{50,63,1224},{53,63,2803},{48,63,1092},{48,63,692},{44,62,58},{43,63,1991},{35,63,16},{59,63,566},{57,63,420},{57,63,356},{55,63,125},{63,55,771},{55,63,386},{52,63,241},{38,63,0},{63,59,771},
{38,63,0},{51,63,2004},{51,63,2004},{51,63,2004},{50,63,1224},{50,63,1802},{48,63,692},{48,63,692},{44,62,42},{40,63,1166},{35,63,16},{57,63,356},{57,63,356},{57,63,356},{55,63,125},{63,50,580},{52,63,241},{52,63,241},{38,63,0},{62,57,580},{38,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{49,0,884},
{49,0,884},{49,0,884},{49,0,884},{46,63,193},{46,63,193},{46,63,193},{44,60,1},{35,63,16},{35,63,16},{54,63,2276},{54,63,1844},{54,63,1700},{52,63,1125},{54,63,2180},{51,63,948},{49,63,649},{46,63,10},{46,63,1551},{39,63,58},{60,63,324},{59,63,257},{59,63,221},{57,63,68},{63,57,452},{57,63,228},{55,63,137},{44,63,0},{63,60,452},{44,63,0},{54,63,1700},{54,63,1700},{54,63,1700},
{52,63,1125},{51,63,1460},{49,63,649},{49,63,649},{46,63,10},{43,63,926},{39,63,58},{59,63,221},{59,63,221},{59,63,221},{57,63,68},{63,53,340},{55,63,137},{55,63,137},{44,63,0},{63,58,340},{44,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{51,0,884},{51,0,884},{51,0,884},{51,0,884},{48,63,260},
{48,63,260},{48,63,260},{46,62,1},{39,63,58},{39,63,58},{57,63,1844},{55,63,1585},{55,63,1464},{54,63,1044},{56,63,1747},{52,63,850},{51,63,596},{48,63,4},{48,63,1204},{43,63,117},{60,63,164},{60,63,116},{60,63,100},{59,63,40},{63,59,216},{58,63,114},{58,63,65},{50,63,0},{63,61,216},{50,63,0},{55,63,1464},{55,63,1464},{55,63,1464},{54,63,1044},{54,63,1188},{51,63,596},{51,63,596},
{48,63,4},{46,63,750},{43,63,117},{60,63,100},{60,63,100},{60,63,100},{59,63,40},{63,56,164},{58,63,65},{58,63,65},{50,63,0},{62,60,164},{50,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{53,0,884},{53,0,884},{53,0,884},{53,0,884},{51,63,340},{51,63,340},{51,63,340},{48,63,4},{43,63,117},
{43,63,117},{0,39,1568},{0,28,194},{0,19,10},{0,16,586},{0,26,3371},{0,17,2169},{0,16,1027},{0,10,2532},{0,12,3648},{0,10,2701},{0,39,1568},{0,28,194},{0,19,10},{0,16,586},{5,16,3371},{0,17,2169},{0,16,1027},{0,10,2532},{26,0,3371},{0,10,2532},{0,18,0},{0,18,0},{0,18,0},{0,9,0},{0,9,288},{0,8,109},{0,8,109},{0,4,164},{0,4,321},{0,4,189},{0,18,0},
{0,18,0},{0,18,0},{0,9,0},{4,1,288},{0,8,109},{0,8,109},{0,4,164},{9,0,288},{0,4,164},{19,1,1568},{0,28,194},{0,19,10},{0,16,586},{19,1,1568},{39,0,1568},{0,16,586},{0,13,1568},{39,0,1568},{0,13,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,45,1568},{0,31,106},{0,22,10},
{0,18,481},{0,30,3968},{0,19,2355},{0,18,1057},{0,13,2852},{0,14,4319},{0,11,3117},{0,45,1568},{0,31,106},{0,22,10},{0,18,481},{15,0,3968},{0,19,2355},{0,18,1057},{0,13,2852},{30,0,3968},{0,13,2852},{0,24,0},{0,24,0},{0,24,0},{0,12,0},{0,12,512},{0,10,205},{0,10,205},{0,5,313},{0,5,566},{0,5,349},{0,24,0},{0,24,0},{0,24,0},{0,12,0},{6,0,512},
{0,10,205},{0,10,205},{0,5,313},{12,0,512},{0,5,313},{22,1,1568},{0,31,106},{1,22,1},{0,18,481},{22,1,1568},{45,0,1568},{0,18,481},{0,15,1568},{45,0,1568},{0,15,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,50,1570},{0,34,52},{1,24,58},{0,21,377},{0,34,4652},{0,22,2571},{0,19,1107},
{0,14,3225},{0,16,5108},{0,13,3525},{0,50,1570},{0,34,52},{1,24,42},{0,21,377},{15,4,4651},{0,22,2571},{0,19,1107},{0,14,3225},{30,2,4651},{0,14,3225},{0,30,0},{0,30,0},{0,30,0},{0,15,0},{0,15,800},{0,11,317},{0,11,317},{0,7,468},{0,7,889},{0,7,549},{0,30,0},{0,30,0},{0,30,0},{0,15,0},{7,1,800},{0,11,317},{0,11,317},{0,7,468},{15,0,800},
{0,7,468},{25,1,1568},{0,34,52},{3,24,1},{0,21,377},{25,1,1568},{47,2,1568},{0,21,377},{0,17,1570},{47,2,1568},{0,17,1570},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,56,1570},{0,38,13},{1,27,138},{0,24,305},{0,38,5419},{0,25,2819},{0,22,1187},{0,16,3659},{0,17,6013},{0,14,4061},{0,56,1570},
{0,38,13},{1,27,122},{0,24,305},{19,0,5419},{0,25,2819},{0,22,1187},{0,16,3659},{38,0,5419},{0,16,3659},{0,36,0},{0,36,0},{0,36,0},{0,18,0},{0,18,1152},{0,14,445},{0,14,445},{0,8,697},{0,8,1270},{0,8,797},{0,36,0},{0,36,0},{0,36,0},{0,18,0},{9,0,1152},{0,14,445},{0,14,445},{0,8,697},{18,0,1152},{0,8,697},{28,1,1568},{0,38,13},{5,26,1},
{0,24,305},{28,1,1568},{47,5,1568},{0,24,305},{0,19,1570},{47,5,1568},{0,19,1570},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,61,1609},{1,41,43},{2,29,250},{1,26,301},{0,45,5419},{0,28,2552},{0,25,820},{0,18,3377},{0,21,6243},{0,18,3953},{2,59,1569},{2,41,10},{3,29,125},{1,26,285},{22,1,5419},
{0,28,2552},{0,25,820},{0,18,3377},{45,0,5419},{0,18,3377},{1,40,42},{1,40,42},{1,40,42},{1,21,42},{0,25,1152},{0,19,292},{0,19,292},{0,11,569},{0,11,1380},{0,10,747},{2,38,2},{2,38,2},{2,38,2},{2,20,2},{12,1,1152},{0,19,292},{0,19,292},{0,11,569},{25,0,1152},{0,11,569},{23,17,1568},{0,42,1},{7,28,1},{0,26,233},{23,17,1568},{63,0,1568},{0,26,233},
{0,21,1568},{63,0,1568},{0,21,1568},{1,0,41},{1,0,41},{1,0,41},{1,0,41},{0,7,0},{0,7,0},{0,7,0},{0,3,1},{0,3,17},{0,3,17},{2,63,1731},{2,44,146},{3,31,441},{1,28,370},{0,50,5420},{0,33,2329},{0,27,554},{0,21,3217},{0,23,6476},{0,19,3861},{4,61,1569},{4,43,10},{5,31,125},{3,28,285},{25,1,5419},{0,33,2329},{0,27,554},{0,21,3217},{47,2,5419},
{0,21,3217},{2,44,146},{2,44,146},{2,44,146},{2,23,146},{0,31,1152},{0,22,180},{0,22,180},{0,13,458},{0,14,1508},{0,13,747},{4,40,2},{4,40,2},{4,40,2},{4,22,2},{15,1,1152},{0,22,180},{0,22,180},{0,13,458},{31,0,1152},{0,13,458},{34,1,1568},{2,44,1},{9,30,1},{0,28,164},{34,1,1568},{63,3,1568},{0,28,164},{0,23,1568},{63,3,1568},{0,23,1568},{2,0,145},
{2,0,145},{2,0,145},{2,0,145},{0,13,0},{0,13,0},{0,13,0},{0,6,1},{0,5,52},{0,5,52},{4,63,1977},{3,47,318},{5,33,675},{3,30,498},{0,56,5420},{0,36,2129},{0,30,338},{0,22,3012},{0,25,6733},{0,21,3825},{6,63,1569},{6,45,10},{8,33,123},{5,30,285},{28,1,5419},{0,36,2129},{0,30,338},{0,22,3012},{47,5,5419},{0,22,3012},{3,48,313},{3,48,313},{3,48,313},
{3,26,314},{0,36,1154},{0,25,100},{0,25,100},{0,16,388},{0,16,1665},{0,14,757},{6,42,2},{6,42,2},{6,42,2},{6,24,2},{15,7,1152},{0,25,100},{0,25,100},{0,16,388},{31,3,1152},{0,16,388},{37,1,1568},{4,46,1},{11,32,2},{0,30,113},{37,1,1568},{63,6,1568},{0,30,113},{0,25,1568},{63,6,1568},{0,25,1568},{3,0,313},{3,0,313},{3,0,313},{3,0,313},{0,19,0},
{0,19,0},{0,19,0},{0,9,1},{0,8,116},{0,8,116},{6,63,2353},{4,49,562},{6,35,1006},{3,32,715},{0,62,5420},{0,39,1961},{0,32,174},{0,24,2817},{0,28,7013},{0,24,3841},{9,63,1577},{8,47,10},{10,35,123},{7,31,290},{31,1,5419},{0,39,1961},{0,32,174},{0,24,2817},{47,8,5419},{0,24,2817},{4,52,545},{4,52,545},{4,52,545},{4,28,546},{0,42,1154},{0,29,45},{0,29,45},
{0,18,289},{0,19,1849},{0,16,797},{8,44,2},{8,44,2},{8,44,2},{8,26,2},{15,13,1152},{0,29,45},{0,29,45},{0,18,289},{31,6,1152},{0,18,289},{40,1,1568},{6,48,1},{13,34,2},{0,33,73},{40,1,1568},{63,9,1568},{0,33,73},{0,27,1568},{63,9,1568},{0,27,1568},{4,0,545},{4,0,545},{4,0,545},{4,0,545},{0,25,0},{0,25,0},{0,25,0},{0,12,1},{0,11,212},
{0,11,212},{8,63,2980},{6,52,920},{7,38,1444},{5,34,1012},{2,63,5504},{0,42,1814},{0,35,57},{0,27,2630},{0,31,7380},{0,25,3860},{11,63,1604},{10,49,13},{12,37,116},{10,33,292},{34,1,5419},{0,42,1814},{0,35,57},{0,27,2630},{63,3,5419},{0,27,2630},{5,57,884},{5,57,884},{5,57,884},{5,31,884},{0,49,1152},{0,33,5},{0,33,5},{0,21,221},{0,22,2091},{0,19,875},{11,45,4},
{11,45,4},{11,45,4},{11,28,4},{24,1,1152},{0,33,5},{0,33,5},{0,21,221},{49,0,1152},{0,21,221},{44,0,1568},{8,50,1},{15,36,4},{0,35,32},{44,0,1568},{62,13,1568},{0,35,32},{0,29,1570},{62,13,1568},{0,29,1570},{5,0,884},{5,0,884},{5,0,884},{5,0,884},{0,31,1},{0,31,1},{0,31,1},{0,16,1},{0,14,349},{0,14,349},{9,63,3638},{7,55,1309},{8,39,1940},
{6,36,1365},{3,63,5739},{0,45,1718},{0,38,17},{0,29,2514},{0,34,7760},{0,28,3900},{14,63,1636},{12,51,13},{14,39,116},{12,35,292},{37,1,5419},{0,45,1718},{0,38,17},{0,29,2514},{63,6,5419},{0,29,2514},{6,61,1252},{6,61,1252},{6,61,1252},{6,33,1253},{0,55,1152},{0,37,4},{0,37,4},{0,22,162},{0,25,2339},{0,21,989},{13,47,4},{13,47,4},{13,47,4},{13,30,4},{27,1,1152},
{1,36,1},{1,36,1},{0,22,162},{55,0,1152},{0,22,162},{47,0,1568},{10,52,1},{17,38,1},{0,38,16},{47,0,1568},{46,24,1568},{0,38,16},{0,31,1570},{46,24,1568},{0,31,1570},{6,0,1252},{6,0,1252},{6,0,1252},{6,0,1252},{0,37,0},{0,37,0},{0,37,0},{0,19,1},{0,16,490},{0,16,490},{11,63,4328},{8,57,1644},{10,41,2360},{7,39,1656},{6,63,6079},{0,49,1644},{1,40,17},
{0,31,2359},{0,36,7943},{0,30,3834},{16,63,1689},{14,53,13},{16,41,129},{14,37,292},{40,1,5419},{0,49,1640},{1,40,13},{0,31,2355},{63,9,5419},{0,31,2355},{8,63,1576},{8,63,1576},{8,63,1576},{7,36,1576},{1,59,1156},{1,39,13},{1,39,13},{0,25,110},{0,28,2475},{0,24,1017},{15,49,4},{15,49,4},{15,49,4},{15,32,4},{30,1,1152},{3,38,1},{3,38,1},{0,25,106},{61,0,1152},
{0,25,106},{49,1,1568},{12,54,1},{19,40,1},{0,40,4},{49,1,1568},{46,27,1568},{0,40,4},{0,33,1568},{46,27,1568},{0,33,1568},{7,0,1576},{7,0,1576},{7,0,1576},{7,0,1576},{1,41,4},{1,41,4},{1,41,4},{1,21,4},{0,19,578},{0,19,578},{14,63,4584},{10,59,1644},{12,43,2360},{9,41,1656},{8,63,6244},{2,51,1644},{3,42,17},{1,33,2308},{0,42,7575},{0,32,3345},{19,63,1761},
{16,55,10},{17,43,125},{16,40,290},{43,1,5419},{0,52,1592},{3,42,13},{0,33,2225},{63,12,5419},{0,33,2225},{10,63,1585},{10,63,1585},{10,63,1585},{9,38,1576},{3,61,1156},{3,41,13},{3,41,13},{2,27,110},{0,31,2211},{0,25,699},{16,52,2},{16,52,2},{16,52,2},{16,34,2},{33,1,1152},{5,40,1},{5,40,1},{0,27,61},{63,2,1152},{0,27,61},{52,1,1568},{14,56,1},{21,42,1},
{1,42,0},{52,1,1568},{46,30,1568},{1,42,0},{0,35,1568},{46,30,1568},{0,35,1568},{9,0,1576},{9,0,1576},{9,0,1576},{9,0,1576},{3,43,4},{3,43,4},{3,43,4},{3,23,4},{0,22,410},{0,22,410},{15,63,4826},{12,62,1650},{14,46,2355},{11,43,1660},{11,63,6452},{4,54,1641},{5,44,15},{3,35,2308},{0,45,7165},{0,35,2875},{22,63,1862},{18,57,9},{20,45,126},{18,42,281},{47,0,5419},
{0,56,1568},{6,44,10},{0,35,2091},{46,24,5419},{0,35,2091},{12,63,1606},{12,63,1606},{12,63,1606},{12,40,1571},{5,63,1155},{5,44,14},{5,44,14},{4,29,109},{0,33,1977},{0,28,424},{19,53,4},{19,53,4},{19,53,4},{19,36,4},{31,12,1152},{7,42,1},{7,42,1},{0,30,29},{62,6,1152},{0,30,29},{56,0,1568},{17,58,1},{24,44,2},{3,44,2},{56,0,1568},{62,25,1568},{3,44,2},
{0,37,1570},{62,25,1568},{0,37,1570},{12,0,1570},{12,0,1570},{12,0,1570},{12,0,1570},{5,46,1},{5,46,1},{5,46,1},{5,25,2},{0,25,260},{0,25,260},{17,63,5096},{14,63,1652},{15,48,2358},{13,45,1660},{14,63,6668},{6,56,1641},{7,46,15},{5,37,2308},{0,47,6891},{0,36,2543},{23,63,1956},{20,59,9},{22,47,126},{20,44,281},{49,1,5419},{2,58,1568},{8,46,10},{0,38,1979},{46,27,5419},
{0,38,1979},{14,63,1651},{14,63,1651},{14,63,1651},{14,42,1571},{8,63,1179},{7,46,14},{7,46,14},{6,31,109},{0,38,1778},{0,31,232},{21,55,4},{21,55,4},{21,55,4},{21,38,4},{31,18,1152},{9,44,1},{9,44,1},{0,32,10},{62,9,1152},{0,32,10},{59,0,1568},{19,60,1},{26,46,2},{5,46,2},{59,0,1568},{62,28,1568},{5,46,2},{0,39,1570},{62,28,1568},{0,39,1570},{14,0,1570},
{14,0,1570},{14,0,1570},{14,0,1570},{7,48,1},{7,48,1},{7,48,1},{7,27,2},{0,28,164},{0,28,164},{20,63,5352},{17,63,1708},{18,49,2360},{15,47,1660},{15,63,6877},{8,58,1641},{9,48,14},{7,39,2308},{0,50,6576},{0,39,2215},{26,63,2052},{22,61,9},{24,49,116},{22,46,281},{52,1,5419},{4,60,1568},{9,48,13},{0,40,1907},{46,30,5419},{0,40,1907},{16,63,1697},{16,63,1697},{16,63,1697},
{16,44,1577},{10,63,1209},{9,48,14},{9,48,14},{8,33,115},{0,42,1601},{0,33,110},{23,57,4},{23,57,4},{23,57,4},{23,40,4},{34,17,1152},{11,46,1},{11,46,1},{0,34,2},{62,12,1152},{0,34,2},{62,0,1568},{21,62,1},{28,48,1},{7,48,0},{62,0,1568},{62,31,1568},{7,48,0},{0,41,1570},{62,31,1568},{0,41,1570},{16,0,1576},{16,0,1576},{16,0,1576},{16,0,1576},{9,50,1},
{9,50,1},{9,50,1},{9,29,2},{0,33,85},{0,33,85},{23,63,5672},{19,63,1825},{20,51,2360},{17,49,1656},{19,63,7135},{10,60,1641},{11,50,14},{9,41,2308},{0,53,6336},{0,42,1983},{29,63,2180},{24,63,9},{26,51,116},{24,48,293},{55,1,5419},{6,62,1568},{11,50,13},{0,42,1814},{47,32,5419},{0,42,1814},{19,63,1761},{19,63,1761},{19,63,1761},{17,46,1576},{12,63,1249},{11,50,14},{11,50,14},
{10,35,115},{0,45,1449},{0,36,30},{25,59,4},{25,59,4},{25,59,4},{25,42,4},{37,17,1152},{13,48,0},{13,48,0},{1,36,2},{62,15,1152},{1,36,2},{63,4,1568},{23,63,4},{30,50,1},{9,50,0},{63,4,1568},{62,34,1568},{9,50,0},{0,43,1570},{62,34,1568},{0,43,1570},{17,0,1576},{17,0,1576},{17,0,1576},{17,0,1576},{11,52,1},{11,52,1},{11,52,1},{11,31,2},{0,36,29},
{0,36,29},{25,63,6066},{21,63,2039},{22,54,2355},{19,51,1660},{20,63,7420},{12,62,1635},{13,52,14},{11,43,2316},{0,57,6109},{0,44,1789},{31,63,2369},{27,63,38},{28,53,115},{26,50,286},{59,0,5419},{9,63,1577},{14,52,11},{0,44,1740},{62,28,5419},{0,44,1740},{21,63,1843},{21,63,1843},{21,63,1843},{20,48,1571},{15,63,1314},{13,52,13},{13,52,13},{12,37,116},{0,48,1329},{0,39,13},{27,62,1},
{27,62,1},{27,62,1},{27,44,2},{49,0,1152},{15,50,2},{15,50,2},{4,38,4},{62,18,1152},{4,38,4},{63,11,1568},{27,63,37},{32,52,2},{12,52,1},{63,11,1568},{47,45,1568},{12,52,1},{0,45,1576},{47,45,1568},{0,45,1576},{20,0,1570},{20,0,1570},{20,0,1570},{20,0,1570},{13,54,1},{13,54,1},{13,54,1},{13,34,1},{0,40,4},{0,40,4},{28,63,6434},{23,63,2268},{24,56,2355},
{21,53,1660},{23,63,7668},{14,63,1652},{15,54,14},{13,45,2316},{0,61,5924},{0,47,1685},{34,63,2502},{29,63,123},{30,55,115},{28,52,286},{62,0,5419},{13,63,1627},{16,54,10},{0,47,1676},{62,31,5419},{0,47,1676},{23,63,1907},{23,63,1907},{23,63,1907},{22,50,1571},{17,63,1395},{15,54,13},{15,54,13},{14,39,116},{0,51,1241},{2,41,13},{29,63,2},{29,63,2},{29,63,2},{29,46,2},{52,0,1152},
{17,52,1},{17,52,1},{6,40,4},{62,21,1152},{6,40,4},{55,32,1568},{31,63,97},{34,54,2},{14,54,1},{55,32,1568},{46,48,1568},{14,54,1},{0,47,1576},{46,48,1568},{0,47,1576},{22,0,1570},{22,0,1570},{22,0,1570},{22,0,1570},{15,56,1},{15,56,1},{15,56,1},{15,36,1},{1,43,1},{1,43,1},{29,63,6756},{26,63,2548},{26,58,2355},{23,55,1660},{26,63,7948},{17,63,1716},{17,56,15},
{15,47,2316},{0,63,5773},{0,49,1638},{37,63,2694},{32,63,262},{32,57,126},{30,54,286},{63,4,5419},{17,63,1715},{18,56,10},{0,49,1634},{62,34,5419},{0,49,1634},{25,63,2018},{25,63,2018},{25,63,2018},{24,52,1571},{20,63,1483},{17,56,14},{17,56,14},{16,41,109},{0,54,1185},{4,43,13},{31,63,17},{31,63,17},{31,63,17},{31,48,2},{55,0,1152},{19,54,1},{19,54,1},{8,42,4},{62,24,1152},
{8,42,4},{58,32,1568},{34,63,169},{36,56,2},{15,56,2},{58,32,1568},{52,48,1568},{15,56,2},{0,49,1570},{52,48,1568},{0,49,1570},{24,0,1570},{24,0,1570},{24,0,1570},{24,0,1570},{17,58,1},{17,58,1},{17,58,1},{17,37,2},{3,45,1},{3,45,1},{31,63,7218},{29,63,2924},{28,60,2355},{25,57,1660},{29,63,8260},{20,63,1884},{19,58,15},{17,49,2308},{3,63,5933},{2,51,1638},{39,63,2892},
{34,63,445},{34,59,126},{32,56,281},{63,10,5419},{21,63,1849},{20,58,10},{0,51,1606},{62,37,5419},{0,51,1606},{28,63,2130},{28,63,2130},{28,63,2130},{26,54,1571},{23,63,1603},{19,58,14},{19,58,14},{18,43,109},{0,58,1156},{6,45,13},{33,63,40},{33,63,40},{33,63,40},{33,50,4},{58,0,1152},{21,56,1},{21,56,1},{10,44,4},{62,27,1152},{10,44,4},{61,32,1568},{38,63,274},{38,58,2},
{17,58,2},{61,32,1568},{58,48,1568},{17,58,2},{0,51,1570},{58,48,1568},{0,51,1570},{26,0,1570},{26,0,1570},{26,0,1570},{26,0,1570},{19,60,1},{19,60,1},{19,60,1},{19,39,2},{5,47,1},{5,47,1},{34,63,7586},{31,63,3453},{30,62,2357},{28,59,1668},{31,63,8699},{23,63,2180},{21,60,21},{19,51,2316},{7,63,6224},{4,53,1634},{42,63,3131},{37,63,722},{36,62,125},{34,58,278},{55,32,5419},
{26,63,2052},{22,60,9},{0,53,1580},{46,48,5419},{0,53,1580},{30,63,2272},{30,63,2272},{30,63,2272},{28,56,1568},{25,63,1746},{21,60,17},{21,60,17},{21,45,113},{1,61,1154},{9,47,14},{36,63,74},{36,63,74},{36,63,74},{35,52,2},{53,16,1152},{23,59,1},{23,59,1},{11,47,1},{63,30,1152},{11,47,1},{63,35,1568},{43,63,433},{40,60,4},{20,60,4},{63,35,1568},{63,49,1568},{20,60,4},
{0,53,1576},{63,49,1568},{0,53,1576},{28,0,1568},{28,0,1568},{28,0,1568},{28,0,1568},{21,63,0},{21,63,0},{21,63,0},{21,42,0},{7,49,1},{7,49,1},{37,63,8018},{34,63,3915},{32,63,2410},{30,61,1668},{34,63,8985},{26,63,2548},{23,62,21},{21,53,2316},{12,63,6555},{6,55,1634},{43,63,3345},{40,63,1026},{38,63,129},{36,60,278},{58,32,5419},{30,63,2274},{24,62,9},{0,55,1577},{52,48,5419},
{0,55,1577},{32,63,2406},{32,63,2406},{32,63,2406},{30,58,1568},{28,63,1890},{23,62,17},{23,62,17},{23,47,113},{3,63,1154},{11,49,12},{38,63,125},{38,63,125},{38,63,125},{37,54,2},{56,16,1152},{25,61,1},{25,61,1},{14,48,5},{62,33,1152},{14,48,5},{63,41,1568},{46,63,585},{42,62,4},{22,62,4},{63,41,1568},{63,52,1568},{22,62,4},{0,55,1576},{63,52,1568},{0,55,1576},{30,0,1568},
{30,0,1568},{30,0,1568},{30,0,1568},{23,63,4},{23,63,4},{23,63,4},{23,44,0},{9,51,1},{9,51,1},{39,63,7700},{35,63,4026},{34,63,2514},{32,62,1611},{37,63,8485},{29,63,2424},{26,63,20},{23,55,1896},{15,63,6115},{9,56,1308},{46,63,2973},{43,63,1034},{40,63,169},{37,61,194},{63,27,4803},{32,63,2024},{27,63,1},{2,57,1253},{63,45,4803},{2,57,1253},{34,63,2514},{34,63,2514},{34,63,2514},
{32,60,1570},{29,63,2056},{26,63,20},{26,63,20},{24,49,116},{7,63,1164},{13,51,12},{40,63,169},{40,63,169},{40,63,169},{39,56,2},{59,16,1152},{27,63,1},{27,63,1},{16,50,4},{62,36,1152},{16,50,4},{63,45,1250},{48,63,500},{45,63,1},{26,63,0},{63,45,1250},{47,62,1250},{26,63,0},{0,57,1252},{47,62,1250},{0,57,1252},{32,0,1570},{32,0,1570},{32,0,1570},{32,0,1570},{26,63,20},
{26,63,20},{26,63,20},{25,46,0},{11,53,1},{11,53,1},{40,63,7062},{37,63,3915},{37,63,2690},{34,63,1579},{37,63,7765},{30,63,2178},{28,63,77},{26,56,1437},{18,63,5499},{11,57,918},{48,63,2504},{44,63,945},{43,63,225},{40,62,89},{63,31,4056},{35,63,1656},{30,63,9},{6,58,885},{63,47,4056},{6,58,885},{37,63,2690},{37,63,2690},{37,63,2690},{34,62,1570},{33,63,2227},{28,63,77},{28,63,77},
{26,51,116},{11,63,1227},{15,53,12},{43,63,225},{43,63,225},{43,63,225},{41,58,2},{62,16,1152},{30,63,9},{30,63,9},{18,52,4},{62,39,1152},{18,52,4},{63,47,884},{51,63,356},{47,63,4},{32,63,0},{63,47,884},{63,55,884},{32,63,0},{0,58,884},{63,55,884},{0,58,884},{34,0,1570},{34,0,1570},{34,0,1570},{34,0,1570},{28,63,41},{28,63,41},{28,63,41},{27,48,1},{13,55,1},
{13,55,1},{43,63,6493},{40,63,3882},{39,63,2880},{36,63,1584},{40,63,6982},{34,63,1966},{31,63,206},{28,57,971},{23,63,4927},{14,59,562},{50,63,2070},{46,63,842},{46,63,313},{43,62,25},{63,35,3318},{38,63,1326},{34,63,45},{11,59,545},{63,49,3318},{11,59,545},{39,63,2880},{39,63,2880},{39,63,2880},{36,63,1584},{34,63,2434},{31,63,206},{31,63,206},{29,53,114},{15,63,1329},{17,55,14},{46,63,313},
{46,63,313},{46,63,313},{43,61,0},{63,21,1152},{34,63,45},{34,63,45},{19,55,1},{63,42,1152},{19,55,1},{63,51,545},{54,63,225},{51,63,1},{39,63,0},{63,51,545},{63,57,545},{39,63,0},{0,59,545},{63,57,545},{0,59,545},{36,0,1568},{36,0,1568},{36,0,1568},{36,0,1568},{31,63,85},{31,63,85},{31,63,85},{29,50,1},{15,57,1},{15,57,1},{45,63,6113},{43,63,3938},{40,63,3065},
{38,63,1649},{43,63,6422},{35,63,1878},{34,63,365},{30,59,651},{26,63,4495},{16,60,318},{51,63,1698},{48,63,794},{48,63,394},{45,63,0},{63,39,2753},{41,63,1094},{38,63,106},{15,60,313},{47,59,2753},{15,60,313},{40,63,3065},{40,63,3065},{40,63,3065},{38,63,1649},{37,63,2610},{34,63,365},{34,63,365},{31,55,114},{20,63,1483},{19,57,14},{48,63,394},{48,63,394},{48,63,394},{45,63,0},{63,27,1152},
{38,63,106},{38,63,106},{21,57,1},{63,45,1152},{21,57,1},{63,54,313},{55,63,130},{54,63,1},{45,63,0},{63,54,313},{62,59,313},{45,63,0},{0,60,313},{62,59,313},{0,60,313},{38,0,1568},{38,0,1568},{38,0,1568},{38,0,1568},{33,63,128},{33,63,128},{33,63,128},{31,52,1},{17,59,1},{17,59,1},{46,63,5677},{43,63,3954},{43,63,3225},{40,63,1764},{45,63,5985},{37,63,1830},{37,63,605},
{32,60,410},{29,63,4159},{19,61,146},{53,63,1454},{51,63,770},{50,63,493},{47,63,25},{63,43,2273},{44,63,926},{41,63,194},{19,61,145},{63,53,2273},{19,61,145},{43,63,3225},{43,63,3225},{43,63,3225},{40,63,1764},{40,63,2818},{37,63,605},{37,63,605},{33,57,113},{23,63,1659},{21,59,14},{50,63,493},{50,63,493},{50,63,493},{47,63,25},{63,33,1152},{41,63,194},{41,63,194},{23,59,1},{47,56,1152},
{23,59,1},{63,57,145},{58,63,58},{57,63,1},{50,63,1},{63,57,145},{63,60,145},{50,63,1},{0,61,145},{63,60,145},{0,61,145},{40,0,1568},{40,0,1568},{40,0,1568},{40,0,1568},{34,63,185},{34,63,185},{34,63,185},{33,54,0},{19,61,1},{19,61,1},{48,63,5379},{46,63,3978},{46,63,3449},{43,63,1924},{46,63,5542},{40,63,1870},{38,63,890},{34,61,225},{32,63,3919},{22,62,43},{54,63,1242},
{53,63,809},{51,63,610},{49,63,101},{63,47,1878},{48,63,810},{44,63,314},{23,62,41},{63,55,1878},{23,62,41},{46,63,3449},{46,63,3449},{46,63,3449},{43,63,1924},{43,63,3058},{38,63,890},{38,63,890},{35,59,113},{27,63,1889},{23,61,14},{51,63,610},{51,63,610},{51,63,610},{49,63,101},{63,39,1152},{44,63,314},{44,63,314},{25,61,1},{47,59,1152},{25,61,1},{63,60,41},{60,63,17},{60,63,1},
{56,63,1},{63,60,41},{62,62,41},{56,63,1},{0,62,41},{62,62,41},{0,62,41},{42,0,1568},{42,0,1568},{42,0,1568},{42,0,1568},{37,63,233},{37,63,233},{37,63,233},{35,56,0},{21,63,1},{21,63,1},{50,63,5170},{48,63,4080},{48,63,3680},{46,63,2195},{48,63,5200},{43,63,2023},{41,63,1269},{37,62,133},{35,63,3772},{25,63,17},{57,63,1088},{55,63,861},{54,63,720},{52,63,241},{63,51,1536},
{51,63,768},{49,63,461},{28,63,0},{63,57,1536},{28,63,0},{48,63,3680},{48,63,3680},{48,63,3680},{46,63,2195},{46,63,3345},{41,63,1269},{41,63,1269},{37,61,114},{32,63,2200},{25,63,17},{54,63,720},{54,63,720},{54,63,720},{52,63,241},{61,49,1152},{49,63,461},{49,63,461},{28,63,0},{63,54,1152},{28,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},
{0,63,0},{63,63,0},{0,63,0},{44,0,1570},{44,0,1570},{44,0,1570},{44,0,1570},{40,63,317},{40,63,317},{40,63,317},{37,58,2},{25,63,17},{25,63,17},{51,63,4416},{50,63,3629},{48,63,3296},{47,63,2070},{50,63,4411},{46,63,1823},{43,63,1150},{39,63,50},{38,63,3132},{29,63,52},{57,63,768},{57,63,576},{56,63,505},{54,63,160},{63,53,1068},{54,63,544},{51,63,320},{33,63,1},{63,58,1068},
{33,63,1},{48,63,3296},{48,63,3296},{48,63,3296},{47,63,2070},{46,63,2881},{43,63,1150},{43,63,1150},{39,62,42},{34,63,1846},{29,63,52},{56,63,505},{56,63,505},{56,63,505},{54,63,160},{63,48,800},{51,63,320},{51,63,320},{33,63,1},{62,56,800},{33,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{46,0,1570},
{46,0,1570},{46,0,1570},{46,0,1570},{42,63,410},{42,63,410},{42,63,410},{39,60,2},{29,63,52},{29,63,52},{53,63,3826},{51,63,3136},{51,63,2880},{49,63,1961},{51,63,3648},{46,63,1615},{46,63,1086},{41,63,5},{41,63,2588},{32,63,116},{59,63,498},{57,63,384},{57,63,320},{56,63,116},{63,55,683},{55,63,342},{54,63,208},{39,63,1},{63,59,683},{39,63,1},{51,63,2880},{51,63,2880},{51,63,2880},
{49,63,1961},{48,63,2448},{46,63,1086},{46,63,1086},{41,63,5},{37,63,1558},{32,63,116},{57,63,320},{57,63,320},{57,63,320},{56,63,116},{63,51,512},{54,63,208},{54,63,208},{39,63,1},{63,57,512},{39,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{48,0,1568},{48,0,1568},{48,0,1568},{48,0,1568},{45,63,514},
{45,63,514},{45,63,514},{41,62,2},{32,63,116},{32,63,116},{54,63,3232},{53,63,2781},{53,63,2585},{51,63,1856},{53,63,3067},{48,63,1456},{48,63,1056},{43,63,10},{44,63,2140},{35,63,212},{60,63,272},{59,63,221},{59,63,185},{57,63,64},{63,57,384},{57,63,192},{55,63,125},{45,63,1},{63,60,384},{45,63,1},{53,63,2585},{53,63,2585},{53,63,2585},{51,63,1856},{51,63,2112},{48,63,1056},{48,63,1056},
{43,63,10},{40,63,1334},{35,63,212},{59,63,185},{59,63,185},{59,63,185},{57,63,64},{63,54,288},{55,63,125},{55,63,125},{45,63,1},{62,59,288},{45,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{50,0,1568},{50,0,1568},{50,0,1568},{50,0,1568},{46,63,605},{46,63,605},{46,63,605},{43,63,10},{35,63,212},
{35,63,212},{0,51,2665},{0,36,306},{0,26,5},{0,22,965},{0,34,5885},{0,22,3726},{0,21,1754},{0,14,4398},{0,16,6359},{0,13,4722},{0,51,2665},{0,36,306},{0,26,5},{0,22,965},{17,0,5885},{0,22,3726},{0,21,1754},{0,14,4398},{34,0,5885},{0,14,4398},{0,25,0},{0,25,0},{0,25,0},{0,12,1},{0,12,545},{0,11,212},{0,11,212},{0,6,337},{0,5,605},{0,5,374},{0,25,0},
{0,25,0},{0,25,0},{0,12,1},{6,0,545},{0,11,212},{0,11,212},{0,6,337},{12,0,545},{0,6,337},{26,0,2665},{0,36,306},{0,26,5},{0,22,965},{26,0,2665},{51,0,2665},{0,22,965},{0,17,2665},{51,0,2665},{0,17,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,57,2665},{0,39,202},{0,29,13},
{0,24,818},{0,39,6669},{0,25,3974},{0,22,1790},{0,16,4826},{0,18,7285},{0,16,5267},{0,57,2665},{0,39,202},{0,29,13},{0,24,818},{19,1,6669},{0,25,3974},{0,22,1790},{0,16,4826},{39,0,6669},{0,16,4826},{0,31,0},{0,31,0},{0,31,0},{0,15,1},{0,15,841},{0,12,337},{0,12,337},{0,7,493},{0,7,934},{0,7,574},{0,31,0},{0,31,0},{0,31,0},{0,15,1},{8,0,841},
{0,12,337},{0,12,337},{0,7,493},{15,0,841},{0,7,493},{29,0,2665},{0,39,202},{1,28,2},{0,24,818},{29,0,2665},{57,0,2665},{0,24,818},{0,19,2665},{57,0,2665},{0,19,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,63,2665},{0,44,116},{1,31,66},{0,27,698},{0,42,7541},{0,28,4254},{0,25,1854},
{0,18,5281},{0,19,8271},{0,16,5795},{0,63,2665},{0,44,116},{1,31,50},{0,27,698},{15,13,7538},{0,28,4254},{0,25,1854},{0,18,5281},{31,6,7538},{0,18,5281},{0,36,1},{0,36,1},{0,36,1},{0,18,1},{0,18,1201},{0,14,468},{0,14,468},{0,8,730},{0,8,1325},{0,8,830},{0,36,1},{0,36,1},{0,36,1},{0,18,1},{9,0,1201},{0,14,468},{0,14,468},{0,8,730},{18,0,1201},
{0,8,730},{32,0,2665},{0,44,116},{3,30,2},{0,27,698},{32,0,2665},{63,0,2665},{0,27,698},{0,21,2665},{63,0,2665},{0,21,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,63,2781},{0,47,52},{1,33,148},{0,29,610},{0,46,8493},{0,31,4566},{0,27,1962},{0,18,5809},{0,22,9367},{0,18,6385},{2,63,2753},
{0,47,52},{2,33,129},{0,29,610},{23,0,8493},{0,31,4566},{0,27,1962},{0,18,5809},{46,0,8493},{0,18,5809},{0,42,1},{0,42,1},{0,42,1},{0,21,1},{0,21,1625},{0,16,637},{0,16,637},{0,10,965},{0,10,1806},{0,10,1134},{0,42,1},{0,42,1},{0,42,1},{0,21,1},{11,0,1625},{0,16,637},{0,16,637},{0,10,965},{21,0,1625},{0,10,965},{35,0,2665},{0,47,52},{5,32,1},
{0,29,610},{35,0,2665},{63,3,2665},{0,29,610},{0,23,2665},{63,3,2665},{0,23,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,63,3105},{0,51,10},{1,35,297},{0,32,481},{0,51,9670},{0,33,4965},{0,30,2120},{0,21,6413},{0,24,10749},{0,19,7191},{3,63,2989},{0,51,10},{2,36,257},{0,32,481},{25,1,9669},
{0,33,4965},{0,30,2120},{0,21,6413},{47,2,9669},{0,21,6413},{0,49,0},{0,49,0},{0,49,0},{0,25,1},{0,25,2178},{0,19,850},{0,19,850},{0,11,1325},{0,11,2406},{0,10,1521},{0,49,0},{0,49,0},{0,49,0},{0,25,1},{12,1,2178},{0,19,850},{0,19,850},{0,11,1325},{25,0,2178},{0,11,1325},{38,0,2665},{0,51,10},{8,34,2},{0,32,481},{38,0,2665},{62,7,2665},{0,32,481},
{0,25,2669},{62,7,2665},{0,25,2669},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{3,63,3437},{1,54,33},{3,37,425},{1,33,457},{0,57,9670},{0,36,4629},{0,31,1654},{0,24,6165},{0,26,11014},{0,22,7031},{6,63,3101},{2,53,10},{4,38,257},{1,33,441},{28,1,9669},{0,36,4629},{0,31,1654},{0,24,6165},{47,5,9669},
{0,24,6165},{1,53,32},{1,53,32},{1,53,32},{1,27,32},{0,31,2178},{0,22,666},{0,22,666},{0,13,1160},{0,14,2534},{0,13,1449},{2,51,0},{2,51,0},{2,51,0},{2,27,1},{15,1,2178},{0,22,666},{0,22,666},{0,13,1160},{31,0,2178},{0,13,1160},{41,0,2665},{1,54,1},{10,36,2},{0,33,394},{41,0,2665},{62,10,2665},{0,33,394},{0,27,2669},{62,10,2665},{0,27,2669},{1,0,32},
{1,0,32},{1,0,32},{1,0,32},{0,6,0},{0,6,0},{0,6,0},{0,3,0},{0,2,13},{0,2,13},{6,63,3917},{2,57,129},{3,40,609},{1,36,497},{0,63,9670},{0,39,4325},{0,33,1274},{0,25,5878},{0,28,11299},{0,24,6917},{8,63,3233},{4,55,10},{6,40,257},{3,35,441},{31,1,9669},{0,39,4325},{0,33,1274},{0,25,5878},{47,8,9669},{0,25,5878},{2,57,128},{2,57,128},{2,57,128},
{2,30,129},{0,36,2180},{0,28,490},{0,28,490},{0,16,1018},{0,16,2691},{0,14,1441},{4,53,0},{4,53,0},{4,53,0},{4,29,1},{15,7,2178},{0,28,490},{0,28,490},{0,16,1018},{31,3,2178},{0,16,1018},{44,0,2665},{3,56,1},{12,38,2},{0,36,306},{44,0,2665},{62,13,2665},{0,36,306},{0,29,2669},{62,13,2665},{0,29,2669},{2,0,128},{2,0,128},{2,0,128},{2,0,128},{0,12,0},
{0,12,0},{0,12,0},{0,6,0},{0,5,45},{0,5,45},{6,63,4541},{3,59,297},{5,42,865},{3,38,625},{2,63,9749},{0,45,4021},{0,36,914},{0,27,5581},{0,31,11611},{0,27,6877},{11,63,3377},{6,57,10},{8,42,257},{5,37,441},{34,0,9669},{0,45,4021},{0,36,914},{0,27,5581},{47,11,9669},{0,27,5581},{3,61,288},{3,61,288},{3,61,288},{3,32,288},{0,42,2180},{0,31,338},{0,31,338},
{0,18,865},{0,19,2875},{0,16,1427},{6,55,0},{6,55,0},{6,55,0},{6,31,1},{15,13,2178},{0,31,338},{0,31,338},{0,18,865},{31,6,2178},{0,18,865},{47,0,2665},{5,58,1},{14,40,2},{0,38,225},{47,0,2665},{62,16,2665},{0,38,225},{0,31,2669},{62,16,2665},{0,31,2669},{3,0,288},{3,0,288},{3,0,288},{3,0,288},{0,18,0},{0,18,0},{0,18,0},{0,9,0},{0,8,109},
{0,8,109},{9,63,5374},{4,62,570},{6,44,1269},{4,40,841},{3,63,10021},{0,47,3745},{0,39,593},{0,30,5294},{0,36,12029},{0,28,6810},{14,63,3558},{8,59,9},{10,44,254},{7,39,446},{37,1,9669},{0,47,3745},{0,39,593},{0,30,5294},{63,6,9669},{0,30,5294},{4,63,561},{4,63,561},{4,63,561},{4,35,546},{0,49,2178},{0,34,218},{0,34,218},{0,21,725},{0,22,3117},{0,19,1433},{8,58,1},
{8,58,1},{8,58,1},{8,33,2},{24,1,2178},{0,34,218},{0,34,218},{0,21,725},{49,0,2178},{0,21,725},{50,0,2665},{7,60,1},{16,42,5},{0,41,157},{50,0,2665},{62,19,2665},{0,41,157},{0,33,2669},{62,19,2665},{0,33,2669},{4,0,545},{4,0,545},{4,0,545},{4,0,545},{0,25,0},{0,25,0},{0,25,0},{0,12,1},{0,11,212},{0,11,212},{11,63,6350},{6,63,905},{7,47,1678},
{4,42,1102},{6,63,10453},{0,50,3485},{0,42,361},{0,32,5054},{0,36,12429},{0,31,6794},{15,63,3710},{10,61,9},{12,46,254},{9,41,446},{40,1,9669},{0,50,3485},{0,42,361},{0,32,5054},{63,9,9669},{0,32,5054},{6,63,901},{6,63,901},{6,63,901},{5,37,842},{0,55,2178},{0,38,125},{0,38,125},{0,24,629},{0,25,3365},{0,22,1489},{10,60,1},{10,60,1},{10,60,1},{10,35,2},{27,1,2178},
{0,38,125},{0,38,125},{0,24,629},{55,0,2178},{0,24,629},{53,0,2665},{9,62,1},{18,44,5},{0,43,117},{53,0,2665},{62,22,2665},{0,43,117},{0,35,2669},{62,22,2665},{0,35,2669},{5,0,841},{5,0,841},{5,0,841},{5,0,841},{0,31,0},{0,31,0},{0,31,0},{0,15,1},{0,12,337},{0,12,337},{12,63,7350},{7,63,1450},{9,48,2190},{6,44,1422},{6,63,11045},{0,53,3293},{0,45,193},
{0,35,4870},{0,39,12829},{0,32,6807},{17,63,3905},{12,63,9},{14,48,267},{11,43,446},{43,1,9669},{0,53,3293},{0,45,193},{0,35,4870},{63,12,9669},{0,35,4870},{7,63,1369},{7,63,1369},{7,63,1369},{6,40,1202},{0,61,2178},{0,42,53},{0,42,53},{0,25,520},{0,28,3645},{0,24,1573},{12,62,1},{12,62,1},{12,62,1},{12,37,2},{30,1,2178},{0,42,53},{0,42,53},{0,25,520},{61,0,2178},
{0,25,520},{56,0,2665},{12,63,8},{20,46,5},{0,45,72},{56,0,2665},{62,25,2665},{0,45,72},{0,37,2669},{62,25,2665},{0,37,2669},{6,0,1201},{6,0,1201},{6,0,1201},{6,0,1201},{0,36,1},{0,36,1},{0,36,1},{0,18,1},{0,14,468},{0,14,468},{14,63,8614},{9,63,2129},{9,51,2758},{6,47,1822},{9,63,11765},{0,57,3125},{0,47,81},{0,36,4629},{0,42,13261},{0,35,6855},{20,63,4081},
{15,63,49},{16,50,257},{13,45,446},{46,1,9669},{0,57,3125},{0,47,81},{0,36,4629},{63,15,9669},{0,36,4629},{9,63,1933},{9,63,1933},{9,63,1933},{7,42,1626},{1,63,2212},{0,45,13},{0,45,13},{0,27,421},{0,31,3957},{0,25,1673},{14,63,2},{14,63,2},{14,63,2},{14,39,2},{33,1,2178},{0,45,13},{0,45,13},{0,27,421},{63,2,2178},{0,27,421},{59,0,2665},{15,63,40},{22,48,2},
{0,47,45},{59,0,2665},{62,28,2665},{0,47,45},{0,39,2669},{62,28,2665},{0,39,2669},{7,0,1625},{7,0,1625},{7,0,1625},{7,0,1625},{0,42,1},{0,42,1},{0,42,1},{0,21,1},{0,16,637},{0,16,637},{15,63,10085},{11,63,3185},{11,53,3481},{8,49,2337},{11,63,12845},{0,62,2958},{0,50,14},{0,38,4381},{0,45,13802},{0,36,6942},{23,63,4318},{17,63,154},{18,52,254},{14,48,456},{49,1,9669},
{0,62,2958},{0,50,14},{0,38,4381},{46,27,9669},{0,38,4381},{10,63,2717},{10,63,2717},{10,63,2717},{8,45,2180},{3,63,2394},{0,49,0},{0,49,0},{0,30,317},{0,33,4314},{0,28,1811},{17,63,10},{17,63,10},{17,63,10},{16,41,1},{31,12,2178},{0,49,0},{0,49,0},{0,30,317},{62,6,2178},{0,30,317},{62,1,2665},{20,63,113},{24,50,4},{0,50,13},{62,1,2665},{63,31,2665},{0,50,13},
{0,42,2669},{63,31,2665},{0,42,2669},{8,0,2180},{8,0,2180},{8,0,2180},{8,0,2180},{0,49,0},{0,49,0},{0,49,0},{0,25,1},{0,19,850},{0,19,850},{17,63,11454},{12,63,4143},{12,55,4141},{9,51,2805},{12,63,13803},{0,63,2871},{1,52,18},{0,41,4182},{0,50,14186},{0,39,6911},{26,63,4550},{20,63,306},{20,54,254},{17,49,446},{52,1,9669},{0,63,2870},{1,52,9},{0,41,4181},{46,30,9669},
{0,41,4181},{12,63,3414},{12,63,3414},{12,63,3414},{9,47,2673},{3,63,2691},{1,52,14},{1,52,14},{0,32,245},{0,36,4587},{0,31,1906},{19,63,25},{19,63,25},{19,63,25},{18,43,1},{31,18,2178},{2,51,0},{2,51,0},{0,32,244},{62,9,2178},{0,32,244},{63,5,2665},{23,63,193},{26,52,4},{0,52,4},{63,5,2665},{63,34,2665},{0,52,4},{0,44,2669},{63,34,2665},{0,44,2669},{9,0,2669},
{9,0,2669},{9,0,2669},{9,0,2669},{0,55,1},{0,55,1},{0,55,1},{0,28,2},{0,22,1009},{0,22,1009},{20,63,11990},{15,63,4575},{14,57,4141},{11,53,2805},{15,63,14195},{3,63,3015},{3,54,18},{0,43,4122},{0,53,13674},{0,41,6249},{29,63,4814},{23,63,522},{22,56,254},{19,51,446},{55,1,9669},{3,63,3006},{3,54,9},{0,43,4041},{47,32,9669},{0,43,4041},{14,63,3561},{14,63,3561},{14,63,3561},
{11,49,2670},{6,63,2795},{3,54,14},{3,54,14},{2,34,245},{0,39,4227},{0,33,1470},{21,63,58},{21,63,58},{21,63,58},{20,45,1},{34,17,2178},{4,53,0},{4,53,0},{0,35,180},{62,12,2178},{0,35,180},{63,11,2665},{26,63,305},{28,54,4},{2,54,4},{63,11,2665},{63,37,2665},{2,54,4},{0,46,2669},{63,37,2665},{0,46,2669},{11,0,2669},{11,0,2669},{11,0,2669},{11,0,2669},{2,57,1},
{2,57,1},{2,57,1},{2,30,2},{0,25,801},{0,25,801},{22,63,12554},{17,63,5066},{16,59,4118},{13,55,2805},{17,63,14614},{6,63,3255},{5,56,18},{1,45,4078},{0,56,13194},{0,44,5633},{31,63,5090},{26,63,802},{24,58,254},{21,53,446},{58,1,9669},{7,63,3198},{5,56,9},{0,45,3846},{53,32,9669},{0,45,3846},{15,63,3710},{15,63,3710},{15,63,3710},{13,51,2670},{9,63,2931},{5,56,14},{5,56,14},
{4,36,245},{0,42,3899},{0,36,1110},{23,63,90},{23,63,90},{23,63,90},{22,47,1},{37,17,2178},{6,55,0},{6,55,0},{0,37,136},{62,15,2178},{0,37,136},{63,17,2665},{30,63,442},{30,56,4},{4,56,4},{63,17,2665},{63,40,2665},{4,56,4},{0,47,2677},{63,40,2665},{0,47,2677},{13,0,2669},{13,0,2669},{13,0,2669},{13,0,2669},{4,59,1},{4,59,1},{4,59,1},{4,31,5},{0,28,625},
{0,28,625},{23,63,13130},{20,63,5706},{18,61,4122},{15,57,2807},{20,63,15102},{9,63,3625},{7,59,15},{3,47,4086},{0,59,12686},{0,47,5027},{34,63,5386},{29,63,1169},{26,60,257},{23,56,446},{62,0,9669},{12,63,3469},{7,59,11},{0,47,3658},{62,31,9669},{0,47,3658},{17,63,3905},{17,63,3905},{17,63,3905},{16,53,2677},{12,63,3112},{7,58,9},{7,58,9},{6,38,246},{0,45,3576},{0,38,755},{26,63,136},
{26,63,136},{26,63,136},{24,49,4},{49,0,2178},{8,57,2},{8,57,2},{0,39,85},{62,18,2178},{0,39,85},{63,23,2665},{34,63,628},{32,59,2},{4,59,2},{63,23,2665},{63,43,2665},{4,59,2},{0,50,2669},{63,43,2665},{0,50,2669},{15,0,2677},{15,0,2677},{15,0,2677},{15,0,2677},{7,60,4},{7,60,4},{7,60,4},{7,33,5},{0,33,424},{0,33,424},{26,63,13650},{23,63,6378},{20,63,4122},
{17,59,2799},{23,63,15558},{12,63,4065},{9,61,15},{6,49,4074},{0,62,12278},{0,49,4534},{36,63,5698},{30,63,1556},{28,62,257},{25,58,446},{63,4,9669},{15,63,3749},{9,61,11},{0,49,3510},{62,34,9669},{0,49,3510},{20,63,4041},{20,63,4041},{20,63,4041},{17,56,2670},{15,63,3304},{9,60,9},{9,60,9},{8,40,246},{0,50,3317},{0,41,499},{29,63,200},{29,63,200},{29,63,200},{26,51,4},{52,0,2178},
{10,59,2},{10,59,2},{0,42,45},{62,21,2178},{0,42,45},{63,29,2665},{37,63,820},{34,61,2},{6,61,2},{63,29,2665},{63,46,2665},{6,61,2},{0,52,2669},{63,46,2665},{0,52,2669},{17,0,2669},{17,0,2669},{17,0,2669},{17,0,2669},{9,62,4},{9,62,4},{9,62,4},{9,35,5},{0,36,296},{0,36,296},{29,63,14234},{23,63,7050},{23,63,4242},{19,61,2799},{26,63,16046},{15,63,4601},{11,63,15},
{8,51,4074},{0,63,12051},{0,50,4110},{37,63,6002},{34,63,1989},{31,63,270},{27,60,446},{63,10,9669},{20,63,4081},{11,63,11},{0,52,3374},{62,37,9669},{0,52,3374},{23,63,4241},{23,63,4241},{23,63,4241},{19,58,2670},{17,63,3485},{11,62,9},{11,62,9},{10,42,246},{0,53,3069},{0,43,306},{31,63,269},{31,63,269},{31,63,269},{28,53,4},{55,0,2178},{12,61,2},{12,61,2},{0,44,18},{62,24,2178},
{0,44,18},{63,35,2665},{41,63,1037},{36,63,2},{8,63,2},{63,35,2665},{63,49,2665},{8,63,2},{0,54,2669},{63,49,2665},{0,54,2669},{19,0,2669},{19,0,2669},{19,0,2669},{19,0,2669},{11,63,5},{11,63,5},{11,63,5},{11,37,5},{0,40,193},{0,40,193},{31,63,13639},{26,63,7005},{25,63,4454},{21,62,2721},{28,63,15204},{17,63,4285},{13,63,24},{9,52,3457},{0,63,11020},{0,52,3109},{40,63,5381},
{34,63,1890},{33,63,346},{29,61,296},{62,16,8712},{23,63,3636},{14,63,2},{0,53,2676},{62,39,8712},{0,53,2676},{25,63,4454},{25,63,4454},{25,63,4454},{21,60,2670},{19,63,3707},{13,63,24},{13,63,24},{12,44,246},{0,56,2853},{0,45,153},{33,63,346},{33,63,346},{33,63,346},{30,55,4},{58,0,2178},{14,63,2},{14,63,2},{0,46,10},{62,27,2178},{0,46,10},{63,39,2178},{44,63,872},{39,63,1},
{14,63,1},{63,39,2178},{47,59,2178},{14,63,1},{0,55,2180},{47,59,2178},{0,55,2180},{21,0,2669},{21,0,2669},{21,0,2669},{21,0,2669},{13,63,20},{13,63,20},{13,63,20},{13,39,5},{0,45,104},{0,45,104},{33,63,12766},{29,63,6930},{26,63,4694},{24,63,2685},{29,63,14014},{20,63,3898},{16,63,101},{13,53,2722},{3,63,10057},{0,54,2129},{43,63,4689},{37,63,1718},{36,63,452},{32,62,164},{63,19,7578},
{26,63,3078},{18,63,17},{0,55,1905},{63,41,7578},{0,55,1905},{26,63,4694},{26,63,4694},{26,63,4694},{24,61,2670},{22,63,3960},{16,63,101},{16,63,101},{14,47,242},{0,59,2650},{0,48,49},{36,63,452},{36,63,452},{36,63,452},{33,57,4},{53,16,2178},{18,63,17},{18,63,17},{0,49,1},{63,30,2178},{0,49,1},{63,42,1625},{46,63,650},{42,63,0},{21,63,0},{63,42,1625},{62,53,1625},{21,63,0},
{0,56,1625},{62,53,1625},{0,56,1625},{24,0,2669},{24,0,2669},{24,0,2669},{24,0,2669},{15,63,50},{15,63,50},{15,63,50},{15,42,2},{0,48,40},{0,48,40},{34,63,11970},{31,63,6969},{29,63,4878},{26,63,2670},{31,63,13161},{21,63,3638},{18,63,229},{15,55,2146},{6,63,9313},{0,56,1410},{43,63,4097},{40,63,1614},{37,63,541},{34,62,69},{58,32,6661},{29,63,2654},{21,63,65},{0,56,1346},{52,48,6661},
{0,56,1346},{29,63,4878},{29,63,4878},{29,63,4878},{26,63,2670},{23,63,4206},{18,63,229},{18,63,229},{15,49,242},{0,62,2506},{0,51,9},{37,63,541},{37,63,541},{37,63,541},{35,59,4},{56,16,2178},{21,63,65},{21,63,65},{2,51,1},{62,33,2178},{2,51,1},{63,45,1201},{48,63,481},{45,63,0},{27,63,0},{63,45,1201},{63,54,1201},{27,63,0},{0,57,1201},{63,54,1201},{0,57,1201},{26,0,2669},
{26,0,2669},{26,0,2669},{26,0,2669},{17,63,85},{17,63,85},{17,63,85},{16,44,4},{0,51,8},{0,51,8},{37,63,11370},{33,63,6958},{31,63,5145},{28,63,2718},{34,63,12263},{23,63,3410},{21,63,405},{17,56,1665},{9,63,8665},{0,57,905},{46,63,3585},{43,63,1574},{40,63,637},{36,63,20},{63,27,5829},{32,63,2294},{26,63,130},{0,57,901},{63,45,5829},{0,57,901},{31,63,5145},{31,63,5145},{31,63,5145},
{28,63,2718},{26,63,4430},{21,63,405},{21,63,405},{18,50,246},{0,63,2520},{2,53,9},{40,63,637},{40,63,637},{40,63,637},{37,61,4},{59,16,2178},{26,63,130},{26,63,130},{4,53,1},{62,36,2178},{4,53,1},{63,48,841},{51,63,337},{48,63,1},{33,63,0},{63,48,841},{62,56,841},{33,63,0},{0,58,841},{62,56,841},{0,58,841},{28,0,2669},{28,0,2669},{28,0,2669},{28,0,2669},{20,63,117},
{20,63,117},{20,63,117},{18,46,4},{1,54,1},{1,54,1},{37,63,10794},{34,63,6895},{34,63,5374},{30,63,2813},{36,63,11574},{26,63,3274},{23,63,622},{19,57,1222},{13,63,8106},{1,59,562},{48,63,3170},{43,63,1494},{43,63,765},{39,63,4},{63,31,5082},{34,63,1998},{29,63,218},{0,59,554},{63,47,5082},{0,59,554},{34,63,5374},{34,63,5374},{34,63,5374},{30,63,2813},{29,63,4686},{23,63,622},{23,63,622},
{20,52,246},{4,63,2714},{4,55,9},{43,63,765},{43,63,765},{43,63,765},{39,63,4},{62,16,2178},{29,63,218},{29,63,218},{6,55,1},{62,39,2178},{6,55,1},{63,51,545},{54,63,225},{51,63,1},{39,63,0},{63,51,545},{63,57,545},{39,63,0},{0,59,545},{63,57,545},{0,59,545},{30,0,2669},{30,0,2669},{30,0,2669},{30,0,2669},{22,63,180},{22,63,180},{22,63,180},{21,47,5},{3,56,1},
{3,56,1},{40,63,10197},{37,63,6930},{36,63,5678},{32,63,2993},{37,63,10780},{29,63,3229},{26,63,945},{21,59,853},{17,63,7593},{4,60,293},{48,63,2756},{46,63,1454},{45,63,914},{41,63,37},{63,35,4344},{38,63,1740},{32,63,360},{2,60,289},{63,49,4344},{2,60,289},{36,63,5678},{36,63,5678},{36,63,5678},{32,63,2993},{31,63,5067},{26,63,945},{26,63,945},{22,55,242},{9,63,2979},{6,57,10},{45,63,914},
{45,63,914},{45,63,914},{41,63,37},{63,21,2178},{32,63,360},{32,63,360},{8,57,1},{63,42,2178},{8,57,1},{63,54,288},{55,63,125},{54,63,0},{45,63,1},{63,54,288},{62,59,288},{45,63,1},{0,60,288},{62,59,288},{0,60,288},{32,0,2669},{32,0,2669},{32,0,2669},{32,0,2669},{25,63,250},{25,63,250},{25,63,250},{23,50,2},{5,59,2},{5,59,2},{43,63,9837},{40,63,7042},{37,63,5945},
{34,63,3198},{40,63,10204},{32,63,3344},{29,63,1289},{24,59,589},{20,63,7225},{7,61,130},{51,63,2436},{48,63,1460},{48,63,1060},{43,63,122},{63,39,3779},{41,63,1580},{37,63,505},{6,61,129},{47,59,3779},{6,61,129},{37,63,5945},{37,63,5945},{37,63,5945},{34,63,3198},{34,63,5304},{29,63,1289},{29,63,1289},{24,57,242},{13,63,3261},{8,59,10},{48,63,1060},{48,63,1060},{48,63,1060},{43,63,122},{63,27,2178},
{37,63,505},{37,63,505},{10,59,1},{63,45,2178},{10,59,1},{63,57,128},{58,63,53},{57,63,0},{51,63,0},{63,57,128},{63,60,128},{51,63,0},{0,61,128},{63,60,128},{0,61,128},{34,0,2669},{34,0,2669},{34,0,2669},{34,0,2669},{27,63,337},{27,63,337},{27,63,337},{25,52,2},{7,61,2},{7,61,2},{43,63,9437},{40,63,7154},{40,63,6193},{37,63,3454},{42,63,9783},{34,63,3398},{32,63,1740},
{26,61,397},{23,63,6953},{9,62,37},{53,63,2246},{51,63,1508},{50,63,1213},{46,63,250},{63,43,3299},{44,63,1484},{40,63,673},{10,62,33},{63,53,3299},{10,62,33},{40,63,6193},{40,63,6193},{40,63,6193},{37,63,3454},{37,63,5624},{32,63,1740},{32,63,1740},{26,59,242},{17,63,3589},{10,61,10},{50,63,1213},{50,63,1213},{50,63,1213},{46,63,250},{63,33,2178},{40,63,673},{40,63,673},{12,61,1},{47,56,2178},
{12,61,1},{63,60,32},{61,63,13},{60,63,0},{57,63,0},{63,60,32},{62,62,32},{57,63,0},{0,62,32},{62,62,32},{0,62,32},{36,0,2669},{36,0,2669},{36,0,2669},{36,0,2669},{29,63,405},{29,63,405},{29,63,405},{27,54,2},{9,63,2},{9,63,2},{46,63,9141},{43,63,7234},{43,63,6505},{40,63,3806},{43,63,9340},{37,63,3622},{34,63,2149},{27,62,277},{27,63,6772},{12,63,10},{54,63,2052},
{53,63,1601},{51,63,1348},{48,63,436},{63,47,2904},{47,63,1452},{44,63,872},{14,63,1},{63,55,2904},{14,63,1},{43,63,6505},{43,63,6505},{43,63,6505},{40,63,3806},{40,63,5976},{34,63,2149},{34,63,2149},{28,61,242},{21,63,3955},{12,63,10},{51,63,1348},{51,63,1348},{51,63,1348},{48,63,436},{63,39,2178},{44,63,872},{44,63,872},{14,63,1},{47,59,2178},{14,63,1},{63,63,0},{63,63,0},{63,63,0},
{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{38,0,2669},{38,0,2669},{38,0,2669},{38,0,2669},{32,63,520},{32,63,520},{32,63,520},{29,56,2},{12,63,10},{12,63,10},{46,63,8097},{46,63,6510},{43,63,5893},{41,63,3605},{46,63,8074},{37,63,3244},{37,63,2019},{30,62,129},{30,63,5794},{16,63,58},{56,63,1563},{54,63,1161},{54,63,1017},{51,63,337},{63,49,2166},
{49,63,1083},{46,63,650},{21,63,0},{63,56,2166},{21,63,0},{43,63,5893},{43,63,5893},{43,63,5893},{41,63,3605},{40,63,5238},{37,63,2019},{37,63,2019},{30,62,113},{24,63,3401},{16,63,58},{54,63,1017},{54,63,1017},{54,63,1017},{51,63,337},{63,42,1625},{46,63,650},{46,63,650},{21,63,0},{62,53,1625},{21,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},
{0,63,0},{63,63,0},{0,63,0},{40,0,2665},{40,0,2665},{40,0,2665},{40,0,2665},{34,63,610},{34,63,610},{34,63,610},{31,58,0},{16,63,58},{16,63,58},{48,63,7165},{46,63,5854},{46,63,5325},{43,63,3434},{46,63,7050},{40,63,2932},{37,63,1955},{32,62,57},{32,63,5021},{20,63,117},{57,63,1137},{54,63,889},{54,63,745},{52,63,250},{63,51,1601},{51,63,801},{48,63,481},{27,63,0},{63,57,1601},
{27,63,0},{46,63,5325},{46,63,5325},{46,63,5325},{43,63,3434},{43,63,4622},{37,63,1955},{37,63,1955},{32,62,41},{27,63,2977},{20,63,117},{54,63,745},{54,63,745},{54,63,745},{52,63,250},{63,45,1201},{48,63,481},{48,63,481},{27,63,0},{63,54,1201},{27,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{42,0,2665},
{42,0,2665},{42,0,2665},{42,0,2665},{37,63,730},{37,63,730},{37,63,730},{33,60,1},{20,63,117},{20,63,117},{50,63,6415},{48,63,5277},{48,63,4877},{45,63,3330},{48,63,6117},{43,63,2716},{40,63,1843},{35,63,10},{35,63,4341},{23,63,205},{57,63,801},{57,63,609},{56,63,530},{54,63,169},{63,53,1121},{52,63,571},{51,63,337},{33,63,0},{63,58,1121},{33,63,0},{48,63,4877},{48,63,4877},{48,63,4877},
{45,63,3330},{45,63,4146},{40,63,1843},{40,63,1843},{35,63,10},{30,63,2617},{23,63,205},{56,63,530},{56,63,530},{56,63,530},{54,63,169},{63,48,841},{51,63,337},{51,63,337},{33,63,0},{62,56,841},{33,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{44,0,2665},{44,0,2665},{44,0,2665},{44,0,2665},{39,63,865},
{39,63,865},{39,63,865},{35,62,1},{23,63,205},{23,63,205},{51,63,5637},{50,63,4826},{48,63,4445},{47,63,3189},{48,63,5365},{43,63,2524},{43,63,1795},{37,63,5},{37,63,3750},{27,63,320},{59,63,531},{57,63,401},{57,63,337},{55,63,122},{63,55,726},{55,63,363},{54,63,225},{39,63,0},{63,59,726},{39,63,0},{48,63,4445},{48,63,4445},{48,63,4445},{47,63,3189},{46,63,3654},{43,63,1795},{43,63,1795},
{37,63,5},{32,63,2329},{27,63,320},{57,63,337},{57,63,337},{57,63,337},{55,63,122},{63,51,545},{54,63,225},{54,63,225},{39,63,0},{63,57,545},{39,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{46,0,2665},{46,0,2665},{46,0,2665},{46,0,2665},{40,63,1010},{40,63,1010},{40,63,1010},{37,63,5},{27,63,320},
{27,63,320},{3,63,10504},{0,62,1552},{0,44,169},{0,38,3866},{0,57,18065},{0,39,12152},{0,35,6099},{0,24,13992},{0,26,19423},{0,22,14922},{6,63,10216},{0,62,1552},{0,44,169},{0,38,3866},{20,17,18065},{0,39,12152},{0,35,6099},{0,24,13992},{57,0,18065},{0,24,13992},{0,35,0},{0,35,0},{0,35,0},{0,17,1},{0,17,1105},{0,14,424},{0,14,424},{0,8,666},{0,8,1217},{0,7,766},{0,35,0},
{0,35,0},{0,35,0},{0,17,1},{9,0,1105},{0,14,424},{0,14,424},{0,8,666},{17,0,1105},{0,8,666},{34,17,9248},{0,62,1552},{0,44,169},{0,38,3866},{34,17,9248},{62,12,9248},{0,38,3866},{0,28,9256},{62,12,9248},{0,28,9256},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{5,63,11298},{0,63,1341},{0,47,65},
{0,38,3578},{0,61,19334},{0,42,12584},{0,36,6093},{0,24,14696},{0,28,20850},{0,24,15720},{6,63,10792},{0,63,1341},{0,47,65},{0,38,3578},{30,1,19334},{0,42,12584},{0,36,6093},{0,24,14696},{61,0,19334},{0,24,14696},{0,41,0},{0,41,0},{0,41,0},{0,20,1},{0,20,1513},{0,16,585},{0,16,585},{0,10,901},{0,9,1681},{0,8,1070},{0,41,0},{0,41,0},{0,41,0},{0,20,1},{10,1,1513},
{0,16,585},{0,16,585},{0,10,901},{20,0,1513},{0,10,901},{37,17,9248},{0,63,1341},{0,47,65},{0,38,3578},{37,17,9248},{62,15,9248},{0,38,3578},{0,30,9256},{62,15,9248},{0,30,9256},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{6,63,12200},{0,63,1325},{0,49,9},{0,41,3298},{0,63,20705},{0,42,13032},{0,38,6147},
{0,27,15400},{0,31,22426},{0,25,16626},{9,63,11512},{0,63,1325},{0,49,9},{0,41,3298},{31,3,20689},{0,42,13032},{0,38,6147},{0,27,15400},{63,1,20689},{0,27,15400},{0,47,0},{0,47,0},{0,47,0},{0,23,1},{0,23,1985},{0,19,769},{0,19,769},{0,11,1202},{0,11,2193},{0,10,1374},{0,47,0},{0,47,0},{0,47,0},{0,23,1},{12,0,1985},{0,19,769},{0,19,769},{0,11,1202},{23,0,1985},
{0,11,1202},{48,1,9248},{0,63,1325},{0,49,9},{0,41,3298},{48,1,9248},{63,17,9248},{0,41,3298},{0,32,9250},{63,17,9248},{0,32,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{6,63,13288},{0,63,1565},{0,51,4},{0,44,3050},{2,63,22214},{0,45,13496},{0,41,6227},{0,28,16225},{0,31,24090},{0,27,17528},{9,63,12344},
{0,63,1565},{0,51,4},{0,44,3050},{34,1,22129},{0,45,13496},{0,41,6227},{0,28,16225},{63,3,22129},{0,28,16225},{0,53,0},{0,53,0},{0,53,0},{0,26,1},{0,26,2521},{0,22,985},{0,22,985},{0,13,1517},{0,11,2801},{0,11,1766},{0,53,0},{0,53,0},{0,53,0},{0,26,1},{13,1,2521},{0,22,985},{0,22,985},{0,13,1517},{26,0,2521},{0,13,1517},{51,1,9248},{1,63,1552},{1,51,0},
{0,44,3050},{51,1,9248},{63,20,9248},{0,44,3050},{0,34,9250},{63,20,9248},{0,34,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{6,63,14818},{1,63,2106},{0,54,58},{0,47,2792},{3,63,24091},{0,50,14075},{0,42,6341},{0,30,17106},{0,33,26067},{0,28,18692},{12,63,13474},{3,63,2077},{1,54,50},{0,47,2792},{31,12,23851},
{0,50,14075},{0,42,6341},{0,30,17106},{62,6,23851},{0,30,17106},{0,59,1},{0,59,1},{0,59,1},{0,30,0},{0,30,3200},{0,25,1258},{0,25,1258},{0,13,1940},{0,14,3542},{0,13,2229},{0,59,1},{0,59,1},{0,59,1},{0,30,0},{15,0,3200},{0,25,1258},{0,25,1258},{0,13,1940},{30,0,3200},{0,13,1940},{55,0,9248},{6,63,1885},{3,53,2},{0,47,2792},{55,0,9248},{62,24,9248},{0,47,2792},
{0,36,9256},{62,24,9248},{0,36,9256},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{9,63,16258},{3,63,2813},{1,56,141},{0,49,2561},{4,63,25971},{0,50,14619},{0,45,6453},{0,32,17996},{0,36,27923},{0,30,19698},{12,63,14594},{3,63,2669},{2,56,122},{0,49,2561},{38,1,25472},{0,50,14619},{0,45,6453},{0,32,17996},{45,16,25472},
{0,32,17996},{0,63,9},{0,63,9},{0,63,9},{0,33,1},{0,33,3872},{0,25,1530},{0,25,1530},{0,16,2378},{0,14,4294},{0,14,2717},{0,63,9},{0,63,9},{0,63,9},{0,33,1},{16,1,3872},{0,25,1530},{0,25,1530},{0,16,2378},{33,0,3872},{0,16,2378},{58,0,9248},{9,63,2205},{5,55,2},{0,49,2561},{58,0,9248},{62,27,9248},{0,49,2561},{0,38,9256},{62,27,9248},{0,38,9256},{0,0,0},
{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{9,63,17750},{3,63,3617},{1,58,254},{0,50,2366},{6,63,27563},{0,53,14891},{0,47,6385},{0,32,18616},{0,39,29523},{0,32,20465},{15,63,15654},{6,63,3425},{2,59,206},{0,50,2366},{41,0,26744},{0,53,14891},{0,47,6385},{0,32,18616},{62,10,26744},{0,32,18616},{1,63,101},{1,63,101},{1,63,101},
{0,36,5},{0,36,4420},{0,28,1666},{0,28,1666},{0,16,2642},{0,16,4931},{0,16,3083},{2,63,72},{2,63,72},{2,63,72},{1,35,4},{15,7,4418},{0,28,1666},{0,28,1666},{0,16,2642},{31,3,4418},{0,16,2642},{61,0,9248},{13,63,2554},{7,57,2},{0,50,2362},{61,0,9248},{62,30,9248},{0,50,2362},{0,40,9256},{62,30,9248},{0,40,9256},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,2,0},
{0,2,0},{0,2,0},{0,1,0},{0,1,1},{0,1,1},{12,63,19046},{4,63,4598},{3,60,382},{0,52,2237},{6,63,28187},{0,59,14347},{0,50,5579},{0,35,18104},{0,39,29955},{0,33,20313},{17,63,16091},{7,63,4046},{4,61,206},{0,52,2237},{44,0,26744},{0,59,14347},{0,50,5579},{0,35,18104},{62,13,26744},{0,35,18104},{2,63,264},{2,63,264},{2,63,264},{1,38,52},{0,42,4420},{0,33,1381},{0,33,1381},
{0,18,2405},{0,19,5115},{0,18,2981},{4,63,117},{4,63,117},{4,63,117},{3,37,4},{15,13,4418},{0,33,1381},{0,33,1381},{0,18,2405},{31,6,4418},{0,18,2405},{63,2,9248},{17,63,2938},{9,59,2},{0,52,2137},{63,2,9248},{46,41,9248},{0,52,2137},{0,42,9256},{46,41,9248},{0,42,9256},{1,0,52},{1,0,52},{1,0,52},{1,0,52},{0,8,0},{0,8,0},{0,8,0},{0,4,0},{0,3,20},
{0,3,20},{12,63,20585},{6,63,5786},{4,63,625},{1,56,2246},{9,63,29012},{0,62,13736},{0,53,4760},{0,38,17595},{0,45,30452},{0,36,20086},{20,63,16620},{9,63,4794},{6,63,205},{1,56,2230},{39,16,26744},{0,62,13736},{0,53,4760},{0,38,17595},{62,16,26744},{0,38,17595},{4,63,589},{4,63,589},{4,63,589},{2,41,185},{0,49,4418},{0,36,1097},{0,36,1097},{0,21,2153},{0,22,5357},{0,21,2937},{6,63,169},
{6,63,169},{6,63,169},{5,39,2},{24,1,4418},{0,36,1097},{0,36,1097},{0,21,2153},{49,0,4418},{0,21,2153},{59,16,9248},{20,63,3380},{11,62,4},{0,55,1901},{59,16,9248},{62,36,9248},{0,55,1901},{0,45,9250},{62,36,9248},{0,45,9250},{2,0,185},{2,0,185},{2,0,185},{2,0,185},{0,14,1},{0,14,1},{0,14,1},{0,7,1},{0,5,72},{0,5,72},{15,63,21605},{9,63,6850},{5,63,978},
{2,57,2226},{11,63,29435},{0,63,12990},{0,55,3962},{0,41,16835},{0,45,30392},{0,38,19516},{22,63,16694},{12,63,5218},{9,63,225},{4,56,2130},{49,1,26259},{0,63,12990},{0,55,3962},{0,41,16835},{46,27,26259},{0,41,16835},{5,63,978},{5,63,978},{5,63,978},{3,43,370},{0,55,4418},{0,39,881},{0,39,881},{0,24,1945},{0,25,5605},{0,22,2889},{9,63,225},{9,63,225},{9,63,225},{7,41,2},{27,1,4418},
{0,39,881},{0,39,881},{0,24,1945},{55,0,4418},{0,24,1945},{63,14,8978},{23,63,3592},{14,63,1},{0,58,1625},{63,14,8978},{46,47,8978},{0,58,1625},{0,46,8986},{46,47,8978},{0,46,8986},{3,0,369},{3,0,369},{3,0,369},{3,0,369},{0,20,0},{0,20,0},{0,20,0},{0,10,1},{0,8,136},{0,8,136},{15,63,21141},{9,63,7026},{6,63,1481},{3,58,2034},{12,63,28216},{0,63,11406},{0,56,2868},
{0,41,14915},{0,48,28876},{0,39,17854},{23,63,15352},{15,63,4866},{11,63,306},{6,58,1746},{52,0,24371},{0,63,11406},{0,56,2868},{0,41,14915},{62,21,24371},{0,41,14915},{6,63,1481},{6,63,1481},{6,63,1481},{4,46,617},{0,61,4418},{0,45,689},{0,45,689},{0,27,1769},{0,28,5885},{0,24,2889},{11,63,306},{11,63,306},{11,63,306},{9,43,2},{30,1,4418},{0,45,689},{0,45,689},{0,27,1769},{61,0,4418},
{0,27,1769},{55,32,7938},{26,63,3176},{16,63,1},{0,58,1129},{55,32,7938},{46,48,7938},{0,58,1129},{0,47,7946},{46,48,7938},{0,47,7946},{4,0,617},{4,0,617},{4,0,617},{4,0,617},{0,26,0},{0,26,0},{0,26,0},{0,13,1},{0,11,232},{0,11,232},{17,63,20849},{9,63,7458},{8,63,2106},{4,59,1970},{12,63,27224},{0,63,10078},{0,58,1986},{0,42,13214},{0,50,27357},{0,41,16276},{26,63,14168},
{17,63,4556},{14,63,394},{9,58,1410},{47,14,22568},{0,63,10078},{0,58,1986},{0,42,13214},{62,23,22568},{0,42,13214},{8,63,2106},{8,63,2106},{8,63,2106},{5,48,930},{1,63,4452},{0,47,521},{0,47,521},{0,28,1600},{0,31,6197},{0,27,2921},{14,63,394},{14,63,394},{14,63,394},{11,45,2},{33,1,4418},{0,47,521},{0,47,521},{0,28,1600},{63,2,4418},{0,28,1600},{56,33,6962},{29,63,2792},{19,63,1},
{0,59,740},{56,33,6962},{49,48,6962},{0,59,740},{0,48,6964},{49,48,6962},{0,48,6964},{5,0,929},{5,0,929},{5,0,929},{5,0,929},{0,32,0},{0,32,0},{0,32,0},{0,16,0},{0,14,360},{0,14,360},{17,63,20651},{12,63,7922},{9,63,2921},{5,60,2052},{15,63,26270},{0,63,8890},{0,59,1154},{0,44,11309},{0,53,25875},{0,42,14658},{26,63,12890},{20,63,4254},{16,63,493},{10,59,1076},{56,0,20642},
{0,63,8890},{0,59,1154},{0,44,11309},{62,25,20642},{0,44,11309},{9,63,2921},{9,63,2921},{9,63,2921},{6,51,1360},{3,63,4634},{0,50,346},{0,50,346},{0,31,1402},{0,33,6554},{0,28,2987},{16,63,493},{16,63,493},{16,63,493},{13,48,1},{31,12,4418},{0,50,346},{0,50,346},{0,31,1402},{62,6,4418},{0,31,1402},{63,23,5941},{30,63,2386},{23,63,1},{0,61,388},{63,23,5941},{63,43,5941},{0,61,388},
{0,49,5945},{63,43,5941},{0,49,5945},{6,0,1360},{6,0,1360},{6,0,1360},{6,0,1360},{0,39,0},{0,39,0},{0,39,0},{0,19,1},{0,16,522},{0,16,522},{20,63,20683},{12,63,8578},{11,63,3792},{7,60,2241},{15,63,25566},{1,63,8090},{0,59,642},{0,45,9834},{0,53,24595},{0,44,13350},{29,63,11794},{20,63,4030},{17,63,610},{13,59,804},{58,0,19021},{3,63,8050},{0,59,642},{0,45,9834},{62,27,19021},
{0,45,9834},{11,63,3792},{11,63,3792},{11,63,3792},{8,53,1808},{3,63,4954},{0,55,232},{0,55,232},{0,33,1241},{0,36,6922},{0,31,3051},{17,63,610},{17,63,610},{17,63,610},{15,50,1},{31,18,4418},{0,55,232},{0,55,232},{0,33,1241},{62,9,4418},{0,33,1241},{63,26,5101},{34,63,2050},{26,63,1},{0,61,164},{63,26,5101},{62,45,5101},{0,61,164},{0,50,5105},{62,45,5101},{0,50,5105},{7,0,1808},
{7,0,1808},{7,0,1808},{7,0,1808},{0,45,0},{0,45,0},{0,45,0},{0,22,1},{0,19,706},{0,19,706},{20,63,20715},{15,63,9258},{12,63,4729},{8,61,2553},{17,63,25067},{3,63,7474},{0,61,264},{0,47,8373},{0,56,23451},{0,45,12138},{31,63,10854},{23,63,3766},{20,63,698},{16,60,594},{60,0,17485},{6,63,7274},{0,61,264},{0,47,8373},{56,32,17485},{0,47,8373},{12,63,4729},{12,63,4729},{12,63,4729},
{8,56,2320},{6,63,5386},{0,59,130},{0,59,130},{0,35,1076},{0,39,7322},{0,33,3161},{20,63,698},{20,63,698},{20,63,698},{17,51,2},{34,17,4418},{0,59,130},{0,59,130},{0,35,1076},{62,12,4418},{0,35,1076},{63,29,4325},{35,63,1733},{29,63,1},{0,62,41},{63,29,4325},{63,46,4325},{0,62,41},{0,51,4329},{63,46,4325},{0,51,4329},{8,0,2320},{8,0,2320},{8,0,2320},{8,0,2320},{0,50,1},
{0,50,1},{0,50,1},{0,25,1},{0,19,914},{0,19,914},{20,63,21003},{15,63,10106},{14,63,5840},{9,62,2993},{17,63,24683},{3,63,7010},{0,62,74},{0,49,7113},{0,56,22427},{0,47,11094},{33,63,9941},{26,63,3566},{23,63,818},{17,61,402},{62,0,16034},{9,63,6562},{0,62,74},{0,49,7113},{62,31,16034},{0,49,7113},{14,63,5840},{14,63,5840},{14,63,5840},{10,58,2896},{6,63,5962},{0,62,58},{0,62,58},
{0,38,932},{0,42,7754},{0,35,3317},{23,63,818},{23,63,818},{23,63,818},{19,53,2},{37,17,4418},{0,62,58},{0,62,58},{0,38,932},{62,15,4418},{0,38,932},{63,32,3613},{38,63,1445},{32,63,1},{0,63,0},{63,32,3613},{61,48,3613},{0,63,0},{0,52,3617},{61,48,3613},{0,52,3617},{9,0,2896},{9,0,2896},{9,0,2896},{9,0,2896},{0,56,1},{0,56,1},{0,56,1},{0,28,1},{0,22,1130},
{0,22,1130},{23,63,21401},{17,63,11165},{15,63,7141},{10,63,3641},{20,63,24533},{6,63,6762},{0,63,81},{0,49,5745},{0,59,21333},{0,47,9996},{34,63,8897},{29,63,3396},{26,63,976},{20,61,224},{56,16,14504},{12,63,5834},{1,63,68},{0,49,5745},{62,33,14504},{0,49,5745},{15,63,7141},{15,63,7141},{15,63,7141},{11,61,3617},{9,63,6772},{0,63,81},{0,63,81},{0,41,794},{0,45,8260},{0,38,3515},{26,63,976},
{26,63,976},{26,63,976},{21,56,0},{49,0,4418},{1,63,68},{1,63,68},{0,41,794},{62,18,4418},{0,41,794},{63,35,2888},{41,63,1156},{35,63,0},{7,63,0},{63,35,2888},{63,49,2888},{7,63,0},{0,53,2896},{63,49,2888},{0,53,2896},{11,0,3617},{11,0,3617},{11,0,3617},{11,0,3617},{0,63,0},{0,63,0},{0,63,0},{0,32,1},{0,25,1413},{0,25,1413},{23,63,21913},{17,63,12317},{17,63,8473},
{12,63,4330},{20,63,24437},{6,63,6650},{2,63,298},{0,52,4721},{0,62,20509},{0,49,9157},{37,63,8153},{29,63,3268},{28,63,1129},{23,61,128},{63,6,13235},{15,63,5258},{6,63,145},{0,52,4721},{62,35,13235},{0,52,4721},{17,63,8473},{17,63,8473},{17,63,8473},{12,63,4330},{9,63,7636},{2,63,298},{2,63,298},{0,42,689},{0,45,8740},{0,39,3689},{28,63,1129},{28,63,1129},{28,63,1129},{23,58,0},{52,0,4418},
{6,63,145},{6,63,145},{0,42,689},{62,21,4418},{0,42,689},{63,38,2312},{43,63,925},{38,63,0},{13,63,0},{63,38,2312},{62,51,2312},{13,63,0},{0,54,2320},{62,51,2312},{0,54,2320},{12,0,4329},{12,0,4329},{12,0,4329},{12,0,4329},{1,63,52},{1,63,52},{1,63,52},{0,34,1},{0,28,1693},{0,28,1693},{26,63,22641},{20,63,13461},{17,63,9881},{13,63,5169},{20,63,24597},{6,63,6794},{3,63,649},
{0,52,3713},{0,62,19773},{0,50,8413},{37,63,7401},{32,63,3181},{29,63,1280},{24,62,48},{63,11,12051},{18,63,4746},{9,63,233},{0,52,3713},{47,45,12051},{0,52,3713},{17,63,9881},{17,63,9881},{17,63,9881},{13,63,5169},{12,63,8644},{3,63,649},{3,63,649},{0,45,569},{0,50,9245},{0,42,3905},{29,63,1280},{29,63,1280},{29,63,1280},{25,60,0},{55,0,4418},{9,63,233},{9,63,233},{0,45,569},{62,24,4418},
{0,45,569},{63,41,1800},{46,63,725},{41,63,0},{19,63,0},{63,41,1800},{63,52,1800},{19,63,0},{0,55,1808},{63,52,1800},{0,55,1808},{13,0,5105},{13,0,5105},{13,0,5105},{13,0,5105},{2,63,185},{2,63,185},{2,63,185},{0,37,1},{0,31,2005},{0,31,2005},{26,63,23345},{20,63,14805},{20,63,11441},{14,63,6170},{23,63,24893},{9,63,7066},{4,63,1236},{0,53,2900},{0,63,19260},{0,52,7861},{40,63,6753},
{34,63,3038},{32,63,1465},{27,63,9},{62,16,10952},{21,63,4298},{12,63,353},{0,53,2900},{62,39,10952},{0,53,2900},{20,63,11441},{20,63,11441},{20,63,11441},{14,63,6170},{12,63,9764},{4,63,1236},{4,63,1236},{0,47,458},{0,50,9789},{0,45,4185},{32,63,1465},{32,63,1465},{32,63,1465},{27,62,0},{58,0,4418},{12,63,353},{12,63,353},{0,47,458},{62,27,4418},{0,47,458},{63,44,1352},{48,63,544},{44,63,0},
{25,63,0},{63,44,1352},{62,54,1352},{25,63,0},{0,56,1360},{62,54,1352},{0,56,1360},{14,0,5945},{14,0,5945},{14,0,5945},{14,0,5945},{3,63,400},{3,63,400},{3,63,400},{0,40,1},{0,33,2336},{0,33,2336},{26,63,24443},{23,63,16415},{20,63,13259},{15,63,7448},{23,63,25379},{9,63,7606},{6,63,2021},{0,55,2045},{0,63,18918},{0,53,7275},{40,63,6141},{37,63,2978},{34,63,1625},{29,63,10},{63,19,9818},
{24,63,3870},{17,63,530},{0,55,2045},{63,41,9818},{0,55,2045},{20,63,13259},{20,63,13259},{20,63,13259},{15,63,7448},{15,63,11218},{6,63,2021},{6,63,2021},{0,49,365},{0,53,10427},{0,45,4509},{34,63,1625},{34,63,1625},{34,63,1625},{29,63,10},{53,16,4418},{17,63,530},{17,63,530},{0,49,365},{63,30,4418},{0,49,365},{63,47,925},{51,63,377},{47,63,1},{31,63,1},{63,47,925},{63,55,925},{31,63,1},
{0,58,929},{63,55,925},{0,58,929},{15,0,6964},{15,0,6964},{15,0,6964},{15,0,6964},{3,63,769},{3,63,769},{3,63,769},{0,44,0},{0,36,2745},{0,36,2745},{29,63,25483},{23,63,17983},{22,63,15066},{17,63,8739},{23,63,26083},{12,63,8302},{6,63,2965},{0,56,1458},{0,63,18886},{0,53,6955},{43,63,5581},{40,63,2986},{37,63,1801},{32,63,65},{58,32,8901},{27,63,3558},{20,63,698},{0,56,1458},{52,48,8901},
{0,56,1458},{22,63,15066},{22,63,15066},{22,63,15066},{17,63,8739},{15,63,12626},{6,63,2965},{6,63,2965},{0,52,277},{0,56,11011},{0,49,4833},{37,63,1801},{37,63,1801},{37,63,1801},{32,63,65},{56,16,4418},{20,63,698},{20,63,698},{0,52,277},{62,33,4418},{0,52,277},{63,50,613},{52,63,250},{50,63,0},{37,63,0},{63,50,613},{62,57,613},{37,63,0},{0,59,617},{62,57,613},{0,59,617},{16,0,7946},
{16,0,7946},{16,0,7946},{16,0,7946},{5,63,1184},{5,63,1184},{5,63,1184},{0,47,0},{0,36,3145},{0,36,3145},{29,63,26667},{26,63,19695},{23,63,16891},{18,63,10206},{26,63,26795},{12,63,9118},{9,63,4037},{0,58,933},{0,63,19110},{0,56,6699},{46,63,5181},{40,63,2970},{40,63,2009},{34,63,160},{63,27,8069},{30,63,3310},{23,63,898},{0,58,933},{63,45,8069},{0,58,933},{23,63,16891},{23,63,16891},{23,63,16891},
{18,63,10206},{17,63,14179},{9,63,4037},{9,63,4037},{0,55,221},{0,59,11627},{0,50,5115},{40,63,2009},{40,63,2009},{40,63,2009},{34,63,160},{59,16,4418},{23,63,898},{23,63,898},{0,55,221},{62,36,4418},{0,55,221},{63,53,365},{55,63,146},{53,63,0},{43,63,0},{63,53,365},{63,58,365},{43,63,0},{0,60,369},{63,58,365},{0,60,369},{16,0,8986},{16,0,8986},{16,0,8986},{16,0,8986},{6,63,1665},
{6,63,1665},{6,63,1665},{0,50,1},{0,39,3545},{0,39,3545},{31,63,26643},{26,63,20231},{26,63,17731},{20,63,10867},{26,63,26531},{15,63,9546},{12,63,4889},{0,59,590},{0,63,18606},{0,56,5707},{46,63,4781},{43,63,2978},{42,63,2228},{37,63,320},{63,31,7322},{32,63,3134},{27,63,1125},{0,59,554},{63,47,7322},{0,59,554},{26,63,17731},{26,63,17731},{26,63,17731},{20,63,10867},{20,63,15043},{12,63,4889},{12,63,4889},
{1,56,185},{0,62,11315},{0,53,4667},{42,63,2228},{42,63,2228},{42,63,2228},{37,63,320},{62,16,4418},{27,63,1125},{27,63,1125},{0,56,162},{62,39,4418},{0,56,162},{63,56,181},{57,63,73},{56,63,0},{49,63,0},{63,56,181},{62,60,181},{49,63,0},{0,61,185},{62,60,181},{0,61,185},{18,0,9250},{18,0,9250},{18,0,9250},{18,0,9250},{8,63,1972},{8,63,1972},{8,63,1972},{1,52,4},{0,42,3341},
{0,42,3341},{34,63,26006},{29,63,20400},{28,63,18273},{23,63,11384},{29,63,25736},{17,63,9864},{12,63,5726},{2,60,366},{0,63,18111},{0,59,4452},{48,63,4436},{46,63,3050},{43,63,2465},{40,63,562},{63,35,6584},{37,63,3006},{30,63,1421},{0,61,237},{63,49,6584},{0,61,237},{28,63,18273},{28,63,18273},{28,63,18273},{23,63,11384},{23,63,15704},{12,63,5726},{12,63,5726},{3,59,189},{0,63,10886},{0,56,3924},{43,63,2465},
{43,63,2465},{43,63,2465},{40,63,562},{63,21,4418},{30,63,1421},{30,63,1421},{0,59,100},{63,42,4418},{0,59,100},{63,59,52},{60,63,20},{59,63,1},{56,63,0},{63,59,52},{63,61,52},{56,63,0},{0,62,52},{63,61,52},{0,62,52},{20,0,9256},{20,0,9256},{20,0,9256},{20,0,9256},{9,63,2205},{9,63,2205},{9,63,2205},{4,54,1},{0,45,2925},{0,45,2925},{34,63,25526},{31,63,20721},{29,63,18700},
{24,63,11921},{31,63,25279},{20,63,10264},{15,63,6494},{4,62,238},{4,63,17924},{0,59,3588},{51,63,4228},{48,63,3140},{46,63,2665},{41,63,833},{63,39,6019},{40,63,2958},{34,63,1693},{0,62,84},{47,59,6019},{0,62,84},{29,63,18700},{29,63,18700},{29,63,18700},{24,63,11921},{23,63,16280},{15,63,6494},{15,63,6494},{5,61,189},{0,63,10854},{0,59,3332},{46,63,2665},{46,63,2665},{46,63,2665},{41,63,833},{63,27,4418},
{34,63,1693},{34,63,1693},{0,61,61},{63,45,4418},{0,61,61},{63,62,4},{63,63,4},{62,63,1},{62,63,0},{63,62,4},{62,63,4},{62,63,0},{0,63,4},{62,63,4},{0,63,4},{22,0,9256},{22,0,9256},{22,0,9256},{22,0,9256},{12,63,2389},{12,63,2389},{12,63,2389},{6,56,1},{0,50,2512},{0,50,2512},{37,63,24250},{34,63,19895},{31,63,18169},{26,63,11820},{34,63,23717},{20,63,10012},{17,63,6584},
{7,62,122},{6,63,16879},{0,62,2736},{51,63,3648},{48,63,2784},{48,63,2384},{43,63,778},{63,42,5163},{41,63,2584},{37,63,1549},{0,63,9},{46,61,5163},{0,63,9},{31,63,18169},{31,63,18169},{31,63,18169},{26,63,11820},{26,63,15620},{17,63,6584},{17,63,6584},{7,62,106},{0,63,10150},{0,59,2624},{48,63,2384},{48,63,2384},{48,63,2384},{43,63,778},{63,31,3872},{37,63,1549},{37,63,1549},{0,63,9},{63,47,3872},
{0,63,9},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{24,0,9256},{24,0,9256},{24,0,9256},{24,0,9256},{15,63,2605},{15,63,2605},{15,63,2605},{8,58,1},{0,53,2176},{0,53,2176},{37,63,22746},{34,63,18775},{34,63,17254},{29,63,11564},{34,63,21973},{23,63,9532},{20,63,6424},{9,62,57},{9,63,15607},{0,62,2032},{53,63,3058},
{51,63,2304},{48,63,2000},{46,63,650},{63,43,4267},{43,63,2134},{40,63,1285},{4,63,0},{63,53,4267},{4,63,0},{34,63,17254},{34,63,17254},{34,63,17254},{29,63,11564},{29,63,14692},{20,63,6424},{20,63,6424},{9,62,41},{0,63,9366},{0,62,2016},{48,63,2000},{48,63,2000},{48,63,2000},{46,63,650},{55,49,3200},{40,63,1285},{40,63,1285},{4,63,0},{63,48,3200},{4,63,0},{63,63,0},{63,63,0},{63,63,0},
{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{26,0,9256},{26,0,9256},{26,0,9256},{26,0,9256},{17,63,2836},{17,63,2836},{17,63,2836},{10,60,1},{0,59,1856},{0,59,1856},{40,63,21168},{37,63,17685},{34,63,16300},{30,63,11323},{37,63,20205},{26,63,9090},{23,63,6310},{12,63,4},{12,63,14287},{0,62,1546},{54,63,2377},{51,63,1809},{51,63,1553},{47,63,520},{61,49,3361},
{46,63,1683},{43,63,1018},{11,63,0},{63,54,3361},{11,63,0},{34,63,16300},{34,63,16300},{34,63,16300},{30,63,11323},{29,63,13666},{23,63,6310},{23,63,6310},{12,63,4},{3,63,8686},{0,62,1530},{51,63,1553},{51,63,1553},{51,63,1553},{47,63,520},{63,37,2521},{43,63,1018},{43,63,1018},{11,63,0},{62,50,2521},{11,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},
{0,63,0},{63,63,0},{0,63,0},{29,0,9250},{29,0,9250},{29,0,9250},{29,0,9250},{20,63,3114},{20,63,3114},{20,63,3114},{12,62,1},{0,62,1514},{0,62,1514},{40,63,19824},{37,63,16725},{37,63,15500},{32,63,11084},{37,63,18685},{26,63,8770},{26,63,6270},{14,63,16},{15,63,13215},{0,63,1325},{54,63,1881},{54,63,1449},{51,63,1249},{48,63,409},{63,47,2649},{47,63,1329},{44,63,797},{16,63,1},{63,55,2649},
{16,63,1},{37,63,15500},{37,63,15500},{37,63,15500},{32,63,11084},{31,63,12906},{26,63,6270},{26,63,6270},{14,63,16},{6,63,8150},{0,63,1325},{51,63,1249},{51,63,1249},{51,63,1249},{48,63,409},{63,40,1985},{44,63,797},{44,63,797},{16,63,1},{63,51,1985},{16,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{31,0,9250},
{31,0,9250},{31,0,9250},{31,0,9250},{22,63,3393},{22,63,3393},{22,63,3393},{14,63,16},{0,63,1325},{0,63,1325},{43,63,18608},{40,63,15853},{37,63,14796},{34,63,10841},{40,63,17341},{29,63,8410},{26,63,6206},{17,63,74},{17,63,12226},{0,63,1341},{56,63,1451},{54,63,1081},{54,63,937},{51,63,305},{63,50,2017},{49,63,1011},{47,63,605},{22,63,1},{62,57,2017},{22,63,1},{37,63,14796},{37,63,14796},{37,63,14796},
{34,63,10841},{34,63,12089},{26,63,6206},{26,63,6206},{17,63,74},{9,63,7678},{0,63,1341},{54,63,937},{54,63,937},{54,63,937},{51,63,305},{63,43,1513},{47,63,605},{47,63,605},{22,63,1},{62,53,1513},{22,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{32,0,9256},{32,0,9256},{32,0,9256},{32,0,9256},{23,63,3650},
{23,63,3650},{23,63,3650},{17,63,74},{0,63,1341},{0,63,1341},{43,63,17392},{40,63,15021},{40,63,14060},{37,63,10673},{40,63,16013},{32,63,8261},{29,63,6166},{19,63,194},{20,63,11338},{1,63,1594},{57,63,1041},{56,63,822},{54,63,697},{52,63,234},{63,51,1473},{51,63,737},{49,63,442},{28,63,1},{63,57,1473},{28,63,1},{40,63,14060},{40,63,14060},{40,63,14060},{37,63,10673},{34,63,11401},{29,63,6166},{29,63,6166},
{19,63,194},{12,63,7270},{1,63,1594},{54,63,697},{54,63,697},{54,63,697},{52,63,234},{63,46,1105},{49,63,442},{49,63,442},{28,63,1},{63,54,1105},{28,63,1},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{63,63,0},{0,63,0},{63,63,0},{0,63,0},{34,0,9256},{34,0,9256},{34,0,9256},{34,0,9256},{26,63,3898},{26,63,3898},{26,63,3898},{19,63,194},{1,63,1594},
{1,63,1594},
/**** ended inlining basisu_transcoder_tables_dxt1_6.inc ****/
	};

	static const etc1_to_dxt1_56_solution g_etc1_to_dxt_5[32 * 8 * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS * NUM_ETC1_TO_DXT1_SELECTOR_RANGES] = {
/**** start inlining basisu_transcoder_tables_dxt1_5.inc ****/
// Copyright (C) 2017-2019 Binomial LLC. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
{0,2,18},{0,1,9},{0,1,0},{0,1,9},{0,1,40},{0,1,22},{0,1,13},{0,1,61},{0,1,47},{0,1,65},{0,2,18},{0,1,9},{0,1,0},{0,1,9},{0,1,40},{0,1,22},{0,1,13},{0,1,61},{1,0,40},{0,1,61},{0,1,0},{0,1,0},{0,1,0},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,1,0},
{0,1,0},{0,1,0},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{1,0,18},{0,1,9},{0,1,0},{0,1,9},{1,0,18},{2,0,18},{0,1,9},{0,1,36},{2,0,18},{0,1,36},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,5,54},{0,3,40},{0,2,61},
{0,2,36},{0,4,51},{0,2,37},{0,2,1},{0,2,52},{0,2,77},{0,1,73},{1,3,22},{1,2,13},{1,2,4},{1,2,13},{0,4,51},{1,2,34},{0,2,1},{0,2,52},{4,0,51},{0,2,52},{0,4,37},{0,4,37},{0,4,37},{0,2,36},{0,3,8},{0,2,1},{0,2,1},{0,1,5},{0,1,30},{0,1,9},{1,2,4},{1,2,4},{1,2,4},{1,1,8},{1,1,8},
{0,2,1},{0,2,1},{0,1,5},{3,0,8},{0,1,5},{2,1,18},{0,3,4},{1,2,0},{0,2,0},{2,1,18},{5,0,18},{0,2,0},{0,2,36},{5,0,18},{0,2,36},{0,0,36},{0,0,36},{0,0,36},{0,0,36},{0,2,1},{0,2,1},{0,2,1},{0,1,1},{0,1,5},{0,1,5},{1,6,54},{1,4,40},{1,3,61},{1,3,36},{1,5,51},{1,3,37},{1,3,1},
{1,3,52},{0,3,72},{0,3,40},{2,4,22},{2,3,13},{2,3,4},{2,3,13},{1,5,51},{0,4,29},{1,3,1},{0,3,40},{7,0,51},{0,3,40},{1,5,37},{1,5,37},{1,5,37},{1,3,36},{1,4,8},{1,3,1},{1,3,1},{1,2,5},{0,3,8},{1,2,9},{2,3,4},{2,3,4},{2,3,4},{2,2,8},{3,0,8},{1,3,1},{1,3,1},{1,2,5},{6,0,8},
{1,2,5},{3,2,18},{1,4,4},{2,3,0},{1,3,0},{3,2,18},{6,1,18},{1,3,0},{0,3,36},{6,1,18},{0,3,36},{1,0,36},{1,0,36},{1,0,36},{1,0,36},{1,3,1},{1,3,1},{1,3,1},{1,2,1},{0,3,4},{0,3,4},{2,7,54},{2,5,40},{2,4,62},{2,4,38},{2,6,51},{2,4,27},{2,4,3},{2,3,69},{0,5,60},{1,4,54},{3,5,22},
{3,4,9},{3,4,6},{3,4,21},{5,0,51},{2,4,27},{2,4,3},{0,4,50},{10,0,51},{0,4,50},{2,6,37},{2,6,37},{2,6,37},{2,4,37},{2,5,8},{2,4,2},{2,4,2},{2,3,5},{1,4,8},{2,3,9},{3,4,5},{3,4,5},{3,4,5},{3,3,8},{4,1,8},{2,4,2},{2,4,2},{2,3,5},{9,0,8},{2,3,5},{3,5,18},{2,5,4},{3,4,2},
{2,4,2},{3,5,18},{7,2,18},{2,4,2},{0,4,50},{7,2,18},{0,4,50},{2,0,36},{2,0,36},{2,0,36},{2,0,36},{2,4,1},{2,4,1},{2,4,1},{2,3,1},{1,4,4},{1,4,4},{3,8,68},{3,6,60},{4,5,68},{3,5,50},{3,7,53},{3,6,28},{3,5,5},{3,5,53},{3,5,68},{2,5,38},{4,6,22},{4,5,13},{4,5,4},{4,5,13},{6,1,52},
{3,6,27},{3,5,4},{1,5,37},{13,0,52},{1,5,37},{3,7,50},{3,7,50},{3,7,50},{3,5,50},{3,6,11},{3,5,5},{3,5,5},{3,4,6},{2,5,11},{3,4,9},{4,5,4},{4,5,4},{4,5,4},{4,4,8},{4,4,8},{3,5,4},{3,5,4},{3,4,5},{12,0,8},{3,4,5},{7,0,18},{4,5,9},{4,5,0},{3,5,0},{7,0,18},{14,0,18},{3,5,0},
{0,5,36},{14,0,18},{0,5,36},{3,0,50},{3,0,50},{3,0,50},{3,0,50},{3,6,2},{3,6,2},{3,6,2},{3,4,2},{2,5,2},{2,5,2},{4,9,54},{4,7,40},{4,6,61},{4,6,36},{4,8,51},{4,6,37},{4,6,1},{4,6,52},{2,7,68},{3,6,38},{5,7,22},{5,6,13},{5,6,4},{5,6,13},{8,0,51},{5,6,34},{4,6,1},{2,6,37},{16,0,51},
{2,6,37},{4,8,37},{4,8,37},{4,8,37},{4,6,36},{4,7,8},{4,6,1},{4,6,1},{4,5,5},{3,6,11},{4,5,9},{5,6,4},{5,6,4},{5,6,4},{5,5,8},{5,5,8},{4,6,1},{4,6,1},{4,5,5},{15,0,8},{4,5,5},{8,1,18},{4,7,4},{5,6,0},{4,6,0},{8,1,18},{17,0,18},{4,6,0},{0,6,36},{17,0,18},{0,6,36},{4,0,36},
{4,0,36},{4,0,36},{4,0,36},{4,6,1},{4,6,1},{4,6,1},{4,5,1},{3,6,2},{3,6,2},{5,10,54},{5,8,40},{5,7,61},{5,7,36},{5,9,51},{5,7,37},{5,7,1},{5,7,52},{3,8,60},{4,7,40},{6,8,22},{6,7,13},{6,7,4},{6,7,13},{9,1,51},{4,8,29},{5,7,1},{3,7,37},{19,0,51},{3,7,37},{5,9,37},{5,9,37},{5,9,37},
{5,7,36},{5,8,8},{5,7,1},{5,7,1},{5,6,5},{4,7,8},{5,6,9},{6,7,4},{6,7,4},{6,7,4},{6,6,8},{9,0,8},{5,7,1},{5,7,1},{5,6,5},{18,0,8},{5,6,5},{10,0,18},{5,8,4},{6,7,0},{5,7,0},{10,0,18},{18,1,18},{5,7,0},{0,7,36},{18,1,18},{0,7,36},{5,0,36},{5,0,36},{5,0,36},{5,0,36},{5,7,1},
{5,7,1},{5,7,1},{5,6,1},{4,7,4},{4,7,4},{6,11,54},{6,9,40},{6,8,62},{6,8,38},{6,10,51},{6,8,27},{6,8,3},{6,7,69},{4,9,60},{5,8,54},{7,9,22},{7,8,9},{7,8,6},{7,8,21},{11,0,51},{6,8,27},{6,8,3},{4,8,50},{22,0,51},{4,8,50},{6,10,37},{6,10,37},{6,10,37},{6,8,37},{6,9,8},{6,8,2},{6,8,2},
{6,7,5},{5,8,8},{6,7,9},{7,8,5},{7,8,5},{7,8,5},{7,7,8},{10,1,8},{6,8,2},{6,8,2},{6,7,5},{21,0,8},{6,7,5},{11,1,18},{6,9,4},{7,8,2},{6,8,2},{11,1,18},{19,2,18},{6,8,2},{0,8,50},{19,2,18},{0,8,50},{6,0,36},{6,0,36},{6,0,36},{6,0,36},{6,8,1},{6,8,1},{6,8,1},{6,7,1},{5,8,4},
{5,8,4},{7,12,68},{7,10,60},{8,9,68},{7,9,50},{7,11,53},{7,10,28},{7,9,5},{7,9,53},{7,9,68},{6,9,38},{8,10,22},{8,9,13},{8,9,4},{8,9,13},{12,1,52},{7,10,27},{7,9,4},{5,9,37},{25,0,52},{5,9,37},{7,11,50},{7,11,50},{7,11,50},{7,9,50},{7,10,11},{7,9,5},{7,9,5},{7,8,6},{6,9,11},{7,8,9},{8,9,4},
{8,9,4},{8,9,4},{8,8,8},{12,0,8},{7,9,4},{7,9,4},{7,8,5},{24,0,8},{7,8,5},{13,0,18},{8,9,9},{8,9,0},{7,9,0},{13,0,18},{26,0,18},{7,9,0},{0,9,36},{26,0,18},{0,9,36},{7,0,50},{7,0,50},{7,0,50},{7,0,50},{7,10,2},{7,10,2},{7,10,2},{7,8,2},{6,9,2},{6,9,2},{8,13,54},{8,11,40},{8,10,61},
{8,10,36},{8,12,51},{8,10,37},{8,10,1},{8,10,52},{6,11,68},{7,10,38},{9,11,22},{9,10,13},{9,10,4},{9,10,13},{12,4,51},{9,10,34},{8,10,1},{6,10,37},{28,0,51},{6,10,37},{8,12,37},{8,12,37},{8,12,37},{8,10,36},{8,11,8},{8,10,1},{8,10,1},{8,9,5},{7,10,11},{8,9,9},{9,10,4},{9,10,4},{9,10,4},{9,9,8},{13,1,8},
{8,10,1},{8,10,1},{8,9,5},{27,0,8},{8,9,5},{14,1,18},{8,11,4},{9,10,0},{8,10,0},{14,1,18},{29,0,18},{8,10,0},{0,10,36},{29,0,18},{0,10,36},{8,0,36},{8,0,36},{8,0,36},{8,0,36},{8,10,1},{8,10,1},{8,10,1},{8,9,1},{7,10,2},{7,10,2},{9,14,54},{9,12,40},{9,11,61},{9,11,36},{9,13,51},{9,11,37},{9,11,1},
{9,11,52},{7,12,60},{8,11,40},{10,12,22},{10,11,13},{10,11,4},{10,11,13},{13,5,51},{8,12,29},{9,11,1},{7,11,37},{31,0,51},{7,11,37},{9,13,37},{9,13,37},{9,13,37},{9,11,36},{9,12,8},{9,11,1},{9,11,1},{9,10,5},{8,11,8},{9,10,9},{10,11,4},{10,11,4},{10,11,4},{10,10,8},{15,0,8},{9,11,1},{9,11,1},{9,10,5},{30,0,8},
{9,10,5},{15,2,18},{9,12,4},{10,11,0},{9,11,0},{15,2,18},{30,1,18},{9,11,0},{0,11,36},{30,1,18},{0,11,36},{9,0,36},{9,0,36},{9,0,36},{9,0,36},{9,11,1},{9,11,1},{9,11,1},{9,10,1},{8,11,4},{8,11,4},{10,15,54},{10,13,40},{10,12,62},{10,12,38},{10,14,51},{10,12,27},{10,12,3},{10,11,69},{8,13,60},{9,12,54},{11,13,22},
{11,12,9},{11,12,6},{11,12,21},{17,0,51},{10,12,27},{10,12,3},{8,12,50},{30,2,51},{8,12,50},{10,14,37},{10,14,37},{10,14,37},{10,12,37},{10,13,8},{10,12,2},{10,12,2},{10,11,5},{9,12,8},{10,11,9},{11,12,5},{11,12,5},{11,12,5},{11,11,8},{16,1,8},{10,12,2},{10,12,2},{10,11,5},{31,1,8},{10,11,5},{15,5,18},{10,13,4},{11,12,2},
{10,12,2},{15,5,18},{31,2,18},{10,12,2},{0,12,50},{31,2,18},{0,12,50},{10,0,36},{10,0,36},{10,0,36},{10,0,36},{10,12,1},{10,12,1},{10,12,1},{10,11,1},{9,12,4},{9,12,4},{11,16,68},{11,14,60},{12,13,68},{11,13,50},{11,15,53},{11,14,28},{11,13,5},{11,13,53},{11,13,68},{10,13,38},{12,14,22},{12,13,13},{12,13,4},{12,13,13},{18,1,52},
{11,14,27},{11,13,4},{9,13,37},{31,3,52},{9,13,37},{11,15,50},{11,15,50},{11,15,50},{11,13,50},{11,14,11},{11,13,5},{11,13,5},{11,12,6},{10,13,11},{11,12,9},{12,13,4},{12,13,4},{12,13,4},{12,12,8},{16,4,8},{11,13,4},{11,13,4},{11,12,5},{28,4,8},{11,12,5},{19,0,18},{12,13,9},{12,13,0},{11,13,0},{19,0,18},{30,4,18},{11,13,0},
{0,13,36},{30,4,18},{0,13,36},{11,0,50},{11,0,50},{11,0,50},{11,0,50},{11,14,2},{11,14,2},{11,14,2},{11,12,2},{10,13,2},{10,13,2},{12,17,54},{12,15,40},{12,14,61},{12,14,36},{12,16,51},{12,14,37},{12,14,1},{12,14,52},{10,15,68},{11,14,38},{13,15,22},{13,14,13},{13,14,4},{13,14,13},{20,0,51},{13,14,34},{12,14,1},{10,14,37},{24,8,51},
{10,14,37},{12,16,37},{12,16,37},{12,16,37},{12,14,36},{12,15,8},{12,14,1},{12,14,1},{12,13,5},{11,14,11},{12,13,9},{13,14,4},{13,14,4},{13,14,4},{13,13,8},{17,5,8},{12,14,1},{12,14,1},{12,13,5},{31,4,8},{12,13,5},{20,1,18},{12,15,4},{13,14,0},{12,14,0},{20,1,18},{31,5,18},{12,14,0},{0,14,36},{31,5,18},{0,14,36},{12,0,36},
{12,0,36},{12,0,36},{12,0,36},{12,14,1},{12,14,1},{12,14,1},{12,13,1},{11,14,2},{11,14,2},{13,18,54},{13,16,40},{13,15,61},{13,15,36},{13,17,51},{13,15,37},{13,15,1},{13,15,52},{11,16,60},{12,15,40},{14,16,22},{14,15,13},{14,15,4},{14,15,13},{21,1,51},{12,16,29},{13,15,1},{11,15,37},{27,8,51},{11,15,37},{13,17,37},{13,17,37},{13,17,37},
{13,15,36},{13,16,8},{13,15,1},{13,15,1},{13,14,5},{12,15,8},{13,14,9},{14,15,4},{14,15,4},{14,15,4},{14,14,8},{21,0,8},{13,15,1},{13,15,1},{13,14,5},{30,6,8},{13,14,5},{22,0,18},{13,16,4},{14,15,0},{13,15,0},{22,0,18},{30,7,18},{13,15,0},{0,15,36},{30,7,18},{0,15,36},{13,0,36},{13,0,36},{13,0,36},{13,0,36},{13,15,1},
{13,15,1},{13,15,1},{13,14,1},{12,15,4},{12,15,4},{14,19,54},{14,17,40},{14,16,62},{14,16,38},{14,18,51},{14,16,27},{14,16,3},{14,15,69},{12,17,60},{13,16,54},{15,17,22},{15,16,9},{15,16,6},{15,16,21},{23,0,51},{14,16,27},{14,16,3},{12,16,50},{30,8,51},{12,16,50},{14,18,37},{14,18,37},{14,18,37},{14,16,37},{14,17,8},{14,16,2},{14,16,2},
{14,15,5},{13,16,8},{14,15,9},{15,16,5},{15,16,5},{15,16,5},{15,15,8},{22,1,8},{14,16,2},{14,16,2},{14,15,5},{31,7,8},{14,15,5},{23,1,18},{14,17,4},{15,16,2},{14,16,2},{23,1,18},{27,10,18},{14,16,2},{0,16,50},{27,10,18},{0,16,50},{14,0,36},{14,0,36},{14,0,36},{14,0,36},{14,16,1},{14,16,1},{14,16,1},{14,15,1},{13,16,4},
{13,16,4},{15,20,68},{15,18,60},{16,17,68},{15,17,50},{15,19,53},{15,18,28},{15,17,5},{15,17,53},{15,17,68},{14,17,38},{16,18,22},{16,17,13},{16,17,4},{16,17,13},{24,1,52},{15,18,27},{15,17,4},{13,17,37},{31,9,52},{13,17,37},{15,19,50},{15,19,50},{15,19,50},{15,17,50},{15,18,11},{15,17,5},{15,17,5},{15,16,6},{14,17,11},{15,16,9},{16,17,4},
{16,17,4},{16,17,4},{16,16,8},{24,0,8},{15,17,4},{15,17,4},{15,16,5},{24,12,8},{15,16,5},{25,0,18},{16,17,9},{16,17,0},{15,17,0},{25,0,18},{30,10,18},{15,17,0},{0,17,36},{30,10,18},{0,17,36},{15,0,50},{15,0,50},{15,0,50},{15,0,50},{15,18,2},{15,18,2},{15,18,2},{15,16,2},{14,17,2},{14,17,2},{16,21,54},{16,19,40},{16,18,61},
{16,18,36},{16,20,51},{16,18,37},{16,18,1},{16,18,52},{14,19,68},{15,18,38},{17,19,22},{17,18,13},{17,18,4},{17,18,13},{24,4,51},{17,18,34},{16,18,1},{14,18,37},{28,12,51},{14,18,37},{16,20,37},{16,20,37},{16,20,37},{16,18,36},{16,19,8},{16,18,1},{16,18,1},{16,17,5},{15,18,11},{16,17,9},{17,18,4},{17,18,4},{17,18,4},{17,17,8},{25,1,8},
{16,18,1},{16,18,1},{16,17,5},{27,12,8},{16,17,5},{26,1,18},{16,19,4},{17,18,0},{16,18,0},{26,1,18},{31,11,18},{16,18,0},{0,18,36},{31,11,18},{0,18,36},{16,0,36},{16,0,36},{16,0,36},{16,0,36},{16,18,1},{16,18,1},{16,18,1},{16,17,1},{15,18,2},{15,18,2},{17,22,54},{17,20,40},{17,19,61},{17,19,36},{17,21,51},{17,19,37},{17,19,1},
{17,19,52},{15,20,60},{16,19,40},{18,20,22},{18,19,13},{18,19,4},{18,19,13},{25,5,51},{16,20,29},{17,19,1},{15,19,37},{31,12,51},{15,19,37},{17,21,37},{17,21,37},{17,21,37},{17,19,36},{17,20,8},{17,19,1},{17,19,1},{17,18,5},{16,19,8},{17,18,9},{18,19,4},{18,19,4},{18,19,4},{18,18,8},{27,0,8},{17,19,1},{17,19,1},{17,18,5},{30,12,8},
{17,18,5},{27,2,18},{17,20,4},{18,19,0},{17,19,0},{27,2,18},{30,13,18},{17,19,0},{0,19,36},{30,13,18},{0,19,36},{17,0,36},{17,0,36},{17,0,36},{17,0,36},{17,19,1},{17,19,1},{17,19,1},{17,18,1},{16,19,4},{16,19,4},{18,23,54},{18,21,40},{18,20,62},{18,20,38},{18,22,51},{18,20,27},{18,20,3},{18,19,69},{16,21,60},{17,20,54},{19,21,22},
{19,20,9},{19,20,6},{19,20,21},{29,0,51},{18,20,27},{18,20,3},{16,20,50},{30,14,51},{16,20,50},{18,22,37},{18,22,37},{18,22,37},{18,20,37},{18,21,8},{18,20,2},{18,20,2},{18,19,5},{17,20,8},{18,19,9},{19,20,5},{19,20,5},{19,20,5},{19,19,8},{28,1,8},{18,20,2},{18,20,2},{18,19,5},{31,13,8},{18,19,5},{27,5,18},{18,21,4},{19,20,2},
{18,20,2},{27,5,18},{31,14,18},{18,20,2},{0,20,50},{31,14,18},{0,20,50},{18,0,36},{18,0,36},{18,0,36},{18,0,36},{18,20,1},{18,20,1},{18,20,1},{18,19,1},{17,20,4},{17,20,4},{19,24,68},{19,22,60},{20,21,68},{19,21,50},{19,23,53},{19,22,28},{19,21,5},{19,21,53},{19,21,68},{18,21,38},{20,22,22},{20,21,13},{20,21,4},{20,21,13},{30,1,52},
{19,22,27},{19,21,4},{17,21,37},{31,15,52},{17,21,37},{19,23,50},{19,23,50},{19,23,50},{19,21,50},{19,22,11},{19,21,5},{19,21,5},{19,20,6},{18,21,11},{19,20,9},{20,21,4},{20,21,4},{20,21,4},{20,20,8},{28,4,8},{19,21,4},{19,21,4},{19,20,5},{28,16,8},{19,20,5},{31,0,18},{20,21,9},{20,21,0},{19,21,0},{31,0,18},{30,16,18},{19,21,0},
{0,21,36},{30,16,18},{0,21,36},{19,0,50},{19,0,50},{19,0,50},{19,0,50},{19,22,2},{19,22,2},{19,22,2},{19,20,2},{18,21,2},{18,21,2},{20,25,54},{20,23,40},{20,22,61},{20,22,36},{20,24,51},{20,22,37},{20,22,1},{20,22,52},{18,23,68},{19,22,38},{21,23,22},{21,22,13},{21,22,4},{21,22,13},{28,8,51},{21,22,34},{20,22,1},{18,22,37},{24,20,51},
{18,22,37},{20,24,37},{20,24,37},{20,24,37},{20,22,36},{20,23,8},{20,22,1},{20,22,1},{20,21,5},{19,22,11},{20,21,9},{21,22,4},{21,22,4},{21,22,4},{21,21,8},{29,5,8},{20,22,1},{20,22,1},{20,21,5},{31,16,8},{20,21,5},{31,3,18},{20,23,4},{21,22,0},{20,22,0},{31,3,18},{31,17,18},{20,22,0},{0,22,36},{31,17,18},{0,22,36},{20,0,36},
{20,0,36},{20,0,36},{20,0,36},{20,22,1},{20,22,1},{20,22,1},{20,21,1},{19,22,2},{19,22,2},{21,26,54},{21,24,40},{21,23,61},{21,23,36},{21,25,51},{21,23,37},{21,23,1},{21,23,52},{19,24,60},{20,23,40},{22,24,22},{22,23,13},{22,23,4},{22,23,13},{29,9,51},{20,24,29},{21,23,1},{19,23,37},{27,20,51},{19,23,37},{21,25,37},{21,25,37},{21,25,37},
{21,23,36},{21,24,8},{21,23,1},{21,23,1},{21,22,5},{20,23,8},{21,22,9},{22,23,4},{22,23,4},{22,23,4},{22,22,8},{31,4,8},{21,23,1},{21,23,1},{21,22,5},{30,18,8},{21,22,5},{31,6,18},{21,24,4},{22,23,0},{21,23,0},{31,6,18},{30,19,18},{21,23,0},{0,23,36},{30,19,18},{0,23,36},{21,0,36},{21,0,36},{21,0,36},{21,0,36},{21,23,1},
{21,23,1},{21,23,1},{21,22,1},{20,23,4},{20,23,4},{22,27,54},{22,25,40},{22,24,62},{22,24,38},{22,26,51},{22,24,27},{22,24,3},{22,23,69},{20,25,60},{21,24,54},{23,25,22},{23,24,9},{23,24,6},{23,24,21},{31,8,51},{22,24,27},{22,24,3},{20,24,50},{30,20,51},{20,24,50},{22,26,37},{22,26,37},{22,26,37},{22,24,37},{22,25,8},{22,24,2},{22,24,2},
{22,23,5},{21,24,8},{22,23,9},{23,24,5},{23,24,5},{23,24,5},{23,23,8},{31,7,8},{22,24,2},{22,24,2},{22,23,5},{31,19,8},{22,23,5},{31,9,18},{22,25,4},{23,24,2},{22,24,2},{31,9,18},{27,22,18},{22,24,2},{0,24,50},{27,22,18},{0,24,50},{22,0,36},{22,0,36},{22,0,36},{22,0,36},{22,24,1},{22,24,1},{22,24,1},{22,23,1},{21,24,4},
{21,24,4},{23,28,68},{23,26,60},{24,25,68},{23,25,50},{23,27,53},{23,26,28},{23,25,5},{23,25,53},{23,25,68},{22,25,38},{24,26,22},{24,25,13},{24,25,4},{24,25,13},{31,11,52},{23,26,27},{23,25,4},{21,25,37},{31,21,52},{21,25,37},{23,27,50},{23,27,50},{23,27,50},{23,25,50},{23,26,11},{23,25,5},{23,25,5},{23,24,6},{22,25,11},{23,24,9},{24,25,4},
{24,25,4},{24,25,4},{24,24,8},{28,16,8},{23,25,4},{23,25,4},{23,24,5},{24,24,8},{23,24,5},{31,12,18},{24,25,9},{24,25,0},{23,25,0},{31,12,18},{30,22,18},{23,25,0},{0,25,36},{30,22,18},{0,25,36},{23,0,50},{23,0,50},{23,0,50},{23,0,50},{23,26,2},{23,26,2},{23,26,2},{23,24,2},{22,25,2},{22,25,2},{24,29,54},{24,27,40},{24,26,61},
{24,26,36},{24,28,51},{24,26,37},{24,26,1},{24,26,52},{22,27,68},{23,26,38},{25,27,22},{25,26,13},{25,26,4},{25,26,13},{28,20,51},{25,26,34},{24,26,1},{22,26,37},{28,24,51},{22,26,37},{24,28,37},{24,28,37},{24,28,37},{24,26,36},{24,27,8},{24,26,1},{24,26,1},{24,25,5},{23,26,11},{24,25,9},{25,26,4},{25,26,4},{25,26,4},{25,25,8},{29,17,8},
{24,26,1},{24,26,1},{24,25,5},{27,24,8},{24,25,5},{31,15,18},{24,27,4},{25,26,0},{24,26,0},{31,15,18},{31,23,18},{24,26,0},{0,26,36},{31,23,18},{0,26,36},{24,0,36},{24,0,36},{24,0,36},{24,0,36},{24,26,1},{24,26,1},{24,26,1},{24,25,1},{23,26,2},{23,26,2},{25,30,54},{25,28,40},{25,27,61},{25,27,36},{25,29,51},{25,27,37},{25,27,1},
{25,27,52},{23,28,60},{24,27,40},{26,28,22},{26,27,13},{26,27,4},{26,27,13},{29,21,51},{24,28,29},{25,27,1},{23,27,37},{31,24,51},{23,27,37},{25,29,37},{25,29,37},{25,29,37},{25,27,36},{25,28,8},{25,27,1},{25,27,1},{25,26,5},{24,27,8},{25,26,9},{26,27,4},{26,27,4},{26,27,4},{26,26,8},{31,16,8},{25,27,1},{25,27,1},{25,26,5},{30,24,8},
{25,26,5},{31,18,18},{25,28,4},{26,27,0},{25,27,0},{31,18,18},{30,25,18},{25,27,0},{0,27,36},{30,25,18},{0,27,36},{25,0,36},{25,0,36},{25,0,36},{25,0,36},{25,27,1},{25,27,1},{25,27,1},{25,26,1},{24,27,4},{24,27,4},{26,31,54},{26,29,40},{26,28,62},{26,28,38},{26,30,51},{26,28,27},{26,28,3},{26,27,69},{24,29,60},{25,28,54},{27,29,22},
{27,28,9},{27,28,6},{27,28,21},{31,20,51},{26,28,27},{26,28,3},{24,28,50},{30,26,51},{24,28,50},{26,30,37},{26,30,37},{26,30,37},{26,28,37},{26,29,8},{26,28,2},{26,28,2},{26,27,5},{25,28,8},{26,27,9},{27,28,5},{27,28,5},{27,28,5},{27,27,8},{31,19,8},{26,28,2},{26,28,2},{26,27,5},{31,25,8},{26,27,5},{31,21,18},{26,29,4},{27,28,2},
{26,28,2},{31,21,18},{31,26,18},{26,28,2},{0,28,50},{31,26,18},{0,28,50},{26,0,36},{26,0,36},{26,0,36},{26,0,36},{26,28,1},{26,28,1},{26,28,1},{26,27,1},{25,28,4},{25,28,4},{28,30,86},{27,30,60},{28,29,68},{27,29,50},{27,31,53},{27,30,28},{27,29,5},{27,29,53},{27,29,68},{26,29,38},{28,30,22},{28,29,13},{28,29,4},{28,29,13},{31,23,52},
{27,30,27},{27,29,4},{25,29,37},{31,27,52},{25,29,37},{27,31,50},{27,31,50},{27,31,50},{27,29,50},{27,30,11},{27,29,5},{27,29,5},{27,28,6},{26,29,11},{27,28,9},{28,29,4},{28,29,4},{28,29,4},{28,28,8},{28,28,8},{27,29,4},{27,29,4},{27,28,5},{28,28,8},{27,28,5},{31,24,18},{28,29,9},{28,29,0},{27,29,0},{31,24,18},{30,28,18},{27,29,0},
{0,29,36},{30,28,18},{0,29,36},{27,0,50},{27,0,50},{27,0,50},{27,0,50},{27,30,2},{27,30,2},{27,30,2},{27,28,2},{26,29,2},{26,29,2},{29,31,86},{28,31,40},{28,30,61},{28,30,36},{28,31,72},{28,30,37},{28,30,1},{28,30,52},{26,31,68},{27,30,38},{29,31,22},{29,30,13},{29,30,4},{29,30,13},{31,26,52},{29,30,34},{28,30,1},{26,30,37},{30,29,52},
{26,30,37},{28,31,40},{28,31,40},{28,31,40},{28,30,36},{28,31,8},{28,30,1},{28,30,1},{28,29,5},{27,30,11},{28,29,9},{29,30,4},{29,30,4},{29,30,4},{29,29,8},{29,29,8},{28,30,1},{28,30,1},{28,29,5},{31,28,8},{28,29,5},{31,27,18},{28,31,4},{29,30,0},{28,30,0},{31,27,18},{31,29,18},{28,30,0},{0,30,36},{31,29,18},{0,30,36},{28,0,36},
{28,0,36},{28,0,36},{28,0,36},{28,30,1},{28,30,1},{28,30,1},{28,29,1},{27,30,2},{27,30,2},{30,31,104},{30,31,77},{29,31,61},{29,31,36},{30,31,116},{29,31,37},{29,31,1},{29,31,52},{28,31,72},{28,31,40},{30,31,40},{30,31,13},{30,31,4},{30,31,13},{31,29,52},{30,31,34},{29,31,1},{27,31,37},{31,30,52},{27,31,37},{29,31,61},{29,31,61},{29,31,61},
{29,31,36},{29,31,26},{29,31,1},{29,31,1},{29,30,5},{28,31,8},{29,30,9},{30,31,4},{30,31,4},{30,31,4},{30,30,8},{31,28,8},{29,31,1},{29,31,1},{29,30,5},{30,30,8},{29,30,5},{31,30,18},{30,31,9},{30,31,0},{29,31,0},{31,30,18},{30,31,18},{29,31,0},{0,31,36},{30,31,18},{0,31,36},{29,0,36},{29,0,36},{29,0,36},{29,0,36},{29,31,1},
{29,31,1},{29,31,1},{29,30,1},{28,31,4},{28,31,4},{31,31,68},{31,31,68},{31,31,68},{30,31,45},{31,31,68},{30,31,34},{30,31,25},{30,31,1},{30,31,23},{30,31,5},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{30,31,1},{31,31,4},{30,31,1},{31,31,68},{31,31,68},{31,31,68},{30,31,45},{30,31,52},{30,31,25},{30,31,25},
{30,31,1},{30,31,14},{30,31,5},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{31,31,4},{30,31,1},{31,31,4},{30,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{30,0,36},{30,0,36},{30,0,36},{30,0,36},{30,31,16},{30,31,16},{30,31,16},{30,31,1},{30,31,5},
{30,31,5},{0,4,72},{0,3,10},{0,2,1},{0,2,26},{0,3,154},{0,2,99},{0,2,51},{0,1,115},{0,1,173},{0,1,119},{0,4,72},{0,3,10},{0,2,1},{0,2,26},{1,1,154},{0,2,99},{0,2,51},{0,1,115},{3,0,154},{0,1,115},{0,2,0},{0,2,0},{0,2,0},{0,1,0},{0,1,13},{0,1,4},{0,1,4},{0,0,25},{0,0,25},{0,0,25},{0,2,0},
{0,2,0},{0,2,0},{0,1,0},{0,1,13},{0,1,4},{0,1,4},{0,0,25},{1,0,13},{0,0,25},{0,4,72},{0,3,10},{0,2,1},{0,2,26},{0,4,72},{4,0,72},{0,2,26},{0,1,90},{4,0,72},{0,1,90},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,7,81},{0,5,13},{0,3,35},
{0,3,19},{0,5,244},{0,3,115},{0,3,35},{0,2,139},{0,2,308},{0,2,164},{0,7,81},{0,5,13},{1,3,26},{0,3,19},{2,1,244},{0,3,115},{0,3,35},{0,2,139},{5,0,244},{0,2,139},{0,5,9},{0,5,9},{0,5,9},{0,2,18},{0,3,50},{0,2,13},{0,2,13},{0,1,29},{0,1,72},{0,1,33},{0,5,9},{0,5,9},{0,5,9},{0,2,18},{1,1,50},
{0,2,13},{0,2,13},{0,1,29},{3,0,50},{0,1,29},{1,5,72},{0,5,4},{1,3,1},{0,3,10},{1,5,72},{7,0,72},{0,3,10},{0,2,90},{7,0,72},{0,2,90},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,1,1},{0,1,1},{0,1,1},{0,1,4},{0,1,8},{0,1,8},{1,8,145},{1,6,77},{1,4,98},{1,4,82},{0,8,243},{0,5,76},{0,4,1},
{0,3,115},{0,4,364},{0,3,179},{1,8,81},{1,6,13},{2,4,25},{1,4,18},{4,0,243},{0,5,76},{0,4,1},{0,3,115},{8,0,243},{0,3,115},{1,6,73},{1,6,73},{1,6,73},{1,3,82},{0,6,50},{0,4,1},{0,4,1},{0,2,26},{0,3,114},{0,2,51},{1,6,9},{1,6,9},{1,6,9},{1,3,18},{3,0,50},{0,4,1},{0,4,1},{0,2,26},{6,0,50},
{0,2,26},{5,0,72},{1,6,4},{2,4,0},{0,4,1},{5,0,72},{10,0,72},{0,4,1},{0,3,90},{10,0,72},{0,3,90},{1,0,73},{1,0,73},{1,0,73},{1,0,73},{0,4,0},{0,4,0},{0,4,0},{0,2,1},{0,2,26},{0,2,26},{2,9,162},{2,7,94},{2,5,115},{2,5,99},{1,9,244},{1,6,77},{1,5,2},{1,4,106},{0,5,300},{0,4,105},{2,9,81},
{2,7,13},{3,5,25},{2,5,18},{5,1,243},{1,6,76},{1,5,1},{0,4,89},{11,0,243},{0,4,89},{2,7,90},{2,7,90},{2,7,90},{2,4,94},{1,7,51},{1,5,2},{1,5,2},{1,3,27},{0,4,74},{0,3,35},{2,7,9},{2,7,9},{2,7,9},{2,4,13},{4,1,50},{1,5,1},{1,5,1},{0,4,25},{9,0,50},{0,4,25},{6,1,72},{2,7,4},{3,5,0},
{1,5,1},{6,1,72},{13,0,72},{1,5,1},{0,4,80},{13,0,72},{0,4,80},{2,0,90},{2,0,90},{2,0,90},{2,0,90},{1,5,1},{1,5,1},{1,5,1},{1,3,2},{0,3,10},{0,3,10},{3,10,154},{3,8,84},{3,6,117},{3,6,85},{2,10,244},{2,7,75},{2,6,2},{2,5,106},{0,7,253},{0,5,100},{3,10,90},{3,8,20},{4,6,26},{3,6,21},{7,0,244},
{2,7,75},{2,6,2},{0,5,91},{14,0,244},{0,5,91},{3,8,80},{3,8,80},{3,8,80},{3,6,84},{2,8,52},{2,6,1},{2,6,1},{2,5,25},{0,6,52},{0,5,19},{3,8,16},{3,8,16},{3,8,16},{3,6,20},{4,4,50},{2,6,1},{2,6,1},{0,5,10},{12,0,50},{0,5,10},{8,0,72},{3,8,4},{4,6,1},{2,6,1},{8,0,72},{16,0,72},{2,6,1},
{0,5,90},{16,0,72},{0,5,90},{3,0,80},{3,0,80},{3,0,80},{3,0,80},{2,6,1},{2,6,1},{2,6,1},{2,4,1},{0,5,10},{0,5,10},{4,11,162},{4,9,94},{4,7,116},{4,7,100},{3,11,244},{3,8,76},{3,7,2},{3,6,106},{1,8,253},{1,6,100},{4,11,81},{4,9,13},{5,7,26},{4,7,19},{8,1,244},{3,8,76},{3,7,2},{1,6,91},{17,0,244},
{1,6,91},{4,9,90},{4,9,90},{4,9,90},{4,6,99},{3,9,52},{3,7,1},{3,7,1},{3,6,25},{1,7,52},{1,6,19},{4,9,9},{4,9,9},{4,9,9},{4,6,18},{5,5,50},{3,7,1},{3,7,1},{1,6,10},{15,0,50},{1,6,10},{9,1,72},{4,9,4},{5,7,1},{3,7,1},{9,1,72},{19,0,72},{3,7,1},{0,6,90},{19,0,72},{0,6,90},{4,0,90},
{4,0,90},{4,0,90},{4,0,90},{3,7,1},{3,7,1},{3,7,1},{3,5,1},{1,6,10},{1,6,10},{5,12,162},{5,10,94},{5,8,115},{5,8,99},{4,12,244},{4,9,77},{4,8,2},{4,7,116},{2,9,253},{2,7,100},{5,12,81},{5,10,13},{6,8,25},{5,8,18},{8,4,243},{4,9,76},{4,8,1},{2,7,91},{20,0,243},{2,7,91},{5,10,90},{5,10,90},{5,10,90},
{5,7,99},{4,10,51},{4,8,2},{4,8,2},{4,6,27},{2,8,50},{2,7,19},{5,10,9},{5,10,9},{5,10,9},{5,7,18},{9,0,50},{4,8,1},{4,8,1},{2,7,10},{18,0,50},{2,7,10},{11,0,72},{5,10,4},{6,8,0},{4,8,1},{11,0,72},{22,0,72},{4,8,1},{0,7,90},{22,0,72},{0,7,90},{5,0,90},{5,0,90},{5,0,90},{5,0,90},{4,8,1},
{4,8,1},{4,8,1},{4,6,2},{2,7,10},{2,7,10},{6,13,162},{6,11,94},{6,9,115},{6,9,99},{5,13,244},{5,10,77},{5,9,2},{5,8,106},{3,10,253},{3,8,84},{6,13,81},{6,11,13},{7,9,25},{6,9,18},{9,5,243},{5,10,76},{5,9,1},{3,8,80},{23,0,243},{3,8,80},{6,11,90},{6,11,90},{6,11,90},{6,8,94},{5,11,51},{5,9,2},{5,9,2},
{5,7,27},{3,9,50},{3,8,20},{6,11,9},{6,11,9},{6,11,9},{6,8,13},{10,1,50},{5,9,1},{5,9,1},{3,8,16},{21,0,50},{3,8,16},{12,1,72},{6,11,4},{7,9,0},{5,9,1},{12,1,72},{25,0,72},{5,9,1},{0,8,80},{25,0,72},{0,8,80},{6,0,90},{6,0,90},{6,0,90},{6,0,90},{5,9,1},{5,9,1},{5,9,1},{5,7,2},{3,8,4},
{3,8,4},{7,14,154},{7,12,84},{7,10,117},{7,10,85},{6,14,244},{6,11,75},{6,10,2},{6,9,106},{4,11,253},{4,9,100},{7,14,90},{7,12,20},{8,10,26},{7,10,21},{13,0,244},{6,11,75},{6,10,2},{4,9,91},{26,0,244},{4,9,91},{7,12,80},{7,12,80},{7,12,80},{7,10,84},{6,12,52},{6,10,1},{6,10,1},{6,9,25},{4,10,52},{4,9,19},{7,12,16},
{7,12,16},{7,12,16},{7,10,20},{12,0,50},{6,10,1},{6,10,1},{4,9,10},{24,0,50},{4,9,10},{12,4,72},{7,12,4},{8,10,1},{6,10,1},{12,4,72},{28,0,72},{6,10,1},{0,9,90},{28,0,72},{0,9,90},{7,0,80},{7,0,80},{7,0,80},{7,0,80},{6,10,1},{6,10,1},{6,10,1},{6,8,1},{4,9,10},{4,9,10},{8,15,162},{8,13,94},{8,11,116},
{8,11,100},{7,15,244},{7,12,76},{7,11,2},{7,10,106},{5,12,253},{5,10,100},{8,15,81},{8,13,13},{9,11,26},{8,11,19},{14,1,244},{7,12,76},{7,11,2},{5,10,91},{29,0,244},{5,10,91},{8,13,90},{8,13,90},{8,13,90},{8,10,99},{7,13,52},{7,11,1},{7,11,1},{7,10,25},{5,11,52},{5,10,19},{8,13,9},{8,13,9},{8,13,9},{8,10,18},{13,1,50},
{7,11,1},{7,11,1},{5,10,10},{27,0,50},{5,10,10},{13,5,72},{8,13,4},{9,11,1},{7,11,1},{13,5,72},{31,0,72},{7,11,1},{0,10,90},{31,0,72},{0,10,90},{8,0,90},{8,0,90},{8,0,90},{8,0,90},{7,11,1},{7,11,1},{7,11,1},{7,9,1},{5,10,10},{5,10,10},{9,16,162},{9,14,94},{9,12,115},{9,12,99},{8,16,244},{8,13,77},{8,12,2},
{8,11,116},{6,13,253},{6,11,100},{9,16,81},{9,14,13},{10,12,25},{9,12,18},{16,0,243},{8,13,76},{8,12,1},{6,11,91},{24,4,243},{6,11,91},{9,14,90},{9,14,90},{9,14,90},{9,11,99},{8,14,51},{8,12,2},{8,12,2},{8,10,27},{6,12,50},{6,11,19},{9,14,9},{9,14,9},{9,14,9},{9,11,18},{15,0,50},{8,12,1},{8,12,1},{6,11,10},{30,0,50},
{6,11,10},{17,0,72},{9,14,4},{10,12,0},{8,12,1},{17,0,72},{30,2,72},{8,12,1},{0,11,90},{30,2,72},{0,11,90},{9,0,90},{9,0,90},{9,0,90},{9,0,90},{8,12,1},{8,12,1},{8,12,1},{8,10,2},{6,11,10},{6,11,10},{10,17,162},{10,15,94},{10,13,115},{10,13,99},{9,17,244},{9,14,77},{9,13,2},{9,12,106},{7,14,253},{7,12,84},{10,17,81},
{10,15,13},{11,13,25},{10,13,18},{17,1,243},{9,14,76},{9,13,1},{7,12,80},{27,4,243},{7,12,80},{10,15,90},{10,15,90},{10,15,90},{10,12,94},{9,15,51},{9,13,2},{9,13,2},{9,11,27},{7,13,50},{7,12,20},{10,15,9},{10,15,9},{10,15,9},{10,12,13},{16,1,50},{9,13,1},{9,13,1},{7,12,16},{31,1,50},{7,12,16},{18,1,72},{10,15,4},{11,13,0},
{9,13,1},{18,1,72},{31,3,72},{9,13,1},{0,12,80},{31,3,72},{0,12,80},{10,0,90},{10,0,90},{10,0,90},{10,0,90},{9,13,1},{9,13,1},{9,13,1},{9,11,2},{7,12,4},{7,12,4},{11,18,154},{11,16,84},{11,14,117},{11,14,85},{10,18,244},{10,15,75},{10,14,2},{10,13,106},{8,15,253},{8,13,100},{11,18,90},{11,16,20},{12,14,26},{11,14,21},{19,0,244},
{10,15,75},{10,14,2},{8,13,91},{30,4,244},{8,13,91},{11,16,80},{11,16,80},{11,16,80},{11,14,84},{10,16,52},{10,14,1},{10,14,1},{10,13,25},{8,14,52},{8,13,19},{11,16,16},{11,16,16},{11,16,16},{11,14,20},{16,4,50},{10,14,1},{10,14,1},{8,13,10},{28,4,50},{8,13,10},{20,0,72},{11,16,4},{12,14,1},{10,14,1},{20,0,72},{24,8,72},{10,14,1},
{0,13,90},{24,8,72},{0,13,90},{11,0,80},{11,0,80},{11,0,80},{11,0,80},{10,14,1},{10,14,1},{10,14,1},{10,12,1},{8,13,10},{8,13,10},{12,19,162},{12,17,94},{12,15,116},{12,15,100},{11,19,244},{11,16,76},{11,15,2},{11,14,106},{9,16,253},{9,14,100},{12,19,81},{12,17,13},{13,15,26},{12,15,19},{20,1,244},{11,16,76},{11,15,2},{9,14,91},{31,5,244},
{9,14,91},{12,17,90},{12,17,90},{12,17,90},{12,14,99},{11,17,52},{11,15,1},{11,15,1},{11,14,25},{9,15,52},{9,14,19},{12,17,9},{12,17,9},{12,17,9},{12,14,18},{17,5,50},{11,15,1},{11,15,1},{9,14,10},{31,4,50},{9,14,10},{21,1,72},{12,17,4},{13,15,1},{11,15,1},{21,1,72},{27,8,72},{11,15,1},{0,14,90},{27,8,72},{0,14,90},{12,0,90},
{12,0,90},{12,0,90},{12,0,90},{11,15,1},{11,15,1},{11,15,1},{11,13,1},{9,14,10},{9,14,10},{13,20,162},{13,18,94},{13,16,115},{13,16,99},{12,20,244},{12,17,77},{12,16,2},{12,15,116},{10,17,253},{10,15,100},{13,20,81},{13,18,13},{14,16,25},{13,16,18},{20,4,243},{12,17,76},{12,16,1},{10,15,91},{28,8,243},{10,15,91},{13,18,90},{13,18,90},{13,18,90},
{13,15,99},{12,18,51},{12,16,2},{12,16,2},{12,14,27},{10,16,50},{10,15,19},{13,18,9},{13,18,9},{13,18,9},{13,15,18},{21,0,50},{12,16,1},{12,16,1},{10,15,10},{30,6,50},{10,15,10},{23,0,72},{13,18,4},{14,16,0},{12,16,1},{23,0,72},{30,8,72},{12,16,1},{0,15,90},{30,8,72},{0,15,90},{13,0,90},{13,0,90},{13,0,90},{13,0,90},{12,16,1},
{12,16,1},{12,16,1},{12,14,2},{10,15,10},{10,15,10},{14,21,162},{14,19,94},{14,17,115},{14,17,99},{13,21,244},{13,18,77},{13,17,2},{13,16,106},{11,18,253},{11,16,84},{14,21,81},{14,19,13},{15,17,25},{14,17,18},{21,5,243},{13,18,76},{13,17,1},{11,16,80},{31,8,243},{11,16,80},{14,19,90},{14,19,90},{14,19,90},{14,16,94},{13,19,51},{13,17,2},{13,17,2},
{13,15,27},{11,17,50},{11,16,20},{14,19,9},{14,19,9},{14,19,9},{14,16,13},{22,1,50},{13,17,1},{13,17,1},{11,16,16},{31,7,50},{11,16,16},{24,1,72},{14,19,4},{15,17,0},{13,17,1},{24,1,72},{31,9,72},{13,17,1},{0,16,80},{31,9,72},{0,16,80},{14,0,90},{14,0,90},{14,0,90},{14,0,90},{13,17,1},{13,17,1},{13,17,1},{13,15,2},{11,16,4},
{11,16,4},{15,22,154},{15,20,84},{15,18,117},{15,18,85},{14,22,244},{14,19,75},{14,18,2},{14,17,106},{12,19,253},{12,17,100},{15,22,90},{15,20,20},{16,18,26},{15,18,21},{25,0,244},{14,19,75},{14,18,2},{12,17,91},{30,10,244},{12,17,91},{15,20,80},{15,20,80},{15,20,80},{15,18,84},{14,20,52},{14,18,1},{14,18,1},{14,17,25},{12,18,52},{12,17,19},{15,20,16},
{15,20,16},{15,20,16},{15,18,20},{24,0,50},{14,18,1},{14,18,1},{12,17,10},{24,12,50},{12,17,10},{24,4,72},{15,20,4},{16,18,1},{14,18,1},{24,4,72},{28,12,72},{14,18,1},{0,17,90},{28,12,72},{0,17,90},{15,0,80},{15,0,80},{15,0,80},{15,0,80},{14,18,1},{14,18,1},{14,18,1},{14,16,1},{12,17,10},{12,17,10},{16,23,162},{16,21,94},{16,19,116},
{16,19,100},{15,23,244},{15,20,76},{15,19,2},{15,18,106},{13,20,253},{13,18,100},{16,23,81},{16,21,13},{17,19,26},{16,19,19},{26,1,244},{15,20,76},{15,19,2},{13,18,91},{31,11,244},{13,18,91},{16,21,90},{16,21,90},{16,21,90},{16,18,99},{15,21,52},{15,19,1},{15,19,1},{15,18,25},{13,19,52},{13,18,19},{16,21,9},{16,21,9},{16,21,9},{16,18,18},{25,1,50},
{15,19,1},{15,19,1},{13,18,10},{27,12,50},{13,18,10},{25,5,72},{16,21,4},{17,19,1},{15,19,1},{25,5,72},{31,12,72},{15,19,1},{0,18,90},{31,12,72},{0,18,90},{16,0,90},{16,0,90},{16,0,90},{16,0,90},{15,19,1},{15,19,1},{15,19,1},{15,17,1},{13,18,10},{13,18,10},{17,24,162},{17,22,94},{17,20,115},{17,20,99},{16,24,244},{16,21,77},{16,20,2},
{16,19,116},{14,21,253},{14,19,100},{17,24,81},{17,22,13},{18,20,25},{17,20,18},{28,0,243},{16,21,76},{16,20,1},{14,19,91},{24,16,243},{14,19,91},{17,22,90},{17,22,90},{17,22,90},{17,19,99},{16,22,51},{16,20,2},{16,20,2},{16,18,27},{14,20,50},{14,19,19},{17,22,9},{17,22,9},{17,22,9},{17,19,18},{27,0,50},{16,20,1},{16,20,1},{14,19,10},{30,12,50},
{14,19,10},{29,0,72},{17,22,4},{18,20,0},{16,20,1},{29,0,72},{30,14,72},{16,20,1},{0,19,90},{30,14,72},{0,19,90},{17,0,90},{17,0,90},{17,0,90},{17,0,90},{16,20,1},{16,20,1},{16,20,1},{16,18,2},{14,19,10},{14,19,10},{18,25,162},{18,23,94},{18,21,115},{18,21,99},{17,25,244},{17,22,77},{17,21,2},{17,20,106},{15,22,253},{15,20,84},{18,25,81},
{18,23,13},{19,21,25},{18,21,18},{29,1,243},{17,22,76},{17,21,1},{15,20,80},{27,16,243},{15,20,80},{18,23,90},{18,23,90},{18,23,90},{18,20,94},{17,23,51},{17,21,2},{17,21,2},{17,19,27},{15,21,50},{15,20,20},{18,23,9},{18,23,9},{18,23,9},{18,20,13},{28,1,50},{17,21,1},{17,21,1},{15,20,16},{31,13,50},{15,20,16},{30,1,72},{18,23,4},{19,21,0},
{17,21,1},{30,1,72},{31,15,72},{17,21,1},{0,20,80},{31,15,72},{0,20,80},{18,0,90},{18,0,90},{18,0,90},{18,0,90},{17,21,1},{17,21,1},{17,21,1},{17,19,2},{15,20,4},{15,20,4},{19,26,154},{19,24,84},{19,22,117},{19,22,85},{18,26,244},{18,23,75},{18,22,2},{18,21,106},{16,23,253},{16,21,100},{19,26,90},{19,24,20},{20,22,26},{19,22,21},{31,0,244},
{18,23,75},{18,22,2},{16,21,91},{30,16,244},{16,21,91},{19,24,80},{19,24,80},{19,24,80},{19,22,84},{18,24,52},{18,22,1},{18,22,1},{18,21,25},{16,22,52},{16,21,19},{19,24,16},{19,24,16},{19,24,16},{19,22,20},{28,4,50},{18,22,1},{18,22,1},{16,21,10},{28,16,50},{16,21,10},{28,8,72},{19,24,4},{20,22,1},{18,22,1},{28,8,72},{24,20,72},{18,22,1},
{0,21,90},{24,20,72},{0,21,90},{19,0,80},{19,0,80},{19,0,80},{19,0,80},{18,22,1},{18,22,1},{18,22,1},{18,20,1},{16,21,10},{16,21,10},{20,27,162},{20,25,94},{20,23,116},{20,23,100},{19,27,244},{19,24,76},{19,23,2},{19,22,106},{17,24,253},{17,22,100},{20,27,81},{20,25,13},{21,23,26},{20,23,19},{31,3,244},{19,24,76},{19,23,2},{17,22,91},{31,17,244},
{17,22,91},{20,25,90},{20,25,90},{20,25,90},{20,22,99},{19,25,52},{19,23,1},{19,23,1},{19,22,25},{17,23,52},{17,22,19},{20,25,9},{20,25,9},{20,25,9},{20,22,18},{29,5,50},{19,23,1},{19,23,1},{17,22,10},{31,16,50},{17,22,10},{29,9,72},{20,25,4},{21,23,1},{19,23,1},{29,9,72},{27,20,72},{19,23,1},{0,22,90},{27,20,72},{0,22,90},{20,0,90},
{20,0,90},{20,0,90},{20,0,90},{19,23,1},{19,23,1},{19,23,1},{19,21,1},{17,22,10},{17,22,10},{21,28,162},{21,26,94},{21,24,115},{21,24,99},{20,28,244},{20,25,77},{20,24,2},{20,23,116},{18,25,253},{18,23,100},{21,28,81},{21,26,13},{22,24,25},{21,24,18},{28,12,243},{20,25,76},{20,24,1},{18,23,91},{28,20,243},{18,23,91},{21,26,90},{21,26,90},{21,26,90},
{21,23,99},{20,26,51},{20,24,2},{20,24,2},{20,22,27},{18,24,50},{18,23,19},{21,26,9},{21,26,9},{21,26,9},{21,23,18},{31,4,50},{20,24,1},{20,24,1},{18,23,10},{30,18,50},{18,23,10},{31,8,72},{21,26,4},{22,24,0},{20,24,1},{31,8,72},{30,20,72},{20,24,1},{0,23,90},{30,20,72},{0,23,90},{21,0,90},{21,0,90},{21,0,90},{21,0,90},{20,24,1},
{20,24,1},{20,24,1},{20,22,2},{18,23,10},{18,23,10},{22,29,162},{22,27,94},{22,25,115},{22,25,99},{21,29,244},{21,26,77},{21,25,2},{21,24,106},{19,26,253},{19,24,84},{22,29,81},{22,27,13},{23,25,25},{22,25,18},{29,13,243},{21,26,76},{21,25,1},{19,24,80},{31,20,243},{19,24,80},{22,27,90},{22,27,90},{22,27,90},{22,24,94},{21,27,51},{21,25,2},{21,25,2},
{21,23,27},{19,25,50},{19,24,20},{22,27,9},{22,27,9},{22,27,9},{22,24,13},{31,7,50},{21,25,1},{21,25,1},{19,24,16},{31,19,50},{19,24,16},{31,11,72},{22,27,4},{23,25,0},{21,25,1},{31,11,72},{31,21,72},{21,25,1},{0,24,80},{31,21,72},{0,24,80},{22,0,90},{22,0,90},{22,0,90},{22,0,90},{21,25,1},{21,25,1},{21,25,1},{21,23,2},{19,24,4},
{19,24,4},{23,30,154},{23,28,84},{23,26,117},{23,26,85},{22,30,244},{22,27,75},{22,26,2},{22,25,106},{20,27,253},{20,25,100},{23,30,90},{23,28,20},{24,26,26},{23,26,21},{31,12,244},{22,27,75},{22,26,2},{20,25,91},{30,22,244},{20,25,91},{23,28,80},{23,28,80},{23,28,80},{23,26,84},{22,28,52},{22,26,1},{22,26,1},{22,25,25},{20,26,52},{20,25,19},{23,28,16},
{23,28,16},{23,28,16},{23,26,20},{28,16,50},{22,26,1},{22,26,1},{20,25,10},{24,24,50},{20,25,10},{28,20,72},{23,28,4},{24,26,1},{22,26,1},{28,20,72},{28,24,72},{22,26,1},{0,25,90},{28,24,72},{0,25,90},{23,0,80},{23,0,80},{23,0,80},{23,0,80},{22,26,1},{22,26,1},{22,26,1},{22,24,1},{20,25,10},{20,25,10},{24,31,162},{24,29,94},{24,27,116},
{24,27,100},{23,31,244},{23,28,76},{23,27,2},{23,26,106},{21,28,253},{21,26,100},{24,31,81},{24,29,13},{25,27,26},{24,27,19},{31,15,244},{23,28,76},{23,27,2},{21,26,91},{31,23,244},{21,26,91},{24,29,90},{24,29,90},{24,29,90},{24,26,99},{23,29,52},{23,27,1},{23,27,1},{23,26,25},{21,27,52},{21,26,19},{24,29,9},{24,29,9},{24,29,9},{24,26,18},{29,17,50},
{23,27,1},{23,27,1},{21,26,10},{27,24,50},{21,26,10},{29,21,72},{24,29,4},{25,27,1},{23,27,1},{29,21,72},{31,24,72},{23,27,1},{0,26,90},{31,24,72},{0,26,90},{24,0,90},{24,0,90},{24,0,90},{24,0,90},{23,27,1},{23,27,1},{23,27,1},{23,25,1},{21,26,10},{21,26,10},{25,31,180},{25,30,94},{25,28,115},{25,28,99},{24,31,265},{24,29,77},{24,28,2},
{24,27,116},{22,29,253},{22,27,100},{26,30,97},{25,30,13},{26,28,25},{25,28,18},{28,24,243},{24,29,76},{24,28,1},{22,27,91},{24,28,243},{22,27,91},{25,30,90},{25,30,90},{25,30,90},{25,27,99},{24,30,51},{24,28,2},{24,28,2},{24,26,27},{22,28,50},{22,27,19},{25,30,9},{25,30,9},{25,30,9},{25,27,18},{31,16,50},{24,28,1},{24,28,1},{22,27,10},{30,24,50},
{22,27,10},{31,20,72},{25,30,4},{26,28,0},{24,28,1},{31,20,72},{30,26,72},{24,28,1},{0,27,90},{30,26,72},{0,27,90},{25,0,90},{25,0,90},{25,0,90},{25,0,90},{24,28,1},{24,28,1},{24,28,1},{24,26,2},{22,27,10},{22,27,10},{26,31,234},{26,31,94},{26,29,115},{26,29,99},{26,31,325},{25,30,77},{25,29,2},{25,28,106},{23,30,253},{23,28,84},{27,31,97},
{26,31,13},{27,29,25},{26,29,18},{29,25,243},{25,30,76},{25,29,1},{23,28,80},{27,28,243},{23,28,80},{26,31,90},{26,31,90},{26,31,90},{26,28,94},{25,31,51},{25,29,2},{25,29,2},{25,27,27},{23,29,50},{23,28,20},{26,31,9},{26,31,9},{26,31,9},{26,28,13},{31,19,50},{25,29,1},{25,29,1},{23,28,16},{31,25,50},{23,28,16},{31,23,72},{26,31,4},{27,29,0},
{25,29,1},{31,23,72},{31,27,72},{25,29,1},{0,28,80},{31,27,72},{0,28,80},{26,0,90},{26,0,90},{26,0,90},{26,0,90},{25,29,1},{25,29,1},{25,29,1},{25,27,2},{23,28,4},{23,28,4},{27,31,314},{27,31,105},{27,30,117},{27,30,85},{27,31,347},{26,31,75},{26,30,2},{26,29,106},{24,31,253},{24,29,100},{28,31,115},{28,31,35},{28,30,26},{27,30,21},{31,24,244},
{26,31,75},{26,30,2},{24,29,91},{30,28,244},{24,29,91},{27,31,89},{27,31,89},{27,31,89},{27,30,84},{26,31,82},{26,30,1},{26,30,1},{26,29,25},{24,30,52},{24,29,19},{28,30,25},{28,30,25},{28,30,25},{27,30,20},{28,28,50},{26,30,1},{26,30,1},{24,29,10},{28,28,50},{24,29,10},{31,26,74},{28,31,10},{28,30,1},{26,30,1},{31,26,74},{30,29,74},{26,30,1},
{0,29,90},{30,29,74},{0,29,90},{27,0,80},{27,0,80},{27,0,80},{27,0,80},{26,30,1},{26,30,1},{26,30,1},{26,28,1},{24,29,10},{24,29,10},{28,31,371},{28,31,179},{28,31,115},{28,31,99},{28,31,387},{27,31,122},{27,31,1},{27,30,89},{26,31,279},{25,30,83},{29,31,146},{29,31,61},{29,31,25},{28,31,18},{31,27,221},{28,31,98},{27,31,1},{25,30,74},{31,29,221},
{25,30,74},{28,31,115},{28,31,115},{28,31,115},{28,30,99},{27,31,122},{27,31,1},{27,31,1},{27,30,25},{25,31,52},{25,30,19},{29,31,25},{29,31,25},{29,31,25},{28,30,18},{29,29,50},{27,31,1},{27,31,1},{25,30,10},{31,28,50},{25,30,10},{31,29,61},{30,31,34},{29,31,0},{27,31,0},{31,29,61},{31,30,61},{27,31,0},{0,30,73},{31,30,61},{0,30,73},{28,0,90},
{28,0,90},{28,0,90},{28,0,90},{27,31,1},{27,31,1},{27,31,1},{27,29,1},{25,30,10},{25,30,10},{29,31,275},{29,31,190},{29,31,154},{29,31,99},{29,31,270},{28,31,99},{28,31,35},{28,31,35},{28,31,195},{26,31,19},{30,31,70},{30,31,43},{30,31,34},{29,31,18},{31,29,94},{29,31,49},{29,31,13},{26,31,10},{31,30,94},{26,31,10},{29,31,154},{29,31,154},{29,31,154},
{29,31,99},{29,31,149},{28,31,35},{28,31,35},{28,30,27},{27,31,77},{26,31,19},{30,31,34},{30,31,34},{30,31,34},{29,31,18},{31,28,50},{29,31,13},{29,31,13},{26,31,10},{30,30,50},{26,31,10},{31,30,9},{31,31,9},{30,31,9},{30,31,0},{31,30,9},{30,31,9},{30,31,0},{0,31,9},{30,31,9},{0,31,9},{29,0,90},{29,0,90},{29,0,90},{29,0,90},{28,31,10},
{28,31,10},{28,31,10},{28,30,2},{26,31,10},{26,31,10},{30,31,162},{30,31,135},{30,31,126},{30,31,99},{30,31,154},{30,31,100},{29,31,73},{29,31,2},{29,31,109},{28,31,10},{31,31,25},{31,31,25},{31,31,25},{30,31,18},{31,30,22},{30,31,19},{30,31,10},{29,31,1},{30,31,22},{29,31,1},{30,31,126},{30,31,126},{30,31,126},{30,31,99},{30,31,118},{29,31,73},{29,31,73},
{29,31,2},{29,31,73},{28,31,10},{31,31,25},{31,31,25},{31,31,25},{30,31,18},{31,30,13},{30,31,10},{30,31,10},{29,31,1},{30,31,13},{29,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{30,0,90},{30,0,90},{30,0,90},{30,0,90},{29,31,37},{29,31,37},{29,31,37},{29,31,2},{28,31,10},
{28,31,10},{0,7,200},{0,5,20},{0,4,20},{0,3,74},{0,5,441},{0,3,282},{0,3,138},{0,2,318},{0,2,487},{0,2,343},{0,7,200},{0,5,20},{0,4,20},{0,3,74},{2,1,441},{0,3,282},{0,3,138},{0,2,318},{5,0,441},{0,2,318},{0,3,1},{0,3,1},{0,3,1},{0,2,1},{0,2,41},{0,1,20},{0,1,20},{0,1,26},{0,1,45},{0,1,30},{0,3,1},
{0,3,1},{0,3,1},{0,2,1},{1,0,41},{0,1,20},{0,1,20},{0,1,26},{2,0,41},{0,1,26},{1,5,200},{0,5,20},{0,4,20},{0,3,74},{1,5,200},{7,0,200},{0,3,74},{0,2,218},{7,0,200},{0,2,218},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,10,200},{0,7,5},{0,5,32},
{0,4,41},{0,7,689},{0,5,369},{0,4,162},{0,3,474},{0,3,762},{0,3,538},{0,10,200},{0,7,5},{0,5,32},{0,4,41},{3,1,686},{0,5,369},{0,4,162},{0,3,474},{3,2,686},{0,3,474},{0,6,1},{0,6,1},{0,6,1},{0,3,1},{0,3,145},{0,3,65},{0,3,65},{0,2,101},{0,1,173},{0,1,110},{0,6,1},{0,6,1},{0,6,1},{0,3,1},{1,1,145},
{0,3,65},{0,3,65},{0,2,101},{3,0,145},{0,2,101},{5,0,200},{0,7,5},{1,4,17},{0,4,41},{5,0,200},{10,0,200},{0,4,41},{0,3,218},{10,0,200},{0,3,218},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,13,249},{0,8,59},{1,6,97},{0,5,69},{0,10,728},{0,6,299},{0,5,56},
{0,4,433},{0,4,884},{0,4,554},{1,11,201},{1,8,6},{1,6,33},{1,5,42},{5,0,728},{0,6,299},{0,5,56},{0,4,433},{10,0,728},{0,4,433},{0,9,50},{0,9,50},{0,9,50},{0,5,53},{0,6,162},{0,4,25},{0,4,25},{0,3,82},{0,3,226},{0,2,115},{1,7,2},{1,7,2},{1,7,2},{1,4,2},{3,0,162},{0,4,25},{0,4,25},{0,3,82},{6,0,162},
{0,3,82},{6,1,200},{1,8,5},{2,5,17},{0,5,20},{6,1,200},{13,0,200},{0,5,20},{0,4,208},{13,0,200},{0,4,208},{0,0,49},{0,0,49},{0,0,49},{0,0,49},{0,3,1},{0,3,1},{0,3,1},{0,1,4},{0,1,8},{0,1,8},{1,14,313},{1,9,123},{1,7,210},{1,6,133},{0,12,724},{0,8,251},{0,6,18},{0,5,352},{0,6,987},{0,5,521},{2,12,201},
{2,9,6},{2,7,33},{2,6,42},{4,4,724},{0,8,251},{0,6,18},{0,5,352},{12,0,724},{0,5,352},{1,10,114},{1,10,114},{1,10,114},{1,6,117},{0,9,162},{0,6,2},{0,6,2},{0,4,49},{0,4,291},{0,3,146},{2,8,2},{2,8,2},{2,8,2},{2,5,2},{4,1,162},{0,6,2},{0,6,2},{0,4,49},{9,0,162},{0,4,49},{7,2,200},{2,9,5},{3,6,17},
{0,6,17},{7,2,200},{14,1,200},{0,6,17},{0,5,208},{14,1,200},{0,5,208},{1,0,113},{1,0,113},{1,0,113},{1,0,113},{0,6,1},{0,6,1},{0,6,1},{0,3,1},{0,2,50},{0,2,50},{2,15,410},{2,11,221},{2,8,324},{2,7,234},{1,14,739},{0,10,233},{0,8,40},{0,6,298},{0,7,1013},{0,6,426},{3,13,202},{3,10,1},{3,8,32},{3,7,41},{7,2,723},
{0,10,217},{0,8,24},{0,6,282},{14,1,723},{0,6,282},{2,12,209},{2,12,209},{2,12,209},{2,7,209},{0,12,178},{1,7,20},{1,7,20},{0,5,26},{0,6,308},{0,5,91},{3,10,1},{3,10,1},{3,10,1},{3,6,4},{4,4,162},{1,7,4},{1,7,4},{0,5,10},{12,0,162},{0,5,10},{9,1,200},{3,10,1},{4,8,20},{0,8,20},{9,1,200},{19,0,200},{0,8,20},
{0,6,218},{19,0,200},{0,6,218},{2,0,208},{2,0,208},{2,0,208},{2,0,208},{1,7,16},{1,7,16},{1,7,16},{1,4,16},{0,4,53},{0,4,53},{3,16,408},{3,12,225},{3,9,324},{3,8,228},{2,15,739},{1,11,233},{2,8,33},{1,7,298},{0,9,875},{0,7,303},{4,14,201},{4,11,6},{4,9,33},{4,8,42},{7,5,723},{0,11,204},{2,8,17},{0,7,254},{15,2,723},
{0,7,254},{3,12,209},{3,12,209},{3,12,209},{3,8,212},{1,13,178},{2,8,17},{2,8,17},{1,6,26},{0,7,229},{0,6,17},{4,10,2},{4,10,2},{4,10,2},{4,7,2},{5,5,162},{2,8,1},{2,8,1},{0,6,1},{15,0,162},{0,6,1},{11,0,200},{4,11,5},{5,8,17},{2,8,17},{11,0,200},{22,0,200},{2,8,17},{0,7,218},{22,0,200},{0,7,218},{3,0,208},
{3,0,208},{3,0,208},{3,0,208},{2,8,16},{2,8,16},{2,8,16},{2,5,16},{0,6,16},{0,6,16},{4,17,418},{4,12,228},{4,10,315},{4,9,238},{3,16,744},{2,11,234},{3,9,33},{2,8,305},{0,11,828},{0,8,228},{5,15,201},{5,12,6},{5,10,33},{5,9,42},{11,0,728},{1,12,204},{3,9,17},{0,8,224},{22,0,728},{0,8,224},{4,13,219},{4,13,219},{4,13,219},
{4,9,222},{3,12,178},{3,9,17},{3,9,17},{2,7,26},{0,9,178},{0,7,6},{5,11,2},{5,11,2},{5,11,2},{5,8,2},{9,0,162},{3,9,1},{3,9,1},{1,7,1},{18,0,162},{1,7,1},{12,1,200},{5,12,5},{6,9,17},{3,9,17},{12,1,200},{25,0,200},{3,9,17},{0,8,208},{25,0,200},{0,8,208},{4,0,218},{4,0,218},{4,0,218},{4,0,218},{3,9,16},
{3,9,16},{3,9,16},{3,6,16},{0,7,5},{0,7,5},{5,18,418},{5,13,228},{5,11,315},{5,10,238},{3,19,744},{3,12,225},{3,11,40},{3,9,305},{0,12,749},{0,9,225},{6,16,201},{6,13,6},{6,11,33},{6,10,42},{12,0,724},{2,13,204},{4,10,18},{0,9,209},{24,0,724},{0,9,209},{5,14,219},{5,14,219},{5,14,219},{5,10,222},{3,15,178},{3,11,24},{3,11,24},
{3,8,32},{1,10,178},{1,8,5},{6,12,2},{6,12,2},{6,12,2},{6,9,2},{10,1,162},{4,10,2},{4,10,2},{2,8,1},{21,0,162},{2,8,1},{14,0,200},{6,13,5},{7,10,17},{4,10,17},{14,0,200},{26,1,200},{4,10,17},{0,9,208},{26,1,200},{0,9,208},{5,0,218},{5,0,218},{5,0,218},{5,0,218},{3,12,17},{3,12,17},{3,12,17},{3,7,20},{1,8,5},
{1,8,5},{6,19,410},{6,15,221},{6,12,324},{6,11,234},{5,18,739},{4,14,233},{4,12,40},{4,10,298},{0,14,724},{2,10,228},{7,17,202},{7,14,1},{7,12,32},{7,11,41},{14,0,723},{3,14,211},{4,12,24},{1,10,218},{26,1,723},{1,10,218},{6,16,209},{6,16,209},{6,16,209},{6,11,209},{4,16,178},{5,11,20},{5,11,20},{4,9,26},{1,12,171},{3,9,10},{7,14,1},
{7,14,1},{7,14,1},{7,10,4},{12,0,162},{5,11,4},{5,11,4},{3,9,1},{24,0,162},{3,9,1},{13,5,200},{7,14,1},{8,12,20},{3,12,17},{13,5,200},{31,0,200},{3,12,17},{0,10,218},{31,0,200},{0,10,218},{6,0,208},{6,0,208},{6,0,208},{6,0,208},{5,11,16},{5,11,16},{5,11,16},{5,8,16},{3,9,9},{3,9,9},{7,20,408},{7,16,225},{7,13,324},
{7,12,228},{6,19,739},{5,15,233},{6,12,33},{5,11,298},{1,15,724},{3,11,228},{8,18,201},{8,15,6},{8,13,33},{8,12,42},{15,1,723},{4,15,204},{6,12,17},{2,11,218},{27,2,723},{2,11,218},{7,16,209},{7,16,209},{7,16,209},{7,12,212},{5,17,178},{6,12,17},{6,12,17},{5,10,26},{2,13,171},{3,10,14},{8,14,2},{8,14,2},{8,14,2},{8,11,2},{13,1,162},
{6,12,1},{6,12,1},{4,10,1},{27,0,162},{4,10,1},{17,0,200},{8,15,5},{9,12,17},{6,12,17},{17,0,200},{30,2,200},{6,12,17},{0,11,218},{30,2,200},{0,11,218},{7,0,208},{7,0,208},{7,0,208},{7,0,208},{6,12,16},{6,12,16},{6,12,16},{6,9,16},{3,11,10},{3,11,10},{8,21,418},{8,16,228},{8,14,315},{8,13,238},{7,20,744},{6,15,234},{7,13,33},
{6,12,305},{2,16,729},{3,12,225},{9,19,201},{9,16,6},{9,14,33},{9,13,42},{17,0,728},{5,16,204},{7,13,17},{3,12,209},{30,2,728},{3,12,209},{8,17,219},{8,17,219},{8,17,219},{8,13,222},{7,16,178},{7,13,17},{7,13,17},{6,11,26},{3,14,171},{4,11,6},{9,15,2},{9,15,2},{9,15,2},{9,12,2},{15,0,162},{7,13,1},{7,13,1},{5,11,1},{30,0,162},
{5,11,1},{18,1,200},{9,16,5},{10,13,17},{7,13,17},{18,1,200},{31,3,200},{7,13,17},{0,12,208},{31,3,200},{0,12,208},{8,0,218},{8,0,218},{8,0,218},{8,0,218},{7,13,16},{7,13,16},{7,13,16},{7,10,16},{4,11,5},{4,11,5},{9,22,418},{9,17,228},{9,15,315},{9,14,238},{7,23,744},{7,16,225},{7,15,40},{7,13,305},{3,17,729},{4,13,225},{10,20,201},
{10,17,6},{10,15,33},{10,14,42},{16,4,724},{6,17,204},{8,14,18},{4,13,209},{28,4,724},{4,13,209},{9,18,219},{9,18,219},{9,18,219},{9,14,222},{7,19,178},{7,15,24},{7,15,24},{7,12,32},{5,14,178},{5,12,5},{10,16,2},{10,16,2},{10,16,2},{10,13,2},{16,1,162},{8,14,2},{8,14,2},{6,12,1},{31,1,162},{6,12,1},{19,2,200},{10,17,5},{11,14,17},
{8,14,17},{19,2,200},{30,5,200},{8,14,17},{0,13,208},{30,5,200},{0,13,208},{9,0,218},{9,0,218},{9,0,218},{9,0,218},{7,16,17},{7,16,17},{7,16,17},{7,11,20},{5,12,5},{5,12,5},{10,23,410},{10,19,221},{10,16,324},{10,15,234},{9,22,739},{8,18,233},{8,16,40},{8,14,298},{4,18,724},{6,14,228},{11,21,202},{11,18,1},{11,16,32},{11,15,41},{19,2,723},
{7,18,211},{8,16,24},{5,14,218},{30,5,723},{5,14,218},{10,20,209},{10,20,209},{10,20,209},{10,15,209},{8,20,178},{9,15,20},{9,15,20},{8,13,26},{5,16,171},{7,13,10},{11,18,1},{11,18,1},{11,18,1},{11,14,4},{16,4,162},{9,15,4},{9,15,4},{7,13,1},{28,4,162},{7,13,1},{21,1,200},{11,18,1},{12,16,20},{7,16,17},{21,1,200},{27,8,200},{7,16,17},
{0,14,218},{27,8,200},{0,14,218},{10,0,208},{10,0,208},{10,0,208},{10,0,208},{9,15,16},{9,15,16},{9,15,16},{9,12,16},{7,13,9},{7,13,9},{11,24,408},{11,20,225},{11,17,324},{11,16,228},{10,23,739},{9,19,233},{10,16,33},{9,15,298},{5,19,724},{7,15,228},{12,22,201},{12,19,6},{12,17,33},{12,16,42},{19,5,723},{8,19,204},{10,16,17},{6,15,218},{31,6,723},
{6,15,218},{11,20,209},{11,20,209},{11,20,209},{11,16,212},{9,21,178},{10,16,17},{10,16,17},{9,14,26},{6,17,171},{7,14,14},{12,18,2},{12,18,2},{12,18,2},{12,15,2},{17,5,162},{10,16,1},{10,16,1},{8,14,1},{31,4,162},{8,14,1},{23,0,200},{12,19,5},{13,16,17},{10,16,17},{23,0,200},{30,8,200},{10,16,17},{0,15,218},{30,8,200},{0,15,218},{11,0,208},
{11,0,208},{11,0,208},{11,0,208},{10,16,16},{10,16,16},{10,16,16},{10,13,16},{7,15,10},{7,15,10},{12,25,418},{12,20,228},{12,18,315},{12,17,238},{11,24,744},{10,19,234},{11,17,33},{10,16,305},{6,20,729},{7,16,225},{13,23,201},{13,20,6},{13,18,33},{13,17,42},{23,0,728},{9,20,204},{11,17,17},{7,16,209},{30,8,728},{7,16,209},{12,21,219},{12,21,219},{12,21,219},
{12,17,222},{11,20,178},{11,17,17},{11,17,17},{10,15,26},{7,18,171},{8,15,6},{13,19,2},{13,19,2},{13,19,2},{13,16,2},{21,0,162},{11,17,1},{11,17,1},{9,15,1},{30,6,162},{9,15,1},{24,1,200},{13,20,5},{14,17,17},{11,17,17},{24,1,200},{31,9,200},{11,17,17},{0,16,208},{31,9,200},{0,16,208},{12,0,218},{12,0,218},{12,0,218},{12,0,218},{11,17,16},
{11,17,16},{11,17,16},{11,14,16},{8,15,5},{8,15,5},{13,26,418},{13,21,228},{13,19,315},{13,18,238},{11,27,744},{11,20,225},{11,19,40},{11,17,305},{7,21,729},{8,17,225},{14,24,201},{14,21,6},{14,19,33},{14,18,42},{24,0,724},{10,21,204},{12,18,18},{8,17,209},{24,12,724},{8,17,209},{13,22,219},{13,22,219},{13,22,219},{13,18,222},{11,23,178},{11,19,24},{11,19,24},
{11,16,32},{9,18,178},{9,16,5},{14,20,2},{14,20,2},{14,20,2},{14,17,2},{22,1,162},{12,18,2},{12,18,2},{10,16,1},{31,7,162},{10,16,1},{26,0,200},{14,21,5},{15,18,17},{12,18,17},{26,0,200},{30,11,200},{12,18,17},{0,17,208},{30,11,200},{0,17,208},{13,0,218},{13,0,218},{13,0,218},{13,0,218},{11,20,17},{11,20,17},{11,20,17},{11,15,20},{9,16,5},
{9,16,5},{14,27,410},{14,23,221},{14,20,324},{14,19,234},{13,26,739},{12,22,233},{12,20,40},{12,18,298},{8,22,724},{10,18,228},{15,25,202},{15,22,1},{15,20,32},{15,19,41},{26,0,723},{11,22,211},{12,20,24},{9,18,218},{30,11,723},{9,18,218},{14,24,209},{14,24,209},{14,24,209},{14,19,209},{12,24,178},{13,19,20},{13,19,20},{12,17,26},{9,20,171},{11,17,10},{15,22,1},
{15,22,1},{15,22,1},{15,18,4},{24,0,162},{13,19,4},{13,19,4},{11,17,1},{24,12,162},{11,17,1},{25,5,200},{15,22,1},{16,20,20},{11,20,17},{25,5,200},{31,12,200},{11,20,17},{0,18,218},{31,12,200},{0,18,218},{14,0,208},{14,0,208},{14,0,208},{14,0,208},{13,19,16},{13,19,16},{13,19,16},{13,16,16},{11,17,9},{11,17,9},{15,28,408},{15,24,225},{15,21,324},
{15,20,228},{14,27,739},{13,23,233},{14,20,33},{13,19,298},{9,23,724},{11,19,228},{16,26,201},{16,23,6},{16,21,33},{16,20,42},{27,1,723},{12,23,204},{14,20,17},{10,19,218},{27,14,723},{10,19,218},{15,24,209},{15,24,209},{15,24,209},{15,20,212},{13,25,178},{14,20,17},{14,20,17},{13,18,26},{10,21,171},{11,18,14},{16,22,2},{16,22,2},{16,22,2},{16,19,2},{25,1,162},
{14,20,1},{14,20,1},{12,18,1},{27,12,162},{12,18,1},{29,0,200},{16,23,5},{17,20,17},{14,20,17},{29,0,200},{30,14,200},{14,20,17},{0,19,218},{30,14,200},{0,19,218},{15,0,208},{15,0,208},{15,0,208},{15,0,208},{14,20,16},{14,20,16},{14,20,16},{14,17,16},{11,19,10},{11,19,10},{16,29,418},{16,24,228},{16,22,315},{16,21,238},{15,28,744},{14,23,234},{15,21,33},
{14,20,305},{10,24,729},{11,20,225},{17,27,201},{17,24,6},{17,22,33},{17,21,42},{29,0,728},{13,24,204},{15,21,17},{11,20,209},{30,14,728},{11,20,209},{16,25,219},{16,25,219},{16,25,219},{16,21,222},{15,24,178},{15,21,17},{15,21,17},{14,19,26},{11,22,171},{12,19,6},{17,23,2},{17,23,2},{17,23,2},{17,20,2},{27,0,162},{15,21,1},{15,21,1},{13,19,1},{30,12,162},
{13,19,1},{30,1,200},{17,24,5},{18,21,17},{15,21,17},{30,1,200},{31,15,200},{15,21,17},{0,20,208},{31,15,200},{0,20,208},{16,0,218},{16,0,218},{16,0,218},{16,0,218},{15,21,16},{15,21,16},{15,21,16},{15,18,16},{12,19,5},{12,19,5},{17,30,418},{17,25,228},{17,23,315},{17,22,238},{15,31,744},{15,24,225},{15,23,40},{15,21,305},{11,25,729},{12,21,225},{18,28,201},
{18,25,6},{18,23,33},{18,22,42},{28,4,724},{14,25,204},{16,22,18},{12,21,209},{28,16,724},{12,21,209},{17,26,219},{17,26,219},{17,26,219},{17,22,222},{15,27,178},{15,23,24},{15,23,24},{15,20,32},{13,22,178},{13,20,5},{18,24,2},{18,24,2},{18,24,2},{18,21,2},{28,1,162},{16,22,2},{16,22,2},{14,20,1},{31,13,162},{14,20,1},{31,2,200},{18,25,5},{19,22,17},
{16,22,17},{31,2,200},{30,17,200},{16,22,17},{0,21,208},{30,17,200},{0,21,208},{17,0,218},{17,0,218},{17,0,218},{17,0,218},{15,24,17},{15,24,17},{15,24,17},{15,19,20},{13,20,5},{13,20,5},{18,31,410},{18,27,221},{18,24,324},{18,23,234},{17,30,739},{16,26,233},{16,24,40},{16,22,298},{12,26,724},{14,22,228},{19,29,202},{19,26,1},{19,24,32},{19,23,41},{31,2,723},
{15,26,211},{16,24,24},{13,22,218},{30,17,723},{13,22,218},{18,28,209},{18,28,209},{18,28,209},{18,23,209},{16,28,178},{17,23,20},{17,23,20},{16,21,26},{13,24,171},{15,21,10},{19,26,1},{19,26,1},{19,26,1},{19,22,4},{28,4,162},{17,23,4},{17,23,4},{15,21,1},{28,16,162},{15,21,1},{29,9,200},{19,26,1},{20,24,20},{15,24,17},{29,9,200},{27,20,200},{15,24,17},
{0,22,218},{27,20,200},{0,22,218},{18,0,208},{18,0,208},{18,0,208},{18,0,208},{17,23,16},{17,23,16},{17,23,16},{17,20,16},{15,21,9},{15,21,9},{19,31,426},{19,28,225},{19,25,324},{19,24,228},{18,31,739},{17,27,233},{18,24,33},{17,23,298},{13,27,724},{15,23,228},{20,30,201},{20,27,6},{20,25,33},{20,24,42},{31,5,723},{16,27,204},{18,24,17},{14,23,218},{31,18,723},
{14,23,218},{19,28,209},{19,28,209},{19,28,209},{19,24,212},{17,29,178},{18,24,17},{18,24,17},{17,22,26},{14,25,171},{15,22,14},{20,26,2},{20,26,2},{20,26,2},{20,23,2},{29,5,162},{18,24,1},{18,24,1},{16,22,1},{31,16,162},{16,22,1},{31,8,200},{20,27,5},{21,24,17},{18,24,17},{31,8,200},{30,20,200},{18,24,17},{0,23,218},{30,20,200},{0,23,218},{19,0,208},
{19,0,208},{19,0,208},{19,0,208},{18,24,16},{18,24,16},{18,24,16},{18,21,16},{15,23,10},{15,23,10},{20,31,468},{20,28,228},{20,26,315},{20,25,238},{19,31,747},{18,27,234},{19,25,33},{18,24,305},{14,28,729},{15,24,225},{21,31,201},{21,28,6},{21,26,33},{21,25,42},{31,8,728},{17,28,204},{19,25,17},{15,24,209},{30,20,728},{15,24,209},{20,29,219},{20,29,219},{20,29,219},
{20,25,222},{19,28,178},{19,25,17},{19,25,17},{18,23,26},{15,26,171},{16,23,6},{21,27,2},{21,27,2},{21,27,2},{21,24,2},{31,4,162},{19,25,1},{19,25,1},{17,23,1},{30,18,162},{17,23,1},{31,11,200},{21,28,5},{22,25,17},{19,25,17},{31,11,200},{31,21,200},{19,25,17},{0,24,208},{31,21,200},{0,24,208},{20,0,218},{20,0,218},{20,0,218},{20,0,218},{19,25,16},
{19,25,16},{19,25,16},{19,22,16},{16,23,5},{16,23,5},{21,31,546},{21,29,228},{21,27,315},{21,26,238},{20,31,788},{19,28,225},{19,27,40},{19,25,305},{15,29,729},{16,25,225},{22,31,219},{22,29,6},{22,27,33},{22,26,42},{28,16,724},{18,29,204},{20,26,18},{16,25,209},{24,24,724},{16,25,209},{21,30,219},{21,30,219},{21,30,219},{21,26,222},{19,31,178},{19,27,24},{19,27,24},
{19,24,32},{17,26,178},{17,24,5},{22,28,2},{22,28,2},{22,28,2},{22,25,2},{31,7,162},{20,26,2},{20,26,2},{18,24,1},{31,19,162},{18,24,1},{31,14,200},{22,29,5},{23,26,17},{20,26,17},{31,14,200},{30,23,200},{20,26,17},{0,25,208},{30,23,200},{0,25,208},{21,0,218},{21,0,218},{21,0,218},{21,0,218},{19,28,17},{19,28,17},{19,28,17},{19,23,20},{17,24,5},
{17,24,5},{23,31,672},{22,31,221},{22,28,324},{22,27,234},{22,31,888},{20,30,233},{20,28,40},{20,26,298},{16,30,724},{18,26,228},{23,31,272},{23,30,1},{23,28,32},{23,27,41},{31,14,723},{19,30,211},{20,28,24},{17,26,218},{30,23,723},{17,26,218},{22,31,212},{22,31,212},{22,31,212},{22,27,209},{21,30,180},{21,27,20},{21,27,20},{20,25,26},{17,28,171},{19,25,10},{23,30,1},
{23,30,1},{23,30,1},{23,26,4},{28,16,162},{21,27,4},{21,27,4},{19,25,1},{24,24,162},{19,25,1},{29,21,200},{23,30,1},{24,28,20},{19,28,17},{29,21,200},{31,24,200},{19,28,17},{0,26,218},{31,24,200},{0,26,218},{22,0,208},{22,0,208},{22,0,208},{22,0,208},{21,27,16},{21,27,16},{21,27,16},{21,24,16},{19,25,9},{19,25,9},{24,31,770},{23,31,228},{23,29,324},
{23,28,228},{23,31,932},{21,31,233},{22,28,33},{21,27,298},{17,31,724},{19,27,228},{25,31,299},{24,31,6},{24,29,33},{24,28,42},{31,17,723},{20,31,204},{22,28,17},{18,27,218},{27,26,723},{18,27,218},{23,31,224},{23,31,224},{23,31,224},{23,28,212},{22,31,180},{22,28,17},{22,28,17},{21,26,26},{18,29,171},{19,26,14},{24,30,2},{24,30,2},{24,30,2},{24,27,2},{29,17,162},
{22,28,1},{22,28,1},{20,26,1},{27,24,162},{20,26,1},{31,20,200},{24,31,5},{25,28,17},{22,28,17},{31,20,200},{30,26,200},{22,28,17},{0,27,218},{30,26,200},{0,27,218},{23,0,208},{23,0,208},{23,0,208},{23,0,208},{22,28,16},{22,28,16},{22,28,16},{22,25,16},{19,27,10},{19,27,10},{25,31,884},{24,31,303},{24,30,315},{24,29,238},{24,31,1025},{22,31,234},{23,29,33},
{22,28,305},{19,31,747},{19,28,225},{26,31,353},{25,31,27},{25,30,33},{25,29,42},{31,20,728},{22,31,218},{23,29,17},{19,28,209},{30,26,728},{19,28,209},{24,31,254},{24,31,254},{24,31,254},{24,29,222},{23,31,196},{23,29,17},{23,29,17},{22,27,26},{19,30,171},{20,27,6},{25,31,2},{25,31,2},{25,31,2},{25,28,2},{31,16,162},{23,29,1},{23,29,1},{21,27,1},{30,24,162},
{21,27,1},{31,23,200},{26,31,20},{26,29,17},{23,29,17},{31,23,200},{31,27,200},{23,29,17},{0,28,208},{31,27,200},{0,28,208},{24,0,218},{24,0,218},{24,0,218},{24,0,218},{23,29,16},{23,29,16},{23,29,16},{23,26,16},{20,27,5},{20,27,5},{26,31,1034},{25,31,468},{25,31,315},{25,30,238},{25,31,1172},{24,31,291},{23,31,40},{23,29,305},{21,31,837},{20,29,225},{27,31,409},
{26,31,117},{26,31,33},{26,30,42},{28,28,724},{24,31,266},{24,30,18},{20,29,209},{28,28,724},{20,29,209},{25,31,299},{25,31,299},{25,31,299},{25,30,222},{24,31,237},{23,31,24},{23,31,24},{23,28,32},{21,30,178},{21,28,5},{26,31,17},{26,31,17},{26,31,17},{26,29,2},{31,19,162},{24,30,2},{24,30,2},{22,28,1},{31,25,162},{22,28,1},{31,26,200},{27,31,53},{27,30,17},
{24,30,17},{31,26,200},{30,29,200},{24,30,17},{0,29,208},{30,29,200},{0,29,208},{25,0,218},{25,0,218},{25,0,218},{25,0,218},{23,31,20},{23,31,20},{23,31,20},{23,27,20},{21,28,5},{21,28,5},{27,31,933},{27,31,570},{26,31,377},{26,31,209},{27,31,1054},{25,31,309},{25,31,20},{24,30,193},{23,31,735},{22,30,123},{28,31,338},{28,31,146},{27,31,49},{27,31,16},{29,29,546},
{26,31,222},{25,31,4},{21,30,113},{31,28,546},{21,30,113},{26,31,377},{26,31,377},{26,31,377},{26,31,209},{26,31,338},{25,31,20},{25,31,20},{24,29,26},{22,31,173},{23,29,10},{27,31,49},{27,31,49},{27,31,49},{27,30,4},{28,28,162},{25,31,4},{25,31,4},{23,29,1},{28,28,162},{23,29,1},{31,28,113},{29,31,52},{28,31,1},{26,31,1},{31,28,113},{30,30,113},{26,31,1},
{0,30,113},{30,30,113},{0,30,113},{26,0,208},{26,0,208},{26,0,208},{26,0,208},{25,31,16},{25,31,16},{25,31,16},{25,28,16},{23,29,9},{23,29,9},{28,31,779},{27,31,554},{27,31,433},{27,31,224},{28,31,859},{26,31,270},{26,31,74},{25,30,90},{24,31,590},{23,31,59},{29,31,218},{29,31,133},{28,31,82},{28,31,2},{31,27,333},{27,31,146},{27,31,25},{22,31,49},{31,29,333},
{22,31,49},{27,31,433},{27,31,433},{27,31,433},{27,31,224},{27,31,378},{26,31,74},{26,31,74},{25,30,26},{24,31,229},{23,30,14},{28,31,82},{28,31,82},{28,31,82},{28,31,2},{29,29,162},{27,31,25},{27,31,25},{24,30,1},{31,28,162},{24,30,1},{31,30,25},{30,31,10},{30,31,1},{28,31,1},{31,30,25},{30,31,25},{28,31,1},{0,31,49},{30,31,25},{0,31,49},{27,0,208},
{27,0,208},{27,0,208},{27,0,208},{26,31,25},{26,31,25},{26,31,25},{26,29,16},{23,31,10},{23,31,10},{29,31,684},{28,31,538},{28,31,474},{28,31,282},{28,31,682},{27,31,283},{27,31,162},{26,31,25},{26,31,482},{24,31,5},{30,31,153},{30,31,126},{30,31,117},{29,31,37},{31,29,193},{29,31,108},{28,31,65},{25,31,0},{31,30,193},{25,31,0},{28,31,474},{28,31,474},{28,31,474},
{28,31,282},{28,31,426},{27,31,162},{27,31,162},{26,31,25},{26,31,286},{24,31,5},{30,31,117},{30,31,117},{30,31,117},{29,31,37},{31,28,145},{28,31,65},{28,31,65},{25,31,0},{30,30,145},{25,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{28,0,218},{28,0,218},{28,0,218},{28,0,218},{27,31,41},
{27,31,41},{27,31,41},{27,30,16},{24,31,5},{24,31,5},{29,31,460},{29,31,375},{29,31,339},{29,31,254},{29,31,415},{28,31,202},{28,31,138},{27,31,20},{27,31,295},{26,31,26},{30,31,73},{30,31,46},{30,31,37},{30,31,10},{31,30,54},{30,31,27},{30,31,18},{28,31,1},{30,31,54},{28,31,1},{29,31,339},{29,31,339},{29,31,339},{29,31,254},{29,31,294},{28,31,138},{28,31,138},
{27,31,20},{27,31,174},{26,31,26},{30,31,37},{30,31,37},{30,31,37},{30,31,10},{31,29,45},{30,31,18},{30,31,18},{28,31,1},{31,30,45},{28,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{29,0,218},{29,0,218},{29,0,218},{29,0,218},{28,31,74},{28,31,74},{28,31,74},{27,31,20},{26,31,26},
{26,31,26},{0,10,421},{0,7,52},{0,5,1},{0,4,162},{0,7,926},{0,5,590},{0,4,283},{0,3,701},{0,3,1005},{0,3,765},{0,10,421},{0,7,52},{0,5,1},{0,4,162},{3,1,925},{0,5,590},{0,4,283},{0,3,701},{3,2,925},{0,3,701},{0,5,0},{0,5,0},{0,5,0},{0,2,9},{0,2,89},{0,2,34},{0,2,34},{0,1,50},{0,1,93},{0,1,54},{0,5,0},
{0,5,0},{0,5,0},{0,2,9},{1,0,89},{0,2,34},{0,2,34},{0,1,50},{2,0,89},{0,1,50},{5,0,421},{0,7,52},{0,5,1},{0,4,162},{5,0,421},{10,0,421},{0,4,162},{0,3,445},{10,0,421},{0,3,445},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,13,421},{0,9,10},{0,6,26},
{0,5,117},{0,9,1261},{0,6,701},{0,5,286},{0,4,917},{0,4,1390},{0,3,1005},{0,13,421},{0,9,10},{0,6,26},{0,5,117},{4,1,1261},{0,6,701},{0,5,286},{0,4,917},{9,0,1261},{0,4,917},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{0,4,221},{0,3,89},{0,3,89},{0,2,125},{0,2,246},{0,2,150},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{2,0,221},
{0,3,89},{0,3,89},{0,2,125},{4,0,221},{0,2,125},{6,1,421},{0,9,10},{1,6,1},{0,5,117},{6,1,421},{13,0,421},{0,5,117},{0,4,433},{13,0,421},{0,4,433},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,16,430},{0,11,14},{0,7,110},{0,7,91},{0,11,1514},{0,7,737},{0,6,259},
{0,4,1002},{0,5,1710},{0,4,1123},{0,16,430},{0,11,14},{1,7,51},{0,7,91},{5,1,1514},{0,7,737},{0,6,259},{0,4,1002},{11,0,1514},{0,4,1002},{0,11,10},{0,11,10},{0,11,10},{0,5,13},{0,6,338},{0,5,104},{0,5,104},{0,3,194},{0,3,402},{0,2,243},{0,11,10},{0,11,10},{0,11,10},{0,5,13},{3,0,338},{0,5,104},{0,5,104},{0,3,194},{6,0,338},
{0,3,194},{8,0,421},{0,11,5},{2,7,1},{0,7,82},{8,0,421},{16,0,421},{0,7,82},{0,5,433},{16,0,421},{0,5,433},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,1,1},{0,1,1},{0,1,1},{0,1,4},{0,1,8},{0,1,8},{1,17,494},{1,12,78},{1,8,173},{1,8,154},{0,14,1514},{0,9,602},{0,7,146},{0,6,874},{0,7,1875},{0,5,1066},{1,17,430},
{1,12,14},{2,8,50},{1,8,90},{7,0,1514},{0,9,602},{0,7,146},{0,6,874},{14,0,1514},{0,6,874},{1,12,74},{1,12,74},{1,12,74},{1,6,77},{0,9,338},{0,6,50},{0,6,50},{0,4,137},{0,4,467},{0,3,258},{1,12,10},{1,12,10},{1,12,10},{1,6,13},{4,1,338},{0,6,50},{0,6,50},{0,4,137},{9,0,338},{0,4,137},{9,1,421},{1,12,5},{3,8,1},
{0,8,49},{9,1,421},{19,0,421},{0,8,49},{0,6,433},{19,0,421},{0,6,433},{1,0,73},{1,0,73},{1,0,73},{1,0,73},{0,4,0},{0,4,0},{0,4,0},{0,2,1},{0,2,26},{0,2,26},{1,20,629},{1,14,233},{2,10,385},{1,9,245},{0,17,1517},{0,11,521},{0,9,26},{0,7,769},{0,8,2025},{0,6,1085},{3,16,437},{2,13,17},{3,9,53},{2,9,81},{8,1,1517},
{0,11,521},{0,9,26},{0,7,769},{17,0,1517},{0,7,769},{1,15,208},{1,15,208},{1,15,208},{1,8,208},{0,12,338},{0,8,8},{0,8,8},{0,5,74},{0,5,579},{0,5,243},{3,11,16},{3,11,16},{3,11,16},{3,7,16},{4,4,338},{0,8,8},{0,8,8},{0,5,74},{12,0,338},{0,5,74},{11,0,421},{2,13,1},{4,9,1},{0,9,10},{11,0,421},{22,0,421},{0,9,10},
{0,7,445},{22,0,421},{0,7,445},{1,0,208},{1,0,208},{1,0,208},{1,0,208},{0,7,1},{0,7,1},{0,7,1},{0,4,4},{0,3,80},{0,3,80},{2,21,821},{2,15,425},{2,11,645},{2,10,437},{0,20,1514},{0,12,458},{0,10,2},{0,8,689},{0,9,2198},{0,7,1146},{3,19,437},{3,14,17},{4,10,51},{3,10,81},{8,4,1514},{0,12,458},{0,10,2},{0,8,689},{20,0,1514},
{0,8,689},{2,16,400},{2,16,400},{2,16,400},{2,9,400},{0,15,338},{0,10,1},{0,10,1},{0,6,41},{0,7,717},{0,6,297},{3,14,16},{3,14,16},{3,14,16},{3,8,20},{5,5,338},{0,10,1},{0,10,1},{0,6,41},{15,0,338},{0,6,41},{12,1,421},{3,14,1},{5,10,1},{0,10,1},{12,1,421},{25,0,421},{0,10,1},{0,8,433},{25,0,421},{0,8,433},{2,0,400},
{2,0,400},{2,0,400},{2,0,400},{0,10,1},{0,10,1},{0,10,1},{0,5,1},{0,4,157},{0,4,157},{3,22,854},{3,15,459},{3,12,666},{3,11,470},{1,21,1515},{1,13,459},{1,11,3},{0,9,651},{0,11,2070},{0,9,891},{4,20,430},{4,15,14},{5,11,51},{4,11,91},{9,5,1514},{0,14,425},{1,11,2},{0,9,602},{23,0,1514},{0,9,602},{3,17,433},{3,17,433},{3,17,433},
{3,10,433},{1,16,339},{1,11,2},{1,11,2},{1,7,42},{0,8,613},{0,7,173},{4,15,10},{4,15,10},{4,15,10},{4,9,13},{9,0,338},{1,11,1},{1,11,1},{0,7,29},{18,0,338},{0,7,29},{14,0,421},{4,15,5},{6,11,1},{1,11,1},{14,0,421},{28,0,421},{1,11,1},{0,9,433},{28,0,421},{0,9,433},{3,0,433},{3,0,433},{3,0,433},{3,0,433},{1,11,2},
{1,11,2},{1,11,2},{1,6,2},{0,6,97},{0,6,97},{4,23,866},{4,16,461},{4,12,670},{4,12,494},{2,22,1515},{2,14,459},{2,12,3},{1,10,651},{0,12,1913},{0,10,677},{5,21,430},{5,16,14},{6,12,50},{5,12,90},{13,0,1514},{1,15,425},{2,12,2},{0,10,533},{26,0,1514},{0,10,533},{4,18,446},{4,18,446},{4,18,446},{4,11,446},{2,17,339},{2,12,3},{2,12,3},
{2,8,42},{0,10,500},{0,8,65},{5,16,10},{5,16,10},{5,16,10},{5,10,13},{10,1,338},{2,12,2},{2,12,2},{0,8,16},{21,0,338},{0,8,16},{15,1,421},{5,16,5},{7,12,1},{2,12,1},{15,1,421},{31,0,421},{2,12,1},{0,10,433},{31,0,421},{0,10,433},{4,0,445},{4,0,445},{4,0,445},{4,0,445},{2,12,2},{2,12,2},{2,12,2},{2,7,2},{0,8,49},
{0,8,49},{5,24,854},{5,18,458},{5,14,678},{5,13,470},{3,23,1517},{3,15,461},{3,13,5},{2,11,653},{0,14,1758},{0,11,530},{7,20,437},{6,17,17},{7,13,53},{6,13,81},{14,1,1517},{2,16,422},{3,13,5},{0,11,494},{29,0,1517},{0,11,494},{5,19,433},{5,19,433},{5,19,433},{5,12,433},{3,18,340},{3,13,4},{3,13,4},{3,9,41},{0,12,419},{0,10,19},{7,15,16},
{7,15,16},{7,15,16},{7,11,16},{12,0,338},{3,13,4},{3,13,4},{0,10,10},{24,0,338},{0,10,10},{17,0,421},{6,17,1},{8,13,1},{3,13,1},{17,0,421},{30,2,421},{3,13,1},{0,11,445},{30,2,421},{0,11,445},{5,0,433},{5,0,433},{5,0,433},{5,0,433},{3,14,1},{3,14,1},{3,14,1},{3,8,1},{0,10,10},{0,10,10},{6,25,854},{6,19,458},{6,15,678},
{6,14,470},{4,24,1515},{4,16,459},{4,14,3},{3,12,666},{0,16,1658},{0,12,459},{7,23,437},{7,18,17},{8,14,51},{7,14,81},{16,0,1514},{3,17,422},{4,14,2},{0,12,458},{24,4,1514},{0,12,458},{6,20,433},{6,20,433},{6,20,433},{6,13,433},{4,19,339},{4,14,2},{4,14,2},{4,10,42},{0,13,365},{0,11,14},{7,18,16},{7,18,16},{7,18,16},{7,12,20},{13,1,338},
{4,14,1},{4,14,1},{0,11,10},{27,0,338},{0,11,10},{18,1,421},{7,18,1},{9,14,1},{4,14,1},{18,1,421},{31,3,421},{4,14,1},{0,12,433},{31,3,421},{0,12,433},{6,0,433},{6,0,433},{6,0,433},{6,0,433},{4,14,2},{4,14,2},{4,14,2},{4,9,2},{0,11,5},{0,11,5},{7,26,854},{7,19,459},{7,16,666},{7,15,470},{5,25,1515},{5,17,459},{5,15,3},
{4,13,651},{0,17,1577},{1,13,459},{8,24,430},{8,19,14},{9,15,51},{8,15,91},{17,1,1514},{4,18,425},{5,15,2},{0,13,437},{27,4,1514},{0,13,437},{7,21,433},{7,21,433},{7,21,433},{7,14,433},{5,20,339},{5,15,2},{5,15,2},{5,11,42},{0,15,339},{1,12,21},{8,19,10},{8,19,10},{8,19,10},{8,13,13},{15,0,338},{5,15,1},{5,15,1},{1,12,17},{30,0,338},
{1,12,17},{20,0,421},{8,19,5},{10,15,1},{5,15,1},{20,0,421},{30,5,421},{5,15,1},{0,13,433},{30,5,421},{0,13,433},{7,0,433},{7,0,433},{7,0,433},{7,0,433},{5,15,2},{5,15,2},{5,15,2},{5,10,2},{1,12,5},{1,12,5},{8,27,866},{8,20,461},{8,16,670},{8,16,494},{6,26,1515},{6,18,459},{6,16,3},{5,14,651},{0,19,1530},{2,14,459},{9,25,430},
{9,20,14},{10,16,50},{9,16,90},{19,0,1514},{5,19,425},{6,16,2},{0,14,434},{30,4,1514},{0,14,434},{8,22,446},{8,22,446},{8,22,446},{8,15,446},{6,21,339},{6,16,3},{6,16,3},{6,12,42},{1,16,339},{2,13,21},{9,20,10},{9,20,10},{9,20,10},{9,14,13},{16,1,338},{6,16,2},{6,16,2},{4,12,16},{31,1,338},{4,12,16},{21,1,421},{9,20,5},{11,16,1},
{6,16,1},{21,1,421},{31,6,421},{6,16,1},{0,14,433},{31,6,421},{0,14,433},{8,0,445},{8,0,445},{8,0,445},{8,0,445},{6,16,2},{6,16,2},{6,16,2},{6,11,2},{2,13,5},{2,13,5},{9,28,854},{9,22,458},{9,18,678},{9,17,470},{7,27,1517},{7,19,461},{7,17,5},{6,15,653},{1,20,1526},{3,15,461},{11,24,437},{10,21,17},{11,17,53},{10,17,81},{20,1,1517},
{6,20,422},{7,17,5},{2,15,446},{31,5,1517},{2,15,446},{9,23,433},{9,23,433},{9,23,433},{9,16,433},{7,22,340},{7,17,4},{7,17,4},{7,13,41},{2,17,340},{4,14,19},{11,19,16},{11,19,16},{11,19,16},{11,15,16},{16,4,338},{7,17,4},{7,17,4},{4,14,10},{28,4,338},{4,14,10},{23,0,421},{10,21,1},{12,17,1},{7,17,1},{23,0,421},{30,8,421},{7,17,1},
{0,15,445},{30,8,421},{0,15,445},{9,0,433},{9,0,433},{9,0,433},{9,0,433},{7,18,1},{7,18,1},{7,18,1},{7,12,1},{4,14,10},{4,14,10},{10,29,854},{10,23,458},{10,19,678},{10,18,470},{8,28,1515},{8,20,459},{8,18,3},{7,16,666},{2,21,1526},{4,16,459},{11,27,437},{11,22,17},{12,18,51},{11,18,81},{20,4,1514},{7,21,422},{8,18,2},{2,16,434},{28,8,1514},
{2,16,434},{10,24,433},{10,24,433},{10,24,433},{10,17,433},{8,23,339},{8,18,2},{8,18,2},{8,14,42},{3,18,340},{4,15,14},{11,22,16},{11,22,16},{11,22,16},{11,16,20},{17,5,338},{8,18,1},{8,18,1},{4,15,10},{31,4,338},{4,15,10},{24,1,421},{11,22,1},{13,18,1},{8,18,1},{24,1,421},{31,9,421},{8,18,1},{0,16,433},{31,9,421},{0,16,433},{10,0,433},
{10,0,433},{10,0,433},{10,0,433},{8,18,2},{8,18,2},{8,18,2},{8,13,2},{4,15,5},{4,15,5},{11,30,854},{11,23,459},{11,20,666},{11,19,470},{9,29,1515},{9,21,459},{9,19,3},{8,17,651},{3,22,1526},{5,17,459},{12,28,430},{12,23,14},{13,19,51},{12,19,91},{21,5,1514},{8,22,425},{9,19,2},{3,17,434},{31,8,1514},{3,17,434},{11,25,433},{11,25,433},{11,25,433},
{11,18,433},{9,24,339},{9,19,2},{9,19,2},{9,15,42},{4,19,339},{5,16,21},{12,23,10},{12,23,10},{12,23,10},{12,17,13},{21,0,338},{9,19,1},{9,19,1},{5,16,17},{30,6,338},{5,16,17},{26,0,421},{12,23,5},{14,19,1},{9,19,1},{26,0,421},{30,11,421},{9,19,1},{0,17,433},{30,11,421},{0,17,433},{11,0,433},{11,0,433},{11,0,433},{11,0,433},{9,19,2},
{9,19,2},{9,19,2},{9,14,2},{5,16,5},{5,16,5},{12,31,866},{12,24,461},{12,20,670},{12,20,494},{10,30,1515},{10,22,459},{10,20,3},{9,18,651},{4,23,1530},{6,18,459},{13,29,430},{13,24,14},{14,20,50},{13,20,90},{25,0,1514},{9,23,425},{10,20,2},{4,18,434},{30,10,1514},{4,18,434},{12,26,446},{12,26,446},{12,26,446},{12,19,446},{10,25,339},{10,20,3},{10,20,3},
{10,16,42},{5,20,339},{6,17,21},{13,24,10},{13,24,10},{13,24,10},{13,18,13},{22,1,338},{10,20,2},{10,20,2},{8,16,16},{31,7,338},{8,16,16},{27,1,421},{13,24,5},{15,20,1},{10,20,1},{27,1,421},{31,12,421},{10,20,1},{0,18,433},{31,12,421},{0,18,433},{12,0,445},{12,0,445},{12,0,445},{12,0,445},{10,20,2},{10,20,2},{10,20,2},{10,15,2},{6,17,5},
{6,17,5},{13,31,878},{13,26,458},{13,22,678},{13,21,470},{11,31,1517},{11,23,461},{11,21,5},{10,19,653},{5,24,1526},{7,19,461},{15,28,437},{14,25,17},{15,21,53},{14,21,81},{26,1,1517},{10,24,422},{11,21,5},{6,19,446},{31,11,1517},{6,19,446},{13,27,433},{13,27,433},{13,27,433},{13,20,433},{11,26,340},{11,21,4},{11,21,4},{11,17,41},{6,21,340},{8,18,19},{15,23,16},
{15,23,16},{15,23,16},{15,19,16},{24,0,338},{11,21,4},{11,21,4},{8,18,10},{24,12,338},{8,18,10},{29,0,421},{14,25,1},{16,21,1},{11,21,1},{29,0,421},{30,14,421},{11,21,1},{0,19,445},{30,14,421},{0,19,445},{13,0,433},{13,0,433},{13,0,433},{13,0,433},{11,22,1},{11,22,1},{11,22,1},{11,16,1},{8,18,10},{8,18,10},{14,31,938},{14,27,458},{14,23,678},
{14,22,470},{12,31,1542},{12,24,459},{12,22,3},{11,20,666},{6,25,1526},{8,20,459},{15,31,437},{15,26,17},{16,22,51},{15,22,81},{28,0,1514},{11,25,422},{12,22,2},{6,20,434},{24,16,1514},{6,20,434},{14,28,433},{14,28,433},{14,28,433},{14,21,433},{12,27,339},{12,22,2},{12,22,2},{12,18,42},{7,22,340},{8,19,14},{15,26,16},{15,26,16},{15,26,16},{15,20,20},{25,1,338},
{12,22,1},{12,22,1},{8,19,10},{27,12,338},{8,19,10},{30,1,421},{15,26,1},{17,22,1},{12,22,1},{30,1,421},{31,15,421},{12,22,1},{0,20,433},{31,15,421},{0,20,433},{14,0,433},{14,0,433},{14,0,433},{14,0,433},{12,22,2},{12,22,2},{12,22,2},{12,17,2},{8,19,5},{8,19,5},{15,31,998},{15,27,459},{15,24,666},{15,23,470},{14,31,1598},{13,25,459},{13,23,3},
{12,21,651},{7,26,1526},{9,21,459},{16,31,442},{16,27,14},{17,23,51},{16,23,91},{29,1,1514},{12,26,425},{13,23,2},{7,21,434},{27,16,1514},{7,21,434},{15,29,433},{15,29,433},{15,29,433},{15,22,433},{13,28,339},{13,23,2},{13,23,2},{13,19,42},{8,23,339},{9,20,21},{16,27,10},{16,27,10},{16,27,10},{16,21,13},{27,0,338},{13,23,1},{13,23,1},{9,20,17},{30,12,338},
{9,20,17},{31,2,421},{16,27,5},{18,23,1},{13,23,1},{31,2,421},{30,17,421},{13,23,1},{0,21,433},{30,17,421},{0,21,433},{15,0,433},{15,0,433},{15,0,433},{15,0,433},{13,23,2},{13,23,2},{13,23,2},{13,18,2},{9,20,5},{9,20,5},{16,31,1086},{16,28,461},{16,24,670},{16,24,494},{15,31,1622},{14,26,459},{14,24,3},{13,22,651},{8,27,1530},{10,22,459},{18,31,446},
{17,28,14},{18,24,50},{17,24,90},{31,0,1514},{13,27,425},{14,24,2},{8,22,434},{30,16,1514},{8,22,434},{16,30,446},{16,30,446},{16,30,446},{16,23,446},{14,29,339},{14,24,3},{14,24,3},{14,20,42},{9,24,339},{10,21,21},{17,28,10},{17,28,10},{17,28,10},{17,22,13},{28,1,338},{14,24,2},{14,24,2},{12,20,16},{31,13,338},{12,20,16},{31,5,421},{17,28,5},{19,24,1},
{14,24,1},{31,5,421},{31,18,421},{14,24,1},{0,22,433},{31,18,421},{0,22,433},{16,0,445},{16,0,445},{16,0,445},{16,0,445},{14,24,2},{14,24,2},{14,24,2},{14,19,2},{10,21,5},{10,21,5},{18,31,1242},{17,30,458},{17,26,678},{17,25,470},{16,31,1703},{15,27,461},{15,25,5},{14,23,653},{9,28,1526},{11,23,461},{19,31,461},{18,29,17},{19,25,53},{18,25,81},{31,3,1517},
{14,28,422},{15,25,5},{10,23,446},{31,17,1517},{10,23,446},{17,31,433},{17,31,433},{17,31,433},{17,24,433},{15,30,340},{15,25,4},{15,25,4},{15,21,41},{10,25,340},{12,22,19},{19,27,16},{19,27,16},{19,27,16},{19,23,16},{28,4,338},{15,25,4},{15,25,4},{12,22,10},{28,16,338},{12,22,10},{31,8,421},{18,29,1},{20,25,1},{15,25,1},{31,8,421},{30,20,421},{15,25,1},
{0,23,445},{30,20,421},{0,23,445},{17,0,433},{17,0,433},{17,0,433},{17,0,433},{15,26,1},{15,26,1},{15,26,1},{15,20,1},{12,22,10},{12,22,10},{19,31,1326},{18,31,458},{18,27,678},{18,26,470},{17,31,1838},{16,28,459},{16,26,3},{15,24,666},{10,29,1526},{12,24,459},{20,31,506},{19,30,17},{20,26,51},{19,26,81},{28,12,1514},{15,29,422},{16,26,2},{10,24,434},{28,20,1514},
{10,24,434},{18,31,442},{18,31,442},{18,31,442},{18,25,433},{16,31,339},{16,26,2},{16,26,2},{16,22,42},{11,26,340},{12,23,14},{19,30,16},{19,30,16},{19,30,16},{19,24,20},{29,5,338},{16,26,1},{16,26,1},{12,23,10},{31,16,338},{12,23,10},{31,11,421},{19,30,1},{21,26,1},{16,26,1},{31,11,421},{31,21,421},{16,26,1},{0,24,433},{31,21,421},{0,24,433},{18,0,433},
{18,0,433},{18,0,433},{18,0,433},{16,26,2},{16,26,2},{16,26,2},{16,21,2},{12,23,5},{12,23,5},{20,31,1470},{19,31,459},{19,28,666},{19,27,470},{19,31,1911},{17,29,459},{17,27,3},{16,25,651},{11,30,1526},{13,25,459},{21,31,590},{20,31,14},{21,27,51},{20,27,91},{29,13,1514},{16,30,425},{17,27,2},{11,25,434},{31,20,1514},{11,25,434},{19,31,458},{19,31,458},{19,31,458},
{19,26,433},{17,31,357},{17,27,2},{17,27,2},{17,23,42},{12,27,339},{13,24,21},{20,31,10},{20,31,10},{20,31,10},{20,25,13},{31,4,338},{17,27,1},{17,27,1},{13,24,17},{30,18,338},{13,24,17},{31,14,421},{20,31,5},{22,27,1},{17,27,1},{31,14,421},{30,23,421},{17,27,1},{0,25,433},{30,23,421},{0,25,433},{19,0,433},{19,0,433},{19,0,433},{19,0,433},{17,27,2},
{17,27,2},{17,27,2},{17,22,2},{13,24,5},{13,24,5},{21,31,1650},{20,31,530},{20,28,670},{20,28,494},{20,31,2030},{18,30,459},{18,28,3},{17,26,651},{12,31,1530},{14,26,459},{23,31,650},{21,31,29},{22,28,50},{21,28,90},{31,12,1514},{17,31,425},{18,28,2},{12,26,434},{30,22,1514},{12,26,434},{20,31,494},{20,31,494},{20,31,494},{20,27,446},{18,31,411},{18,28,3},{18,28,3},
{18,24,42},{13,28,339},{14,25,21},{21,31,13},{21,31,13},{21,31,13},{21,26,13},{31,7,338},{18,28,2},{18,28,2},{16,24,16},{31,19,338},{16,24,16},{31,17,421},{22,31,10},{23,28,1},{18,28,1},{31,17,421},{31,24,421},{18,28,1},{0,26,433},{31,24,421},{0,26,433},{20,0,445},{20,0,445},{20,0,445},{20,0,445},{18,28,2},{18,28,2},{18,28,2},{18,23,2},{14,25,5},
{14,25,5},{22,31,1902},{21,31,723},{21,30,678},{21,29,470},{21,31,2235},{19,31,461},{19,29,5},{18,27,653},{14,31,1587},{15,27,461},{24,31,734},{23,31,65},{23,29,53},{22,29,81},{31,15,1517},{19,31,461},{19,29,5},{14,27,446},{31,23,1517},{14,27,446},{21,31,554},{21,31,554},{21,31,554},{21,28,433},{20,31,437},{19,29,4},{19,29,4},{19,25,41},{14,29,340},{16,26,19},{23,31,16},
{23,31,16},{23,31,16},{23,27,16},{28,16,338},{19,29,4},{19,29,4},{16,26,10},{24,24,338},{16,26,10},{31,20,421},{23,31,49},{24,29,1},{19,29,1},{31,20,421},{30,26,421},{19,29,1},{0,27,445},{30,26,421},{0,27,445},{21,0,433},{21,0,433},{21,0,433},{21,0,433},{19,30,1},{19,30,1},{19,30,1},{19,24,1},{16,26,10},{16,26,10},{23,31,2074},{23,31,930},{22,31,678},
{22,30,470},{23,31,2382},{20,31,570},{20,30,3},{19,28,666},{16,31,1703},{16,28,459},{25,31,854},{24,31,173},{24,30,51},{23,30,81},{28,24,1514},{21,31,554},{20,30,2},{14,28,434},{24,28,1514},{14,28,434},{22,31,629},{22,31,629},{22,31,629},{22,29,433},{21,31,491},{20,30,2},{20,30,2},{20,26,42},{15,30,340},{16,27,14},{24,31,29},{24,31,29},{24,31,29},{23,28,20},{29,17,338},
{20,30,1},{20,30,1},{16,27,10},{27,24,338},{16,27,10},{31,23,421},{25,31,109},{25,30,1},{20,30,1},{31,23,421},{31,27,421},{20,30,1},{0,28,433},{31,27,421},{0,28,433},{22,0,433},{22,0,433},{22,0,433},{22,0,433},{20,30,2},{20,30,2},{20,30,2},{20,25,2},{16,27,5},{16,27,5},{24,31,2229},{24,31,1146},{23,31,689},{23,31,469},{24,31,2476},{22,31,731},{21,31,2},
{20,29,618},{18,31,1805},{17,29,426},{26,31,953},{25,31,339},{25,31,50},{24,30,82},{29,25,1459},{23,31,620},{21,31,1},{15,29,401},{27,28,1459},{15,29,401},{23,31,689},{23,31,689},{23,31,689},{23,30,433},{22,31,581},{21,31,2},{21,31,2},{21,27,42},{16,31,339},{17,28,21},{25,31,50},{25,31,50},{25,31,50},{24,29,13},{31,16,338},{21,31,1},{21,31,1},{17,28,17},{30,24,338},
{17,28,17},{31,26,392},{27,31,157},{26,31,0},{21,31,0},{31,26,392},{30,29,392},{21,31,0},{0,29,400},{30,29,392},{0,29,400},{23,0,433},{23,0,433},{23,0,433},{23,0,433},{21,31,2},{21,31,2},{21,31,2},{21,26,2},{17,28,5},{17,28,5},{25,31,1943},{24,31,1130},{24,31,769},{24,31,446},{25,31,2103},{23,31,573},{22,31,42},{22,29,373},{20,31,1481},{18,30,234},{27,31,657},
{26,31,285},{26,31,89},{25,31,18},{31,23,1064},{24,31,426},{23,31,8},{16,30,209},{31,27,1064},{16,30,209},{24,31,769},{24,31,769},{24,31,769},{24,31,446},{23,31,661},{22,31,42},{22,31,42},{22,28,42},{18,31,365},{18,29,21},{26,31,89},{26,31,89},{26,31,89},{25,30,13},{31,19,338},{23,31,8},{23,31,8},{20,28,16},{31,25,338},{20,28,16},{31,27,202},{28,31,80},{27,31,4},
{24,31,1},{31,27,202},{31,29,202},{24,31,1},{0,30,208},{31,29,202},{0,30,208},{24,0,445},{24,0,445},{24,0,445},{24,0,445},{22,31,17},{22,31,17},{22,31,17},{22,27,2},{18,29,5},{18,29,5},{27,31,1710},{26,31,1126},{25,31,917},{25,31,469},{26,31,1779},{24,31,507},{24,31,146},{23,30,154},{22,31,1221},{20,30,83},{28,31,450},{27,31,258},{27,31,137},{27,31,16},{29,29,722},
{26,31,286},{25,31,52},{20,30,74},{31,28,722},{20,30,74},{25,31,917},{25,31,917},{25,31,917},{25,31,469},{25,31,789},{24,31,146},{24,31,146},{23,29,41},{20,31,446},{20,30,19},{27,31,137},{27,31,137},{27,31,137},{27,31,16},{28,28,338},{25,31,52},{25,31,52},{20,30,10},{28,28,338},{20,30,10},{31,29,61},{30,31,34},{29,31,0},{27,31,0},{31,29,61},{31,30,61},{27,31,0},
{0,30,73},{31,30,61},{0,30,73},{25,0,433},{25,0,433},{25,0,433},{25,0,433},{23,31,49},{23,31,49},{23,31,49},{23,28,1},{20,30,10},{20,30,10},{27,31,1486},{27,31,1123},{27,31,1002},{26,31,554},{27,31,1519},{25,31,582},{25,31,293},{24,30,106},{23,31,1090},{20,31,14},{29,31,354},{28,31,258},{28,31,194},{28,31,50},{31,27,509},{27,31,234},{27,31,113},{20,31,10},{31,29,509},
{20,31,10},{27,31,1002},{27,31,1002},{27,31,1002},{26,31,554},{26,31,915},{25,31,293},{25,31,293},{24,30,42},{22,31,564},{20,31,14},{28,31,194},{28,31,194},{28,31,194},{28,31,50},{29,29,338},{27,31,113},{27,31,113},{20,31,10},{31,28,338},{20,31,10},{31,30,9},{31,31,9},{30,31,9},{30,31,0},{31,30,9},{30,31,9},{30,31,0},{0,31,9},{30,31,9},{0,31,9},{26,0,433},
{26,0,433},{26,0,433},{26,0,433},{24,31,82},{24,31,82},{24,31,82},{24,29,2},{20,31,5},{20,31,5},{28,31,1197},{28,31,1005},{27,31,917},{27,31,554},{28,31,1213},{26,31,522},{26,31,326},{25,31,17},{24,31,870},{22,31,16},{30,31,241},{29,31,182},{29,31,146},{29,31,61},{31,28,297},{28,31,153},{28,31,89},{23,31,1},{30,30,297},{23,31,1},{27,31,917},{27,31,917},{27,31,917},
{27,31,554},{27,31,774},{26,31,326},{26,31,326},{25,31,17},{24,31,509},{22,31,16},{29,31,146},{29,31,146},{29,31,146},{29,31,61},{31,27,221},{28,31,89},{28,31,89},{23,31,1},{31,29,221},{23,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{27,0,433},{27,0,433},{27,0,433},{27,0,433},{26,31,130},
{26,31,130},{26,31,130},{25,30,2},{22,31,16},{22,31,16},{29,31,927},{28,31,765},{28,31,701},{28,31,509},{28,31,845},{27,31,404},{27,31,283},{26,31,2},{26,31,589},{24,31,52},{30,31,97},{30,31,70},{30,31,61},{30,31,34},{31,30,118},{30,31,67},{29,31,40},{26,31,1},{30,31,118},{26,31,1},{28,31,701},{28,31,701},{28,31,701},{28,31,509},{28,31,589},{27,31,283},{27,31,283},
{26,31,2},{25,31,386},{24,31,52},{30,31,61},{30,31,61},{30,31,61},{30,31,34},{31,29,85},{29,31,40},{29,31,40},{26,31,1},{31,30,85},{26,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{28,0,445},{28,0,445},{28,0,445},{28,0,445},{27,31,162},{27,31,162},{27,31,162},{26,31,2},{24,31,52},
{24,31,52},{0,14,884},{0,10,117},{0,7,10},{0,6,317},{0,10,1899},{0,6,1236},{0,6,573},{0,4,1438},{0,4,2065},{0,4,1559},{0,14,884},{0,10,117},{0,7,10},{0,6,317},{5,0,1899},{0,6,1236},{0,6,573},{0,4,1438},{10,0,1899},{0,4,1438},{0,7,1},{0,7,1},{0,7,1},{0,3,4},{0,3,164},{0,3,68},{0,3,68},{0,2,104},{0,2,189},{0,1,129},{0,7,1},
{0,7,1},{0,7,1},{0,3,4},{2,0,164},{0,3,68},{0,3,68},{0,2,104},{3,0,164},{0,2,104},{7,0,884},{0,10,117},{0,7,10},{0,6,317},{7,0,884},{14,0,884},{0,6,317},{0,5,890},{14,0,884},{0,5,890},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,17,884},{0,12,53},{0,8,20},
{0,7,265},{0,11,2360},{0,8,1384},{0,7,626},{0,5,1683},{0,5,2580},{0,5,1852},{0,17,884},{0,12,53},{0,8,20},{0,7,265},{6,0,2356},{0,8,1384},{0,7,626},{0,5,1683},{10,1,2356},{0,5,1683},{0,10,1},{0,10,1},{0,10,1},{0,5,1},{0,5,338},{0,4,137},{0,4,137},{0,2,200},{0,2,381},{0,2,225},{0,10,1},{0,10,1},{0,10,1},{0,5,1},{2,1,338},
{0,4,137},{0,4,137},{0,2,200},{5,0,338},{0,2,200},{8,1,884},{0,12,53},{1,8,5},{0,7,265},{8,1,884},{17,0,884},{0,7,265},{0,6,890},{17,0,884},{0,6,890},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,20,882},{0,14,16},{0,10,85},{0,8,200},{0,14,2904},{0,9,1530},{0,8,684},
{0,6,1978},{0,6,3220},{0,5,2172},{0,20,882},{0,14,16},{1,9,84},{0,8,200},{7,0,2904},{0,9,1530},{0,8,684},{0,6,1978},{14,0,2904},{0,6,1978},{0,12,1},{0,12,1},{0,12,1},{0,6,4},{0,6,580},{0,5,218},{0,5,218},{0,3,356},{0,3,644},{0,3,420},{0,12,1},{0,12,1},{0,12,1},{0,6,4},{3,0,580},{0,5,218},{0,5,218},{0,3,356},{6,0,580},
{0,3,356},{8,4,882},{0,14,16},{2,9,5},{0,8,200},{8,4,882},{20,0,882},{0,8,200},{0,7,890},{20,0,882},{0,7,890},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,23,918},{0,15,41},{1,11,153},{0,10,184},{0,16,3048},{0,11,1476},{0,9,473},{0,6,1950},{0,7,3517},{0,6,2206},{1,21,886},
{1,15,20},{1,11,89},{0,10,184},{8,0,3048},{0,11,1476},{0,9,473},{0,6,1950},{16,0,3048},{0,6,1950},{0,15,37},{0,15,37},{0,15,37},{0,8,40},{0,9,648},{0,7,185},{0,7,185},{0,4,337},{0,4,777},{0,4,458},{1,13,5},{1,13,5},{1,13,5},{1,7,8},{4,1,648},{0,7,185},{0,7,185},{0,4,337},{9,0,648},{0,4,337},{9,5,882},{0,15,5},{3,10,5},
{0,10,148},{9,5,882},{23,0,882},{0,10,148},{0,8,900},{23,0,882},{0,8,900},{0,0,36},{0,0,36},{0,0,36},{0,0,36},{0,2,1},{0,2,1},{0,2,1},{0,1,1},{0,1,5},{0,1,5},{1,24,997},{1,17,123},{1,12,318},{1,11,243},{0,19,3051},{0,12,1278},{0,10,274},{0,8,1795},{0,9,3672},{0,7,2222},{2,22,885},{2,16,10},{2,12,101},{1,11,179},{10,0,3051},
{0,12,1278},{0,10,274},{0,8,1795},{19,0,3051},{0,8,1795},{1,17,114},{1,17,114},{1,17,114},{1,9,114},{0,12,648},{0,9,85},{0,9,85},{0,5,244},{0,5,889},{0,5,413},{2,15,1},{2,15,1},{2,15,1},{2,8,5},{4,4,648},{0,9,85},{0,9,85},{0,5,244},{12,0,648},{0,5,244},{13,0,884},{2,16,9},{4,11,10},{0,11,90},{13,0,884},{26,0,884},{0,11,90},
{0,9,890},{26,0,884},{0,9,890},{1,0,113},{1,0,113},{1,0,113},{1,0,113},{0,6,1},{0,6,1},{0,6,1},{0,3,1},{0,2,50},{0,2,50},{1,27,1173},{1,18,314},{2,13,510},{1,12,374},{0,22,3051},{0,14,1139},{0,12,153},{0,9,1630},{0,10,3924},{0,8,2199},{3,23,885},{3,17,10},{3,13,101},{2,12,197},{11,0,3051},{0,14,1139},{0,12,153},{0,9,1630},{22,0,3051},
{0,9,1630},{1,20,290},{1,20,290},{1,20,290},{1,10,293},{0,15,648},{0,10,41},{0,10,41},{0,6,181},{0,7,1027},{0,6,437},{3,16,2},{3,16,2},{3,16,2},{3,9,5},{5,5,648},{0,10,41},{0,10,41},{0,6,181},{15,0,648},{0,6,181},{14,1,884},{3,17,9},{5,12,5},{0,12,53},{14,1,884},{29,0,884},{0,12,53},{0,10,890},{29,0,884},{0,10,890},{1,0,289},
{1,0,289},{1,0,289},{1,0,289},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{0,3,113},{0,3,113},{2,28,1365},{2,19,506},{3,14,830},{2,13,566},{0,25,3051},{0,16,1051},{0,13,36},{0,10,1483},{0,12,4164},{0,9,2174},{4,24,886},{4,18,20},{4,14,89},{3,13,197},{12,1,3051},{0,16,1051},{0,13,36},{0,10,1483},{25,0,3051},{0,10,1483},{2,21,482},{2,21,482},{2,21,482},
{2,11,485},{0,18,648},{0,12,5},{0,12,5},{0,7,149},{0,8,1182},{0,7,510},{4,16,5},{4,16,5},{4,16,5},{4,10,8},{9,0,648},{0,12,5},{0,12,5},{0,7,149},{18,0,648},{0,7,149},{16,0,882},{3,19,10},{6,13,5},{0,13,20},{16,0,882},{24,4,882},{0,13,20},{0,11,890},{24,4,882},{0,11,890},{2,0,481},{2,0,481},{2,0,481},{2,0,481},{0,11,1},
{0,11,1},{0,11,1},{0,6,1},{0,5,185},{0,5,185},{2,31,1669},{3,20,818},{3,15,1161},{2,14,838},{0,28,3048},{0,17,949},{0,14,6},{0,11,1395},{0,13,4381},{0,10,2228},{5,25,886},{5,19,20},{5,15,89},{4,14,184},{12,4,3048},{0,17,949},{0,14,6},{0,11,1395},{28,0,3048},{0,11,1395},{2,24,786},{2,24,786},{2,24,786},{2,13,786},{0,21,648},{0,14,2},{0,14,2},
{0,9,101},{0,9,1352},{0,8,590},{5,17,5},{5,17,5},{5,17,5},{5,11,8},{10,1,648},{0,14,2},{0,14,2},{0,9,101},{21,0,648},{0,9,101},{17,1,882},{4,19,5},{7,14,5},{0,14,5},{17,1,882},{27,4,882},{0,14,5},{0,12,900},{27,4,882},{0,12,900},{2,0,785},{2,0,785},{2,0,785},{2,0,785},{0,14,1},{0,14,1},{0,14,1},{0,7,1},{0,6,305},
{0,6,305},{3,31,1814},{3,22,968},{4,16,1314},{3,15,945},{1,29,3055},{0,19,936},{1,15,17},{0,12,1314},{0,15,4321},{0,12,2007},{6,26,885},{6,20,10},{6,16,101},{5,15,179},{15,2,3051},{0,19,900},{1,15,13},{0,12,1278},{31,0,3051},{0,12,1278},{3,25,900},{3,25,900},{3,25,900},{3,14,900},{1,22,654},{1,15,8},{1,15,8},{0,10,76},{0,11,1296},{0,9,425},{6,19,1},
{6,19,1},{6,19,1},{6,12,5},{12,0,648},{1,15,4},{1,15,4},{0,10,40},{24,0,648},{0,10,40},{19,0,884},{6,20,9},{8,15,10},{1,15,9},{19,0,884},{30,4,884},{1,15,9},{0,13,890},{30,4,884},{0,13,890},{3,0,900},{3,0,900},{3,0,900},{3,0,900},{1,16,5},{1,16,5},{1,16,5},{1,8,8},{0,7,269},{0,7,269},{5,31,1838},{4,23,948},{5,17,1314},
{4,16,943},{2,30,3055},{1,20,936},{2,16,13},{1,13,1314},{0,16,4056},{0,13,1620},{7,27,885},{7,21,10},{7,17,101},{6,16,197},{17,0,3051},{0,21,891},{2,16,9},{0,13,1179},{30,2,3051},{0,13,1179},{4,26,891},{4,26,891},{4,26,891},{4,15,891},{2,23,654},{2,16,9},{2,16,9},{1,11,76},{0,13,1107},{0,10,273},{7,20,2},{7,20,2},{7,20,2},{7,13,5},{13,1,648},
{3,15,5},{3,15,5},{0,11,20},{27,0,648},{0,11,20},{20,1,884},{7,21,9},{9,16,5},{2,16,5},{20,1,884},{31,5,884},{2,16,5},{0,14,890},{31,5,884},{0,14,890},{4,0,890},{4,0,890},{4,0,890},{4,0,890},{2,17,5},{2,17,5},{2,17,5},{2,9,8},{0,9,149},{0,9,149},{6,31,1868},{5,24,948},{6,18,1314},{5,17,943},{3,31,3055},{2,21,936},{3,17,13},
{2,14,1314},{0,18,3825},{0,14,1354},{8,28,886},{8,22,20},{8,18,89},{7,17,197},{18,1,3051},{1,22,891},{3,17,9},{0,14,1098},{31,3,3051},{0,14,1098},{5,27,891},{5,27,891},{5,27,891},{5,16,891},{3,24,652},{3,17,9},{3,17,9},{2,12,86},{0,14,976},{0,12,126},{8,20,5},{8,20,5},{8,20,5},{8,14,8},{15,0,648},{4,16,5},{4,16,5},{0,12,5},{30,0,648},
{0,12,5},{20,4,882},{7,23,10},{10,17,5},{3,17,5},{20,4,882},{28,8,882},{3,17,5},{0,15,890},{28,8,882},{0,15,890},{5,0,890},{5,0,890},{5,0,890},{5,0,890},{3,18,5},{3,18,5},{3,18,5},{3,10,8},{0,11,80},{0,11,80},{7,31,1908},{6,25,948},{7,19,1314},{6,18,943},{4,31,3084},{3,22,936},{4,18,15},{3,15,1314},{0,19,3640},{0,15,1175},{9,29,886},
{9,23,20},{9,19,89},{8,18,184},{20,0,3048},{2,23,891},{4,18,6},{0,15,1054},{24,8,3048},{0,15,1054},{6,28,891},{6,28,891},{6,28,891},{6,17,891},{4,25,657},{4,18,11},{4,18,11},{3,13,86},{0,16,852},{0,13,27},{9,21,5},{9,21,5},{9,21,5},{9,15,8},{16,1,648},{4,18,2},{4,18,2},{0,13,2},{31,1,648},{0,13,2},{21,5,882},{8,23,5},{11,18,5},
{4,18,5},{21,5,882},{31,8,882},{4,18,5},{0,16,900},{31,8,882},{0,16,900},{6,0,890},{6,0,890},{6,0,890},{6,0,890},{4,18,10},{4,18,10},{4,18,10},{4,11,10},{0,13,26},{0,13,26},{8,31,1998},{7,26,968},{8,20,1314},{7,19,945},{6,31,3160},{4,23,936},{5,19,17},{4,16,1314},{0,21,3420},{0,16,1028},{10,30,885},{10,24,10},{10,20,101},{9,19,179},{22,0,3051},
{3,24,891},{5,19,13},{0,16,1003},{30,7,3051},{0,16,1003},{7,29,900},{7,29,900},{7,29,900},{7,18,900},{5,26,654},{5,19,8},{5,19,8},{4,14,76},{0,18,750},{1,14,24},{10,23,1},{10,23,1},{10,23,1},{10,16,5},{16,4,648},{5,19,4},{5,19,4},{2,14,5},{28,4,648},{2,14,5},{25,0,884},{10,24,9},{12,19,10},{5,19,9},{25,0,884},{30,10,884},{5,19,9},
{0,17,890},{30,10,884},{0,17,890},{7,0,900},{7,0,900},{7,0,900},{7,0,900},{5,20,5},{5,20,5},{5,20,5},{5,12,8},{0,15,5},{0,15,5},{9,31,2124},{8,27,948},{9,21,1314},{8,20,943},{7,31,3196},{5,24,936},{6,20,13},{5,17,1314},{0,23,3307},{0,17,971},{11,31,885},{11,25,10},{11,21,101},{10,20,197},{23,0,3051},{4,25,891},{6,20,9},{0,17,970},{30,8,3051},
{0,17,970},{8,30,891},{8,30,891},{8,30,891},{8,19,891},{6,27,654},{6,20,9},{6,20,9},{5,15,76},{0,19,691},{2,15,24},{11,24,2},{11,24,2},{11,24,2},{11,17,5},{17,5,648},{7,19,5},{7,19,5},{3,15,5},{31,4,648},{3,15,5},{26,1,884},{11,25,9},{13,20,5},{6,20,5},{26,1,884},{31,11,884},{6,20,5},{0,18,890},{31,11,884},{0,18,890},{8,0,890},
{8,0,890},{8,0,890},{8,0,890},{6,21,5},{6,21,5},{6,21,5},{6,13,8},{1,16,5},{1,16,5},{10,31,2286},{9,28,948},{10,22,1314},{9,21,943},{8,31,3277},{6,25,936},{7,21,13},{6,18,1314},{0,24,3196},{0,19,948},{12,31,904},{12,26,20},{12,22,89},{11,21,197},{24,1,3051},{5,26,891},{7,21,9},{0,19,939},{31,9,3051},{0,19,939},{9,31,891},{9,31,891},{9,31,891},
{9,20,891},{7,28,652},{7,21,9},{7,21,9},{6,16,86},{0,21,652},{2,16,18},{12,24,5},{12,24,5},{12,24,5},{12,18,8},{21,0,648},{8,20,5},{8,20,5},{3,16,2},{30,6,648},{3,16,2},{28,0,882},{11,27,10},{14,21,5},{7,21,5},{28,0,882},{24,16,882},{7,21,5},{0,19,890},{24,16,882},{0,19,890},{9,0,890},{9,0,890},{9,0,890},{9,0,890},{7,22,5},
{7,22,5},{7,22,5},{7,14,8},{2,17,5},{2,17,5},{11,31,2414},{10,29,948},{11,23,1314},{10,22,943},{9,31,3412},{7,26,936},{8,22,15},{7,19,1314},{0,26,3115},{1,20,958},{13,31,958},{13,27,20},{13,23,89},{12,22,184},{24,4,3048},{6,27,891},{8,22,6},{0,20,925},{28,12,3048},{0,20,925},{10,31,894},{10,31,894},{10,31,894},{10,21,891},{8,29,657},{8,22,11},{8,22,11},
{7,17,86},{1,22,652},{3,17,18},{13,25,5},{13,25,5},{13,25,5},{13,19,8},{22,1,648},{8,22,2},{8,22,2},{4,17,2},{31,7,648},{4,17,2},{29,1,882},{12,27,5},{15,22,5},{8,22,5},{29,1,882},{27,16,882},{8,22,5},{0,20,900},{27,16,882},{0,20,900},{10,0,890},{10,0,890},{10,0,890},{10,0,890},{8,22,10},{8,22,10},{8,22,10},{8,15,10},{3,18,5},
{3,18,5},{13,31,2606},{11,30,968},{12,24,1314},{11,23,945},{11,31,3519},{8,27,936},{9,23,17},{8,20,1314},{0,28,3085},{2,21,942},{15,31,995},{14,28,10},{14,24,101},{13,23,179},{27,2,3051},{7,28,891},{9,23,13},{0,21,891},{31,12,3051},{0,21,891},{11,31,925},{11,31,925},{11,31,925},{11,22,900},{9,30,654},{9,23,8},{9,23,8},{8,18,76},{2,23,651},{5,18,24},{14,27,1},
{14,27,1},{14,27,1},{14,20,5},{24,0,648},{9,23,4},{9,23,4},{6,18,5},{24,12,648},{6,18,5},{31,0,884},{14,28,9},{16,23,10},{9,23,9},{31,0,884},{30,16,884},{9,23,9},{0,21,890},{30,16,884},{0,21,890},{11,0,900},{11,0,900},{11,0,900},{11,0,900},{9,24,5},{9,24,5},{9,24,5},{9,16,8},{4,19,5},{4,19,5},{14,31,2804},{12,31,948},{13,25,1314},
{12,24,943},{12,31,3652},{9,28,936},{10,24,13},{9,21,1314},{0,29,3052},{3,22,942},{16,31,1054},{15,29,10},{15,25,101},{14,24,197},{29,0,3051},{8,29,891},{10,24,9},{1,22,891},{30,14,3051},{1,22,891},{12,31,939},{12,31,939},{12,31,939},{12,23,891},{10,31,654},{10,24,9},{10,24,9},{9,19,76},{3,24,652},{6,19,24},{15,28,2},{15,28,2},{15,28,2},{15,21,5},{25,1,648},
{11,23,5},{11,23,5},{7,19,5},{27,12,648},{7,19,5},{31,3,884},{15,29,9},{17,24,5},{10,24,5},{31,3,884},{31,17,884},{10,24,5},{0,22,890},{31,17,884},{0,22,890},{12,0,890},{12,0,890},{12,0,890},{12,0,890},{10,25,5},{10,25,5},{10,25,5},{10,17,8},{5,20,5},{5,20,5},{15,31,2956},{14,31,979},{14,26,1314},{13,25,943},{13,31,3841},{10,29,936},{11,25,13},
{10,22,1314},{1,30,3052},{4,23,948},{17,31,1144},{16,30,20},{16,26,89},{15,25,197},{30,1,3051},{9,30,891},{11,25,9},{2,23,891},{31,15,3051},{2,23,891},{14,31,979},{14,31,979},{14,31,979},{13,24,891},{11,31,670},{11,25,9},{11,25,9},{10,20,86},{4,25,652},{6,20,18},{16,28,5},{16,28,5},{16,28,5},{16,22,8},{27,0,648},{12,24,5},{12,24,5},{7,20,2},{30,12,648},
{7,20,2},{28,12,882},{15,31,10},{18,25,5},{11,25,5},{28,12,882},{28,20,882},{11,25,5},{0,23,890},{28,20,882},{0,23,890},{13,0,890},{13,0,890},{13,0,890},{13,0,890},{11,26,5},{11,26,5},{11,26,5},{11,18,8},{6,21,5},{6,21,5},{16,31,3182},{15,31,1028},{15,27,1314},{14,26,943},{15,31,4020},{11,30,936},{12,26,15},{11,23,1314},{2,31,3052},{5,24,958},{18,31,1270},
{17,31,20},{17,27,89},{16,26,184},{28,8,3048},{10,31,891},{12,26,6},{2,24,901},{24,20,3048},{2,24,901},{15,31,1003},{15,31,1003},{15,31,1003},{14,25,891},{12,31,707},{12,26,11},{12,26,11},{11,21,86},{5,26,652},{7,21,18},{17,29,5},{17,29,5},{17,29,5},{17,23,8},{28,1,648},{12,26,2},{12,26,2},{8,21,2},{31,13,648},{8,21,2},{29,13,882},{16,31,5},{19,26,5},
{12,26,5},{29,13,882},{31,20,882},{12,26,5},{0,24,900},{31,20,882},{0,24,900},{14,0,890},{14,0,890},{14,0,890},{14,0,890},{12,26,10},{12,26,10},{12,26,10},{12,19,10},{7,22,5},{7,22,5},{17,31,3508},{16,31,1175},{16,28,1314},{15,27,945},{16,31,4209},{12,31,936},{13,27,17},{12,24,1314},{4,31,3100},{6,25,942},{20,31,1368},{18,31,37},{18,28,101},{17,27,179},{31,6,3051},
{12,31,900},{13,27,13},{4,25,891},{30,19,3051},{4,25,891},{16,31,1054},{16,31,1054},{16,31,1054},{15,26,900},{14,31,780},{13,27,8},{13,27,8},{12,22,76},{6,27,651},{9,22,24},{18,31,1},{18,31,1},{18,31,1},{18,24,5},{28,4,648},{13,27,4},{13,27,4},{10,22,5},{28,16,648},{10,22,5},{31,12,884},{18,31,36},{20,27,10},{13,27,9},{31,12,884},{30,22,884},{13,27,9},
{0,25,890},{30,22,884},{0,25,890},{15,0,900},{15,0,900},{15,0,900},{15,0,900},{13,28,5},{13,28,5},{13,28,5},{13,20,8},{8,23,5},{8,23,5},{19,31,3790},{17,31,1412},{17,29,1314},{16,28,943},{17,31,4452},{14,31,954},{14,28,13},{13,25,1314},{7,31,3196},{7,26,942},{21,31,1494},{19,31,126},{19,29,101},{18,28,197},{31,8,3051},{14,31,950},{14,28,9},{5,26,891},{30,20,3051},
{5,26,891},{17,31,1123},{17,31,1123},{17,31,1123},{16,27,891},{15,31,820},{14,28,9},{14,28,9},{13,23,76},{7,28,652},{10,23,24},{19,31,5},{19,31,5},{19,31,5},{19,25,5},{29,5,648},{15,27,5},{15,27,5},{11,23,5},{31,16,648},{11,23,5},{31,15,884},{20,31,80},{21,28,5},{14,28,5},{31,15,884},{31,23,884},{14,28,5},{0,26,890},{31,23,884},{0,26,890},{16,0,890},
{16,0,890},{16,0,890},{16,0,890},{14,29,5},{14,29,5},{14,29,5},{14,21,8},{9,24,5},{9,24,5},{20,31,4072},{18,31,1694},{18,30,1314},{17,29,943},{19,31,4705},{15,31,1064},{15,29,13},{14,26,1314},{8,31,3355},{8,27,948},{22,31,1656},{20,31,276},{20,30,89},{19,29,197},{31,11,3051},{16,31,1054},{15,29,9},{6,27,891},{31,21,3051},{6,27,891},{18,31,1210},{18,31,1210},{18,31,1210},
{17,28,891},{16,31,897},{15,29,9},{15,29,9},{14,24,86},{8,29,652},{10,24,18},{20,31,20},{20,31,20},{20,31,20},{20,26,8},{31,4,648},{16,28,5},{16,28,5},{11,24,2},{30,18,648},{11,24,2},{28,24,882},{22,31,157},{22,29,5},{15,29,5},{28,24,882},{24,28,882},{15,29,5},{0,27,890},{24,28,882},{0,27,890},{17,0,890},{17,0,890},{17,0,890},{17,0,890},{15,30,5},
{15,30,5},{15,30,5},{15,22,8},{10,25,5},{10,25,5},{21,31,4390},{19,31,2007},{19,31,1314},{18,30,943},{20,31,4932},{16,31,1287},{16,30,15},{15,27,1314},{11,31,3547},{9,28,958},{23,31,1784},{22,31,465},{21,31,89},{20,30,184},{28,20,3048},{18,31,1188},{16,30,6},{6,28,901},{28,24,3048},{6,28,901},{19,31,1278},{19,31,1278},{19,31,1278},{18,29,891},{17,31,1011},{16,30,11},{16,30,11},
{15,25,86},{9,30,652},{11,25,18},{21,31,53},{21,31,53},{21,31,53},{21,27,8},{31,7,648},{16,30,2},{16,30,2},{12,25,2},{31,19,648},{12,25,2},{29,25,882},{24,31,269},{23,30,5},{16,30,5},{29,25,882},{27,28,882},{16,30,5},{0,28,900},{27,28,882},{0,28,900},{18,0,890},{18,0,890},{18,0,890},{18,0,890},{16,30,10},{16,30,10},{16,30,10},{16,23,10},{11,26,5},
{11,26,5},{22,31,4471},{20,31,2295},{20,31,1395},{19,31,936},{21,31,4906},{18,31,1414},{17,31,8},{16,28,1161},{13,31,3570},{11,28,818},{24,31,1769},{23,31,590},{23,31,106},{22,30,146},{29,21,2814},{19,31,1206},{17,31,4},{8,29,786},{31,24,2814},{8,29,786},{20,31,1395},{20,31,1395},{20,31,1395},{19,30,900},{19,31,1134},{17,31,8},{17,31,8},{16,26,76},{10,31,651},{13,26,24},{23,31,106},
{23,31,106},{23,31,106},{22,28,5},{28,16,648},{17,31,4},{17,31,4},{14,26,5},{24,24,648},{14,26,5},{31,24,761},{26,31,317},{24,31,1},{17,31,0},{31,24,761},{30,28,761},{17,31,0},{0,29,785},{30,28,761},{0,29,785},{19,0,900},{19,0,900},{19,0,900},{19,0,900},{17,31,8},{17,31,8},{17,31,8},{17,24,8},{12,27,5},{12,27,5},{23,31,3955},{22,31,2260},{21,31,1530},
{20,31,891},{22,31,4375},{19,31,1194},{18,31,54},{17,28,805},{15,31,3075},{12,29,498},{25,31,1417},{24,31,510},{24,31,149},{23,30,74},{31,19,2249},{21,31,937},{19,31,5},{11,29,482},{31,25,2249},{11,29,482},{21,31,1530},{21,31,1530},{21,31,1530},{20,31,891},{20,31,1251},{18,31,54},{18,31,54},{17,27,76},{12,31,691},{14,27,24},{24,31,149},{24,31,149},{24,31,149},{23,29,5},{29,17,648},
{19,31,5},{19,31,5},{15,27,5},{27,24,648},{15,27,5},{29,29,481},{27,31,202},{25,31,4},{20,31,1},{29,29,481},{31,28,481},{20,31,1},{0,29,481},{31,28,481},{0,29,481},{20,0,890},{20,0,890},{20,0,890},{20,0,890},{18,31,29},{18,31,29},{18,31,29},{18,25,8},{13,28,5},{13,28,5},{24,31,3609},{23,31,2199},{22,31,1683},{21,31,915},{23,31,3827},{20,31,1071},{19,31,153},
{18,29,485},{16,31,2690},{13,30,306},{26,31,1133},{25,31,489},{25,31,200},{24,31,20},{29,25,1769},{22,31,710},{21,31,41},{12,30,290},{27,28,1769},{12,30,290},{22,31,1683},{22,31,1683},{22,31,1683},{21,31,915},{21,31,1401},{19,31,153},{19,31,153},{18,28,86},{14,31,769},{14,28,18},{25,31,200},{25,31,200},{25,31,200},{24,30,8},{31,16,648},{21,31,41},{21,31,41},{15,28,2},{30,24,648},
{15,28,2},{31,27,265},{28,31,113},{27,31,1},{23,31,1},{31,27,265},{31,29,265},{23,31,1},{0,30,289},{31,29,265},{0,30,289},{21,0,890},{21,0,890},{21,0,890},{21,0,890},{19,31,53},{19,31,53},{19,31,53},{19,26,8},{14,29,5},{14,29,5},{24,31,3305},{24,31,2222},{23,31,1795},{22,31,990},{24,31,3438},{22,31,1087},{21,31,306},{19,30,293},{18,31,2403},{15,30,118},{27,31,857},
{26,31,465},{26,31,269},{25,31,8},{31,23,1374},{24,31,546},{23,31,98},{14,30,114},{31,27,1374},{14,30,114},{23,31,1795},{23,31,1795},{23,31,1795},{22,31,990},{22,31,1587},{21,31,306},{21,31,306},{19,29,86},{16,31,897},{15,29,18},{26,31,269},{26,31,269},{26,31,269},{25,31,8},{31,19,648},{23,31,98},{23,31,98},{16,29,2},{31,25,648},{16,29,2},{31,28,113},{29,31,52},{28,31,1},
{26,31,1},{31,28,113},{30,30,113},{26,31,1},{0,30,113},{30,30,113},{0,30,113},{22,0,890},{22,0,890},{22,0,890},{22,0,890},{20,31,90},{20,31,90},{20,31,90},{20,27,10},{15,30,5},{15,30,5},{25,31,3092},{25,31,2292},{24,31,1978},{24,31,1123},{25,31,3124},{23,31,1068},{22,31,525},{20,30,140},{20,31,2196},{16,31,41},{28,31,680},{27,31,458},{27,31,337},{26,31,65},{29,29,1032},
{26,31,456},{24,31,185},{16,31,37},{31,28,1032},{16,31,37},{24,31,1978},{24,31,1978},{24,31,1978},{24,31,1123},{24,31,1769},{22,31,525},{22,31,525},{20,30,76},{18,31,1080},{17,30,24},{27,31,337},{27,31,337},{27,31,337},{26,31,65},{28,28,648},{24,31,185},{24,31,185},{18,30,5},{28,28,648},{18,30,5},{31,30,18},{30,31,9},{30,31,0},{29,31,0},{31,30,18},{30,31,18},{29,31,0},
{0,31,36},{30,31,18},{0,31,36},{23,0,900},{23,0,900},{23,0,900},{23,0,900},{22,31,164},{22,31,164},{22,31,164},{21,28,8},{16,31,5},{16,31,5},{27,31,2818},{26,31,2254},{25,31,2043},{25,31,1243},{26,31,2829},{24,31,1099},{23,31,684},{21,31,72},{22,31,2007},{18,31,20},{29,31,566},{28,31,420},{28,31,356},{27,31,122},{31,27,771},{27,31,386},{26,31,232},{19,31,1},{31,29,771},
{19,31,1},{25,31,2043},{25,31,2043},{25,31,2043},{25,31,1243},{25,31,1819},{23,31,684},{23,31,684},{21,31,72},{20,31,1172},{18,31,20},{28,31,356},{28,31,356},{28,31,356},{27,31,122},{31,25,578},{26,31,232},{26,31,232},{19,31,1},{27,30,578},{19,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{24,0,890},
{24,0,890},{24,0,890},{24,0,890},{23,31,200},{23,31,200},{23,31,200},{22,29,8},{18,31,20},{18,31,20},{27,31,2242},{27,31,1879},{26,31,1738},{26,31,1150},{27,31,2209},{24,31,987},{24,31,626},{23,31,20},{23,31,1560},{19,31,53},{29,31,342},{29,31,257},{29,31,221},{28,31,68},{31,28,452},{28,31,228},{27,31,137},{22,31,1},{30,30,452},{22,31,1},{26,31,1738},{26,31,1738},{26,31,1738},
{26,31,1150},{25,31,1499},{24,31,626},{24,31,626},{23,31,20},{22,31,950},{19,31,53},{29,31,221},{29,31,221},{29,31,221},{28,31,68},{31,26,340},{27,31,137},{27,31,137},{22,31,1},{30,29,340},{22,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{25,0,890},{25,0,890},{25,0,890},{25,0,890},{24,31,265},
{24,31,265},{24,31,265},{23,30,8},{19,31,53},{19,31,53},{28,31,1844},{27,31,1559},{27,31,1438},{27,31,1075},{27,31,1713},{26,31,853},{25,31,635},{24,31,10},{24,31,1207},{21,31,125},{30,31,172},{30,31,145},{29,31,125},{29,31,40},{31,29,216},{29,31,121},{28,31,68},{24,31,1},{31,30,216},{24,31,1},{27,31,1438},{27,31,1438},{27,31,1438},{27,31,1075},{27,31,1229},{25,31,635},{25,31,635},
{24,31,10},{23,31,756},{21,31,125},{29,31,125},{29,31,125},{29,31,125},{29,31,40},{31,28,164},{28,31,68},{28,31,68},{24,31,1},{30,30,164},{24,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{26,0,890},{26,0,890},{26,0,890},{26,0,890},{25,31,346},{25,31,346},{25,31,346},{24,31,10},{21,31,125},
{21,31,125},{0,19,1568},{0,13,202},{0,10,13},{0,8,596},{0,13,3371},{0,9,2162},{0,8,1080},{0,5,2539},{0,6,3648},{0,5,2708},{0,19,1568},{0,13,202},{0,10,13},{0,8,596},{6,1,3371},{0,9,2162},{0,8,1080},{0,5,2539},{13,0,3371},{0,5,2539},{0,9,0},{0,9,0},{0,9,0},{0,4,4},{0,4,290},{0,4,125},{0,4,125},{0,2,164},{0,2,321},{0,2,189},{0,9,0},
{0,9,0},{0,9,0},{0,4,4},{2,1,290},{0,4,125},{0,4,125},{0,2,164},{4,0,290},{0,2,164},{9,1,1568},{0,13,202},{0,10,13},{0,8,596},{9,1,1568},{19,0,1568},{0,8,596},{0,6,1586},{19,0,1568},{0,6,1586},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,22,1568},{0,15,117},{0,11,8},
{0,9,485},{0,15,3971},{0,10,2369},{0,9,1061},{0,6,2834},{0,7,4341},{0,6,3090},{0,22,1568},{0,15,117},{0,11,8},{0,9,485},{7,1,3968},{0,10,2369},{0,9,1061},{0,6,2834},{11,2,3968},{0,6,2834},{0,12,1},{0,12,1},{0,12,1},{0,6,0},{0,6,512},{0,5,194},{0,5,194},{0,3,320},{0,3,576},{0,2,381},{0,12,1},{0,12,1},{0,12,1},{0,6,0},{3,0,512},
{0,5,194},{0,5,194},{0,3,320},{6,0,512},{0,3,320},{11,0,1568},{0,15,117},{0,11,8},{0,9,485},{11,0,1568},{22,0,1568},{0,9,485},{0,7,1586},{22,0,1568},{0,7,1586},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,25,1568},{0,17,45},{0,12,58},{0,10,392},{0,17,4652},{0,11,2596},{0,10,1121},
{0,7,3254},{0,8,5140},{0,6,3570},{0,25,1568},{0,17,45},{0,12,58},{0,10,392},{8,1,4652},{0,11,2596},{0,10,1121},{0,7,3254},{17,0,4652},{0,7,3254},{0,15,1},{0,15,1},{0,15,1},{0,7,4},{0,7,802},{0,6,320},{0,6,320},{0,3,512},{0,3,896},{0,3,576},{0,15,1},{0,15,1},{0,15,1},{0,7,4},{3,2,802},{0,6,320},{0,6,320},{0,3,512},{7,0,802},
{0,3,512},{12,1,1568},{0,17,45},{1,12,13},{0,10,392},{12,1,1568},{25,0,1568},{0,10,392},{0,8,1576},{25,0,1568},{0,8,1576},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,28,1570},{0,19,16},{0,13,178},{0,11,340},{0,19,5424},{0,13,2843},{0,11,1240},{0,8,3704},{0,9,6003},{0,7,4095},{0,28,1570},
{0,19,16},{1,13,122},{0,11,340},{7,5,5419},{0,13,2843},{0,11,1240},{0,8,3704},{15,2,5419},{0,8,3704},{0,18,1},{0,18,1},{0,18,1},{0,9,1},{0,9,1152},{0,7,461},{0,7,461},{0,4,709},{0,4,1281},{0,4,830},{0,18,1},{0,18,1},{0,18,1},{0,9,1},{4,1,1152},{0,7,461},{0,7,461},{0,4,709},{9,0,1152},{0,4,709},{14,0,1568},{0,19,16},{2,13,13},
{0,11,340},{14,0,1568},{26,1,1568},{0,11,340},{0,9,1576},{26,1,1568},{0,9,1576},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,29,1633},{1,20,75},{1,14,265},{0,13,313},{0,22,5419},{0,14,2571},{0,12,853},{0,9,3410},{0,10,6244},{0,9,3986},{1,29,1569},{1,20,11},{2,14,117},{0,13,313},{11,0,5419},
{0,14,2571},{0,12,853},{0,9,3410},{22,0,5419},{0,9,3410},{1,19,65},{1,19,65},{1,19,65},{1,10,65},{0,12,1152},{0,9,289},{0,9,289},{0,5,580},{0,5,1393},{0,5,749},{1,19,1},{1,19,1},{1,19,1},{1,10,1},{4,4,1152},{0,9,289},{0,9,289},{0,5,580},{12,0,1152},{0,5,580},{13,5,1568},{0,21,4},{3,14,13},{0,13,232},{13,5,1568},{31,0,1568},{0,13,232},
{0,10,1586},{31,0,1568},{0,10,1586},{1,0,65},{1,0,65},{1,0,65},{1,0,65},{0,3,1},{0,3,1},{0,3,1},{0,2,1},{0,1,20},{0,1,20},{1,31,1731},{1,22,149},{2,15,457},{1,14,377},{0,25,5419},{0,16,2347},{0,13,556},{0,10,3179},{0,11,6495},{0,9,3890},{2,30,1569},{2,21,11},{3,15,117},{1,14,313},{12,1,5419},{0,16,2347},{0,13,556},{0,10,3179},{25,0,5419},
{0,10,3179},{1,22,145},{1,22,145},{1,22,145},{1,11,154},{0,15,1152},{0,11,194},{0,11,194},{0,6,481},{0,7,1531},{0,6,737},{2,20,1},{2,20,1},{2,20,1},{2,11,1},{5,5,1152},{0,11,194},{0,11,194},{0,6,481},{15,0,1152},{0,6,481},{17,0,1568},{1,22,4},{4,15,8},{0,14,157},{17,0,1568},{30,2,1568},{0,14,157},{0,11,1586},{30,2,1568},{0,11,1586},{1,0,145},
{1,0,145},{1,0,145},{1,0,145},{0,6,1},{0,6,1},{0,6,1},{0,3,1},{0,3,65},{0,3,65},{2,31,1977},{2,23,341},{2,16,707},{1,15,510},{0,28,5424},{0,17,2137},{0,15,373},{0,11,3035},{0,13,6709},{0,10,3860},{3,31,1569},{3,22,11},{4,16,122},{2,15,313},{14,0,5419},{0,17,2137},{0,15,373},{0,11,3035},{26,1,5419},{0,11,3035},{2,23,337},{2,23,337},{2,23,337},
{2,12,341},{0,18,1152},{0,13,104},{0,13,104},{0,8,410},{0,8,1686},{0,7,786},{3,21,1},{3,21,1},{3,21,1},{3,12,2},{9,0,1152},{0,13,104},{0,13,104},{0,8,410},{18,0,1152},{0,8,410},{18,1,1568},{2,23,4},{5,16,13},{0,15,117},{18,1,1568},{31,3,1568},{0,15,117},{0,12,1576},{31,3,1568},{0,12,1576},{2,0,337},{2,0,337},{2,0,337},{2,0,337},{0,9,1},
{0,9,1},{0,9,1},{0,5,4},{0,4,130},{0,4,130},{3,31,2353},{2,24,570},{3,17,1027},{2,16,714},{0,31,5424},{0,19,1979},{0,16,195},{0,12,2857},{0,14,7039},{0,11,3919},{4,31,1576},{4,23,16},{5,17,122},{3,16,331},{15,1,5419},{0,19,1979},{0,16,195},{0,12,2857},{27,2,5419},{0,12,2857},{2,26,546},{2,26,546},{2,26,546},{2,14,546},{0,21,1152},{0,14,50},{0,14,50},
{0,9,305},{0,9,1856},{0,8,830},{4,22,1},{4,22,1},{4,22,1},{4,13,1},{10,1,1152},{0,14,50},{0,14,50},{0,9,305},{21,0,1152},{0,9,305},{19,2,1568},{3,24,10},{6,17,13},{0,16,74},{19,2,1568},{30,5,1568},{0,16,74},{0,13,1576},{30,5,1568},{0,13,1576},{2,0,545},{2,0,545},{2,0,545},{2,0,545},{0,12,0},{0,12,0},{0,12,0},{0,6,1},{0,5,205},
{0,5,205},{4,31,2980},{3,26,925},{3,19,1484},{2,17,1032},{1,31,5504},{0,21,1811},{0,17,61},{0,13,2651},{0,16,7420},{0,13,3876},{6,31,1619},{5,24,11},{6,18,117},{4,17,313},{17,0,5419},{0,21,1811},{0,17,61},{0,13,2651},{30,2,5419},{0,13,2651},{3,27,900},{3,27,900},{3,27,900},{3,15,900},{0,24,1152},{0,16,8},{0,16,8},{0,10,208},{0,11,2124},{0,9,881},{5,23,1},
{5,23,1},{5,23,1},{5,14,1},{12,0,1152},{0,16,8},{0,16,8},{0,10,208},{24,0,1152},{0,10,208},{21,1,1568},{4,25,4},{7,18,13},{0,17,45},{21,1,1568},{27,8,1568},{0,17,45},{0,14,1586},{27,8,1568},{0,14,1586},{3,0,900},{3,0,900},{3,0,900},{3,0,900},{0,15,1},{0,15,1},{0,15,1},{0,8,4},{0,6,356},{0,6,356},{4,31,3668},{3,27,1320},{4,20,1940},
{3,18,1352},{2,31,5771},{0,23,1737},{0,18,37},{0,14,2486},{0,17,7711},{0,14,3930},{7,31,1635},{6,25,11},{7,19,117},{5,18,313},{18,1,5419},{0,23,1737},{0,18,37},{0,14,2486},{31,3,5419},{0,14,2486},{3,30,1252},{3,30,1252},{3,30,1252},{3,16,1256},{0,27,1152},{0,18,1},{0,18,1},{0,11,164},{0,12,2369},{0,10,1001},{6,24,1},{6,24,1},{6,24,1},{6,15,1},{13,1,1152},
{0,18,1},{0,18,1},{0,11,164},{27,0,1152},{0,11,164},{23,0,1568},{5,26,4},{8,19,8},{0,18,36},{23,0,1568},{30,8,1568},{0,18,36},{0,15,1586},{30,8,1568},{0,15,1586},{3,0,1252},{3,0,1252},{3,0,1252},{3,0,1252},{0,18,1},{0,18,1},{0,18,1},{0,9,1},{0,7,505},{0,7,505},{5,31,4346},{4,28,1644},{5,21,2372},{4,19,1703},{3,31,6079},{0,24,1644},{0,20,38},
{0,15,2390},{0,18,7969},{0,15,3907},{8,31,1682},{7,26,11},{8,20,122},{6,19,313},{19,2,5419},{0,24,1640},{0,20,34},{0,15,2386},{30,5,5419},{0,15,2386},{4,31,1587},{4,31,1587},{4,31,1587},{4,17,1590},{0,30,1156},{0,20,29},{0,20,29},{0,12,117},{0,13,2483},{0,11,1044},{7,25,1},{7,25,1},{7,25,1},{7,16,2},{15,0,1152},{1,19,1},{1,19,1},{0,12,113},{30,0,1152},
{0,12,113},{24,1,1568},{6,27,4},{9,20,13},{0,20,25},{24,1,1568},{31,9,1568},{0,20,25},{0,16,1576},{31,9,1568},{0,16,1576},{4,0,1586},{4,0,1586},{4,0,1586},{4,0,1586},{0,21,5},{0,21,5},{0,21,5},{0,11,8},{0,9,565},{0,9,565},{7,31,4580},{5,29,1644},{6,22,2372},{4,20,1692},{4,31,6228},{1,25,1644},{1,21,38},{1,16,2348},{0,20,7577},{0,16,3408},{9,31,1760},
{8,27,16},{9,21,122},{7,20,331},{19,5,5419},{0,26,1593},{2,20,27},{0,16,2252},{31,6,5419},{0,16,2252},{5,31,1590},{5,31,1590},{5,31,1590},{5,18,1590},{1,31,1156},{1,21,29},{1,21,29},{1,13,117},{0,15,2241},{0,13,699},{8,26,1},{8,26,1},{8,26,1},{8,17,1},{16,1,1152},{2,20,2},{2,20,2},{0,13,74},{31,1,1152},{0,13,74},{26,0,1568},{7,28,10},{10,21,13},
{0,21,10},{26,0,1568},{30,11,1568},{0,21,10},{0,17,1576},{30,11,1568},{0,17,1576},{5,0,1586},{5,0,1586},{5,0,1586},{5,0,1586},{1,22,5},{1,22,5},{1,22,5},{1,12,8},{0,10,433},{0,10,433},{8,31,4826},{6,30,1644},{7,23,2352},{6,21,1676},{5,31,6463},{2,26,1644},{2,22,38},{2,17,2348},{0,22,7196},{0,17,2863},{11,31,1865},{9,28,11},{10,22,117},{8,21,313},{23,0,5419},
{1,27,1586},{2,22,29},{0,18,2115},{30,8,5419},{0,18,2115},{6,31,1612},{6,31,1612},{6,31,1612},{6,19,1585},{3,30,1179},{3,21,29},{3,21,29},{2,14,122},{0,17,1953},{0,14,426},{9,27,1},{9,27,1},{9,27,1},{9,18,1},{16,4,1152},{3,21,4},{3,21,4},{0,15,49},{28,4,1152},{0,15,49},{25,5,1568},{8,29,4},{11,22,13},{1,22,4},{25,5,1568},{31,12,1568},{1,22,4},
{0,18,1586},{31,12,1568},{0,18,1586},{6,0,1576},{6,0,1576},{6,0,1576},{6,0,1576},{2,24,10},{2,24,10},{2,24,10},{2,13,10},{0,13,272},{0,13,272},{9,31,5108},{7,31,1644},{8,24,2372},{7,22,1676},{7,31,6660},{3,27,1644},{3,23,38},{3,18,2348},{0,23,6891},{0,18,2519},{12,31,1952},{10,29,11},{11,23,117},{9,22,313},{24,1,5419},{2,28,1584},{3,23,29},{0,19,2027},{31,9,5419},
{0,19,2027},{7,31,1640},{7,31,1640},{7,31,1640},{7,20,1580},{4,31,1188},{3,23,34},{3,23,34},{3,15,122},{0,19,1795},{0,15,261},{10,28,1},{10,28,1},{10,28,1},{10,19,1},{17,5,1152},{4,22,1},{4,22,1},{0,16,26},{31,4,1152},{0,16,26},{29,0,1568},{9,30,4},{12,23,8},{2,23,4},{29,0,1568},{30,14,1568},{2,23,4},{0,19,1586},{30,14,1568},{0,19,1586},{7,0,1576},
{7,0,1576},{7,0,1576},{7,0,1576},{3,24,10},{3,24,10},{3,24,10},{3,14,10},{0,14,170},{0,14,170},{10,31,5426},{8,31,1695},{9,25,2372},{8,23,1703},{8,31,6861},{4,28,1644},{4,24,38},{3,19,2372},{0,25,6573},{0,19,2268},{13,31,2066},{11,30,11},{12,24,122},{10,23,313},{26,0,5419},{3,29,1584},{4,24,34},{0,20,1937},{30,11,5419},{0,20,1937},{8,31,1686},{8,31,1686},{8,31,1686},
{8,21,1590},{5,31,1206},{4,24,29},{4,24,29},{4,16,117},{0,20,1602},{0,17,126},{11,29,1},{11,29,1},{11,29,1},{11,20,2},{21,0,1152},{5,23,1},{5,23,1},{0,17,5},{30,6,1152},{0,17,5},{30,1,1568},{10,31,4},{13,24,13},{3,24,10},{30,1,1568},{31,15,1568},{3,24,10},{0,20,1576},{31,15,1568},{0,20,1576},{8,0,1586},{8,0,1586},{8,0,1586},{8,0,1586},{4,25,5},
{4,25,5},{4,25,5},{4,15,8},{0,16,90},{0,16,90},{11,31,5658},{9,31,1836},{10,26,2372},{8,24,1692},{9,31,7116},{5,29,1644},{5,25,38},{5,20,2348},{0,26,6379},{0,21,2028},{14,31,2216},{12,31,16},{13,25,122},{11,24,331},{27,1,5419},{2,31,1584},{6,24,27},{0,21,1832},{27,14,5419},{0,21,1832},{9,31,1755},{9,31,1755},{9,31,1755},{9,22,1590},{6,31,1260},{5,25,29},{5,25,29},
{5,17,117},{0,22,1459},{0,18,38},{12,30,1},{12,30,1},{12,30,1},{12,21,1},{22,1,1152},{6,24,2},{6,24,2},{0,18,2},{31,7,1152},{0,18,2},{31,2,1568},{11,31,13},{14,25,13},{4,25,10},{31,2,1568},{30,17,1568},{4,25,10},{0,21,1576},{30,17,1568},{0,21,1576},{9,0,1586},{9,0,1586},{9,0,1586},{9,0,1586},{5,26,5},{5,26,5},{5,26,5},{5,16,8},{0,18,37},
{0,18,37},{12,31,6036},{11,31,2033},{11,27,2352},{10,25,1676},{11,31,7423},{6,30,1644},{6,26,38},{6,21,2348},{0,28,6109},{0,22,1794},{15,31,2371},{13,31,50},{14,26,117},{12,25,313},{29,0,5419},{5,31,1586},{6,26,29},{0,22,1730},{30,14,5419},{0,22,1730},{10,31,1865},{10,31,1865},{10,31,1865},{10,23,1585},{7,31,1339},{7,25,29},{7,25,29},{6,18,122},{0,23,1345},{0,19,16},{13,31,1},
{13,31,1},{13,31,1},{13,22,1},{24,0,1152},{7,25,4},{7,25,4},{2,19,1},{24,12,1152},{2,19,1},{29,9,1568},{14,31,41},{15,26,13},{5,26,4},{29,9,1568},{27,20,1568},{5,26,4},{0,22,1586},{27,20,1568},{0,22,1586},{10,0,1576},{10,0,1576},{10,0,1576},{10,0,1576},{6,28,10},{6,28,10},{6,28,10},{6,17,10},{0,20,13},{0,20,13},{13,31,6450},{12,31,2268},{12,28,2372},
{11,26,1676},{12,31,7676},{7,31,1644},{7,27,38},{7,22,2348},{0,30,5924},{0,23,1695},{17,31,2536},{15,31,139},{15,27,117},{13,26,313},{30,1,5419},{7,31,1635},{7,27,29},{0,23,1686},{31,15,5419},{0,23,1686},{11,31,1937},{11,31,1937},{11,31,1937},{11,24,1580},{9,31,1420},{7,27,34},{7,27,34},{7,19,122},{0,25,1234},{1,20,17},{14,31,10},{14,31,10},{14,31,10},{14,23,1},{25,1,1152},
{8,26,1},{8,26,1},{2,20,2},{27,12,1152},{2,20,2},{31,8,1568},{15,31,90},{16,27,8},{6,27,4},{31,8,1568},{30,20,1568},{6,27,4},{0,23,1586},{30,20,1568},{0,23,1586},{11,0,1576},{11,0,1576},{11,0,1576},{11,0,1576},{7,28,10},{7,28,10},{7,28,10},{7,18,10},{0,21,10},{0,21,10},{15,31,6772},{13,31,2595},{13,29,2372},{12,27,1703},{13,31,7985},{8,31,1725},{8,28,38},
{7,23,2372},{0,31,5773},{0,24,1644},{18,31,2722},{16,31,261},{16,28,122},{14,27,313},{31,2,5419},{8,31,1721},{8,28,34},{0,24,1640},{30,17,5419},{0,24,1640},{12,31,2027},{12,31,2027},{12,31,2027},{12,25,1590},{10,31,1510},{8,28,29},{8,28,29},{8,20,117},{0,27,1188},{2,21,17},{15,31,26},{15,31,26},{15,31,26},{15,24,2},{27,0,1152},{9,27,1},{9,27,1},{3,21,2},{30,12,1152},
{3,21,2},{31,11,1568},{17,31,180},{17,28,13},{7,28,10},{31,11,1568},{31,21,1568},{7,28,10},{0,24,1576},{31,21,1568},{0,24,1576},{12,0,1586},{12,0,1586},{12,0,1586},{12,0,1586},{8,29,5},{8,29,5},{8,29,5},{8,19,8},{1,22,10},{1,22,10},{16,31,7154},{14,31,2955},{14,30,2372},{12,28,1692},{14,31,8348},{10,31,1895},{9,29,38},{9,24,2348},{1,31,5956},{1,25,1644},{19,31,2866},
{17,31,468},{17,29,122},{15,28,331},{31,5,5419},{10,31,1859},{10,28,27},{0,25,1601},{31,18,5419},{0,25,1601},{13,31,2162},{13,31,2162},{13,31,2162},{13,26,1590},{11,31,1590},{9,29,29},{9,29,29},{9,21,117},{0,28,1161},{3,22,17},{16,31,49},{16,31,49},{16,31,49},{16,25,1},{28,1,1152},{10,28,2},{10,28,2},{4,22,2},{31,13,1152},{4,22,2},{31,14,1568},{19,31,277},{18,29,13},
{8,29,10},{31,14,1568},{30,23,1568},{8,29,10},{0,25,1576},{30,23,1568},{0,25,1576},{13,0,1586},{13,0,1586},{13,0,1586},{13,0,1586},{9,30,5},{9,30,5},{9,30,5},{9,20,8},{2,23,10},{2,23,10},{17,31,7636},{15,31,3408},{15,31,2352},{14,29,1676},{16,31,8673},{11,31,2187},{10,30,38},{10,25,2348},{3,31,6235},{2,26,1644},{20,31,3112},{19,31,754},{18,30,117},{16,29,313},{31,8,5419},
{12,31,2060},{10,30,29},{0,26,1587},{30,20,5419},{0,26,1587},{15,31,2252},{15,31,2252},{15,31,2252},{14,27,1585},{12,31,1740},{11,29,29},{11,29,29},{10,22,122},{0,30,1163},{4,23,16},{18,31,85},{18,31,85},{18,31,85},{17,26,1},{28,4,1152},{11,29,4},{11,29,4},{6,23,1},{28,16,1152},{6,23,1},{29,21,1568},{20,31,436},{19,30,13},{9,30,4},{29,21,1568},{31,24,1568},{9,30,4},
{0,26,1586},{31,24,1568},{0,26,1586},{14,0,1576},{14,0,1576},{14,0,1576},{14,0,1576},{10,31,13},{10,31,13},{10,31,13},{10,21,10},{3,24,10},{3,24,10},{18,31,8122},{16,31,3907},{16,31,2390},{15,30,1676},{17,31,9036},{12,31,2576},{11,31,38},{11,26,2348},{6,31,6555},{3,27,1644},{22,31,3392},{20,31,1044},{19,31,117},{17,30,313},{31,11,5419},{15,31,2284},{11,31,29},{0,27,1587},{31,21,5419},
{0,27,1587},{16,31,2386},{16,31,2386},{16,31,2386},{15,28,1580},{14,31,1924},{11,31,34},{11,31,34},{11,23,122},{1,31,1163},{5,24,17},{19,31,113},{19,31,113},{19,31,113},{18,27,1},{29,5,1152},{12,30,1},{12,30,1},{6,24,2},{31,16,1152},{6,24,2},{31,20,1568},{22,31,593},{20,31,8},{10,31,4},{31,20,1568},{30,26,1568},{10,31,4},{0,27,1586},{30,26,1568},{0,27,1586},{15,0,1576},
{15,0,1576},{15,0,1576},{15,0,1576},{11,31,25},{11,31,25},{11,31,25},{11,22,10},{4,25,10},{4,25,10},{19,31,7638},{17,31,4060},{17,31,2539},{16,30,1659},{18,31,8553},{14,31,2430},{13,31,37},{11,27,1940},{7,31,6120},{4,28,1320},{23,31,2996},{20,31,1064},{20,31,164},{19,30,186},{29,17,4803},{16,31,2018},{13,31,1},{1,28,1253},{27,24,4803},{1,28,1253},{17,31,2539},{17,31,2539},{17,31,2539},
{16,29,1590},{15,31,2028},{13,31,37},{13,31,37},{12,24,117},{3,31,1170},{6,25,17},{20,31,164},{20,31,164},{20,31,164},{19,28,2},{31,4,1152},{13,31,1},{13,31,1},{7,25,2},{30,18,1152},{7,25,2},{31,22,1250},{24,31,505},{22,31,0},{13,31,0},{31,22,1250},{30,27,1250},{13,31,0},{0,28,1252},{30,27,1250},{0,28,1252},{16,0,1586},{16,0,1586},{16,0,1586},{16,0,1586},{13,31,37},
{13,31,37},{13,31,37},{12,23,8},{5,26,10},{5,26,10},{20,31,7060},{19,31,3955},{18,31,2710},{17,31,1595},{19,31,7717},{15,31,2140},{14,31,77},{13,28,1480},{8,31,5539},{6,28,925},{23,31,2516},{22,31,945},{21,31,233},{20,30,100},{31,15,4056},{18,31,1656},{15,31,8},{4,28,900},{31,23,4056},{4,28,900},{18,31,2710},{18,31,2710},{18,31,2710},{17,30,1590},{16,31,2193},{14,31,77},{14,31,77},
{13,25,117},{5,31,1233},{7,26,17},{21,31,233},{21,31,233},{21,31,233},{20,29,1},{31,7,1152},{15,31,8},{15,31,8},{8,26,2},{31,19,1152},{8,26,2},{31,23,884},{25,31,370},{23,31,4},{16,31,1},{31,23,884},{31,27,884},{16,31,1},{0,28,900},{31,27,884},{0,28,900},{17,0,1586},{17,0,1586},{17,0,1586},{17,0,1586},{14,31,52},{14,31,52},{14,31,52},{13,24,8},{6,27,10},
{6,27,10},{21,31,6535},{20,31,3919},{19,31,2857},{18,31,1585},{20,31,6979},{16,31,1942},{15,31,195},{14,28,990},{11,31,4914},{7,29,562},{24,31,2045},{23,31,830},{22,31,338},{21,31,26},{29,21,3318},{19,31,1314},{17,31,52},{5,29,546},{31,24,3318},{5,29,546},{19,31,2857},{19,31,2857},{19,31,2857},{18,31,1585},{17,31,2436},{15,31,195},{15,31,195},{14,26,122},{7,31,1339},{8,27,16},{22,31,338},
{22,31,338},{22,31,338},{21,30,1},{28,16,1152},{17,31,52},{17,31,52},{10,27,1},{24,24,1152},{10,27,1},{31,25,545},{26,31,221},{25,31,0},{19,31,0},{31,25,545},{31,28,545},{19,31,0},{0,29,545},{31,28,545},{0,29,545},{18,0,1576},{18,0,1576},{18,0,1576},{18,0,1576},{15,31,74},{15,31,74},{15,31,74},{14,25,10},{7,28,10},{7,28,10},{22,31,6151},{20,31,3935},{20,31,3035},
{19,31,1640},{21,31,6458},{18,31,1902},{16,31,373},{15,29,670},{12,31,4499},{8,29,347},{25,31,1729},{24,31,786},{23,31,410},{22,31,1},{31,19,2753},{20,31,1094},{19,31,113},{8,29,338},{31,25,2753},{8,29,338},{20,31,3035},{20,31,3035},{20,31,3035},{19,31,1640},{19,31,2630},{16,31,373},{16,31,373},{15,27,122},{10,31,1483},{9,28,17},{23,31,410},{23,31,410},{23,31,410},{22,31,1},{29,17,1152},
{19,31,113},{19,31,113},{10,28,2},{27,24,1152},{10,28,2},{31,26,317},{27,31,130},{26,31,9},{22,31,0},{31,26,317},{30,29,317},{22,31,0},{0,29,337},{30,29,317},{0,29,337},{19,0,1576},{19,0,1576},{19,0,1576},{19,0,1576},{16,31,117},{16,31,117},{16,31,117},{15,26,10},{8,29,10},{8,29,10},{23,31,5691},{22,31,4004},{21,31,3254},{20,31,1755},{22,31,6023},{19,31,1830},{18,31,606},
{16,29,430},{14,31,4162},{9,30,155},{26,31,1481},{25,31,801},{24,31,505},{23,31,26},{29,25,2273},{22,31,914},{20,31,194},{9,30,146},{27,28,2273},{9,30,146},{21,31,3254},{21,31,3254},{21,31,3254},{20,31,1755},{20,31,2835},{18,31,606},{18,31,606},{16,28,117},{11,31,1665},{10,29,17},{24,31,505},{24,31,505},{24,31,505},{23,31,26},{31,16,1152},{20,31,194},{20,31,194},{11,29,2},{30,24,1152},
{11,29,2},{31,28,145},{28,31,65},{28,31,1},{25,31,0},{31,28,145},{30,30,145},{25,31,0},{0,30,145},{30,30,145},{0,30,145},{20,0,1586},{20,0,1586},{20,0,1586},{20,0,1586},{17,31,180},{17,31,180},{17,31,180},{16,27,8},{9,30,10},{9,30,10},{24,31,5421},{23,31,3999},{22,31,3491},{21,31,1947},{23,31,5539},{20,31,1879},{19,31,853},{17,30,238},{16,31,3922},{11,30,81},{27,31,1229},
{26,31,813},{26,31,617},{24,31,100},{31,23,1878},{23,31,822},{22,31,305},{12,30,66},{31,27,1878},{12,30,66},{22,31,3491},{22,31,3491},{22,31,3491},{21,31,1947},{21,31,3081},{19,31,853},{19,31,853},{17,29,117},{14,31,1905},{11,30,17},{26,31,617},{26,31,617},{26,31,617},{24,31,100},{31,19,1152},{22,31,305},{22,31,305},{12,30,2},{31,25,1152},{12,30,2},{31,29,45},{30,31,18},{29,31,4},
{28,31,1},{31,29,45},{31,30,45},{28,31,1},{0,30,65},{31,30,45},{0,30,65},{21,0,1586},{21,0,1586},{21,0,1586},{21,0,1586},{18,31,261},{18,31,261},{18,31,261},{17,28,8},{10,31,10},{10,31,10},{24,31,5178},{24,31,4095},{23,31,3704},{22,31,2201},{24,31,5197},{21,31,2101},{20,31,1240},{18,31,153},{18,31,3760},{12,31,16},{28,31,1088},{27,31,830},{27,31,709},{26,31,245},{29,29,1536},
{26,31,792},{24,31,461},{14,31,1},{31,28,1536},{14,31,1},{23,31,3704},{23,31,3704},{23,31,3704},{22,31,2201},{23,31,3396},{20,31,1240},{20,31,1240},{18,30,122},{15,31,2208},{12,31,16},{27,31,709},{27,31,709},{27,31,709},{26,31,245},{28,28,1152},{24,31,461},{24,31,461},{14,31,1},{28,28,1152},{14,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},
{0,31,0},{31,31,0},{0,31,0},{22,0,1576},{22,0,1576},{22,0,1576},{22,0,1576},{20,31,340},{20,31,340},{20,31,340},{18,29,10},{12,31,16},{12,31,16},{25,31,4468},{24,31,3615},{24,31,3254},{23,31,2060},{24,31,4413},{22,31,1834},{21,31,1205},{19,31,58},{19,31,3127},{14,31,53},{28,31,768},{28,31,576},{28,31,512},{27,31,170},{31,26,1068},{26,31,536},{25,31,338},{16,31,1},{30,29,1068},
{16,31,1},{24,31,3254},{24,31,3254},{24,31,3254},{23,31,2060},{23,31,2852},{21,31,1205},{21,31,1205},{19,31,58},{16,31,1878},{14,31,53},{28,31,512},{28,31,512},{28,31,512},{27,31,170},{31,24,802},{25,31,338},{25,31,338},{16,31,1},{30,28,802},{16,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{23,0,1576},
{23,0,1576},{23,0,1576},{23,0,1576},{21,31,421},{21,31,421},{21,31,421},{19,30,10},{14,31,53},{14,31,53},{26,31,3858},{25,31,3188},{25,31,2899},{24,31,1947},{25,31,3700},{23,31,1596},{23,31,1112},{20,31,8},{20,31,2588},{16,31,117},{29,31,498},{28,31,384},{28,31,320},{27,31,122},{31,27,683},{27,31,342},{26,31,212},{19,31,1},{31,29,683},{19,31,1},{25,31,2899},{25,31,2899},{25,31,2899},
{24,31,1947},{24,31,2441},{23,31,1112},{23,31,1112},{20,31,8},{18,31,1568},{16,31,117},{28,31,320},{28,31,320},{28,31,320},{27,31,122},{29,29,512},{26,31,212},{26,31,212},{19,31,1},{31,28,512},{19,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{24,0,1586},{24,0,1586},{24,0,1586},{24,0,1586},{22,31,520},
{22,31,520},{22,31,520},{20,31,8},{16,31,117},{16,31,117},{27,31,3258},{26,31,2790},{26,31,2594},{25,31,1875},{26,31,3105},{24,31,1447},{23,31,1080},{21,31,20},{22,31,2151},{18,31,208},{29,31,306},{29,31,221},{29,31,185},{28,31,64},{31,28,384},{28,31,192},{27,31,125},{22,31,1},{30,30,384},{22,31,1},{26,31,2594},{26,31,2594},{26,31,2594},{25,31,1875},{25,31,2131},{23,31,1080},{23,31,1080},
{21,31,20},{20,31,1336},{18,31,208},{29,31,185},{29,31,185},{29,31,185},{28,31,64},{31,27,290},{27,31,125},{27,31,125},{22,31,1},{31,29,290},{22,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{25,0,1586},{25,0,1586},{25,0,1586},{25,0,1586},{23,31,596},{23,31,596},{23,31,596},{21,31,20},{18,31,208},
{18,31,208},{0,25,2669},{0,18,320},{0,13,5},{0,11,985},{0,17,5885},{0,11,3745},{0,10,1746},{0,7,4421},{0,8,6385},{0,7,4782},{0,25,2669},{0,18,320},{0,13,5},{0,11,985},{8,1,5885},{0,11,3745},{0,10,1746},{0,7,4421},{17,0,5885},{0,7,4421},{0,12,0},{0,12,0},{0,12,0},{0,6,1},{0,6,545},{0,5,205},{0,5,205},{0,3,337},{0,3,609},{0,3,401},{0,12,0},
{0,12,0},{0,12,0},{0,6,1},{3,0,545},{0,5,205},{0,5,205},{0,3,337},{6,0,545},{0,3,337},{13,0,2669},{0,18,320},{0,13,5},{0,11,985},{13,0,2669},{25,0,2669},{0,11,985},{0,8,2689},{25,0,2669},{0,8,2689},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,28,2665},{0,20,212},{0,14,10},
{0,12,850},{0,19,6669},{0,13,3974},{0,11,1837},{0,8,4865},{0,9,7266},{0,7,5310},{0,28,2665},{0,20,212},{0,14,10},{0,12,850},{9,1,6669},{0,13,3974},{0,11,1837},{0,8,4865},{19,0,6669},{0,8,4865},{0,15,0},{0,15,0},{0,15,0},{0,7,9},{0,7,845},{0,6,337},{0,6,337},{0,3,545},{0,3,945},{0,3,609},{0,15,0},{0,15,0},{0,15,0},{0,7,9},{3,2,841},
{0,6,337},{0,6,337},{0,3,545},{6,1,841},{0,3,545},{12,4,2665},{0,20,212},{1,14,5},{0,12,850},{12,4,2665},{28,0,2665},{0,12,850},{0,9,2689},{28,0,2665},{0,9,2689},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,31,2665},{0,21,113},{0,15,65},{0,13,709},{0,21,7538},{0,14,4289},{0,12,1907},
{0,9,5330},{0,10,8294},{0,8,5845},{0,31,2665},{0,21,113},{0,15,65},{0,13,709},{10,1,7538},{0,14,4289},{0,12,1907},{0,9,5330},{21,0,7538},{0,9,5330},{0,18,0},{0,18,0},{0,18,0},{0,9,0},{0,9,1201},{0,7,482},{0,7,482},{0,4,740},{0,4,1334},{0,4,861},{0,18,0},{0,18,0},{0,18,0},{0,9,0},{4,1,1201},{0,7,482},{0,7,482},{0,4,740},{9,0,1201},
{0,4,740},{13,5,2665},{0,21,113},{2,15,5},{0,13,709},{13,5,2665},{31,0,2665},{0,13,709},{0,10,2689},{31,0,2665},{0,10,2689},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,31,2809},{0,23,52},{0,16,173},{0,14,586},{0,23,8494},{0,15,4610},{0,13,1950},{0,9,5826},{0,11,9409},{0,9,6402},{1,31,2753},
{0,23,52},{1,16,132},{0,14,586},{11,1,8493},{0,15,4610},{0,13,1950},{0,9,5826},{19,2,8493},{0,9,5826},{0,21,0},{0,21,0},{0,21,0},{0,10,9},{0,10,1629},{0,9,640},{0,9,640},{0,5,985},{0,5,1798},{0,5,1154},{0,21,0},{0,21,0},{0,21,0},{0,10,9},{3,5,1625},{0,9,640},{0,9,640},{0,5,985},{7,2,1625},{0,5,985},{17,0,2665},{0,23,52},{2,16,13},
{0,14,586},{17,0,2665},{30,2,2665},{0,14,586},{0,11,2689},{30,2,2665},{0,11,2689},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,31,3105},{0,25,8},{1,17,325},{0,15,512},{0,25,9669},{0,17,4961},{0,14,2120},{0,10,6421},{0,12,10774},{0,10,7150},{2,31,3033},{0,25,8},{1,17,261},{0,15,512},{12,1,9669},
{0,17,4961},{0,14,2120},{0,10,6421},{25,0,9669},{0,10,6421},{0,24,0},{0,24,0},{0,24,0},{0,12,0},{0,12,2178},{0,9,865},{0,9,865},{0,6,1313},{0,5,2419},{0,5,1523},{0,24,0},{0,24,0},{0,24,0},{0,12,0},{4,4,2178},{0,9,865},{0,9,865},{0,6,1313},{12,0,2178},{0,6,1313},{19,0,2669},{0,25,8},{4,17,5},{0,15,512},{19,0,2669},{31,3,2669},{0,15,512},
{0,12,2689},{31,3,2669},{0,12,2689},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,31,3465},{0,27,65},{1,19,434},{0,17,474},{0,28,9670},{0,18,4658},{0,16,1717},{0,11,6209},{0,13,10979},{0,11,7109},{3,31,3101},{1,26,8},{2,18,261},{0,17,474},{14,0,9669},{0,18,4658},{0,16,1717},{0,11,6209},{26,1,9669},
{0,11,6209},{1,25,64},{1,25,64},{1,25,64},{1,13,64},{0,15,2178},{0,11,680},{0,11,680},{0,7,1189},{0,7,2557},{0,6,1457},{1,25,0},{1,25,0},{1,25,0},{1,13,0},{5,5,2178},{0,11,680},{0,11,680},{0,7,1189},{15,0,2178},{0,7,1189},{20,0,2665},{0,27,1},{5,18,5},{0,17,410},{20,0,2665},{24,8,2665},{0,17,410},{0,13,2689},{24,8,2665},{0,13,2689},{0,0,64},
{0,0,64},{0,0,64},{0,0,64},{0,3,0},{0,3,0},{0,3,0},{0,2,4},{0,1,13},{0,1,13},{3,31,3917},{1,28,129},{2,20,645},{1,18,538},{0,31,9670},{0,20,4330},{0,17,1246},{0,12,5925},{0,14,11321},{0,12,7014},{4,31,3218},{2,27,8},{3,19,261},{1,18,474},{15,1,9669},{0,20,4330},{0,17,1246},{0,12,5925},{27,2,9669},{0,12,5925},{1,28,128},{1,28,128},{1,28,128},
{1,15,132},{0,18,2178},{0,13,482},{0,13,482},{0,8,1040},{0,8,2712},{0,7,1470},{2,26,0},{2,26,0},{2,26,0},{2,14,0},{9,0,2178},{0,13,482},{0,13,482},{0,8,1040},{18,0,2178},{0,8,1040},{21,1,2665},{1,28,1},{6,19,5},{0,18,305},{21,1,2665},{27,8,2665},{0,18,305},{0,14,2689},{27,8,2665},{0,14,2689},{1,0,128},{1,0,128},{1,0,128},{1,0,128},{0,6,0},
{0,6,0},{0,6,0},{0,3,0},{0,2,61},{0,2,61},{3,31,4541},{2,29,321},{2,21,904},{1,19,625},{1,31,9749},{0,21,4018},{0,18,914},{0,14,5633},{0,16,11661},{0,13,6859},{5,31,3374},{3,28,10},{4,20,254},{2,19,474},{17,0,9670},{0,21,4018},{0,18,914},{0,14,5633},{30,2,9670},{0,14,5633},{2,29,320},{2,29,320},{2,29,320},{1,16,320},{0,21,2178},{0,15,353},{0,15,353},
{0,9,881},{0,9,2882},{0,9,1457},{3,27,0},{3,27,0},{3,27,0},{3,15,0},{10,1,2178},{0,15,353},{0,15,353},{0,9,881},{21,0,2178},{0,9,881},{23,0,2665},{2,29,1},{6,20,13},{0,19,245},{23,0,2665},{30,8,2665},{0,19,245},{0,15,2689},{30,8,2665},{0,15,2689},{1,0,320},{1,0,320},{1,0,320},{1,0,320},{0,9,0},{0,9,0},{0,9,0},{0,4,4},{0,4,125},
{0,4,125},{4,31,5378},{2,31,570},{3,22,1267},{2,20,850},{2,31,10045},{0,23,3745},{0,19,642},{0,14,5354},{0,17,11993},{0,14,6798},{7,31,3553},{4,29,8},{5,21,261},{3,20,491},{18,1,9669},{0,23,3745},{0,19,642},{0,14,5354},{31,3,9669},{0,14,5354},{2,31,561},{2,31,561},{2,31,561},{2,17,546},{0,24,2178},{0,17,205},{0,17,205},{0,10,730},{0,11,3150},{0,9,1457},{4,28,0},
{4,28,0},{4,28,0},{4,16,0},{12,0,2178},{0,17,205},{0,17,205},{0,10,730},{24,0,2178},{0,10,730},{25,0,2669},{3,30,1},{8,21,5},{0,20,170},{25,0,2669},{31,9,2669},{0,20,170},{0,16,2689},{31,9,2669},{0,16,2689},{2,0,545},{2,0,545},{2,0,545},{2,0,545},{0,12,0},{0,12,0},{0,12,0},{0,6,1},{0,5,205},{0,5,205},{5,31,6330},{3,31,905},{4,23,1718},
{2,21,1099},{3,31,10453},{0,25,3478},{0,21,369},{0,16,5138},{0,18,12455},{0,15,6867},{8,31,3710},{5,30,8},{6,22,261},{4,21,474},{19,2,9669},{0,25,3478},{0,21,369},{0,16,5138},{30,5,9669},{0,16,5138},{3,31,901},{3,31,901},{3,31,901},{3,18,866},{0,27,2178},{0,19,130},{0,19,130},{0,11,650},{0,12,3395},{0,10,1523},{5,29,0},{5,29,0},{5,29,0},{5,17,0},{13,1,2178},
{0,19,130},{0,19,130},{0,11,650},{27,0,2178},{0,11,650},{24,4,2665},{4,31,1},{9,22,5},{0,21,113},{24,4,2665},{28,12,2665},{0,21,113},{0,17,2689},{28,12,2665},{0,17,2689},{3,0,865},{3,0,865},{3,0,865},{3,0,865},{0,15,0},{0,15,0},{0,15,0},{0,7,9},{0,6,337},{0,6,337},{6,31,7446},{4,31,1458},{4,24,2174},{3,22,1419},{3,31,11045},{0,27,3314},{0,22,195},
{0,17,4865},{0,20,12855},{0,16,6882},{9,31,3902},{6,31,8},{7,23,261},{5,22,474},{19,5,9669},{0,27,3314},{0,22,195},{0,17,4865},{31,6,9669},{0,17,4865},{3,31,1397},{3,31,1397},{3,31,1397},{3,19,1210},{0,30,2178},{0,20,61},{0,20,61},{0,12,545},{0,13,3645},{0,12,1634},{6,30,0},{6,30,0},{6,30,0},{6,18,0},{15,0,2178},{0,20,61},{0,20,61},{0,12,545},{30,0,2178},
{0,12,545},{25,5,2665},{6,31,8},{10,23,5},{0,22,74},{25,5,2665},{31,12,2665},{0,22,74},{0,18,2689},{31,12,2665},{0,18,2689},{3,0,1201},{3,0,1201},{3,0,1201},{3,0,1201},{0,18,0},{0,18,0},{0,18,0},{0,9,0},{0,7,482},{0,7,482},{7,31,8578},{4,31,2146},{5,25,2766},{3,23,1835},{4,31,11766},{0,28,3125},{0,23,94},{0,18,4610},{0,21,13238},{0,17,6837},{10,31,4130},
{7,31,49},{8,24,254},{6,23,474},{23,0,9670},{0,28,3125},{0,23,94},{0,18,4610},{30,8,9670},{0,18,4610},{4,31,1921},{4,31,1921},{4,31,1921},{3,21,1665},{0,31,2228},{0,22,18},{0,22,18},{0,14,425},{0,15,3987},{0,13,1677},{7,31,0},{7,31,0},{7,31,0},{7,19,0},{16,1,2178},{0,22,18},{0,22,18},{0,14,425},{31,1,2178},{0,14,425},{29,0,2665},{7,31,49},{10,24,13},
{0,23,58},{29,0,2665},{30,14,2665},{0,23,58},{0,19,2689},{30,14,2665},{0,19,2689},{3,0,1665},{3,0,1665},{3,0,1665},{3,0,1665},{0,21,0},{0,21,0},{0,21,0},{0,10,9},{0,9,640},{0,9,640},{7,31,10135},{5,31,3190},{5,26,3543},{4,24,2348},{5,31,12846},{0,30,2961},{0,25,30},{0,19,4421},{0,23,13846},{0,18,6930},{11,31,4345},{8,31,178},{9,25,261},{7,24,491},{24,1,9669},
{0,30,2961},{0,25,30},{0,19,4421},{31,9,9669},{0,19,4421},{5,31,2706},{5,31,2706},{5,31,2706},{4,22,2179},{1,31,2404},{0,24,0},{0,24,0},{0,15,337},{0,16,4356},{0,14,1833},{8,31,9},{8,31,9},{8,31,9},{8,20,0},{16,4,2178},{0,24,0},{0,24,0},{0,15,337},{28,4,2178},{0,15,337},{31,0,2669},{10,31,113},{12,25,5},{0,25,26},{31,0,2669},{31,15,2669},{0,25,26},
{0,20,2689},{31,15,2669},{0,20,2689},{4,0,2178},{4,0,2178},{4,0,2178},{4,0,2178},{0,24,0},{0,24,0},{0,24,0},{0,12,0},{0,9,865},{0,9,865},{8,31,11466},{6,31,4187},{6,27,4166},{4,25,2818},{6,31,13898},{0,31,2871},{0,26,15},{0,20,4226},{0,24,14214},{0,19,7006},{12,31,4590},{10,31,321},{10,26,261},{8,25,474},{26,0,9669},{0,31,2870},{0,26,14},{0,20,4225},{30,11,9669},
{0,20,4225},{6,31,3458},{6,31,3458},{6,31,3458},{5,23,2690},{2,31,2691},{0,26,14},{0,26,14},{0,16,261},{0,17,4595},{0,15,1953},{9,31,36},{9,31,36},{9,31,36},{9,21,0},{17,5,2178},{1,25,0},{1,25,0},{0,16,260},{31,4,2178},{0,16,260},{28,8,2665},{11,31,194},{13,26,5},{0,26,5},{28,8,2665},{24,20,2665},{0,26,5},{0,21,2689},{24,20,2665},{0,21,2689},{5,0,2689},
{5,0,2689},{5,0,2689},{5,0,2689},{0,27,1},{0,27,1},{0,27,1},{0,14,5},{0,11,1037},{0,11,1037},{9,31,12054},{7,31,4578},{7,28,4118},{5,26,2818},{7,31,14214},{1,31,3042},{1,27,15},{0,21,4099},{0,25,13683},{0,21,6322},{14,31,4858},{11,31,514},{11,27,261},{9,26,474},{27,1,9669},{2,31,3014},{1,27,14},{0,21,4018},{27,14,9669},{0,21,4018},{7,31,3554},{7,31,3554},{7,31,3554},
{6,24,2689},{3,31,2795},{1,27,14},{1,27,14},{1,17,261},{0,19,4269},{0,16,1517},{11,31,73},{11,31,73},{11,31,73},{10,22,0},{21,0,2178},{2,26,0},{2,26,0},{0,17,185},{30,6,2178},{0,17,185},{29,9,2665},{13,31,320},{14,27,5},{0,27,1},{29,9,2665},{27,20,2665},{0,27,1},{0,22,2689},{27,20,2665},{0,22,2689},{6,0,2689},{6,0,2689},{6,0,2689},{6,0,2689},{1,28,1},
{1,28,1},{1,28,1},{1,15,5},{0,13,797},{0,13,797},{11,31,12506},{8,31,5075},{8,29,4166},{6,27,2818},{8,31,14651},{3,31,3255},{2,28,23},{1,22,4099},{0,27,13238},{0,21,5650},{15,31,5054},{12,31,782},{12,28,254},{10,27,474},{29,0,9670},{3,31,3206},{2,28,22},{0,22,3829},{30,14,9670},{0,22,3829},{8,31,3706},{8,31,3706},{8,31,3706},{7,25,2689},{4,31,2946},{2,28,19},{2,28,19},
{2,18,261},{0,21,3906},{0,17,1106},{12,31,106},{12,31,106},{12,31,106},{11,23,0},{22,1,2178},{3,27,0},{3,27,0},{0,18,128},{31,7,2178},{0,18,128},{31,8,2665},{15,31,445},{14,28,13},{1,28,4},{31,8,2665},{30,20,2665},{1,28,4},{0,23,2689},{30,20,2665},{0,23,2689},{7,0,2689},{7,0,2689},{7,0,2689},{7,0,2689},{2,29,1},{2,29,1},{2,29,1},{2,15,10},{0,14,637},
{0,14,637},{12,31,13094},{10,31,5782},{9,30,4166},{7,28,2838},{10,31,15213},{4,31,3618},{3,29,25},{2,23,4101},{0,29,12686},{0,23,5075},{16,31,5378},{14,31,1172},{13,29,261},{11,28,491},{30,1,9669},{6,31,3469},{3,29,21},{0,23,3706},{31,15,9669},{0,23,3706},{9,31,3890},{9,31,3890},{9,31,3890},{8,26,2690},{6,31,3157},{3,29,24},{3,29,24},{3,19,254},{0,23,3619},{0,19,782},{13,31,145},
{13,31,145},{13,31,145},{12,24,0},{24,0,2178},{4,28,0},{4,28,0},{0,19,106},{24,12,2178},{0,19,106},{31,11,2669},{16,31,640},{16,29,5},{2,29,2},{31,11,2669},{31,21,2669},{2,29,2},{0,24,2689},{31,21,2669},{0,24,2689},{8,0,2689},{8,0,2689},{8,0,2689},{8,0,2689},{3,30,5},{3,30,5},{3,30,5},{3,17,8},{0,17,436},{0,17,436},{13,31,13718},{11,31,6325},{10,31,4166},
{8,29,2818},{11,31,15565},{6,31,4094},{4,30,15},{3,24,4118},{0,31,12355},{0,24,4578},{17,31,5738},{15,31,1517},{14,30,261},{12,29,474},{31,2,9669},{7,31,3761},{4,30,14},{0,24,3554},{30,17,9669},{0,24,3554},{10,31,4085},{10,31,4085},{10,31,4085},{9,27,2690},{7,31,3285},{4,30,14},{4,30,14},{4,20,261},{0,24,3330},{0,20,514},{14,31,208},{14,31,208},{14,31,208},{13,25,0},{25,1,2178},
{5,29,0},{5,29,0},{0,21,64},{27,12,2178},{0,21,64},{28,20,2665},{18,31,829},{17,30,5},{3,30,2},{28,20,2665},{28,24,2665},{3,30,2},{0,25,2689},{28,24,2665},{0,25,2689},{9,0,2689},{9,0,2689},{9,0,2689},{9,0,2689},{4,31,1},{4,31,1},{4,31,1},{4,18,5},{0,18,306},{0,18,306},{14,31,14378},{12,31,7006},{11,31,4226},{9,30,2818},{12,31,16054},{7,31,4578},{5,31,15},
{4,25,4099},{0,31,12051},{0,25,4089},{19,31,5970},{16,31,1953},{15,31,261},{13,30,474},{31,5,9669},{10,31,4081},{5,31,14},{0,25,3413},{31,18,9669},{0,25,3413},{11,31,4225},{11,31,4225},{11,31,4225},{10,28,2689},{8,31,3476},{5,31,14},{5,31,14},{5,21,261},{0,26,3091},{0,21,289},{15,31,260},{15,31,260},{15,31,260},{14,26,0},{27,0,2178},{6,30,0},{6,30,0},{0,22,25},{30,12,2178},
{0,22,25},{29,21,2665},{20,31,1037},{18,31,5},{4,31,1},{29,21,2665},{31,24,2665},{4,31,1},{0,26,2689},{31,24,2665},{0,26,2689},{10,0,2689},{10,0,2689},{10,0,2689},{10,0,2689},{5,31,10},{5,31,10},{5,31,10},{5,19,5},{0,20,194},{0,20,194},{15,31,13557},{13,31,7094},{12,31,4421},{10,31,2769},{13,31,15228},{8,31,4270},{6,31,46},{5,26,3476},{0,31,11020},{0,26,3108},{20,31,5400},
{17,31,1931},{16,31,337},{15,30,320},{31,7,8712},{11,31,3630},{7,31,0},{0,26,2667},{31,19,8712},{0,26,2667},{12,31,4421},{12,31,4421},{12,31,4421},{11,29,2689},{9,31,3722},{6,31,46},{6,31,46},{6,22,261},{0,27,2882},{0,22,173},{16,31,337},{16,31,337},{16,31,337},{15,27,0},{28,1,2178},{7,31,0},{7,31,0},{0,23,9},{31,13,2178},{0,23,9},{31,19,2178},{22,31,881},{19,31,0},
{7,31,0},{31,19,2178},{31,25,2178},{7,31,0},{0,27,2178},{31,25,2178},{0,27,2178},{11,0,2689},{11,0,2689},{11,0,2689},{11,0,2689},{6,31,37},{6,31,37},{6,31,37},{6,19,10},{0,22,109},{0,22,109},{16,31,12678},{14,31,7003},{13,31,4693},{12,31,2714},{15,31,14026},{10,31,3943},{8,31,94},{6,26,2766},{2,31,10074},{0,27,2146},{20,31,4698},{19,31,1746},{17,31,464},{16,30,164},{29,13,7578},
{12,31,3090},{9,31,20},{0,27,1921},{31,20,7578},{0,27,1921},{13,31,4693},{13,31,4693},{13,31,4693},{12,30,2690},{11,31,3939},{8,31,94},{8,31,94},{7,23,254},{0,29,2650},{0,24,49},{17,31,464},{17,31,464},{17,31,464},{16,28,0},{28,4,2178},{9,31,20},{9,31,20},{0,24,0},{28,16,2178},{0,24,0},{31,21,1625},{23,31,653},{21,31,4},{10,31,1},{31,21,1625},{31,26,1625},{10,31,1},
{0,27,1665},{31,26,1625},{0,27,1665},{12,0,2689},{12,0,2689},{12,0,2689},{12,0,2689},{8,31,58},{8,31,58},{8,31,58},{7,21,8},{0,24,49},{0,24,49},{17,31,12042},{15,31,6882},{15,31,4946},{13,31,2690},{16,31,13127},{11,31,3615},{9,31,229},{7,27,2174},{3,31,9313},{0,27,1458},{21,31,4150},{19,31,1634},{19,31,545},{17,30,89},{31,11,6661},{14,31,2654},{11,31,61},{0,28,1397},{31,21,6661},
{0,28,1397},{15,31,4946},{15,31,4946},{15,31,4946},{13,31,2690},{12,31,4170},{9,31,229},{9,31,229},{8,24,261},{0,31,2520},{0,25,10},{19,31,545},{19,31,545},{19,31,545},{17,29,0},{29,5,2178},{11,31,61},{11,31,61},{1,25,0},{31,16,2178},{1,25,0},{31,22,1201},{24,31,482},{22,31,1},{13,31,1},{31,22,1201},{30,27,1201},{13,31,1},{0,28,1201},{30,27,1201},{0,28,1201},{13,0,2689},
{13,0,2689},{13,0,2689},{13,0,2689},{9,31,85},{9,31,85},{9,31,85},{8,22,5},{0,25,10},{0,25,10},{18,31,11474},{16,31,6867},{15,31,5138},{14,31,2725},{16,31,12279},{12,31,3410},{10,31,419},{8,27,1718},{4,31,8678},{0,28,905},{23,31,3626},{20,31,1550},{20,31,650},{18,31,25},{29,17,5829},{15,31,2306},{12,31,130},{0,28,901},{27,24,5829},{0,28,901},{15,31,5138},{15,31,5138},{15,31,5138},
{14,31,2725},{13,31,4452},{10,31,419},{10,31,419},{9,25,261},{0,31,2520},{1,26,10},{20,31,650},{20,31,650},{20,31,650},{18,30,0},{31,4,2178},{12,31,130},{12,31,130},{2,26,0},{30,18,2178},{2,26,0},{31,23,845},{25,31,353},{23,31,9},{16,31,0},{31,23,845},{31,27,845},{16,31,0},{0,28,865},{31,27,845},{0,28,865},{14,0,2689},{14,0,2689},{14,0,2689},{14,0,2689},{10,31,130},
{10,31,130},{10,31,130},{9,23,5},{0,27,1},{0,27,1},{19,31,10774},{17,31,6962},{16,31,5378},{15,31,2810},{17,31,11598},{12,31,3346},{12,31,642},{9,28,1222},{6,31,8113},{0,29,570},{23,31,3146},{22,31,1539},{21,31,773},{19,31,0},{31,15,5082},{16,31,2034},{14,31,221},{0,29,554},{31,23,5082},{0,29,554},{16,31,5378},{16,31,5378},{16,31,5378},{15,31,2810},{15,31,4746},{12,31,642},{12,31,642},
{10,26,261},{2,31,2714},{2,27,10},{21,31,773},{21,31,773},{21,31,773},{19,31,0},{31,7,2178},{14,31,221},{14,31,221},{3,27,0},{31,19,2178},{3,27,0},{31,25,545},{26,31,221},{25,31,0},{19,31,0},{31,25,545},{31,28,545},{19,31,0},{0,29,545},{31,28,545},{0,29,545},{15,0,2689},{15,0,2689},{15,0,2689},{15,0,2689},{11,31,170},{11,31,170},{11,31,170},{10,23,10},{1,28,1},
{1,28,1},{20,31,10225},{18,31,7025},{18,31,5729},{16,31,2978},{19,31,10792},{14,31,3283},{13,31,982},{10,29,861},{8,31,7588},{2,29,325},{24,31,2729},{23,31,1460},{22,31,932},{20,31,36},{29,21,4344},{19,31,1746},{16,31,353},{2,29,321},{31,24,4344},{2,29,321},{18,31,5729},{18,31,5729},{18,31,5729},{16,31,2978},{16,31,5028},{13,31,982},{13,31,982},{11,27,254},{4,31,2981},{3,28,10},{22,31,932},
{22,31,932},{22,31,932},{20,31,36},{28,16,2178},{16,31,353},{16,31,353},{4,28,0},{24,24,2178},{4,28,0},{31,27,290},{27,31,125},{27,31,4},{22,31,1},{31,27,290},{31,29,290},{22,31,1},{0,29,320},{31,29,290},{0,29,320},{16,0,2689},{16,0,2689},{16,0,2689},{16,0,2689},{12,31,245},{12,31,245},{12,31,245},{11,25,8},{2,29,5},{2,29,5},{20,31,9825},{19,31,7014},{19,31,5925},
{17,31,3218},{20,31,10245},{15,31,3285},{14,31,1330},{11,29,612},{10,31,7225},{3,30,133},{25,31,2467},{24,31,1470},{23,31,1040},{21,31,144},{31,19,3779},{20,31,1580},{18,31,500},{3,30,129},{31,25,3779},{3,30,129},{19,31,5925},{19,31,5925},{19,31,5925},{17,31,3218},{17,31,5346},{14,31,1330},{14,31,1330},{12,28,261},{6,31,3267},{4,29,10},{23,31,1040},{23,31,1040},{23,31,1040},{21,31,144},{29,17,2178},
{18,31,500},{18,31,500},{5,29,0},{27,24,2178},{5,29,0},{31,28,128},{29,31,61},{28,31,0},{25,31,1},{31,28,128},{30,30,128},{25,31,1},{0,30,128},{30,30,128},{0,30,128},{17,0,2689},{17,0,2689},{17,0,2689},{17,0,2689},{13,31,338},{13,31,338},{13,31,338},{12,26,5},{3,30,5},{3,30,5},{21,31,9523},{20,31,7109},{20,31,6209},{18,31,3473},{20,31,9749},{16,31,3410},{15,31,1717},
{12,30,401},{11,31,6964},{4,31,65},{26,31,2273},{25,31,1539},{24,31,1189},{23,31,260},{29,25,3299},{22,31,1490},{20,31,680},{6,30,64},{27,28,3299},{6,30,64},{20,31,6209},{20,31,6209},{20,31,6209},{18,31,3473},{18,31,5700},{15,31,1717},{15,31,1717},{13,29,261},{8,31,3587},{5,30,10},{24,31,1189},{24,31,1189},{24,31,1189},{23,31,260},{31,16,2178},{20,31,680},{20,31,680},{6,30,0},{30,24,2178},
{6,30,0},{31,30,34},{30,31,13},{30,31,4},{28,31,0},{31,30,34},{30,31,34},{28,31,0},{0,30,64},{30,31,34},{0,30,64},{18,0,2689},{18,0,2689},{18,0,2689},{18,0,2689},{15,31,421},{15,31,421},{15,31,421},{13,27,5},{4,31,1},{4,31,1},{23,31,9201},{21,31,7314},{21,31,6530},{19,31,3778},{21,31,9420},{18,31,3652},{16,31,2193},{14,30,325},{14,31,6804},{6,31,10},{27,31,2057},
{26,31,1605},{25,31,1378},{24,31,442},{31,23,2904},{23,31,1452},{22,31,881},{7,31,0},{31,27,2904},{7,31,0},{21,31,6530},{21,31,6530},{21,31,6530},{19,31,3778},{19,31,5956},{16,31,2193},{16,31,2193},{14,30,261},{10,31,3957},{6,31,10},{25,31,1378},{25,31,1378},{25,31,1378},{24,31,442},{31,19,2178},{22,31,881},{22,31,881},{7,31,0},{31,25,2178},{7,31,0},{31,31,0},{31,31,0},{31,31,0},
{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{19,0,2689},{19,0,2689},{19,0,2689},{19,0,2689},{16,31,512},{16,31,512},{16,31,512},{14,27,10},{6,31,10},{6,31,10},{23,31,8049},{22,31,6550},{22,31,5925},{20,31,3589},{23,31,8137},{19,31,3220},{18,31,2050},{15,31,173},{15,31,5805},{8,31,52},{27,31,1544},{27,31,1181},{26,31,1040},{25,31,353},{31,24,2166},
{24,31,1083},{23,31,653},{10,31,1},{30,28,2166},{10,31,1},{22,31,5925},{22,31,5925},{22,31,5925},{20,31,3589},{20,31,5209},{18,31,2050},{18,31,2050},{15,30,117},{12,31,3405},{8,31,52},{26,31,1040},{26,31,1040},{26,31,1040},{25,31,353},{31,21,1625},{23,31,653},{23,31,653},{10,31,1},{31,26,1625},{10,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},
{0,31,0},{31,31,0},{0,31,0},{20,0,2689},{20,0,2689},{20,0,2689},{20,0,2689},{17,31,625},{17,31,625},{17,31,625},{15,29,8},{8,31,52},{8,31,52},{24,31,7177},{23,31,5845},{23,31,5361},{21,31,3473},{23,31,7033},{20,31,2945},{19,31,1907},{16,31,65},{16,31,5026},{10,31,117},{28,31,1137},{27,31,861},{27,31,740},{26,31,260},{29,29,1601},{26,31,833},{24,31,482},{13,31,1},{31,28,1601},
{13,31,1},{23,31,5361},{23,31,5361},{23,31,5361},{21,31,3473},{21,31,4661},{19,31,1907},{19,31,1907},{16,31,65},{14,31,2997},{10,31,117},{27,31,740},{27,31,740},{27,31,740},{26,31,260},{31,22,1201},{24,31,482},{24,31,482},{13,31,1},{30,27,1201},{13,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{21,0,2689},
{21,0,2689},{21,0,2689},{21,0,2689},{18,31,754},{18,31,754},{18,31,754},{16,30,5},{10,31,117},{10,31,117},{24,31,6393},{24,31,5310},{23,31,4865},{22,31,3314},{24,31,6146},{20,31,2737},{20,31,1837},{17,31,5},{18,31,4381},{11,31,212},{28,31,801},{28,31,609},{28,31,545},{27,31,185},{31,26,1121},{26,31,561},{25,31,353},{16,31,0},{30,29,1121},{16,31,0},{23,31,4865},{23,31,4865},{23,31,4865},
{22,31,3314},{22,31,4181},{20,31,1837},{20,31,1837},{17,31,5},{15,31,2621},{11,31,212},{28,31,545},{28,31,545},{28,31,545},{27,31,185},{31,23,845},{25,31,353},{25,31,353},{16,31,0},{31,27,845},{16,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{22,0,2689},{22,0,2689},{22,0,2689},{22,0,2689},{19,31,850},
{19,31,850},{19,31,850},{17,31,5},{11,31,212},{11,31,212},{25,31,5683},{24,31,4782},{24,31,4421},{23,31,3173},{24,31,5314},{22,31,2563},{21,31,1844},{18,31,10},{18,31,3757},{13,31,338},{29,31,531},{28,31,401},{28,31,337},{27,31,121},{31,27,726},{27,31,363},{26,31,221},{19,31,0},{31,29,726},{19,31,0},{24,31,4421},{24,31,4421},{24,31,4421},{23,31,3173},{23,31,3657},{21,31,1844},{21,31,1844},
{18,31,10},{16,31,2321},{13,31,338},{28,31,337},{28,31,337},{28,31,337},{27,31,121},{31,25,545},{26,31,221},{26,31,221},{19,31,0},{31,28,545},{19,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{23,0,2689},{23,0,2689},{23,0,2689},{23,0,2689},{20,31,985},{20,31,985},{20,31,985},{18,31,10},{13,31,338},
{13,31,338},{2,31,10560},{0,30,1586},{0,22,173},{0,18,3826},{0,28,18065},{0,19,12194},{0,17,6081},{0,11,14098},{0,13,19386},{0,11,14998},{3,31,10216},{0,30,1586},{0,22,173},{0,18,3826},{12,4,18065},{0,19,12194},{0,17,6081},{0,11,14098},{28,0,18065},{0,11,14098},{0,17,1},{0,17,1},{0,17,1},{0,9,4},{0,9,1105},{0,7,442},{0,7,442},{0,4,680},{0,4,1230},{0,4,801},{0,17,1},
{0,17,1},{0,17,1},{0,9,4},{4,1,1105},{0,7,442},{0,7,442},{0,4,680},{9,0,1105},{0,4,680},{21,0,9248},{0,30,1586},{0,22,173},{0,18,3826},{21,0,9248},{30,6,9248},{0,18,3826},{0,14,9248},{30,6,9248},{0,14,9248},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,31,11328},{0,31,1341},{0,23,68},
{0,19,3626},{0,30,19334},{0,20,12611},{0,18,6099},{0,12,14756},{0,14,20886},{0,12,15845},{3,31,10792},{0,31,1341},{0,23,68},{0,19,3626},{15,0,19334},{0,20,12611},{0,18,6099},{0,12,14756},{30,0,19334},{0,12,14756},{0,20,0},{0,20,0},{0,20,0},{0,10,1},{0,10,1513},{0,8,605},{0,8,605},{0,5,925},{0,5,1682},{0,4,1089},{0,20,0},{0,20,0},{0,20,0},{0,10,1},{5,0,1513},
{0,8,605},{0,8,605},{0,5,925},{10,0,1513},{0,5,925},{22,1,9248},{0,31,1341},{0,23,68},{0,19,3626},{22,1,9248},{31,7,9248},{0,19,3626},{0,15,9248},{31,7,9248},{0,15,9248},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{3,31,12200},{0,31,1325},{0,24,10},{0,20,3365},{0,31,20705},{0,21,13009},{0,18,6227},
{0,13,15441},{0,15,22455},{0,13,16666},{4,31,11489},{0,31,1325},{0,24,10},{0,20,3365},{15,2,20689},{0,21,13009},{0,18,6227},{0,13,15441},{30,1,20689},{0,13,15441},{0,23,0},{0,23,0},{0,23,0},{0,11,9},{0,11,1989},{0,9,772},{0,9,772},{0,5,1213},{0,5,2194},{0,5,1382},{0,23,0},{0,23,0},{0,23,0},{0,11,9},{6,0,1985},{0,9,772},{0,9,772},{0,5,1213},{10,1,1985},
{0,5,1213},{23,2,9248},{0,31,1325},{0,24,10},{0,20,3365},{23,2,9248},{30,9,9248},{0,20,3365},{0,16,9250},{30,9,9248},{0,16,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{3,31,13288},{0,31,1565},{0,25,10},{0,22,3077},{1,31,22214},{0,23,13555},{0,20,6310},{0,14,16176},{0,16,24130},{0,13,17562},{4,31,12385},
{0,31,1565},{0,25,10},{0,22,3077},{17,0,22129},{0,23,13555},{0,20,6310},{0,14,16176},{30,2,22129},{0,14,16176},{0,26,0},{0,26,0},{0,26,0},{0,13,0},{0,13,2521},{0,10,1018},{0,10,1018},{0,6,1508},{0,6,2801},{0,6,1764},{0,26,0},{0,26,0},{0,26,0},{0,13,0},{6,1,2521},{0,10,1018},{0,10,1018},{0,6,1508},{13,0,2521},{0,6,1508},{23,5,9248},{0,31,1565},{0,25,10},
{0,22,3077},{23,5,9248},{31,10,9248},{0,22,3077},{0,17,9250},{31,10,9248},{0,17,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{4,31,14788},{0,31,2141},{0,27,72},{0,22,2825},{1,31,24140},{0,25,14114},{0,21,6323},{0,14,17202},{0,17,26063},{0,14,18646},{5,31,13521},{1,31,2100},{0,27,72},{0,22,2825},{16,4,23851},
{0,25,14114},{0,21,6323},{0,14,17202},{28,4,23851},{0,14,17202},{0,29,1},{0,29,1},{0,29,1},{0,15,4},{0,15,3202},{0,12,1285},{0,12,1285},{0,7,1973},{0,7,3569},{0,6,2241},{0,29,1},{0,29,1},{0,29,1},{0,15,4},{7,1,3200},{0,12,1285},{0,12,1285},{0,7,1973},{11,2,3200},{0,7,1973},{27,0,9248},{3,31,1885},{2,26,8},{0,22,2825},{27,0,9248},{30,12,9248},{0,22,2825},
{0,18,9248},{30,12,9248},{0,18,9248},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{4,31,16228},{1,31,2836},{0,28,170},{0,24,2612},{2,31,25971},{0,25,14594},{0,22,6473},{0,16,18144},{0,17,27951},{0,14,19830},{7,31,14675},{2,31,2706},{1,27,149},{0,24,2612},{19,0,25472},{0,25,14594},{0,22,6473},{0,16,18144},{30,4,25472},
{0,16,18144},{0,31,9},{0,31,9},{0,31,9},{0,16,0},{0,16,3872},{0,13,1514},{0,13,1514},{0,7,2405},{0,7,4305},{0,7,2766},{0,31,9},{0,31,9},{0,31,9},{0,16,0},{8,0,3872},{0,13,1514},{0,13,1514},{0,7,2405},{16,0,3872},{0,7,2405},{28,1,9248},{4,31,2210},{3,27,8},{0,24,2612},{28,1,9248},{31,13,9248},{0,24,2612},{0,19,9248},{31,13,9248},{0,19,9248},{0,0,0},
{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{4,31,17796},{2,31,3702},{1,29,270},{0,25,2361},{3,31,27563},{0,27,14944},{0,23,6470},{0,17,18681},{0,19,29556},{0,16,20628},{7,31,15635},{3,31,3425},{1,29,206},{0,25,2361},{20,0,26744},{0,27,14944},{0,23,6470},{0,17,18681},{24,8,26744},{0,17,18681},{0,31,125},{0,31,125},{0,31,125},
{0,18,8},{0,18,4418},{0,14,1696},{0,14,1696},{0,8,2664},{0,8,4952},{0,8,3148},{1,31,72},{1,31,72},{1,31,72},{0,18,8},{9,0,4418},{0,14,1696},{0,14,1696},{0,8,2664},{18,0,4418},{0,8,2664},{30,0,9248},{6,31,2557},{3,28,10},{0,25,2357},{30,0,9248},{30,15,9248},{0,25,2357},{0,20,9250},{30,15,9248},{0,20,9250},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,1,0},
{0,1,0},{0,1,0},{0,0,4},{0,0,4},{0,0,4},{5,31,19090},{2,31,4598},{1,30,406},{0,26,2220},{3,31,28187},{0,29,14347},{0,25,5609},{0,18,18144},{0,20,29961},{0,17,20301},{8,31,16082},{4,31,4019},{2,30,206},{0,26,2220},{21,1,26744},{0,29,14347},{0,25,5609},{0,18,18144},{27,8,26744},{0,18,18144},{1,31,264},{1,31,264},{1,31,264},{1,19,72},{0,21,4418},{0,16,1412},{0,16,1412},
{0,10,2420},{0,9,5122},{0,9,2997},{2,31,117},{2,31,117},{2,31,117},{1,19,8},{10,1,4418},{0,16,1412},{0,16,1412},{0,10,2420},{21,0,4418},{0,10,2420},{31,1,9248},{8,31,2929},{4,29,10},{0,26,2120},{31,1,9248},{27,18,9248},{0,26,2120},{0,21,9250},{27,18,9248},{0,21,9250},{1,0,68},{1,0,68},{1,0,68},{1,0,68},{0,4,1},{0,4,1},{0,4,1},{0,2,0},{0,2,25},
{0,2,25},{7,31,20669},{3,31,5786},{2,31,625},{1,27,2283},{4,31,29033},{0,30,13795},{0,26,4770},{0,18,17667},{0,21,30425},{0,18,20068},{9,31,16691},{5,31,4841},{3,31,205},{1,27,2219},{23,0,26747},{0,30,13795},{0,26,4770},{0,18,17667},{30,8,26747},{0,18,17667},{2,31,589},{2,31,589},{2,31,589},{1,20,186},{0,24,4418},{0,18,1125},{0,18,1125},{0,10,2186},{0,11,5390},{0,10,2915},{3,31,169},
{3,31,169},{3,31,169},{2,20,13},{12,0,4418},{0,18,1125},{0,18,1125},{0,10,2186},{24,0,4418},{0,10,2186},{31,4,9248},{10,31,3380},{6,30,8},{0,27,1954},{31,4,9248},{30,18,9248},{0,27,1954},{0,22,9248},{30,18,9248},{0,22,9248},{1,0,185},{1,0,185},{1,0,185},{1,0,185},{0,7,0},{0,7,0},{0,7,0},{0,3,9},{0,3,73},{0,3,73},{7,31,21577},{4,31,6797},{3,31,985},
{1,28,2226},{5,31,29436},{0,31,12990},{0,27,4027},{0,20,16925},{0,23,30436},{0,18,19620},{11,31,16645},{7,31,5260},{4,31,232},{2,28,2141},{24,1,26259},{0,31,12990},{0,27,4027},{0,20,16925},{31,9,26259},{0,20,16925},{3,31,985},{3,31,985},{3,31,985},{2,21,378},{0,27,4418},{0,20,905},{0,20,905},{0,12,2005},{0,12,5635},{0,11,2950},{4,31,232},{4,31,232},{4,31,232},{3,21,13},{13,1,4418},
{0,20,905},{0,20,905},{0,12,2005},{27,0,4418},{0,12,2005},{31,7,8980},{11,31,3601},{7,31,4},{0,28,1665},{31,7,8980},{31,19,8980},{0,28,1665},{0,23,8980},{31,19,8980},{0,23,8980},{2,0,377},{2,0,377},{2,0,377},{2,0,377},{0,10,0},{0,10,0},{0,10,0},{0,5,0},{0,4,146},{0,4,146},{7,31,21193},{4,31,7053},{3,31,1481},{2,28,2093},{6,31,28313},{0,31,11406},{0,28,2965},
{0,20,15021},{0,23,28900},{0,19,17995},{11,31,15381},{7,31,4876},{5,31,325},{3,28,1786},{25,1,24371},{0,31,11406},{0,28,2965},{0,20,15021},{27,12,24371},{0,20,15021},{3,31,1481},{3,31,1481},{3,31,1481},{2,23,621},{0,30,4418},{0,21,680},{0,21,680},{0,13,1780},{0,13,5885},{0,12,2950},{5,31,325},{5,31,325},{5,31,325},{4,22,8},{15,0,4418},{0,21,680},{0,21,680},{0,13,1780},{30,0,4418},
{0,13,1780},{31,8,7938},{12,31,3188},{8,31,0},{0,29,1156},{31,8,7938},{30,20,7938},{0,29,1156},{0,23,7956},{30,20,7938},{0,23,7956},{2,0,617},{2,0,617},{2,0,617},{2,0,617},{0,13,0},{0,13,0},{0,13,0},{0,6,9},{0,5,233},{0,5,233},{8,31,20825},{5,31,7494},{4,31,2089},{2,29,1970},{7,31,27269},{0,31,10078},{0,28,2021},{0,21,13204},{0,25,27384},{0,20,16398},{12,31,14148},
{8,31,4562},{7,31,392},{4,29,1445},{26,1,22568},{0,31,10078},{0,28,2021},{0,21,13204},{31,11,22568},{0,21,13204},{4,31,2089},{4,31,2089},{4,31,2089},{3,23,946},{0,31,4468},{0,23,521},{0,23,521},{0,14,1573},{0,15,6227},{0,13,2909},{7,31,392},{7,31,392},{7,31,392},{5,23,8},{16,1,4418},{0,23,521},{0,23,521},{0,14,1573},{31,1,4418},{0,14,1573},{31,10,6964},{14,31,2785},{10,31,4},
{0,29,740},{31,10,6964},{31,20,6964},{0,29,740},{0,24,6970},{31,20,6964},{0,24,6970},{3,0,937},{3,0,937},{3,0,937},{3,0,937},{0,16,1},{0,16,1},{0,16,1},{0,8,1},{0,6,377},{0,6,377},{8,31,20717},{6,31,8012},{5,31,2958},{3,29,2057},{7,31,26243},{0,31,8890},{0,29,1154},{0,22,11325},{0,25,25854},{0,21,14671},{13,31,12926},{10,31,4313},{8,31,485},{5,29,1106},{25,5,20642},
{0,31,8890},{0,29,1154},{0,22,11325},{31,12,20642},{0,22,11325},{5,31,2958},{5,31,2958},{5,31,2958},{3,25,1361},{1,31,4644},{0,25,337},{0,25,337},{0,15,1429},{0,16,6596},{0,14,2981},{8,31,485},{8,31,485},{8,31,485},{6,24,13},{16,4,4418},{0,25,337},{0,25,337},{0,15,1429},{28,4,4418},{0,15,1429},{31,11,5941},{15,31,2377},{11,31,1},{0,30,388},{31,11,5941},{31,21,5941},{0,30,388},
{0,24,5953},{31,21,5941},{0,24,5953},{3,0,1360},{3,0,1360},{3,0,1360},{3,0,1360},{0,19,0},{0,19,0},{0,19,0},{0,10,4},{0,8,548},{0,8,548},{9,31,20713},{7,31,8575},{5,31,3806},{3,30,2260},{7,31,25603},{0,31,8106},{0,29,642},{0,22,9805},{0,26,24678},{0,21,13359},{15,31,11882},{11,31,4006},{9,31,596},{6,29,818},{28,1,19021},{2,31,8066},{0,29,642},{0,22,9805},{31,13,19021},
{0,22,9805},{5,31,3806},{5,31,3806},{5,31,3806},{4,26,1819},{2,31,4962},{0,27,232},{0,27,232},{0,16,1268},{0,17,6926},{0,15,3126},{9,31,596},{9,31,596},{9,31,596},{7,25,13},{17,5,4418},{0,27,232},{0,27,232},{0,16,1268},{31,4,4418},{0,16,1268},{31,13,5101},{16,31,2042},{13,31,4},{0,30,164},{31,13,5101},{31,22,5101},{0,30,164},{0,25,5105},{31,22,5101},{0,25,5105},{4,0,1818},
{4,0,1818},{4,0,1818},{4,0,1818},{0,22,0},{0,22,0},{0,22,0},{0,11,0},{0,9,697},{0,9,697},{10,31,20905},{7,31,9247},{6,31,4787},{4,30,2547},{8,31,25042},{1,31,7537},{0,30,264},{0,23,8449},{0,27,23521},{0,22,12141},{15,31,10794},{12,31,3786},{10,31,725},{8,29,621},{29,1,17485},{3,31,7274},{0,30,264},{0,23,8449},{27,16,17485},{0,23,8449},{6,31,4787},{6,31,4787},{6,31,4787},
{4,27,2323},{3,31,5386},{0,29,130},{0,29,130},{0,18,1096},{0,19,7364},{0,17,3225},{10,31,725},{10,31,725},{10,31,725},{8,26,8},{21,0,4418},{0,29,130},{0,29,130},{0,18,1096},{30,6,4418},{0,18,1096},{31,14,4325},{18,31,1737},{14,31,1},{0,31,64},{31,14,4325},{30,23,4325},{0,31,64},{0,25,4337},{30,23,4325},{0,25,4337},{4,0,2314},{4,0,2314},{4,0,2314},{4,0,2314},{0,25,0},
{0,25,0},{0,25,0},{0,12,4},{0,10,925},{0,10,925},{11,31,21021},{8,31,10106},{7,31,5819},{4,30,3027},{8,31,24722},{2,31,7042},{0,31,81},{0,24,7169},{0,29,22467},{0,22,11133},{16,31,9869},{12,31,3594},{11,31,821},{8,30,420},{30,1,16034},{4,31,6558},{0,31,81},{0,24,7169},{31,15,16034},{0,24,7169},{7,31,5819},{7,31,5819},{7,31,5819},{5,28,2915},{3,31,5962},{0,30,72},{0,30,72},
{0,18,968},{0,20,7781},{0,17,3305},{11,31,821},{11,31,821},{11,31,821},{9,27,8},{22,1,4418},{0,30,72},{0,30,72},{0,18,968},{31,7,4418},{0,18,968},{31,15,3617},{19,31,1450},{15,31,9},{0,31,0},{31,15,3617},{31,23,3617},{0,31,0},{0,26,3617},{31,23,3617},{0,26,3617},{5,0,2906},{5,0,2906},{5,0,2906},{5,0,2906},{0,28,1},{0,28,1},{0,28,1},{0,14,0},{0,11,1156},
{0,11,1156},{11,31,21381},{8,31,11186},{7,31,7169},{5,31,3633},{9,31,24543},{3,31,6762},{0,31,81},{0,24,5819},{0,29,21333},{0,23,10106},{17,31,8955},{14,31,3433},{12,31,980},{10,30,242},{29,5,14504},{6,31,5834},{1,31,74},{0,24,5819},{31,16,14504},{0,24,5819},{7,31,7169},{7,31,7169},{7,31,7169},{5,30,3618},{4,31,6757},{0,31,81},{0,31,81},{0,20,821},{0,21,8245},{0,18,3531},{12,31,980},
{12,31,980},{12,31,980},{10,28,13},{24,0,4418},{1,31,74},{1,31,74},{0,20,821},{24,12,4418},{0,20,821},{29,21,2888},{20,31,1156},{17,31,1},{3,31,1},{29,21,2888},{31,24,2888},{3,31,1},{0,26,2906},{31,24,2888},{0,26,2906},{5,0,3617},{5,0,3617},{5,0,3617},{5,0,3617},{0,31,0},{0,31,0},{0,31,0},{0,15,9},{0,13,1421},{0,13,1421},{12,31,21949},{9,31,12367},{8,31,8449},
{6,31,4338},{10,31,24600},{3,31,6650},{1,31,298},{0,25,4698},{0,30,20575},{0,24,9247},{17,31,8219},{15,31,3236},{13,31,1157},{11,30,122},{31,3,13235},{7,31,5260},{3,31,145},{0,25,4698},{31,17,13235},{0,25,4698},{8,31,8449},{8,31,8449},{8,31,8449},{6,31,4338},{5,31,7667},{1,31,298},{1,31,298},{0,21,680},{0,23,8779},{0,19,3786},{13,31,1157},{13,31,1157},{13,31,1157},{11,29,13},{25,1,4418},
{3,31,145},{3,31,145},{0,21,680},{27,12,4418},{0,21,680},{31,19,2314},{22,31,949},{19,31,4},{6,31,1},{31,19,2314},{31,25,2314},{6,31,1},{0,27,2314},{31,25,2314},{0,27,2314},{6,0,4337},{6,0,4337},{6,0,4337},{6,0,4337},{0,31,64},{0,31,64},{0,31,64},{0,17,0},{0,13,1709},{0,13,1709},{12,31,22557},{10,31,13585},{9,31,9926},{6,31,5186},{11,31,24636},{3,31,6794},{2,31,692},
{0,26,3723},{0,31,19836},{0,25,8442},{19,31,7417},{16,31,3126},{15,31,1268},{12,31,68},{29,9,12051},{8,31,4762},{4,31,232},{0,26,3723},{27,20,12051},{0,26,3723},{9,31,9926},{9,31,9926},{9,31,9926},{6,31,5186},{5,31,8691},{2,31,692},{2,31,692},{0,22,557},{0,25,9284},{0,21,3929},{15,31,1268},{15,31,1268},{15,31,1268},{12,30,8},{27,0,4418},{4,31,232},{4,31,232},{0,22,557},{30,12,4418},
{0,22,557},{31,20,1800},{22,31,725},{20,31,0},{9,31,1},{31,20,1800},{30,26,1800},{9,31,1},{0,27,1818},{30,26,1800},{0,27,1818},{6,0,5105},{6,0,5105},{6,0,5105},{6,0,5105},{1,31,185},{1,31,185},{1,31,185},{0,18,9},{0,15,2042},{0,15,2042},{12,31,23421},{11,31,14850},{9,31,11462},{7,31,6149},{11,31,24860},{4,31,7053},{2,31,1236},{0,26,2891},{0,31,19260},{0,25,7818},{19,31,6761},
{17,31,3107},{16,31,1429},{13,31,8},{31,7,10952},{10,31,4300},{6,31,353},{0,26,2891},{31,19,10952},{0,26,2891},{9,31,11462},{9,31,11462},{9,31,11462},{7,31,6149},{7,31,9845},{2,31,1236},{2,31,1236},{0,23,485},{0,25,9764},{0,21,4185},{16,31,1429},{16,31,1429},{16,31,1429},{13,31,8},{28,1,4418},{6,31,353},{6,31,353},{0,23,485},{31,13,4418},{0,23,485},{31,22,1354},{23,31,548},{22,31,4},
{12,31,0},{31,22,1354},{30,27,1354},{12,31,0},{0,28,1360},{30,27,1354},{0,28,1360},{7,0,5953},{7,0,5953},{7,0,5953},{7,0,5953},{1,31,425},{1,31,425},{1,31,425},{0,20,1},{0,17,2372},{0,17,2372},{13,31,24507},{11,31,16398},{10,31,13349},{8,31,7460},{11,31,25418},{4,31,7647},{3,31,2021},{0,27,2089},{0,31,18918},{0,26,7302},{20,31,6098},{18,31,3037},{17,31,1640},{14,31,25},{29,13,9818},
{12,31,3874},{8,31,521},{0,27,2089},{31,20,9818},{0,27,2089},{10,31,13349},{10,31,13349},{10,31,13349},{8,31,7460},{7,31,11195},{3,31,2021},{3,31,2021},{0,24,392},{0,27,10472},{0,23,4562},{17,31,1640},{17,31,1640},{17,31,1640},{14,31,25},{28,4,4418},{8,31,521},{8,31,521},{0,24,392},{28,16,4418},{0,24,392},{31,23,925},{24,31,386},{23,31,1},{15,31,1},{31,23,925},{31,27,925},{15,31,1},
{0,28,937},{31,27,925},{0,28,937},{7,0,6970},{7,0,6970},{7,0,6970},{7,0,6970},{2,31,785},{2,31,785},{2,31,785},{0,22,4},{0,17,2741},{0,17,2741},{14,31,25663},{12,31,17995},{11,31,15021},{8,31,8740},{12,31,26003},{6,31,8399},{3,31,2965},{0,28,1481},{0,31,18886},{0,26,7014},{21,31,5634},{19,31,2950},{18,31,1853},{16,31,72},{31,11,8901},{14,31,3578},{10,31,698},{0,28,1481},{31,21,8901},
{0,28,1481},{11,31,15021},{11,31,15021},{11,31,15021},{8,31,8740},{8,31,12646},{3,31,2965},{3,31,2965},{0,26,292},{0,29,11051},{0,24,4876},{18,31,1853},{18,31,1853},{18,31,1853},{16,31,72},{29,5,4418},{10,31,698},{10,31,698},{0,26,292},{31,16,4418},{0,26,292},{31,25,613},{26,31,245},{25,31,4},{18,31,1},{31,25,613},{27,30,613},{18,31,1},{0,29,617},{27,30,613},{0,29,617},{8,0,7956},
{8,0,7956},{8,0,7956},{8,0,7956},{3,31,1201},{3,31,1201},{3,31,1201},{0,23,0},{0,18,3185},{0,18,3185},{15,31,26715},{12,31,19659},{11,31,16925},{9,31,10232},{12,31,26835},{6,31,9215},{4,31,4027},{0,28,985},{0,31,19110},{0,27,6797},{22,31,5238},{20,31,2950},{19,31,2005},{17,31,180},{29,17,8069},{15,31,3314},{11,31,905},{0,28,985},{27,24,8069},{0,28,985},{11,31,16925},{11,31,16925},{11,31,16925},
{9,31,10232},{8,31,14182},{4,31,4027},{4,31,4027},{0,27,232},{0,29,11627},{0,25,5117},{19,31,2005},{19,31,2005},{19,31,2005},{17,31,180},{31,4,4418},{11,31,905},{11,31,905},{0,27,232},{30,18,4418},{0,27,232},{31,26,365},{27,31,146},{26,31,1},{21,31,1},{31,26,365},{30,29,365},{21,31,1},{0,29,377},{30,29,365},{0,29,377},{8,0,8980},{8,0,8980},{8,0,8980},{8,0,8980},{3,31,1665},
{3,31,1665},{3,31,1665},{0,24,4},{0,20,3601},{0,20,3601},{15,31,26555},{13,31,20326},{12,31,17723},{10,31,10897},{13,31,26598},{7,31,9527},{5,31,4934},{0,29,590},{0,31,18606},{0,28,5786},{23,31,4770},{21,31,3041},{20,31,2210},{18,31,325},{31,15,7322},{16,31,3126},{14,31,1145},{0,29,554},{31,23,7322},{0,29,554},{12,31,17723},{12,31,17723},{12,31,17723},{10,31,10897},{9,31,15092},{5,31,4934},{5,31,4934},
{0,28,205},{0,31,11381},{0,26,4709},{20,31,2210},{20,31,2210},{20,31,2210},{18,31,325},{31,7,4418},{14,31,1145},{14,31,1145},{0,28,169},{31,19,4418},{0,28,169},{31,27,185},{28,31,73},{27,31,9},{24,31,0},{31,27,185},{31,29,185},{24,31,0},{0,30,185},{31,29,185},{0,30,185},{9,0,9248},{9,0,9248},{9,0,9248},{9,0,9248},{4,31,1954},{4,31,1954},{4,31,1954},{1,25,8},{0,21,3330},
{0,21,3330},{16,31,25958},{15,31,20468},{13,31,18321},{11,31,11371},{15,31,25748},{8,31,9863},{7,31,5684},{1,30,373},{0,31,18111},{0,29,4452},{24,31,4437},{23,31,3084},{22,31,2500},{19,31,554},{29,21,6584},{18,31,2996},{15,31,1412},{0,30,237},{31,24,6584},{0,30,237},{13,31,18321},{13,31,18321},{13,31,18321},{11,31,11371},{11,31,15661},{7,31,5684},{7,31,5684},{1,29,206},{0,31,10886},{0,27,4019},{22,31,2500},
{22,31,2500},{22,31,2500},{19,31,554},{28,16,4418},{15,31,1412},{15,31,1412},{0,29,100},{24,24,4418},{0,29,100},{31,29,52},{30,31,25},{29,31,1},{27,31,1},{31,29,52},{31,30,52},{27,31,1},{0,30,68},{31,30,52},{0,30,68},{10,0,9250},{10,0,9250},{10,0,9250},{10,0,9250},{5,31,2197},{5,31,2197},{5,31,2197},{2,27,5},{0,23,2929},{0,23,2929},{17,31,25604},{15,31,20628},{15,31,18692},
{12,31,11876},{16,31,25201},{10,31,10381},{8,31,6470},{2,30,270},{2,31,17924},{0,29,3588},{25,31,4259},{23,31,3148},{23,31,2664},{20,31,820},{31,19,6019},{19,31,2950},{16,31,1717},{0,30,125},{31,25,6019},{0,30,125},{15,31,18692},{15,31,18692},{15,31,18692},{12,31,11876},{12,31,16244},{8,31,6470},{8,31,6470},{2,30,206},{0,31,10854},{0,29,3332},{23,31,2664},{23,31,2664},{23,31,2664},{20,31,820},{29,17,4418},
{16,31,1717},{16,31,1717},{0,30,61},{27,24,4418},{0,30,61},{31,31,4},{31,31,4},{31,31,4},{30,31,1},{31,31,4},{31,31,4},{30,31,1},{0,31,4},{31,31,4},{0,31,4},{11,0,9250},{11,0,9250},{11,0,9250},{11,0,9250},{6,31,2440},{6,31,2440},{6,31,2440},{3,27,10},{0,25,2509},{0,25,2509},{18,31,24418},{16,31,19831},{15,31,18144},{13,31,11876},{16,31,23685},{11,31,10015},{8,31,6638},
{3,31,170},{3,31,16879},{0,30,2738},{25,31,3699},{24,31,2766},{24,31,2405},{21,31,820},{31,20,5164},{20,31,2584},{18,31,1552},{0,31,9},{30,26,5164},{0,31,9},{15,31,18144},{15,31,18144},{15,31,18144},{13,31,11876},{12,31,15696},{8,31,6638},{8,31,6638},{4,30,132},{0,31,10150},{0,29,2624},{24,31,2405},{24,31,2405},{24,31,2405},{21,31,820},{31,15,3872},{18,31,1552},{18,31,1552},{0,31,9},{31,23,3872},
{0,31,9},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{12,0,9248},{12,0,9248},{12,0,9248},{12,0,9248},{7,31,2612},{7,31,2612},{7,31,2612},{4,28,8},{0,27,2210},{0,27,2210},{19,31,22762},{17,31,18876},{16,31,17222},{14,31,11585},{17,31,22068},{11,31,9551},{10,31,6509},{5,31,68},{4,31,15612},{0,30,2098},{26,31,3089},
{24,31,2334},{24,31,1973},{22,31,661},{29,25,4267},{22,31,2150},{19,31,1285},{2,31,0},{27,28,4267},{2,31,0},{16,31,17222},{16,31,17222},{16,31,17222},{14,31,11585},{13,31,14786},{10,31,6509},{10,31,6509},{5,31,68},{0,31,9366},{0,30,2034},{24,31,1973},{24,31,1973},{24,31,1973},{22,31,661},{31,16,3202},{19,31,1285},{19,31,1285},{2,31,0},{30,24,3202},{2,31,0},{31,31,0},{31,31,0},{31,31,0},
{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{13,0,9248},{13,0,9248},{13,0,9248},{13,0,9248},{8,31,2845},{8,31,2845},{8,31,2845},{5,29,8},{0,29,1856},{0,29,1856},{19,31,21160},{18,31,17776},{17,31,16329},{15,31,11282},{17,31,20358},{12,31,9098},{11,31,6310},{6,31,5},{6,31,14287},{0,31,1565},{27,31,2412},{25,31,1862},{25,31,1573},{23,31,509},{28,28,3361},
{22,31,1691},{20,31,1021},{5,31,1},{28,28,3361},{5,31,1},{17,31,16329},{17,31,16329},{17,31,16329},{15,31,11282},{15,31,13658},{11,31,6310},{11,31,6310},{6,31,5},{2,31,8690},{0,31,1565},{25,31,1573},{25,31,1573},{25,31,1573},{23,31,509},{31,18,2521},{20,31,1021},{20,31,1021},{5,31,1},{30,25,2521},{5,31,1},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},
{0,31,0},{31,31,0},{0,31,0},{14,0,9250},{14,0,9250},{14,0,9250},{14,0,9250},{9,31,3176},{9,31,3176},{9,31,3176},{6,31,5},{0,30,1556},{0,30,1556},{20,31,19810},{19,31,16741},{18,31,15584},{16,31,11057},{19,31,18721},{14,31,8860},{12,31,6234},{7,31,10},{7,31,13210},{0,31,1325},{27,31,1868},{26,31,1464},{26,31,1268},{24,31,397},{31,23,2649},{23,31,1329},{22,31,794},{8,31,0},{31,27,2649},
{8,31,0},{18,31,15584},{18,31,15584},{18,31,15584},{16,31,11057},{15,31,12858},{12,31,6234},{12,31,6234},{7,31,10},{3,31,8150},{0,31,1325},{26,31,1268},{26,31,1268},{26,31,1268},{24,31,397},{31,19,1989},{22,31,794},{22,31,794},{8,31,0},{31,25,1989},{8,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{15,0,9250},
{15,0,9250},{15,0,9250},{15,0,9250},{11,31,3365},{11,31,3365},{11,31,3365},{7,31,10},{0,31,1325},{0,31,1325},{20,31,18626},{19,31,15845},{19,31,14756},{17,31,10897},{19,31,17297},{15,31,8442},{13,31,6285},{8,31,68},{8,31,12227},{0,31,1341},{27,31,1452},{27,31,1089},{27,31,968},{25,31,325},{31,24,2018},{24,31,1011},{23,31,605},{11,31,0},{30,28,2018},{11,31,0},{19,31,14756},{19,31,14756},{19,31,14756},
{17,31,10897},{16,31,12077},{13,31,6285},{13,31,6285},{8,31,68},{4,31,7686},{0,31,1341},{27,31,968},{27,31,968},{27,31,968},{25,31,325},{31,21,1513},{23,31,605},{23,31,605},{11,31,0},{31,26,1513},{11,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{16,0,9248},{16,0,9248},{16,0,9248},{16,0,9248},{12,31,3626},
{12,31,3626},{12,31,3626},{8,31,68},{0,31,1341},{0,31,1341},{21,31,17476},{20,31,14998},{20,31,14098},{18,31,10672},{20,31,16018},{15,31,8154},{15,31,6218},{9,31,200},{10,31,11338},{0,31,1613},{28,31,1041},{27,31,801},{27,31,680},{26,31,232},{29,29,1473},{26,31,753},{24,31,442},{14,31,0},{31,28,1473},{14,31,0},{20,31,14098},{20,31,14098},{20,31,14098},{18,31,10672},{17,31,11453},{15,31,6218},{15,31,6218},
{9,31,200},{6,31,7270},{0,31,1613},{27,31,680},{27,31,680},{27,31,680},{26,31,232},{28,28,1105},{24,31,442},{24,31,442},{14,31,0},{28,28,1105},{14,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{31,31,0},{0,31,0},{31,31,0},{0,31,0},{17,0,9248},{17,0,9248},{17,0,9248},{17,0,9248},{13,31,3929},{13,31,3929},{13,31,3929},{9,31,200},{0,31,1613},
{0,31,1613},
/**** ended inlining basisu_transcoder_tables_dxt1_5.inc ****/
	};
#endif // BASISD_SUPPORT_DXT1

#if BASISD_SUPPORT_DXT1 || BASISD_SUPPORT_UASTC
	// First saw the idea for optimal BC1 single-color block encoding using lookup tables in ryg_dxt.
	struct bc1_match_entry
	{
		uint8_t m_hi;
		uint8_t m_lo;
	};
	static bc1_match_entry g_bc1_match5_equals_1[256], g_bc1_match6_equals_1[256]; // selector 1, allow equals hi/lo
	static bc1_match_entry g_bc1_match5_equals_0[256], g_bc1_match6_equals_0[256]; // selector 0, allow equals hi/lo

	static void prepare_bc1_single_color_table(bc1_match_entry* pTable, const uint8_t* pExpand, int size0, int size1, int sel)
	{
		for (int i = 0; i < 256; i++)
		{
			int lowest_e = 256;
			for (int lo = 0; lo < size0; lo++)
			{
				for (int hi = 0; hi < size1; hi++)
				{
					const int lo_e = pExpand[lo], hi_e = pExpand[hi];
					int e;

					if (sel == 1)
					{
						// Selector 1
						e = basisu::iabs(((hi_e * 2 + lo_e) / 3) - i);
						e += (basisu::iabs(hi_e - lo_e) * 3) / 100;
					}
					else
					{
						assert(sel == 0);

						// Selector 0
						e = basisu::iabs(hi_e - i);
					}

					if (e < lowest_e)
					{
						pTable[i].m_hi = static_cast<uint8_t>(hi);
						pTable[i].m_lo = static_cast<uint8_t>(lo);

						lowest_e = e;
					}

				} // hi
			} // lo
		}
	}
#endif

#if BASISD_WRITE_NEW_DXT1_TABLES
	static void create_etc1_to_dxt1_5_conversion_table()
	{
		FILE* pFile = nullptr;
		fopen_s(&pFile, "basisu_transcoder_tables_dxt1_5.inc", "w");

		uint32_t n = 0;

		for (int inten = 0; inten < 8; inten++)
		{
			for (uint32_t g = 0; g < 32; g++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(g, g, g, 255), false), inten);

				for (uint32_t sr = 0; sr < NUM_ETC1_TO_DXT1_SELECTOR_RANGES; sr++)
				{
					const uint32_t low_selector = g_etc1_to_dxt1_selector_ranges[sr].m_low;
					const uint32_t high_selector = g_etc1_to_dxt1_selector_ranges[sr].m_high;

					for (uint32_t m = 0; m < NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS; m++)
					{
						uint32_t best_lo = 0;
						uint32_t best_hi = 0;
						uint64_t best_err = UINT64_MAX;

						for (uint32_t hi = 0; hi <= 31; hi++)
						{
							for (uint32_t lo = 0; lo <= 31; lo++)
							{
								//if (lo == hi) continue;

								uint32_t colors[4];

								colors[0] = (lo << 3) | (lo >> 2);
								colors[3] = (hi << 3) | (hi >> 2);

								colors[1] = (colors[0] * 2 + colors[3]) / 3;
								colors[2] = (colors[3] * 2 + colors[0]) / 3;

								uint64_t total_err = 0;

								for (uint32_t s = low_selector; s <= high_selector; s++)
								{
									int err = block_colors[s].g - colors[g_etc1_to_dxt1_selector_mappings[m][s]];

									total_err += err * err;
								}

								if (total_err < best_err)
								{
									best_err = total_err;
									best_lo = lo;
									best_hi = hi;
								}
							}
						}

						assert(best_err <= 0xFFFF);

						//table[g + inten * 32].m_solutions[sr][m].m_lo = static_cast<uint8_t>(best_lo);
						//table[g + inten * 32].m_solutions[sr][m].m_hi = static_cast<uint8_t>(best_hi);
						//table[g + inten * 32].m_solutions[sr][m].m_err = static_cast<uint16_t>(best_err);

						//assert(best_lo != best_hi);
						fprintf(pFile, "{%u,%u,%u},", best_lo, best_hi, (uint32_t)best_err);
						n++;
						if ((n & 31) == 31)
							fprintf(pFile, "\n");
					} // m
				} // sr
			} // g
		} // inten

		fclose(pFile);
	}

	static void create_etc1_to_dxt1_6_conversion_table()
	{
		FILE* pFile = nullptr;
		fopen_s(&pFile, "basisu_transcoder_tables_dxt1_6.inc", "w");

		uint32_t n = 0;

		for (int inten = 0; inten < 8; inten++)
		{
			for (uint32_t g = 0; g < 32; g++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(g, g, g, 255), false), inten);

				for (uint32_t sr = 0; sr < NUM_ETC1_TO_DXT1_SELECTOR_RANGES; sr++)
				{
					const uint32_t low_selector = g_etc1_to_dxt1_selector_ranges[sr].m_low;
					const uint32_t high_selector = g_etc1_to_dxt1_selector_ranges[sr].m_high;

					for (uint32_t m = 0; m < NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS; m++)
					{
						uint32_t best_lo = 0;
						uint32_t best_hi = 0;
						uint64_t best_err = UINT64_MAX;

						for (uint32_t hi = 0; hi <= 63; hi++)
						{
							for (uint32_t lo = 0; lo <= 63; lo++)
							{
								//if (lo == hi) continue;

								uint32_t colors[4];

								colors[0] = (lo << 2) | (lo >> 4);
								colors[3] = (hi << 2) | (hi >> 4);

								colors[1] = (colors[0] * 2 + colors[3]) / 3;
								colors[2] = (colors[3] * 2 + colors[0]) / 3;

								uint64_t total_err = 0;

								for (uint32_t s = low_selector; s <= high_selector; s++)
								{
									int err = block_colors[s].g - colors[g_etc1_to_dxt1_selector_mappings[m][s]];

									total_err += err * err;
								}

								if (total_err < best_err)
								{
									best_err = total_err;
									best_lo = lo;
									best_hi = hi;
								}
							}
						}

						assert(best_err <= 0xFFFF);

						//table[g + inten * 32].m_solutions[sr][m].m_lo = static_cast<uint8_t>(best_lo);
						//table[g + inten * 32].m_solutions[sr][m].m_hi = static_cast<uint8_t>(best_hi);
						//table[g + inten * 32].m_solutions[sr][m].m_err = static_cast<uint16_t>(best_err);

						//assert(best_lo != best_hi);
						fprintf(pFile, "{%u,%u,%u},", best_lo, best_hi, (uint32_t)best_err);
						n++;
						if ((n & 31) == 31)
							fprintf(pFile, "\n");

					} // m
				} // sr
			} // g
		} // inten

		fclose(pFile);
	}
#endif


#if BASISD_SUPPORT_UASTC || BASISD_SUPPORT_ETC2_EAC_A8 || BASISD_SUPPORT_ETC2_EAC_RG11
	static const int8_t g_eac_modifier_table[16][8] =
	{
		{ -3, -6, -9, -15, 2, 5, 8, 14 },
		{ -3, -7, -10, -13, 2, 6, 9, 12 },
		{ -2, -5, -8, -13, 1, 4, 7, 12 },
		{ -2, -4, -6, -13, 1, 3, 5, 12 },
		{ -3, -6, -8, -12, 2, 5, 7, 11 },
		{ -3, -7, -9, -11, 2, 6, 8, 10 },
		{ -4, -7, -8, -11, 3, 6, 7, 10 },
		{ -3, -5, -8, -11, 2, 4, 7, 10 },

		{ -2, -6, -8, -10, 1, 5, 7, 9 },
		{ -2, -5, -8, -10, 1, 4, 7, 9 },
		{ -2, -4, -8, -10, 1, 3, 7, 9 },
		{ -2, -5, -7, -10, 1, 4, 6, 9 },
		{ -3, -4, -7, -10, 2, 3, 6, 9 },
		{ -1, -2, -3, -10, 0, 1, 2, 9 }, // entry 13
		{ -4, -6, -8, -9, 3, 5, 7, 8 },
		{ -3, -5, -7, -9, 2, 4, 6, 8 }
	};

	// Used by ETC2 EAC A8 and ETC2 EAC R11/RG11.
	struct eac_block
	{
		uint16_t m_base : 8;

		uint16_t m_table : 4;
		uint16_t m_multiplier : 4;

		uint8_t m_selectors[6];

		uint32_t get_selector(uint32_t x, uint32_t y) const
		{
			assert((x < 4) && (y < 4));

			const uint32_t ofs = 45 - (y + x * 4) * 3;

			const uint64_t pixels = get_selector_bits();

			return (pixels >> ofs) & 7;
		}

		void set_selector(uint32_t x, uint32_t y, uint32_t s)
		{
			assert((x < 4) && (y < 4) && (s < 8));

			const uint32_t ofs = 45 - (y + x * 4) * 3;

			uint64_t pixels = get_selector_bits();

			pixels &= ~(7ULL << ofs);
			pixels |= (static_cast<uint64_t>(s) << ofs);

			set_selector_bits(pixels);
		}

		uint64_t get_selector_bits() const
		{
			uint64_t pixels = ((uint64_t)m_selectors[0] << 40) | ((uint64_t)m_selectors[1] << 32) |
				((uint64_t)m_selectors[2] << 24) |
				((uint64_t)m_selectors[3] << 16) | ((uint64_t)m_selectors[4] << 8) | m_selectors[5];
			return pixels;
		}

		void set_selector_bits(uint64_t pixels)
		{
			m_selectors[0] = (uint8_t)(pixels >> 40);
			m_selectors[1] = (uint8_t)(pixels >> 32);
			m_selectors[2] = (uint8_t)(pixels >> 24);
			m_selectors[3] = (uint8_t)(pixels >> 16);
			m_selectors[4] = (uint8_t)(pixels >> 8);
			m_selectors[5] = (uint8_t)(pixels);
		}
	};

#endif // #if BASISD_SUPPORT_UASTC BASISD_SUPPORT_ETC2_EAC_A8 || BASISD_SUPPORT_ETC2_EAC_RG11

#if BASISD_SUPPORT_ETC2_EAC_A8 || BASISD_SUPPORT_ETC2_EAC_RG11
	static const dxt_selector_range s_etc2_eac_selector_ranges[] =
	{
		{ 0, 3 },

		{ 1, 3 },
		{ 0, 2 },

		{ 1, 2 },
	};

	const uint32_t NUM_ETC2_EAC_SELECTOR_RANGES = sizeof(s_etc2_eac_selector_ranges) / sizeof(s_etc2_eac_selector_ranges[0]);

	struct etc1_g_to_eac_conversion
	{
		uint8_t m_base;
		uint8_t m_table_mul; // mul*16+table
		uint16_t m_trans; // translates ETC1 selectors to ETC2_EAC_A8
	};
#endif // BASISD_SUPPORT_ETC2_EAC_A8 || BASISD_SUPPORT_ETC2_EAC_RG11

#if BASISD_SUPPORT_ETC2_EAC_A8

#if BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES
	struct pack_eac_a8_results
	{
		uint32_t m_base;
		uint32_t m_table;
		uint32_t m_multiplier;
		basisu::vector<uint8_t> m_selectors;
		basisu::vector<uint8_t> m_selectors_temp;
	};

	static uint64_t pack_eac_a8_exhaustive(pack_eac_a8_results& results, const uint8_t* pPixels, uint32_t num_pixels)
	{
		results.m_selectors.resize(num_pixels);
		results.m_selectors_temp.resize(num_pixels);

		uint64_t best_err = UINT64_MAX;

		for (uint32_t base_color = 0; base_color < 256; base_color++)
		{
			for (uint32_t multiplier = 1; multiplier < 16; multiplier++)
			{
				for (uint32_t table = 0; table < 16; table++)
				{
					uint64_t total_err = 0;

					for (uint32_t i = 0; i < num_pixels; i++)
					{
						const int a = pPixels[i];

						uint32_t best_s_err = UINT32_MAX;
						uint32_t best_s = 0;
						for (uint32_t s = 0; s < 8; s++)
						{
							int v = (int)multiplier * g_eac_modifier_table[table][s] + (int)base_color;
							if (v < 0)
								v = 0;
							else if (v > 255)
								v = 255;

							uint32_t err = abs(a - v);
							if (err < best_s_err)
							{
								best_s_err = err;
								best_s = s;
							}
						}

						results.m_selectors_temp[i] = static_cast<uint8_t>(best_s);

						total_err += best_s_err * best_s_err;
						if (total_err >= best_err)
							break;
					}

					if (total_err < best_err)
					{
						best_err = total_err;
						results.m_base = base_color;
						results.m_multiplier = multiplier;
						results.m_table = table;
						results.m_selectors.swap(results.m_selectors_temp);
					}

				} // table

			} // multiplier

		} // base_color

		return best_err;
	}
#endif // BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES
		
	static
#if !BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES		
		const
#endif
		etc1_g_to_eac_conversion s_etc1_g_to_etc2_a8[32 * 8][NUM_ETC2_EAC_SELECTOR_RANGES] =
	{
		{ { 0,1,3328 },{ 0,1,3328 },{ 0,1,256 },{ 0,1,256 } },
		{ { 0,226,3936 },{ 0,226,3936 },{ 0,81,488 },{ 0,81,488 } },
		{ { 6,178,4012 },{ 6,178,4008 },{ 0,146,501 },{ 0,130,496 } },
		{ { 14,178,4012 },{ 14,178,4008 },{ 8,146,501 },{ 6,82,496 } },
		{ { 23,178,4012 },{ 23,178,4008 },{ 17,146,501 },{ 3,228,496 } },
		{ { 31,178,4012 },{ 31,178,4008 },{ 25,146,501 },{ 11,228,496 } },
		{ { 39,178,4012 },{ 39,178,4008 },{ 33,146,501 },{ 19,228,496 } },
		{ { 47,178,4012 },{ 47,178,4008 },{ 41,146,501 },{ 27,228,496 } },
		{ { 56,178,4012 },{ 56,178,4008 },{ 50,146,501 },{ 36,228,496 } },
		{ { 64,178,4012 },{ 64,178,4008 },{ 58,146,501 },{ 44,228,496 } },
		{ { 72,178,4012 },{ 72,178,4008 },{ 66,146,501 },{ 52,228,496 } },
		{ { 80,178,4012 },{ 80,178,4008 },{ 74,146,501 },{ 60,228,496 } },
		{ { 89,178,4012 },{ 89,178,4008 },{ 83,146,501 },{ 69,228,496 } },
		{ { 97,178,4012 },{ 97,178,4008 },{ 91,146,501 },{ 77,228,496 } },
		{ { 105,178,4012 },{ 105,178,4008 },{ 99,146,501 },{ 85,228,496 } },
		{ { 113,178,4012 },{ 113,178,4008 },{ 107,146,501 },{ 93,228,496 } },
		{ { 122,178,4012 },{ 122,178,4008 },{ 116,146,501 },{ 102,228,496 } },
		{ { 130,178,4012 },{ 130,178,4008 },{ 124,146,501 },{ 110,228,496 } },
		{ { 138,178,4012 },{ 138,178,4008 },{ 132,146,501 },{ 118,228,496 } },
		{ { 146,178,4012 },{ 146,178,4008 },{ 140,146,501 },{ 126,228,496 } },
		{ { 155,178,4012 },{ 155,178,4008 },{ 149,146,501 },{ 135,228,496 } },
		{ { 163,178,4012 },{ 163,178,4008 },{ 157,146,501 },{ 143,228,496 } },
		{ { 171,178,4012 },{ 171,178,4008 },{ 165,146,501 },{ 151,228,496 } },
		{ { 179,178,4012 },{ 179,178,4008 },{ 173,146,501 },{ 159,228,496 } },
		{ { 188,178,4012 },{ 188,178,4008 },{ 182,146,501 },{ 168,228,496 } },
		{ { 196,178,4012 },{ 196,178,4008 },{ 190,146,501 },{ 176,228,496 } },
		{ { 204,178,4012 },{ 204,178,4008 },{ 198,146,501 },{ 184,228,496 } },
		{ { 212,178,4012 },{ 212,178,4008 },{ 206,146,501 },{ 192,228,496 } },
		{ { 221,178,4012 },{ 221,178,4008 },{ 215,146,501 },{ 201,228,496 } },
		{ { 229,178,4012 },{ 229,178,4008 },{ 223,146,501 },{ 209,228,496 } },
		{ { 235,66,4012 },{ 221,100,4008 },{ 231,146,501 },{ 217,228,496 } },
		{ { 211,102,4085 },{ 118,31,4080 },{ 211,102,501 },{ 118,31,496 } },
		{ { 1,2,3328 },{ 1,2,3328 },{ 0,1,320 },{ 0,1,320 } },
		{ { 7,162,3905 },{ 7,162,3904 },{ 1,17,480 },{ 1,17,480 } },
		{ { 15,162,3906 },{ 15,162,3904 },{ 1,117,352 },{ 1,117,352 } },
		{ { 23,162,3906 },{ 23,162,3904 },{ 5,34,500 },{ 4,53,424 } },
		{ { 32,162,3906 },{ 32,162,3904 },{ 14,34,500 },{ 3,69,424 } },
		{ { 40,162,3906 },{ 40,162,3904 },{ 22,34,500 },{ 1,133,496 } },
		{ { 48,162,3906 },{ 48,162,3904 },{ 30,34,500 },{ 4,85,496 } },
		{ { 56,162,3906 },{ 56,162,3904 },{ 38,34,500 },{ 12,85,496 } },
		{ { 65,162,3906 },{ 65,162,3904 },{ 47,34,500 },{ 1,106,424 } },
		{ { 73,162,3906 },{ 73,162,3904 },{ 55,34,500 },{ 9,106,424 } },
		{ { 81,162,3906 },{ 81,162,3904 },{ 63,34,500 },{ 7,234,496 } },
		{ { 89,162,3906 },{ 89,162,3904 },{ 71,34,500 },{ 15,234,496 } },
		{ { 98,162,3906 },{ 98,162,3904 },{ 80,34,500 },{ 24,234,496 } },
		{ { 106,162,3906 },{ 106,162,3904 },{ 88,34,500 },{ 32,234,496 } },
		{ { 114,162,3906 },{ 114,162,3904 },{ 96,34,500 },{ 40,234,496 } },
		{ { 122,162,3906 },{ 122,162,3904 },{ 104,34,500 },{ 48,234,496 } },
		{ { 131,162,3906 },{ 131,162,3904 },{ 113,34,500 },{ 57,234,496 } },
		{ { 139,162,3906 },{ 139,162,3904 },{ 121,34,500 },{ 65,234,496 } },
		{ { 147,162,3906 },{ 147,162,3904 },{ 129,34,500 },{ 73,234,496 } },
		{ { 155,162,3906 },{ 155,162,3904 },{ 137,34,500 },{ 81,234,496 } },
		{ { 164,162,3906 },{ 164,162,3904 },{ 146,34,500 },{ 90,234,496 } },
		{ { 172,162,3906 },{ 172,162,3904 },{ 154,34,500 },{ 98,234,496 } },
		{ { 180,162,3906 },{ 180,162,3904 },{ 162,34,500 },{ 106,234,496 } },
		{ { 188,162,3906 },{ 188,162,3904 },{ 170,34,500 },{ 114,234,496 } },
		{ { 197,162,3906 },{ 197,162,3904 },{ 179,34,500 },{ 123,234,496 } },
		{ { 205,162,3906 },{ 205,162,3904 },{ 187,34,500 },{ 131,234,496 } },
		{ { 213,162,3906 },{ 213,162,3904 },{ 195,34,500 },{ 139,234,496 } },
		{ { 221,162,3906 },{ 221,162,3904 },{ 203,34,500 },{ 147,234,496 } },
		{ { 230,162,3906 },{ 230,162,3904 },{ 212,34,500 },{ 156,234,496 } },
		{ { 238,162,3906 },{ 174,106,4008 },{ 220,34,500 },{ 164,234,496 } },
		{ { 240,178,4001 },{ 182,106,4008 },{ 228,34,500 },{ 172,234,496 } },
		{ { 166,108,4085 },{ 115,31,4080 },{ 166,108,501 },{ 115,31,496 } },
		{ { 1,68,3328 },{ 1,68,3328 },{ 0,17,384 },{ 0,17,384 } },
		{ { 1,148,3904 },{ 1,148,3904 },{ 1,2,384 },{ 1,2,384 } },
		{ { 21,18,3851 },{ 21,18,3848 },{ 1,50,488 },{ 1,50,488 } },
		{ { 27,195,3851 },{ 29,18,3848 },{ 0,67,488 },{ 0,67,488 } },
		{ { 34,195,3907 },{ 38,18,3848 },{ 20,66,482 },{ 0,3,496 } },
		{ { 42,195,3907 },{ 46,18,3848 },{ 28,66,482 },{ 2,6,424 } },
		{ { 50,195,3907 },{ 54,18,3848 },{ 36,66,482 },{ 4,22,424 } },
		{ { 58,195,3907 },{ 62,18,3848 },{ 44,66,482 },{ 3,73,424 } },
		{ { 67,195,3907 },{ 71,18,3848 },{ 53,66,482 },{ 3,22,496 } },
		{ { 75,195,3907 },{ 79,18,3848 },{ 61,66,482 },{ 2,137,496 } },
		{ { 83,195,3907 },{ 87,18,3848 },{ 69,66,482 },{ 1,89,496 } },
		{ { 91,195,3907 },{ 95,18,3848 },{ 77,66,482 },{ 9,89,496 } },
		{ { 100,195,3907 },{ 104,18,3848 },{ 86,66,482 },{ 18,89,496 } },
		{ { 108,195,3907 },{ 112,18,3848 },{ 94,66,482 },{ 26,89,496 } },
		{ { 116,195,3907 },{ 120,18,3848 },{ 102,66,482 },{ 34,89,496 } },
		{ { 124,195,3907 },{ 128,18,3848 },{ 110,66,482 },{ 42,89,496 } },
		{ { 133,195,3907 },{ 137,18,3848 },{ 119,66,482 },{ 51,89,496 } },
		{ { 141,195,3907 },{ 145,18,3848 },{ 127,66,482 },{ 59,89,496 } },
		{ { 149,195,3907 },{ 153,18,3848 },{ 135,66,482 },{ 67,89,496 } },
		{ { 157,195,3907 },{ 161,18,3848 },{ 143,66,482 },{ 75,89,496 } },
		{ { 166,195,3907 },{ 170,18,3848 },{ 152,66,482 },{ 84,89,496 } },
		{ { 174,195,3907 },{ 178,18,3848 },{ 160,66,482 },{ 92,89,496 } },
		{ { 182,195,3907 },{ 186,18,3848 },{ 168,66,482 },{ 100,89,496 } },
		{ { 190,195,3907 },{ 194,18,3848 },{ 176,66,482 },{ 108,89,496 } },
		{ { 199,195,3907 },{ 203,18,3848 },{ 185,66,482 },{ 117,89,496 } },
		{ { 207,195,3907 },{ 211,18,3848 },{ 193,66,482 },{ 125,89,496 } },
		{ { 215,195,3907 },{ 219,18,3848 },{ 201,66,482 },{ 133,89,496 } },
		{ { 223,195,3907 },{ 227,18,3848 },{ 209,66,482 },{ 141,89,496 } },
		{ { 231,195,3907 },{ 168,89,4008 },{ 218,66,482 },{ 150,89,496 } },
		{ { 236,18,3907 },{ 176,89,4008 },{ 226,66,482 },{ 158,89,496 } },
		{ { 158,90,4085 },{ 103,31,4080 },{ 158,90,501 },{ 103,31,496 } },
		{ { 166,90,4085 },{ 111,31,4080 },{ 166,90,501 },{ 111,31,496 } },
		{ { 0,70,3328 },{ 0,70,3328 },{ 0,45,256 },{ 0,45,256 } },
		{ { 0,117,3904 },{ 0,117,3904 },{ 0,35,384 },{ 0,35,384 } },
		{ { 13,165,3905 },{ 13,165,3904 },{ 3,221,416 },{ 3,221,416 } },
		{ { 21,165,3906 },{ 21,165,3904 },{ 11,221,416 },{ 11,221,416 } },
		{ { 30,165,3906 },{ 30,165,3904 },{ 7,61,352 },{ 7,61,352 } },
		{ { 38,165,3906 },{ 38,165,3904 },{ 2,125,352 },{ 2,125,352 } },
		{ { 46,165,3906 },{ 46,165,3904 },{ 2,37,500 },{ 10,125,352 } },
		{ { 54,165,3906 },{ 54,165,3904 },{ 10,37,500 },{ 5,61,424 } },
		{ { 63,165,3906 },{ 63,165,3904 },{ 19,37,500 },{ 1,189,424 } },
		{ { 4,254,4012 },{ 71,165,3904 },{ 27,37,500 },{ 9,189,424 } },
		{ { 12,254,4012 },{ 79,165,3904 },{ 35,37,500 },{ 4,77,424 } },
		{ { 20,254,4012 },{ 87,165,3904 },{ 43,37,500 },{ 12,77,424 } },
		{ { 29,254,4012 },{ 96,165,3904 },{ 52,37,500 },{ 8,93,424 } },
		{ { 37,254,4012 },{ 104,165,3904 },{ 60,37,500 },{ 3,141,496 } },
		{ { 45,254,4012 },{ 112,165,3904 },{ 68,37,500 },{ 11,141,496 } },
		{ { 53,254,4012 },{ 120,165,3904 },{ 76,37,500 },{ 6,93,496 } },
		{ { 62,254,4012 },{ 129,165,3904 },{ 85,37,500 },{ 15,93,496 } },
		{ { 70,254,4012 },{ 137,165,3904 },{ 93,37,500 },{ 23,93,496 } },
		{ { 78,254,4012 },{ 145,165,3904 },{ 101,37,500 },{ 31,93,496 } },
		{ { 86,254,4012 },{ 153,165,3904 },{ 109,37,500 },{ 39,93,496 } },
		{ { 95,254,4012 },{ 162,165,3904 },{ 118,37,500 },{ 48,93,496 } },
		{ { 103,254,4012 },{ 170,165,3904 },{ 126,37,500 },{ 56,93,496 } },
		{ { 111,254,4012 },{ 178,165,3904 },{ 134,37,500 },{ 64,93,496 } },
		{ { 119,254,4012 },{ 186,165,3904 },{ 142,37,500 },{ 72,93,496 } },
		{ { 128,254,4012 },{ 195,165,3904 },{ 151,37,500 },{ 81,93,496 } },
		{ { 136,254,4012 },{ 203,165,3904 },{ 159,37,500 },{ 89,93,496 } },
		{ { 212,165,3906 },{ 136,77,4008 },{ 167,37,500 },{ 97,93,496 } },
		{ { 220,165,3394 },{ 131,93,4008 },{ 175,37,500 },{ 105,93,496 } },
		{ { 214,181,4001 },{ 140,93,4008 },{ 184,37,500 },{ 114,93,496 } },
		{ { 222,181,4001 },{ 148,93,4008 },{ 192,37,500 },{ 122,93,496 } },
		{ { 114,95,4085 },{ 99,31,4080 },{ 114,95,501 },{ 99,31,496 } },
		{ { 122,95,4085 },{ 107,31,4080 },{ 122,95,501 },{ 107,31,496 } },
		{ { 0,102,3840 },{ 0,102,3840 },{ 0,18,384 },{ 0,18,384 } },
		{ { 5,167,3904 },{ 5,167,3904 },{ 0,13,256 },{ 0,13,256 } },
		{ { 4,54,3968 },{ 4,54,3968 },{ 1,67,448 },{ 1,67,448 } },
		{ { 30,198,3850 },{ 30,198,3848 },{ 0,3,480 },{ 0,3,480 } },
		{ { 39,198,3850 },{ 39,198,3848 },{ 3,52,488 },{ 3,52,488 } },
		{ { 47,198,3851 },{ 47,198,3848 },{ 3,4,488 },{ 3,4,488 } },
		{ { 55,198,3851 },{ 55,198,3848 },{ 1,70,488 },{ 1,70,488 } },
		{ { 54,167,3906 },{ 63,198,3848 },{ 3,22,488 },{ 3,22,488 } },
		{ { 62,167,3906 },{ 72,198,3848 },{ 24,118,488 },{ 0,6,496 } },
		{ { 70,167,3906 },{ 80,198,3848 },{ 32,118,488 },{ 2,89,488 } },
		{ { 78,167,3906 },{ 88,198,3848 },{ 40,118,488 },{ 1,73,496 } },
		{ { 86,167,3906 },{ 96,198,3848 },{ 48,118,488 },{ 0,28,424 } },
		{ { 95,167,3906 },{ 105,198,3848 },{ 57,118,488 },{ 9,28,424 } },
		{ { 103,167,3906 },{ 113,198,3848 },{ 65,118,488 },{ 5,108,496 } },
		{ { 111,167,3906 },{ 121,198,3848 },{ 73,118,488 },{ 13,108,496 } },
		{ { 119,167,3906 },{ 129,198,3848 },{ 81,118,488 },{ 21,108,496 } },
		{ { 128,167,3906 },{ 138,198,3848 },{ 90,118,488 },{ 6,28,496 } },
		{ { 136,167,3906 },{ 146,198,3848 },{ 98,118,488 },{ 14,28,496 } },
		{ { 144,167,3906 },{ 154,198,3848 },{ 106,118,488 },{ 22,28,496 } },
		{ { 152,167,3906 },{ 162,198,3848 },{ 114,118,488 },{ 30,28,496 } },
		{ { 161,167,3906 },{ 171,198,3848 },{ 123,118,488 },{ 39,28,496 } },
		{ { 169,167,3906 },{ 179,198,3848 },{ 131,118,488 },{ 47,28,496 } },
		{ { 177,167,3906 },{ 187,198,3848 },{ 139,118,488 },{ 55,28,496 } },
		{ { 185,167,3906 },{ 195,198,3848 },{ 147,118,488 },{ 63,28,496 } },
		{ { 194,167,3906 },{ 120,12,4008 },{ 156,118,488 },{ 72,28,496 } },
		{ { 206,198,3907 },{ 116,28,4008 },{ 164,118,488 },{ 80,28,496 } },
		{ { 214,198,3907 },{ 124,28,4008 },{ 172,118,488 },{ 88,28,496 } },
		{ { 222,198,3395 },{ 132,28,4008 },{ 180,118,488 },{ 96,28,496 } },
		{ { 207,134,4001 },{ 141,28,4008 },{ 189,118,488 },{ 105,28,496 } },
		{ { 95,30,4085 },{ 86,31,4080 },{ 95,30,501 },{ 86,31,496 } },
		{ { 103,30,4085 },{ 94,31,4080 },{ 103,30,501 },{ 94,31,496 } },
		{ { 111,30,4085 },{ 102,31,4080 },{ 111,30,501 },{ 102,31,496 } },
		{ { 0,104,3840 },{ 0,104,3840 },{ 0,18,448 },{ 0,18,448 } },
		{ { 4,39,3904 },{ 4,39,3904 },{ 0,4,384 },{ 0,4,384 } },
		{ { 0,56,3968 },{ 0,56,3968 },{ 0,84,448 },{ 0,84,448 } },
		{ { 6,110,3328 },{ 6,110,3328 },{ 0,20,448 },{ 0,20,448 } },
		{ { 41,200,3850 },{ 41,200,3848 },{ 1,4,480 },{ 1,4,480 } },
		{ { 49,200,3850 },{ 49,200,3848 },{ 1,8,416 },{ 1,8,416 } },
		{ { 57,200,3851 },{ 57,200,3848 },{ 1,38,488 },{ 1,38,488 } },
		{ { 65,200,3851 },{ 65,200,3848 },{ 1,120,488 },{ 1,120,488 } },
		{ { 74,200,3851 },{ 74,200,3848 },{ 2,72,488 },{ 2,72,488 } },
		{ { 69,6,3907 },{ 82,200,3848 },{ 2,24,488 },{ 2,24,488 } },
		{ { 77,6,3907 },{ 90,200,3848 },{ 26,120,488 },{ 10,24,488 } },
		{ { 97,63,3330 },{ 98,200,3848 },{ 34,120,488 },{ 2,8,496 } },
		{ { 106,63,3330 },{ 107,200,3848 },{ 43,120,488 },{ 3,92,488 } },
		{ { 114,63,3330 },{ 115,200,3848 },{ 51,120,488 },{ 11,92,488 } },
		{ { 122,63,3330 },{ 123,200,3848 },{ 59,120,488 },{ 7,76,496 } },
		{ { 130,63,3330 },{ 131,200,3848 },{ 67,120,488 },{ 15,76,496 } },
		{ { 139,63,3330 },{ 140,200,3848 },{ 76,120,488 },{ 24,76,496 } },
		{ { 147,63,3330 },{ 148,200,3848 },{ 84,120,488 },{ 32,76,496 } },
		{ { 155,63,3330 },{ 156,200,3848 },{ 92,120,488 },{ 40,76,496 } },
		{ { 163,63,3330 },{ 164,200,3848 },{ 100,120,488 },{ 48,76,496 } },
		{ { 172,63,3330 },{ 173,200,3848 },{ 109,120,488 },{ 57,76,496 } },
		{ { 184,6,3851 },{ 181,200,3848 },{ 117,120,488 },{ 65,76,496 } },
		{ { 192,6,3851 },{ 133,28,3936 },{ 125,120,488 },{ 73,76,496 } },
		{ { 189,200,3907 },{ 141,28,3936 },{ 133,120,488 },{ 81,76,496 } },
		{ { 198,200,3907 },{ 138,108,4000 },{ 142,120,488 },{ 90,76,496 } },
		{ { 206,200,3907 },{ 146,108,4000 },{ 150,120,488 },{ 98,76,496 } },
		{ { 214,200,3395 },{ 154,108,4000 },{ 158,120,488 },{ 106,76,496 } },
		{ { 190,136,4001 },{ 162,108,4000 },{ 166,120,488 },{ 114,76,496 } },
		{ { 123,30,4076 },{ 87,15,4080 },{ 123,30,492 },{ 87,15,496 } },
		{ { 117,110,4084 },{ 80,31,4080 },{ 117,110,500 },{ 80,31,496 } },
		{ { 125,110,4084 },{ 88,31,4080 },{ 125,110,500 },{ 88,31,496 } },
		{ { 133,110,4084 },{ 96,31,4080 },{ 133,110,500 },{ 96,31,496 } },
		{ { 9,56,3904 },{ 9,56,3904 },{ 0,67,448 },{ 0,67,448 } },
		{ { 1,8,3904 },{ 1,8,3904 },{ 1,84,448 },{ 1,84,448 } },
		{ { 1,124,3904 },{ 1,124,3904 },{ 0,39,384 },{ 0,39,384 } },
		{ { 9,124,3904 },{ 9,124,3904 },{ 1,4,448 },{ 1,4,448 } },
		{ { 6,76,3904 },{ 6,76,3904 },{ 0,70,448 },{ 0,70,448 } },
		{ { 62,6,3859 },{ 62,6,3856 },{ 2,38,480 },{ 2,38,480 } },
		{ { 70,6,3859 },{ 70,6,3856 },{ 5,43,416 },{ 5,43,416 } },
		{ { 78,6,3859 },{ 78,6,3856 },{ 2,11,416 },{ 2,11,416 } },
		{ { 87,6,3859 },{ 87,6,3856 },{ 0,171,488 },{ 0,171,488 } },
		{ { 67,8,3906 },{ 95,6,3856 },{ 8,171,488 },{ 8,171,488 } },
		{ { 75,8,3907 },{ 103,6,3856 },{ 5,123,488 },{ 5,123,488 } },
		{ { 83,8,3907 },{ 111,6,3856 },{ 2,75,488 },{ 2,75,488 } },
		{ { 92,8,3907 },{ 120,6,3856 },{ 0,27,488 },{ 0,27,488 } },
		{ { 100,8,3907 },{ 128,6,3856 },{ 8,27,488 },{ 8,27,488 } },
		{ { 120,106,3843 },{ 136,6,3856 },{ 100,6,387 },{ 16,27,488 } },
		{ { 128,106,3843 },{ 144,6,3856 },{ 108,6,387 },{ 2,11,496 } },
		{ { 137,106,3843 },{ 153,6,3856 },{ 117,6,387 },{ 11,11,496 } },
		{ { 145,106,3843 },{ 161,6,3856 },{ 125,6,387 },{ 19,11,496 } },
		{ { 163,8,3851 },{ 137,43,3904 },{ 133,6,387 },{ 27,11,496 } },
		{ { 171,8,3851 },{ 101,11,4000 },{ 141,6,387 },{ 35,11,496 } },
		{ { 180,8,3851 },{ 110,11,4000 },{ 150,6,387 },{ 44,11,496 } },
		{ { 188,8,3851 },{ 118,11,4000 },{ 158,6,387 },{ 52,11,496 } },
		{ { 172,72,3907 },{ 126,11,4000 },{ 166,6,387 },{ 60,11,496 } },
		{ { 174,6,3971 },{ 134,11,4000 },{ 174,6,387 },{ 68,11,496 } },
		{ { 183,6,3971 },{ 143,11,4000 },{ 183,6,387 },{ 77,11,496 } },
		{ { 191,6,3971 },{ 151,11,4000 },{ 191,6,387 },{ 85,11,496 } },
		{ { 199,6,3971 },{ 159,11,4000 },{ 199,6,387 },{ 93,11,496 } },
		{ { 92,12,4084 },{ 69,15,4080 },{ 92,12,500 },{ 69,15,496 } },
		{ { 101,12,4084 },{ 78,15,4080 },{ 101,12,500 },{ 78,15,496 } },
		{ { 109,12,4084 },{ 86,15,4080 },{ 109,12,500 },{ 86,15,496 } },
		{ { 117,12,4084 },{ 79,31,4080 },{ 117,12,500 },{ 79,31,496 } },
		{ { 125,12,4084 },{ 87,31,4080 },{ 125,12,500 },{ 87,31,496 } },
		{ { 71,8,3602 },{ 71,8,3600 },{ 2,21,384 },{ 2,21,384 } },
		{ { 79,8,3611 },{ 79,8,3608 },{ 0,69,448 },{ 0,69,448 } },
		{ { 87,8,3611 },{ 87,8,3608 },{ 0,23,384 },{ 0,23,384 } },
		{ { 95,8,3611 },{ 95,8,3608 },{ 1,5,448 },{ 1,5,448 } },
		{ { 104,8,3611 },{ 104,8,3608 },{ 0,88,448 },{ 0,88,448 } },
		{ { 112,8,3611 },{ 112,8,3608 },{ 0,72,448 },{ 0,72,448 } },
		{ { 120,8,3611 },{ 121,8,3608 },{ 36,21,458 },{ 36,21,456 } },
		{ { 133,47,3091 },{ 129,8,3608 },{ 44,21,458 },{ 44,21,456 } },
		{ { 142,47,3091 },{ 138,8,3608 },{ 53,21,459 },{ 53,21,456 } },
		{ { 98,12,3850 },{ 98,12,3848 },{ 61,21,459 },{ 61,21,456 } },
		{ { 106,12,3850 },{ 106,12,3848 },{ 10,92,480 },{ 69,21,456 } },
		{ { 114,12,3851 },{ 114,12,3848 },{ 18,92,480 },{ 77,21,456 } },
		{ { 87,12,3906 },{ 87,12,3904 },{ 3,44,488 },{ 86,21,456 } },
		{ { 95,12,3906 },{ 95,12,3904 },{ 11,44,488 },{ 94,21,456 } },
		{ { 103,12,3906 },{ 103,12,3904 },{ 19,44,488 },{ 102,21,456 } },
		{ { 111,12,3907 },{ 111,12,3904 },{ 27,44,489 },{ 110,21,456 } },
		{ { 120,12,3907 },{ 120,12,3904 },{ 36,44,489 },{ 119,21,456 } },
		{ { 128,12,3907 },{ 128,12,3904 },{ 44,44,489 },{ 127,21,456 } },
		{ { 136,12,3907 },{ 136,12,3904 },{ 52,44,489 },{ 135,21,456 } },
		{ { 144,12,3907 },{ 144,12,3904 },{ 60,44,489 },{ 143,21,456 } },
		{ { 153,12,3907 },{ 153,12,3904 },{ 69,44,490 },{ 152,21,456 } },
		{ { 161,12,3395 },{ 149,188,3968 },{ 77,44,490 },{ 160,21,456 } },
		{ { 169,12,3395 },{ 198,21,3928 },{ 85,44,490 },{ 168,21,456 } },
		{ { 113,95,4001 },{ 201,69,3992 },{ 125,8,483 },{ 176,21,456 } },
		{ { 122,95,4001 },{ 200,21,3984 },{ 134,8,483 },{ 185,21,456 } },
		{ { 142,8,4067 },{ 208,21,3984 },{ 142,8,483 },{ 193,21,456 } },
		{ { 151,8,4067 },{ 47,15,4080 },{ 151,8,483 },{ 47,15,496 } },
		{ { 159,8,4067 },{ 55,15,4080 },{ 159,8,483 },{ 55,15,496 } },
		{ { 168,8,4067 },{ 64,15,4080 },{ 168,8,483 },{ 64,15,496 } },
		{ { 160,40,4075 },{ 72,15,4080 },{ 160,40,491 },{ 72,15,496 } },
		{ { 168,40,4075 },{ 80,15,4080 },{ 168,40,491 },{ 80,15,496 } },
		{ { 144,8,4082 },{ 88,15,4080 },{ 144,8,498 },{ 88,15,496 } }
	};
#endif // BASISD_SUPPORT_ETC2_EAC_A8

#if BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES
	static void create_etc2_eac_a8_conversion_table()
	{
		FILE* pFile = fopen("basisu_decoder_tables_etc2_eac_a8.inc", "w");

		for (uint32_t inten = 0; inten < 8; inten++)
		{
			for (uint32_t base = 0; base < 32; base++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(base, base, base, 255), false), inten);

				fprintf(pFile, "{");

				for (uint32_t sel_range = 0; sel_range < NUM_ETC2_EAC_SELECTOR_RANGES; sel_range++)
				{
					const uint32_t low_selector = s_etc2_eac_selector_ranges[sel_range].m_low;
					const uint32_t high_selector = s_etc2_eac_selector_ranges[sel_range].m_high;

					// We have a ETC1 base color and intensity, and a used selector range from low_selector-high_selector.
					// Now find the best ETC2 EAC A8 base/table/multiplier that fits these colors.

					uint8_t pixels[4];
					uint32_t num_pixels = 0;
					for (uint32_t s = low_selector; s <= high_selector; s++)
						pixels[num_pixels++] = block_colors[s].g;

					pack_eac_a8_results pack_results;
					pack_eac_a8_exhaustive(pack_results, pixels, num_pixels);

					etc1_g_to_eac_conversion& c = s_etc1_g_to_etc2_a8[base + inten * 32][sel_range];

					c.m_base = pack_results.m_base;
					c.m_table_mul = pack_results.m_table * 16 + pack_results.m_multiplier;
					c.m_trans = 0;

					for (uint32_t s = 0; s < 4; s++)
					{
						if ((s < low_selector) || (s > high_selector))
							continue;

						uint32_t etc2_selector = pack_results.m_selectors[s - low_selector];

						c.m_trans |= (etc2_selector << (s * 3));
					}

					fprintf(pFile, "{%u,%u,%u}", c.m_base, c.m_table_mul, c.m_trans);
					if (sel_range < (NUM_ETC2_EAC_SELECTOR_RANGES - 1))
						fprintf(pFile, ",");
				}

				fprintf(pFile, "},\n");
			}
		}

		fclose(pFile);
	}
#endif

#if BASISD_WRITE_NEW_ETC2_EAC_R11_TABLES
	struct pack_eac_r11_results
	{
		uint32_t m_base;
		uint32_t m_table;
		uint32_t m_multiplier;
		basisu::vector<uint8_t> m_selectors;
		basisu::vector<uint8_t> m_selectors_temp;
	};

	static uint64_t pack_eac_r11_exhaustive(pack_eac_r11_results& results, const uint8_t* pPixels, uint32_t num_pixels)
	{
		results.m_selectors.resize(num_pixels);
		results.m_selectors_temp.resize(num_pixels);

		uint64_t best_err = UINT64_MAX;

		for (uint32_t base_color = 0; base_color < 256; base_color++)
		{
			for (uint32_t multiplier = 0; multiplier < 16; multiplier++)
			{
				for (uint32_t table = 0; table < 16; table++)
				{
					uint64_t total_err = 0;

					for (uint32_t i = 0; i < num_pixels; i++)
					{
						// Convert 8-bit input to 11-bits
						const int a = (pPixels[i] * 2047 + 128) / 255;

						uint32_t best_s_err = UINT32_MAX;
						uint32_t best_s = 0;
						for (uint32_t s = 0; s < 8; s++)
						{
							int v = (int)(multiplier ? (multiplier * 8) : 1) * g_eac_modifier_table[table][s] + (int)base_color * 8 + 4;
							if (v < 0)
								v = 0;
							else if (v > 2047)
								v = 2047;

							uint32_t err = abs(a - v);
							if (err < best_s_err)
							{
								best_s_err = err;
								best_s = s;
							}
						}

						results.m_selectors_temp[i] = static_cast<uint8_t>(best_s);

						total_err += best_s_err * best_s_err;
						if (total_err >= best_err)
							break;
					}

					if (total_err < best_err)
					{
						best_err = total_err;
						results.m_base = base_color;
						results.m_multiplier = multiplier;
						results.m_table = table;
						results.m_selectors.swap(results.m_selectors_temp);
					}

				} // table

			} // multiplier

		} // base_color

		return best_err;
	}

	static void create_etc2_eac_r11_conversion_table()
	{
		FILE* pFile = nullptr;
		fopen_s(&pFile, "basisu_decoder_tables_etc2_eac_r11.inc", "w");

		for (uint32_t inten = 0; inten < 8; inten++)
		{
			for (uint32_t base = 0; base < 32; base++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(base, base, base, 255), false), inten);

				fprintf(pFile, "{");

				for (uint32_t sel_range = 0; sel_range < NUM_ETC2_EAC_SELECTOR_RANGES; sel_range++)
				{
					const uint32_t low_selector = s_etc2_eac_selector_ranges[sel_range].m_low;
					const uint32_t high_selector = s_etc2_eac_selector_ranges[sel_range].m_high;

					// We have a ETC1 base color and intensity, and a used selector range from low_selector-high_selector.
					// Now find the best ETC2 EAC R11 base/table/multiplier that fits these colors.

					uint8_t pixels[4];
					uint32_t num_pixels = 0;
					for (uint32_t s = low_selector; s <= high_selector; s++)
						pixels[num_pixels++] = block_colors[s].g;

					pack_eac_r11_results pack_results;
					pack_eac_r11_exhaustive(pack_results, pixels, num_pixels);

					etc1_g_to_eac_conversion c;

					c.m_base = (uint8_t)pack_results.m_base;
					c.m_table_mul = (uint8_t)(pack_results.m_table * 16 + pack_results.m_multiplier);
					c.m_trans = 0;

					for (uint32_t s = 0; s < 4; s++)
					{
						if ((s < low_selector) || (s > high_selector))
							continue;

						uint32_t etc2_selector = pack_results.m_selectors[s - low_selector];

						c.m_trans |= (etc2_selector << (s * 3));
					}

					fprintf(pFile, "{%u,%u,%u}", c.m_base, c.m_table_mul, c.m_trans);
					if (sel_range < (NUM_ETC2_EAC_SELECTOR_RANGES - 1))
						fprintf(pFile, ",");
				}

				fprintf(pFile, "},\n");
			}
		}

		fclose(pFile);
	}
#endif // BASISD_WRITE_NEW_ETC2_EAC_R11_TABLES

#if BASISD_WRITE_NEW_ASTC_TABLES
	static void create_etc1_to_astc_conversion_table_0_47();
	static void create_etc1_to_astc_conversion_table_0_255();
#endif

#if BASISD_SUPPORT_ASTC
	static void transcoder_init_astc();
#endif

#if BASISD_WRITE_NEW_BC7_MODE5_TABLES
	static void create_etc1_to_bc7_m5_color_conversion_table();
	static void create_etc1_to_bc7_m5_alpha_conversion_table();
#endif

#if BASISD_SUPPORT_BC7_MODE5
	static void transcoder_init_bc7_mode5();
#endif

#if BASISD_WRITE_NEW_ATC_TABLES
	static void create_etc1s_to_atc_conversion_tables();
#endif

#if BASISD_SUPPORT_ATC
	static void transcoder_init_atc();
#endif

#if BASISD_SUPPORT_PVRTC2
	static void transcoder_init_pvrtc2();
#endif

#if BASISD_SUPPORT_UASTC
	void uastc_init();
#endif

	static bool g_transcoder_initialized;
		
	// Library global initialization. Requires ~9 milliseconds when compiled and executed natively on a Core i7 2.2 GHz.
	// If this is too slow, these computed tables can easilky be moved to be compiled in.
	void basisu_transcoder_init()
	{
		if (g_transcoder_initialized)
      {
         BASISU_DEVEL_ERROR("basisu_transcoder::basisu_transcoder_init: Called more than once\n");      
			return;
      }
         
     BASISU_DEVEL_ERROR("basisu_transcoder::basisu_transcoder_init: Initializing (this is not an error)\n");      

#if BASISD_SUPPORT_UASTC
		uastc_init();
#endif

#if BASISD_SUPPORT_ASTC
		transcoder_init_astc();
#endif
				
#if BASISD_WRITE_NEW_ASTC_TABLES
		create_etc1_to_astc_conversion_table_0_47();
		create_etc1_to_astc_conversion_table_0_255();
		exit(0);
#endif

#if BASISD_WRITE_NEW_BC7_MODE5_TABLES
		create_etc1_to_bc7_m5_color_conversion_table();
		create_etc1_to_bc7_m5_alpha_conversion_table();
		exit(0);
#endif

#if BASISD_WRITE_NEW_DXT1_TABLES
		create_etc1_to_dxt1_5_conversion_table();
		create_etc1_to_dxt1_6_conversion_table();
		exit(0);
#endif

#if BASISD_WRITE_NEW_ETC2_EAC_A8_TABLES
		create_etc2_eac_a8_conversion_table();
		exit(0);
#endif

#if BASISD_WRITE_NEW_ATC_TABLES
		create_etc1s_to_atc_conversion_tables();
		exit(0);
#endif

#if BASISD_WRITE_NEW_ETC2_EAC_R11_TABLES
		create_etc2_eac_r11_conversion_table();
		exit(0);
#endif

#if BASISD_SUPPORT_DXT1 || BASISD_SUPPORT_UASTC
		uint8_t bc1_expand5[32];
		for (int i = 0; i < 32; i++)
			bc1_expand5[i] = static_cast<uint8_t>((i << 3) | (i >> 2));
		prepare_bc1_single_color_table(g_bc1_match5_equals_1, bc1_expand5, 32, 32, 1);
		prepare_bc1_single_color_table(g_bc1_match5_equals_0, bc1_expand5, 1, 32, 0);

		uint8_t bc1_expand6[64];
		for (int i = 0; i < 64; i++)
			bc1_expand6[i] = static_cast<uint8_t>((i << 2) | (i >> 4));
		prepare_bc1_single_color_table(g_bc1_match6_equals_1, bc1_expand6, 64, 64, 1);
		prepare_bc1_single_color_table(g_bc1_match6_equals_0, bc1_expand6, 1, 64, 0);

#if 0
		for (uint32_t i = 0; i < 256; i++)
		{
			printf("%u %u %u\n", i, (i * 63 + 127) / 255, g_bc1_match6_equals_0[i].m_hi);
		}
		exit(0);
#endif

#endif

#if BASISD_SUPPORT_DXT1
		for (uint32_t i = 0; i < NUM_ETC1_TO_DXT1_SELECTOR_RANGES; i++)
		{
			uint32_t l = g_etc1_to_dxt1_selector_ranges[i].m_low;
			uint32_t h = g_etc1_to_dxt1_selector_ranges[i].m_high;
			g_etc1_to_dxt1_selector_range_index[l][h] = i;
		}

		for (uint32_t sm = 0; sm < NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS; sm++)
		{
			uint8_t etc1_to_dxt1_selector_mappings_raw_dxt1[4];
			uint8_t etc1_to_dxt1_selector_mappings_raw_dxt1_inv[4];

			for (uint32_t j = 0; j < 4; j++)
			{
				static const uint8_t s_linear_dxt1_to_dxt1[4] = { 0, 2, 3, 1 };
				static const uint8_t s_dxt1_inverted_xlat[4] = { 1, 0, 3, 2 };

				etc1_to_dxt1_selector_mappings_raw_dxt1[j] = (uint8_t)s_linear_dxt1_to_dxt1[g_etc1_to_dxt1_selector_mappings[sm][j]];
				etc1_to_dxt1_selector_mappings_raw_dxt1_inv[j] = (uint8_t)s_dxt1_inverted_xlat[etc1_to_dxt1_selector_mappings_raw_dxt1[j]];
			}

			for (uint32_t i = 0; i < 256; i++)
			{
				uint32_t k = 0, k_inv = 0;
				for (uint32_t s = 0; s < 4; s++)
				{
					k |= (etc1_to_dxt1_selector_mappings_raw_dxt1[(i >> (s * 2)) & 3] << (s * 2));
					k_inv |= (etc1_to_dxt1_selector_mappings_raw_dxt1_inv[(i >> (s * 2)) & 3] << (s * 2));
				}
				g_etc1_to_dxt1_selector_mappings_raw_dxt1_256[sm][i] = (uint8_t)k;
				g_etc1_to_dxt1_selector_mappings_raw_dxt1_inv_256[sm][i] = (uint8_t)k_inv;
			}
		}
#endif

#if BASISD_SUPPORT_BC7_MODE5
		transcoder_init_bc7_mode5();
#endif

#if BASISD_SUPPORT_ATC
		transcoder_init_atc();
#endif

#if BASISD_SUPPORT_PVRTC2
		transcoder_init_pvrtc2();
#endif

		g_transcoder_initialized = true;
	}

#if BASISD_SUPPORT_DXT1
	static void convert_etc1s_to_dxt1(dxt1_block* pDst_block, const endpoint *pEndpoints, const selector* pSelector, bool use_threecolor_blocks)
	{
#if !BASISD_WRITE_NEW_DXT1_TABLES
		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const color32& base_color = pEndpoints->m_color5;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (low_selector == high_selector)
		{
			uint32_t r, g, b;
			decoder_etc_block::get_block_color5(base_color, inten_table, low_selector, r, g, b);

			uint32_t mask = 0xAA;
			uint32_t max16 = (g_bc1_match5_equals_1[r].m_hi << 11) | (g_bc1_match6_equals_1[g].m_hi << 5) | g_bc1_match5_equals_1[b].m_hi;
			uint32_t min16 = (g_bc1_match5_equals_1[r].m_lo << 11) | (g_bc1_match6_equals_1[g].m_lo << 5) | g_bc1_match5_equals_1[b].m_lo;

			if ((!use_threecolor_blocks) && (min16 == max16))
			{
				// This is an annoying edge case that impacts BC3.
				// This is to guarantee that BC3 blocks never use punchthrough alpha (3 color) mode, which isn't supported on some (all?) GPU's.
				mask = 0;

				// Make l > h
				if (min16 > 0)
					min16--;
				else
				{
					// l = h = 0
					assert(min16 == max16 && max16 == 0);

					max16 = 1;
					min16 = 0;
					mask = 0x55;
				}

				assert(max16 > min16);
			}

			if (max16 < min16)
			{
				std::swap(max16, min16);
				mask ^= 0x55;
			}

			pDst_block->set_low_color(static_cast<uint16_t>(max16));
			pDst_block->set_high_color(static_cast<uint16_t>(min16));
			pDst_block->m_selectors[0] = static_cast<uint8_t>(mask);
			pDst_block->m_selectors[1] = static_cast<uint8_t>(mask);
			pDst_block->m_selectors[2] = static_cast<uint8_t>(mask);
			pDst_block->m_selectors[3] = static_cast<uint8_t>(mask);

			return;
		}
		else if ((inten_table >= 7) && (pSelector->m_num_unique_selectors == 2) && (pSelector->m_lo_selector == 0) && (pSelector->m_hi_selector == 3))
		{
			color32 block_colors[4];

			decoder_etc_block::get_block_colors5(block_colors, base_color, inten_table);

			const uint32_t r0 = block_colors[0].r;
			const uint32_t g0 = block_colors[0].g;
			const uint32_t b0 = block_colors[0].b;

			const uint32_t r1 = block_colors[3].r;
			const uint32_t g1 = block_colors[3].g;
			const uint32_t b1 = block_colors[3].b;

			uint32_t max16 = (g_bc1_match5_equals_0[r0].m_hi << 11) | (g_bc1_match6_equals_0[g0].m_hi << 5) | g_bc1_match5_equals_0[b0].m_hi;
			uint32_t min16 = (g_bc1_match5_equals_0[r1].m_hi << 11) | (g_bc1_match6_equals_0[g1].m_hi << 5) | g_bc1_match5_equals_0[b1].m_hi;

			uint32_t l = 0, h = 1;

			if (min16 == max16)
			{
				// Make l > h
				if (min16 > 0)
				{
					min16--;

					l = 0;
					h = 0;
				}
				else
				{
					// l = h = 0
					assert(min16 == max16 && max16 == 0);

					max16 = 1;
					min16 = 0;

					l = 1;
					h = 1;
				}

				assert(max16 > min16);
			}

			if (max16 < min16)
			{
				std::swap(max16, min16);
				l = 1;
				h = 0;
			}

			pDst_block->set_low_color((uint16_t)max16);
			pDst_block->set_high_color((uint16_t)min16);

			for (uint32_t y = 0; y < 4; y++)
			{
				for (uint32_t x = 0; x < 4; x++)
				{
					uint32_t s = pSelector->get_selector(x, y);
					pDst_block->set_selector(x, y, (s == 3) ? h : l);
				}
			}

			return;
		}

		const uint32_t selector_range_table = g_etc1_to_dxt1_selector_range_index[low_selector][high_selector];

		//[32][8][RANGES][MAPPING]
		const etc1_to_dxt1_56_solution* pTable_r = &g_etc1_to_dxt_5[(inten_table * 32 + base_color.r) * (NUM_ETC1_TO_DXT1_SELECTOR_RANGES * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS];
		const etc1_to_dxt1_56_solution* pTable_g = &g_etc1_to_dxt_6[(inten_table * 32 + base_color.g) * (NUM_ETC1_TO_DXT1_SELECTOR_RANGES * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS];
		const etc1_to_dxt1_56_solution* pTable_b = &g_etc1_to_dxt_5[(inten_table * 32 + base_color.b) * (NUM_ETC1_TO_DXT1_SELECTOR_RANGES * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS];

		uint32_t best_err = UINT_MAX;
		uint32_t best_mapping = 0;

		assert(NUM_ETC1_TO_DXT1_SELECTOR_MAPPINGS == 10);
#define DO_ITER(m) { uint32_t total_err = pTable_r[m].m_err + pTable_g[m].m_err + pTable_b[m].m_err; if (total_err < best_err) { best_err = total_err; best_mapping = m; } }
		DO_ITER(0); DO_ITER(1); DO_ITER(2); DO_ITER(3); DO_ITER(4);
		DO_ITER(5); DO_ITER(6); DO_ITER(7); DO_ITER(8); DO_ITER(9);
#undef DO_ITER

		uint32_t l = dxt1_block::pack_unscaled_color(pTable_r[best_mapping].m_lo, pTable_g[best_mapping].m_lo, pTable_b[best_mapping].m_lo);
		uint32_t h = dxt1_block::pack_unscaled_color(pTable_r[best_mapping].m_hi, pTable_g[best_mapping].m_hi, pTable_b[best_mapping].m_hi);

		const uint8_t* pSelectors_xlat_256 = &g_etc1_to_dxt1_selector_mappings_raw_dxt1_256[best_mapping][0];

		if (l < h)
		{
			std::swap(l, h);
			pSelectors_xlat_256 = &g_etc1_to_dxt1_selector_mappings_raw_dxt1_inv_256[best_mapping][0];
		}
				
		pDst_block->set_low_color(static_cast<uint16_t>(l));
		pDst_block->set_high_color(static_cast<uint16_t>(h));

		if (l == h)
		{
			uint8_t mask = 0;

			if (!use_threecolor_blocks)
			{
				// This is an annoying edge case that impacts BC3.

				// Make l > h
				if (h > 0)
					h--;
				else
				{
					// l = h = 0
					assert(l == h && h == 0);

					h = 0;
					l = 1;
					mask = 0x55;
				}

				assert(l > h);
				pDst_block->set_low_color(static_cast<uint16_t>(l));
				pDst_block->set_high_color(static_cast<uint16_t>(h));
			}

			pDst_block->m_selectors[0] = mask;
			pDst_block->m_selectors[1] = mask;
			pDst_block->m_selectors[2] = mask;
			pDst_block->m_selectors[3] = mask;

			return;
		}

		pDst_block->m_selectors[0] = pSelectors_xlat_256[pSelector->m_selectors[0]];
		pDst_block->m_selectors[1] = pSelectors_xlat_256[pSelector->m_selectors[1]];
		pDst_block->m_selectors[2] = pSelectors_xlat_256[pSelector->m_selectors[2]];
		pDst_block->m_selectors[3] = pSelectors_xlat_256[pSelector->m_selectors[3]];
#endif
	}

#if BASISD_ENABLE_DEBUG_FLAGS
	static void convert_etc1s_to_dxt1_vis(dxt1_block* pDst_block, const endpoint* pEndpoints, const selector* pSelector, bool use_threecolor_blocks)
	{
		convert_etc1s_to_dxt1(pDst_block, pEndpoints, pSelector, use_threecolor_blocks);

		if (g_debug_flags & cDebugFlagVisBC1Sels)
		{
			uint32_t l = dxt1_block::pack_unscaled_color(31, 63, 31);
			uint32_t h = dxt1_block::pack_unscaled_color(0, 0, 0);
			pDst_block->set_low_color(static_cast<uint16_t>(l));
			pDst_block->set_high_color(static_cast<uint16_t>(h));
		}
		else if (g_debug_flags & cDebugFlagVisBC1Endpoints)
		{
			for (uint32_t y = 0; y < 4; y++)
				for (uint32_t x = 0; x < 4; x++)
					pDst_block->set_selector(x, y, (y < 2) ? 0 : 1);
		}
	}
#endif
#endif

#if BASISD_SUPPORT_FXT1
	struct fxt1_block
	{
		union
		{
			struct
			{
				uint64_t m_t00 : 2;
				uint64_t m_t01 : 2;
				uint64_t m_t02 : 2;
				uint64_t m_t03 : 2;
				uint64_t m_t04 : 2;
				uint64_t m_t05 : 2;
				uint64_t m_t06 : 2;
				uint64_t m_t07 : 2;
				uint64_t m_t08 : 2;
				uint64_t m_t09 : 2;
				uint64_t m_t10 : 2;
				uint64_t m_t11 : 2;
				uint64_t m_t12 : 2;
				uint64_t m_t13 : 2;
				uint64_t m_t14 : 2;
				uint64_t m_t15 : 2;
				uint64_t m_t16 : 2;
				uint64_t m_t17 : 2;
				uint64_t m_t18 : 2;
				uint64_t m_t19 : 2;
				uint64_t m_t20 : 2;
				uint64_t m_t21 : 2;
				uint64_t m_t22 : 2;
				uint64_t m_t23 : 2;
				uint64_t m_t24 : 2;
				uint64_t m_t25 : 2;
				uint64_t m_t26 : 2;
				uint64_t m_t27 : 2;
				uint64_t m_t28 : 2;
				uint64_t m_t29 : 2;
				uint64_t m_t30 : 2;
				uint64_t m_t31 : 2;
			} m_lo;
			uint64_t m_lo_bits;
			uint8_t m_sels[8];
		};
		union
		{
			struct
			{
#ifdef BASISU_USE_ORIGINAL_3DFX_FXT1_ENCODING
				uint64_t m_b1 : 5;
				uint64_t m_g1 : 5;
				uint64_t m_r1 : 5;
				uint64_t m_b0 : 5;
				uint64_t m_g0 : 5;
				uint64_t m_r0 : 5;
				uint64_t m_b3 : 5;
				uint64_t m_g3 : 5;
				uint64_t m_r3 : 5;
				uint64_t m_b2 : 5;
				uint64_t m_g2 : 5;
				uint64_t m_r2 : 5;
#else
				uint64_t m_b0 : 5;
				uint64_t m_g0 : 5;
				uint64_t m_r0 : 5;
				uint64_t m_b1 : 5;
				uint64_t m_g1 : 5;
				uint64_t m_r1 : 5;
				uint64_t m_b2 : 5;
				uint64_t m_g2 : 5;
				uint64_t m_r2 : 5;
				uint64_t m_b3 : 5;
				uint64_t m_g3 : 5;
				uint64_t m_r3 : 5;
#endif
				uint64_t m_alpha : 1;
				uint64_t m_glsb : 2;
				uint64_t m_mode : 1;
			} m_hi;
			uint64_t m_hi_bits;
		};
	};

	static uint8_t conv_dxt1_to_fxt1_sels(uint32_t sels)
	{
		static uint8_t s_conv_table[16] = { 0, 3, 1, 2, 12, 15, 13, 14, 4, 7, 5, 6, 8, 11, 9, 10 };
		return s_conv_table[sels & 15] | (s_conv_table[sels >> 4] << 4);
	}

	static void convert_etc1s_to_fxt1(void *pDst, const endpoint *pEndpoints, const selector *pSelectors, uint32_t fxt1_subblock)
	{
		fxt1_block* pBlock = static_cast<fxt1_block*>(pDst);

		// CC_MIXED is basically DXT1 with different encoding tricks.
		// So transcode ETC1S to DXT1, then transcode that to FXT1 which is easy and nearly lossless. 
		// (It's not completely lossless because FXT1 rounds in its color lerps while DXT1 doesn't, but it should be good enough.)
		dxt1_block blk;
		convert_etc1s_to_dxt1(&blk, pEndpoints, pSelectors, false);

		const uint32_t l = blk.get_low_color();
		const uint32_t h = blk.get_high_color();

		color32 color0((l >> 11) & 31, (l >> 5) & 63, l & 31, 255);
		color32 color1((h >> 11) & 31, (h >> 5) & 63, h & 31, 255);

		uint32_t g0 = color0.g & 1;
		uint32_t g1 = color1.g & 1;
		
		color0.g >>= 1;
		color1.g >>= 1;

		blk.m_selectors[0] = conv_dxt1_to_fxt1_sels(blk.m_selectors[0]);
		blk.m_selectors[1] = conv_dxt1_to_fxt1_sels(blk.m_selectors[1]);
		blk.m_selectors[2] = conv_dxt1_to_fxt1_sels(blk.m_selectors[2]);
		blk.m_selectors[3] = conv_dxt1_to_fxt1_sels(blk.m_selectors[3]);
		
		if ((blk.get_selector(0, 0) >> 1) != (g0 ^ g1))
		{
			std::swap(color0, color1);
			std::swap(g0, g1);

			blk.m_selectors[0] ^= 0xFF;
			blk.m_selectors[1] ^= 0xFF;
			blk.m_selectors[2] ^= 0xFF;
			blk.m_selectors[3] ^= 0xFF;
		}

		if (fxt1_subblock == 0)
		{
			pBlock->m_hi.m_mode = 1; 
			pBlock->m_hi.m_alpha = 0;
			pBlock->m_hi.m_glsb = g1 | (g1 << 1);
			pBlock->m_hi.m_r0 = color0.r;
			pBlock->m_hi.m_g0 = color0.g;
			pBlock->m_hi.m_b0 = color0.b;
			pBlock->m_hi.m_r1 = color1.r;
			pBlock->m_hi.m_g1 = color1.g;
			pBlock->m_hi.m_b1 = color1.b;
			pBlock->m_hi.m_r2 = color0.r;
			pBlock->m_hi.m_g2 = color0.g;
			pBlock->m_hi.m_b2 = color0.b;
			pBlock->m_hi.m_r3 = color1.r;
			pBlock->m_hi.m_g3 = color1.g;
			pBlock->m_hi.m_b3 = color1.b;
			pBlock->m_sels[0] = blk.m_selectors[0];
			pBlock->m_sels[1] = blk.m_selectors[1];
			pBlock->m_sels[2] = blk.m_selectors[2];
			pBlock->m_sels[3] = blk.m_selectors[3];

			static const uint8_t s_border_dup[4] = { 0, 85, 170, 255 };
			pBlock->m_sels[4] = s_border_dup[blk.m_selectors[0] >> 6];
			pBlock->m_sels[5] = s_border_dup[blk.m_selectors[1] >> 6];
			pBlock->m_sels[6] = s_border_dup[blk.m_selectors[2] >> 6];
			pBlock->m_sels[7] = s_border_dup[blk.m_selectors[3] >> 6];
		}
		else
		{
			pBlock->m_hi.m_glsb = (pBlock->m_hi.m_glsb & 1) | (g1 << 1);
			pBlock->m_hi.m_r2 = color0.r;
			pBlock->m_hi.m_g2 = color0.g;
			pBlock->m_hi.m_b2 = color0.b;
			pBlock->m_hi.m_r3 = color1.r;
			pBlock->m_hi.m_g3 = color1.g;
			pBlock->m_hi.m_b3 = color1.b;
			pBlock->m_sels[4] = blk.m_selectors[0];
			pBlock->m_sels[5] = blk.m_selectors[1];
			pBlock->m_sels[6] = blk.m_selectors[2];
			pBlock->m_sels[7] = blk.m_selectors[3];
		}
	}
#endif // BASISD_SUPPORT_FXT1
#if BASISD_SUPPORT_DXT5A
	static dxt_selector_range s_dxt5a_selector_ranges[] =
	{
		{ 0, 3 },

		{ 1, 3 },
		{ 0, 2 },

		{ 1, 2 },
	};

	const uint32_t NUM_DXT5A_SELECTOR_RANGES = sizeof(s_dxt5a_selector_ranges) / sizeof(s_dxt5a_selector_ranges[0]);

	struct etc1_g_to_dxt5a_conversion
	{
		uint8_t m_lo, m_hi;
		uint16_t m_trans;
	};

	static etc1_g_to_dxt5a_conversion g_etc1_g_to_dxt5a[32 * 8][NUM_DXT5A_SELECTOR_RANGES] =
	{
		{ { 8, 0, 393 },{ 8, 0, 392 },{ 2, 0, 9 },{ 2, 0, 8 }, }, { { 6, 16, 710 },{ 16, 6, 328 },{ 0, 10, 96 },{ 10, 6, 8 }, },
		{ { 28, 5, 1327 },{ 24, 14, 328 },{ 8, 18, 96 },{ 18, 14, 8 }, }, { { 36, 13, 1327 },{ 32, 22, 328 },{ 16, 26, 96 },{ 26, 22, 8 }, },
		{ { 45, 22, 1327 },{ 41, 31, 328 },{ 25, 35, 96 },{ 35, 31, 8 }, }, { { 53, 30, 1327 },{ 49, 39, 328 },{ 33, 43, 96 },{ 43, 39, 8 }, },
		{ { 61, 38, 1327 },{ 57, 47, 328 },{ 41, 51, 96 },{ 51, 47, 8 }, }, { { 69, 46, 1327 },{ 65, 55, 328 },{ 49, 59, 96 },{ 59, 55, 8 }, },
		{ { 78, 55, 1327 },{ 74, 64, 328 },{ 58, 68, 96 },{ 68, 64, 8 }, }, { { 86, 63, 1327 },{ 82, 72, 328 },{ 66, 76, 96 },{ 76, 72, 8 }, },
		{ { 94, 71, 1327 },{ 90, 80, 328 },{ 74, 84, 96 },{ 84, 80, 8 }, }, { { 102, 79, 1327 },{ 98, 88, 328 },{ 82, 92, 96 },{ 92, 88, 8 }, },
		{ { 111, 88, 1327 },{ 107, 97, 328 },{ 91, 101, 96 },{ 101, 97, 8 }, }, { { 119, 96, 1327 },{ 115, 105, 328 },{ 99, 109, 96 },{ 109, 105, 8 }, },
		{ { 127, 104, 1327 },{ 123, 113, 328 },{ 107, 117, 96 },{ 117, 113, 8 }, }, { { 135, 112, 1327 },{ 131, 121, 328 },{ 115, 125, 96 },{ 125, 121, 8 }, },
		{ { 144, 121, 1327 },{ 140, 130, 328 },{ 124, 134, 96 },{ 134, 130, 8 }, }, { { 152, 129, 1327 },{ 148, 138, 328 },{ 132, 142, 96 },{ 142, 138, 8 }, },
		{ { 160, 137, 1327 },{ 156, 146, 328 },{ 140, 150, 96 },{ 150, 146, 8 }, }, { { 168, 145, 1327 },{ 164, 154, 328 },{ 148, 158, 96 },{ 158, 154, 8 }, },
		{ { 177, 154, 1327 },{ 173, 163, 328 },{ 157, 167, 96 },{ 167, 163, 8 }, }, { { 185, 162, 1327 },{ 181, 171, 328 },{ 165, 175, 96 },{ 175, 171, 8 }, },
		{ { 193, 170, 1327 },{ 189, 179, 328 },{ 173, 183, 96 },{ 183, 179, 8 }, }, { { 201, 178, 1327 },{ 197, 187, 328 },{ 181, 191, 96 },{ 191, 187, 8 }, },
		{ { 210, 187, 1327 },{ 206, 196, 328 },{ 190, 200, 96 },{ 200, 196, 8 }, }, { { 218, 195, 1327 },{ 214, 204, 328 },{ 198, 208, 96 },{ 208, 204, 8 }, },
		{ { 226, 203, 1327 },{ 222, 212, 328 },{ 206, 216, 96 },{ 216, 212, 8 }, }, { { 234, 211, 1327 },{ 230, 220, 328 },{ 214, 224, 96 },{ 224, 220, 8 }, },
		{ { 243, 220, 1327 },{ 239, 229, 328 },{ 223, 233, 96 },{ 233, 229, 8 }, }, { { 251, 228, 1327 },{ 247, 237, 328 },{ 231, 241, 96 },{ 241, 237, 8 }, },
		{ { 239, 249, 3680 },{ 245, 249, 3648 },{ 239, 249, 96 },{ 249, 245, 8 }, }, { { 247, 253, 4040 },{ 255, 253, 8 },{ 247, 253, 456 },{ 255, 253, 8 }, },
		{ { 5, 17, 566 },{ 5, 17, 560 },{ 5, 0, 9 },{ 5, 0, 8 }, }, { { 25, 0, 313 },{ 25, 3, 328 },{ 13, 0, 49 },{ 13, 3, 8 }, },
		{ { 39, 0, 1329 },{ 33, 11, 328 },{ 11, 21, 70 },{ 21, 11, 8 }, }, { { 47, 7, 1329 },{ 41, 19, 328 },{ 29, 7, 33 },{ 29, 19, 8 }, },
		{ { 50, 11, 239 },{ 50, 28, 328 },{ 38, 16, 33 },{ 38, 28, 8 }, }, { { 92, 13, 2423 },{ 58, 36, 328 },{ 46, 24, 33 },{ 46, 36, 8 }, },
		{ { 100, 21, 2423 },{ 66, 44, 328 },{ 54, 32, 33 },{ 54, 44, 8 }, }, { { 86, 7, 1253 },{ 74, 52, 328 },{ 62, 40, 33 },{ 62, 52, 8 }, },
		{ { 95, 16, 1253 },{ 83, 61, 328 },{ 71, 49, 33 },{ 71, 61, 8 }, }, { { 103, 24, 1253 },{ 91, 69, 328 },{ 79, 57, 33 },{ 79, 69, 8 }, },
		{ { 111, 32, 1253 },{ 99, 77, 328 },{ 87, 65, 33 },{ 87, 77, 8 }, }, { { 119, 40, 1253 },{ 107, 85, 328 },{ 95, 73, 33 },{ 95, 85, 8 }, },
		{ { 128, 49, 1253 },{ 116, 94, 328 },{ 104, 82, 33 },{ 104, 94, 8 }, }, { { 136, 57, 1253 },{ 124, 102, 328 },{ 112, 90, 33 },{ 112, 102, 8 }, },
		{ { 144, 65, 1253 },{ 132, 110, 328 },{ 120, 98, 33 },{ 120, 110, 8 }, }, { { 152, 73, 1253 },{ 140, 118, 328 },{ 128, 106, 33 },{ 128, 118, 8 }, },
		{ { 161, 82, 1253 },{ 149, 127, 328 },{ 137, 115, 33 },{ 137, 127, 8 }, }, { { 169, 90, 1253 },{ 157, 135, 328 },{ 145, 123, 33 },{ 145, 135, 8 }, },
		{ { 177, 98, 1253 },{ 165, 143, 328 },{ 153, 131, 33 },{ 153, 143, 8 }, }, { { 185, 106, 1253 },{ 173, 151, 328 },{ 161, 139, 33 },{ 161, 151, 8 }, },
		{ { 194, 115, 1253 },{ 182, 160, 328 },{ 170, 148, 33 },{ 170, 160, 8 }, }, { { 202, 123, 1253 },{ 190, 168, 328 },{ 178, 156, 33 },{ 178, 168, 8 }, },
		{ { 210, 131, 1253 },{ 198, 176, 328 },{ 186, 164, 33 },{ 186, 176, 8 }, }, { { 218, 139, 1253 },{ 206, 184, 328 },{ 194, 172, 33 },{ 194, 184, 8 }, },
		{ { 227, 148, 1253 },{ 215, 193, 328 },{ 203, 181, 33 },{ 203, 193, 8 }, }, { { 235, 156, 1253 },{ 223, 201, 328 },{ 211, 189, 33 },{ 211, 201, 8 }, },
		{ { 243, 164, 1253 },{ 231, 209, 328 },{ 219, 197, 33 },{ 219, 209, 8 }, }, { { 183, 239, 867 },{ 239, 217, 328 },{ 227, 205, 33 },{ 227, 217, 8 }, },
		{ { 254, 214, 1329 },{ 248, 226, 328 },{ 236, 214, 33 },{ 236, 226, 8 }, }, { { 222, 244, 3680 },{ 234, 244, 3648 },{ 244, 222, 33 },{ 244, 234, 8 }, },
		{ { 230, 252, 3680 },{ 242, 252, 3648 },{ 252, 230, 33 },{ 252, 242, 8 }, }, { { 238, 250, 4040 },{ 255, 250, 8 },{ 238, 250, 456 },{ 255, 250, 8 }, },
		{ { 9, 29, 566 },{ 9, 29, 560 },{ 9, 0, 9 },{ 9, 0, 8 }, }, { { 17, 37, 566 },{ 17, 37, 560 },{ 17, 0, 9 },{ 17, 0, 8 }, },
		{ { 45, 0, 313 },{ 45, 0, 312 },{ 25, 0, 49 },{ 25, 7, 8 }, }, { { 14, 63, 2758 },{ 5, 53, 784 },{ 15, 33, 70 },{ 33, 15, 8 }, },
		{ { 71, 6, 1329 },{ 72, 4, 1328 },{ 42, 4, 33 },{ 42, 24, 8 }, }, { { 70, 3, 239 },{ 70, 2, 232 },{ 50, 12, 33 },{ 50, 32, 8 }, },
		{ { 0, 98, 2842 },{ 78, 10, 232 },{ 58, 20, 33 },{ 58, 40, 8 }, }, { { 97, 27, 1329 },{ 86, 18, 232 },{ 66, 28, 33 },{ 66, 48, 8 }, },
		{ { 0, 94, 867 },{ 95, 27, 232 },{ 75, 37, 33 },{ 75, 57, 8 }, }, { { 8, 102, 867 },{ 103, 35, 232 },{ 83, 45, 33 },{ 83, 65, 8 }, },
		{ { 12, 112, 867 },{ 111, 43, 232 },{ 91, 53, 33 },{ 91, 73, 8 }, }, { { 139, 2, 1253 },{ 119, 51, 232 },{ 99, 61, 33 },{ 99, 81, 8 }, },
		{ { 148, 13, 1253 },{ 128, 60, 232 },{ 108, 70, 33 },{ 108, 90, 8 }, }, { { 156, 21, 1253 },{ 136, 68, 232 },{ 116, 78, 33 },{ 116, 98, 8 }, },
		{ { 164, 29, 1253 },{ 144, 76, 232 },{ 124, 86, 33 },{ 124, 106, 8 }, }, { { 172, 37, 1253 },{ 152, 84, 232 },{ 132, 94, 33 },{ 132, 114, 8 }, },
		{ { 181, 46, 1253 },{ 161, 93, 232 },{ 141, 103, 33 },{ 141, 123, 8 }, }, { { 189, 54, 1253 },{ 169, 101, 232 },{ 149, 111, 33 },{ 149, 131, 8 }, },
		{ { 197, 62, 1253 },{ 177, 109, 232 },{ 157, 119, 33 },{ 157, 139, 8 }, }, { { 205, 70, 1253 },{ 185, 117, 232 },{ 165, 127, 33 },{ 165, 147, 8 }, },
		{ { 214, 79, 1253 },{ 194, 126, 232 },{ 174, 136, 33 },{ 174, 156, 8 }, }, { { 222, 87, 1253 },{ 202, 134, 232 },{ 182, 144, 33 },{ 182, 164, 8 }, },
		{ { 230, 95, 1253 },{ 210, 142, 232 },{ 190, 152, 33 },{ 190, 172, 8 }, }, { { 238, 103, 1253 },{ 218, 150, 232 },{ 198, 160, 33 },{ 198, 180, 8 }, },
		{ { 247, 112, 1253 },{ 227, 159, 232 },{ 207, 169, 33 },{ 207, 189, 8 }, }, { { 255, 120, 1253 },{ 235, 167, 232 },{ 215, 177, 33 },{ 215, 197, 8 }, },
		{ { 146, 243, 867 },{ 243, 175, 232 },{ 223, 185, 33 },{ 223, 205, 8 }, }, { { 184, 231, 3682 },{ 203, 251, 784 },{ 231, 193, 33 },{ 231, 213, 8 }, },
		{ { 193, 240, 3682 },{ 222, 240, 3648 },{ 240, 202, 33 },{ 240, 222, 8 }, }, { { 255, 210, 169 },{ 230, 248, 3648 },{ 248, 210, 33 },{ 248, 230, 8 }, },
		{ { 218, 238, 4040 },{ 255, 238, 8 },{ 218, 238, 456 },{ 255, 238, 8 }, }, { { 226, 246, 4040 },{ 255, 246, 8 },{ 226, 246, 456 },{ 255, 246, 8 }, },
		{ { 13, 42, 566 },{ 13, 42, 560 },{ 13, 0, 9 },{ 13, 0, 8 }, }, { { 50, 0, 329 },{ 50, 0, 328 },{ 21, 0, 9 },{ 21, 0, 8 }, },
		{ { 29, 58, 566 },{ 67, 2, 1352 },{ 3, 29, 70 },{ 29, 3, 8 }, }, { { 10, 79, 2758 },{ 76, 11, 1352 },{ 11, 37, 70 },{ 37, 11, 8 }, },
		{ { 7, 75, 790 },{ 7, 75, 784 },{ 20, 46, 70 },{ 46, 20, 8 }, }, { { 15, 83, 790 },{ 97, 1, 1328 },{ 28, 54, 70 },{ 54, 28, 8 }, },
		{ { 101, 7, 1329 },{ 105, 9, 1328 },{ 62, 0, 39 },{ 62, 36, 8 }, }, { { 99, 1, 239 },{ 99, 3, 232 },{ 1, 71, 98 },{ 70, 44, 8 }, },
		{ { 107, 11, 239 },{ 108, 12, 232 },{ 10, 80, 98 },{ 79, 53, 8 }, }, { { 115, 19, 239 },{ 116, 20, 232 },{ 18, 88, 98 },{ 87, 61, 8 }, },
		{ { 123, 27, 239 },{ 124, 28, 232 },{ 26, 96, 98 },{ 95, 69, 8 }, }, { { 131, 35, 239 },{ 132, 36, 232 },{ 34, 104, 98 },{ 103, 77, 8 }, },
		{ { 140, 44, 239 },{ 141, 45, 232 },{ 43, 113, 98 },{ 112, 86, 8 }, }, { { 148, 52, 239 },{ 149, 53, 232 },{ 51, 121, 98 },{ 120, 94, 8 }, },
		{ { 156, 60, 239 },{ 157, 61, 232 },{ 59, 129, 98 },{ 128, 102, 8 }, }, { { 164, 68, 239 },{ 165, 69, 232 },{ 67, 137, 98 },{ 136, 110, 8 }, },
		{ { 173, 77, 239 },{ 174, 78, 232 },{ 76, 146, 98 },{ 145, 119, 8 }, }, { { 181, 85, 239 },{ 182, 86, 232 },{ 84, 154, 98 },{ 153, 127, 8 }, },
		{ { 189, 93, 239 },{ 190, 94, 232 },{ 92, 162, 98 },{ 161, 135, 8 }, }, { { 197, 101, 239 },{ 198, 102, 232 },{ 100, 170, 98 },{ 169, 143, 8 }, },
		{ { 206, 110, 239 },{ 207, 111, 232 },{ 109, 179, 98 },{ 178, 152, 8 }, }, { { 214, 118, 239 },{ 215, 119, 232 },{ 117, 187, 98 },{ 186, 160, 8 }, },
		{ { 222, 126, 239 },{ 223, 127, 232 },{ 125, 195, 98 },{ 194, 168, 8 }, }, { { 230, 134, 239 },{ 231, 135, 232 },{ 133, 203, 98 },{ 202, 176, 8 }, },
		{ { 239, 143, 239 },{ 240, 144, 232 },{ 142, 212, 98 },{ 211, 185, 8 }, }, { { 247, 151, 239 },{ 180, 248, 784 },{ 150, 220, 98 },{ 219, 193, 8 }, },
		{ { 159, 228, 3682 },{ 201, 227, 3648 },{ 158, 228, 98 },{ 227, 201, 8 }, }, { { 181, 249, 3928 },{ 209, 235, 3648 },{ 166, 236, 98 },{ 235, 209, 8 }, },
		{ { 255, 189, 169 },{ 218, 244, 3648 },{ 175, 245, 98 },{ 244, 218, 8 }, }, { { 197, 226, 4040 },{ 226, 252, 3648 },{ 183, 253, 98 },{ 252, 226, 8 }, },
		{ { 205, 234, 4040 },{ 255, 234, 8 },{ 205, 234, 456 },{ 255, 234, 8 }, }, { { 213, 242, 4040 },{ 255, 242, 8 },{ 213, 242, 456 },{ 255, 242, 8 }, },
		{ { 18, 60, 566 },{ 18, 60, 560 },{ 18, 0, 9 },{ 18, 0, 8 }, }, { { 26, 68, 566 },{ 26, 68, 560 },{ 26, 0, 9 },{ 26, 0, 8 }, },
		{ { 34, 76, 566 },{ 34, 76, 560 },{ 34, 0, 9 },{ 34, 0, 8 }, }, { { 5, 104, 2758 },{ 98, 5, 1352 },{ 42, 0, 57 },{ 42, 6, 8 }, },
		{ { 92, 0, 313 },{ 93, 1, 312 },{ 15, 51, 70 },{ 51, 15, 8 }, }, { { 3, 101, 790 },{ 3, 101, 784 },{ 0, 59, 88 },{ 59, 23, 8 }, },
		{ { 14, 107, 790 },{ 11, 109, 784 },{ 31, 67, 70 },{ 67, 31, 8 }, }, { { 19, 117, 790 },{ 19, 117, 784 },{ 39, 75, 70 },{ 75, 39, 8 }, },
		{ { 28, 126, 790 },{ 28, 126, 784 },{ 83, 5, 33 },{ 84, 48, 8 }, }, { { 132, 0, 239 },{ 36, 134, 784 },{ 91, 13, 33 },{ 92, 56, 8 }, },
		{ { 142, 4, 239 },{ 44, 142, 784 },{ 99, 21, 33 },{ 100, 64, 8 }, }, { { 150, 12, 239 },{ 52, 150, 784 },{ 107, 29, 33 },{ 108, 72, 8 }, },
		{ { 159, 21, 239 },{ 61, 159, 784 },{ 116, 38, 33 },{ 117, 81, 8 }, }, { { 167, 29, 239 },{ 69, 167, 784 },{ 124, 46, 33 },{ 125, 89, 8 }, },
		{ { 175, 37, 239 },{ 77, 175, 784 },{ 132, 54, 33 },{ 133, 97, 8 }, }, { { 183, 45, 239 },{ 85, 183, 784 },{ 140, 62, 33 },{ 141, 105, 8 }, },
		{ { 192, 54, 239 },{ 94, 192, 784 },{ 149, 71, 33 },{ 150, 114, 8 }, }, { { 200, 62, 239 },{ 102, 200, 784 },{ 157, 79, 33 },{ 158, 122, 8 }, },
		{ { 208, 70, 239 },{ 110, 208, 784 },{ 165, 87, 33 },{ 166, 130, 8 }, }, { { 216, 78, 239 },{ 118, 216, 784 },{ 173, 95, 33 },{ 174, 138, 8 }, },
		{ { 225, 87, 239 },{ 127, 225, 784 },{ 182, 104, 33 },{ 183, 147, 8 }, }, { { 233, 95, 239 },{ 135, 233, 784 },{ 190, 112, 33 },{ 191, 155, 8 }, },
		{ { 241, 103, 239 },{ 143, 241, 784 },{ 198, 120, 33 },{ 199, 163, 8 }, }, { { 111, 208, 3682 },{ 151, 249, 784 },{ 206, 128, 33 },{ 207, 171, 8 }, },
		{ { 120, 217, 3682 },{ 180, 216, 3648 },{ 215, 137, 33 },{ 216, 180, 8 }, }, { { 128, 225, 3682 },{ 188, 224, 3648 },{ 223, 145, 33 },{ 224, 188, 8 }, },
		{ { 155, 253, 3928 },{ 196, 232, 3648 },{ 231, 153, 33 },{ 232, 196, 8 }, }, { { 144, 241, 3682 },{ 204, 240, 3648 },{ 239, 161, 33 },{ 240, 204, 8 }, },
		{ { 153, 250, 3682 },{ 213, 249, 3648 },{ 248, 170, 33 },{ 249, 213, 8 }, }, { { 179, 221, 4040 },{ 255, 221, 8 },{ 179, 221, 456 },{ 255, 221, 8 }, },
		{ { 187, 229, 4040 },{ 255, 229, 8 },{ 187, 229, 456 },{ 255, 229, 8 }, }, { { 195, 237, 4040 },{ 255, 237, 8 },{ 195, 237, 456 },{ 255, 237, 8 }, },
		{ { 24, 80, 566 },{ 24, 80, 560 },{ 24, 0, 9 },{ 24, 0, 8 }, }, { { 32, 88, 566 },{ 32, 88, 560 },{ 32, 0, 9 },{ 32, 0, 8 }, },
		{ { 40, 96, 566 },{ 40, 96, 560 },{ 40, 0, 9 },{ 40, 0, 8 }, }, { { 48, 104, 566 },{ 48, 104, 560 },{ 48, 0, 9 },{ 48, 0, 8 }, },
		{ { 9, 138, 2758 },{ 130, 7, 1352 },{ 9, 57, 70 },{ 57, 9, 8 }, }, { { 119, 0, 313 },{ 120, 0, 312 },{ 17, 65, 70 },{ 65, 17, 8 }, },
		{ { 0, 128, 784 },{ 128, 6, 312 },{ 25, 73, 70 },{ 73, 25, 8 }, }, { { 6, 137, 790 },{ 5, 136, 784 },{ 33, 81, 70 },{ 81, 33, 8 }, },
		{ { 42, 171, 2758 },{ 14, 145, 784 },{ 42, 90, 70 },{ 90, 42, 8 }, }, { { 50, 179, 2758 },{ 22, 153, 784 },{ 50, 98, 70 },{ 98, 50, 8 }, },
		{ { 58, 187, 2758 },{ 30, 161, 784 },{ 58, 106, 70 },{ 106, 58, 8 }, }, { { 191, 18, 1329 },{ 38, 169, 784 },{ 112, 9, 33 },{ 114, 66, 8 }, },
		{ { 176, 0, 239 },{ 47, 178, 784 },{ 121, 18, 33 },{ 123, 75, 8 }, }, { { 187, 1, 239 },{ 55, 186, 784 },{ 129, 26, 33 },{ 131, 83, 8 }, },
		{ { 195, 10, 239 },{ 63, 194, 784 },{ 137, 34, 33 },{ 139, 91, 8 }, }, { { 203, 18, 239 },{ 71, 202, 784 },{ 145, 42, 33 },{ 147, 99, 8 }, },
		{ { 212, 27, 239 },{ 80, 211, 784 },{ 154, 51, 33 },{ 156, 108, 8 }, }, { { 220, 35, 239 },{ 88, 219, 784 },{ 162, 59, 33 },{ 164, 116, 8 }, },
		{ { 228, 43, 239 },{ 96, 227, 784 },{ 170, 67, 33 },{ 172, 124, 8 }, }, { { 236, 51, 239 },{ 104, 235, 784 },{ 178, 75, 33 },{ 180, 132, 8 }, },
		{ { 245, 60, 239 },{ 113, 244, 784 },{ 187, 84, 33 },{ 189, 141, 8 }, }, { { 91, 194, 3680 },{ 149, 197, 3648 },{ 195, 92, 33 },{ 197, 149, 8 }, },
		{ { 99, 202, 3680 },{ 157, 205, 3648 },{ 203, 100, 33 },{ 205, 157, 8 }, }, { { 107, 210, 3680 },{ 165, 213, 3648 },{ 211, 108, 33 },{ 213, 165, 8 }, },
		{ { 119, 249, 3928 },{ 174, 222, 3648 },{ 220, 117, 33 },{ 222, 174, 8 }, }, { { 127, 255, 856 },{ 182, 230, 3648 },{ 228, 125, 33 },{ 230, 182, 8 }, },
		{ { 255, 135, 169 },{ 190, 238, 3648 },{ 236, 133, 33 },{ 238, 190, 8 }, }, { { 140, 243, 3680 },{ 198, 246, 3648 },{ 244, 141, 33 },{ 246, 198, 8 }, },
		{ { 151, 207, 4040 },{ 255, 207, 8 },{ 151, 207, 456 },{ 255, 207, 8 }, }, { { 159, 215, 4040 },{ 255, 215, 8 },{ 159, 215, 456 },{ 255, 215, 8 }, },
		{ { 167, 223, 4040 },{ 255, 223, 8 },{ 167, 223, 456 },{ 255, 223, 8 }, }, { { 175, 231, 4040 },{ 255, 231, 8 },{ 175, 231, 456 },{ 255, 231, 8 }, },
		{ { 33, 106, 566 },{ 33, 106, 560 },{ 33, 0, 9 },{ 33, 0, 8 }, }, { { 41, 114, 566 },{ 41, 114, 560 },{ 41, 0, 9 },{ 41, 0, 8 }, },
		{ { 49, 122, 566 },{ 49, 122, 560 },{ 49, 0, 9 },{ 49, 0, 8 }, }, { { 57, 130, 566 },{ 57, 130, 560 },{ 57, 0, 9 },{ 57, 0, 8 }, },
		{ { 66, 139, 566 },{ 66, 139, 560 },{ 66, 0, 9 },{ 66, 0, 8 }, }, { { 74, 147, 566 },{ 170, 7, 1352 },{ 8, 74, 70 },{ 74, 8, 8 }, },
		{ { 152, 0, 313 },{ 178, 15, 1352 },{ 0, 82, 80 },{ 82, 16, 8 }, }, { { 162, 0, 313 },{ 186, 23, 1352 },{ 24, 90, 70 },{ 90, 24, 8 }, },
		{ { 0, 171, 784 },{ 195, 32, 1352 },{ 33, 99, 70 },{ 99, 33, 8 }, }, { { 6, 179, 790 },{ 203, 40, 1352 },{ 41, 107, 70 },{ 107, 41, 8 }, },
		{ { 15, 187, 790 },{ 211, 48, 1352 },{ 115, 0, 41 },{ 115, 49, 8 }, }, { { 61, 199, 710 },{ 219, 56, 1352 },{ 57, 123, 70 },{ 123, 57, 8 }, },
		{ { 70, 208, 710 },{ 228, 65, 1352 },{ 66, 132, 70 },{ 132, 66, 8 }, }, { { 78, 216, 710 },{ 236, 73, 1352 },{ 74, 140, 70 },{ 140, 74, 8 }, },
		{ { 86, 224, 710 },{ 244, 81, 1352 },{ 145, 7, 33 },{ 148, 82, 8 }, }, { { 222, 8, 233 },{ 252, 89, 1352 },{ 153, 15, 33 },{ 156, 90, 8 }, },
		{ { 235, 0, 239 },{ 241, 101, 328 },{ 166, 6, 39 },{ 165, 99, 8 }, }, { { 32, 170, 3680 },{ 249, 109, 328 },{ 0, 175, 98 },{ 173, 107, 8 }, },
		{ { 40, 178, 3680 },{ 115, 181, 3648 },{ 8, 183, 98 },{ 181, 115, 8 }, }, { { 48, 186, 3680 },{ 123, 189, 3648 },{ 16, 191, 98 },{ 189, 123, 8 }, },
		{ { 57, 195, 3680 },{ 132, 198, 3648 },{ 25, 200, 98 },{ 198, 132, 8 }, }, { { 67, 243, 3928 },{ 140, 206, 3648 },{ 33, 208, 98 },{ 206, 140, 8 }, },
		{ { 76, 251, 3928 },{ 148, 214, 3648 },{ 41, 216, 98 },{ 214, 148, 8 }, }, { { 86, 255, 856 },{ 156, 222, 3648 },{ 49, 224, 98 },{ 222, 156, 8 }, },
		{ { 255, 93, 169 },{ 165, 231, 3648 },{ 58, 233, 98 },{ 231, 165, 8 }, }, { { 98, 236, 3680 },{ 173, 239, 3648 },{ 66, 241, 98 },{ 239, 173, 8 }, },
		{ { 108, 181, 4040 },{ 181, 247, 3648 },{ 74, 249, 98 },{ 247, 181, 8 }, }, { { 116, 189, 4040 },{ 255, 189, 8 },{ 116, 189, 456 },{ 255, 189, 8 }, },
		{ { 125, 198, 4040 },{ 255, 198, 8 },{ 125, 198, 456 },{ 255, 198, 8 }, }, { { 133, 206, 4040 },{ 255, 206, 8 },{ 133, 206, 456 },{ 255, 206, 8 }, },
		{ { 141, 214, 4040 },{ 255, 214, 8 },{ 141, 214, 456 },{ 255, 214, 8 }, }, { { 149, 222, 4040 },{ 255, 222, 8 },{ 149, 222, 456 },{ 255, 222, 8 }, },
		{ { 47, 183, 566 },{ 47, 183, 560 },{ 47, 0, 9 },{ 47, 0, 8 }, }, { { 55, 191, 566 },{ 55, 191, 560 },{ 55, 0, 9 },{ 55, 0, 8 }, },
		{ { 63, 199, 566 },{ 63, 199, 560 },{ 63, 0, 9 },{ 63, 0, 8 }, }, { { 71, 207, 566 },{ 71, 207, 560 },{ 71, 0, 9 },{ 71, 0, 8 }, },
		{ { 80, 216, 566 },{ 80, 216, 560 },{ 80, 0, 9 },{ 80, 0, 8 }, }, { { 88, 224, 566 },{ 88, 224, 560 },{ 88, 0, 9 },{ 88, 0, 8 }, },
		{ { 3, 233, 710 },{ 3, 233, 704 },{ 2, 96, 70 },{ 96, 2, 8 }, }, { { 11, 241, 710 },{ 11, 241, 704 },{ 10, 104, 70 },{ 104, 10, 8 }, },
		{ { 20, 250, 710 },{ 20, 250, 704 },{ 19, 113, 70 },{ 113, 19, 8 }, }, { { 27, 121, 3654 },{ 27, 121, 3648 },{ 27, 121, 70 },{ 121, 27, 8 }, },
		{ { 35, 129, 3654 },{ 35, 129, 3648 },{ 35, 129, 70 },{ 129, 35, 8 }, }, { { 43, 137, 3654 },{ 43, 137, 3648 },{ 43, 137, 70 },{ 137, 43, 8 }, },
		{ { 52, 146, 3654 },{ 52, 146, 3648 },{ 52, 146, 70 },{ 146, 52, 8 }, }, { { 60, 154, 3654 },{ 60, 154, 3648 },{ 60, 154, 70 },{ 154, 60, 8 }, },
		{ { 68, 162, 3654 },{ 68, 162, 3648 },{ 68, 162, 70 },{ 162, 68, 8 }, }, { { 76, 170, 3654 },{ 76, 170, 3648 },{ 76, 170, 70 },{ 170, 76, 8 }, },
		{ { 85, 179, 3654 },{ 85, 179, 3648 },{ 85, 179, 70 },{ 179, 85, 8 }, }, { { 93, 187, 3654 },{ 93, 187, 3648 },{ 93, 187, 70 },{ 187, 93, 8 }, },
		{ { 101, 195, 3654 },{ 101, 195, 3648 },{ 101, 195, 70 },{ 195, 101, 8 }, }, { { 109, 203, 3654 },{ 109, 203, 3648 },{ 109, 203, 70 },{ 203, 109, 8 }, },
		{ { 118, 212, 3654 },{ 118, 212, 3648 },{ 118, 212, 70 },{ 212, 118, 8 }, }, { { 126, 220, 3654 },{ 126, 220, 3648 },{ 126, 220, 70 },{ 220, 126, 8 }, },
		{ { 134, 228, 3654 },{ 134, 228, 3648 },{ 134, 228, 70 },{ 228, 134, 8 }, }, { { 5, 236, 3680 },{ 142, 236, 3648 },{ 5, 236, 96 },{ 236, 142, 8 }, },
		{ { 14, 245, 3680 },{ 151, 245, 3648 },{ 14, 245, 96 },{ 245, 151, 8 }, }, { { 23, 159, 4040 },{ 159, 253, 3648 },{ 23, 159, 456 },{ 253, 159, 8 }, },
		{ { 31, 167, 4040 },{ 255, 167, 8 },{ 31, 167, 456 },{ 255, 167, 8 }, }, { { 39, 175, 4040 },{ 255, 175, 8 },{ 39, 175, 456 },{ 255, 175, 8 }, },
		{ { 48, 184, 4040 },{ 255, 184, 8 },{ 48, 184, 456 },{ 255, 184, 8 }, }, { { 56, 192, 4040 },{ 255, 192, 8 },{ 56, 192, 456 },{ 255, 192, 8 }, },
		{ { 64, 200, 4040 },{ 255, 200, 8 },{ 64, 200, 456 },{ 255, 200, 8 }, },{ { 72, 208, 4040 },{ 255, 208, 8 },{ 72, 208, 456 },{ 255, 208, 8 }, },

	};

	struct dxt5a_block
	{
		uint8_t m_endpoints[2];

		enum { cTotalSelectorBytes = 6 };
		uint8_t m_selectors[cTotalSelectorBytes];

		inline void clear()
		{
			basisu::clear_obj(*this);
		}

		inline uint32_t get_low_alpha() const
		{
			return m_endpoints[0];
		}

		inline uint32_t get_high_alpha() const
		{
			return m_endpoints[1];
		}

		inline void set_low_alpha(uint32_t i)
		{
			assert(i <= UINT8_MAX);
			m_endpoints[0] = static_cast<uint8_t>(i);
		}

		inline void set_high_alpha(uint32_t i)
		{
			assert(i <= UINT8_MAX);
			m_endpoints[1] = static_cast<uint8_t>(i);
		}

		inline bool is_alpha6_block() const { return get_low_alpha() <= get_high_alpha(); }

		uint32_t get_endpoints_as_word() const { return m_endpoints[0] | (m_endpoints[1] << 8); }
		uint32_t get_selectors_as_word(uint32_t index) { assert(index < 3); return m_selectors[index * 2] | (m_selectors[index * 2 + 1] << 8); }

		inline uint32_t get_selector(uint32_t x, uint32_t y) const
		{
			assert((x < 4U) && (y < 4U));

			uint32_t selector_index = (y * 4) + x;
			uint32_t bit_index = selector_index * cDXT5SelectorBits;

			uint32_t byte_index = bit_index >> 3;
			uint32_t bit_ofs = bit_index & 7;

			uint32_t v = m_selectors[byte_index];
			if (byte_index < (cTotalSelectorBytes - 1))
				v |= (m_selectors[byte_index + 1] << 8);

			return (v >> bit_ofs) & 7;
		}

		inline void set_selector(uint32_t x, uint32_t y, uint32_t val)
		{
			assert((x < 4U) && (y < 4U) && (val < 8U));

			uint32_t selector_index = (y * 4) + x;
			uint32_t bit_index = selector_index * cDXT5SelectorBits;

			uint32_t byte_index = bit_index >> 3;
			uint32_t bit_ofs = bit_index & 7;

			uint32_t v = m_selectors[byte_index];
			if (byte_index < (cTotalSelectorBytes - 1))
				v |= (m_selectors[byte_index + 1] << 8);

			v &= (~(7 << bit_ofs));
			v |= (val << bit_ofs);

			m_selectors[byte_index] = static_cast<uint8_t>(v);
			if (byte_index < (cTotalSelectorBytes - 1))
				m_selectors[byte_index + 1] = static_cast<uint8_t>(v >> 8);
		}

		enum { cMaxSelectorValues = 8 };

		static uint32_t get_block_values6(color32* pDst, uint32_t l, uint32_t h)
		{
			pDst[0].a = static_cast<uint8_t>(l);
			pDst[1].a = static_cast<uint8_t>(h);
			pDst[2].a = static_cast<uint8_t>((l * 4 + h) / 5);
			pDst[3].a = static_cast<uint8_t>((l * 3 + h * 2) / 5);
			pDst[4].a = static_cast<uint8_t>((l * 2 + h * 3) / 5);
			pDst[5].a = static_cast<uint8_t>((l + h * 4) / 5);
			pDst[6].a = 0;
			pDst[7].a = 255;
			return 6;
		}

		static uint32_t get_block_values8(color32* pDst, uint32_t l, uint32_t h)
		{
			pDst[0].a = static_cast<uint8_t>(l);
			pDst[1].a = static_cast<uint8_t>(h);
			pDst[2].a = static_cast<uint8_t>((l * 6 + h) / 7);
			pDst[3].a = static_cast<uint8_t>((l * 5 + h * 2) / 7);
			pDst[4].a = static_cast<uint8_t>((l * 4 + h * 3) / 7);
			pDst[5].a = static_cast<uint8_t>((l * 3 + h * 4) / 7);
			pDst[6].a = static_cast<uint8_t>((l * 2 + h * 5) / 7);
			pDst[7].a = static_cast<uint8_t>((l + h * 6) / 7);
			return 8;
		}

		static uint32_t get_block_values(color32* pDst, uint32_t l, uint32_t h)
		{
			if (l > h)
				return get_block_values8(pDst, l, h);
			else
				return get_block_values6(pDst, l, h);
		}
	};

	static void convert_etc1s_to_dxt5a(dxt5a_block* pDst_block, const endpoint* pEndpoints, const selector* pSelector)
	{
		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const color32& base_color = pEndpoints->m_color5;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (low_selector == high_selector)
		{
			uint32_t r;
			decoder_etc_block::get_block_color5_r(base_color, inten_table, low_selector, r);
						
			pDst_block->set_low_alpha(r);
			pDst_block->set_high_alpha(r);
			pDst_block->m_selectors[0] = 0;
			pDst_block->m_selectors[1] = 0;
			pDst_block->m_selectors[2] = 0;
			pDst_block->m_selectors[3] = 0;
			pDst_block->m_selectors[4] = 0;
			pDst_block->m_selectors[5] = 0;
			return;
		}
		else if (pSelector->m_num_unique_selectors == 2)
		{
			color32 block_colors[4];

			decoder_etc_block::get_block_colors5(block_colors, base_color, inten_table);

			const uint32_t r0 = block_colors[low_selector].r;
			const uint32_t r1 = block_colors[high_selector].r;

			pDst_block->set_low_alpha(r0);
			pDst_block->set_high_alpha(r1);

			// TODO: Optimize this
			for (uint32_t y = 0; y < 4; y++)
			{
				for (uint32_t x = 0; x < 4; x++)
				{
					uint32_t s = pSelector->get_selector(x, y);
					pDst_block->set_selector(x, y, (s == high_selector) ? 1 : 0);
				}
			}

			return;
		}

		uint32_t selector_range_table = 0;
		for (selector_range_table = 0; selector_range_table < NUM_DXT5A_SELECTOR_RANGES; selector_range_table++)
			if ((low_selector == s_dxt5a_selector_ranges[selector_range_table].m_low) && (high_selector == s_dxt5a_selector_ranges[selector_range_table].m_high))
				break;
		if (selector_range_table >= NUM_DXT5A_SELECTOR_RANGES)
			selector_range_table = 0;

		const etc1_g_to_dxt5a_conversion* pTable_entry = &g_etc1_g_to_dxt5a[base_color.r + inten_table * 32][selector_range_table];

		pDst_block->set_low_alpha(pTable_entry->m_lo);
		pDst_block->set_high_alpha(pTable_entry->m_hi);

		// TODO: Optimize this (like ETC1->BC1)
		for (uint32_t y = 0; y < 4; y++)
		{
			for (uint32_t x = 0; x < 4; x++)
			{
				uint32_t s = pSelector->get_selector(x, y);

				uint32_t ds = (pTable_entry->m_trans >> (s * 3)) & 7;

				pDst_block->set_selector(x, y, ds);
			}
		}
	}
#endif //BASISD_SUPPORT_DXT5A

	// PVRTC

#if BASISD_SUPPORT_PVRTC1 || BASISD_SUPPORT_UASTC
	static const  uint16_t g_pvrtc_swizzle_table[256] =
	{
		0x0000, 0x0001, 0x0004, 0x0005, 0x0010, 0x0011, 0x0014, 0x0015, 0x0040, 0x0041, 0x0044, 0x0045, 0x0050, 0x0051, 0x0054, 0x0055, 0x0100, 0x0101, 0x0104, 0x0105, 0x0110, 0x0111, 0x0114, 0x0115, 0x0140, 0x0141, 0x0144, 0x0145, 0x0150, 0x0151, 0x0154, 0x0155,
		0x0400, 0x0401, 0x0404, 0x0405, 0x0410, 0x0411, 0x0414, 0x0415, 0x0440, 0x0441, 0x0444, 0x0445, 0x0450, 0x0451, 0x0454, 0x0455, 0x0500, 0x0501, 0x0504, 0x0505, 0x0510, 0x0511, 0x0514, 0x0515, 0x0540, 0x0541, 0x0544, 0x0545, 0x0550, 0x0551, 0x0554, 0x0555,
		0x1000, 0x1001, 0x1004, 0x1005, 0x1010, 0x1011, 0x1014, 0x1015, 0x1040, 0x1041, 0x1044, 0x1045, 0x1050, 0x1051, 0x1054, 0x1055, 0x1100, 0x1101, 0x1104, 0x1105, 0x1110, 0x1111, 0x1114, 0x1115, 0x1140, 0x1141, 0x1144, 0x1145, 0x1150, 0x1151, 0x1154, 0x1155,
		0x1400, 0x1401, 0x1404, 0x1405, 0x1410, 0x1411, 0x1414, 0x1415, 0x1440, 0x1441, 0x1444, 0x1445, 0x1450, 0x1451, 0x1454, 0x1455, 0x1500, 0x1501, 0x1504, 0x1505, 0x1510, 0x1511, 0x1514, 0x1515, 0x1540, 0x1541, 0x1544, 0x1545, 0x1550, 0x1551, 0x1554, 0x1555,
		0x4000, 0x4001, 0x4004, 0x4005, 0x4010, 0x4011, 0x4014, 0x4015, 0x4040, 0x4041, 0x4044, 0x4045, 0x4050, 0x4051, 0x4054, 0x4055, 0x4100, 0x4101, 0x4104, 0x4105, 0x4110, 0x4111, 0x4114, 0x4115, 0x4140, 0x4141, 0x4144, 0x4145, 0x4150, 0x4151, 0x4154, 0x4155,
		0x4400, 0x4401, 0x4404, 0x4405, 0x4410, 0x4411, 0x4414, 0x4415, 0x4440, 0x4441, 0x4444, 0x4445, 0x4450, 0x4451, 0x4454, 0x4455, 0x4500, 0x4501, 0x4504, 0x4505, 0x4510, 0x4511, 0x4514, 0x4515, 0x4540, 0x4541, 0x4544, 0x4545, 0x4550, 0x4551, 0x4554, 0x4555,
		0x5000, 0x5001, 0x5004, 0x5005, 0x5010, 0x5011, 0x5014, 0x5015, 0x5040, 0x5041, 0x5044, 0x5045, 0x5050, 0x5051, 0x5054, 0x5055, 0x5100, 0x5101, 0x5104, 0x5105, 0x5110, 0x5111, 0x5114, 0x5115, 0x5140, 0x5141, 0x5144, 0x5145, 0x5150, 0x5151, 0x5154, 0x5155,
		0x5400, 0x5401, 0x5404, 0x5405, 0x5410, 0x5411, 0x5414, 0x5415, 0x5440, 0x5441, 0x5444, 0x5445, 0x5450, 0x5451, 0x5454, 0x5455, 0x5500, 0x5501, 0x5504, 0x5505, 0x5510, 0x5511, 0x5514, 0x5515, 0x5540, 0x5541, 0x5544, 0x5545, 0x5550, 0x5551, 0x5554, 0x5555
	};

	// Note we can't use simple calculations to convert PVRTC1 encoded endpoint components to/from 8-bits, due to hardware approximations.
	static const uint8_t g_pvrtc_5[32] = { 0,8,16,24,33,41,49,57,66,74,82,90,99,107,115,123,132,140,148,156,165,173,181,189,198,206,214,222,231,239,247,255 };
	static const uint8_t g_pvrtc_4[16] = { 0,16,33,49,66,82,99,115,140,156,173,189,206,222,239,255 };
	static const uint8_t g_pvrtc_3[8] = { 0,33,74,107,148,181,222,255 };
	static const uint8_t g_pvrtc_alpha[9] = { 0,34,68,102,136,170,204,238,255 };
		
	static const uint8_t g_pvrtc_5_floor[256] =
	{
		0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,
		3,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,
		7,7,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,11,11,11,11,11,11,
		11,11,11,12,12,12,12,12,12,12,12,13,13,13,13,13,13,13,13,14,14,14,14,14,14,14,14,15,15,15,15,15,
		15,15,15,15,16,16,16,16,16,16,16,16,17,17,17,17,17,17,17,17,18,18,18,18,18,18,18,18,19,19,19,19,
		19,19,19,19,19,20,20,20,20,20,20,20,20,21,21,21,21,21,21,21,21,22,22,22,22,22,22,22,22,23,23,23,
		23,23,23,23,23,23,24,24,24,24,24,24,24,24,25,25,25,25,25,25,25,25,26,26,26,26,26,26,26,26,27,27,
		27,27,27,27,27,27,27,28,28,28,28,28,28,28,28,29,29,29,29,29,29,29,29,30,30,30,30,30,30,30,30,31
	};

	static const uint8_t g_pvrtc_5_ceil[256] =
	{
		0,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,
		4,4,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,8,8,8,8,8,8,
		8,8,8,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,12,12,12,12,12,
		12,12,12,12,13,13,13,13,13,13,13,13,14,14,14,14,14,14,14,14,15,15,15,15,15,15,15,15,16,16,16,16,
		16,16,16,16,16,17,17,17,17,17,17,17,17,18,18,18,18,18,18,18,18,19,19,19,19,19,19,19,19,20,20,20,
		20,20,20,20,20,20,21,21,21,21,21,21,21,21,22,22,22,22,22,22,22,22,23,23,23,23,23,23,23,23,24,24,
		24,24,24,24,24,24,24,25,25,25,25,25,25,25,25,26,26,26,26,26,26,26,26,27,27,27,27,27,27,27,27,28,
		28,28,28,28,28,28,28,28,29,29,29,29,29,29,29,29,30,30,30,30,30,30,30,30,31,31,31,31,31,31,31,31
	};
		
	static const uint8_t g_pvrtc_4_floor[256] =
	{
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
		3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
		5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,
		7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,
		9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,
		11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,
		13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,15
	};

	static const uint8_t g_pvrtc_4_ceil[256] =
	{
		0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
		4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,
		6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,
		8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,
		10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,12,
		12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,14,
		14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15
	};
		
	static const uint8_t g_pvrtc_3_floor[256] =
	{
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
		3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,
		4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,
		5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,
		6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7
	};

	static const uint8_t g_pvrtc_3_ceil[256] =
	{
		0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
		3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
		4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,
		5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,
		6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
	};
		
	static const uint8_t g_pvrtc_alpha_floor[256] =
	{
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
		3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
		4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
		5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
		6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8
	};

	static const uint8_t g_pvrtc_alpha_ceil[256] =
	{
		0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
		1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
		2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
		3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
		4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
		5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
		6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
		7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8
	};

	struct pvrtc4_block
	{
		uint32_t m_modulation;
		uint32_t m_endpoints;

		pvrtc4_block() : m_modulation(0), m_endpoints(0) { }

		inline bool operator== (const pvrtc4_block& rhs) const
		{
			return (m_modulation == rhs.m_modulation) && (m_endpoints == rhs.m_endpoints);
		}

		inline void clear()
		{
			m_modulation = 0;
			m_endpoints = 0;
		}

		inline bool get_block_uses_transparent_modulation() const
		{
			return (m_endpoints & 1) != 0;
		}

		inline void set_block_uses_transparent_modulation(bool m)
		{
			m_endpoints = (m_endpoints & ~1U) | static_cast<uint32_t>(m);
		}

		inline bool is_endpoint_opaque(uint32_t endpoint_index) const
		{
			static const uint32_t s_bitmasks[2] = { 0x8000U, 0x80000000U };
			return (m_endpoints & s_bitmasks[basisu::open_range_check(endpoint_index, 2U)]) != 0;
		}

		inline void set_endpoint_opaque(uint32_t endpoint_index, bool opaque)
		{
			assert(endpoint_index < 2);
			static const uint32_t s_bitmasks[2] = { 0x8000U, 0x80000000U };
			if (opaque)
				m_endpoints |= s_bitmasks[endpoint_index];
			else
				m_endpoints &= ~s_bitmasks[endpoint_index];
		}

		inline color32 get_endpoint_5554(uint32_t endpoint_index) const
		{
			assert(endpoint_index < 2);
			static const uint32_t s_endpoint_mask[2] = { 0xFFFE, 0xFFFF };
			uint32_t packed = (m_endpoints >> (basisu::open_range_check(endpoint_index, 2U) ? 16 : 0)) & s_endpoint_mask[endpoint_index];

			uint32_t r, g, b, a;
			if (packed & 0x8000)
			{
				// opaque 554 or 555
				r = (packed >> 10) & 31;
				g = (packed >> 5) & 31;
				b = packed & 31;

				if (!endpoint_index)
					b |= (b >> 4);

				a = 0xF;
			}
			else
			{
				// translucent 4433 or 4443
				r = (packed >> 7) & 0x1E;
				g = (packed >> 3) & 0x1E;
				b = (packed & 0xF) << 1;

				r |= (r >> 4);
				g |= (g >> 4);

				if (!endpoint_index)
					b |= (b >> 3);
				else
					b |= (b >> 4);

				a = (packed >> 11) & 0xE;
			}

			assert((r < 32) && (g < 32) && (b < 32) && (a < 16));
					
			return color32(r, g, b, a);
		}
				
		inline color32 get_endpoint_8888(uint32_t endpoint_index) const
		{
			assert(endpoint_index < 2);
			static const uint32_t s_endpoint_mask[2] = { 0xFFFE, 0xFFFF };
			uint32_t packed = (m_endpoints >> (basisu::open_range_check(endpoint_index, 2U) ? 16 : 0)) & s_endpoint_mask[endpoint_index];

			uint32_t r, g, b, a;
			if (packed & 0x8000)
			{
				// opaque 554 or 555
				// 1RRRRRGGGGGBBBBM
				// 1RRRRRGGGGGBBBBB
				r = (packed >> 10) & 31;
				g = (packed >> 5) & 31;
				b = packed & 31;

				r = g_pvrtc_5[r];
				g = g_pvrtc_5[g];

				if (!endpoint_index)
					b = g_pvrtc_4[b >> 1];
				else
					b = g_pvrtc_5[b];

				a = 255;
			}
			else
			{
				// translucent 4433 or 4443
				// 0AAA RRRR GGGG BBBM
				// 0AAA RRRR GGGG BBBB
				r = (packed >> 8) & 0xF;
				g = (packed >> 4) & 0xF;
				b = packed & 0xF;
				a = (packed >> 12) & 7;

				r = g_pvrtc_4[r];
				g = g_pvrtc_4[g];

				if (!endpoint_index)
					b = g_pvrtc_3[b >> 1];
				else
					b = g_pvrtc_4[b];

				a = g_pvrtc_alpha[a];
			}
			
			return color32(r, g, b, a);
		}

		inline uint32_t get_endpoint_l8(uint32_t endpoint_index) const
		{
			color32 c(get_endpoint_8888(endpoint_index));
			return c.r + c.g + c.b + c.a;
		}
				
		inline uint32_t get_opaque_endpoint_l0() const
		{
			uint32_t packed = m_endpoints & 0xFFFE;

			uint32_t r, g, b;
			assert(packed & 0x8000);

			// opaque 554 or 555
			r = (packed >> 10) & 31;
			g = (packed >> 5) & 31;
			b = packed & 31;
			b |= (b >> 4);

			return r + g + b;
		}

		inline uint32_t get_opaque_endpoint_l1() const
		{
			uint32_t packed = m_endpoints >> 16;

			uint32_t r, g, b;
			assert(packed & 0x8000);

			// opaque 554 or 555
			r = (packed >> 10) & 31;
			g = (packed >> 5) & 31;
			b = packed & 31;

			return r + g + b;
		}

		static uint32_t get_component_precision_in_bits(uint32_t c, uint32_t endpoint_index, bool opaque_endpoint)
		{
			static const uint32_t s_comp_prec[4][4] =
			{
				// R0 G0 B0 A0      R1 G1 B1 A1
				{ 4, 4, 3, 3 },{ 4, 4, 4, 3 }, // transparent endpoint

				{ 5, 5, 4, 0 },{ 5, 5, 5, 0 }  // opaque endpoint
			};
			return s_comp_prec[basisu::open_range_check(endpoint_index, 2U) + (opaque_endpoint * 2)][basisu::open_range_check(c, 4U)];
		}

		static color32 get_color_precision_in_bits(uint32_t endpoint_index, bool opaque_endpoint)
		{
			static const color32 s_color_prec[4] =
			{
				color32(4, 4, 3, 3), color32(4, 4, 4, 3), // transparent endpoint
				color32(5, 5, 4, 0), color32(5, 5, 5, 0)  // opaque endpoint
			};
			return s_color_prec[basisu::open_range_check(endpoint_index, 2U) + (opaque_endpoint * 2)];
		}

		inline void set_opaque_endpoint_floor(uint32_t endpoint_index, const color32& c)
		{
			assert(endpoint_index < 2);
			const uint32_t m = m_endpoints & 1;

			uint32_t r = g_pvrtc_5_floor[c[0]], g = g_pvrtc_5_floor[c[1]], b = c[2];

			if (!endpoint_index)
				b = g_pvrtc_4_floor[b] << 1;
			else
				b = g_pvrtc_5_floor[b];

			// rgba=555 here
			assert((r < 32) && (g < 32) && (b < 32));

			// 1RRRRRGGGGGBBBBM
			// 1RRRRRGGGGGBBBBB

			// opaque 554 or 555
			uint32_t packed = 0x8000 | (r << 10) | (g << 5) | b;
			if (!endpoint_index)
				packed = (packed & ~1) | m;

			assert(packed <= 0xFFFF);

			if (endpoint_index)
				m_endpoints = (m_endpoints & 0xFFFFU) | (packed << 16);
			else
				m_endpoints = (m_endpoints & 0xFFFF0000U) | packed;
		}

		inline void set_opaque_endpoint_ceil(uint32_t endpoint_index, const color32& c)
		{
			assert(endpoint_index < 2);
			const uint32_t m = m_endpoints & 1;

			uint32_t r = g_pvrtc_5_ceil[c[0]], g = g_pvrtc_5_ceil[c[1]], b = c[2];

			if (!endpoint_index)
				b = g_pvrtc_4_ceil[b] << 1;
			else
				b = g_pvrtc_5_ceil[b];

			// rgba=555 here
			assert((r < 32) && (g < 32) && (b < 32));

			// 1RRRRRGGGGGBBBBM
			// 1RRRRRGGGGGBBBBB

			// opaque 554 or 555
			uint32_t packed = 0x8000 | (r << 10) | (g << 5) | b;
			if (!endpoint_index)
				packed |= m;

			assert(packed <= 0xFFFF);

			if (endpoint_index)
				m_endpoints = (m_endpoints & 0xFFFFU) | (packed << 16);
			else
				m_endpoints = (m_endpoints & 0xFFFF0000U) | packed;
		}
				
		// opaque endpoints:	554 or 555
		// transparent endpoints: 3443 or 3444
		inline void set_endpoint_raw(uint32_t endpoint_index, const color32& c, bool opaque_endpoint)
		{
			assert(endpoint_index < 2);
			const uint32_t m = m_endpoints & 1;
			uint32_t r = c[0], g = c[1], b = c[2], a = c[3];

			uint32_t packed;

			if (opaque_endpoint)
			{
				if (!endpoint_index)
				{
					// 554
					// 1RRRRRGGGGGBBBBM
					assert((r < 32) && (g < 32) && (b < 16));
					packed = 0x8000 | (r << 10) | (g << 5) | (b << 1) | m;
				}
				else
				{
					// 555
					// 1RRRRRGGGGGBBBBB
					assert((r < 32) && (g < 32) && (b < 32));
					packed = 0x8000 | (r << 10) | (g << 5) | b;
				}
			}
			else
			{
				if (!endpoint_index)
				{
					// 3443
					// 0AAA RRRR GGGG BBBM
					assert((r < 16) && (g < 16) && (b < 8) && (a < 8));
					packed = (a << 12) | (r << 8) | (g << 4) | (b << 1) | m;
				}
				else
				{
					// 3444
					// 0AAA RRRR GGGG BBBB
					assert((r < 16) && (g < 16) && (b < 16) && (a < 8));
					packed = (a << 12) | (r << 8) | (g << 4) | b;
				}
			}

			assert(packed <= 0xFFFF);

			if (endpoint_index)
				m_endpoints = (m_endpoints & 0xFFFFU) | (packed << 16);
			else
				m_endpoints = (m_endpoints & 0xFFFF0000U) | packed;
		}
				
		inline void set_endpoint_floor(uint32_t endpoint_index, const color32& c)
		{
			assert(endpoint_index < 2);

			int a = g_pvrtc_alpha_floor[c.a];
			if (a == 8)
			{
				// 554 or 555
				uint32_t r = g_pvrtc_5_floor[c[0]], g = g_pvrtc_5_floor[c[1]], b = c[2];

				if (!endpoint_index)
					b = g_pvrtc_4_floor[b];
				else
					b = g_pvrtc_5_floor[b];

				set_endpoint_raw(endpoint_index, color32(r, g, b, a), true);
			}
			else
			{
				// 4433 or 4443
				uint32_t r = g_pvrtc_4_floor[c[0]], g = g_pvrtc_4_floor[c[1]], b = c[2];

				if (!endpoint_index)
					b = g_pvrtc_3_floor[b];
				else
					b = g_pvrtc_4_floor[b];

				set_endpoint_raw(endpoint_index, color32(r, g, b, a), false);
			}
		}

		inline void set_endpoint_ceil(uint32_t endpoint_index, const color32& c)
		{
			assert(endpoint_index < 2);

			int a = g_pvrtc_alpha_ceil[c.a];
			if (a == 8)
			{
				// 554 or 555
				uint32_t r = g_pvrtc_5_ceil[c[0]], g = g_pvrtc_5_ceil[c[1]], b = c[2];

				if (!endpoint_index)
					b = g_pvrtc_4_ceil[b];
				else
					b = g_pvrtc_5_ceil[b];

				set_endpoint_raw(endpoint_index, color32(r, g, b, a), true);
			}
			else
			{
				// 4433 or 4443
				uint32_t r = g_pvrtc_4_ceil[c[0]], g = g_pvrtc_4_ceil[c[1]], b = c[2];

				if (!endpoint_index)
					b = g_pvrtc_3_ceil[b];
				else
					b = g_pvrtc_4_ceil[b];

				set_endpoint_raw(endpoint_index, color32(r, g, b, a), false);
			}
		}

		inline uint32_t get_modulation(uint32_t x, uint32_t y) const
		{
			assert((x < 4) && (y < 4));
			return (m_modulation >> ((y * 4 + x) * 2)) & 3;
		}

		// Scaled by 8
		inline const uint32_t* get_scaled_modulation_values(bool block_uses_transparent_modulation) const
		{
			static const uint32_t s_block_scales[2][4] = { { 0, 3, 5, 8 },{ 0, 4, 4, 8 } };
			return s_block_scales[block_uses_transparent_modulation];
		}

		// Scaled by 8
		inline uint32_t get_scaled_modulation(uint32_t x, uint32_t y) const
		{
			return get_scaled_modulation_values(get_block_uses_transparent_modulation())[get_modulation(x, y)];
		}

		inline void set_modulation(uint32_t x, uint32_t y, uint32_t s)
		{
			assert((x < 4) && (y < 4) && (s < 4));
			uint32_t n = (y * 4 + x) * 2;
			m_modulation = (m_modulation & (~(3 << n))) | (s << n);
			assert(get_modulation(x, y) == s);
		}

		// Assumes modulation was initialized to 0
		inline void set_modulation_fast(uint32_t x, uint32_t y, uint32_t s)
		{
			assert((x < 4) && (y < 4) && (s < 4));
			uint32_t n = (y * 4 + x) * 2;
			m_modulation |= (s << n);
			assert(get_modulation(x, y) == s);
		}
	};

#if 0
	static const uint8_t g_pvrtc_bilinear_weights[16][4] =
	{
		{ 4, 4, 4, 4 }, { 2, 6, 2, 6 }, { 8, 0, 8, 0 }, { 6, 2, 6, 2 },
		{ 2, 2, 6, 6 }, { 1, 3, 3, 9 }, { 4, 0, 12, 0 }, { 3, 1, 9, 3 },
		{ 8, 8, 0, 0 }, { 4, 12, 0, 0 }, { 16, 0, 0, 0 }, { 12, 4, 0, 0 },
		{ 6, 6, 2, 2 }, { 3, 9, 1, 3 }, { 12, 0, 4, 0 }, { 9, 3, 3, 1 },
	};
#endif

	struct pvrtc1_temp_block
	{
		decoder_etc_block m_etc1_block;
		uint32_t m_pvrtc_endpoints;
	};

	static inline uint32_t get_opaque_endpoint_l0(uint32_t endpoints)
	{
		uint32_t packed = endpoints;

		uint32_t r, g, b;
		assert(packed & 0x8000);

		r = (packed >> 10) & 31;
		g = (packed >> 5) & 31;
		b = packed & 30;
		b |= (b >> 4);

		return r + g + b;
	}

	static inline uint32_t get_opaque_endpoint_l1(uint32_t endpoints)
	{
		uint32_t packed = endpoints >> 16;

		uint32_t r, g, b;
		assert(packed & 0x8000);

		r = (packed >> 10) & 31;
		g = (packed >> 5) & 31;
		b = packed & 31;

		return r + g + b;
	}

	static color32 get_endpoint_8888(uint32_t endpoints, uint32_t endpoint_index)
	{
		assert(endpoint_index < 2);
		static const uint32_t s_endpoint_mask[2] = { 0xFFFE, 0xFFFF };
		uint32_t packed = (endpoints >> (basisu::open_range_check(endpoint_index, 2U) ? 16 : 0)) & s_endpoint_mask[endpoint_index];

		uint32_t r, g, b, a;
		if (packed & 0x8000)
		{
			// opaque 554 or 555
			// 1RRRRRGGGGGBBBBM
			// 1RRRRRGGGGGBBBBB
			r = (packed >> 10) & 31;
			g = (packed >> 5) & 31;
			b = packed & 31;

			r = g_pvrtc_5[r];
			g = g_pvrtc_5[g];

			if (!endpoint_index)
				b = g_pvrtc_4[b >> 1];
			else
				b = g_pvrtc_5[b];

			a = 255;
		}
		else
		{
			// translucent 4433 or 4443
			// 0AAA RRRR GGGG BBBM
			// 0AAA RRRR GGGG BBBB
			r = (packed >> 8) & 0xF;
			g = (packed >> 4) & 0xF;
			b = packed & 0xF;
			a = (packed >> 12) & 7;

			r = g_pvrtc_4[r];
			g = g_pvrtc_4[g];

			if (!endpoint_index)
				b = g_pvrtc_3[b >> 1];
			else
				b = g_pvrtc_4[b];

			a = g_pvrtc_alpha[a];
		}

		return color32(r, g, b, a);
	}

	static uint32_t get_endpoint_l8(uint32_t endpoints, uint32_t endpoint_index)
	{
		color32 c(get_endpoint_8888(endpoints, endpoint_index));
		return c.r + c.g + c.b + c.a;
	}
#endif

#if BASISD_SUPPORT_PVRTC1
	// TODO: Support decoding a non-pow2 ETC1S texture into the next larger pow2 PVRTC texture.
	static void fixup_pvrtc1_4_modulation_rgb(const decoder_etc_block* pETC_Blocks, const uint32_t* pPVRTC_endpoints, void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y)
	{
		const uint32_t x_mask = num_blocks_x - 1;
		const uint32_t y_mask = num_blocks_y - 1;
		const uint32_t x_bits = basisu::total_bits(x_mask);
		const uint32_t y_bits = basisu::total_bits(y_mask);
		const uint32_t min_bits = basisu::minimum(x_bits, y_bits);
		//const uint32_t max_bits = basisu::maximum(x_bits, y_bits);
		const uint32_t swizzle_mask = (1 << (min_bits * 2)) - 1;

		uint32_t block_index = 0;

		// really 3x3
		int e0[4][4], e1[4][4];

		for (int y = 0; y < static_cast<int>(num_blocks_y); y++)
		{
			const uint32_t* pE_rows[3];

			for (int ey = 0; ey < 3; ey++)
			{
				int by = y + ey - 1; 

				const uint32_t* pE = &pPVRTC_endpoints[(by & y_mask) * num_blocks_x];

				pE_rows[ey] = pE;

				for (int ex = 0; ex < 3; ex++)
				{
					int bx = 0 + ex - 1; 

					const uint32_t e = pE[bx & x_mask];

					e0[ex][ey] = (get_opaque_endpoint_l0(e) * 255) / 31;
					e1[ex][ey] = (get_opaque_endpoint_l1(e) * 255) / 31;
				}
			}

			const uint32_t y_swizzle = (g_pvrtc_swizzle_table[y >> 8] << 16) | g_pvrtc_swizzle_table[y & 0xFF];

			for (int x = 0; x < static_cast<int>(num_blocks_x); x++, block_index++)
			{
				const decoder_etc_block& src_block = pETC_Blocks[block_index];

				const uint32_t x_swizzle = (g_pvrtc_swizzle_table[x >> 8] << 17) | (g_pvrtc_swizzle_table[x & 0xFF] << 1);

				uint32_t swizzled = x_swizzle | y_swizzle;
				if (num_blocks_x != num_blocks_y)
				{
					swizzled &= swizzle_mask;

					if (num_blocks_x > num_blocks_y)
						swizzled |= ((x >> min_bits) << (min_bits * 2));
					else
						swizzled |= ((y >> min_bits) << (min_bits * 2));
				}

				pvrtc4_block* pDst_block = static_cast<pvrtc4_block*>(pDst_blocks) + swizzled;
				pDst_block->m_endpoints = pPVRTC_endpoints[block_index];

				uint32_t base_r = g_etc_5_to_8[src_block.m_differential.m_red1];
				uint32_t base_g = g_etc_5_to_8[src_block.m_differential.m_green1];
				uint32_t base_b = g_etc_5_to_8[src_block.m_differential.m_blue1];

				const int* pInten_table48 = g_etc1_inten_tables48[src_block.m_differential.m_cw1];
				int by = (base_r + base_g + base_b) * 16;
				int block_colors_y_x16[4];
				block_colors_y_x16[0] = by + pInten_table48[2];
				block_colors_y_x16[1] = by + pInten_table48[3];
				block_colors_y_x16[2] = by + pInten_table48[1];
				block_colors_y_x16[3] = by + pInten_table48[0];

				{
					const uint32_t ex = 2;
					int bx = x + ex - 1;
					bx &= x_mask;

#define DO_ROW(ey) \
					{ \
						const uint32_t e = pE_rows[ey][bx]; \
						e0[ex][ey] = (get_opaque_endpoint_l0(e) * 255) / 31; \
						e1[ex][ey] = (get_opaque_endpoint_l1(e) * 255) / 31; \
					}

					DO_ROW(0);
					DO_ROW(1);
					DO_ROW(2);
#undef DO_ROW
				}

				uint32_t mod = 0;

				uint32_t lookup_x[4];

#define DO_LOOKUP(lx) { \
					const uint32_t byte_ofs = 7 - (((lx) * 4) >> 3); \
					const uint32_t lsb_bits = src_block.m_bytes[byte_ofs] >> (((lx) & 1) * 4); \
					const uint32_t msb_bits = src_block.m_bytes[byte_ofs - 2] >> (((lx) & 1) * 4); \
					lookup_x[lx] = (lsb_bits & 0xF) | ((msb_bits & 0xF) << 4); }

				DO_LOOKUP(0);
				DO_LOOKUP(1);
				DO_LOOKUP(2);
				DO_LOOKUP(3);
#undef DO_LOOKUP

#define DO_PIX(lx, ly, w0, w1, w2, w3) \
				{ \
					int ca_l = a0 * w0 + a1 * w1 + a2 * w2 + a3 * w3; \
					int cb_l = b0 * w0 + b1 * w1 + b2 * w2 + b3 * w3; \
					int cl = block_colors_y_x16[g_etc1_x_selector_unpack[ly][lookup_x[lx]]]; \
					int dl = cb_l - ca_l; \
					int vl = cl - ca_l; \
					int p = vl * 16; \
					if (ca_l > cb_l) { p = -p; dl = -dl; } \
					uint32_t m = 0; \
					if (p > 3 * dl) m = (uint32_t)(1 << ((ly) * 8 + (lx) * 2)); \
					if (p > 8 * dl) m = (uint32_t)(2 << ((ly) * 8 + (lx) * 2)); \
					if (p > 13 * dl) m = (uint32_t)(3 << ((ly) * 8 + (lx) * 2)); \
					mod |= m; \
				}

				{
					const uint32_t ex = 0, ey = 0;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(0, 0, 4, 4, 4, 4);
					DO_PIX(1, 0, 2, 6, 2, 6);
					DO_PIX(0, 1, 2, 2, 6, 6);
					DO_PIX(1, 1, 1, 3, 3, 9);
				}

				{
					const uint32_t ex = 1, ey = 0;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(2, 0, 8, 0, 8, 0);
					DO_PIX(3, 0, 6, 2, 6, 2);
					DO_PIX(2, 1, 4, 0, 12, 0);
					DO_PIX(3, 1, 3, 1, 9, 3);
				}

				{
					const uint32_t ex = 0, ey = 1;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(0, 2, 8, 8, 0, 0);
					DO_PIX(1, 2, 4, 12, 0, 0);
					DO_PIX(0, 3, 6, 6, 2, 2);
					DO_PIX(1, 3, 3, 9, 1, 3);
				}

				{
					const uint32_t ex = 1, ey = 1;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(2, 2, 16, 0, 0, 0);
					DO_PIX(3, 2, 12, 4, 0, 0);
					DO_PIX(2, 3, 12, 0, 4, 0);
					DO_PIX(3, 3, 9, 3, 3, 1);
				}
#undef DO_PIX

				pDst_block->m_modulation = mod;

				e0[0][0] = e0[1][0]; e0[1][0] = e0[2][0];
				e0[0][1] = e0[1][1]; e0[1][1] = e0[2][1];
				e0[0][2] = e0[1][2]; e0[1][2] = e0[2][2];

				e1[0][0] = e1[1][0]; e1[1][0] = e1[2][0];
				e1[0][1] = e1[1][1]; e1[1][1] = e1[2][1];
				e1[0][2] = e1[1][2]; e1[1][2] = e1[2][2];

			} // x
		} // y
	}

	static void fixup_pvrtc1_4_modulation_rgba(
		const decoder_etc_block* pETC_Blocks, 
		const uint32_t* pPVRTC_endpoints, 
		void* pDst_blocks, uint32_t num_blocks_x, uint32_t num_blocks_y, void *pAlpha_blocks,
		const endpoint* pEndpoints, const selector* pSelectors)
	{
		const uint32_t x_mask = num_blocks_x - 1;
		const uint32_t y_mask = num_blocks_y - 1;
		const uint32_t x_bits = basisu::total_bits(x_mask);
		const uint32_t y_bits = basisu::total_bits(y_mask);
		const uint32_t min_bits = basisu::minimum(x_bits, y_bits);
		//const uint32_t max_bits = basisu::maximum(x_bits, y_bits);
		const uint32_t swizzle_mask = (1 << (min_bits * 2)) - 1;

		uint32_t block_index = 0;

		// really 3x3
		int e0[4][4], e1[4][4];

		for (int y = 0; y < static_cast<int>(num_blocks_y); y++)
		{
			const uint32_t* pE_rows[3];

			for (int ey = 0; ey < 3; ey++)
			{
				int by = y + ey - 1; 

				const uint32_t* pE = &pPVRTC_endpoints[(by & y_mask) * num_blocks_x];

				pE_rows[ey] = pE;

				for (int ex = 0; ex < 3; ex++)
				{
					int bx = 0 + ex - 1; 

					const uint32_t e = pE[bx & x_mask];

					e0[ex][ey] = get_endpoint_l8(e, 0);
					e1[ex][ey] = get_endpoint_l8(e, 1);
				}
			}

			const uint32_t y_swizzle = (g_pvrtc_swizzle_table[y >> 8] << 16) | g_pvrtc_swizzle_table[y & 0xFF];

			for (int x = 0; x < static_cast<int>(num_blocks_x); x++, block_index++)
			{
				const decoder_etc_block& src_block = pETC_Blocks[block_index];
				
				const uint16_t* pSrc_alpha_block = reinterpret_cast<const uint16_t*>(static_cast<const uint32_t*>(pAlpha_blocks) + x + (y * num_blocks_x));
				const endpoint* pAlpha_endpoints = &pEndpoints[pSrc_alpha_block[0]];
				const selector* pAlpha_selectors = &pSelectors[pSrc_alpha_block[1]];
				
				const uint32_t x_swizzle = (g_pvrtc_swizzle_table[x >> 8] << 17) | (g_pvrtc_swizzle_table[x & 0xFF] << 1);
				
				uint32_t swizzled = x_swizzle | y_swizzle;
				if (num_blocks_x != num_blocks_y)
				{
					swizzled &= swizzle_mask;

					if (num_blocks_x > num_blocks_y)
						swizzled |= ((x >> min_bits) << (min_bits * 2));
					else
						swizzled |= ((y >> min_bits) << (min_bits * 2));
				}

				pvrtc4_block* pDst_block = static_cast<pvrtc4_block*>(pDst_blocks) + swizzled;
				pDst_block->m_endpoints = pPVRTC_endpoints[block_index];

				uint32_t base_r = g_etc_5_to_8[src_block.m_differential.m_red1];
				uint32_t base_g = g_etc_5_to_8[src_block.m_differential.m_green1];
				uint32_t base_b = g_etc_5_to_8[src_block.m_differential.m_blue1];

				const int* pInten_table48 = g_etc1_inten_tables48[src_block.m_differential.m_cw1];
				int by = (base_r + base_g + base_b) * 16;
				int block_colors_y_x16[4];
				block_colors_y_x16[0] = basisu::clamp<int>(by + pInten_table48[0], 0, 48 * 255);
				block_colors_y_x16[1] = basisu::clamp<int>(by + pInten_table48[1], 0, 48 * 255);
				block_colors_y_x16[2] = basisu::clamp<int>(by + pInten_table48[2], 0, 48 * 255);
				block_colors_y_x16[3] = basisu::clamp<int>(by + pInten_table48[3], 0, 48 * 255);

				uint32_t alpha_base_g = g_etc_5_to_8[pAlpha_endpoints->m_color5.g] * 16;
				const int* pInten_table16 = g_etc1_inten_tables16[pAlpha_endpoints->m_inten5];
				int alpha_block_colors_x16[4];
				alpha_block_colors_x16[0] = basisu::clamp<int>(alpha_base_g + pInten_table16[0], 0, 16 * 255);
				alpha_block_colors_x16[1] = basisu::clamp<int>(alpha_base_g + pInten_table16[1], 0, 16 * 255);
				alpha_block_colors_x16[2] = basisu::clamp<int>(alpha_base_g + pInten_table16[2], 0, 16 * 255);
				alpha_block_colors_x16[3] = basisu::clamp<int>(alpha_base_g + pInten_table16[3], 0, 16 * 255);

				// clamp((base_r + base_g + base_b) * 16 + color_inten[s] * 48) + clamp(alpha_base_g * 16 + alpha_inten[as] * 16)

				{
					const uint32_t ex = 2;
					int bx = x + ex - 1;
					bx &= x_mask;

#define DO_ROW(ey) \
					{ \
						const uint32_t e = pE_rows[ey][bx]; \
						e0[ex][ey] = get_endpoint_l8(e, 0); \
						e1[ex][ey] = get_endpoint_l8(e, 1); \
					}

					DO_ROW(0);
					DO_ROW(1);
					DO_ROW(2);
#undef DO_ROW
				}

				uint32_t mod = 0;

#define DO_PIX(lx, ly, w0, w1, w2, w3) \
				{ \
					int ca_l = a0 * w0 + a1 * w1 + a2 * w2 + a3 * w3; \
					int cb_l = b0 * w0 + b1 * w1 + b2 * w2 + b3 * w3; \
					int cl = block_colors_y_x16[(src_block.m_bytes[4 + ly] >> (lx * 2)) & 3] + alpha_block_colors_x16[(pAlpha_selectors->m_selectors[ly] >> (lx * 2)) & 3]; \
					int dl = cb_l - ca_l; \
					int vl = cl - ca_l; \
					int p = vl * 16; \
					if (ca_l > cb_l) { p = -p; dl = -dl; } \
					uint32_t m = 0; \
					if (p > 3 * dl) m = (uint32_t)(1 << ((ly) * 8 + (lx) * 2)); \
					if (p > 8 * dl) m = (uint32_t)(2 << ((ly) * 8 + (lx) * 2)); \
					if (p > 13 * dl) m = (uint32_t)(3 << ((ly) * 8 + (lx) * 2)); \
					mod |= m; \
				}

				{
					const uint32_t ex = 0, ey = 0;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(0, 0, 4, 4, 4, 4);
					DO_PIX(1, 0, 2, 6, 2, 6);
					DO_PIX(0, 1, 2, 2, 6, 6);
					DO_PIX(1, 1, 1, 3, 3, 9);
				}

				{
					const uint32_t ex = 1, ey = 0;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(2, 0, 8, 0, 8, 0);
					DO_PIX(3, 0, 6, 2, 6, 2);
					DO_PIX(2, 1, 4, 0, 12, 0);
					DO_PIX(3, 1, 3, 1, 9, 3);
				}

				{
					const uint32_t ex = 0, ey = 1;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(0, 2, 8, 8, 0, 0);
					DO_PIX(1, 2, 4, 12, 0, 0);
					DO_PIX(0, 3, 6, 6, 2, 2);
					DO_PIX(1, 3, 3, 9, 1, 3);
				}

				{
					const uint32_t ex = 1, ey = 1;
					const int a0 = e0[ex][ey], a1 = e0[ex + 1][ey], a2 = e0[ex][ey + 1], a3 = e0[ex + 1][ey + 1];
					const int b0 = e1[ex][ey], b1 = e1[ex + 1][ey], b2 = e1[ex][ey + 1], b3 = e1[ex + 1][ey + 1];
					DO_PIX(2, 2, 16, 0, 0, 0);
					DO_PIX(3, 2, 12, 4, 0, 0);
					DO_PIX(2, 3, 12, 0, 4, 0);
					DO_PIX(3, 3, 9, 3, 3, 1);
				}
#undef DO_PIX

				pDst_block->m_modulation = mod;

				e0[0][0] = e0[1][0]; e0[1][0] = e0[2][0];
				e0[0][1] = e0[1][1]; e0[1][1] = e0[2][1];
				e0[0][2] = e0[1][2]; e0[1][2] = e0[2][2];

				e1[0][0] = e1[1][0]; e1[1][0] = e1[2][0];
				e1[0][1] = e1[1][1]; e1[1][1] = e1[2][1];
				e1[0][2] = e1[1][2]; e1[1][2] = e1[2][2];

			} // x
		} // y
	}
#endif // BASISD_SUPPORT_PVRTC1

#if BASISD_SUPPORT_BC7_MODE5
	static dxt_selector_range g_etc1_to_bc7_m5_selector_ranges[] =
	{
		{ 0, 3 },
		{ 1, 3 },
		{ 0, 2 },
		{ 1, 2 },
		{ 2, 3 },
		{ 0, 1 },
	};

	const uint32_t NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES = sizeof(g_etc1_to_bc7_m5_selector_ranges) / sizeof(g_etc1_to_bc7_m5_selector_ranges[0]);

	static uint32_t g_etc1_to_bc7_m5_selector_range_index[4][4];
	
	const uint32_t NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS = 10;
	static const uint8_t g_etc1_to_bc7_m5_selector_mappings[NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS][4] =
	{
		{ 0, 0, 1, 1 },
		{ 0, 0, 1, 2 },
		{ 0, 0, 1, 3 },
		{ 0, 0, 2, 3 },
		{ 0, 1, 1, 1 },
		{ 0, 1, 2, 2 },
		{ 0, 1, 2, 3 },
		{ 0, 2, 3, 3 },
		{ 1, 2, 2, 2 },
		{ 1, 2, 3, 3 },
	};

	struct etc1_to_bc7_m5_solution
	{
		uint8_t m_lo;
		uint8_t m_hi;
		uint16_t m_err;
	};
		
	static const etc1_to_bc7_m5_solution g_etc1_to_bc7_m5_color[32 * 8 * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS * NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES] = {
/**** start inlining basisu_transcoder_tables_bc7_m5_color.inc ****/
{0,7,18},{0,5,2},{0,4,1},{0,3,8},{0,4,35},{0,3,24},{0,3,12},{0,2,29},{0,2,36},{0,2,30},{0,7,18},{0,5,2},{0,4,1},{0,3,8},{2,0,35},{0,3,24},{0,3,12},{0,2,29},{4,0,35},{0,2,29},{0,3,0},{0,3,0},{0,3,0},{0,1,1},{0,1,2},{0,1,2},{0,1,2},{0,1,1},{1,0,3},{0,1,2},{0,3,0},
{0,3,0},{0,3,0},{0,1,1},{1,0,2},{1,0,2},{1,0,2},{0,1,1},{1,0,2},{0,1,1},{4,0,18},{0,5,2},{0,4,1},{0,3,8},{4,0,18},{7,0,18},{0,3,8},{0,2,20},{7,0,18},{0,2,20},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,15,38},{2,11,20},{2,8,24},
{1,8,21},{0,16,51},{0,10,19},{0,8,2},{0,6,24},{0,8,76},{0,6,40},{3,13,18},{3,10,2},{3,8,1},{3,7,5},{8,0,51},{1,9,19},{0,8,2},{0,6,24},{16,0,51},{0,6,24},{2,11,20},{2,11,20},{2,11,20},{1,7,20},{0,12,8},{0,7,2},{0,7,2},{0,5,1},{0,5,19},{0,5,10},{3,9,0},{3,9,0},{3,9,0},{3,6,0},{6,0,8},
{1,7,0},{1,7,0},{0,5,1},{12,0,8},{0,5,1},{10,0,18},{2,11,0},{3,8,1},{0,8,1},{10,0,18},{20,0,18},{0,8,1},{0,6,20},{20,0,18},{0,6,20},{1,0,20},{1,0,20},{1,0,20},{1,0,20},{0,9,0},{0,9,0},{0,9,0},{0,4,1},{0,3,8},{0,3,8},{6,19,38},{6,15,20},{6,12,24},{5,12,21},{4,20,51},{4,14,19},{4,12,2},
{4,10,24},{0,14,52},{1,10,20},{7,17,18},{7,14,2},{7,12,1},{7,11,5},{14,0,51},{5,13,19},{4,12,2},{1,10,20},{29,0,51},{1,10,20},{6,15,20},{6,15,20},{6,15,20},{5,11,20},{4,16,8},{4,11,2},{4,11,2},{4,9,1},{0,12,8},{1,9,2},{7,13,0},{7,13,0},{7,13,0},{7,10,0},{12,0,8},{5,11,0},{5,11,0},{3,9,0},{24,0,8},
{3,9,0},{15,1,18},{6,15,0},{7,12,1},{3,12,0},{15,1,18},{32,0,18},{3,12,0},{0,10,20},{32,0,18},{0,10,20},{5,0,20},{5,0,20},{5,0,20},{5,0,20},{4,13,0},{4,13,0},{4,13,0},{4,8,1},{1,10,0},{1,10,0},{10,23,38},{10,19,20},{10,16,24},{9,16,21},{8,24,51},{8,18,19},{8,16,2},{8,14,24},{3,18,52},{5,14,20},{11,21,18},
{11,18,2},{11,16,1},{11,15,5},{20,0,51},{9,17,19},{8,16,2},{5,14,20},{41,0,51},{5,14,20},{10,19,20},{10,19,20},{10,19,20},{9,15,20},{8,20,8},{8,15,2},{8,15,2},{8,13,1},{4,16,8},{5,13,2},{11,17,0},{11,17,0},{11,17,0},{11,14,0},{18,0,8},{9,15,0},{9,15,0},{7,13,0},{36,0,8},{7,13,0},{21,1,18},{10,19,0},{11,16,1},
{7,16,0},{21,1,18},{44,0,18},{7,16,0},{0,14,20},{44,0,18},{0,14,20},{9,0,20},{9,0,20},{9,0,20},{9,0,20},{8,17,0},{8,17,0},{8,17,0},{8,12,1},{5,14,0},{5,14,0},{14,29,36},{14,24,18},{14,21,23},{14,20,20},{13,28,52},{13,22,20},{13,20,2},{12,19,23},{7,23,51},{9,19,18},{16,25,19},{15,23,1},{16,20,2},{15,20,6},{27,0,51},
{13,22,19},{13,20,1},{8,19,18},{55,0,51},{8,19,18},{14,24,18},{14,24,18},{14,24,18},{14,19,18},{13,23,9},{13,20,1},{13,20,1},{12,17,2},{9,20,8},{10,18,1},{16,20,1},{16,20,1},{16,20,1},{16,18,1},{24,1,8},{14,19,0},{14,19,0},{12,17,1},{50,0,8},{12,17,1},{28,0,18},{15,23,0},{16,20,1},{12,20,1},{28,0,18},{58,0,18},{12,20,1},
{0,19,18},{58,0,18},{0,19,18},{14,0,18},{14,0,18},{14,0,18},{14,0,18},{13,20,1},{13,20,1},{13,20,1},{12,17,1},{10,18,0},{10,18,0},{18,33,36},{18,28,18},{18,25,23},{18,24,20},{17,32,52},{17,26,20},{17,24,2},{16,23,23},{11,27,51},{13,23,18},{20,29,19},{19,27,1},{20,24,2},{19,24,6},{33,0,51},{17,26,19},{17,24,1},{12,23,18},{66,0,51},
{12,23,18},{18,28,18},{18,28,18},{18,28,18},{18,23,18},{17,27,9},{17,24,1},{17,24,1},{16,21,2},{13,24,8},{14,22,1},{20,24,1},{20,24,1},{20,24,1},{20,22,1},{30,1,8},{18,23,0},{18,23,0},{16,21,1},{62,0,8},{16,21,1},{34,0,18},{19,27,0},{20,24,1},{16,24,1},{34,0,18},{69,0,18},{16,24,1},{0,23,18},{69,0,18},{0,23,18},{18,0,18},
{18,0,18},{18,0,18},{18,0,18},{17,24,1},{17,24,1},{17,24,1},{16,21,1},{14,22,0},{14,22,0},{22,37,36},{22,32,18},{22,29,23},{22,28,20},{21,36,52},{21,30,20},{21,28,2},{20,27,23},{15,31,51},{17,27,18},{24,33,19},{23,31,1},{24,28,2},{23,28,6},{39,0,51},{21,30,19},{21,28,1},{16,27,18},{78,0,51},{16,27,18},{22,32,18},{22,32,18},{22,32,18},
{22,27,18},{21,31,9},{21,28,1},{21,28,1},{20,25,2},{17,28,8},{18,26,1},{24,28,1},{24,28,1},{24,28,1},{24,26,1},{36,1,8},{22,27,0},{22,27,0},{20,25,1},{74,0,8},{20,25,1},{40,0,18},{23,31,0},{24,28,1},{20,28,1},{40,0,18},{82,0,18},{20,28,1},{0,27,18},{82,0,18},{0,27,18},{22,0,18},{22,0,18},{22,0,18},{22,0,18},{21,28,1},
{21,28,1},{21,28,1},{20,25,1},{18,26,0},{18,26,0},{26,41,36},{26,36,18},{26,33,23},{26,32,20},{25,40,52},{25,34,20},{25,32,2},{24,31,23},{19,35,51},{21,31,18},{28,37,19},{27,35,1},{28,32,2},{27,32,6},{45,0,51},{25,34,19},{25,32,1},{20,31,18},{91,0,51},{20,31,18},{26,36,18},{26,36,18},{26,36,18},{26,31,18},{25,35,9},{25,32,1},{25,32,1},
{24,29,2},{21,32,8},{22,30,1},{28,32,1},{28,32,1},{28,32,1},{28,30,1},{42,1,8},{26,31,0},{26,31,0},{24,29,1},{86,0,8},{24,29,1},{46,0,18},{27,35,0},{28,32,1},{24,32,1},{46,0,18},{94,0,18},{24,32,1},{0,31,18},{94,0,18},{0,31,18},{26,0,18},{26,0,18},{26,0,18},{26,0,18},{25,32,1},{25,32,1},{25,32,1},{24,29,1},{22,30,0},
{22,30,0},{31,44,38},{31,40,20},{31,37,24},{30,37,21},{29,45,51},{29,39,19},{29,37,2},{29,35,24},{24,39,52},{26,35,20},{32,42,18},{32,39,2},{32,37,1},{32,36,5},{51,0,51},{30,38,19},{29,37,2},{26,35,20},{104,0,51},{26,35,20},{31,40,20},{31,40,20},{31,40,20},{30,36,20},{29,41,8},{29,36,2},{29,36,2},{29,34,1},{25,37,8},{26,34,2},{32,38,0},
{32,38,0},{32,38,0},{32,35,0},{49,0,8},{30,36,0},{30,36,0},{28,34,0},{100,0,8},{28,34,0},{53,0,18},{31,40,0},{32,37,1},{28,37,0},{53,0,18},{107,0,18},{28,37,0},{0,35,20},{107,0,18},{0,35,20},{30,0,20},{30,0,20},{30,0,20},{30,0,20},{29,38,0},{29,38,0},{29,38,0},{29,33,1},{26,35,0},{26,35,0},{35,48,38},{35,44,20},{35,41,24},
{34,41,21},{33,49,51},{33,43,19},{33,41,2},{33,39,24},{28,43,52},{30,39,20},{36,46,18},{36,43,2},{36,41,1},{36,40,5},{57,0,51},{34,42,19},{33,41,2},{30,39,20},{117,0,51},{30,39,20},{35,44,20},{35,44,20},{35,44,20},{34,40,20},{33,45,8},{33,40,2},{33,40,2},{33,38,1},{29,41,8},{30,38,2},{36,42,0},{36,42,0},{36,42,0},{36,39,0},{55,0,8},
{34,40,0},{34,40,0},{32,38,0},{112,0,8},{32,38,0},{59,0,18},{35,44,0},{36,41,1},{32,41,0},{59,0,18},{120,0,18},{32,41,0},{0,39,20},{120,0,18},{0,39,20},{34,0,20},{34,0,20},{34,0,20},{34,0,20},{33,42,0},{33,42,0},{33,42,0},{33,37,1},{30,39,0},{30,39,0},{39,52,38},{39,48,20},{39,45,24},{38,45,21},{37,53,51},{37,47,19},{37,45,2},
{37,43,24},{32,47,52},{34,43,20},{40,50,18},{40,47,2},{40,45,1},{40,44,5},{63,0,51},{38,46,19},{37,45,2},{34,43,20},{127,1,51},{34,43,20},{39,48,20},{39,48,20},{39,48,20},{38,44,20},{37,49,8},{37,44,2},{37,44,2},{37,42,1},{33,45,8},{34,42,2},{40,46,0},{40,46,0},{40,46,0},{40,43,0},{61,0,8},{38,44,0},{38,44,0},{36,42,0},{124,0,8},
{36,42,0},{64,0,18},{39,48,0},{40,45,1},{36,45,0},{64,0,18},{126,3,18},{36,45,0},{0,43,20},{126,3,18},{0,43,20},{38,0,20},{38,0,20},{38,0,20},{38,0,20},{37,46,0},{37,46,0},{37,46,0},{37,41,1},{34,43,0},{34,43,0},{43,56,38},{43,52,20},{43,49,24},{42,49,21},{41,57,51},{41,51,19},{41,49,2},{41,47,24},{36,51,52},{38,47,20},{44,54,18},
{44,51,2},{44,49,1},{44,48,5},{69,0,51},{42,50,19},{41,49,2},{38,47,20},{127,7,51},{38,47,20},{43,52,20},{43,52,20},{43,52,20},{42,48,20},{41,53,8},{41,48,2},{41,48,2},{41,46,1},{37,49,8},{38,46,2},{44,50,0},{44,50,0},{44,50,0},{44,47,0},{66,1,8},{42,48,0},{42,48,0},{40,46,0},{126,5,8},{40,46,0},{70,0,18},{43,52,0},{44,49,1},
{40,49,0},{70,0,18},{126,9,18},{40,49,0},{0,47,20},{126,9,18},{0,47,20},{42,0,20},{42,0,20},{42,0,20},{42,0,20},{41,50,0},{41,50,0},{41,50,0},{41,45,1},{38,47,0},{38,47,0},{47,62,36},{47,57,18},{47,54,23},{47,53,20},{46,61,52},{46,55,20},{46,53,2},{45,52,23},{40,56,51},{42,52,18},{49,58,19},{48,56,1},{49,53,2},{48,53,6},{75,1,51},
{46,55,19},{46,53,1},{41,52,18},{126,14,51},{41,52,18},{47,57,18},{47,57,18},{47,57,18},{47,52,18},{46,56,9},{46,53,1},{46,53,1},{45,50,2},{42,53,8},{43,51,1},{49,53,1},{49,53,1},{49,53,1},{49,51,1},{73,0,8},{47,52,0},{47,52,0},{45,50,1},{126,12,8},{45,50,1},{77,0,18},{48,56,0},{49,53,1},{45,53,1},{77,0,18},{127,15,18},{45,53,1},
{0,52,18},{127,15,18},{0,52,18},{47,0,18},{47,0,18},{47,0,18},{47,0,18},{46,53,1},{46,53,1},{46,53,1},{45,50,1},{43,51,0},{43,51,0},{51,65,36},{51,61,18},{51,58,23},{51,57,20},{50,64,52},{50,59,20},{50,57,2},{49,56,23},{44,60,51},{46,56,18},{53,62,19},{52,60,1},{53,57,2},{52,57,6},{81,0,51},{50,59,19},{50,57,1},{45,56,18},{126,20,51},
{45,56,18},{51,61,18},{51,61,18},{51,61,18},{51,56,18},{50,60,9},{50,57,1},{50,57,1},{49,54,2},{46,57,8},{47,55,1},{53,57,1},{53,57,1},{53,57,1},{53,55,1},{79,0,8},{51,56,0},{51,56,0},{49,54,1},{127,17,8},{49,54,1},{83,0,18},{52,60,0},{53,57,1},{49,57,1},{83,0,18},{127,21,18},{49,57,1},{0,56,18},{127,21,18},{0,56,18},{51,0,18},
{51,0,18},{51,0,18},{51,0,18},{50,57,1},{50,57,1},{50,57,1},{49,54,1},{47,55,0},{47,55,0},{55,69,36},{55,64,19},{55,62,23},{55,61,20},{54,68,52},{54,63,20},{54,61,2},{53,60,23},{48,64,51},{50,60,18},{57,65,19},{56,64,2},{57,61,2},{56,61,6},{87,0,51},{54,63,19},{54,61,1},{49,60,18},{126,26,51},{49,60,18},{55,65,18},{55,65,18},{55,65,18},
{55,60,18},{54,64,9},{54,61,1},{54,61,1},{53,58,2},{50,61,8},{51,59,1},{57,61,1},{57,61,1},{57,61,1},{57,59,1},{85,0,8},{55,60,0},{55,60,0},{53,58,1},{127,23,8},{53,58,1},{89,0,18},{55,64,1},{57,61,1},{53,61,1},{89,0,18},{127,27,18},{53,61,1},{0,60,18},{127,27,18},{0,60,18},{55,0,18},{55,0,18},{55,0,18},{55,0,18},{54,61,1},
{54,61,1},{54,61,1},{53,58,1},{51,59,0},{51,59,0},{59,73,36},{59,68,19},{59,66,26},{59,64,22},{58,72,52},{57,67,19},{57,65,2},{57,63,28},{52,68,51},{53,64,21},{61,69,19},{60,67,2},{61,65,1},{60,65,5},{93,0,51},{57,67,18},{57,65,1},{52,64,20},{126,32,51},{52,64,20},{59,69,18},{59,69,18},{59,69,18},{59,64,18},{58,68,9},{58,64,2},{58,64,2},
{57,62,2},{55,64,9},{55,63,1},{61,65,1},{61,65,1},{61,65,1},{61,63,1},{91,0,8},{58,64,1},{58,64,1},{57,62,1},{127,29,8},{57,62,1},{95,0,18},{60,67,1},{61,65,0},{56,65,0},{95,0,18},{127,33,18},{56,65,0},{0,64,20},{127,33,18},{0,64,20},{59,0,18},{59,0,18},{59,0,18},{59,0,18},{58,65,1},{58,65,1},{58,65,1},{57,62,1},{55,63,0},
{55,63,0},{63,79,38},{63,73,21},{64,70,28},{63,69,22},{62,78,51},{62,71,18},{62,69,2},{61,68,26},{57,72,51},{58,68,19},{65,74,19},{64,72,1},{65,69,2},{64,69,6},{100,0,51},{62,71,18},{62,69,2},{58,68,18},{127,38,51},{58,68,18},{63,74,20},{63,74,20},{63,74,20},{63,68,21},{62,73,8},{62,69,1},{62,69,1},{62,66,1},{59,69,9},{60,66,2},{65,69,1},
{65,69,1},{65,69,1},{65,67,1},{98,0,8},{63,68,1},{63,68,1},{62,66,1},{127,36,8},{62,66,1},{101,1,18},{64,72,0},{65,69,1},{62,69,1},{101,1,18},{126,40,18},{62,69,1},{0,68,18},{126,40,18},{0,68,18},{63,0,20},{63,0,20},{63,0,20},{63,0,20},{62,70,0},{62,70,0},{62,70,0},{62,66,0},{60,66,1},{60,66,1},{67,82,36},{67,77,18},{67,74,23},
{67,73,20},{66,81,52},{66,75,20},{66,73,2},{65,72,23},{61,76,51},{62,72,19},{69,78,19},{68,76,1},{69,73,2},{68,73,6},{106,0,51},{66,75,19},{66,73,1},{62,72,18},{127,44,51},{62,72,18},{67,77,18},{67,77,18},{67,77,18},{67,72,18},{66,76,9},{66,73,1},{66,73,1},{65,70,2},{63,73,9},{63,71,2},{69,73,1},{69,73,1},{69,73,1},{69,71,1},{104,0,8},
{67,72,0},{67,72,0},{65,70,1},{127,42,8},{65,70,1},{107,1,18},{68,76,0},{69,73,1},{65,73,1},{107,1,18},{126,46,18},{65,73,1},{0,72,18},{126,46,18},{0,72,18},{67,0,18},{67,0,18},{67,0,18},{67,0,18},{66,73,1},{66,73,1},{66,73,1},{65,70,1},{63,71,1},{63,71,1},{71,86,36},{71,81,18},{71,78,23},{71,77,20},{70,85,52},{70,79,20},{70,77,2},
{69,76,23},{64,80,51},{66,76,18},{73,82,19},{72,80,1},{73,77,2},{72,77,6},{112,0,51},{70,79,19},{70,77,1},{65,76,18},{127,50,51},{65,76,18},{71,81,18},{71,81,18},{71,81,18},{71,76,18},{70,80,9},{70,77,1},{70,77,1},{69,74,2},{66,77,8},{67,75,1},{73,77,1},{73,77,1},{73,77,1},{73,75,1},{110,0,8},{71,76,0},{71,76,0},{69,74,1},{126,48,8},
{69,74,1},{113,0,18},{72,80,0},{73,77,1},{69,77,1},{113,0,18},{126,52,18},{69,77,1},{0,76,18},{126,52,18},{0,76,18},{71,0,18},{71,0,18},{71,0,18},{71,0,18},{70,77,1},{70,77,1},{70,77,1},{69,74,1},{67,75,0},{67,75,0},{75,90,36},{75,85,18},{75,82,23},{75,81,20},{74,89,52},{74,83,20},{74,81,2},{73,80,23},{68,84,51},{70,80,18},{77,86,19},
{76,84,1},{77,81,2},{76,81,6},{118,0,51},{74,83,19},{74,81,1},{69,80,18},{127,56,51},{69,80,18},{75,85,18},{75,85,18},{75,85,18},{75,80,18},{74,84,9},{74,81,1},{74,81,1},{73,78,2},{70,81,8},{71,79,1},{77,81,1},{77,81,1},{77,81,1},{77,79,1},{115,1,8},{75,80,0},{75,80,0},{73,78,1},{126,54,8},{73,78,1},{119,0,18},{76,84,0},{77,81,1},
{73,81,1},{119,0,18},{126,58,18},{73,81,1},{0,80,18},{126,58,18},{0,80,18},{75,0,18},{75,0,18},{75,0,18},{75,0,18},{74,81,1},{74,81,1},{74,81,1},{73,78,1},{71,79,0},{71,79,0},{80,93,38},{80,89,20},{80,86,24},{79,86,21},{78,94,51},{78,88,19},{78,86,2},{78,84,24},{73,88,52},{75,84,20},{81,91,18},{81,88,2},{81,86,1},{81,85,5},{124,1,51},
{79,87,19},{78,86,2},{75,84,20},{126,63,51},{75,84,20},{80,89,20},{80,89,20},{80,89,20},{79,85,20},{78,90,8},{78,85,2},{78,85,2},{78,83,1},{74,86,8},{75,83,2},{81,87,0},{81,87,0},{81,87,0},{81,84,0},{122,0,8},{79,85,0},{79,85,0},{77,83,0},{126,61,8},{77,83,0},{126,0,18},{80,89,0},{81,86,1},{77,86,0},{126,0,18},{126,64,18},{77,86,0},
{0,84,20},{126,64,18},{0,84,20},{79,0,20},{79,0,20},{79,0,20},{79,0,20},{78,87,0},{78,87,0},{78,87,0},{78,82,1},{75,84,0},{75,84,0},{84,97,38},{84,93,20},{84,90,24},{83,90,21},{82,98,51},{82,92,19},{82,90,2},{82,88,24},{77,92,52},{79,88,20},{85,95,18},{85,92,2},{85,90,1},{85,89,5},{127,7,51},{83,91,19},{82,90,2},{79,88,20},{127,68,51},
{79,88,20},{84,93,20},{84,93,20},{84,93,20},{83,89,20},{82,94,8},{82,89,2},{82,89,2},{82,87,1},{78,90,8},{79,87,2},{85,91,0},{85,91,0},{85,91,0},{85,88,0},{127,2,8},{83,89,0},{83,89,0},{81,87,0},{127,66,8},{81,87,0},{127,10,18},{84,93,0},{85,90,1},{81,90,0},{127,10,18},{126,70,18},{81,90,0},{0,88,20},{126,70,18},{0,88,20},{83,0,20},
{83,0,20},{83,0,20},{83,0,20},{82,91,0},{82,91,0},{82,91,0},{82,86,1},{79,88,0},{79,88,0},{88,101,38},{88,97,20},{88,94,24},{87,94,21},{86,102,51},{86,96,19},{86,94,2},{86,92,24},{81,96,52},{83,92,20},{89,99,18},{89,96,2},{89,94,1},{89,93,5},{127,19,51},{87,95,19},{86,94,2},{83,92,20},{127,74,51},{83,92,20},{88,97,20},{88,97,20},{88,97,20},
{87,93,20},{86,98,8},{86,93,2},{86,93,2},{86,91,1},{82,94,8},{83,91,2},{89,95,0},{89,95,0},{89,95,0},{89,92,0},{127,14,8},{87,93,0},{87,93,0},{85,91,0},{127,72,8},{85,91,0},{127,22,18},{88,97,0},{89,94,1},{85,94,0},{127,22,18},{126,76,18},{85,94,0},{0,92,20},{126,76,18},{0,92,20},{87,0,20},{87,0,20},{87,0,20},{87,0,20},{86,95,0},
{86,95,0},{86,95,0},{86,90,1},{83,92,0},{83,92,0},{92,105,38},{92,101,20},{92,98,24},{91,98,21},{90,106,51},{90,100,19},{90,98,2},{90,96,24},{85,100,52},{87,96,20},{93,103,18},{93,100,2},{93,98,1},{93,97,5},{127,31,51},{91,99,19},{90,98,2},{87,96,20},{127,80,51},{87,96,20},{92,101,20},{92,101,20},{92,101,20},{91,97,20},{90,102,8},{90,97,2},{90,97,2},
{90,95,1},{86,98,8},{87,95,2},{93,99,0},{93,99,0},{93,99,0},{93,96,0},{127,27,8},{91,97,0},{91,97,0},{89,95,0},{126,78,8},{89,95,0},{127,34,18},{92,101,0},{93,98,1},{89,98,0},{127,34,18},{126,82,18},{89,98,0},{0,96,20},{126,82,18},{0,96,20},{91,0,20},{91,0,20},{91,0,20},{91,0,20},{90,99,0},{90,99,0},{90,99,0},{90,94,1},{87,96,0},
{87,96,0},{96,111,36},{96,106,18},{96,103,23},{96,102,20},{95,110,52},{95,104,20},{95,102,2},{94,101,23},{89,105,51},{91,101,18},{98,107,19},{97,105,1},{98,102,2},{97,102,6},{127,45,51},{95,104,19},{95,102,1},{90,101,18},{126,87,51},{90,101,18},{96,106,18},{96,106,18},{96,106,18},{96,101,18},{95,105,9},{95,102,1},{95,102,1},{94,99,2},{91,102,8},{92,100,1},{98,102,1},
{98,102,1},{98,102,1},{98,100,1},{127,40,8},{96,101,0},{96,101,0},{94,99,1},{126,85,8},{94,99,1},{127,48,18},{97,105,0},{98,102,1},{94,102,1},{127,48,18},{127,88,18},{94,102,1},{0,101,18},{127,88,18},{0,101,18},{96,0,18},{96,0,18},{96,0,18},{96,0,18},{95,102,1},{95,102,1},{95,102,1},{94,99,1},{92,100,0},{92,100,0},{100,115,36},{100,110,18},{100,107,23},
{100,106,20},{99,114,52},{99,108,20},{99,106,2},{98,105,23},{93,109,51},{95,105,18},{102,111,19},{101,109,1},{102,106,2},{101,106,6},{127,57,51},{99,108,19},{99,106,1},{94,105,18},{126,93,51},{94,105,18},{100,110,18},{100,110,18},{100,110,18},{100,105,18},{99,109,9},{99,106,1},{99,106,1},{98,103,2},{95,106,8},{96,104,1},{102,106,1},{102,106,1},{102,106,1},{102,104,1},{127,53,8},
{100,105,0},{100,105,0},{98,103,1},{126,91,8},{98,103,1},{127,60,18},{101,109,0},{102,106,1},{98,106,1},{127,60,18},{127,94,18},{98,106,1},{0,105,18},{127,94,18},{0,105,18},{100,0,18},{100,0,18},{100,0,18},{100,0,18},{99,106,1},{99,106,1},{99,106,1},{98,103,1},{96,104,0},{96,104,0},{104,119,36},{104,114,18},{104,111,23},{104,110,20},{103,118,52},{103,112,20},{103,110,2},
{102,109,23},{97,113,51},{99,109,18},{106,115,19},{105,113,1},{106,110,2},{105,110,6},{127,69,51},{103,112,19},{103,110,1},{98,109,18},{126,99,51},{98,109,18},{104,114,18},{104,114,18},{104,114,18},{104,109,18},{103,113,9},{103,110,1},{103,110,1},{102,107,2},{99,110,8},{100,108,1},{106,110,1},{106,110,1},{106,110,1},{106,108,1},{127,64,8},{104,109,0},{104,109,0},{102,107,1},{126,97,8},
{102,107,1},{127,72,18},{105,113,0},{106,110,1},{102,110,1},{127,72,18},{127,100,18},{102,110,1},{0,109,18},{127,100,18},{0,109,18},{104,0,18},{104,0,18},{104,0,18},{104,0,18},{103,110,1},{103,110,1},{103,110,1},{102,107,1},{100,108,0},{100,108,0},{108,123,36},{108,118,18},{108,115,23},{108,114,20},{107,122,52},{107,116,20},{107,114,2},{106,113,23},{101,117,51},{103,113,18},{110,119,19},
{109,117,1},{110,114,2},{109,114,6},{127,81,51},{107,116,19},{107,114,1},{102,113,18},{126,105,51},{102,113,18},{108,118,18},{108,118,18},{108,118,18},{108,113,18},{107,117,9},{107,114,1},{107,114,1},{106,111,2},{103,114,8},{104,112,1},{110,114,1},{110,114,1},{110,114,1},{110,112,1},{127,76,8},{108,113,0},{108,113,0},{106,111,1},{126,103,8},{106,111,1},{127,84,18},{109,117,0},{110,114,1},
{106,114,1},{127,84,18},{127,106,18},{106,114,1},{0,113,18},{127,106,18},{0,113,18},{108,0,18},{108,0,18},{108,0,18},{108,0,18},{107,114,1},{107,114,1},{107,114,1},{106,111,1},{104,112,0},{104,112,0},{113,126,38},{113,122,20},{113,119,24},{112,119,21},{111,127,51},{111,121,19},{111,119,2},{111,117,24},{106,121,52},{108,117,20},{114,124,18},{114,121,2},{114,119,1},{114,118,5},{127,95,51},
{112,120,19},{111,119,2},{108,117,20},{127,111,51},{108,117,20},{113,122,20},{113,122,20},{113,122,20},{112,118,20},{111,123,8},{111,118,2},{111,118,2},{111,116,1},{107,119,8},{108,116,2},{114,120,0},{114,120,0},{114,120,0},{114,117,0},{127,90,8},{112,118,0},{112,118,0},{110,116,0},{127,109,8},{110,116,0},{127,98,18},{113,122,0},{114,119,1},{110,119,0},{127,98,18},{126,113,18},{110,119,0},
{0,117,20},{126,113,18},{0,117,20},{112,0,20},{112,0,20},{112,0,20},{112,0,20},{111,120,0},{111,120,0},{111,120,0},{111,115,1},{108,117,0},{108,117,0},{117,127,46},{117,126,20},{117,123,24},{116,123,21},{116,126,63},{115,125,19},{115,123,2},{115,121,24},{110,125,52},{112,121,20},{118,127,20},{118,125,2},{118,123,1},{118,122,5},{127,107,51},{116,124,19},{115,123,2},{112,121,20},{127,117,51},
{112,121,20},{117,126,20},{117,126,20},{117,126,20},{116,122,20},{115,127,8},{115,122,2},{115,122,2},{115,120,1},{111,123,8},{112,120,2},{118,124,0},{118,124,0},{118,124,0},{118,121,0},{127,102,8},{116,122,0},{116,122,0},{114,120,0},{127,115,8},{114,120,0},{127,110,18},{117,126,0},{118,123,1},{114,123,0},{127,110,18},{126,119,18},{114,123,0},{0,121,20},{126,119,18},{0,121,20},{116,0,20},
{116,0,20},{116,0,20},{116,0,20},{115,124,0},{115,124,0},{115,124,0},{115,119,1},{112,121,0},{112,121,0},{122,126,86},{121,127,40},{121,127,24},{120,127,21},{121,127,88},{119,127,27},{119,127,2},{119,125,24},{116,127,60},{116,125,20},{123,127,30},{122,127,10},{122,127,1},{122,126,5},{127,119,51},{121,127,24},{119,127,2},{116,125,20},{127,123,51},{116,125,20},{121,127,24},{121,127,24},{121,127,24},
{120,126,20},{120,127,14},{119,126,2},{119,126,2},{119,124,1},{115,127,8},{116,124,2},{122,126,1},{122,126,1},{122,126,1},{122,125,0},{127,115,8},{120,126,0},{120,126,0},{118,124,0},{127,121,8},{118,124,0},{127,122,18},{124,127,8},{122,127,1},{118,127,0},{127,122,18},{126,125,18},{118,127,0},{0,125,20},{126,125,18},{0,125,20},{120,0,20},{120,0,20},{120,0,20},{120,0,20},{119,126,1},
{119,126,1},{119,126,1},{119,123,1},{116,125,0},{116,125,0},{125,126,38},{125,127,30},{125,127,29},{125,127,21},{125,126,35},{124,127,16},{124,127,12},{123,127,1},{122,127,20},{122,127,2},{126,127,2},{126,127,2},{126,127,1},{126,127,1},{127,125,3},{127,126,3},{125,127,2},{124,127,0},{127,126,3},{124,127,0},{125,126,29},{125,126,29},{125,126,29},{125,127,21},{124,127,24},{124,127,12},{124,127,12},
{123,127,1},{122,127,11},{122,127,2},{126,127,1},{126,127,1},{126,127,1},{126,127,1},{127,125,2},{127,126,2},{127,126,2},{124,127,0},{127,126,2},{124,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{124,0,20},{124,0,20},{124,0,20},{124,0,20},{124,127,8},{124,127,8},{124,127,8},{123,127,1},{122,127,2},
{122,127,2},{0,16,72},{0,11,8},{0,8,1},{0,7,25},{0,10,153},{0,8,97},{0,6,50},{0,4,115},{0,5,162},{0,4,124},{0,16,72},{0,11,8},{0,8,1},{0,7,25},{5,0,153},{0,8,97},{0,6,50},{0,4,115},{10,0,153},{0,4,115},{0,7,0},{0,7,0},{0,7,0},{0,4,0},{0,3,13},{0,2,5},{0,2,5},{0,1,10},{0,2,14},{0,1,11},{0,7,0},
{0,7,0},{0,7,0},{0,4,0},{2,0,13},{0,2,5},{0,2,5},{0,1,10},{3,0,13},{0,1,10},{8,0,72},{0,11,8},{0,8,1},{0,7,25},{8,0,72},{16,0,72},{0,7,25},{0,5,74},{16,0,72},{0,5,74},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,27,77},{1,18,5},{1,13,15},
{1,11,18},{0,21,243},{0,14,108},{0,11,29},{0,8,154},{0,9,287},{0,8,179},{2,25,73},{1,18,1},{2,12,6},{1,11,14},{10,1,243},{0,14,108},{0,11,29},{0,8,154},{21,0,243},{0,8,154},{1,17,5},{1,17,5},{1,17,5},{1,9,5},{0,12,50},{0,8,8},{0,8,8},{0,5,25},{0,5,61},{0,5,34},{2,15,1},{2,15,1},{2,15,1},{1,9,1},{6,0,50},
{0,8,8},{0,8,8},{0,5,25},{12,0,50},{0,5,25},{14,0,72},{1,18,0},{4,12,1},{0,12,10},{14,0,72},{29,0,72},{0,12,10},{0,9,74},{29,0,72},{0,9,74},{1,0,5},{1,0,5},{1,0,5},{1,0,5},{0,4,0},{0,4,0},{0,4,0},{0,2,0},{0,2,1},{0,2,1},{3,35,133},{3,24,63},{4,17,90},{3,16,66},{0,33,243},{0,19,75},{0,16,2},
{0,13,110},{0,14,345},{0,12,179},{6,29,73},{5,22,1},{6,16,6},{5,15,14},{16,0,243},{0,19,75},{0,16,2},{0,13,110},{33,0,243},{0,13,110},{3,26,61},{3,26,61},{3,26,61},{3,14,61},{0,24,50},{0,16,1},{0,16,1},{0,10,5},{0,11,101},{0,9,46},{6,19,1},{6,19,1},{6,19,1},{5,13,1},{12,0,50},{1,15,0},{1,15,0},{0,10,5},{24,0,50},
{0,10,5},{20,0,72},{5,22,0},{8,16,1},{0,16,2},{20,0,72},{41,0,72},{0,16,2},{0,13,74},{41,0,72},{0,13,74},{3,0,61},{3,0,61},{3,0,61},{3,0,61},{0,16,0},{0,16,0},{0,16,0},{0,8,0},{0,5,25},{0,5,25},{7,39,146},{7,28,76},{8,21,107},{6,20,79},{4,37,244},{4,23,76},{4,20,3},{3,16,107},{0,22,292},{0,17,106},{10,33,73},
{9,26,1},{10,20,6},{9,19,14},{22,0,243},{2,25,72},{4,20,2},{0,17,90},{45,0,243},{0,17,90},{7,30,74},{7,30,74},{7,30,74},{7,18,74},{4,28,51},{4,20,2},{4,20,2},{3,14,6},{0,17,68},{0,14,5},{10,23,1},{10,23,1},{10,23,1},{9,17,1},{18,0,50},{5,19,0},{5,19,0},{0,14,1},{36,0,50},{0,14,1},{26,0,72},{9,26,0},{12,20,1},
{3,20,1},{26,0,72},{53,0,72},{3,20,1},{0,17,74},{53,0,72},{0,17,74},{6,0,74},{6,0,74},{6,0,74},{6,0,74},{4,20,1},{4,20,1},{4,20,1},{4,12,1},{0,14,4},{0,14,4},{11,44,144},{11,32,76},{12,25,105},{11,24,77},{8,43,243},{8,28,75},{8,25,2},{8,21,105},{0,28,248},{1,22,76},{14,38,72},{14,30,1},{15,25,5},{14,24,13},{29,0,243},
{7,29,72},{8,25,2},{0,22,76},{59,0,243},{0,22,76},{11,35,72},{11,35,72},{11,35,72},{11,23,72},{8,34,50},{8,24,1},{8,24,1},{8,18,5},{1,24,51},{3,19,1},{14,29,0},{14,29,0},{14,29,0},{14,21,1},{24,1,50},{9,24,0},{9,24,0},{3,19,0},{50,0,50},{3,19,0},{33,0,72},{13,31,0},{16,25,0},{6,25,0},{33,0,72},{66,0,72},{6,25,0},
{0,22,72},{66,0,72},{0,22,72},{11,0,72},{11,0,72},{11,0,72},{11,0,72},{8,26,0},{8,26,0},{8,26,0},{8,17,0},{2,20,0},{2,20,0},{15,48,144},{15,36,76},{16,29,105},{15,28,77},{12,47,243},{12,32,75},{12,29,2},{12,25,105},{2,33,244},{5,26,76},{18,42,72},{18,34,1},{19,29,5},{18,28,13},{35,0,243},{11,33,72},{12,29,2},{1,26,72},{71,0,243},
{1,26,72},{15,39,72},{15,39,72},{15,39,72},{15,27,72},{12,38,50},{12,28,1},{12,28,1},{12,22,5},{5,28,51},{7,23,1},{18,33,0},{18,33,0},{18,33,0},{18,25,1},{30,1,50},{13,28,0},{13,28,0},{7,23,0},{62,0,50},{7,23,0},{39,0,72},{17,35,0},{20,29,0},{10,29,0},{39,0,72},{78,0,72},{10,29,0},{0,26,72},{78,0,72},{0,26,72},{15,0,72},
{15,0,72},{15,0,72},{15,0,72},{12,30,0},{12,30,0},{12,30,0},{12,21,0},{6,24,0},{6,24,0},{19,52,144},{19,40,76},{20,33,105},{19,32,77},{16,51,243},{16,36,75},{16,33,2},{16,29,105},{6,37,244},{9,30,76},{22,46,72},{22,38,1},{23,33,5},{22,32,13},{41,0,243},{15,37,72},{16,33,2},{5,30,72},{83,0,243},{5,30,72},{19,43,72},{19,43,72},{19,43,72},
{19,31,72},{16,42,50},{16,32,1},{16,32,1},{16,26,5},{9,32,51},{11,27,1},{22,37,0},{22,37,0},{22,37,0},{22,29,1},{36,1,50},{17,32,0},{17,32,0},{11,27,0},{74,0,50},{11,27,0},{45,0,72},{21,39,0},{24,33,0},{14,33,0},{45,0,72},{91,0,72},{14,33,0},{0,30,72},{91,0,72},{0,30,72},{19,0,72},{19,0,72},{19,0,72},{19,0,72},{16,34,0},
{16,34,0},{16,34,0},{16,25,0},{10,28,0},{10,28,0},{23,56,144},{23,44,76},{24,37,105},{23,36,77},{20,55,243},{20,40,75},{20,37,2},{20,33,105},{10,41,244},{13,34,76},{26,50,72},{26,42,1},{27,37,5},{26,36,13},{47,0,243},{19,41,72},{20,37,2},{9,34,72},{95,0,243},{9,34,72},{23,47,72},{23,47,72},{23,47,72},{23,35,72},{20,46,50},{20,36,1},{20,36,1},
{20,30,5},{13,36,51},{15,31,1},{26,41,0},{26,41,0},{26,41,0},{26,33,1},{42,1,50},{21,36,0},{21,36,0},{15,31,0},{86,0,50},{15,31,0},{50,1,72},{25,43,0},{28,37,0},{18,37,0},{50,1,72},{103,0,72},{18,37,0},{0,34,72},{103,0,72},{0,34,72},{23,0,72},{23,0,72},{23,0,72},{23,0,72},{20,38,0},{20,38,0},{20,38,0},{20,29,0},{14,32,0},
{14,32,0},{28,60,146},{28,49,76},{29,42,107},{27,41,79},{25,58,244},{25,44,76},{25,41,3},{24,37,107},{14,46,244},{17,38,76},{31,54,73},{30,47,1},{31,41,6},{30,40,14},{53,1,243},{23,46,72},{25,41,2},{15,38,74},{109,0,243},{15,38,74},{28,51,74},{28,51,74},{28,51,74},{28,39,74},{25,49,51},{25,41,2},{25,41,2},{24,35,6},{18,40,51},{19,36,1},{31,44,1},
{31,44,1},{31,44,1},{30,38,1},{49,0,50},{26,40,0},{26,40,0},{21,35,1},{100,0,50},{21,35,1},{57,0,72},{30,47,0},{33,41,1},{24,41,1},{57,0,72},{117,0,72},{24,41,1},{0,38,74},{117,0,72},{0,38,74},{27,0,74},{27,0,74},{27,0,74},{27,0,74},{25,41,1},{25,41,1},{25,41,1},{25,33,1},{19,36,0},{19,36,0},{32,64,146},{32,53,76},{33,46,107},
{31,45,79},{29,62,244},{29,48,76},{29,45,3},{28,41,107},{18,50,244},{21,42,76},{35,58,73},{34,51,1},{35,45,6},{34,44,14},{59,1,243},{27,50,72},{29,45,2},{19,42,74},{121,0,243},{19,42,74},{32,55,74},{32,55,74},{32,55,74},{32,43,74},{29,53,51},{29,45,2},{29,45,2},{28,39,6},{22,44,51},{23,40,1},{35,48,1},{35,48,1},{35,48,1},{34,42,1},{55,0,50},
{30,44,0},{30,44,0},{25,39,1},{112,0,50},{25,39,1},{63,0,72},{34,51,0},{37,45,1},{28,45,1},{63,0,72},{127,1,72},{28,45,1},{0,42,74},{127,1,72},{0,42,74},{31,0,74},{31,0,74},{31,0,74},{31,0,74},{29,45,1},{29,45,1},{29,45,1},{29,37,1},{23,40,0},{23,40,0},{36,67,146},{36,57,76},{37,50,107},{35,49,79},{33,66,244},{33,52,76},{33,49,3},
{32,45,107},{22,54,244},{25,46,76},{39,62,73},{38,55,1},{39,49,6},{38,48,14},{65,0,243},{31,54,72},{33,49,2},{23,46,74},{127,3,243},{23,46,74},{36,59,74},{36,59,74},{36,59,74},{36,47,74},{33,57,51},{33,49,2},{33,49,2},{32,43,6},{26,48,51},{27,44,1},{39,52,1},{39,52,1},{39,52,1},{38,46,1},{61,0,50},{34,48,0},{34,48,0},{29,43,1},{124,0,50},
{29,43,1},{69,0,72},{38,55,0},{41,49,1},{32,49,1},{69,0,72},{127,7,72},{32,49,1},{0,46,74},{127,7,72},{0,46,74},{35,0,74},{35,0,74},{35,0,74},{35,0,74},{33,49,1},{33,49,1},{33,49,1},{33,41,1},{27,44,0},{27,44,0},{40,71,146},{40,61,76},{41,54,107},{39,53,79},{37,70,244},{37,56,76},{37,53,3},{36,49,107},{26,58,244},{29,50,76},{43,65,73},
{42,59,1},{43,53,6},{42,52,14},{71,0,243},{35,58,72},{37,53,2},{27,50,74},{127,9,243},{27,50,74},{40,63,74},{40,63,74},{40,63,74},{40,51,74},{37,61,51},{37,53,2},{37,53,2},{36,47,6},{30,52,51},{31,48,1},{43,56,1},{43,56,1},{43,56,1},{42,50,1},{66,1,50},{38,52,0},{38,52,0},{33,47,1},{126,5,50},{33,47,1},{75,0,72},{42,59,0},{45,53,1},
{36,53,1},{75,0,72},{127,13,72},{36,53,1},{0,50,74},{127,13,72},{0,50,74},{39,0,74},{39,0,74},{39,0,74},{39,0,74},{37,53,1},{37,53,1},{37,53,1},{37,45,1},{31,48,0},{31,48,0},{44,77,144},{44,65,77},{45,58,105},{44,57,77},{41,75,243},{41,61,75},{41,58,2},{41,54,105},{31,62,244},{34,55,76},{47,71,72},{47,63,1},{48,58,5},{47,57,13},{78,0,243},
{40,62,72},{41,58,2},{30,55,72},{126,16,243},{30,55,72},{44,68,72},{44,68,72},{44,68,72},{44,56,72},{41,66,50},{41,57,1},{41,57,1},{41,51,5},{34,57,51},{36,52,1},{47,62,0},{47,62,0},{47,62,0},{47,54,1},{73,0,50},{42,57,0},{42,57,0},{36,52,0},{126,12,50},{36,52,0},{81,0,72},{47,63,1},{49,58,0},{39,58,0},{81,0,72},{126,20,72},{39,58,0},
{0,55,72},{126,20,72},{0,55,72},{44,0,72},{44,0,72},{44,0,72},{44,0,72},{41,59,0},{41,59,0},{41,59,0},{41,50,0},{35,53,0},{35,53,0},{48,81,144},{48,69,77},{49,62,105},{48,61,77},{45,79,243},{45,65,73},{45,62,2},{45,58,105},{34,67,243},{38,59,76},{51,75,72},{51,66,1},{52,62,5},{51,61,13},{83,1,243},{45,65,73},{45,62,2},{34,59,72},{126,22,243},
{34,59,72},{48,72,72},{48,72,72},{48,72,72},{48,60,72},{45,70,50},{45,61,1},{45,61,1},{45,55,5},{38,61,51},{40,56,1},{51,65,0},{51,65,0},{51,65,0},{51,58,1},{79,0,50},{46,61,0},{46,61,0},{40,56,0},{127,17,50},{40,56,0},{87,0,72},{51,66,1},{53,62,0},{43,62,0},{87,0,72},{126,26,72},{43,62,0},{0,59,72},{126,26,72},{0,59,72},{48,0,72},
{48,0,72},{48,0,72},{48,0,72},{45,63,0},{45,63,0},{45,63,0},{45,54,0},{39,57,0},{39,57,0},{52,85,144},{52,73,77},{53,67,105},{52,65,77},{49,83,243},{49,69,73},{49,65,2},{49,62,105},{38,71,243},{42,63,76},{55,79,72},{55,70,1},{55,66,5},{54,65,14},{89,1,243},{49,69,73},{49,65,2},{38,63,72},{126,28,243},{38,63,72},{52,76,72},{52,76,72},{52,76,72},
{52,63,73},{49,74,50},{49,65,1},{49,65,1},{49,59,5},{41,65,50},{44,60,1},{55,69,0},{55,69,0},{55,69,0},{55,62,1},{85,0,50},{50,64,1},{50,64,1},{44,60,0},{127,23,50},{44,60,0},{93,0,72},{55,70,1},{57,65,1},{48,65,1},{93,0,72},{126,32,72},{48,65,1},{0,63,72},{126,32,72},{0,63,72},{52,0,72},{52,0,72},{52,0,72},{52,0,72},{49,67,0},
{49,67,0},{49,67,0},{49,58,0},{43,61,0},{43,61,0},{56,89,144},{56,77,77},{57,71,105},{56,69,77},{53,87,243},{53,73,73},{53,69,2},{52,66,103},{42,75,243},{45,66,79},{59,83,72},{59,74,1},{59,70,5},{58,69,14},{95,1,243},{53,73,73},{53,69,2},{43,66,74},{126,34,243},{43,66,74},{56,80,72},{56,80,72},{56,80,72},{56,67,73},{53,78,50},{53,69,1},{53,69,1},
{53,63,5},{45,69,50},{47,64,2},{59,73,0},{59,73,0},{59,73,0},{59,66,0},{91,0,50},{55,67,1},{55,67,1},{47,64,1},{127,29,50},{47,64,1},{99,0,72},{59,74,1},{61,69,1},{52,69,1},{99,0,72},{126,38,72},{52,69,1},{0,66,74},{126,38,72},{0,66,74},{56,0,72},{56,0,72},{56,0,72},{56,0,72},{53,71,0},{53,71,0},{53,71,0},{53,62,0},{47,64,1},
{47,64,1},{61,92,146},{61,80,79},{61,75,103},{60,74,78},{58,91,244},{57,78,74},{58,74,2},{56,70,105},{47,79,243},{49,71,77},{63,88,73},{63,79,2},{64,74,5},{63,73,14},{102,0,243},{57,78,73},{58,74,1},{47,71,72},{126,41,243},{47,71,72},{61,83,74},{61,83,74},{61,83,74},{60,72,74},{58,82,51},{58,73,2},{58,73,2},{57,68,5},{50,73,50},{52,68,1},{63,79,1},
{63,79,1},{63,79,1},{63,71,1},{98,0,50},{59,72,1},{59,72,1},{53,68,0},{127,36,50},{53,68,0},{106,0,72},{63,79,1},{65,74,0},{56,74,0},{106,0,72},{127,44,72},{56,74,0},{0,71,72},{127,44,72},{0,71,72},{60,0,74},{60,0,74},{60,0,74},{60,0,74},{58,74,1},{58,74,1},{58,74,1},{57,66,1},{52,68,1},{52,68,1},{64,97,144},{64,85,76},{65,78,105},
{64,77,77},{62,95,244},{61,82,74},{62,78,2},{60,74,105},{51,83,243},{53,75,77},{67,91,72},{67,83,1},{68,78,5},{67,77,13},{108,0,243},{61,82,73},{62,78,1},{51,75,72},{125,47,243},{51,75,72},{64,88,72},{64,88,72},{64,88,72},{64,76,72},{62,86,51},{62,77,2},{62,77,2},{61,72,5},{54,77,50},{56,72,1},{67,82,0},{67,82,0},{67,82,0},{67,74,1},{104,0,50},
{63,76,1},{63,76,1},{57,72,0},{127,42,50},{57,72,0},{112,0,72},{66,84,0},{69,78,0},{60,78,0},{112,0,72},{127,50,72},{60,78,0},{0,75,72},{127,50,72},{0,75,72},{64,0,72},{64,0,72},{64,0,72},{64,0,72},{62,78,1},{62,78,1},{62,78,1},{61,70,1},{56,72,1},{56,72,1},{68,101,144},{68,89,76},{69,82,105},{68,81,77},{65,100,243},{65,85,75},{65,82,2},
{65,78,105},{55,87,243},{57,79,77},{71,95,72},{71,87,1},{72,82,5},{71,81,13},{114,0,243},{64,86,72},{65,82,2},{55,79,72},{127,52,243},{55,79,72},{68,92,72},{68,92,72},{68,92,72},{68,80,72},{65,91,50},{65,81,1},{65,81,1},{65,75,5},{58,81,50},{60,76,1},{71,86,0},{71,86,0},{71,86,0},{71,78,1},{110,0,50},{66,81,0},{66,81,0},{61,76,0},{126,48,50},
{61,76,0},{118,0,72},{70,88,0},{73,82,0},{64,82,0},{118,0,72},{127,56,72},{64,82,0},{0,79,72},{127,56,72},{0,79,72},{68,0,72},{68,0,72},{68,0,72},{68,0,72},{65,83,0},{65,83,0},{65,83,0},{65,74,0},{60,76,1},{60,76,1},{72,105,144},{72,93,76},{73,86,105},{72,85,77},{69,104,243},{69,89,75},{69,86,2},{69,82,105},{59,91,243},{61,83,77},{75,99,72},
{75,91,1},{76,86,5},{75,85,13},{120,0,243},{68,90,72},{69,86,2},{59,83,72},{127,58,243},{59,83,72},{72,96,72},{72,96,72},{72,96,72},{72,84,72},{69,95,50},{69,85,1},{69,85,1},{69,79,5},{62,85,50},{64,80,1},{75,90,0},{75,90,0},{75,90,0},{75,82,1},{115,1,50},{70,85,0},{70,85,0},{64,80,0},{126,54,50},{64,80,0},{124,0,72},{74,92,0},{77,86,0},
{67,86,0},{124,0,72},{127,62,72},{67,86,0},{0,83,72},{127,62,72},{0,83,72},{72,0,72},{72,0,72},{72,0,72},{72,0,72},{69,87,0},{69,87,0},{69,87,0},{69,78,0},{64,80,1},{64,80,1},{77,109,146},{77,98,76},{78,91,107},{76,90,79},{74,107,244},{74,93,76},{74,90,3},{73,86,107},{63,96,243},{66,87,76},{80,103,73},{79,96,1},{80,90,6},{79,89,14},{127,0,243},
{72,95,72},{74,90,2},{64,87,74},{126,65,243},{64,87,74},{77,100,74},{77,100,74},{77,100,74},{77,88,74},{74,98,51},{74,90,2},{74,90,2},{73,84,6},{67,89,51},{68,85,1},{80,93,1},{80,93,1},{80,93,1},{79,87,1},{122,0,50},{75,89,0},{75,89,0},{70,84,1},{126,61,50},{70,84,1},{127,7,72},{79,96,0},{82,90,1},{73,90,1},{127,7,72},{127,68,72},{73,90,1},
{0,87,74},{127,68,72},{0,87,74},{76,0,74},{76,0,74},{76,0,74},{76,0,74},{74,90,1},{74,90,1},{74,90,1},{74,82,1},{68,85,0},{68,85,0},{81,113,146},{81,102,76},{82,95,107},{80,94,79},{78,111,244},{78,97,76},{78,94,3},{77,90,107},{67,99,244},{70,91,76},{84,107,73},{83,100,1},{84,94,6},{83,93,14},{127,11,243},{76,99,72},{78,94,2},{68,91,74},{126,71,243},
{68,91,74},{81,104,74},{81,104,74},{81,104,74},{81,92,74},{78,102,51},{78,94,2},{78,94,2},{77,88,6},{71,93,51},{72,89,1},{84,97,1},{84,97,1},{84,97,1},{83,91,1},{127,2,50},{79,93,0},{79,93,0},{74,88,1},{127,66,50},{74,88,1},{127,19,72},{83,100,0},{86,94,1},{77,94,1},{127,19,72},{127,74,72},{77,94,1},{0,91,74},{127,74,72},{0,91,74},{80,0,74},
{80,0,74},{80,0,74},{80,0,74},{78,94,1},{78,94,1},{78,94,1},{78,86,1},{72,89,0},{72,89,0},{85,117,146},{85,106,76},{86,99,107},{84,98,79},{82,115,244},{82,101,76},{82,98,3},{81,94,107},{71,103,244},{74,95,76},{88,111,73},{87,104,1},{88,98,6},{87,97,14},{127,24,243},{80,103,72},{82,98,2},{72,95,74},{125,77,243},{72,95,74},{85,108,74},{85,108,74},{85,108,74},
{85,96,74},{82,106,51},{82,98,2},{82,98,2},{81,92,6},{75,97,51},{76,93,1},{88,101,1},{88,101,1},{88,101,1},{87,95,1},{127,14,50},{83,97,0},{83,97,0},{78,92,1},{127,72,50},{78,92,1},{127,31,72},{87,104,0},{90,98,1},{81,98,1},{127,31,72},{127,80,72},{81,98,1},{0,95,74},{127,80,72},{0,95,74},{84,0,74},{84,0,74},{84,0,74},{84,0,74},{82,98,1},
{82,98,1},{82,98,1},{82,90,1},{76,93,0},{76,93,0},{89,121,146},{89,110,76},{90,103,107},{88,102,79},{86,119,244},{86,105,76},{86,102,3},{85,98,107},{75,107,244},{78,99,76},{92,115,73},{91,108,1},{92,102,6},{91,101,14},{127,36,243},{84,107,72},{86,102,2},{76,99,74},{127,82,243},{76,99,74},{89,112,74},{89,112,74},{89,112,74},{89,100,74},{86,110,51},{86,102,2},{86,102,2},
{85,96,6},{79,101,51},{80,97,1},{92,105,1},{92,105,1},{92,105,1},{91,99,1},{127,27,50},{87,101,0},{87,101,0},{82,96,1},{126,78,50},{82,96,1},{127,43,72},{91,108,0},{94,102,1},{85,102,1},{127,43,72},{127,86,72},{85,102,1},{0,99,74},{127,86,72},{0,99,74},{88,0,74},{88,0,74},{88,0,74},{88,0,74},{86,102,1},{86,102,1},{86,102,1},{86,94,1},{80,97,0},
{80,97,0},{93,126,144},{93,114,76},{94,107,105},{93,106,77},{90,125,243},{90,110,75},{90,107,2},{90,103,105},{80,111,244},{83,104,76},{96,120,72},{96,112,1},{97,107,5},{96,106,13},{127,50,243},{89,111,72},{90,107,2},{79,104,72},{127,89,243},{79,104,72},{93,117,72},{93,117,72},{93,117,72},{93,105,72},{90,116,50},{90,106,1},{90,106,1},{90,100,5},{83,106,51},{85,101,1},{96,111,0},
{96,111,0},{96,111,0},{96,103,1},{127,40,50},{91,106,0},{91,106,0},{85,101,0},{126,85,50},{85,101,0},{127,57,72},{95,113,0},{98,107,0},{88,107,0},{127,57,72},{126,93,72},{88,107,0},{0,104,72},{126,93,72},{0,104,72},{93,0,72},{93,0,72},{93,0,72},{93,0,72},{90,108,0},{90,108,0},{90,108,0},{90,99,0},{84,102,0},{84,102,0},{97,127,152},{97,118,76},{98,111,105},
{97,110,77},{94,127,244},{94,114,75},{94,111,2},{94,107,105},{84,115,244},{87,108,76},{100,124,72},{100,116,1},{101,111,5},{100,110,13},{127,62,243},{93,115,72},{94,111,2},{83,108,72},{127,95,243},{83,108,72},{97,121,72},{97,121,72},{97,121,72},{97,109,72},{94,120,50},{94,110,1},{94,110,1},{94,104,5},{87,110,51},{89,105,1},{100,115,0},{100,115,0},{100,115,0},{100,107,1},{127,53,50},
{95,110,0},{95,110,0},{89,105,0},{126,91,50},{89,105,0},{127,69,72},{99,117,0},{102,111,0},{92,111,0},{127,69,72},{126,99,72},{92,111,0},{0,108,72},{126,99,72},{0,108,72},{97,0,72},{97,0,72},{97,0,72},{97,0,72},{94,112,0},{94,112,0},{94,112,0},{94,103,0},{88,106,0},{88,106,0},{102,126,184},{101,122,76},{102,115,105},{101,114,77},{100,127,260},{98,118,75},{98,115,2},
{98,111,105},{88,119,244},{91,112,76},{104,127,74},{104,120,1},{105,115,5},{104,114,13},{127,73,243},{97,119,72},{98,115,2},{87,112,72},{127,101,243},{87,112,72},{101,125,72},{101,125,72},{101,125,72},{101,113,72},{98,124,50},{98,114,1},{98,114,1},{98,108,5},{91,114,51},{93,109,1},{104,119,0},{104,119,0},{104,119,0},{104,111,1},{127,64,50},{99,114,0},{99,114,0},{93,109,0},{126,97,50},
{93,109,0},{127,81,72},{103,121,0},{106,115,0},{96,115,0},{127,81,72},{126,105,72},{96,115,0},{0,112,72},{126,105,72},{0,112,72},{101,0,72},{101,0,72},{101,0,72},{101,0,72},{98,116,0},{98,116,0},{98,116,0},{98,107,0},{92,110,0},{92,110,0},{106,127,224},{105,126,76},{106,119,105},{105,118,77},{104,127,299},{102,122,75},{102,119,2},{102,115,105},{92,123,244},{95,116,76},{109,127,84},
{108,124,1},{109,119,5},{108,118,13},{127,86,243},{101,123,72},{102,119,2},{91,116,72},{127,107,243},{91,116,72},{105,126,76},{105,126,76},{105,126,76},{105,117,72},{102,126,52},{102,118,1},{102,118,1},{102,112,5},{95,118,51},{97,113,1},{108,123,0},{108,123,0},{108,123,0},{108,115,1},{127,76,50},{103,118,0},{103,118,0},{97,113,0},{126,103,50},{97,113,0},{127,93,72},{107,125,0},{110,119,0},
{100,119,0},{127,93,72},{126,111,72},{100,119,0},{0,116,72},{126,111,72},{0,116,72},{105,0,72},{105,0,72},{105,0,72},{105,0,72},{102,120,0},{102,120,0},{102,120,0},{102,111,0},{96,114,0},{96,114,0},{111,127,290},{111,127,103},{111,124,107},{109,123,79},{111,127,345},{107,126,76},{107,123,3},{106,119,107},{98,127,248},{99,120,76},{114,127,113},{113,127,5},{113,123,6},{112,122,14},{127,99,243},
{107,126,75},{107,123,2},{97,120,74},{126,114,243},{97,120,74},{110,126,90},{110,126,90},{110,126,90},{110,121,74},{108,126,67},{107,123,2},{107,123,2},{106,117,6},{100,122,51},{101,118,1},{113,126,1},{113,126,1},{113,126,1},{112,120,1},{127,90,50},{108,122,0},{108,122,0},{103,117,1},{127,109,50},{103,117,1},{127,107,72},{113,127,4},{115,123,1},{106,123,1},{127,107,72},{127,117,72},{106,123,1},
{0,120,74},{127,117,72},{0,120,74},{109,0,74},{109,0,74},{109,0,74},{109,0,74},{107,123,1},{107,123,1},{107,123,1},{107,115,1},{101,118,0},{101,118,0},{117,127,343},{115,127,179},{114,127,110},{113,126,78},{115,127,387},{111,127,102},{111,127,2},{110,123,90},{105,127,263},{103,124,63},{120,127,134},{118,127,46},{117,127,5},{116,126,9},{127,110,221},{113,127,89},{111,127,1},{101,124,61},{126,119,221},
{101,124,61},{114,127,110},{114,127,110},{114,127,110},{114,125,74},{112,127,91},{111,127,2},{111,127,2},{110,121,6},{104,126,51},{105,122,1},{117,127,5},{117,127,5},{117,127,5},{116,124,1},{127,102,50},{112,126,0},{112,126,0},{107,121,1},{127,115,50},{107,121,1},{127,118,61},{119,127,25},{119,127,0},{110,127,0},{127,118,61},{126,123,61},{110,127,0},{0,124,61},{126,123,61},{0,124,61},{113,0,74},
{113,0,74},{113,0,74},{113,0,74},{111,127,1},{111,127,1},{111,127,1},{111,119,1},{105,122,0},{105,122,0},{120,127,239},{119,127,179},{119,127,154},{118,127,83},{120,127,254},{116,127,78},{116,127,29},{114,126,15},{113,127,169},{109,126,5},{123,127,54},{122,127,34},{122,127,25},{121,127,2},{127,119,93},{119,127,33},{119,127,8},{109,126,5},{127,123,93},{109,126,5},{119,126,154},{119,126,154},{119,126,154},
{118,127,83},{117,127,125},{116,127,29},{116,127,29},{114,125,6},{110,127,72},{109,126,1},{122,126,25},{122,126,25},{122,126,25},{121,127,2},{127,115,50},{119,127,8},{119,127,8},{111,125,1},{127,121,50},{111,125,1},{127,124,5},{125,127,1},{125,127,0},{122,127,0},{127,124,5},{126,126,5},{122,127,0},{0,126,5},{126,126,5},{0,126,5},{117,0,74},{117,0,74},{117,0,74},{117,0,74},{115,127,10},
{115,127,10},{115,127,10},{115,123,1},{109,126,0},{109,126,0},{123,127,140},{123,127,124},{123,127,115},{122,127,83},{123,127,131},{121,127,66},{120,127,50},{119,127,1},{119,127,86},{116,127,8},{126,127,11},{125,127,11},{125,127,10},{125,127,2},{127,124,17},{125,127,6},{124,127,5},{119,127,0},{126,126,17},{119,127,0},{123,127,115},{123,127,115},{123,127,115},{122,127,83},{121,127,98},{120,127,50},{120,127,50},
{119,127,1},{116,127,57},{116,127,8},{125,126,10},{125,126,10},{125,126,10},{125,127,2},{127,122,13},{124,127,5},{124,127,5},{119,127,0},{126,125,13},{119,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{121,0,74},{121,0,74},{121,0,74},{121,0,74},{120,127,25},{120,127,25},{120,127,25},{119,127,1},{116,127,8},
{116,127,8},{0,29,200},{0,19,18},{0,14,1},{0,13,73},{0,20,441},{0,11,281},{0,11,134},{0,7,331},{0,8,474},{0,7,356},{0,29,200},{0,19,18},{0,14,1},{0,13,73},{10,0,441},{0,11,281},{0,11,134},{0,7,331},{20,0,441},{0,7,331},{0,13,0},{0,13,0},{0,13,0},{0,7,0},{0,6,41},{0,5,13},{0,5,13},{0,3,25},{0,3,45},{0,3,29},{0,13,0},
{0,13,0},{0,13,0},{0,7,0},{3,0,41},{0,5,13},{0,5,13},{0,3,25},{6,0,41},{0,3,25},{14,0,200},{0,19,18},{0,14,1},{0,13,73},{14,0,200},{29,0,200},{0,13,73},{0,9,202},{29,0,200},{0,9,202},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,41,200},{0,27,2},{1,19,18},
{0,17,45},{0,27,686},{0,19,346},{0,15,158},{0,10,467},{0,11,762},{0,10,516},{0,41,200},{0,27,2},{1,19,14},{0,17,45},{13,1,686},{0,19,346},{0,15,158},{0,10,467},{27,0,686},{0,10,467},{0,26,0},{0,26,0},{0,26,0},{0,13,0},{0,12,145},{0,11,53},{0,11,53},{0,6,89},{0,5,158},{0,6,105},{0,26,0},{0,26,0},{0,26,0},{0,13,0},{6,0,145},
{0,11,53},{0,11,53},{0,6,89},{12,0,145},{0,6,89},{20,0,200},{1,26,2},{4,18,1},{0,17,45},{20,0,200},{41,0,200},{0,17,45},{0,13,202},{41,0,200},{0,13,202},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,49,225},{2,32,27},{3,24,71},{1,21,54},{0,39,723},{0,25,283},{0,20,65},
{0,15,419},{0,17,865},{0,15,519},{4,45,201},{3,32,1},{5,23,19},{3,21,35},{19,0,723},{0,25,283},{0,20,65},{0,15,419},{39,0,723},{0,15,419},{2,34,25},{2,34,25},{2,34,25},{2,17,26},{0,24,162},{0,17,20},{0,17,20},{0,10,61},{0,11,213},{0,9,110},{4,30,1},{4,30,1},{4,30,1},{3,17,1},{12,0,162},{0,17,20},{0,17,20},{0,10,61},{24,0,162},
{0,10,61},{26,0,200},{3,32,0},{8,22,1},{0,21,18},{26,0,200},{53,0,200},{0,21,18},{0,17,202},{53,0,200},{0,17,202},{2,0,25},{2,0,25},{2,0,25},{2,0,25},{0,10,0},{0,10,0},{0,10,0},{0,5,0},{0,5,9},{0,5,9},{4,57,313},{4,39,121},{5,28,198},{4,25,131},{0,52,723},{0,31,227},{0,26,9},{0,20,362},{0,22,959},{0,19,515},{8,49,201},
{7,36,1},{9,27,19},{7,25,35},{25,0,723},{0,31,227},{0,26,9},{0,20,362},{52,0,723},{0,20,362},{4,42,113},{4,42,113},{4,42,113},{4,22,114},{0,36,162},{0,24,2},{0,24,2},{0,15,34},{0,17,285},{0,14,122},{8,34,1},{8,34,1},{8,34,1},{7,21,1},{18,0,162},{1,23,2},{1,23,2},{0,15,34},{36,0,162},{0,15,34},{32,0,200},{7,36,0},{12,26,1},
{0,26,5},{32,0,200},{65,0,200},{0,26,5},{0,21,202},{65,0,200},{0,21,202},{4,0,113},{4,0,113},{4,0,113},{4,0,113},{0,23,0},{0,23,0},{0,23,0},{0,11,0},{0,8,41},{0,8,41},{7,64,400},{7,43,216},{8,32,312},{7,30,213},{2,61,723},{1,38,208},{2,31,2},{0,24,312},{0,31,980},{0,24,440},{12,54,200},{12,40,0},{13,31,20},{11,29,36},{32,0,723},
{0,38,203},{2,31,2},{0,24,296},{65,0,723},{0,24,296},{7,49,200},{7,49,200},{7,49,200},{7,28,200},{2,46,162},{2,31,2},{2,31,2},{2,20,20},{0,22,280},{0,19,84},{12,39,0},{12,39,0},{12,39,0},{12,25,1},{24,1,162},{4,29,0},{4,29,0},{0,20,13},{50,0,162},{0,20,13},{39,0,200},{12,40,0},{16,31,0},{0,31,0},{39,0,200},{78,0,200},{0,31,0},
{0,26,200},{78,0,200},{0,26,200},{7,0,200},{7,0,200},{7,0,200},{7,0,200},{2,32,0},{2,32,0},{2,32,0},{2,17,0},{0,14,50},{0,14,50},{11,68,400},{11,47,216},{12,36,312},{11,34,213},{6,65,723},{5,42,208},{6,35,2},{4,28,312},{0,37,868},{0,28,304},{16,58,200},{16,44,0},{17,35,20},{15,33,36},{38,0,723},{3,43,200},{6,35,2},{0,29,257},{77,0,723},
{0,29,257},{11,53,200},{11,53,200},{11,53,200},{11,32,200},{6,50,162},{6,35,2},{6,35,2},{6,24,20},{0,28,216},{0,24,20},{16,43,0},{16,43,0},{16,43,0},{16,29,1},{30,1,162},{8,33,0},{8,33,0},{0,24,4},{62,0,162},{0,24,4},{45,0,200},{16,44,0},{20,35,0},{4,35,0},{45,0,200},{91,0,200},{4,35,0},{0,30,200},{91,0,200},{0,30,200},{11,0,200},
{11,0,200},{11,0,200},{11,0,200},{6,36,0},{6,36,0},{6,36,0},{6,21,0},{0,22,8},{0,22,8},{15,72,400},{15,51,216},{16,40,312},{15,38,213},{10,69,723},{9,46,208},{10,39,2},{8,32,312},{0,43,788},{0,33,228},{20,62,200},{20,48,0},{21,39,20},{19,37,36},{44,0,723},{7,47,200},{10,39,2},{0,33,224},{89,0,723},{0,33,224},{15,57,200},{15,57,200},{15,57,200},
{15,36,200},{10,54,162},{10,39,2},{10,39,2},{10,28,20},{0,35,179},{1,29,0},{20,47,0},{20,47,0},{20,47,0},{20,33,1},{36,1,162},{12,37,0},{12,37,0},{1,29,0},{74,0,162},{1,29,0},{50,1,200},{20,48,0},{24,39,0},{8,39,0},{50,1,200},{103,0,200},{8,39,0},{0,34,200},{103,0,200},{0,34,200},{15,0,200},{15,0,200},{15,0,200},{15,0,200},{10,40,0},
{10,40,0},{10,40,0},{10,25,0},{1,29,0},{1,29,0},{19,76,400},{19,55,216},{20,44,312},{19,42,213},{14,73,723},{13,50,208},{14,43,2},{12,36,312},{0,49,740},{1,37,216},{24,66,200},{24,52,0},{25,43,20},{23,41,36},{50,0,723},{11,51,200},{14,43,2},{0,38,209},{101,0,723},{0,38,209},{19,61,200},{19,61,200},{19,61,200},{19,40,200},{14,58,162},{14,43,2},{14,43,2},
{14,32,20},{1,42,163},{5,33,0},{24,51,0},{24,51,0},{24,51,0},{24,37,1},{42,1,162},{16,41,0},{16,41,0},{5,33,0},{86,0,162},{5,33,0},{56,1,200},{24,52,0},{28,43,0},{12,43,0},{56,1,200},{115,0,200},{12,43,0},{0,38,200},{115,0,200},{0,38,200},{19,0,200},{19,0,200},{19,0,200},{19,0,200},{14,44,0},{14,44,0},{14,44,0},{14,29,0},{5,33,0},
{5,33,0},{24,80,402},{24,60,212},{25,49,308},{23,47,212},{19,76,724},{19,53,212},{19,47,4},{17,41,308},{1,55,724},{6,42,212},{29,69,201},{28,57,1},{30,48,19},{28,46,35},{56,1,723},{15,56,200},{19,47,3},{1,42,202},{115,0,723},{1,42,202},{24,64,202},{24,64,202},{24,64,202},{24,44,202},{19,61,163},{19,46,3},{19,46,3},{18,36,19},{6,46,163},{9,38,1},{29,55,1},
{29,55,1},{29,55,1},{28,42,1},{49,0,162},{20,46,0},{20,46,0},{11,37,1},{100,0,162},{11,37,1},{63,0,200},{28,57,0},{33,47,1},{18,47,1},{63,0,200},{127,1,200},{18,47,1},{0,42,202},{127,1,200},{0,42,202},{23,0,202},{23,0,202},{23,0,202},{23,0,202},{19,48,1},{19,48,1},{19,48,1},{19,33,1},{9,38,0},{9,38,0},{28,84,402},{28,64,215},{29,53,308},
{27,51,212},{23,80,724},{23,57,212},{23,51,4},{21,45,308},{5,59,724},{10,46,212},{33,73,201},{32,61,1},{34,52,19},{32,50,35},{62,1,723},{19,60,200},{23,51,3},{5,46,202},{127,0,723},{5,46,202},{28,68,202},{28,68,202},{28,68,202},{28,48,202},{23,65,163},{23,50,3},{23,50,3},{22,40,19},{10,50,163},{13,42,1},{33,59,1},{33,59,1},{33,59,1},{32,46,1},{55,0,162},
{24,50,0},{24,50,0},{15,41,1},{112,0,162},{15,41,1},{69,0,200},{32,61,0},{37,51,1},{22,51,1},{69,0,200},{127,7,200},{22,51,1},{0,46,202},{127,7,200},{0,46,202},{27,0,202},{27,0,202},{27,0,202},{27,0,202},{23,52,1},{23,52,1},{23,52,1},{23,37,1},{13,42,0},{13,42,0},{32,88,402},{32,68,215},{33,57,308},{31,55,212},{27,84,724},{27,61,212},{27,55,4},
{25,49,308},{8,64,723},{14,50,212},{37,77,201},{36,64,2},{38,56,19},{36,54,35},{68,0,723},{22,64,201},{27,55,3},{9,50,202},{127,6,723},{9,50,202},{32,72,202},{32,72,202},{32,72,202},{32,52,202},{27,69,163},{27,54,3},{27,54,3},{26,44,19},{14,54,163},{17,46,1},{37,63,1},{37,63,1},{37,63,1},{36,50,1},{61,0,162},{28,54,0},{28,54,0},{19,45,1},{124,0,162},
{19,45,1},{75,0,200},{36,64,1},{41,55,1},{26,55,1},{75,0,200},{127,13,200},{26,55,1},{0,50,202},{127,13,200},{0,50,202},{31,0,202},{31,0,202},{31,0,202},{31,0,202},{27,56,1},{27,56,1},{27,56,1},{27,41,1},{17,46,0},{17,46,0},{36,92,402},{36,72,215},{37,61,308},{35,59,212},{31,88,724},{30,65,207},{31,59,4},{29,53,308},{12,68,723},{18,54,212},{41,81,201},
{41,67,2},{42,60,19},{40,58,35},{74,0,723},{28,67,201},{31,59,3},{13,54,202},{127,12,723},{13,54,202},{36,76,202},{36,76,202},{36,76,202},{36,56,202},{31,73,163},{31,58,3},{31,58,3},{30,48,19},{18,58,163},{21,50,1},{41,66,1},{41,66,1},{41,66,1},{40,54,1},{66,1,162},{32,58,0},{32,58,0},{23,49,1},{126,5,162},{23,49,1},{80,1,200},{41,67,1},{45,59,1},
{30,59,1},{80,1,200},{126,19,200},{30,59,1},{0,54,202},{126,19,200},{0,54,202},{35,0,202},{35,0,202},{35,0,202},{35,0,202},{31,60,1},{31,60,1},{31,60,1},{31,45,1},{21,50,0},{21,50,0},{40,97,400},{40,77,213},{42,64,314},{40,63,213},{35,94,723},{35,70,209},{35,64,6},{33,57,312},{17,72,723},{22,58,216},{45,87,200},{45,72,1},{46,64,21},{44,62,36},{80,1,723},
{33,71,201},{36,63,4},{16,59,200},{126,19,723},{16,59,200},{40,82,200},{40,82,200},{40,82,200},{40,61,200},{35,78,162},{35,63,4},{35,63,4},{35,53,20},{22,63,163},{26,54,0},{45,72,0},{45,72,0},{45,72,0},{45,58,1},{73,0,162},{37,62,0},{37,62,0},{26,54,0},{126,12,162},{26,54,0},{87,0,200},{45,72,1},{49,64,1},{32,64,1},{87,0,200},{126,26,200},{32,64,1},
{0,59,200},{126,26,200},{0,59,200},{40,0,200},{40,0,200},{40,0,200},{40,0,200},{35,65,0},{35,65,0},{35,65,0},{35,50,0},{26,54,0},{26,54,0},{44,101,400},{44,81,213},{46,68,314},{44,66,213},{39,98,723},{39,74,209},{39,67,2},{37,61,312},{21,76,723},{26,62,216},{49,91,200},{49,76,1},{50,68,21},{48,66,38},{86,1,723},{37,75,201},{39,67,2},{20,63,200},{126,25,723},
{20,63,200},{44,86,200},{44,86,200},{44,86,200},{44,64,200},{39,82,162},{39,67,1},{39,67,1},{39,57,20},{25,67,162},{30,58,0},{49,76,0},{49,76,0},{49,76,0},{49,62,1},{79,0,162},{41,65,1},{41,65,1},{30,58,0},{127,17,162},{30,58,0},{93,0,200},{49,76,1},{53,67,1},{38,67,1},{93,0,200},{126,32,200},{38,67,1},{0,63,200},{126,32,200},{0,63,200},{44,0,200},
{44,0,200},{44,0,200},{44,0,200},{39,69,0},{39,69,0},{39,69,0},{39,54,0},{30,58,0},{30,58,0},{48,105,400},{48,85,213},{50,72,314},{48,70,213},{43,102,723},{43,78,209},{43,71,2},{41,65,315},{25,80,723},{29,66,215},{53,95,200},{53,80,1},{54,72,21},{52,70,38},{92,1,723},{41,79,201},{43,71,2},{25,66,202},{126,31,723},{25,66,202},{48,90,200},{48,90,200},{48,90,200},
{48,68,200},{43,86,162},{43,71,1},{43,71,1},{43,61,20},{29,71,162},{34,62,0},{53,80,0},{53,80,0},{53,80,0},{53,66,0},{85,0,162},{45,69,1},{45,69,1},{34,62,0},{127,23,162},{34,62,0},{99,0,200},{53,80,1},{57,71,1},{42,71,1},{99,0,200},{126,38,200},{42,71,1},{0,66,202},{126,38,200},{0,66,202},{48,0,200},{48,0,200},{48,0,200},{48,0,200},{43,73,0},
{43,73,0},{43,73,0},{43,58,0},{34,62,0},{34,62,0},{52,109,400},{52,89,213},{54,76,314},{52,74,213},{47,106,723},{47,82,209},{47,75,2},{45,69,315},{29,84,723},{33,70,215},{57,99,200},{57,84,1},{58,76,21},{56,74,38},{98,1,723},{45,83,201},{47,75,2},{29,70,202},{126,37,723},{29,70,202},{52,94,200},{52,94,200},{52,94,200},{52,72,200},{47,90,162},{47,75,1},{47,75,1},
{46,65,21},{33,75,162},{38,65,2},{57,84,0},{57,84,0},{57,84,0},{57,70,0},{91,0,162},{49,73,1},{49,73,1},{39,65,1},{127,29,162},{39,65,1},{105,0,200},{57,84,1},{61,75,1},{46,75,1},{105,0,200},{126,44,200},{46,75,1},{0,70,202},{126,44,200},{0,70,202},{52,0,200},{52,0,200},{52,0,200},{52,0,200},{47,77,0},{47,77,0},{47,77,0},{47,62,0},{38,65,1},
{38,65,1},{57,113,402},{57,93,215},{58,82,315},{56,79,215},{52,109,724},{51,86,207},{52,80,2},{51,73,314},{33,89,723},{38,75,213},{62,102,201},{62,88,2},{62,81,21},{60,79,38},{105,0,723},{49,88,201},{52,80,1},{33,75,200},{126,44,723},{33,75,200},{57,97,202},{57,97,202},{57,97,202},{56,77,202},{52,94,163},{52,79,2},{52,79,2},{51,69,21},{38,79,162},{42,70,1},{62,87,1},
{62,87,1},{62,87,1},{62,74,1},{98,0,162},{54,77,1},{54,77,1},{43,70,0},{127,36,162},{43,70,0},{112,0,200},{62,88,1},{65,80,0},{50,80,0},{112,0,200},{127,50,200},{50,80,0},{0,75,200},{127,50,200},{0,75,200},{56,0,202},{56,0,202},{56,0,202},{56,0,202},{52,80,1},{52,80,1},{52,80,1},{51,66,1},{43,69,1},{43,69,1},{61,117,402},{61,97,215},{62,86,315},
{60,83,215},{56,113,724},{55,90,207},{56,84,2},{55,77,314},{37,93,723},{42,79,213},{65,107,200},{65,93,0},{66,84,20},{64,82,36},{111,0,723},{53,92,201},{56,84,1},{37,79,200},{127,49,723},{37,79,200},{61,101,202},{61,101,202},{61,101,202},{60,81,202},{56,98,163},{56,83,2},{56,83,2},{55,73,21},{42,83,162},{46,74,1},{65,92,0},{65,92,0},{65,92,0},{65,78,1},{104,0,162},
{58,81,1},{58,81,1},{47,74,0},{127,42,162},{47,74,0},{118,0,200},{65,93,0},{69,84,0},{54,84,0},{118,0,200},{127,56,200},{54,84,0},{0,79,200},{127,56,200},{0,79,200},{60,0,202},{60,0,202},{60,0,202},{60,0,202},{56,84,1},{56,84,1},{56,84,1},{55,70,1},{47,73,1},{47,73,1},{64,122,400},{64,100,216},{65,89,312},{64,87,213},{60,117,724},{59,94,207},{60,88,2},
{59,81,314},{41,97,723},{46,83,213},{69,111,200},{69,97,0},{70,88,20},{68,86,36},{117,0,723},{57,96,201},{60,88,1},{41,83,200},{127,55,723},{41,83,200},{64,106,200},{64,106,200},{64,106,200},{64,85,200},{60,102,163},{60,87,2},{60,87,2},{59,77,21},{46,87,162},{50,78,1},{69,96,0},{69,96,0},{69,96,0},{69,82,1},{110,0,162},{62,85,1},{62,85,1},{51,78,0},{126,48,162},
{51,78,0},{124,0,200},{69,97,0},{73,88,0},{58,88,0},{124,0,200},{127,62,200},{58,88,0},{0,83,200},{127,62,200},{0,83,200},{64,0,200},{64,0,200},{64,0,200},{64,0,200},{60,88,1},{60,88,1},{60,88,1},{59,74,1},{51,77,1},{51,77,1},{68,126,400},{68,104,216},{69,93,312},{68,91,213},{63,123,724},{63,98,207},{63,92,6},{63,85,314},{45,101,723},{50,87,213},{73,115,200},
{73,101,0},{74,92,20},{72,90,36},{123,0,723},{61,100,201},{64,91,4},{45,87,200},{127,61,723},{45,87,200},{68,110,200},{68,110,200},{68,110,200},{68,89,200},{63,108,163},{64,91,4},{64,91,4},{63,81,21},{50,91,162},{54,82,1},{73,100,0},{73,100,0},{73,100,0},{73,86,1},{115,1,162},{65,90,0},{65,90,0},{55,82,0},{126,54,162},{55,82,0},{127,5,200},{73,101,0},{77,92,0},
{62,92,0},{127,5,200},{126,68,200},{62,92,0},{0,87,200},{126,68,200},{0,87,200},{68,0,200},{68,0,200},{68,0,200},{68,0,200},{63,94,1},{63,94,1},{63,94,1},{63,78,1},{55,81,1},{55,81,1},{73,126,410},{73,109,212},{74,98,308},{72,96,212},{68,126,724},{68,102,212},{68,96,4},{66,90,308},{50,105,723},{54,91,215},{78,119,201},{77,106,1},{79,97,19},{77,95,35},{127,5,723},
{64,105,200},{68,96,3},{50,91,202},{126,68,723},{50,91,202},{73,114,202},{73,114,202},{73,114,202},{73,93,202},{68,110,163},{68,95,3},{68,95,3},{67,85,19},{54,96,162},{59,86,2},{78,104,1},{78,104,1},{78,104,1},{77,91,1},{122,0,162},{69,95,0},{69,95,0},{60,86,1},{126,61,162},{60,86,1},{127,19,200},{77,106,0},{82,96,1},{67,96,1},{127,19,200},{127,74,200},{67,96,1},
{0,91,202},{127,74,200},{0,91,202},{72,0,202},{72,0,202},{72,0,202},{72,0,202},{68,97,1},{68,97,1},{68,97,1},{68,82,1},{59,86,1},{59,86,1},{77,127,434},{77,113,212},{78,102,308},{76,100,212},{72,127,732},{72,106,212},{72,100,4},{70,94,308},{54,109,723},{58,95,215},{82,123,201},{81,110,1},{83,101,19},{81,99,35},{127,18,723},{68,109,200},{72,100,3},{54,95,202},{126,74,723},
{54,95,202},{77,118,202},{77,118,202},{77,118,202},{77,97,202},{72,114,163},{72,99,3},{72,99,3},{71,89,19},{58,100,162},{63,90,2},{82,108,1},{82,108,1},{82,108,1},{81,95,1},{127,2,162},{73,99,0},{73,99,0},{64,90,1},{127,66,162},{64,90,1},{127,31,200},{81,110,0},{86,100,1},{71,100,1},{127,31,200},{127,80,200},{71,100,1},{0,95,202},{127,80,200},{0,95,202},{76,0,202},
{76,0,202},{76,0,202},{76,0,202},{72,101,1},{72,101,1},{72,101,1},{72,86,1},{63,90,1},{63,90,1},{82,127,468},{81,117,212},{82,106,308},{80,104,212},{77,127,753},{76,110,212},{76,104,4},{74,98,308},{58,113,723},{62,99,215},{86,127,201},{85,114,1},{87,105,19},{85,103,35},{127,30,723},{72,113,200},{76,104,3},{58,99,202},{127,79,723},{58,99,202},{81,122,202},{81,122,202},{81,122,202},
{81,101,202},{76,118,163},{76,103,3},{76,103,3},{75,93,19},{62,104,162},{66,95,1},{86,112,1},{86,112,1},{86,112,1},{85,99,1},{127,14,162},{77,103,0},{77,103,0},{68,94,1},{127,72,162},{68,94,1},{127,43,200},{85,114,0},{90,104,1},{75,104,1},{127,43,200},{127,86,200},{75,104,1},{0,99,202},{127,86,200},{0,99,202},{80,0,202},{80,0,202},{80,0,202},{80,0,202},{76,105,1},
{76,105,1},{76,105,1},{76,90,1},{66,95,0},{66,95,0},{86,127,546},{85,121,212},{86,110,308},{84,108,212},{82,127,788},{80,114,212},{80,108,4},{78,102,308},{62,117,723},{67,103,212},{91,127,211},{89,118,1},{91,109,19},{89,107,35},{127,42,723},{76,117,200},{80,108,3},{62,103,202},{127,85,723},{62,103,202},{85,126,202},{85,126,202},{85,126,202},{85,105,202},{80,122,163},{80,107,3},{80,107,3},
{79,97,19},{67,107,163},{70,99,1},{90,116,1},{90,116,1},{90,116,1},{89,103,1},{127,27,162},{81,107,0},{81,107,0},{72,98,1},{126,78,162},{72,98,1},{127,56,200},{89,118,0},{94,108,1},{79,108,1},{127,56,200},{127,92,200},{79,108,1},{0,103,202},{127,92,200},{0,103,202},{84,0,202},{84,0,202},{84,0,202},{84,0,202},{80,109,1},{80,109,1},{80,109,1},{80,94,1},{70,99,0},
{70,99,0},{91,127,632},{89,125,216},{90,114,312},{89,112,213},{88,127,852},{83,120,208},{84,113,2},{82,106,312},{66,121,724},{71,107,216},{97,127,244},{94,122,0},{95,113,20},{93,111,36},{127,56,723},{81,121,200},{84,113,2},{65,108,200},{127,92,723},{65,108,200},{89,127,209},{89,127,209},{89,127,209},{89,110,200},{84,126,164},{84,113,2},{84,113,2},{84,102,20},{71,112,163},{75,103,0},{94,121,0},
{94,121,0},{94,121,0},{94,107,1},{127,40,162},{86,111,0},{86,111,0},{75,103,0},{126,85,162},{75,103,0},{127,69,200},{94,122,0},{98,113,0},{82,113,0},{127,69,200},{126,99,200},{82,113,0},{0,108,200},{126,99,200},{0,108,200},{89,0,200},{89,0,200},{89,0,200},{89,0,200},{84,114,0},{84,114,0},{84,114,0},{84,99,0},{75,103,0},{75,103,0},{97,127,728},{94,127,228},{94,118,312},
{93,116,213},{94,127,932},{87,124,208},{88,117,2},{86,110,312},{70,125,724},{75,111,216},{101,127,286},{98,126,0},{99,117,20},{97,115,36},{127,67,723},{85,125,200},{88,117,2},{69,112,200},{127,98,723},{69,112,200},{94,127,224},{94,127,224},{94,127,224},{93,114,200},{89,127,174},{88,117,2},{88,117,2},{88,106,20},{75,116,163},{79,107,0},{98,125,0},{98,125,0},{98,125,0},{98,111,1},{127,53,162},
{90,115,0},{90,115,0},{79,107,0},{126,91,162},{79,107,0},{127,81,200},{98,126,0},{102,117,0},{86,117,0},{127,81,200},{126,105,200},{86,117,0},{0,112,200},{126,105,200},{0,112,200},{93,0,200},{93,0,200},{93,0,200},{93,0,200},{88,118,0},{88,118,0},{88,118,0},{88,103,0},{79,107,0},{79,107,0},{100,127,864},{99,127,304},{98,122,312},{97,120,213},{98,127,1043},{92,127,211},{92,121,2},
{90,114,312},{76,127,732},{79,115,216},{106,127,336},{102,127,20},{103,121,20},{101,119,36},{127,79,723},{90,127,208},{92,121,2},{73,116,200},{127,104,723},{73,116,200},{98,127,257},{98,127,257},{98,127,257},{97,118,200},{94,127,196},{92,121,2},{92,121,2},{92,110,20},{79,120,163},{83,111,0},{102,126,4},{102,126,4},{102,126,4},{102,115,1},{127,64,162},{94,119,0},{94,119,0},{83,111,0},{126,97,162},
{83,111,0},{127,93,200},{105,127,8},{106,121,0},{90,121,0},{127,93,200},{126,111,200},{90,121,0},{0,116,200},{126,111,200},{0,116,200},{97,0,200},{97,0,200},{97,0,200},{97,0,200},{92,122,0},{92,122,0},{92,122,0},{92,107,0},{83,111,0},{83,111,0},{106,127,992},{103,127,440},{102,126,312},{101,124,213},{103,127,1144},{96,127,272},{96,125,2},{94,118,312},{84,127,804},{83,119,216},{111,127,398},
{108,127,84},{107,125,20},{105,123,36},{127,92,723},{99,127,248},{96,125,2},{77,120,200},{126,110,723},{77,120,200},{103,127,296},{103,127,296},{103,127,296},{101,122,200},{100,127,244},{96,125,2},{96,125,2},{96,114,20},{83,124,163},{87,115,0},{107,127,13},{107,127,13},{107,127,13},{106,119,1},{127,76,162},{98,123,0},{98,123,0},{87,115,0},{126,103,162},{87,115,0},{127,105,200},{111,127,45},{110,125,0},
{94,125,0},{127,105,200},{126,117,200},{94,125,0},{0,120,200},{126,117,200},{0,120,200},{101,0,200},{101,0,200},{101,0,200},{101,0,200},{96,126,0},{96,126,0},{96,126,0},{96,111,0},{87,115,0},{87,115,0},{111,127,919},{108,127,515},{107,127,362},{105,127,203},{109,127,1027},{102,127,267},{101,127,9},{99,122,198},{93,127,703},{88,123,121},{115,127,338},{113,127,122},{112,127,34},{110,126,14},{127,102,546},
{108,127,202},{102,127,2},{85,123,113},{127,115,546},{85,123,113},{107,127,362},{107,127,362},{107,127,362},{106,126,202},{104,127,309},{101,127,9},{101,127,9},{100,118,19},{89,127,168},{91,120,1},{112,127,34},{112,127,34},{112,127,34},{110,124,1},{127,90,162},{104,126,2},{104,126,2},{93,119,1},{127,109,162},{93,119,1},{127,115,113},{119,127,41},{116,127,0},{104,127,0},{127,115,113},{127,121,113},{104,127,0},
{0,123,113},{127,121,113},{0,123,113},{105,0,202},{105,0,202},{105,0,202},{105,0,202},{101,127,5},{101,127,5},{101,127,5},{101,115,1},{91,120,0},{91,120,0},{114,127,751},{111,127,514},{111,127,414},{110,127,218},{114,127,814},{108,127,219},{107,127,65},{103,124,71},{102,127,539},{94,125,27},{120,127,206},{117,127,110},{117,127,61},{115,127,2},{127,110,333},{113,127,121},{110,127,20},{93,125,25},{126,119,333},
{93,125,25},{111,127,414},{111,127,414},{111,127,414},{110,127,218},{109,127,371},{107,127,65},{107,127,65},{104,122,19},{96,127,203},{95,124,1},{117,127,61},{117,127,61},{117,127,61},{115,127,2},{127,102,162},{110,127,20},{110,127,20},{97,123,1},{127,115,162},{97,123,1},{127,121,25},{122,127,9},{122,127,0},{116,127,0},{127,121,25},{127,124,25},{116,127,0},{0,125,25},{127,124,25},{0,125,25},{109,0,202},
{109,0,202},{109,0,202},{109,0,202},{106,127,18},{106,127,18},{106,127,18},{105,119,1},{95,124,0},{95,124,0},{117,127,636},{117,127,516},{117,127,467},{115,127,282},{117,127,643},{111,127,241},{111,127,141},{108,126,18},{108,127,434},{99,127,2},{123,127,131},{121,127,105},{121,127,89},{119,127,26},{127,118,193},{119,127,86},{116,127,53},{101,127,0},{126,123,193},{101,127,0},{117,127,467},{117,127,467},{117,127,467},
{115,127,282},{114,127,410},{111,127,141},{111,127,141},{108,126,14},{105,127,254},{99,127,2},{121,127,89},{121,127,89},{121,127,89},{119,127,26},{127,113,145},{116,127,53},{116,127,53},{101,127,0},{127,120,145},{101,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{113,0,202},{113,0,202},{113,0,202},{113,0,202},{111,127,41},
{111,127,41},{111,127,41},{109,123,1},{101,126,2},{101,126,2},{120,127,412},{120,127,356},{120,127,331},{118,127,238},{120,127,387},{116,127,183},{116,127,134},{113,127,1},{113,127,262},{108,127,18},{124,127,41},{124,127,29},{124,127,25},{122,127,10},{127,122,54},{122,127,22},{122,127,13},{113,127,0},{126,125,54},{113,127,0},{120,127,331},{120,127,331},{120,127,331},{118,127,238},{117,127,266},{116,127,134},{116,127,134},
{113,127,1},{110,127,161},{108,127,18},{124,127,25},{124,127,25},{124,127,25},{122,127,10},{127,119,41},{122,127,13},{122,127,13},{113,127,0},{127,123,41},{113,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{117,0,202},{117,0,202},{117,0,202},{117,0,202},{114,127,73},{114,127,73},{114,127,73},{113,127,1},{108,127,18},
{108,127,18},{0,41,421},{0,28,41},{0,21,1},{0,18,157},{0,27,925},{0,19,569},{0,17,285},{0,12,701},{0,12,1005},{0,10,751},{0,41,421},{0,28,41},{0,21,1},{0,18,157},{13,1,925},{0,19,569},{0,17,285},{0,12,701},{27,0,925},{0,12,701},{0,20,0},{0,20,0},{0,20,0},{0,10,0},{0,9,85},{0,8,29},{0,8,29},{0,4,50},{0,5,94},{0,4,59},{0,20,0},
{0,20,0},{0,20,0},{0,10,0},{5,0,85},{0,8,29},{0,8,29},{0,4,50},{9,0,85},{0,4,50},{20,0,421},{0,28,41},{0,21,1},{0,18,157},{20,0,421},{41,0,421},{0,18,157},{0,14,421},{41,0,421},{0,14,421},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,53,421},{0,37,9},{1,25,17},
{0,21,113},{0,36,1261},{0,22,677},{0,20,305},{0,15,881},{0,17,1383},{0,13,971},{0,53,421},{0,37,9},{1,25,13},{0,21,113},{18,0,1261},{0,22,677},{0,20,305},{0,15,881},{36,0,1261},{0,15,881},{0,32,0},{0,32,0},{0,32,0},{0,15,1},{0,15,221},{0,11,85},{0,11,85},{0,7,130},{0,8,246},{0,7,155},{0,32,0},{0,32,0},{0,32,0},{0,15,1},{8,0,221},
{0,11,85},{0,11,85},{0,7,130},{15,0,221},{0,7,130},{26,0,421},{0,37,9},{3,25,1},{0,21,113},{26,0,421},{53,0,421},{0,21,113},{0,18,421},{53,0,421},{0,18,421},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{1,63,426},{1,42,6},{2,29,66},{0,27,74},{0,45,1514},{0,28,706},{0,25,258},
{0,18,974},{0,19,1706},{0,18,1118},{2,61,422},{1,42,2},{3,30,38},{0,27,74},{22,0,1514},{0,28,706},{0,25,258},{0,18,974},{45,0,1514},{0,18,974},{1,42,5},{1,42,5},{1,42,5},{1,21,5},{0,24,338},{0,19,90},{0,19,90},{0,10,181},{0,11,389},{0,10,230},{2,40,1},{2,40,1},{2,40,1},{1,21,1},{12,0,338},{0,19,90},{0,19,90},{0,10,181},{24,0,338},
{0,10,181},{32,0,421},{1,42,1},{7,29,1},{0,27,65},{32,0,421},{65,0,421},{0,27,65},{0,22,421},{65,0,421},{0,22,421},{1,0,5},{1,0,5},{1,0,5},{1,0,5},{0,4,0},{0,4,0},{0,4,0},{0,2,0},{0,2,1},{0,2,1},{3,71,482},{3,48,62},{4,34,158},{2,31,117},{0,58,1514},{0,37,594},{0,31,122},{0,23,881},{0,25,1818},{0,21,1086},{6,65,422},
{5,46,2},{7,34,38},{4,31,74},{28,0,1514},{0,37,594},{0,31,122},{0,23,881},{58,0,1514},{0,23,881},{3,50,61},{3,50,61},{3,50,61},{3,26,61},{0,36,338},{0,25,34},{0,25,34},{0,15,130},{0,17,461},{0,14,226},{6,44,1},{6,44,1},{6,44,1},{5,25,1},{18,0,338},{0,25,34},{0,25,34},{0,15,130},{36,0,338},{0,15,130},{38,0,421},{5,46,1},{11,33,1},
{0,32,40},{38,0,421},{77,0,421},{0,32,40},{0,26,421},{77,0,421},{0,26,421},{3,0,61},{3,0,61},{3,0,61},{3,0,61},{0,16,0},{0,16,0},{0,16,0},{0,8,0},{0,5,25},{0,5,25},{5,80,621},{5,55,213},{7,40,357},{5,35,241},{0,71,1514},{0,43,497},{0,36,29},{0,27,761},{0,31,1989},{0,26,1079},{10,70,421},{10,50,1},{11,38,41},{8,35,77},{35,0,1514},
{0,43,497},{0,36,29},{0,27,761},{71,0,1514},{0,27,761},{5,60,200},{5,60,200},{5,60,200},{5,32,200},{0,50,338},{0,34,4},{0,34,4},{0,21,80},{0,22,552},{0,19,244},{10,49,0},{10,49,0},{10,49,0},{10,29,0},{24,1,338},{0,34,4},{0,34,4},{0,21,80},{50,0,338},{0,21,80},{45,0,421},{10,50,1},{16,37,1},{0,36,13},{45,0,421},{91,0,421},{0,36,13},
{0,30,421},{91,0,421},{0,30,421},{5,0,200},{5,0,200},{5,0,200},{5,0,200},{0,30,0},{0,30,0},{0,30,0},{0,15,0},{0,11,74},{0,11,74},{7,89,813},{7,60,421},{10,43,617},{7,40,417},{0,83,1514},{0,51,446},{0,41,3},{0,32,686},{0,37,2165},{0,31,1101},{14,74,421},{14,54,1},{15,42,41},{12,39,77},{41,0,1514},{0,51,446},{0,41,3},{0,32,686},{83,0,1514},
{0,32,686},{7,67,392},{7,67,392},{7,67,392},{7,37,392},{0,62,338},{0,41,2},{0,41,2},{0,24,52},{0,28,680},{0,23,298},{14,53,0},{14,53,0},{14,53,0},{14,33,0},{30,1,338},{2,39,0},{2,39,0},{0,24,52},{62,0,338},{0,24,52},{51,0,421},{14,54,1},{20,41,1},{0,41,2},{51,0,421},{103,0,421},{0,41,2},{0,34,421},{103,0,421},{0,34,421},{7,0,392},
{7,0,392},{7,0,392},{7,0,392},{0,42,0},{0,42,0},{0,42,0},{0,21,0},{0,17,146},{0,17,146},{11,93,842},{11,65,453},{12,48,654},{11,44,446},{4,87,1515},{3,54,435},{4,45,4},{0,36,654},{0,43,2030},{0,35,882},{18,78,421},{18,58,1},{19,46,41},{16,43,77},{47,0,1514},{0,57,422},{4,45,3},{0,36,605},{95,0,1514},{0,36,605},{11,71,421},{11,71,421},{11,71,421},
{11,41,421},{4,66,339},{4,45,3},{4,45,3},{2,29,41},{0,34,581},{0,28,161},{18,57,0},{18,57,0},{18,57,0},{18,37,0},{36,1,338},{6,43,0},{6,43,0},{0,30,20},{74,0,338},{0,30,20},{57,0,421},{18,58,1},{24,45,1},{2,45,1},{57,0,421},{115,0,421},{2,45,1},{0,38,421},{115,0,421},{0,38,421},{11,0,421},{11,0,421},{11,0,421},{11,0,421},{4,46,1},
{4,46,1},{4,46,1},{3,25,1},{0,22,85},{0,22,85},{15,97,842},{16,67,453},{16,52,654},{15,48,446},{8,91,1515},{7,58,435},{8,49,4},{4,40,654},{0,51,1850},{0,40,682},{22,82,421},{22,62,1},{23,50,41},{20,47,77},{53,0,1514},{3,61,421},{8,49,3},{0,41,554},{107,0,1514},{0,41,554},{15,75,421},{15,75,421},{15,75,421},{15,45,421},{8,70,339},{8,49,3},{8,49,3},
{6,33,41},{0,40,477},{0,34,57},{22,61,0},{22,61,0},{22,61,0},{22,41,0},{42,1,338},{10,47,0},{10,47,0},{0,34,8},{86,0,338},{0,34,8},{63,0,421},{22,62,1},{28,49,1},{6,49,1},{63,0,421},{127,0,421},{6,49,1},{0,42,421},{127,0,421},{0,42,421},{15,0,421},{15,0,421},{15,0,421},{15,0,421},{8,50,1},{8,50,1},{8,50,1},{7,29,1},{0,31,29},
{0,31,29},{19,102,842},{20,71,451},{21,57,650},{19,52,446},{12,97,1514},{12,63,437},{12,54,2},{9,45,650},{0,57,1710},{0,45,530},{27,86,422},{26,67,1},{28,55,38},{25,52,74},{59,1,1514},{7,66,421},{12,54,2},{0,45,494},{121,0,1514},{0,45,494},{19,81,421},{19,81,421},{19,81,421},{19,49,422},{12,75,338},{12,54,2},{12,54,2},{11,38,38},{0,49,402},{0,39,6},{27,64,1},
{27,64,1},{27,64,1},{26,46,1},{49,0,338},{14,52,0},{14,52,0},{1,39,1},{100,0,338},{1,39,1},{69,0,421},{26,67,0},{32,54,1},{10,54,0},{69,0,421},{127,7,421},{10,54,0},{0,47,421},{127,7,421},{0,47,421},{19,0,421},{19,0,421},{19,0,421},{19,0,421},{12,56,0},{12,56,0},{12,56,0},{12,33,1},{0,38,5},{0,38,5},{23,106,842},{24,75,451},{25,61,650},
{23,56,446},{16,101,1514},{15,67,434},{16,58,2},{13,49,650},{0,64,1617},{0,50,462},{31,90,422},{30,71,1},{32,59,38},{29,56,74},{65,0,1514},{11,70,421},{16,58,2},{0,50,461},{127,3,1514},{0,50,461},{23,85,421},{23,85,421},{23,85,421},{23,53,422},{16,79,338},{16,58,2},{16,58,2},{15,42,38},{0,54,349},{3,43,2},{31,68,1},{31,68,1},{31,68,1},{30,50,1},{55,0,338},
{18,56,0},{18,56,0},{5,43,1},{112,0,338},{5,43,1},{75,0,421},{30,71,0},{36,58,1},{14,58,0},{75,0,421},{127,13,421},{14,58,0},{0,51,421},{127,13,421},{0,51,421},{23,0,421},{23,0,421},{23,0,421},{23,0,421},{16,60,0},{16,60,0},{16,60,0},{16,37,1},{3,43,1},{3,43,1},{27,110,842},{28,79,451},{30,64,650},{27,60,446},{20,105,1514},{19,71,434},{20,62,2},
{17,53,650},{0,70,1553},{1,54,450},{35,94,422},{34,75,1},{36,63,38},{33,60,74},{71,0,1514},{15,74,421},{20,62,2},{0,54,441},{127,9,1514},{0,54,441},{27,89,421},{27,89,421},{27,89,421},{27,57,422},{20,83,338},{20,62,2},{20,62,2},{19,46,38},{0,61,338},{7,47,2},{35,72,1},{35,72,1},{35,72,1},{34,54,1},{61,0,338},{22,60,0},{22,60,0},{9,47,1},{124,0,338},
{9,47,1},{81,0,421},{34,75,0},{40,62,1},{18,62,0},{81,0,421},{126,19,421},{18,62,0},{0,55,421},{126,19,421},{0,55,421},{27,0,421},{27,0,421},{27,0,421},{27,0,421},{20,64,0},{20,64,0},{20,64,0},{20,41,1},{7,47,1},{7,47,1},{31,114,842},{32,83,451},{33,67,650},{31,64,446},{24,109,1514},{23,75,434},{24,65,6},{21,57,650},{0,76,1521},{5,58,450},{39,98,422},
{38,79,1},{41,66,42},{37,63,81},{77,0,1514},{19,78,421},{25,65,2},{0,59,425},{127,15,1514},{0,59,425},{31,93,421},{31,93,421},{31,93,421},{31,61,422},{24,87,338},{24,64,5},{24,64,5},{23,50,38},{5,64,339},{11,51,2},{39,76,1},{39,76,1},{39,76,1},{38,58,1},{66,1,338},{25,64,1},{25,64,1},{13,51,1},{126,5,338},{13,51,1},{87,0,421},{38,79,0},{45,65,1},
{23,65,1},{87,0,421},{126,25,421},{23,65,1},{0,59,421},{126,25,421},{0,59,421},{31,0,421},{31,0,421},{31,0,421},{31,0,421},{24,67,0},{24,67,0},{24,67,0},{24,45,1},{11,51,1},{11,51,1},{36,118,842},{37,88,453},{38,72,646},{35,69,445},{29,112,1515},{27,79,438},{29,69,6},{25,61,654},{3,82,1515},{10,63,450},{43,103,421},{43,83,2},{45,70,41},{42,67,76},{83,1,1514},
{24,82,421},{29,69,5},{2,63,421},{126,22,1514},{2,63,421},{36,96,421},{36,96,421},{36,96,421},{36,65,421},{29,91,339},{29,69,2},{29,69,2},{27,54,41},{10,68,339},{15,56,1},{43,82,0},{43,82,0},{43,82,0},{43,62,0},{73,0,338},{31,67,1},{31,67,1},{16,56,0},{126,12,338},{16,56,0},{93,0,421},{42,84,0},{49,70,0},{26,70,0},{93,0,421},{126,32,421},{26,70,0},
{0,63,421},{126,32,421},{0,63,421},{36,0,421},{36,0,421},{36,0,421},{36,0,421},{29,71,1},{29,71,1},{29,71,1},{28,50,1},{15,56,1},{15,56,1},{40,122,842},{41,92,453},{42,76,646},{39,73,445},{33,116,1515},{31,83,438},{33,73,6},{30,64,650},{7,86,1515},{15,66,451},{47,107,421},{47,87,2},{49,74,41},{46,71,76},{89,1,1514},{28,86,421},{33,73,5},{5,67,421},{126,28,1514},
{5,67,421},{40,100,421},{40,100,421},{40,100,421},{40,69,421},{33,95,339},{33,73,2},{33,73,2},{31,58,41},{14,72,339},{19,60,1},{47,86,0},{47,86,0},{47,86,0},{47,66,0},{79,0,338},{35,71,1},{35,71,1},{20,60,0},{127,17,338},{20,60,0},{99,0,421},{46,88,0},{53,74,0},{30,74,0},{99,0,421},{126,38,421},{30,74,0},{0,67,421},{126,38,421},{0,67,421},{40,0,421},
{40,0,421},{40,0,421},{40,0,421},{33,75,1},{33,75,1},{33,75,1},{32,54,1},{19,60,1},{19,60,1},{44,126,842},{45,96,453},{46,80,646},{43,77,445},{37,120,1515},{35,87,438},{37,77,6},{34,68,650},{11,90,1515},{19,70,451},{51,111,421},{51,91,2},{53,78,41},{50,75,76},{95,1,1514},{32,90,421},{37,77,5},{9,71,421},{126,34,1514},{9,71,421},{44,104,421},{44,104,421},{44,104,421},
{44,73,421},{37,99,339},{37,77,2},{37,77,2},{35,62,41},{18,76,339},{23,64,1},{51,90,0},{51,90,0},{51,90,0},{51,70,0},{85,0,338},{39,75,1},{39,75,1},{23,64,1},{127,23,338},{23,64,1},{105,0,421},{50,92,0},{57,78,0},{34,78,0},{105,0,421},{126,44,421},{34,78,0},{0,71,421},{126,44,421},{0,71,421},{44,0,421},{44,0,421},{44,0,421},{44,0,421},{37,79,1},
{37,79,1},{37,79,1},{36,58,1},{23,64,0},{23,64,0},{48,126,854},{49,100,453},{50,84,646},{47,81,445},{41,124,1515},{39,91,438},{41,81,6},{38,72,650},{15,94,1515},{23,74,451},{55,115,421},{55,95,2},{57,82,41},{54,79,76},{101,1,1514},{36,94,421},{41,81,5},{13,75,421},{126,40,1514},{13,75,421},{48,108,421},{48,108,421},{48,108,421},{48,77,421},{41,103,339},{41,81,2},{41,81,2},
{40,65,42},{22,80,339},{27,68,1},{55,94,0},{55,94,0},{55,94,0},{55,74,0},{91,0,338},{43,79,1},{43,79,1},{29,67,1},{127,29,338},{29,67,1},{111,0,421},{54,96,0},{61,82,0},{38,82,0},{111,0,421},{127,49,421},{38,82,0},{0,75,421},{127,49,421},{0,75,421},{48,0,421},{48,0,421},{48,0,421},{48,0,421},{41,83,1},{41,83,1},{41,83,1},{40,62,1},{27,68,0},
{27,68,0},{53,127,886},{53,104,451},{54,88,650},{52,85,446},{46,126,1521},{44,96,434},{45,86,6},{43,77,646},{18,99,1515},{27,78,453},{60,119,422},{59,100,1},{62,87,42},{59,84,75},{108,0,1514},{40,99,421},{46,86,2},{18,79,421},{125,47,1514},{18,79,421},{52,114,421},{52,114,421},{52,114,421},{52,82,421},{45,108,338},{45,85,5},{45,85,5},{45,70,41},{26,85,339},{31,72,2},{60,97,1},
{60,97,1},{60,97,1},{60,78,1},{98,0,338},{48,83,1},{48,83,1},{33,72,0},{127,36,338},{33,72,0},{118,0,421},{59,100,0},{65,86,1},{44,86,1},{118,0,421},{127,56,421},{44,86,1},{0,79,421},{127,56,421},{0,79,421},{52,0,421},{52,0,421},{52,0,421},{52,0,421},{45,88,0},{45,88,0},{45,88,0},{45,66,0},{31,73,0},{31,73,0},{58,127,926},{57,108,451},{58,92,650},
{56,89,446},{51,126,1542},{48,100,434},{49,90,6},{47,81,646},{22,103,1515},{31,82,453},{63,125,422},{63,104,1},{64,91,41},{63,88,75},{114,0,1514},{44,103,421},{50,90,2},{22,83,421},{127,52,1514},{22,83,421},{56,118,421},{56,118,421},{56,118,421},{56,86,421},{49,112,338},{49,89,5},{49,89,5},{49,74,41},{30,89,339},{35,76,2},{63,103,1},{63,103,1},{63,103,1},{63,82,2},{104,0,338},
{52,87,1},{52,87,1},{37,76,0},{127,42,338},{37,76,0},{124,0,421},{63,104,0},{69,90,1},{48,90,1},{124,0,421},{127,62,421},{48,90,1},{0,83,421},{127,62,421},{0,83,421},{56,0,421},{56,0,421},{56,0,421},{56,0,421},{49,92,0},{49,92,0},{49,92,0},{49,70,0},{35,77,0},{35,77,0},{62,127,1010},{61,112,451},{62,96,650},{60,93,446},{56,127,1577},{52,104,434},{53,94,6},
{51,85,646},{26,107,1515},{35,86,453},{67,126,425},{67,107,1},{68,95,41},{65,92,77},{120,0,1514},{48,107,421},{54,94,2},{26,87,421},{127,58,1514},{26,87,421},{60,122,421},{60,122,421},{60,122,421},{60,90,421},{53,116,338},{53,93,5},{53,93,5},{53,78,41},{34,93,339},{39,80,2},{67,106,0},{67,106,0},{67,106,0},{67,86,0},{110,0,338},{56,91,1},{56,91,1},{41,80,0},{126,48,338},
{41,80,0},{127,5,421},{67,107,1},{73,94,1},{52,94,1},{127,5,421},{126,68,421},{52,94,1},{0,87,421},{126,68,421},{0,87,421},{60,0,421},{60,0,421},{60,0,421},{60,0,421},{53,96,0},{53,96,0},{53,96,0},{53,74,0},{39,81,0},{39,81,0},{68,127,1098},{64,117,450},{65,101,654},{64,97,446},{61,127,1626},{56,108,434},{57,98,6},{55,89,646},{30,111,1515},{39,90,453},{72,127,437},
{71,111,1},{72,99,41},{69,96,77},{126,0,1514},{52,111,421},{58,98,2},{30,91,421},{126,64,1514},{30,91,421},{64,125,421},{64,125,421},{64,125,421},{64,94,421},{57,120,338},{57,97,5},{57,97,5},{57,82,41},{38,97,339},{43,84,2},{71,110,0},{71,110,0},{71,110,0},{71,90,0},{115,1,338},{60,95,1},{60,95,1},{45,84,0},{126,54,338},{45,84,0},{127,18,421},{71,111,1},{77,98,1},
{56,98,1},{127,18,421},{126,74,421},{56,98,1},{0,91,421},{126,74,421},{0,91,421},{64,0,421},{64,0,421},{64,0,421},{64,0,421},{57,100,0},{57,100,0},{57,100,0},{57,78,0},{43,85,0},{43,85,0},{71,127,1214},{69,122,450},{70,106,650},{68,101,446},{66,127,1722},{60,112,438},{62,102,6},{59,93,650},{36,115,1515},{44,95,451},{78,126,470},{75,116,2},{77,104,38},{74,101,74},{127,11,1514},
{57,115,421},{62,102,5},{34,96,421},{126,71,1514},{34,96,421},{68,127,425},{68,127,425},{68,127,425},{68,98,422},{62,124,339},{62,102,2},{62,102,2},{61,86,42},{43,101,339},{48,89,1},{76,114,1},{76,114,1},{76,114,1},{75,95,1},{122,0,338},{63,101,1},{63,101,1},{50,88,1},{126,61,338},{50,88,1},{127,31,421},{75,116,1},{81,103,1},{59,103,0},{127,31,421},{127,80,421},{59,103,0},
{0,96,421},{127,80,421},{0,96,421},{68,0,421},{68,0,421},{68,0,421},{68,0,421},{62,104,1},{62,104,1},{62,104,1},{62,82,1},{48,89,0},{48,89,0},{77,127,1334},{73,126,450},{74,110,650},{72,105,446},{71,127,1805},{65,116,437},{65,107,2},{63,97,650},{40,119,1515},{48,99,451},{82,127,506},{79,120,2},{81,108,38},{78,105,74},{127,24,1514},{61,119,421},{65,107,2},{38,100,421},{125,77,1514},
{38,100,421},{73,126,441},{73,126,441},{73,126,441},{72,102,422},{65,127,340},{65,107,2},{65,107,2},{64,91,38},{47,105,339},{52,93,1},{80,118,1},{80,118,1},{80,118,1},{79,99,1},{127,2,338},{67,105,0},{67,105,0},{54,92,1},{127,66,338},{54,92,1},{127,43,421},{79,120,1},{85,107,1},{63,107,0},{127,43,421},{127,86,421},{63,107,0},{0,100,421},{127,86,421},{0,100,421},{72,0,421},
{72,0,421},{72,0,421},{72,0,421},{65,109,0},{65,109,0},{65,109,0},{65,86,1},{52,93,0},{52,93,0},{82,127,1470},{77,127,462},{78,114,650},{76,109,446},{77,127,1917},{69,120,437},{69,111,2},{66,102,650},{44,123,1515},{52,103,451},{88,127,562},{83,124,2},{85,112,38},{82,109,74},{127,36,1514},{65,122,421},{69,111,2},{42,104,421},{127,82,1514},{42,104,421},{77,127,461},{77,127,461},{77,127,461},
{76,106,422},{71,127,356},{69,111,2},{69,111,2},{68,95,38},{51,109,339},{56,97,1},{84,122,1},{84,122,1},{84,122,1},{83,103,1},{127,14,338},{71,109,0},{71,109,0},{58,96,1},{127,72,338},{58,96,1},{127,56,421},{83,124,1},{89,111,1},{67,111,0},{127,56,421},{127,92,421},{67,111,0},{0,104,421},{127,92,421},{0,104,421},{76,0,421},{76,0,421},{76,0,421},{76,0,421},{69,113,0},
{69,113,0},{69,113,0},{69,90,1},{56,97,0},{56,97,0},{85,127,1634},{82,127,530},{82,118,650},{80,113,446},{82,127,2030},{73,124,437},{73,115,2},{70,106,650},{48,127,1515},{56,107,451},{92,127,646},{87,127,6},{89,116,38},{86,113,74},{127,48,1514},{69,126,421},{73,115,2},{46,108,421},{127,88,1514},{46,108,421},{82,127,494},{82,127,494},{82,127,494},{80,110,422},{75,127,386},{73,115,2},{73,115,2},
{72,99,38},{55,113,339},{60,101,1},{88,126,1},{88,126,1},{88,126,1},{87,107,1},{127,27,338},{75,113,0},{75,113,0},{62,100,1},{126,78,338},{62,100,1},{127,67,421},{89,126,5},{93,115,1},{71,115,0},{127,67,421},{127,98,421},{71,115,0},{0,108,421},{127,98,421},{0,108,421},{80,0,421},{80,0,421},{80,0,421},{80,0,421},{73,117,0},{73,117,0},{73,117,0},{73,94,1},{60,101,0},
{60,101,0},{91,127,1838},{87,127,682},{86,122,654},{85,118,446},{88,127,2198},{78,127,450},{78,119,4},{74,110,654},{57,127,1557},{60,111,453},{97,127,741},{93,127,57},{93,120,41},{90,117,77},{127,62,1514},{76,127,446},{78,119,3},{51,112,421},{127,95,1514},{51,112,421},{86,127,554},{86,127,554},{86,127,554},{85,115,421},{80,127,435},{78,119,3},{78,119,3},{76,103,41},{59,118,339},{64,105,1},{93,126,8},
{93,126,8},{93,126,8},{92,111,0},{127,40,338},{80,117,0},{80,117,0},{65,105,0},{126,85,338},{65,105,0},{127,81,421},{96,127,29},{98,119,1},{76,119,1},{127,81,421},{126,105,421},{76,119,1},{0,112,421},{126,105,421},{0,112,421},{85,0,421},{85,0,421},{85,0,421},{85,0,421},{78,120,1},{78,120,1},{78,120,1},{77,99,1},{64,105,1},{64,105,1},{97,127,2070},{92,127,882},{90,126,654},
{89,122,446},{91,127,2382},{84,127,546},{82,123,4},{78,114,654},{64,127,1658},{62,116,453},{103,127,837},{99,127,161},{97,124,41},{94,121,77},{127,73,1514},{84,127,521},{82,123,3},{55,116,421},{127,101,1514},{55,116,421},{91,127,605},{91,127,605},{91,127,605},{89,119,421},{85,127,485},{82,123,3},{82,123,3},{80,107,41},{63,122,339},{68,109,1},{97,127,20},{97,127,20},{97,127,20},{96,115,0},{127,53,338},
{84,121,0},{84,121,0},{69,109,0},{126,91,338},{69,109,0},{127,93,421},{102,127,85},{102,123,1},{80,123,1},{127,93,421},{126,111,421},{80,123,1},{0,116,421},{126,111,421},{0,116,421},{89,0,421},{89,0,421},{89,0,421},{89,0,421},{82,124,1},{82,124,1},{82,124,1},{81,103,1},{68,109,1},{68,109,1},{100,127,2201},{96,127,1101},{95,127,686},{93,126,441},{97,127,2469},{87,127,689},{86,127,3},
{84,117,617},{73,127,1735},{67,120,421},{109,127,916},{104,127,298},{102,127,52},{98,125,68},{127,86,1459},{93,127,584},{86,127,2},{59,120,392},{127,107,1459},{59,120,392},{95,127,686},{95,127,686},{95,127,686},{93,123,421},{91,127,557},{86,127,3},{86,127,3},{84,111,41},{66,126,338},{72,113,1},{102,126,52},{102,126,52},{102,126,52},{100,119,0},{127,64,338},{88,125,0},{88,125,0},{73,113,0},{126,97,338},
{73,113,0},{127,105,392},{110,127,146},{106,127,0},{84,127,0},{127,105,392},{126,117,392},{84,127,0},{0,120,392},{126,117,392},{0,120,392},{93,0,421},{93,0,421},{93,0,421},{93,0,421},{86,127,2},{86,127,2},{86,127,2},{85,107,1},{72,113,1},{72,113,1},{103,127,1901},{101,127,1079},{100,127,761},{97,127,421},{100,127,2093},{93,127,537},{91,127,29},{87,120,357},{81,127,1425},{72,122,213},{111,127,638},
{108,127,244},{106,127,80},{103,126,17},{127,93,1064},{99,127,392},{93,127,4},{67,122,200},{126,111,1064},{67,122,200},{100,127,761},{100,127,761},{100,127,761},{97,127,421},{95,127,659},{91,127,29},{91,127,29},{88,115,41},{73,127,350},{76,117,1},{106,127,80},{106,127,80},{106,127,80},{104,123,0},{127,76,338},{93,127,4},{93,127,4},{77,117,0},{126,103,338},{77,117,0},{127,111,200},{116,127,74},{112,127,0},
{96,127,0},{127,111,200},{126,120,200},{96,127,0},{0,122,200},{126,120,200},{0,122,200},{97,0,421},{97,0,421},{97,0,421},{97,0,421},{91,127,13},{91,127,13},{91,127,13},{89,111,1},{76,117,1},{76,117,1},{109,127,1646},{106,127,1086},{104,127,881},{102,127,450},{106,127,1730},{99,127,474},{96,127,122},{93,123,158},{87,127,1166},{79,124,62},{114,127,446},{111,127,225},{111,127,125},{108,127,2},{127,102,722},
{105,127,254},{102,127,34},{76,124,61},{127,115,722},{76,124,61},{104,127,881},{104,127,881},{104,127,881},{102,127,450},{100,127,770},{96,127,122},{96,127,122},{93,120,38},{81,127,426},{81,121,2},{111,127,125},{111,127,125},{111,127,125},{108,127,2},{127,90,338},{102,127,34},{102,127,34},{83,121,1},{127,109,338},{83,121,1},{127,118,61},{119,127,25},{119,127,0},{110,127,0},{127,118,61},{126,123,61},{110,127,0},
{0,124,61},{126,123,61},{0,124,61},{101,0,421},{101,0,421},{101,0,421},{101,0,421},{95,127,40},{95,127,40},{95,127,40},{94,115,1},{81,121,1},{81,121,1},{111,127,1450},{111,127,1109},{109,127,974},{105,127,542},{111,127,1505},{102,127,514},{102,127,258},{98,125,66},{96,127,1026},{84,126,6},{120,127,342},{117,127,230},{117,127,181},{113,127,37},{127,110,509},{111,127,213},{108,127,90},{85,126,5},{126,119,509},
{85,126,5},{109,127,974},{109,127,974},{109,127,974},{105,127,542},{106,127,882},{102,127,258},{102,127,258},{97,124,38},{90,127,530},{85,125,2},{117,127,181},{117,127,181},{117,127,181},{113,127,37},{127,102,338},{108,127,90},{108,127,90},{87,125,1},{127,115,338},{87,125,1},{127,124,5},{125,127,1},{125,127,0},{122,127,0},{127,124,5},{126,126,5},{122,127,0},{0,126,5},{126,126,5},{0,126,5},{105,0,421},
{105,0,421},{105,0,421},{105,0,421},{100,127,65},{100,127,65},{100,127,65},{98,119,1},{85,125,1},{85,125,1},{114,127,1179},{114,127,971},{112,127,881},{111,127,545},{114,127,1170},{108,127,457},{107,127,305},{102,126,17},{102,127,793},{90,127,9},{121,127,209},{120,127,155},{120,127,130},{118,127,45},{127,116,294},{116,127,134},{113,127,85},{95,127,0},{126,122,294},{95,127,0},{112,127,881},{112,127,881},{112,127,881},
{111,127,545},{111,127,765},{107,127,305},{107,127,305},{102,126,13},{96,127,465},{90,127,9},{120,127,130},{120,127,130},{120,127,130},{118,127,45},{127,110,221},{113,127,85},{113,127,85},{95,127,0},{126,119,221},{95,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{109,0,421},{109,0,421},{109,0,421},{109,0,421},{106,127,113},
{106,127,113},{106,127,113},{102,123,1},{90,127,9},{90,127,9},{117,127,871},{117,127,751},{115,127,701},{113,127,502},{117,127,822},{111,127,364},{111,127,264},{106,127,1},{105,127,537},{99,127,41},{123,127,75},{123,127,59},{123,127,50},{121,127,17},{127,121,113},{119,127,54},{119,127,29},{107,127,0},{127,124,113},{107,127,0},{115,127,701},{115,127,701},{115,127,701},{113,127,502},{114,127,561},{111,127,264},{111,127,264},
{106,127,1},{102,127,329},{99,127,41},{123,127,50},{123,127,50},{123,127,50},{121,127,17},{127,116,85},{119,127,29},{119,127,29},{107,127,0},{126,122,85},{107,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{113,0,421},{113,0,421},{113,0,421},{113,0,421},{109,127,157},{109,127,157},{109,127,157},{106,127,1},{99,127,41},
{99,127,41},{0,59,882},{0,40,100},{0,30,4},{0,24,340},{0,39,1896},{0,25,1188},{0,23,590},{0,15,1444},{0,19,2040},{0,15,1544},{0,59,882},{0,40,100},{0,30,4},{0,24,340},{19,0,1896},{0,25,1188},{0,23,590},{0,15,1444},{39,0,1896},{0,15,1444},{0,27,0},{0,27,0},{0,27,0},{0,13,1},{0,13,162},{0,11,58},{0,11,58},{0,7,97},{0,5,179},{0,6,116},{0,27,0},
{0,27,0},{0,27,0},{0,13,1},{7,0,162},{0,11,58},{0,11,58},{0,7,97},{13,0,162},{0,7,97},{29,0,882},{0,40,100},{0,30,4},{0,24,340},{29,0,882},{59,0,882},{0,24,340},{0,19,884},{59,0,882},{0,19,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,71,882},{0,49,40},{1,34,12},
{0,30,260},{0,47,2355},{0,31,1332},{0,28,612},{0,18,1700},{0,22,2556},{0,18,1844},{0,71,882},{0,49,40},{1,34,8},{0,30,260},{23,0,2355},{0,31,1332},{0,28,612},{0,18,1700},{47,0,2355},{0,18,1700},{0,39,0},{0,39,0},{0,39,0},{0,19,0},{0,20,338},{0,14,130},{0,14,130},{0,10,205},{0,8,371},{0,9,244},{0,39,0},{0,39,0},{0,39,0},{0,19,0},{10,0,338},
{0,14,130},{0,14,130},{0,10,205},{20,0,338},{0,10,205},{35,0,882},{0,49,40},{3,34,0},{0,30,260},{35,0,882},{71,0,882},{0,30,260},{0,23,884},{71,0,882},{0,23,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,83,882},{0,55,8},{1,40,60},{0,33,200},{0,56,2899},{0,37,1508},{0,31,660},
{0,21,2004},{0,25,3176},{0,21,2200},{0,83,882},{0,55,8},{2,38,52},{0,33,200},{27,1,2899},{0,37,1508},{0,31,660},{0,21,2004},{56,0,2899},{0,21,2004},{0,52,0},{0,52,0},{0,52,0},{0,25,0},{0,26,578},{0,19,208},{0,19,208},{0,13,353},{0,11,635},{0,10,414},{0,52,0},{0,52,0},{0,52,0},{0,25,0},{13,0,578},{0,19,208},{0,19,208},{0,13,353},{26,0,578},
{0,13,353},{41,0,882},{0,55,8},{7,38,0},{0,33,200},{41,0,882},{83,0,882},{0,33,200},{0,27,884},{83,0,882},{0,27,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,91,902},{1,62,20},{3,44,133},{0,39,172},{0,66,3048},{0,43,1416},{0,37,488},{0,27,1944},{0,31,3460},{0,26,2254},{3,89,882},
{3,60,2},{5,44,68},{2,38,168},{33,0,3048},{0,43,1416},{0,37,488},{0,27,1944},{66,0,3048},{0,27,1944},{2,60,20},{2,60,20},{2,60,20},{2,30,20},{0,36,648},{0,28,160},{0,28,160},{0,15,340},{0,17,771},{0,15,440},{3,58,0},{3,58,0},{3,58,0},{3,30,0},{18,0,648},{0,28,160},{0,28,160},{0,15,340},{36,0,648},{0,15,340},{47,0,882},{1,62,0},{11,42,0},
{0,39,136},{47,0,882},{95,0,882},{0,39,136},{0,31,884},{95,0,882},{0,31,884},{1,0,20},{1,0,20},{1,0,20},{1,0,20},{0,9,0},{0,9,0},{0,9,0},{0,4,1},{0,3,8},{0,3,8},{4,101,995},{4,67,114},{6,48,283},{3,44,242},{0,80,3048},{0,51,1224},{0,43,267},{0,32,1794},{0,37,3651},{0,30,2195},{8,93,883},{7,65,2},{9,48,67},{6,42,171},{39,1,3048},
{0,51,1224},{0,43,267},{0,32,1794},{80,0,3048},{0,32,1794},{4,69,113},{4,69,113},{4,69,113},{4,36,113},{0,50,648},{0,34,74},{0,34,74},{0,21,250},{0,22,862},{0,19,434},{8,61,1},{8,61,1},{8,61,1},{8,34,1},{24,1,648},{0,34,74},{0,34,74},{0,21,250},{50,0,648},{0,21,250},{53,1,882},{6,65,1},{15,47,1},{0,44,89},{53,1,882},{109,0,882},{0,44,89},
{0,36,882},{109,0,882},{0,36,882},{4,0,113},{4,0,113},{4,0,113},{4,0,113},{0,23,0},{0,23,0},{0,23,0},{0,11,0},{0,8,41},{0,8,41},{6,109,1147},{6,72,274},{8,53,499},{5,47,367},{0,92,3048},{0,57,1096},{0,47,129},{0,36,1635},{0,43,3859},{0,34,2183},{12,97,883},{11,69,2},{13,52,67},{10,46,171},{45,1,3048},{0,57,1096},{0,47,129},{0,36,1635},{92,0,3048},
{0,36,1635},{6,77,265},{6,77,265},{6,77,265},{6,41,265},{0,62,648},{0,43,26},{0,43,26},{0,26,193},{0,28,990},{0,23,458},{12,65,1},{12,65,1},{12,65,1},{12,38,1},{30,1,648},{0,43,26},{0,43,26},{0,26,193},{62,0,648},{0,26,193},{59,1,882},{10,69,1},{19,51,1},{0,49,58},{59,1,882},{121,0,882},{0,49,58},{0,40,882},{121,0,882},{0,40,882},{6,0,265},
{6,0,265},{6,0,265},{6,0,265},{0,35,0},{0,35,0},{0,35,0},{0,17,0},{0,14,97},{0,14,97},{8,117,1363},{8,77,506},{10,58,795},{7,52,559},{0,104,3048},{0,63,1000},{0,53,41},{0,39,1523},{0,46,4095},{0,38,2195},{16,101,883},{15,73,2},{17,56,67},{14,50,171},{51,0,3048},{0,63,1000},{0,53,41},{0,39,1523},{104,0,3048},{0,39,1523},{8,85,481},{8,85,481},{8,85,481},
{8,46,481},{0,74,648},{0,49,2},{0,49,2},{0,30,130},{0,31,1146},{0,28,494},{16,69,1},{16,69,1},{16,69,1},{16,42,1},{36,1,648},{0,49,2},{0,49,2},{0,30,130},{74,0,648},{0,30,130},{65,0,882},{14,73,1},{23,55,1},{0,53,25},{65,0,882},{127,3,882},{0,53,25},{0,44,882},{127,3,882},{0,44,882},{8,0,481},{8,0,481},{8,0,481},{8,0,481},{0,47,0},
{0,47,0},{0,47,0},{0,23,0},{0,19,169},{0,19,169},{10,125,1643},{11,83,802},{13,61,1159},{9,56,826},{0,117,3048},{0,70,933},{0,58,11},{0,45,1395},{0,54,4321},{0,43,2227},{20,105,883},{19,77,2},{21,60,67},{18,54,171},{57,0,3048},{0,70,933},{0,58,11},{0,45,1395},{117,0,3048},{0,45,1395},{10,93,761},{10,93,761},{10,93,761},{10,51,761},{0,86,648},{0,56,4},{0,56,4},
{0,35,89},{0,37,1322},{0,31,578},{20,73,1},{20,73,1},{20,73,1},{20,46,1},{42,1,648},{3,54,0},{3,54,0},{0,35,89},{86,0,648},{0,35,89},{71,0,882},{18,77,1},{27,59,1},{0,58,10},{71,0,882},{127,9,882},{0,58,10},{0,48,882},{127,9,882},{0,48,882},{10,0,761},{10,0,761},{10,0,761},{10,0,761},{0,59,0},{0,59,0},{0,59,0},{0,29,0},{0,22,277},
{0,22,277},{14,126,1784},{14,89,929},{17,65,1329},{13,60,945},{3,124,3048},{2,75,910},{3,62,14},{0,50,1329},{0,60,4212},{0,47,1974},{24,110,882},{24,80,5},{27,64,70},{23,59,168},{63,2,3048},{0,77,893},{4,63,10},{0,50,1293},{126,2,3048},{0,50,1293},{14,99,884},{14,99,884},{14,99,884},{13,56,884},{3,94,648},{3,62,10},{3,62,10},{1,40,68},{0,46,1256},{0,37,428},{24,78,0},
{24,78,0},{24,78,0},{24,51,0},{49,0,648},{7,59,0},{7,59,0},{0,39,52},{100,0,648},{0,39,52},{78,0,882},{23,81,1},{32,63,0},{0,63,1},{78,0,882},{126,16,882},{0,63,1},{0,52,884},{126,16,882},{0,52,884},{13,0,884},{13,0,884},{13,0,884},{13,0,884},{3,66,0},{3,66,0},{3,66,0},{3,34,0},{0,31,232},{0,31,232},{19,126,1814},{18,93,929},{21,69,1329},
{17,65,942},{7,127,3051},{6,79,910},{7,66,14},{4,54,1329},{0,67,3975},{0,52,1640},{28,114,882},{28,84,5},{30,67,66},{27,63,168},{69,1,3048},{1,83,882},{9,66,10},{0,53,1205},{126,8,3048},{0,53,1205},{18,103,884},{18,103,884},{18,103,884},{17,60,884},{7,98,648},{8,65,9},{8,65,9},{5,44,68},{0,51,1059},{0,43,236},{28,82,0},{28,82,0},{28,82,0},{28,55,0},{55,0,648},
{11,63,0},{11,63,0},{0,45,20},{112,0,648},{0,45,20},{83,1,882},{27,85,1},{36,66,1},{4,66,1},{83,1,882},{126,22,882},{4,66,1},{0,56,884},{126,22,882},{0,56,884},{17,0,884},{17,0,884},{17,0,884},{17,0,884},{7,70,0},{7,70,0},{7,70,0},{7,38,0},{0,37,136},{0,37,136},{23,127,1854},{22,97,929},{25,73,1329},{21,69,942},{12,127,3064},{10,83,910},{11,70,14},
{8,58,1329},{0,73,3751},{0,57,1374},{32,118,882},{32,88,5},{34,71,66},{30,67,165},{75,1,3048},{5,87,882},{13,70,10},{0,59,1125},{126,14,3048},{0,59,1125},{22,107,884},{22,107,884},{22,107,884},{21,64,884},{11,102,648},{12,69,9},{12,69,9},{9,48,68},{0,57,923},{0,49,108},{32,86,0},{32,86,0},{32,86,0},{32,59,0},{61,0,648},{16,65,1},{16,65,1},{0,49,8},{124,0,648},
{0,49,8},{89,1,882},{31,89,1},{40,70,1},{8,70,1},{89,1,882},{126,28,882},{8,70,1},{0,60,884},{126,28,882},{0,60,884},{21,0,884},{21,0,884},{21,0,884},{21,0,884},{11,74,0},{11,74,0},{11,74,0},{11,42,0},{0,46,68},{0,46,68},{29,127,1934},{26,101,929},{29,77,1329},{25,73,942},{17,127,3091},{14,87,910},{15,74,14},{12,62,1329},{0,79,3559},{0,62,1174},{36,122,882},
{36,92,5},{38,75,66},{34,71,165},{81,0,3048},{9,91,882},{17,74,10},{0,62,1053},{126,20,3048},{0,62,1053},{26,111,884},{26,111,884},{26,111,884},{26,67,884},{15,106,648},{16,73,9},{16,73,9},{13,52,68},{0,64,810},{0,54,26},{36,90,0},{36,90,0},{36,90,0},{36,63,0},{66,1,648},{20,69,1},{20,69,1},{0,54,1},{126,5,648},{0,54,1},{95,1,882},{35,93,1},{44,74,1},
{12,74,1},{95,1,882},{126,34,882},{12,74,1},{0,64,882},{126,34,882},{0,64,882},{25,0,884},{25,0,884},{25,0,884},{25,0,884},{15,78,0},{15,78,0},{15,78,0},{15,46,0},{0,51,18},{0,51,18},{33,127,2036},{31,106,931},{33,81,1331},{29,77,942},{23,127,3145},{19,92,910},{19,79,14},{17,65,1329},{0,87,3375},{0,66,1021},{41,126,883},{40,98,2},{43,80,65},{39,76,166},{88,0,3048},
{14,95,882},{21,78,9},{0,66,996},{127,26,3048},{0,66,996},{30,116,882},{30,116,882},{30,116,882},{30,72,882},{20,109,649},{20,77,10},{20,77,10},{18,57,67},{0,70,729},{1,59,3},{41,94,1},{41,94,1},{41,94,1},{40,67,1},{73,0,648},{24,74,1},{24,74,1},{4,58,1},{126,12,648},{4,58,1},{102,0,882},{39,98,1},{48,79,0},{15,79,0},{102,0,882},{126,41,882},{15,79,0},
{0,68,884},{126,41,882},{0,68,884},{30,0,882},{30,0,882},{30,0,882},{30,0,882},{20,82,1},{20,82,1},{20,82,1},{19,51,1},{0,60,0},{0,60,0},{38,127,2134},{35,110,931},{37,85,1331},{33,81,942},{29,127,3217},{23,96,910},{23,83,14},{21,69,1329},{0,93,3247},{0,71,949},{45,127,891},{44,102,2},{47,84,65},{43,80,166},{94,0,3048},{18,99,882},{25,82,9},{0,72,948},{127,32,3048},
{0,72,948},{34,120,882},{34,120,882},{34,120,882},{34,76,882},{24,113,649},{24,81,10},{24,81,10},{22,61,67},{0,79,673},{5,63,3},{45,98,1},{45,98,1},{45,98,1},{44,71,1},{79,0,648},{28,78,1},{28,78,1},{8,62,1},{127,17,648},{8,62,1},{108,0,882},{43,102,1},{52,83,0},{19,83,0},{108,0,882},{125,47,882},{19,83,0},{0,72,884},{125,47,882},{0,72,884},{34,0,882},
{34,0,882},{34,0,882},{34,0,882},{24,86,1},{24,86,1},{24,86,1},{23,55,1},{3,64,1},{3,64,1},{42,127,2284},{39,114,931},{41,89,1331},{37,85,942},{33,127,3316},{27,100,910},{27,87,14},{25,73,1329},{0,99,3151},{1,76,929},{50,127,909},{48,106,2},{51,88,65},{47,84,166},{100,0,3048},{22,103,882},{29,86,9},{0,75,916},{127,38,3048},{0,75,916},{38,124,882},{38,124,882},{38,124,882},
{38,80,882},{28,117,649},{28,85,10},{28,85,10},{27,64,66},{0,85,649},{8,66,5},{49,102,1},{49,102,1},{49,102,1},{48,75,1},{85,0,648},{32,82,1},{32,82,1},{11,66,0},{127,23,648},{11,66,0},{114,0,882},{47,106,1},{56,87,0},{23,87,0},{114,0,882},{127,52,882},{23,87,0},{0,76,884},{127,52,882},{0,76,884},{38,0,882},{38,0,882},{38,0,882},{38,0,882},{28,90,1},
{28,90,1},{28,90,1},{27,59,1},{8,67,1},{8,67,1},{47,127,2414},{43,118,931},{45,93,1331},{41,89,942},{38,127,3409},{31,104,910},{31,91,14},{29,77,1329},{0,105,3087},{5,80,929},{55,127,939},{52,110,2},{55,92,65},{51,88,166},{106,0,3048},{26,107,882},{33,90,9},{0,80,893},{127,44,3048},{0,80,893},{42,127,883},{42,127,883},{42,127,883},{42,84,882},{32,121,649},{32,89,10},{32,89,10},
{31,68,66},{4,89,649},{12,70,5},{53,106,1},{53,106,1},{53,106,1},{52,79,1},{91,0,648},{36,86,1},{36,86,1},{15,70,0},{127,29,648},{15,70,0},{120,0,882},{51,110,1},{60,91,0},{27,91,0},{120,0,882},{127,58,882},{27,91,0},{0,80,884},{127,58,882},{0,80,884},{42,0,882},{42,0,882},{42,0,882},{42,0,882},{32,94,1},{32,94,1},{32,94,1},{31,63,1},{12,71,1},
{12,71,1},{52,127,2584},{47,122,929},{50,98,1329},{46,94,942},{44,127,3547},{35,108,910},{36,95,14},{34,82,1331},{0,112,3055},{9,84,931},{61,127,996},{57,113,5},{59,96,66},{55,92,165},{112,1,3048},{30,112,882},{38,95,10},{0,85,883},{126,51,3048},{0,85,883},{47,127,893},{47,127,893},{47,127,893},{47,88,884},{36,127,648},{37,94,9},{37,94,9},{35,72,65},{9,93,649},{17,75,2},{57,111,0},
{57,111,0},{57,111,0},{57,83,0},{98,0,648},{41,90,1},{41,90,1},{21,74,1},{127,36,648},{21,74,1},{127,0,882},{56,114,1},{64,96,1},{33,95,1},{127,0,882},{126,65,882},{33,95,1},{0,85,882},{126,65,882},{0,85,882},{46,0,884},{46,0,884},{46,0,884},{46,0,884},{36,99,0},{36,99,0},{36,99,0},{36,67,0},{17,75,1},{17,75,1},{58,127,2792},{51,126,929},{54,102,1329},
{50,98,942},{49,127,3672},{39,112,910},{40,99,14},{38,86,1331},{3,117,3049},{13,88,931},{65,127,1061},{61,117,5},{63,100,66},{59,96,165},{118,1,3048},{34,116,882},{42,99,10},{2,89,882},{126,57,3048},{2,89,882},{52,127,916},{52,127,916},{52,127,916},{51,92,884},{41,127,654},{41,98,9},{41,98,9},{39,76,65},{13,97,649},{21,79,2},{61,115,0},{61,115,0},{61,115,0},{61,87,0},{104,0,648},
{45,94,1},{45,94,1},{25,78,1},{127,42,648},{25,78,1},{127,11,882},{60,118,1},{68,100,1},{37,99,1},{127,11,882},{126,71,882},{37,99,1},{0,89,882},{126,71,882},{0,89,882},{50,0,884},{50,0,884},{50,0,884},{50,0,884},{40,103,0},{40,103,0},{40,103,0},{40,71,0},{21,79,1},{21,79,1},{61,127,2984},{55,127,949},{58,106,1329},{54,102,942},{55,127,3832},{43,116,910},{44,103,14},
{42,90,1331},{7,121,3049},{17,92,931},{71,127,1149},{64,122,3},{66,105,67},{63,100,165},{124,1,3048},{38,120,882},{46,103,10},{6,93,882},{126,63,3048},{6,93,882},{55,127,948},{55,127,948},{55,127,948},{55,96,884},{46,126,682},{45,102,9},{45,102,9},{43,80,65},{17,101,649},{25,83,2},{65,118,1},{65,118,1},{65,118,1},{65,91,1},{110,0,648},{49,98,1},{49,98,1},{29,82,1},{126,48,648},
{29,82,1},{127,24,882},{63,123,1},{72,104,1},{41,103,1},{127,24,882},{125,77,882},{41,103,1},{0,93,882},{125,77,882},{0,93,882},{54,0,884},{54,0,884},{54,0,884},{54,0,884},{44,107,0},{44,107,0},{44,107,0},{44,75,0},{25,83,1},{25,83,1},{65,127,3214},{61,127,1021},{62,110,1329},{58,106,942},{61,127,4024},{47,120,910},{48,107,14},{46,94,1331},{11,125,3049},{21,96,931},{74,127,1245},
{68,126,3},{70,109,67},{67,103,171},{127,7,3048},{42,124,882},{50,107,10},{10,97,882},{127,68,3048},{10,97,882},{61,127,996},{61,127,996},{61,127,996},{59,100,884},{50,127,714},{49,106,9},{49,106,9},{47,84,65},{21,105,649},{29,87,2},{69,122,1},{69,122,1},{69,122,1},{69,95,1},{115,1,648},{53,102,1},{53,102,1},{33,86,1},{126,54,648},{33,86,1},{127,36,882},{67,127,0},{76,108,1},
{45,107,1},{127,36,882},{127,82,882},{45,107,1},{0,97,882},{127,82,882},{0,97,882},{58,0,884},{58,0,884},{58,0,884},{58,0,884},{48,111,0},{48,111,0},{48,111,0},{48,79,0},{29,87,1},{29,87,1},{71,127,3494},{65,127,1174},{66,114,1320},{62,110,942},{65,127,4231},{52,125,910},{52,112,14},{50,98,1329},{19,127,3067},{26,101,929},{79,127,1364},{73,127,26},{75,114,68},{72,108,168},{127,21,3048},
{48,127,885},{54,111,9},{16,101,884},{127,75,3048},{16,101,884},{65,127,1053},{65,127,1053},{65,127,1053},{63,105,882},{56,127,769},{53,110,10},{53,110,10},{52,89,66},{25,110,649},{33,91,5},{73,126,1},{73,126,1},{73,126,1},{73,100,0},{122,0,648},{57,107,1},{57,107,1},{36,91,0},{126,61,648},{36,91,0},{127,50,882},{76,127,18},{81,112,0},{48,112,0},{127,50,882},{127,89,882},{48,112,0},
{0,101,884},{127,89,882},{0,101,884},{63,0,882},{63,0,882},{63,0,882},{63,0,882},{53,115,1},{53,115,1},{53,115,1},{53,83,1},{33,92,1},{33,92,1},{74,127,3782},{70,127,1374},{70,118,1320},{66,113,945},{71,127,4455},{57,127,926},{56,116,14},{54,102,1329},{25,127,3139},{30,105,929},{85,127,1476},{79,127,97},{79,118,68},{76,112,168},{127,33,3048},{57,127,925},{58,115,9},{20,105,884},{126,81,3048},
{20,105,884},{68,127,1125},{68,127,1125},{68,127,1125},{66,109,884},{61,127,827},{57,114,10},{57,114,10},{56,93,66},{29,114,649},{37,95,5},{78,126,8},{78,126,8},{78,126,8},{77,104,0},{127,2,648},{61,111,1},{61,111,1},{40,95,0},{127,66,648},{40,95,0},{127,62,882},{81,127,68},{85,116,0},{52,116,0},{127,62,882},{127,95,882},{52,116,0},{0,105,884},{127,95,882},{0,105,884},{66,0,884},
{66,0,884},{66,0,884},{66,0,884},{57,119,1},{57,119,1},{57,119,1},{57,87,1},{37,96,1},{37,96,1},{79,127,4024},{75,127,1640},{74,122,1320},{70,117,945},{74,127,4699},{63,127,1030},{60,120,14},{58,106,1329},{34,127,3259},{34,109,929},{91,127,1620},{84,127,236},{83,122,68},{80,116,168},{127,45,3048},{64,127,1003},{62,119,9},{24,109,884},{126,87,3048},{24,109,884},{74,127,1205},{74,127,1205},{74,127,1205},
{70,113,884},{65,127,910},{61,118,10},{61,118,10},{60,97,66},{33,118,649},{41,99,5},{82,127,20},{82,127,20},{82,127,20},{81,108,0},{127,14,648},{64,116,0},{64,116,0},{44,99,0},{127,72,648},{44,99,0},{127,73,882},{90,127,136},{89,120,0},{56,120,0},{127,73,882},{127,101,882},{56,120,0},{0,109,884},{127,101,882},{0,109,884},{70,0,884},{70,0,884},{70,0,884},{70,0,884},{61,123,1},
{61,123,1},{61,123,1},{61,91,1},{41,100,1},{41,100,1},{85,127,4328},{79,127,1925},{78,126,1320},{74,121,945},{79,127,4920},{67,127,1215},{64,123,14},{62,110,1329},{43,127,3435},{38,113,929},{94,127,1784},{90,127,428},{87,126,68},{84,120,168},{127,57,3048},{70,127,1131},{65,124,10},{28,113,884},{126,93,3048},{28,113,884},{77,127,1293},{77,127,1293},{77,127,1293},{74,117,884},{71,127,1006},{64,123,10},{64,123,10},
{63,100,70},{37,122,649},{45,103,5},{87,126,52},{87,126,52},{87,126,52},{85,112,0},{127,27,648},{68,120,0},{68,120,0},{48,103,0},{126,78,648},{48,103,0},{127,86,882},{96,127,232},{93,124,0},{60,124,0},{127,86,882},{127,107,882},{60,124,0},{0,113,884},{127,107,882},{0,113,884},{74,0,884},{74,0,884},{74,0,884},{74,0,884},{64,126,1},{64,126,1},{64,126,1},{64,95,0},{45,104,1},
{45,104,1},{88,127,4403},{84,127,2227},{82,127,1395},{78,125,922},{85,127,4875},{73,127,1326},{69,127,11},{65,115,1174},{54,127,3438},{44,116,802},{100,127,1746},{95,127,578},{92,127,89},{88,124,126},{127,69,2814},{79,127,1146},{71,127,4},{33,117,761},{126,99,2814},{33,117,761},{82,127,1395},{82,127,1395},{82,127,1395},{79,121,883},{77,127,1137},{69,127,11},{69,127,11},{67,106,67},{42,126,649},{50,108,2},{92,127,89},
{92,127,89},{92,127,89},{90,116,1},{127,40,648},{73,124,0},{73,124,0},{54,107,1},{126,85,648},{54,107,1},{127,96,761},{105,127,277},{98,127,0},{67,127,0},{127,96,761},{127,112,761},{67,127,0},{0,117,761},{127,112,761},{0,117,761},{79,0,882},{79,0,882},{79,0,882},{79,0,882},{69,127,10},{69,127,10},{69,127,10},{68,100,1},{50,108,1},{50,108,1},{94,127,3955},{89,127,2195},{88,127,1523},
{83,127,886},{91,127,4323},{79,127,1139},{74,127,41},{71,116,799},{60,127,2958},{49,118,506},{103,127,1386},{99,127,494},{97,127,130},{92,125,50},{127,76,2249},{87,127,870},{78,127,2},{42,119,481},{126,103,2249},{42,119,481},{88,127,1523},{88,127,1523},{88,127,1523},{83,125,883},{82,127,1251},{74,127,41},{74,127,41},{71,110,67},{48,127,670},{54,112,2},{97,127,130},{97,127,130},{97,127,130},{94,120,1},{127,53,648},
{78,127,2},{78,127,2},{58,111,1},{126,91,648},{58,111,1},{127,102,481},{108,127,169},{104,127,0},{79,127,0},{127,102,481},{127,115,481},{79,127,0},{0,119,481},{127,115,481},{0,119,481},{83,0,882},{83,0,882},{83,0,882},{83,0,882},{74,127,25},{74,127,25},{74,127,25},{72,104,1},{54,112,1},{54,112,1},{97,127,3571},{93,127,2183},{91,127,1635},{87,127,891},{94,127,3827},{84,127,1035},{79,127,114},
{74,119,499},{67,127,2577},{55,121,274},{106,127,1098},{102,127,458},{101,127,193},{97,126,9},{127,86,1769},{93,127,654},{84,127,26},{50,121,265},{127,107,1769},{50,121,265},{91,127,1635},{91,127,1635},{91,127,1635},{87,127,891},{85,127,1387},{79,127,114},{79,127,114},{75,114,67},{57,127,734},{58,116,2},{101,127,193},{101,127,193},{101,127,193},{98,124,1},{127,64,648},{84,127,26},{84,127,26},{62,115,1},{126,97,648},
{62,115,1},{127,108,265},{113,127,97},{110,127,0},{92,127,0},{127,108,265},{127,118,265},{92,127,0},{0,121,265},{127,118,265},{0,121,265},{87,0,882},{87,0,882},{87,0,882},{87,0,882},{79,127,50},{79,127,50},{79,127,50},{76,108,1},{58,116,1},{58,116,1},{100,127,3267},{97,127,2195},{95,127,1794},{92,127,954},{97,127,3435},{87,127,975},{84,127,267},{79,121,283},{73,127,2281},{60,123,114},{111,127,838},
{107,127,434},{106,127,250},{102,127,2},{127,93,1374},{99,127,502},{93,127,74},{58,123,113},{126,111,1374},{58,123,113},{95,127,1794},{95,127,1794},{95,127,1794},{92,127,954},{91,127,1539},{84,127,267},{84,127,267},{79,118,67},{64,127,840},{62,120,2},{106,127,250},{106,127,250},{106,127,250},{102,127,2},{127,76,648},{93,127,74},{93,127,74},{65,119,1},{126,103,648},{65,119,1},{127,115,113},{119,127,41},{116,127,0},
{104,127,0},{127,115,113},{127,121,113},{104,127,0},{0,123,113},{127,121,113},{0,123,113},{91,0,882},{91,0,882},{91,0,882},{91,0,882},{83,127,89},{83,127,89},{83,127,89},{80,112,1},{62,120,1},{62,120,1},{103,127,3032},{101,127,2254},{100,127,1944},{96,127,1080},{103,127,3096},{93,127,1012},{90,127,488},{83,124,133},{81,127,2104},{65,126,20},{114,127,666},{111,127,425},{111,127,325},{108,127,52},{127,102,1032},
{105,127,404},{99,127,160},{67,125,20},{127,115,1032},{67,125,20},{100,127,1944},{100,127,1944},{100,127,1944},{96,127,1080},{97,127,1736},{90,127,488},{90,127,488},{83,122,68},{73,127,1011},{67,124,2},{111,127,325},{111,127,325},{111,127,325},{108,127,52},{127,90,648},{99,127,160},{99,127,160},{69,124,0},{127,109,648},{69,124,0},{127,122,18},{124,127,8},{122,127,1},{118,127,0},{127,122,18},{126,125,18},{118,127,0},
{0,125,20},{126,125,18},{0,125,20},{95,0,884},{95,0,884},{95,0,884},{95,0,884},{88,127,136},{88,127,136},{88,127,136},{85,116,0},{65,126,0},{65,126,0},{109,127,2756},{106,127,2200},{103,127,2004},{102,127,1188},{106,127,2760},{96,127,1060},{96,127,660},{87,126,60},{87,127,1900},{72,127,8},{117,127,534},{117,127,414},{114,127,353},{111,127,104},{127,110,771},{108,127,352},{108,127,208},{75,127,0},{126,119,771},
{75,127,0},{103,127,2004},{103,127,2004},{103,127,2004},{102,127,1188},{100,127,1784},{96,127,660},{96,127,660},{89,125,52},{81,127,1112},{72,127,8},{114,127,353},{114,127,353},{114,127,353},{111,127,104},{127,101,578},{108,127,208},{108,127,208},{75,127,0},{127,114,578},{75,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{99,0,884},
{99,0,884},{99,0,884},{99,0,884},{94,127,200},{94,127,200},{94,127,200},{89,120,0},{72,127,8},{72,127,8},{111,127,2214},{109,127,1844},{109,127,1700},{105,127,1096},{109,127,2180},{102,127,900},{99,127,612},{93,126,12},{93,127,1468},{78,127,40},{120,127,306},{118,127,244},{117,127,205},{116,127,65},{127,113,451},{113,127,211},{111,127,125},{87,127,0},{127,120,451},{87,127,0},{109,127,1700},{109,127,1700},{109,127,1700},
{105,127,1096},{103,127,1460},{99,127,612},{99,127,612},{93,126,8},{87,127,872},{78,127,40},{117,127,205},{117,127,205},{117,127,205},{116,127,65},{127,107,338},{111,127,125},{111,127,125},{87,127,0},{127,117,338},{87,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{103,0,884},{103,0,884},{103,0,884},{103,0,884},{97,127,260},
{97,127,260},{97,127,260},{93,124,0},{78,127,40},{78,127,40},{114,127,1798},{111,127,1521},{111,127,1421},{108,127,1028},{111,127,1675},{105,127,792},{104,127,590},{97,127,4},{96,127,1128},{87,127,100},{123,127,150},{121,127,116},{120,127,97},{119,127,29},{127,118,216},{119,127,99},{116,127,58},{99,127,0},{126,123,216},{99,127,0},{111,127,1421},{111,127,1421},{111,127,1421},{108,127,1028},{109,127,1188},{104,127,590},{104,127,590},
{97,127,4},{93,127,696},{87,127,100},{120,127,97},{120,127,97},{120,127,97},{119,127,29},{127,113,162},{116,127,58},{116,127,58},{99,127,0},{127,120,162},{99,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{107,0,884},{107,0,884},{107,0,884},{107,0,884},{103,127,340},{103,127,340},{103,127,340},{97,127,4},{87,127,100},
{87,127,100},{0,78,1568},{0,54,170},{0,40,4},{0,33,596},{0,53,3371},{0,34,2124},{0,31,1048},{0,21,2552},{0,25,3628},{0,21,2748},{0,78,1568},{0,54,170},{0,40,4},{0,33,596},{26,0,3371},{0,34,2124},{0,31,1048},{0,21,2552},{53,0,3371},{0,21,2552},{0,36,0},{0,36,0},{0,36,0},{0,18,0},{0,18,288},{0,14,106},{0,14,106},{0,9,180},{0,8,315},{0,7,206},{0,36,0},
{0,36,0},{0,36,0},{0,18,0},{9,0,288},{0,14,106},{0,14,106},{0,9,180},{18,0,288},{0,9,180},{39,0,1568},{0,54,170},{0,40,4},{0,33,596},{39,0,1568},{78,0,1568},{0,33,596},{0,26,1568},{78,0,1568},{0,26,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,91,1568},{0,63,90},{0,45,8},
{0,39,500},{0,61,3968},{0,40,2316},{0,35,1078},{0,24,2880},{0,28,4304},{0,24,3136},{0,91,1568},{0,63,90},{1,44,8},{0,39,500},{30,0,3968},{0,40,2316},{0,35,1078},{0,24,2880},{61,0,3968},{0,24,2880},{0,48,0},{0,48,0},{0,48,0},{0,24,0},{0,24,512},{0,19,180},{0,19,180},{0,10,313},{0,11,563},{0,10,362},{0,48,0},{0,48,0},{0,48,0},{0,24,0},{12,0,512},
{0,19,180},{0,19,180},{0,10,313},{24,0,512},{0,10,313},{45,0,1568},{0,63,90},{2,44,0},{0,39,500},{45,0,1568},{91,0,1568},{0,39,500},{0,30,1568},{91,0,1568},{0,30,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,103,1568},{0,70,37},{1,49,48},{0,42,400},{0,68,4651},{0,46,2540},{0,40,1128},
{0,27,3256},{0,31,5084},{0,27,3580},{0,103,1568},{0,70,37},{1,49,44},{0,42,400},{33,1,4651},{0,46,2540},{0,40,1128},{0,27,3256},{68,0,4651},{0,27,3256},{0,61,0},{0,61,0},{0,61,0},{0,30,0},{0,30,800},{0,25,292},{0,25,292},{0,13,485},{0,14,883},{0,13,566},{0,61,0},{0,61,0},{0,61,0},{0,30,0},{15,0,800},{0,25,292},{0,25,292},{0,13,485},{30,0,800},
{0,13,485},{50,1,1568},{0,70,37},{6,48,0},{0,42,400},{50,1,1568},{103,0,1568},{0,42,400},{0,34,1568},{103,0,1568},{0,34,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,115,1568},{0,76,5},{1,54,145},{0,47,325},{0,77,5419},{0,51,2755},{0,43,1208},{0,30,3680},{0,34,5968},{0,30,4080},{0,115,1568},
{0,76,5},{3,53,121},{0,47,325},{38,0,5419},{0,51,2755},{0,43,1208},{0,30,3680},{77,0,5419},{0,30,3680},{0,72,0},{0,72,0},{0,72,0},{0,36,0},{0,36,1152},{0,28,424},{0,28,424},{0,18,720},{0,17,1275},{0,15,824},{0,72,0},{0,72,0},{0,72,0},{0,36,0},{18,0,1152},{0,28,424},{0,28,424},{0,18,720},{36,0,1152},{0,18,720},{56,1,1568},{0,76,5},{10,52,0},
{0,47,325},{56,1,1568},{115,0,1568},{0,47,325},{0,38,1568},{115,0,1568},{0,38,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,125,1609},{2,82,42},{5,58,254},{1,51,318},{0,91,5419},{0,57,2480},{0,49,835},{0,36,3427},{0,40,6191},{0,35,3991},{5,119,1569},{4,80,6},{8,58,122},{2,52,302},{45,0,5419},
{0,57,2480},{0,49,835},{0,36,3427},{91,0,5419},{0,36,3427},{2,82,41},{2,82,41},{2,82,41},{2,41,42},{0,50,1152},{0,37,274},{0,37,274},{0,21,586},{0,22,1366},{0,20,782},{5,76,1},{5,76,1},{5,76,1},{5,40,1},{24,1,1152},{0,37,274},{0,37,274},{0,21,586},{50,0,1152},{0,21,586},{63,0,1568},{2,82,1},{15,56,1},{0,53,245},{63,0,1568},{127,1,1568},{0,53,245},
{0,42,1570},{127,1,1568},{0,42,1570},{2,0,41},{2,0,41},{2,0,41},{2,0,41},{0,13,0},{0,13,0},{0,13,0},{0,7,0},{0,5,13},{0,5,13},{5,126,1735},{4,87,150},{7,63,426},{3,56,382},{0,103,5419},{0,64,2265},{0,54,557},{0,39,3243},{0,46,6415},{0,39,3919},{9,123,1569},{8,84,6},{12,62,122},{6,56,302},{50,1,5419},{0,64,2265},{0,54,557},{0,39,3243},{103,0,5419},
{0,39,3243},{4,90,145},{4,90,145},{4,90,145},{4,46,146},{0,62,1152},{0,43,170},{0,43,170},{0,27,482},{0,28,1494},{0,24,770},{9,80,1},{9,80,1},{9,80,1},{9,44,1},{30,1,1152},{0,43,170},{0,43,170},{0,27,482},{62,0,1152},{0,27,482},{69,0,1568},{6,86,1},{19,60,1},{0,56,181},{69,0,1568},{127,7,1568},{0,56,181},{0,46,1570},{127,7,1568},{0,46,1570},{4,0,145},
{4,0,145},{4,0,145},{4,0,145},{0,26,0},{0,26,0},{0,26,0},{0,13,0},{0,11,53},{0,11,53},{9,127,1991},{7,93,322},{9,66,678},{5,61,518},{0,115,5419},{0,70,2081},{0,60,341},{0,45,3035},{0,51,6641},{0,42,3891},{13,127,1569},{12,88,6},{15,66,122},{10,60,302},{56,1,5419},{0,70,2081},{0,60,341},{0,45,3035},{115,0,5419},{0,45,3035},{6,98,313},{6,98,313},{6,98,313},
{6,51,314},{0,74,1152},{0,51,80},{0,51,80},{0,30,394},{0,31,1650},{0,28,782},{13,84,1},{13,84,1},{13,84,1},{13,48,1},{36,1,1152},{0,51,80},{0,51,80},{0,30,394},{74,0,1152},{0,30,394},{75,0,1568},{10,90,1},{23,64,0},{0,62,125},{75,0,1568},{127,13,1568},{0,62,125},{0,50,1570},{127,13,1568},{0,50,1570},{6,0,313},{6,0,313},{6,0,313},{6,0,313},{0,38,0},
{0,38,0},{0,38,0},{0,18,1},{0,14,117},{0,14,117},{12,127,2387},{9,98,566},{11,71,1002},{7,64,721},{0,127,5419},{0,76,1929},{0,64,181},{0,50,2886},{0,57,6921},{0,47,3879},{17,127,1577},{16,92,6},{19,70,122},{13,64,309},{62,1,5419},{0,76,1929},{0,64,181},{0,50,2886},{127,0,5419},{0,50,2886},{8,106,545},{8,106,545},{8,106,545},{8,56,545},{0,86,1152},{0,57,32},{0,57,32},
{0,36,306},{0,37,1826},{0,34,822},{17,88,1},{17,88,1},{17,88,1},{17,52,1},{42,1,1152},{0,57,32},{0,57,32},{0,36,306},{86,0,1152},{0,36,306},{80,1,1568},{14,94,1},{27,68,0},{0,66,80},{80,1,1568},{126,19,1568},{0,66,80},{0,54,1570},{126,19,1568},{0,54,1570},{8,0,545},{8,0,545},{8,0,545},{8,0,545},{0,50,0},{0,50,0},{0,50,0},{0,24,1},{0,19,193},
{0,19,193},{15,127,3004},{11,106,925},{14,77,1461},{9,69,1030},{3,127,5520},{0,84,1769},{0,70,66},{0,53,2697},{0,63,7276},{0,51,3898},{23,127,1602},{21,97,5},{23,74,125},{19,67,306},{69,0,5419},{0,84,1769},{0,70,66},{0,53,2697},{127,7,5419},{0,53,2697},{11,114,884},{11,114,884},{11,114,884},{10,62,884},{0,100,1152},{0,65,5},{0,65,5},{0,42,232},{0,43,2064},{0,37,896},{21,93,0},
{21,93,0},{21,93,0},{21,57,0},{49,0,1152},{0,65,5},{0,65,5},{0,42,232},{100,0,1152},{0,42,232},{87,0,1568},{19,98,1},{31,73,1},{0,71,45},{87,0,1568},{126,26,1568},{0,71,45},{0,59,1568},{126,26,1568},{0,59,1568},{10,0,884},{10,0,884},{10,0,884},{10,0,884},{0,64,0},{0,64,0},{0,64,0},{0,31,0},{0,25,320},{0,25,320},{17,127,3702},{13,111,1317},{17,80,1945},
{11,74,1374},{6,127,5788},{0,90,1681},{0,75,18},{0,59,2537},{0,70,7631},{0,55,3952},{29,127,1650},{25,101,5},{27,78,125},{23,71,306},{75,0,5419},{0,90,1681},{0,75,18},{0,59,2537},{127,13,5419},{0,59,2537},{13,122,1252},{13,122,1252},{13,122,1252},{13,66,1252},{0,112,1152},{0,72,5},{0,72,5},{0,45,164},{0,51,2291},{0,43,992},{25,97,0},{25,97,0},{25,97,0},{25,61,0},{55,0,1152},
{3,70,1},{3,70,1},{0,45,164},{112,0,1152},{0,45,164},{93,0,1568},{23,102,1},{35,77,1},{0,75,18},{93,0,1568},{126,32,1568},{0,75,18},{0,63,1568},{126,32,1568},{0,63,1568},{12,0,1252},{12,0,1252},{12,0,1252},{12,0,1252},{0,75,0},{0,75,0},{0,75,0},{0,37,0},{0,31,464},{0,31,464},{23,127,4370},{16,114,1661},{19,85,2353},{14,77,1674},{12,127,6128},{0,99,1621},{2,80,18},
{0,62,2397},{0,73,7815},{0,60,3858},{32,127,1702},{29,105,5},{31,82,125},{27,75,306},{80,1,5419},{0,99,1617},{2,80,14},{0,62,2393},{126,19,5419},{0,62,2393},{15,127,1572},{15,127,1572},{15,127,1572},{15,71,1568},{1,122,1152},{1,79,17},{1,79,17},{0,50,121},{0,54,2403},{0,46,996},{29,101,0},{29,101,0},{29,101,0},{29,64,0},{61,0,1152},{7,74,1},{7,74,1},{0,50,117},{124,0,1152},
{0,50,117},{99,0,1568},{27,106,1},{39,81,1},{0,80,5},{99,0,1568},{126,38,1568},{0,80,5},{0,66,1570},{126,38,1568},{0,66,1570},{15,0,1568},{15,0,1568},{15,0,1568},{15,0,1568},{1,86,0},{1,86,0},{1,86,0},{1,43,0},{0,34,544},{0,34,544},{26,127,4586},{20,118,1661},{23,89,2353},{18,81,1674},{17,127,6275},{3,102,1617},{6,84,18},{1,67,2355},{0,81,7401},{0,64,3371},{38,127,1766},
{33,109,5},{35,86,125},{31,79,306},{86,1,5419},{0,105,1577},{6,84,14},{0,66,2259},{126,25,5419},{0,66,2259},{20,127,1585},{20,127,1585},{20,127,1585},{19,75,1568},{5,126,1152},{5,83,17},{5,83,17},{4,54,121},{0,60,2155},{0,51,682},{33,105,0},{33,105,0},{33,105,0},{33,68,0},{66,1,1152},{11,78,1},{11,78,1},{0,55,80},{126,5,1152},{0,55,80},{105,0,1568},{31,110,1},{43,85,1},
{0,84,1},{105,0,1568},{126,44,1568},{0,84,1},{0,70,1570},{126,44,1568},{0,70,1570},{19,0,1568},{19,0,1568},{19,0,1568},{19,0,1568},{5,90,0},{5,90,0},{5,90,0},{5,47,0},{0,43,388},{0,43,388},{32,127,4866},{25,123,1659},{27,93,2355},{23,86,1670},{23,127,6489},{8,107,1615},{10,88,18},{5,71,2353},{0,87,7039},{0,70,2897},{44,127,1865},{37,113,6},{40,91,122},{36,84,305},{93,0,5419},
{2,111,1569},{10,88,17},{0,72,2124},{126,32,5419},{0,72,2124},{24,127,1619},{24,127,1619},{24,127,1619},{24,79,1570},{10,127,1161},{10,88,14},{10,88,14},{8,58,122},{0,67,1905},{0,57,409},{38,109,1},{38,109,1},{38,109,1},{37,73,1},{73,0,1152},{15,83,1},{15,83,1},{0,59,41},{126,12,1152},{0,59,41},{112,0,1568},{35,115,1},{48,89,0},{4,89,0},{112,0,1568},{127,50,1568},{4,89,0},
{0,75,1568},{127,50,1568},{0,75,1568},{23,0,1570},{23,0,1570},{23,0,1570},{23,0,1570},{10,93,1},{10,93,1},{10,93,1},{10,51,1},{0,51,232},{0,51,232},{35,127,5154},{29,127,1659},{31,97,2355},{27,90,1670},{26,127,6689},{12,111,1615},{14,92,18},{9,75,2353},{0,93,6751},{0,74,2541},{49,127,1955},{41,117,6},{44,95,122},{40,88,305},{99,0,5419},{6,115,1569},{14,92,17},{0,75,2020},{126,38,5419},
{0,75,2020},{29,127,1650},{29,127,1650},{29,127,1650},{28,83,1570},{15,127,1179},{14,92,14},{14,92,14},{12,62,122},{0,76,1713},{0,63,225},{42,113,1},{42,113,1},{42,113,1},{41,77,1},{79,0,1152},{19,87,1},{19,87,1},{0,64,20},{127,17,1152},{0,64,20},{118,0,1568},{39,119,1},{52,93,0},{8,93,0},{118,0,1568},{127,56,1568},{8,93,0},{0,79,1568},{127,56,1568},{0,79,1568},{27,0,1570},
{27,0,1570},{27,0,1570},{27,0,1570},{14,97,1},{14,97,1},{14,97,1},{14,55,1},{0,57,136},{0,57,136},{41,127,5426},{33,127,1711},{35,101,2355},{31,94,1670},{32,127,6905},{16,115,1615},{18,96,18},{13,79,2353},{0,102,6483},{0,79,2241},{52,127,2075},{45,121,6},{48,99,122},{44,92,305},{105,0,5419},{10,119,1569},{18,96,17},{0,80,1929},{126,44,5419},{0,80,1929},{33,127,1707},{33,127,1707},{33,127,1707},
{32,87,1570},{20,127,1209},{18,96,14},{18,96,14},{15,66,125},{0,81,1526},{0,67,97},{46,117,1},{46,117,1},{46,117,1},{45,81,1},{85,0,1152},{23,91,1},{23,91,1},{0,69,4},{127,23,1152},{0,69,4},{124,0,1568},{43,123,1},{56,97,0},{12,97,0},{124,0,1568},{127,62,1568},{12,97,0},{0,83,1568},{127,62,1568},{0,83,1568},{31,0,1570},{31,0,1570},{31,0,1570},{31,0,1570},{18,101,1},
{18,101,1},{18,101,1},{18,59,1},{0,64,65},{0,64,65},{44,127,5762},{38,127,1811},{39,105,2355},{35,98,1670},{38,127,7153},{20,119,1615},{22,100,18},{17,83,2353},{0,108,6243},{0,83,2009},{58,127,2195},{49,125,6},{52,103,122},{48,96,305},{111,0,5419},{14,123,1569},{22,100,17},{0,85,1856},{127,49,5419},{0,85,1856},{38,127,1762},{38,127,1762},{38,127,1762},{36,91,1570},{26,127,1265},{22,100,14},{22,100,14},
{19,70,125},{0,87,1398},{0,73,25},{50,121,1},{50,121,1},{50,121,1},{49,85,1},{91,0,1152},{27,95,1},{27,95,1},{0,73,0},{127,29,1152},{0,73,0},{127,5,1568},{47,127,1},{60,101,0},{16,101,0},{127,5,1568},{126,68,1568},{16,101,0},{0,87,1568},{126,68,1568},{0,87,1568},{35,0,1570},{35,0,1570},{35,0,1570},{35,0,1570},{22,105,1},{22,105,1},{22,105,1},{22,63,1},{0,70,25},
{0,70,25},{52,127,6088},{43,127,2009},{44,110,2353},{39,102,1674},{44,127,7451},{24,123,1617},{27,105,18},{22,88,2355},{0,113,5999},{0,89,1811},{65,127,2385},{54,127,25},{56,107,125},{52,100,306},{118,0,5419},{19,127,1569},{27,105,14},{0,89,1762},{127,56,5419},{0,89,1762},{42,127,1856},{42,127,1856},{42,127,1856},{40,96,1568},{30,127,1344},{26,104,17},{26,104,17},{24,75,122},{0,96,1281},{1,78,6},{54,126,0},
{54,126,0},{54,126,0},{54,89,0},{98,0,1152},{32,99,1},{32,99,1},{6,77,1},{127,36,1152},{6,77,1},{127,19,1568},{54,127,25},{64,105,1},{21,105,1},{127,19,1568},{127,74,1568},{21,105,1},{0,91,1570},{127,74,1568},{0,91,1570},{40,0,1568},{40,0,1568},{40,0,1568},{40,0,1568},{26,111,0},{26,111,0},{26,111,0},{26,67,0},{0,79,1},{0,79,1},{55,127,6416},{48,127,2241},{48,114,2353},
{43,106,1674},{49,127,7696},{28,127,1617},{31,109,18},{26,92,2355},{0,122,5827},{0,94,1711},{68,127,2521},{60,127,97},{60,111,125},{56,104,306},{124,0,5419},{25,127,1601},{31,109,14},{0,94,1707},{127,62,5419},{0,94,1707},{47,127,1929},{47,127,1929},{47,127,1929},{44,100,1568},{35,127,1414},{30,108,17},{30,108,17},{28,79,122},{0,102,1209},{5,82,6},{58,127,4},{58,127,4},{58,127,4},{58,93,0},{104,0,1152},
{36,103,1},{36,103,1},{10,81,1},{127,42,1152},{10,81,1},{127,31,1568},{63,127,65},{68,109,1},{25,109,1},{127,31,1568},{127,80,1568},{25,109,1},{0,95,1570},{127,80,1568},{0,95,1570},{44,0,1568},{44,0,1568},{44,0,1568},{44,0,1568},{30,115,0},{30,115,0},{30,115,0},{30,71,0},{4,83,1},{4,83,1},{61,127,6800},{53,127,2541},{52,118,2353},{47,110,1674},{52,127,7996},{34,127,1665},{35,113,18},
{30,96,2355},{0,126,5709},{0,98,1659},{74,127,2705},{64,127,225},{65,115,122},{60,108,306},{127,5,5419},{34,127,1665},{35,113,14},{0,98,1650},{126,68,5419},{0,98,1650},{52,127,2020},{52,127,2020},{52,127,2020},{48,104,1568},{41,127,1510},{34,112,17},{34,112,17},{32,83,122},{0,108,1169},{9,86,6},{63,126,20},{63,126,20},{63,126,20},{62,97,0},{110,0,1152},{40,107,1},{40,107,1},{14,85,1},{126,48,1152},
{14,85,1},{127,43,1568},{70,127,136},{72,113,1},{29,113,1},{127,43,1568},{127,86,1568},{29,113,1},{0,99,1570},{127,86,1568},{0,99,1570},{48,0,1568},{48,0,1568},{48,0,1568},{48,0,1568},{34,119,0},{34,119,0},{34,119,0},{34,75,0},{8,87,1},{8,87,1},{65,127,7186},{57,127,2897},{56,122,2353},{51,114,1674},{58,127,8300},{40,127,1809},{39,117,18},{34,100,2355},{5,127,5791},{4,102,1659},{79,127,2875},
{70,127,409},{69,119,122},{63,114,310},{127,18,5419},{43,127,1777},{39,117,14},{0,103,1619},{126,74,5419},{0,103,1619},{55,127,2124},{55,127,2124},{55,127,2124},{52,108,1568},{44,127,1634},{38,116,17},{38,116,17},{36,87,122},{1,115,1153},{13,90,6},{68,127,41},{68,127,41},{68,127,41},{66,101,1},{115,1,1152},{44,111,1},{44,111,1},{18,89,1},{126,54,1152},{18,89,1},{127,56,1568},{76,127,232},{76,117,1},
{33,117,1},{127,56,1568},{127,92,1568},{33,117,1},{0,103,1570},{127,92,1568},{0,103,1570},{52,0,1568},{52,0,1568},{52,0,1568},{52,0,1568},{38,123,0},{38,123,0},{38,123,0},{38,79,0},{12,91,1},{12,91,1},{68,127,7650},{63,127,3371},{60,126,2355},{56,119,1670},{65,127,8695},{46,127,2083},{43,121,18},{38,104,2353},{14,127,6049},{6,107,1661},{82,127,3112},{76,127,682},{73,123,121},{67,117,305},{127,31,5419},
{51,127,1977},{43,121,17},{0,107,1585},{127,80,5419},{0,107,1585},{61,127,2259},{61,127,2259},{61,127,2259},{57,112,1570},{52,127,1755},{43,121,14},{43,121,14},{40,91,125},{6,119,1153},{18,94,5},{72,127,80},{72,127,80},{72,127,80},{70,106,0},{122,0,1152},{48,116,1},{48,116,1},{21,94,0},{126,61,1152},{21,94,0},{127,69,1568},{84,127,388},{80,122,0},{37,122,0},{127,69,1568},{126,99,1568},{37,122,0},
{0,108,1568},{126,99,1568},{0,108,1568},{56,0,1570},{56,0,1570},{56,0,1570},{56,0,1570},{43,126,1},{43,126,1},{43,126,1},{42,84,1},{16,96,1},{16,96,1},{74,127,8066},{67,127,3858},{65,127,2397},{60,123,1670},{68,127,9035},{51,127,2458},{47,125,18},{42,108,2353},{22,127,6379},{10,111,1661},{88,127,3320},{79,127,977},{77,127,121},{71,121,305},{127,43,5419},{60,127,2185},{47,125,17},{0,112,1572},{127,86,5419},
{0,112,1572},{65,127,2393},{65,127,2393},{65,127,2393},{61,116,1570},{55,127,1891},{47,125,14},{47,125,14},{44,95,125},{10,123,1153},{22,98,5},{77,127,117},{77,127,117},{77,127,117},{74,110,0},{127,2,1152},{52,120,1},{52,120,1},{25,98,0},{127,66,1152},{25,98,0},{127,81,1568},{93,127,544},{84,126,0},{41,126,0},{127,81,1568},{126,105,1568},{41,126,0},{0,112,1568},{126,105,1568},{0,112,1568},{60,0,1570},
{60,0,1570},{60,0,1570},{60,0,1570},{47,127,5},{47,127,5},{47,127,5},{46,88,1},{20,100,1},{20,100,1},{79,127,7660},{72,127,3952},{68,127,2537},{64,125,1620},{74,127,8515},{57,127,2310},{52,127,18},{47,110,1945},{28,127,5939},{16,114,1317},{91,127,2984},{84,127,992},{82,127,164},{76,123,201},{127,53,4803},{64,127,1931},{54,126,5},{5,114,1252},{126,91,4803},{5,114,1252},{68,127,2537},{68,127,2537},{68,127,2537},
{64,120,1569},{61,127,2043},{52,127,18},{52,127,18},{48,99,125},{14,127,1153},{26,102,5},{82,127,164},{82,127,164},{82,127,164},{78,114,0},{127,14,1152},{56,124,1},{56,124,1},{29,102,0},{127,72,1152},{29,102,0},{127,89,1250},{96,127,464},{90,127,0},{51,127,0},{127,89,1250},{125,109,1250},{51,127,0},{0,114,1252},{125,109,1250},{0,114,1252},{64,0,1568},{64,0,1568},{64,0,1568},{64,0,1568},{52,127,18},
{52,127,18},{52,127,18},{50,92,1},{24,104,1},{24,104,1},{82,127,7060},{76,127,3898},{74,127,2697},{68,126,1576},{79,127,7756},{60,127,2062},{57,127,66},{50,113,1461},{37,127,5307},{21,116,925},{97,127,2504},{90,127,896},{85,127,232},{80,125,96},{127,60,4056},{70,127,1563},{60,127,5},{13,116,884},{127,94,4056},{13,116,884},{74,127,2697},{74,127,2697},{74,127,2697},{68,124,1569},{65,127,2214},{57,127,66},{57,127,66},
{52,103,125},{22,127,1186},{30,106,5},{85,127,232},{85,127,232},{85,127,232},{82,118,0},{127,27,1152},{62,126,5},{62,126,5},{33,106,0},{126,78,1152},{33,106,0},{127,95,882},{102,127,320},{96,127,0},{63,127,0},{127,95,882},{127,111,882},{63,127,0},{0,116,884},{127,111,882},{0,116,884},{68,0,1568},{68,0,1568},{68,0,1568},{68,0,1568},{56,127,45},{56,127,45},{56,127,45},{54,96,1},{28,108,1},
{28,108,1},{85,127,6483},{79,127,3828},{79,127,2867},{73,127,1574},{82,127,6979},{67,127,1846},{63,127,181},{56,116,1002},{43,127,4714},{29,118,566},{100,127,2034},{93,127,822},{91,127,306},{86,125,26},{127,69,3318},{76,127,1233},{70,127,32},{20,119,545},{126,99,3318},{20,119,545},{79,127,2867},{79,127,2867},{79,127,2867},{73,127,1574},{71,127,2425},{63,127,181},{63,127,181},{57,108,122},{31,127,1273},{34,111,6},{91,127,306},
{91,127,306},{91,127,306},{87,122,1},{127,40,1152},{70,127,32},{70,127,32},{39,110,1},{126,85,1152},{39,110,1},{127,101,545},{108,127,193},{102,127,1},{76,127,0},{127,101,545},{127,114,545},{76,127,0},{0,119,545},{127,114,545},{0,119,545},{72,0,1570},{72,0,1570},{72,0,1570},{72,0,1570},{61,127,80},{61,127,80},{61,127,80},{59,100,0},{33,112,1},{33,112,1},{88,127,6059},{84,127,3891},{82,127,3035},
{76,127,1634},{85,127,6411},{70,127,1754},{67,127,341},{61,118,678},{51,127,4330},{34,120,322},{103,127,1698},{98,127,782},{97,127,394},{90,127,2},{127,76,2753},{84,127,1018},{76,127,80},{28,121,313},{126,103,2753},{28,121,313},{82,127,3035},{82,127,3035},{82,127,3035},{76,127,1634},{77,127,2641},{67,127,341},{67,127,341},{61,112,122},{37,127,1401},{38,115,6},{97,127,394},{97,127,394},{97,127,394},{91,126,1},{127,53,1152},
{76,127,80},{76,127,80},{43,114,1},{126,91,1152},{43,114,1},{127,107,313},{111,127,116},{108,127,1},{89,127,0},{127,107,313},{127,117,313},{89,127,0},{0,121,313},{127,117,313},{0,121,313},{76,0,1570},{76,0,1570},{76,0,1570},{76,0,1570},{65,127,125},{65,127,125},{65,127,125},{63,104,0},{37,116,1},{37,116,1},{94,127,5691},{88,127,3919},{88,127,3243},{81,127,1739},{88,127,5947},{76,127,1722},{73,127,557},
{64,120,426},{57,127,3994},{39,122,150},{106,127,1434},{102,127,770},{100,127,482},{95,127,17},{127,86,2273},{90,127,850},{84,127,170},{36,123,145},{127,107,2273},{36,123,145},{88,127,3243},{88,127,3243},{88,127,3243},{81,127,1739},{79,127,2835},{73,127,557},{73,127,557},{65,115,122},{46,127,1561},{42,119,6},{100,127,482},{100,127,482},{100,127,482},{95,127,17},{127,64,1152},{84,127,170},{84,127,170},{47,118,1},{126,97,1152},
{47,118,1},{127,113,145},{116,127,53},{114,127,0},{101,127,0},{127,113,145},{127,120,145},{101,127,0},{0,123,145},{127,120,145},{0,123,145},{80,0,1570},{80,0,1570},{80,0,1570},{80,0,1570},{71,127,181},{71,127,181},{71,127,181},{67,108,1},{41,120,1},{41,120,1},{97,127,5379},{92,127,3991},{91,127,3427},{87,127,1907},{94,127,5539},{79,127,1783},{79,127,822},{69,122,254},{64,127,3745},{44,125,42},{111,127,1210},
{106,127,782},{106,127,586},{99,127,82},{127,93,1878},{96,127,746},{90,127,274},{44,125,41},{126,111,1878},{44,125,41},{91,127,3427},{91,127,3427},{91,127,3427},{87,127,1907},{85,127,3051},{79,127,822},{79,127,822},{69,119,122},{54,127,1798},{46,123,6},{106,127,586},{106,127,586},{106,127,586},{99,127,82},{127,76,1152},{90,127,274},{90,127,274},{51,122,1},{126,103,1152},{51,122,1},{127,119,41},{122,127,13},{120,127,0},
{113,127,0},{127,119,41},{127,123,41},{113,127,0},{0,125,41},{127,123,41},{0,125,41},{84,0,1570},{84,0,1570},{84,0,1570},{84,0,1570},{74,127,245},{74,127,245},{74,127,245},{71,112,1},{45,124,1},{45,124,1},{100,127,5128},{97,127,4080},{97,127,3680},{90,127,2152},{97,127,5200},{87,127,1964},{84,127,1208},{73,126,145},{73,127,3580},{51,127,5},{114,127,1062},{111,127,797},{111,127,697},{105,127,212},{127,102,1536},
{102,127,708},{99,127,424},{54,127,0},{127,115,1536},{54,127,0},{97,127,3680},{97,127,3680},{97,127,3680},{90,127,2152},{91,127,3328},{84,127,1208},{84,127,1208},{74,124,121},{63,127,2089},{51,127,5},{111,127,697},{111,127,697},{111,127,697},{105,127,212},{127,90,1152},{99,127,424},{99,127,424},{54,127,0},{127,109,1152},{54,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},
{0,127,0},{127,127,0},{0,127,0},{89,0,1568},{89,0,1568},{89,0,1568},{89,0,1568},{79,127,320},{79,127,320},{79,127,320},{75,117,0},{51,127,5},{51,127,5},{103,127,4416},{100,127,3580},{100,127,3256},{96,127,2040},{100,127,4372},{90,127,1736},{87,127,1128},{79,126,57},{76,127,2956},{57,127,37},{117,127,738},{114,127,566},{114,127,485},{108,127,148},{127,107,1067},{108,127,484},{102,127,292},{66,127,0},{127,117,1067},
{66,127,0},{100,127,3256},{100,127,3256},{100,127,3256},{96,127,2040},{94,127,2852},{87,127,1128},{87,127,1128},{78,125,45},{70,127,1739},{57,127,37},{114,127,485},{114,127,485},{114,127,485},{108,127,148},{127,96,800},{102,127,292},{102,127,292},{66,127,0},{127,112,800},{66,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{93,0,1568},
{93,0,1568},{93,0,1568},{93,0,1568},{85,127,400},{85,127,400},{85,127,400},{79,121,0},{57,127,37},{57,127,37},{106,127,3784},{103,127,3136},{103,127,2880},{99,127,1924},{103,127,3648},{93,127,1564},{92,127,1078},{82,127,8},{81,127,2480},{64,127,90},{117,127,482},{117,127,362},{117,127,313},{113,127,97},{127,110,683},{111,127,321},{108,127,180},{78,127,0},{126,119,683},{78,127,0},{103,127,2880},{103,127,2880},{103,127,2880},
{99,127,1924},{97,127,2448},{92,127,1078},{92,127,1078},{83,126,8},{76,127,1451},{64,127,90},{117,127,313},{117,127,313},{117,127,313},{113,127,97},{127,102,512},{108,127,180},{108,127,180},{78,127,0},{127,115,512},{78,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{97,0,1568},{97,0,1568},{97,0,1568},{97,0,1568},{88,127,500},
{88,127,500},{88,127,500},{83,125,0},{64,127,90},{64,127,90},{109,127,3232},{106,127,2748},{106,127,2552},{102,127,1832},{106,127,3028},{99,127,1420},{96,127,1048},{87,127,4},{87,127,2032},{73,127,170},{120,127,262},{120,127,206},{118,127,180},{116,127,53},{127,115,384},{116,127,179},{113,127,106},{90,127,0},{127,121,384},{90,127,0},{106,127,2552},{106,127,2552},{106,127,2552},{102,127,1832},{103,127,2112},{96,127,1048},{96,127,1048},
{87,127,4},{81,127,1260},{73,127,170},{118,127,180},{118,127,180},{118,127,180},{116,127,53},{127,108,288},{113,127,106},{113,127,106},{90,127,0},{127,118,288},{90,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{101,0,1568},{101,0,1568},{101,0,1568},{101,0,1568},{94,127,596},{94,127,596},{94,127,596},{87,127,4},{73,127,170},
{73,127,170},{0,104,2665},{0,73,274},{0,53,4},{0,45,985},{0,69,5885},{0,46,3677},{0,41,1789},{0,27,4441},{0,31,6341},{0,27,4765},{0,104,2665},{0,73,274},{0,53,4},{0,45,985},{34,0,5885},{0,46,3677},{0,41,1789},{0,27,4441},{69,0,5885},{0,27,4441},{0,50,0},{0,50,0},{0,50,0},{0,24,1},{0,24,545},{0,19,193},{0,19,193},{0,12,337},{0,11,598},{0,10,387},{0,50,0},
{0,50,0},{0,50,0},{0,24,1},{12,0,545},{0,19,193},{0,19,193},{0,12,337},{24,0,545},{0,12,337},{51,0,2665},{0,73,274},{0,53,4},{0,45,985},{51,0,2665},{104,0,2665},{0,45,985},{0,35,2665},{104,0,2665},{0,35,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,117,2665},{0,81,160},{1,57,12},
{0,50,872},{0,78,6669},{0,51,3898},{0,46,1825},{0,33,4865},{0,34,7249},{0,30,5277},{0,117,2665},{0,81,160},{1,57,8},{0,50,872},{39,0,6669},{0,51,3898},{0,46,1825},{0,33,4865},{78,0,6669},{0,33,4865},{0,62,0},{0,62,0},{0,62,0},{0,30,1},{0,30,841},{0,25,305},{0,25,305},{0,13,514},{0,14,926},{0,13,595},{0,62,0},{0,62,0},{0,62,0},{0,30,1},{15,0,841},
{0,25,305},{0,25,305},{0,13,514},{30,0,841},{0,13,514},{57,0,2665},{0,81,160},{3,57,0},{0,50,872},{57,0,2665},{117,0,2665},{0,50,872},{0,39,2665},{117,0,2665},{0,39,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,127,2669},{0,87,80},{1,63,60},{0,53,740},{0,86,7538},{0,57,4178},{0,49,1889},
{0,36,5333},{0,40,8241},{0,33,5845},{0,127,2669},{0,87,80},{2,61,52},{0,53,740},{42,1,7538},{0,57,4178},{0,49,1889},{0,36,5333},{86,0,7538},{0,36,5333},{0,74,0},{0,74,0},{0,74,0},{0,36,1},{0,36,1201},{0,28,445},{0,28,445},{0,18,745},{0,17,1326},{0,15,861},{0,74,0},{0,74,0},{0,74,0},{0,36,1},{18,0,1201},{0,28,445},{0,28,445},{0,18,745},{36,0,1201},
{0,18,745},{63,0,2665},{0,87,80},{7,61,0},{0,53,740},{63,0,2665},{127,1,2665},{0,53,740},{0,43,2665},{127,1,2665},{0,43,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{3,127,2797},{0,93,32},{2,65,154},{0,59,628},{0,94,8493},{0,63,4490},{0,54,1965},{0,39,5849},{0,43,9333},{0,36,6469},{3,127,2761},
{0,93,32},{3,66,126},{0,59,628},{46,0,8493},{0,63,4490},{0,54,1965},{0,39,5849},{94,0,8493},{0,39,5849},{0,86,0},{0,86,0},{0,86,0},{0,42,1},{0,42,1625},{0,34,605},{0,34,605},{0,21,1009},{0,19,1781},{0,18,1161},{0,86,0},{0,86,0},{0,86,0},{0,42,1},{21,0,1625},{0,34,605},{0,34,605},{0,21,1009},{42,0,1625},{0,21,1009},{69,0,2665},{0,93,32},{11,64,1},
{0,59,628},{69,0,2665},{127,7,2665},{0,59,628},{0,47,2665},{127,7,2665},{0,47,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{3,127,3157},{0,102,2},{2,71,304},{0,62,514},{0,103,9669},{0,67,4865},{0,60,2107},{0,42,6485},{0,46,10689},{0,39,7245},{6,127,3029},{0,102,2},{5,71,260},{0,62,514},{50,1,9669},
{0,67,4865},{0,60,2107},{0,42,6485},{103,0,9669},{0,42,6485},{0,100,0},{0,100,0},{0,100,0},{0,49,0},{0,50,2178},{0,40,820},{0,40,820},{0,24,1348},{0,22,2392},{0,21,1556},{0,100,0},{0,100,0},{0,100,0},{0,49,0},{24,1,2178},{0,40,820},{0,40,820},{0,24,1348},{50,0,2178},{0,24,1348},{76,0,2665},{0,102,2},{15,69,0},{0,62,514},{76,0,2665},{126,14,2665},{0,62,514},
{0,51,2665},{126,14,2665},{0,51,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{6,127,3509},{2,108,32},{4,76,424},{1,67,473},{0,115,9669},{0,73,4529},{0,64,1649},{0,45,6209},{0,51,10919},{0,45,7109},{12,127,3141},{4,106,2},{9,75,260},{1,67,469},{56,1,9669},{0,73,4529},{0,64,1649},{0,45,6209},{115,0,9669},
{0,45,6209},{2,108,32},{2,108,32},{2,108,32},{2,54,32},{0,62,2178},{0,46,628},{0,46,628},{0,27,1184},{0,28,2520},{0,26,1502},{4,104,0},{4,104,0},{4,104,0},{4,53,0},{30,1,2178},{0,46,628},{0,46,628},{0,27,1184},{62,0,2178},{0,27,1184},{81,0,2665},{2,108,0},{19,73,0},{0,66,421},{81,0,2665},{126,20,2665},{0,66,421},{0,55,2665},{126,20,2665},{0,55,2665},{2,0,32},
{2,0,32},{2,0,32},{2,0,32},{0,12,0},{0,12,0},{0,12,0},{0,6,0},{0,5,10},{0,5,10},{12,127,3989},{4,114,130},{7,79,616},{3,72,525},{0,127,9669},{0,81,4178},{0,67,1265},{0,50,5950},{0,57,11199},{0,49,7029},{17,127,3253},{8,110,2},{13,79,260},{5,71,469},{62,1,9669},{0,81,4178},{0,67,1265},{0,50,5950},{127,0,9669},{0,50,5950},{4,116,128},{4,116,128},{4,116,128},
{4,59,128},{0,74,2178},{0,54,442},{0,54,442},{0,33,1040},{0,31,2676},{0,30,1460},{8,108,0},{8,108,0},{8,108,0},{8,57,0},{36,1,2178},{0,54,442},{0,54,442},{0,33,1040},{74,0,2178},{0,33,1040},{87,0,2665},{6,112,0},{23,77,0},{0,72,325},{87,0,2665},{126,26,2665},{0,72,325},{0,59,2665},{126,26,2665},{0,59,2665},{4,0,128},{4,0,128},{4,0,128},{4,0,128},{0,24,0},
{0,24,0},{0,24,0},{0,12,0},{0,8,50},{0,8,50},{15,127,4613},{6,118,304},{9,84,880},{4,76,645},{3,127,9761},{0,87,3898},{0,73,913},{0,56,5686},{0,63,11511},{0,53,6939},{20,127,3401},{12,114,2},{17,83,260},{9,75,469},{68,0,9669},{0,87,3898},{0,73,913},{0,56,5686},{127,6,9669},{0,56,5686},{6,124,288},{6,124,288},{6,124,288},{6,63,289},{0,86,2178},{0,60,306},{0,60,306},
{0,36,900},{0,37,2852},{0,34,1452},{12,112,0},{12,112,0},{12,112,0},{12,61,0},{42,1,2178},{0,60,306},{0,60,306},{0,36,900},{86,0,2178},{0,36,900},{93,0,2665},{10,116,0},{27,81,0},{0,77,260},{93,0,2665},{126,32,2665},{0,77,260},{0,63,2665},{126,32,2665},{0,63,2665},{6,0,288},{6,0,288},{6,0,288},{6,0,288},{0,36,0},{0,36,0},{0,36,0},{0,18,0},{0,14,106},
{0,14,106},{17,127,5446},{9,124,575},{12,90,1267},{8,80,862},{7,127,10073},{0,96,3625},{0,79,594},{0,59,5393},{0,70,11905},{0,57,6894},{26,127,3570},{16,119,3},{21,87,259},{14,80,470},{75,0,9669},{0,96,3625},{0,79,594},{0,59,5393},{127,13,9669},{0,59,5393},{9,127,558},{9,127,558},{9,127,558},{8,69,545},{0,100,2178},{0,67,185},{0,67,185},{0,42,754},{0,43,3090},{0,38,1478},{17,115,1},
{17,115,1},{17,115,1},{16,65,1},{49,0,2178},{0,67,185},{0,67,185},{0,42,754},{100,0,2178},{0,42,754},{100,0,2665},{15,120,0},{32,85,1},{0,82,193},{100,0,2665},{127,38,2665},{0,82,193},{0,67,2665},{127,38,2665},{0,67,2665},{8,0,545},{8,0,545},{8,0,545},{8,0,545},{0,50,0},{0,50,0},{0,50,0},{0,24,1},{0,19,193},{0,19,193},{20,127,6378},{12,127,915},{15,93,1695},
{9,84,1118},{12,127,10506},{0,102,3401},{0,84,366},{0,65,5186},{0,76,12289},{0,61,6870},{32,127,3738},{20,123,3},{25,91,259},{18,84,470},{80,1,9669},{0,102,3401},{0,84,366},{0,65,5186},{126,19,9669},{0,65,5186},{12,127,914},{12,127,914},{12,127,914},{10,74,841},{0,112,2178},{0,76,97},{0,76,97},{0,45,650},{0,51,3317},{0,43,1514},{21,119,1},{21,119,1},{21,119,1},{20,69,1},{55,0,2178},
{0,76,97},{0,76,97},{0,45,650},{112,0,2178},{0,45,650},{106,0,2665},{19,124,0},{36,89,1},{0,86,130},{106,0,2665},{127,44,2665},{0,86,130},{0,71,2665},{127,44,2665},{0,71,2665},{10,0,841},{10,0,841},{10,0,841},{10,0,841},{0,62,0},{0,62,0},{0,62,0},{0,30,1},{0,25,305},{0,25,305},{23,127,7454},{15,127,1431},{17,97,2214},{11,89,1438},{15,127,11102},{0,108,3209},{0,90,198},
{0,69,4913},{0,81,12639},{0,65,6875},{38,127,3938},{24,127,3},{29,95,259},{22,88,470},{86,1,9669},{0,108,3209},{0,90,198},{0,69,4913},{126,25,9669},{0,69,4913},{15,127,1382},{15,127,1382},{15,127,1382},{12,79,1201},{0,124,2178},{0,84,37},{0,84,37},{0,50,549},{0,54,3565},{0,46,1598},{25,123,1},{25,123,1},{25,123,1},{24,73,1},{61,0,2178},{0,84,37},{0,84,37},{0,50,549},{124,0,2178},
{0,50,549},{112,0,2665},{24,127,2},{40,93,1},{0,91,89},{112,0,2665},{127,50,2665},{0,91,89},{0,75,2665},{127,50,2665},{0,75,2665},{12,0,1201},{12,0,1201},{12,0,1201},{12,0,1201},{0,74,0},{0,74,0},{0,74,0},{0,36,1},{0,28,445},{0,28,445},{26,127,8674},{18,127,2131},{19,102,2786},{13,93,1839},{17,127,11833},{0,113,3038},{0,95,86},{0,72,4685},{0,84,13079},{0,70,6909},{41,127,4130},
{30,127,35},{33,99,259},{26,92,470},{92,1,9669},{0,113,3038},{0,95,86},{0,72,4685},{126,31,9669},{0,72,4685},{17,127,1941},{17,127,1941},{17,127,1941},{14,84,1625},{3,127,2222},{0,90,5},{0,90,5},{0,56,445},{0,60,3861},{0,51,1674},{29,127,1},{29,127,1},{29,127,1},{28,77,1},{66,1,2178},{0,90,5},{0,90,5},{0,56,445},{126,5,2178},{0,56,445},{118,0,2665},{31,127,20},{44,97,1},
{0,95,50},{118,0,2665},{127,56,2665},{0,95,50},{0,79,2665},{127,56,2665},{0,79,2665},{14,0,1625},{14,0,1625},{14,0,1625},{14,0,1625},{0,86,0},{0,86,0},{0,86,0},{0,42,1},{0,34,605},{0,34,605},{32,127,10209},{21,127,3140},{22,108,3525},{15,98,2360},{20,127,12902},{0,122,2885},{0,101,21},{0,78,4450},{0,93,13598},{0,73,6982},{47,127,4373},{34,127,134},{38,104,260},{30,96,469},{99,0,9669},
{0,122,2885},{0,101,21},{0,78,4450},{126,38,9669},{0,78,4450},{20,127,2717},{20,127,2717},{20,127,2717},{16,90,2180},{6,127,2420},{0,98,1},{0,98,1},{0,62,353},{0,64,4230},{0,57,1814},{34,126,13},{34,126,13},{34,126,13},{33,81,1},{73,0,2178},{2,96,1},{2,96,1},{0,62,353},{126,12,2178},{0,62,353},{125,0,2665},{40,127,74},{48,102,0},{0,101,20},{125,0,2665},{126,63,2665},{0,101,20},
{0,84,2665},{126,63,2665},{0,84,2665},{16,0,2180},{16,0,2180},{16,0,2180},{16,0,2180},{0,100,0},{0,100,0},{0,100,0},{0,49,0},{0,40,820},{0,40,820},{35,127,11582},{25,127,4131},{25,111,4166},{17,103,2825},{26,127,13903},{1,126,2826},{1,105,18},{0,83,4290},{0,96,13969},{0,79,6967},{52,127,4577},{40,127,270},{42,108,260},{34,100,469},{105,0,9669},{1,126,2825},{1,105,17},{0,83,4289},{126,44,9669},
{0,83,4289},{23,127,3434},{23,127,3434},{23,127,3434},{19,94,2666},{9,127,2709},{1,105,14},{1,105,14},{0,66,275},{0,70,4491},{0,60,1893},{38,127,29},{38,127,29},{38,127,29},{37,85,1},{79,0,2178},{6,100,1},{6,100,1},{0,66,274},{127,17,2178},{0,66,274},{127,7,2665},{46,127,146},{52,106,0},{0,105,8},{127,7,2665},{127,68,2665},{0,105,8},{0,88,2665},{127,68,2665},{0,88,2665},{19,0,2665},
{19,0,2665},{19,0,2665},{19,0,2665},{1,110,1},{1,110,1},{1,110,1},{0,55,1},{0,43,985},{0,43,985},{38,127,12090},{30,127,4561},{29,115,4166},{21,107,2825},{29,127,14311},{5,127,2910},{5,109,18},{0,86,4166},{0,105,13477},{0,84,6285},{58,127,4833},{46,127,470},{46,112,260},{38,104,469},{111,0,9669},{5,127,2909},{5,109,17},{0,86,4085},{127,49,9669},{0,86,4085},{26,127,3590},{26,127,3590},{26,127,3590},
{23,98,2666},{12,127,2853},{5,109,14},{5,109,14},{3,69,259},{0,76,4147},{0,64,1475},{44,127,61},{44,127,61},{44,127,61},{41,89,1},{85,0,2178},{10,104,1},{10,104,1},{0,69,202},{127,23,2178},{0,69,202},{127,19,2665},{54,127,260},{56,110,0},{0,110,0},{127,19,2665},{127,74,2665},{0,110,0},{0,92,2665},{127,74,2665},{0,92,2665},{23,0,2665},{23,0,2665},{23,0,2665},{23,0,2665},{5,114,1},
{5,114,1},{5,114,1},{4,59,1},{0,51,745},{0,51,745},{44,127,12610},{34,127,5039},{33,119,4166},{25,111,2825},{35,127,14719},{11,127,3118},{9,113,18},{4,90,4166},{0,113,13003},{0,87,5661},{61,127,5093},{51,127,736},{50,116,260},{42,108,469},{117,0,9669},{14,127,3073},{9,113,17},{0,89,3929},{127,55,9669},{0,89,3929},{32,127,3726},{32,127,3726},{32,127,3726},{27,102,2666},{17,127,2979},{9,113,14},{9,113,14},
{7,73,259},{0,84,3784},{0,70,1091},{47,127,97},{47,127,97},{47,127,97},{45,93,1},{91,0,2178},{14,108,1},{14,108,1},{0,75,146},{127,29,2178},{0,75,146},{127,31,2665},{60,127,388},{60,114,0},{3,114,0},{127,31,2665},{127,80,2665},{3,114,0},{0,96,2665},{127,80,2665},{0,96,2665},{27,0,2665},{27,0,2665},{27,0,2665},{27,0,2665},{9,118,1},{9,118,1},{9,118,1},{8,63,1},{0,60,565},
{0,60,565},{49,127,13154},{40,127,5661},{37,123,4166},{30,116,2823},{41,127,15213},{16,127,3497},{14,118,18},{8,94,4166},{0,116,12489},{0,93,5039},{68,127,5409},{57,127,1091},{54,120,259},{47,113,470},{124,0,9669},{22,127,3341},{14,118,14},{0,95,3726},{127,62,9669},{0,95,3726},{38,127,3929},{38,127,3929},{38,127,3929},{31,107,2665},{23,127,3156},{13,117,17},{13,117,17},{11,77,260},{0,90,3467},{0,76,736},{52,127,146},
{52,127,146},{52,127,146},{49,98,1},{98,0,2178},{19,112,1},{19,112,1},{0,80,97},{127,36,2178},{0,80,97},{127,45,2665},{67,127,565},{64,119,1},{9,118,1},{127,45,2665},{126,87,2665},{9,118,1},{0,100,2665},{126,87,2665},{0,100,2665},{31,0,2665},{31,0,2665},{31,0,2665},{31,0,2665},{13,123,0},{13,123,0},{13,123,0},{13,67,0},{0,67,388},{0,67,388},{52,127,13734},{43,127,6285},{41,127,4166},
{34,120,2823},{47,127,15677},{22,127,3905},{18,122,18},{12,98,4166},{0,125,12093},{0,97,4561},{71,127,5701},{60,127,1475},{58,124,259},{51,117,470},{127,5,9669},{31,127,3601},{18,122,14},{0,101,3590},{126,68,9669},{0,101,3590},{41,127,4085},{41,127,4085},{41,127,4085},{35,111,2665},{29,127,3332},{17,121,17},{17,121,17},{15,81,260},{0,99,3211},{0,81,470},{58,127,202},{58,127,202},{58,127,202},{53,102,1},{104,0,2178},
{23,116,1},{23,116,1},{0,83,61},{127,42,2178},{0,83,61},{127,57,2665},{76,127,745},{68,123,1},{13,122,1},{127,57,2665},{126,93,2665},{13,122,1},{0,104,2665},{126,93,2665},{0,104,2665},{35,0,2665},{35,0,2665},{35,0,2665},{35,0,2665},{17,127,0},{17,127,0},{17,127,0},{17,71,0},{0,73,260},{0,73,260},{58,127,14302},{48,127,6967},{44,127,4290},{38,124,2823},{52,127,16094},{28,127,4409},{22,126,18},
{16,102,4166},{0,126,11883},{0,102,4131},{77,127,6005},{67,127,1893},{61,127,275},{55,121,470},{127,18,9669},{40,127,3909},{22,126,14},{0,104,3434},{126,74,9669},{0,104,3434},{44,127,4289},{44,127,4289},{44,127,4289},{39,115,2665},{35,127,3540},{21,125,17},{21,125,17},{19,85,260},{0,105,2979},{0,87,270},{61,127,274},{61,127,274},{61,127,274},{57,106,1},{110,0,2178},{27,120,1},{27,120,1},{0,89,29},{126,48,2178},
{0,89,29},{127,69,2665},{81,127,985},{72,127,1},{17,126,1},{127,69,2665},{126,99,2665},{17,126,1},{0,108,2665},{126,99,2665},{0,108,2665},{39,0,2665},{39,0,2665},{39,0,2665},{39,0,2665},{22,126,8},{22,126,8},{22,126,8},{21,75,0},{0,81,146},{0,81,146},{61,127,13635},{53,127,6982},{49,127,4450},{42,125,2742},{55,127,15195},{34,127,4106},{26,127,21},{19,105,3525},{2,127,10776},{0,106,3140},{79,127,5396},
{70,127,1814},{65,127,353},{59,122,321},{127,27,8712},{46,127,3462},{29,127,1},{0,107,2717},{126,78,8712},{0,107,2717},{49,127,4450},{49,127,4450},{49,127,4450},{43,119,2665},{38,127,3736},{26,127,21},{26,127,21},{23,89,260},{0,113,2772},{0,93,134},{65,127,353},{65,127,353},{65,127,353},{61,110,1},{115,1,2178},{31,124,1},{31,124,1},{0,93,13},{126,54,2178},{0,93,13},{127,76,2178},{87,127,820},{78,127,0},
{27,127,0},{127,76,2178},{126,103,2178},{27,127,0},{0,110,2180},{126,103,2178},{0,110,2180},{43,0,2665},{43,0,2665},{43,0,2665},{43,0,2665},{26,127,20},{26,127,20},{26,127,20},{25,79,0},{0,87,74},{0,87,74},{65,127,12750},{57,127,6909},{55,127,4685},{48,126,2678},{61,127,14070},{37,127,3711},{32,127,86},{25,108,2786},{5,127,9739},{0,109,2131},{85,127,4658},{76,127,1674},{71,127,445},{64,123,173},{127,36,7578},
{51,127,2949},{37,127,5},{0,110,1941},{127,82,7578},{0,110,1941},{55,127,4685},{55,127,4685},{55,127,4685},{48,123,2666},{44,127,3987},{32,127,86},{32,127,86},{28,94,259},{0,119,2571},{0,97,35},{71,127,445},{71,127,445},{71,127,445},{66,114,1},{122,0,2178},{37,127,5},{37,127,5},{0,98,1},{126,61,2178},{0,98,1},{127,83,1625},{93,127,605},{84,127,1},{40,127,0},{127,83,1625},{126,106,1625},{40,127,0},
{0,113,1625},{126,106,1625},{0,113,1625},{48,0,2665},{48,0,2665},{48,0,2665},{48,0,2665},{32,127,50},{32,127,50},{32,127,50},{30,83,1},{0,96,20},{0,96,20},{68,127,12050},{62,127,6875},{58,127,4913},{52,127,2666},{65,127,13165},{43,127,3423},{37,127,198},{30,110,2214},{11,127,8987},{0,112,1431},{88,127,4082},{79,127,1547},{77,127,549},{68,124,81},{127,43,6661},{57,127,2525},{43,127,37},{0,112,1382},{127,86,6661},
{0,112,1382},{58,127,4913},{58,127,4913},{58,127,4913},{52,127,2666},{49,127,4197},{37,127,198},{37,127,198},{32,98,259},{0,125,2435},{0,103,3},{77,127,549},{77,127,549},{77,127,549},{70,118,1},{127,2,2178},{43,127,37},{43,127,37},{3,102,1},{127,66,2178},{3,102,1},{127,89,1201},{99,127,445},{90,127,1},{53,127,0},{127,89,1201},{125,109,1201},{53,127,0},{0,115,1201},{125,109,1201},{0,115,1201},{52,0,2665},
{52,0,2665},{52,0,2665},{52,0,2665},{36,127,89},{36,127,89},{36,127,89},{34,87,1},{0,103,2},{0,103,2},{74,127,11418},{65,127,6870},{62,127,5186},{56,127,2701},{68,127,12313},{46,127,3251},{43,127,366},{34,112,1695},{14,127,8383},{0,115,915},{91,127,3578},{84,127,1514},{79,127,650},{73,126,21},{127,53,5829},{63,127,2165},{51,127,97},{0,115,914},{126,91,5829},{0,115,914},{62,127,5186},{62,127,5186},{62,127,5186},
{56,127,2701},{55,127,4461},{43,127,366},{43,127,366},{36,102,259},{2,127,2427},{4,107,3},{79,127,650},{79,127,650},{79,127,650},{74,122,1},{127,14,2178},{51,127,97},{51,127,97},{7,106,1},{127,72,2178},{7,106,1},{127,95,841},{102,127,305},{96,127,1},{64,127,0},{127,95,841},{127,111,841},{64,127,0},{0,117,841},{127,111,841},{0,117,841},{56,0,2665},{56,0,2665},{56,0,2665},{56,0,2665},{41,127,130},
{41,127,130},{41,127,130},{38,91,1},{3,108,0},{3,108,0},{77,127,10830},{69,127,6894},{68,127,5393},{60,127,2786},{71,127,11565},{54,127,3154},{48,127,594},{37,115,1267},{25,127,7825},{3,118,575},{94,127,3146},{87,127,1478},{85,127,754},{77,127,1},{127,60,5082},{70,127,1869},{57,127,185},{0,118,558},{127,94,5082},{0,118,558},{68,127,5393},{68,127,5393},{68,127,5393},{60,127,2786},{58,127,4725},{48,127,594},{48,127,594},
{40,106,259},{8,127,2587},{8,111,3},{85,127,754},{85,127,754},{85,127,754},{78,126,1},{127,27,2178},{57,127,185},{57,127,185},{11,110,1},{126,78,2178},{11,110,1},{127,101,545},{108,127,193},{102,127,1},{76,127,0},{127,101,545},{127,114,545},{76,127,0},{0,119,545},{127,114,545},{0,119,545},{60,0,2665},{60,0,2665},{60,0,2665},{60,0,2665},{45,127,193},{45,127,193},{45,127,193},{42,95,1},{7,112,0},
{7,112,0},{79,127,10221},{74,127,6939},{71,127,5686},{64,127,2954},{77,127,10836},{57,127,3109},{54,127,913},{42,117,880},{34,127,7300},{8,120,304},{100,127,2736},{93,127,1452},{91,127,900},{83,127,29},{127,69,4344},{76,127,1611},{67,127,306},{2,121,288},{126,99,4344},{2,121,288},{71,127,5686},{71,127,5686},{71,127,5686},{64,127,2954},{65,127,5051},{54,127,913},{54,127,913},{44,110,260},{14,127,2856},{13,115,2},{91,127,900},
{91,127,900},{91,127,900},{83,127,29},{127,40,2178},{67,127,306},{67,127,306},{15,115,0},{126,85,2178},{15,115,0},{127,108,288},{113,127,106},{109,127,0},{90,127,0},{127,108,288},{127,118,288},{90,127,0},{0,121,288},{127,118,288},{0,121,288},{64,0,2665},{64,0,2665},{64,0,2665},{64,0,2665},{50,127,260},{50,127,260},{50,127,260},{46,100,0},{11,117,0},{11,117,0},{85,127,9781},{79,127,6942},{77,127,5950},
{70,127,3146},{79,127,10205},{63,127,3141},{57,127,1265},{47,119,616},{40,127,6924},{13,123,130},{103,127,2436},{97,127,1460},{94,127,1040},{87,127,104},{127,76,3779},{81,127,1496},{73,127,442},{10,123,128},{126,103,3779},{10,123,128},{77,127,5950},{77,127,5950},{77,127,5950},{70,127,3146},{68,127,5347},{57,127,1265},{57,127,1265},{48,114,260},{25,127,3115},{17,119,2},{94,127,1040},{94,127,1040},{94,127,1040},{87,127,104},{127,53,2178},
{73,127,442},{73,127,442},{19,119,0},{126,91,2178},{19,119,0},{127,115,128},{116,127,50},{115,127,0},{102,127,0},{127,115,128},{127,121,128},{102,127,0},{0,123,128},{127,121,128},{0,123,128},{68,0,2665},{68,0,2665},{68,0,2665},{68,0,2665},{55,127,325},{55,127,325},{55,127,325},{50,104,0},{15,121,0},{15,121,0},{88,127,9417},{82,127,7109},{82,127,6209},{73,127,3406},{85,127,9733},{67,127,3260},{63,127,1649},
{50,122,424},{46,127,6644},{19,125,32},{106,127,2208},{101,127,1502},{100,127,1184},{93,127,232},{127,86,3299},{87,127,1400},{81,127,628},{19,125,32},{127,107,3299},{19,125,32},{82,127,6209},{82,127,6209},{82,127,6209},{73,127,3406},{74,127,5659},{63,127,1649},{63,127,1649},{52,118,260},{34,127,3419},{21,123,2},{100,127,1184},{100,127,1184},{100,127,1184},{93,127,232},{127,64,2178},{81,127,628},{81,127,628},{23,123,0},{126,97,2178},
{23,123,0},{127,121,32},{122,127,10},{121,127,0},{115,127,0},{127,121,32},{127,124,32},{115,127,0},{0,125,32},{127,124,32},{0,125,32},{72,0,2665},{72,0,2665},{72,0,2665},{72,0,2665},{61,127,421},{61,127,421},{61,127,421},{54,108,0},{19,125,0},{19,125,0},{91,127,9133},{88,127,7245},{85,127,6485},{79,127,3710},{88,127,9325},{73,127,3460},{67,127,2107},{55,124,304},{54,127,6532},{25,127,2},{111,127,2038},
{106,127,1556},{103,127,1348},{96,127,416},{127,93,2904},{93,127,1368},{87,127,820},{27,127,0},{126,111,2904},{27,127,0},{85,127,6485},{85,127,6485},{85,127,6485},{79,127,3710},{79,127,5949},{67,127,2107},{67,127,2107},{56,122,260},{40,127,3771},{25,127,2},{103,127,1348},{103,127,1348},{103,127,1348},{96,127,416},{127,76,2178},{87,127,820},{87,127,820},{27,127,0},{126,103,2178},{27,127,0},{127,127,0},{127,127,0},{127,127,0},
{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{76,0,2665},{76,0,2665},{76,0,2665},{76,0,2665},{65,127,514},{65,127,514},{65,127,514},{58,112,0},{25,127,2},{25,127,2},{94,127,8049},{91,127,6469},{88,127,5849},{84,127,3561},{91,127,8053},{76,127,3106},{73,127,1965},{62,125,154},{60,127,5562},{34,127,32},{111,127,1507},{109,127,1161},{106,127,1009},{102,127,305},{127,98,2166},
{99,127,1009},{93,127,605},{40,127,0},{126,113,2166},{40,127,0},{88,127,5849},{88,127,5849},{88,127,5849},{84,127,3561},{82,127,5209},{73,127,1965},{73,127,1965},{61,124,126},{46,127,3225},{34,127,32},{106,127,1009},{106,127,1009},{106,127,1009},{102,127,305},{127,83,1625},{93,127,605},{93,127,605},{40,127,0},{126,106,1625},{40,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},
{0,127,0},{127,127,0},{0,127,0},{80,0,2665},{80,0,2665},{80,0,2665},{80,0,2665},{68,127,628},{68,127,628},{68,127,628},{63,116,1},{34,127,32},{34,127,32},{97,127,7165},{94,127,5845},{91,127,5333},{87,127,3401},{94,127,7033},{79,127,2823},{79,127,1862},{64,126,60},{64,127,4795},{40,127,80},{114,127,1107},{111,127,830},{111,127,730},{105,127,221},{127,102,1601},{102,127,737},{99,127,445},{53,127,0},{127,115,1601},
{53,127,0},{91,127,5333},{91,127,5333},{91,127,5333},{87,127,3401},{85,127,4629},{79,127,1862},{79,127,1862},{66,125,52},{54,127,2834},{40,127,80},{111,127,730},{111,127,730},{111,127,730},{105,127,221},{127,89,1201},{99,127,445},{99,127,445},{53,127,0},{125,109,1201},{53,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{84,0,2665},
{84,0,2665},{84,0,2665},{84,0,2665},{74,127,740},{74,127,740},{74,127,740},{66,120,0},{40,127,80},{40,127,80},{100,127,6361},{97,127,5277},{94,127,4865},{90,127,3265},{97,127,6117},{84,127,2641},{79,127,1814},{70,126,12},{70,127,4123},{46,127,160},{117,127,779},{114,127,595},{114,127,514},{108,127,153},{127,105,1121},{105,127,513},{102,127,305},{64,127,0},{126,117,1121},{64,127,0},{94,127,4865},{94,127,4865},{94,127,4865},
{90,127,3265},{91,127,4117},{79,127,1814},{79,127,1814},{70,126,8},{60,127,2474},{46,127,160},{114,127,514},{114,127,514},{114,127,514},{108,127,153},{127,95,841},{102,127,305},{102,127,305},{64,127,0},{127,111,841},{64,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{88,0,2665},{88,0,2665},{88,0,2665},{88,0,2665},{79,127,853},
{79,127,853},{79,127,853},{70,124,0},{46,127,160},{46,127,160},{103,127,5637},{100,127,4765},{100,127,4441},{93,127,3153},{100,127,5305},{90,127,2465},{86,127,1789},{74,127,4},{73,127,3543},{54,127,274},{117,127,507},{117,127,387},{115,127,337},{111,127,101},{127,110,726},{108,127,337},{108,127,193},{76,127,0},{126,119,726},{76,127,0},{100,127,4441},{100,127,4441},{100,127,4441},{93,127,3153},{94,127,3657},{86,127,1789},{86,127,1789},
{74,127,4},{67,127,2182},{54,127,274},{115,127,337},{115,127,337},{115,127,337},{111,127,101},{127,101,545},{108,127,193},{108,127,193},{76,127,0},{127,114,545},{76,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{92,0,2665},{92,0,2665},{92,0,2665},{92,0,2665},{82,127,985},{82,127,985},{82,127,985},{74,127,4},{54,127,274},
{54,127,274},{17,127,39416},{1,127,2073},{0,90,164},{0,86,5261},{17,127,49709},{0,113,21212},{0,85,8480},{0,75,26137},{0,87,56335},{0,67,37225},{12,127,10274},{0,122,1413},{0,90,148},{0,75,3940},{57,0,18065},{0,76,12036},{0,68,6166},{0,45,14098},{117,0,18065},{0,45,14098},{0,71,0},{0,71,0},{0,71,0},{0,35,0},{0,35,1105},{0,28,405},{0,28,405},{0,15,689},{0,14,1226},{0,15,789},{0,71,0},
{0,71,0},{0,71,0},{0,35,0},{17,0,1105},{0,28,405},{0,28,405},{0,15,689},{35,0,1105},{0,15,689},{85,0,9248},{0,122,1413},{0,90,148},{0,75,3940},{85,0,9248},{127,23,9248},{0,75,3940},{0,57,9250},{127,23,9248},{0,57,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{20,127,44736},{2,127,2677},{0,95,64},
{0,92,4941},{17,127,55213},{0,119,21820},{0,89,8426},{0,78,27209},{0,90,60583},{0,73,39465},{15,127,10870},{0,125,1213},{0,93,52},{0,78,3656},{61,0,19334},{0,81,12395},{0,73,6176},{0,50,14795},{124,0,19334},{0,50,14795},{0,83,0},{0,83,0},{0,83,0},{0,41,0},{0,41,1513},{0,34,565},{0,34,565},{0,18,937},{0,19,1661},{0,18,1081},{0,83,0},{0,83,0},{0,83,0},{0,41,0},{20,0,1513},
{0,34,565},{0,34,565},{0,18,937},{41,0,1513},{0,18,937},{91,0,9248},{0,125,1213},{0,93,52},{0,78,3656},{91,0,9248},{127,29,9248},{0,78,3656},{0,61,9250},{127,29,9248},{0,61,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{20,127,50624},{2,127,4005},{0,99,4},{0,95,4545},{20,127,61345},{0,125,22460},{0,93,8410},
{0,83,28480},{0,96,65039},{0,76,41789},{17,127,11556},{1,127,1217},{0,99,4},{0,83,3425},{64,0,20689},{0,87,12835},{0,76,6216},{0,53,15539},{126,3,20689},{0,53,15539},{0,95,0},{0,95,0},{0,95,0},{0,47,0},{0,47,1985},{0,37,745},{0,37,745},{0,21,1225},{0,22,2185},{0,21,1421},{0,95,0},{0,95,0},{0,95,0},{0,47,0},{23,0,1985},{0,37,745},{0,37,745},{0,21,1225},{47,0,1985},
{0,21,1225},{97,0,9248},{2,127,1205},{0,99,4},{0,83,3425},{97,0,9248},{127,35,9248},{0,83,3425},{0,65,9248},{127,35,9248},{0,65,9248},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{23,127,56952},{5,127,6081},{0,103,9},{0,98,4221},{20,127,65535},{0,126,23535},{0,97,8416},{0,86,29640},{0,99,65535},{0,79,44233},{20,127,12416},
{2,127,1397},{1,103,8},{0,86,3181},{69,0,22129},{0,90,13307},{0,81,6266},{0,56,16331},{127,7,22129},{0,56,16331},{0,107,0},{0,107,0},{0,107,0},{0,53,0},{0,53,2521},{0,43,953},{0,43,953},{0,24,1553},{0,22,2777},{0,24,1809},{0,107,0},{0,107,0},{0,107,0},{0,53,0},{26,0,2521},{0,43,953},{0,43,953},{0,24,1553},{53,0,2521},{0,24,1553},{103,0,9248},{2,127,1381},{2,103,0},
{0,86,3181},{103,0,9248},{127,41,9248},{0,86,3181},{0,69,9248},{127,41,9248},{0,69,9248},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{26,127,64790},{5,127,9105},{1,108,78},{0,104,3845},{23,127,65535},{2,127,25370},{0,102,8458},{0,89,31046},{0,105,65535},{0,81,47079},{23,127,13542},{5,127,1881},{2,109,50},{0,92,2897},{73,0,23851},
{0,96,13865},{0,87,6374},{0,59,17289},{126,12,23851},{0,59,17289},{0,121,0},{0,121,0},{0,121,0},{0,59,1},{0,61,3200},{0,51,1210},{0,51,1210},{0,27,1972},{0,25,3528},{0,27,2296},{0,121,0},{0,121,0},{0,121,0},{0,59,1},{30,0,3200},{0,51,1210},{0,51,1210},{0,27,1972},{61,0,3200},{0,27,1972},{110,0,9248},{11,127,1693},{7,107,1},{0,92,2897},{110,0,9248},{126,48,9248},{0,92,2897},
{0,73,9250},{126,48,9248},{0,73,9250},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{26,127,65535},{5,127,12609},{1,111,198},{0,107,3541},{26,127,65535},{2,127,27706},{0,105,8522},{0,92,32390},{0,113,65535},{0,87,49743},{26,127,14690},{8,127,2501},{3,113,126},{0,98,2665},{77,0,25472},{0,102,14385},{0,90,6486},{0,62,18185},{127,15,25472},
{0,62,18185},{0,127,16},{0,127,16},{0,127,16},{0,65,0},{0,66,3872},{0,51,1450},{0,51,1450},{0,30,2384},{0,28,4268},{0,27,2776},{1,127,13},{1,127,13},{1,127,13},{0,65,0},{33,0,3872},{0,51,1450},{0,51,1450},{0,30,2384},{66,0,3872},{0,30,2384},{115,1,9248},{19,127,2041},{11,111,1},{0,98,2665},{115,1,9248},{126,54,9248},{0,98,2665},{0,77,9250},{126,54,9248},{0,77,9250},{0,0,0},
{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{29,127,65535},{8,127,16605},{1,116,349},{0,112,3274},{26,127,65535},{2,127,30470},{0,109,8346},{0,98,33226},{0,113,65535},{0,90,52035},{29,127,15746},{11,127,3213},{5,117,217},{0,101,2445},{81,0,26744},{0,108,14657},{0,96,6398},{0,66,18739},{126,20,26744},{0,66,18739},{3,127,116},{3,127,116},{3,127,116},
{0,71,4},{0,74,4418},{0,57,1586},{0,57,1586},{0,33,2664},{0,31,4916},{0,30,3140},{3,127,80},{3,127,80},{3,127,80},{1,70,1},{36,1,4418},{0,57,1586},{0,57,1586},{0,33,2664},{74,0,4418},{0,33,2664},{121,1,9248},{25,127,2377},{15,115,1},{0,101,2441},{121,1,9248},{126,60,9248},{0,101,2441},{0,81,9250},{126,60,9248},{0,81,9250},{0,0,4},{0,0,4},{0,0,4},{0,0,4},{0,3,0},
{0,3,0},{0,3,0},{0,1,1},{1,0,2},{1,0,2},{32,127,65535},{11,127,21205},{2,120,605},{0,115,3106},{29,127,65535},{2,127,33062},{0,114,7330},{0,101,32522},{0,119,65535},{0,93,52995},{35,127,16210},{14,127,3809},{9,121,217},{0,107,2325},{87,0,26744},{0,116,14054},{0,99,5606},{0,72,18275},{126,26,26744},{0,72,18275},{6,127,292},{6,127,292},{6,127,292},{2,76,52},{0,86,4418},{0,64,1313},{0,64,1313},
{0,39,2440},{0,37,5092},{0,36,3044},{9,127,128},{9,127,128},{9,127,128},{5,74,1},{42,1,4418},{0,64,1313},{0,64,1313},{0,39,2440},{86,0,4418},{0,39,2440},{127,1,9248},{34,127,2741},{19,119,1},{0,107,2225},{127,1,9248},{127,65,9248},{0,107,2225},{0,85,9250},{127,65,9248},{0,85,9250},{2,0,52},{2,0,52},{2,0,52},{2,0,52},{0,15,0},{0,15,0},{0,15,0},{0,7,1},{0,5,18},
{0,5,18},{35,127,65535},{11,127,27154},{3,124,1078},{0,121,3041},{32,127,65535},{5,127,36765},{0,119,6285},{0,107,31809},{0,125,65535},{0,96,54246},{38,127,16729},{22,127,4594},{14,126,218},{4,111,2318},{94,0,26744},{0,122,13481},{0,105,4785},{0,75,17772},{127,32,26744},{0,75,17772},{9,127,614},{9,127,614},{9,127,614},{5,81,181},{0,100,4418},{0,73,1037},{0,73,1037},{0,42,2210},{0,43,5330},{0,41,2986},{13,127,193},
{13,127,193},{13,127,193},{9,79,1},{49,0,4418},{0,73,1037},{0,73,1037},{0,42,2210},{100,0,4418},{0,42,2210},{127,14,9248},{40,127,3177},{23,124,0},{0,110,2009},{127,14,9248},{127,72,9248},{0,110,2009},{0,90,9248},{127,72,9248},{0,90,9248},{5,0,181},{5,0,181},{5,0,181},{5,0,181},{0,29,0},{0,29,0},{0,29,0},{0,14,0},{0,11,65},{0,11,65},{35,127,65535},{14,127,31662},{4,127,1705},
{0,124,3025},{35,127,65535},{8,127,39325},{0,122,5245},{0,110,30421},{0,125,65535},{0,102,54322},{44,127,16749},{25,127,5034},{17,127,245},{7,114,2214},{99,0,26259},{0,125,12725},{0,111,3981},{0,78,16984},{126,38,26259},{0,78,16984},{10,127,1025},{10,127,1025},{10,127,1025},{7,86,365},{0,112,4418},{0,81,797},{0,81,797},{0,47,2025},{0,51,5557},{0,45,2950},{17,127,245},{17,127,245},{17,127,245},{13,83,1},{55,0,4418},
{0,81,797},{0,81,797},{0,47,2025},{112,0,4418},{0,47,2025},{127,25,8978},{46,127,3385},{28,127,0},{0,115,1732},{127,25,8978},{127,77,8978},{0,115,1732},{0,93,8980},{127,77,8978},{0,93,8980},{7,0,365},{7,0,365},{7,0,365},{7,0,365},{0,41,0},{0,41,0},{0,41,0},{0,20,0},{0,17,137},{0,17,137},{38,127,65535},{14,127,32078},{6,127,2618},{1,125,2789},{35,127,65535},{8,127,37485},{0,123,3805},
{0,110,27013},{0,125,65535},{0,102,51330},{49,127,15435},{31,127,4714},{23,127,317},{12,116,1830},{104,0,24371},{0,125,11205},{0,113,2881},{0,83,15117},{127,42,24371},{0,83,15117},{12,127,1538},{12,127,1538},{12,127,1538},{9,91,613},{0,124,4418},{0,87,605},{0,87,605},{0,53,1825},{0,54,5805},{0,49,2946},{23,127,317},{23,127,317},{23,127,317},{17,87,1},{61,0,4418},{0,87,605},{0,87,605},{0,53,1825},{124,0,4418},
{0,53,1825},{127,31,7938},{51,127,3029},{34,127,0},{0,118,1224},{127,31,7938},{127,80,7938},{0,118,1224},{0,95,7940},{127,80,7938},{0,95,7940},{9,0,613},{9,0,613},{9,0,613},{9,0,613},{0,53,0},{0,53,0},{0,53,0},{0,26,0},{0,22,221},{0,22,221},{38,127,65535},{14,127,32750},{6,127,3706},{3,124,2753},{35,127,65535},{8,127,35901},{0,124,2601},{0,110,23861},{0,125,65535},{0,102,48594},{52,127,14211},
{34,127,4410},{26,127,405},{16,117,1482},{107,1,22568},{2,127,9869},{0,116,1973},{0,86,13349},{126,46,22568},{0,86,13349},{17,127,2137},{17,127,2137},{17,127,2137},{11,96,925},{3,127,4462},{0,93,445},{0,93,445},{0,56,1649},{0,60,6101},{0,54,2966},{26,127,405},{26,127,405},{26,127,405},{21,91,1},{66,1,4418},{0,93,445},{0,93,445},{0,56,1649},{126,5,4418},{0,56,1649},{127,37,6962},{57,127,2645},{40,127,0},
{0,118,808},{127,37,6962},{127,83,6962},{0,118,808},{0,97,6964},{127,83,6962},{0,97,6964},{11,0,925},{11,0,925},{11,0,925},{11,0,925},{0,65,0},{0,65,0},{0,65,0},{0,32,0},{0,25,337},{0,25,337},{38,127,65535},{14,127,33812},{7,127,5233},{4,125,2961},{35,127,65535},{8,127,34425},{0,125,1509},{0,112,20542},{0,125,65535},{0,105,45810},{55,127,12917},{40,127,4114},{32,127,505},{22,117,1140},{112,0,20642},
{2,127,8609},{0,119,1157},{0,89,11489},{127,50,20642},{0,89,11489},{20,127,2969},{20,127,2969},{20,127,2969},{13,102,1352},{6,127,4660},{0,102,289},{0,102,289},{0,62,1445},{0,64,6470},{0,57,3018},{32,127,505},{32,127,505},{32,127,505},{26,95,1},{73,0,4418},{0,102,289},{0,102,289},{0,62,1445},{126,12,4418},{0,62,1445},{127,43,5941},{63,127,2248},{46,127,1},{0,121,433},{127,43,5941},{127,86,5941},{0,121,433},
{0,100,5941},{127,86,5941},{0,100,5941},{13,0,1352},{13,0,1352},{13,0,1352},{13,0,1352},{0,78,0},{0,78,0},{0,78,0},{0,39,0},{0,31,500},{0,31,500},{38,127,65535},{19,127,34934},{9,127,6748},{5,126,3381},{38,127,65535},{11,127,33369},{0,125,789},{0,112,17822},{0,125,65535},{0,105,43458},{58,127,11849},{43,127,3870},{38,127,617},{27,119,868},{115,1,19021},{5,127,7741},{0,120,621},{0,92,9957},{126,54,19021},
{0,92,9957},{23,127,3833},{23,127,3833},{23,127,3833},{15,107,1800},{9,127,4984},{0,108,185},{0,108,185},{0,66,1282},{0,70,6822},{0,63,3090},{38,127,617},{38,127,617},{38,127,617},{30,99,1},{79,0,4418},{0,108,185},{0,108,185},{0,66,1282},{127,17,4418},{0,66,1282},{127,50,5101},{67,127,1921},{52,127,0},{0,124,205},{127,50,5101},{127,89,5101},{0,124,205},{0,102,5101},{127,89,5101},{0,102,5101},{15,0,1800},
{15,0,1800},{15,0,1800},{15,0,1800},{0,91,0},{0,91,0},{0,91,0},{0,45,0},{0,37,676},{0,37,676},{41,127,65535},{19,127,36134},{9,127,8476},{5,126,4005},{38,127,65535},{11,127,32489},{0,126,309},{0,115,15210},{0,125,65535},{0,105,41362},{61,127,10853},{46,127,3674},{41,127,725},{31,120,632},{120,0,17485},{11,127,6965},{0,123,249},{0,95,8537},{127,58,17485},{0,95,8537},{23,127,4809},{23,127,4809},{23,127,4809},
{17,112,2312},{12,127,5444},{0,116,89},{0,116,89},{0,72,1130},{0,76,7206},{0,67,3194},{41,127,725},{41,127,725},{41,127,725},{34,103,1},{85,0,4418},{0,116,89},{0,116,89},{0,72,1130},{127,23,4418},{0,72,1130},{127,56,4325},{70,127,1625},{58,127,0},{0,124,61},{127,56,4325},{127,92,4325},{0,124,61},{0,104,4325},{127,92,4325},{0,104,4325},{17,0,2312},{17,0,2312},{17,0,2312},{17,0,2312},{0,103,0},
{0,103,0},{0,103,0},{0,50,1},{0,40,872},{0,40,872},{41,127,65535},{19,127,37590},{9,127,10460},{6,127,4841},{38,127,65535},{11,127,31865},{0,127,65},{0,115,12842},{0,125,65535},{0,105,39522},{65,127,9918},{51,127,3500},{47,127,853},{35,121,436},{124,0,16034},{14,127,6273},{0,125,53},{0,98,7229},{127,62,16034},{0,98,7229},{26,127,5881},{26,127,5881},{26,127,5881},{19,117,2888},{15,127,6040},{0,125,37},{0,125,37},
{0,75,986},{0,81,7565},{0,70,3318},{47,127,853},{47,127,853},{47,127,853},{38,107,1},{91,0,4418},{0,125,37},{0,125,37},{0,75,986},{127,29,4418},{0,75,986},{127,62,3613},{76,127,1345},{64,127,0},{0,127,1},{127,62,3613},{127,95,3613},{0,127,1},{0,106,3613},{127,95,3613},{0,106,3613},{19,0,2888},{19,0,2888},{19,0,2888},{19,0,2888},{0,115,0},{0,115,0},{0,115,0},{0,56,1},{0,46,1096},
{0,46,1096},{41,127,65535},{22,127,39522},{12,127,12842},{7,127,6030},{38,127,65535},{11,127,31469},{0,127,65},{0,118,10460},{0,125,65535},{0,108,37590},{68,127,8954},{57,127,3318},{52,127,986},{40,123,258},{127,2,14504},{22,127,5594},{2,127,37},{0,101,5881},{127,66,14504},{0,101,5881},{29,127,7229},{29,127,7229},{29,127,7229},{21,122,3614},{17,127,6824},{1,127,53},{1,127,53},{0,80,853},{0,87,8059},{0,76,3500},{52,127,986},
{52,127,986},{52,127,986},{42,112,1},{98,0,4418},{2,127,37},{2,127,37},{0,80,853},{127,36,4418},{0,80,853},{127,69,2888},{81,127,1096},{70,127,1},{11,127,0},{127,69,2888},{126,99,2888},{11,127,0},{0,108,2888},{126,99,2888},{0,108,2888},{21,0,3613},{21,0,3613},{21,0,3613},{21,0,3613},{0,127,1},{0,127,1},{0,127,1},{0,63,0},{0,51,1345},{0,51,1345},{41,127,65535},{22,127,41362},{12,127,15210},
{8,127,7309},{41,127,65535},{11,127,31389},{1,127,309},{0,118,8476},{0,125,65535},{0,108,36134},{71,127,8174},{60,127,3194},{55,127,1130},{45,123,137},{127,10,13235},{28,127,5018},{11,127,89},{0,101,4809},{126,70,13235},{0,101,4809},{32,127,8537},{32,127,8537},{32,127,8537},{23,127,4326},{20,127,7700},{4,127,249},{4,127,249},{0,86,725},{0,93,8531},{0,81,3674},{55,127,1130},{55,127,1130},{55,127,1130},{46,116,1},{104,0,4418},
{11,127,89},{11,127,89},{0,86,725},{127,42,4418},{0,86,725},{127,75,2312},{87,127,872},{76,127,1},{24,127,0},{127,75,2312},{126,102,2312},{24,127,0},{0,110,2312},{126,102,2312},{0,110,2312},{23,0,4325},{23,0,4325},{23,0,4325},{23,0,4325},{3,127,61},{3,127,61},{3,127,61},{0,69,0},{0,54,1625},{0,54,1625},{44,127,65535},{22,127,43458},{15,127,17822},{8,127,8829},{41,127,65535},{14,127,31497},{2,127,789},
{0,118,6748},{0,125,65535},{0,108,34934},{77,127,7454},{64,127,3090},{61,127,1282},{50,125,53},{127,19,12051},{34,127,4506},{16,127,185},{0,104,3833},{127,74,12051},{0,104,3833},{35,127,9957},{35,127,9957},{35,127,9957},{25,127,5150},{23,127,8712},{5,127,621},{5,127,621},{0,89,617},{0,96,9035},{0,84,3870},{61,127,1282},{61,127,1282},{61,127,1282},{50,120,1},{110,0,4418},{16,127,185},{16,127,185},{0,89,617},{126,48,4418},
{0,89,617},{127,81,1800},{90,127,676},{82,127,0},{36,127,0},{127,81,1800},{126,105,1800},{36,127,0},{0,112,1800},{126,105,1800},{0,112,1800},{25,0,5101},{25,0,5101},{25,0,5101},{25,0,5101},{3,127,205},{3,127,205},{3,127,205},{0,75,0},{0,60,1921},{0,60,1921},{44,127,65535},{22,127,45810},{15,127,20542},{10,127,10546},{41,127,65535},{14,127,31833},{2,127,1509},{0,120,5233},{0,125,65535},{0,113,33812},{79,127,6740},
{70,127,3018},{65,127,1445},{54,126,9},{127,27,10952},{40,127,4058},{25,127,289},{0,107,2969},{126,78,10952},{0,107,2969},{38,127,11489},{38,127,11489},{38,127,11489},{28,127,6114},{26,127,9860},{8,127,1157},{8,127,1157},{0,95,505},{0,102,9555},{0,87,4114},{65,127,1445},{65,127,1445},{65,127,1445},{54,124,1},{115,1,4418},{25,127,289},{25,127,289},{0,95,505},{126,54,4418},{0,95,505},{127,87,1352},{96,127,500},{88,127,0},
{48,127,0},{127,87,1352},{126,108,1352},{48,127,0},{0,114,1352},{126,108,1352},{0,114,1352},{27,0,5941},{27,0,5941},{27,0,5941},{27,0,5941},{6,127,433},{6,127,433},{6,127,433},{0,80,1},{0,64,2248},{0,64,2248},{44,127,65535},{25,127,48594},{17,127,23861},{11,127,12725},{41,127,65535},{14,127,32517},{3,127,2601},{0,121,3706},{0,126,65535},{0,113,32750},{82,127,6098},{73,127,2966},{71,127,1649},{59,127,4},{127,36,9818},
{46,127,3638},{34,127,445},{0,110,2137},{127,82,9818},{0,110,2137},{41,127,13349},{41,127,13349},{41,127,13349},{31,127,7380},{29,127,11310},{11,127,1973},{11,127,1973},{0,101,405},{0,108,10181},{0,93,4410},{71,127,1649},{71,127,1649},{71,127,1649},{59,127,4},{122,0,4418},{34,127,445},{34,127,445},{0,101,405},{126,61,4418},{0,101,405},{127,93,925},{102,127,337},{95,127,0},{62,127,0},{127,93,925},{126,111,925},{62,127,0},
{0,116,925},{126,111,925},{0,116,925},{29,0,6964},{29,0,6964},{29,0,6964},{29,0,6964},{9,127,808},{9,127,808},{9,127,808},{0,87,0},{0,70,2645},{0,70,2645},{44,127,65535},{25,127,51330},{17,127,27013},{11,127,14917},{41,127,65535},{14,127,33397},{4,127,3805},{0,121,2618},{1,126,65535},{0,113,32078},{88,127,5594},{79,127,2891},{74,127,1825},{64,127,45},{127,43,8901},{54,127,3373},{40,127,605},{0,115,1538},{127,86,8901},
{0,115,1538},{44,127,15117},{44,127,15117},{44,127,15117},{34,127,8664},{32,127,12746},{14,127,2881},{14,127,2881},{0,104,317},{0,113,10694},{0,96,4714},{74,127,1825},{74,127,1825},{74,127,1825},{64,127,45},{127,2,4418},{40,127,605},{40,127,605},{0,104,317},{127,66,4418},{0,104,317},{127,99,613},{105,127,221},{101,127,0},{73,127,0},{127,99,613},{126,114,613},{73,127,0},{0,118,613},{126,114,613},{0,118,613},{31,0,7940},
{31,0,7940},{31,0,7940},{31,0,7940},{9,127,1224},{9,127,1224},{9,127,1224},{0,93,0},{0,76,3029},{0,76,3029},{49,127,65535},{25,127,54322},{17,127,30421},{13,127,17348},{44,127,65535},{14,127,34533},{5,127,5245},{0,123,1705},{2,127,65535},{0,113,31662},{91,127,5146},{82,127,2950},{79,127,1994},{67,127,145},{127,53,8069},{60,127,3125},{46,127,797},{0,117,1025},{126,91,8069},{0,117,1025},{49,127,16984},{49,127,16984},{49,127,16984},
{37,127,10100},{35,127,14318},{16,127,3981},{16,127,3981},{0,110,245},{0,119,11302},{0,102,5034},{79,127,1994},{79,127,1994},{79,127,1994},{67,127,145},{127,14,4418},{46,127,797},{46,127,797},{0,110,245},{127,72,4418},{0,110,245},{127,105,365},{111,127,136},{107,127,0},{86,127,0},{127,105,365},{126,117,365},{86,127,0},{0,120,365},{126,117,365},{0,120,365},{33,0,8980},{33,0,8980},{33,0,8980},{33,0,8980},{12,127,1732},
{12,127,1732},{12,127,1732},{0,99,0},{0,81,3385},{0,81,3385},{49,127,65535},{28,127,54246},{20,127,31809},{16,127,18396},{49,127,65535},{19,127,34118},{8,127,6285},{3,124,1078},{2,127,65535},{0,116,27154},{94,127,4770},{86,127,2986},{82,127,2210},{73,127,281},{127,60,7322},{67,127,2945},{54,127,1037},{0,118,614},{127,94,7322},{0,118,614},{52,127,17772},{52,127,17772},{52,127,17772},{40,127,10764},{38,127,15150},{22,127,4785},{22,127,4785},
{1,113,218},{0,125,10994},{0,105,4594},{82,127,2210},{82,127,2210},{82,127,2210},{73,127,281},{127,27,4418},{54,127,1037},{54,127,1037},{0,114,193},{126,78,4418},{0,114,193},{127,111,181},{116,127,65},{113,127,0},{98,127,0},{127,111,181},{126,120,181},{98,127,0},{0,122,181},{126,120,181},{0,122,181},{37,0,9248},{37,0,9248},{37,0,9248},{37,0,9248},{17,127,2009},{17,127,2009},{17,127,2009},{3,104,0},{0,87,3177},
{0,87,3177},{52,127,65535},{34,127,52995},{26,127,32522},{21,127,19126},{49,127,65535},{22,127,32935},{13,127,7330},{7,125,605},{2,127,64493},{0,116,21205},{97,127,4436},{91,127,3044},{88,127,2440},{79,127,505},{127,69,6584},{73,127,2811},{63,127,1313},{0,121,292},{126,99,6584},{0,121,292},{55,127,18275},{55,127,18275},{55,127,18275},{46,127,11259},{44,127,15797},{28,127,5606},{28,127,5606},{6,118,217},{0,125,10589},{0,113,3809},{88,127,2440},
{88,127,2440},{88,127,2440},{79,127,505},{127,40,4418},{63,127,1313},{63,127,1313},{0,118,128},{126,85,4418},{0,118,128},{127,119,50},{122,127,18},{119,127,1},{111,127,0},{127,119,50},{127,123,50},{111,127,0},{0,124,52},{127,123,50},{0,124,52},{41,0,9250},{41,0,9250},{41,0,9250},{41,0,9250},{20,127,2225},{20,127,2225},{20,127,2225},{8,108,1},{0,93,2741},{0,93,2741},{55,127,65535},{37,127,52035},{29,127,33226},
{25,127,19751},{52,127,65535},{28,127,31983},{18,127,8346},{11,126,349},{2,127,60061},{0,119,16605},{100,127,4216},{97,127,3140},{94,127,2664},{84,127,776},{127,76,6019},{76,127,2779},{70,127,1586},{0,124,116},{126,103,6019},{0,124,116},{61,127,18739},{61,127,18739},{61,127,18739},{51,127,11794},{49,127,16315},{31,127,6398},{31,127,6398},{10,122,217},{2,127,10565},{0,116,3213},{94,127,2664},{94,127,2664},{94,127,2664},{84,127,776},{127,53,4418},
{70,127,1586},{70,127,1586},{0,124,80},{126,91,4418},{0,124,80},{127,125,2},{127,126,2},{125,127,1},{124,127,0},{127,125,2},{127,126,2},{124,127,0},{0,126,4},{127,126,2},{0,126,4},{45,0,9250},{45,0,9250},{45,0,9250},{45,0,9250},{26,127,2441},{26,127,2441},{26,127,2441},{12,112,1},{0,102,2377},{0,102,2377},{58,127,65535},{40,127,49743},{35,127,32390},{28,127,19671},{55,127,65535},{31,127,30383},{22,127,8522},
{15,126,198},{2,127,55505},{0,122,12609},{103,127,3648},{100,127,2776},{97,127,2384},{87,127,740},{127,83,5163},{84,127,2468},{73,127,1450},{0,127,16},{126,106,5163},{0,127,16},{65,127,18185},{65,127,18185},{65,127,18185},{54,127,11714},{52,127,15699},{37,127,6486},{37,127,6486},{14,124,126},{2,127,9785},{0,119,2501},{97,127,2384},{97,127,2384},{97,127,2384},{87,127,740},{127,60,3872},{73,127,1450},{73,127,1450},{0,126,13},{127,94,3872},
{0,126,13},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{49,0,9250},{49,0,9250},{49,0,9250},{49,0,9250},{29,127,2665},{29,127,2665},{29,127,2665},{16,116,1},{0,108,2041},{0,108,2041},{61,127,65535},{46,127,47079},{38,127,31046},{33,127,19370},{58,127,65535},{34,127,28647},{25,127,8458},{19,126,78},{2,127,51393},{0,122,9105},{106,127,3012},
{100,127,2296},{100,127,1972},{90,127,612},{127,86,4267},{87,127,2028},{76,127,1210},{5,127,0},{127,107,4267},{5,127,0},{65,127,17289},{65,127,17289},{65,127,17289},{57,127,11462},{55,127,14739},{40,127,6374},{40,127,6374},{18,125,50},{2,127,8937},{0,122,1881},{100,127,1972},{100,127,1972},{100,127,1972},{90,127,612},{127,66,3200},{76,127,1210},{76,127,1210},{5,127,0},{127,97,3200},{5,127,0},{127,127,0},{127,127,0},{127,127,0},
{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{53,0,9250},{53,0,9250},{53,0,9250},{53,0,9250},{35,127,2897},{35,127,2897},{35,127,2897},{20,120,1},{0,116,1693},{0,116,1693},{65,127,65535},{48,127,44233},{41,127,29640},{37,127,18961},{61,127,65535},{40,127,26823},{30,127,8416},{24,127,9},{8,127,47133},{0,122,6081},{109,127,2377},{103,127,1809},{103,127,1553},{96,127,481},{127,90,3361},
{90,127,1593},{84,127,953},{19,127,0},{127,109,3361},{19,127,0},{71,127,16331},{71,127,16331},{71,127,16331},{63,127,11212},{61,127,13721},{46,127,6266},{46,127,6266},{24,126,8},{5,127,8245},{0,125,1397},{103,127,1553},{103,127,1553},{103,127,1553},{96,127,481},{127,72,2521},{84,127,953},{84,127,953},{19,127,0},{127,100,2521},{19,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},
{0,127,0},{127,127,0},{0,127,0},{58,0,9248},{58,0,9248},{58,0,9248},{58,0,9248},{41,127,3181},{41,127,3181},{41,127,3181},{24,125,0},{0,125,1381},{0,125,1381},{65,127,65535},{51,127,41789},{44,127,28480},{40,127,18673},{65,127,65535},{43,127,25251},{34,127,8410},{28,127,4},{14,127,43557},{0,125,4005},{111,127,1843},{106,127,1421},{106,127,1225},{99,127,373},{127,95,2646},{96,127,1241},{90,127,745},{31,127,0},{127,111,2646},
{31,127,0},{74,127,15539},{74,127,15539},{74,127,15539},{64,127,10979},{65,127,12914},{51,127,6216},{51,127,6216},{28,127,4},{11,127,7709},{0,126,1217},{106,127,1225},{106,127,1225},{106,127,1225},{99,127,373},{127,78,1985},{90,127,745},{90,127,745},{31,127,0},{127,103,1985},{31,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{62,0,9248},
{62,0,9248},{62,0,9248},{62,0,9248},{44,127,3425},{44,127,3425},{44,127,3425},{28,127,4},{0,125,1205},{0,125,1205},{68,127,65535},{54,127,39465},{49,127,27209},{46,127,18393},{65,127,64563},{46,127,23799},{38,127,8426},{32,127,64},{14,127,40437},{0,125,2677},{111,127,1411},{111,127,1070},{109,127,937},{102,127,281},{127,99,2017},{99,127,937},{93,127,565},{43,127,0},{126,114,2017},{43,127,0},{77,127,14795},{77,127,14795},{77,127,14795},
{70,127,10779},{68,127,12146},{54,127,6176},{54,127,6176},{34,127,52},{14,127,7281},{2,127,1213},{109,127,937},{109,127,937},{109,127,937},{102,127,281},{127,84,1513},{93,127,565},{93,127,565},{43,127,0},{127,106,1513},{43,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{65,0,9250},{65,0,9250},{65,0,9250},{65,0,9250},{49,127,3656},
{49,127,3656},{49,127,3656},{34,127,52},{2,127,1213},{2,127,1213},{71,127,63180},{60,127,37225},{52,127,26137},{48,127,18128},{68,127,59595},{51,127,22636},{42,127,8480},{37,127,164},{22,127,37455},{0,126,2073},{114,127,1019},{111,127,766},{111,127,666},{105,127,205},{127,102,1473},{102,127,681},{99,127,405},{56,127,0},{127,115,1473},{56,127,0},{79,127,14066},{79,127,14066},{79,127,14066},{73,127,10571},{71,127,11450},{59,127,6166},{59,127,6166},
{37,127,148},{25,127,6914},{5,127,1413},{111,127,666},{111,127,666},{111,127,666},{105,127,205},{127,90,1105},{99,127,405},{99,127,405},{56,127,0},{127,109,1105},{56,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{127,127,0},{0,127,0},{127,127,0},{0,127,0},{69,0,9250},{69,0,9250},{69,0,9250},{69,0,9250},{52,127,3940},{52,127,3940},{52,127,3940},{37,127,148},{5,127,1413},
{5,127,1413},
/**** ended inlining basisu_transcoder_tables_bc7_m5_color.inc ****/
	};
	
	static dxt_selector_range g_etc1_to_bc7_m5a_selector_ranges[] =
	{
		{ 0, 3 },
		{ 1, 3 },
		{ 0, 2 },
		{ 1, 2 },
		{ 2, 3 },
		{ 0, 1 }
	};

	const uint32_t NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES = sizeof(g_etc1_to_bc7_m5a_selector_ranges) / sizeof(g_etc1_to_bc7_m5a_selector_ranges[0]);

	static uint32_t g_etc1_to_bc7_m5a_selector_range_index[4][4];

	struct etc1_g_to_bc7_m5a_conversion
	{
		uint8_t m_lo, m_hi;
		uint8_t m_trans;
	};

	static etc1_g_to_bc7_m5a_conversion g_etc1_g_to_bc7_m5a[8 * 32 * NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES] =
	{
/**** start inlining basisu_transcoder_tables_bc7_m5_alpha.inc ****/
{7,0,47},{7,0,44},{2,0,15},{2,0,12},{8,2,48},{0,0,0},{15,0,27},{15,1,24},{9,0,7},{15,1,24},{23,3,96},{6,0,3},{24,7,27},{24,7,24},{18,3,6},{18,5,4},{24,5,16},{21,1,6},{32,15,27},{32,15,24},{26,11,6},{26,13,4},{32,13,16},{22,3,1},{41,24,27},{41,24,24},{35,20,6},{35,22,4},{41,22,16},{31,12,1},{49,32,27},
{49,32,24},{43,28,6},{43,30,4},{49,30,16},{39,20,1},{57,40,27},{57,40,24},{51,36,6},{51,38,4},{57,38,16},{47,28,1},{65,48,27},{65,48,24},{59,44,6},{59,46,4},{65,46,16},{55,36,1},{74,57,27},{74,57,24},{68,53,6},{68,55,4},{74,55,16},{64,45,1},{82,65,27},{82,65,24},{76,61,6},{76,63,4},{82,63,16},{72,53,1},{90,73,27},{90,73,24},{84,69,6},
{84,71,4},{90,71,16},{80,61,1},{98,81,27},{98,81,24},{92,77,6},{92,79,4},{98,79,16},{88,69,1},{107,90,27},{107,90,24},{101,86,6},{101,88,4},{107,88,16},{97,78,1},{115,98,27},{115,98,24},{109,94,6},{109,96,4},{115,96,16},{105,86,1},{123,106,27},{123,106,24},{117,102,6},{117,104,4},{123,104,16},{113,94,1},{131,114,27},{131,114,24},{125,110,6},{125,112,4},{131,112,16},
{121,102,1},{140,123,27},{140,123,24},{134,119,6},{134,121,4},{140,121,16},{130,111,1},{148,131,27},{148,131,24},{142,127,6},{142,129,4},{148,129,16},{138,119,1},{156,139,27},{156,139,24},{150,135,6},{150,137,4},{156,137,16},{146,127,1},{164,147,27},{164,147,24},{158,143,6},{158,145,4},{164,145,16},{154,135,1},{173,156,27},{173,156,24},{167,152,6},{167,154,4},{173,154,16},{163,144,1},{181,164,27},
{181,164,24},{175,160,6},{175,162,4},{181,162,16},{171,152,1},{189,172,27},{189,172,24},{183,168,6},{183,170,4},{189,170,16},{179,160,1},{197,180,27},{197,180,24},{191,176,6},{191,178,4},{197,178,16},{187,168,1},{206,189,27},{206,189,24},{200,185,6},{200,187,4},{206,187,16},{196,177,1},{214,197,27},{214,197,24},{208,193,6},{208,195,4},{214,195,16},{204,185,1},{222,205,27},{222,205,24},{216,201,6},
{216,203,4},{222,203,16},{212,193,1},{230,213,27},{230,213,24},{224,209,6},{224,211,4},{230,211,16},{220,201,1},{239,222,27},{239,222,24},{233,218,6},{233,220,4},{239,220,16},{229,210,1},{247,230,27},{247,230,24},{241,226,6},{241,228,4},{247,228,16},{237,218,1},{255,238,27},{255,238,24},{249,234,6},{249,236,4},{255,236,16},{245,226,1},{255,247,7},{255,248,4},{255,247,7},{255,248,4},{255,0,0},
{253,234,1},{16,0,47},{16,0,44},{5,0,15},{5,0,12},{17,5,48},{0,0,0},{37,1,111},{36,2,108},{13,0,11},{13,3,12},{37,1,96},{3,0,3},{33,0,27},{33,0,24},{31,0,27},{31,1,24},{45,9,96},{11,0,3},{40,7,27},{41,8,24},{39,7,27},{29,0,4},{41,3,16},{19,1,2},{49,16,27},{50,17,24},{38,5,6},{38,6,4},{50,12,16},{40,4,6},{57,24,27},
{58,25,24},{46,13,6},{46,14,4},{58,20,16},{36,0,1},{65,32,27},{66,33,24},{54,21,6},{54,22,4},{66,28,16},{44,6,1},{73,40,27},{74,41,24},{62,29,6},{62,30,4},{74,36,16},{52,14,1},{82,49,27},{83,50,24},{71,38,6},{71,39,4},{83,45,16},{61,23,1},{90,57,27},{91,58,24},{79,46,6},{79,47,4},{91,53,16},{69,31,1},{98,65,27},{99,66,24},{87,54,6},
{87,55,4},{99,61,16},{77,39,1},{106,73,27},{107,74,24},{95,62,6},{95,63,4},{107,69,16},{85,47,1},{115,82,27},{116,83,24},{104,71,6},{104,72,4},{116,78,16},{94,56,1},{123,90,27},{124,91,24},{112,79,6},{112,80,4},{124,86,16},{102,64,1},{131,98,27},{132,99,24},{120,87,6},{120,88,4},{132,94,16},{110,72,1},{139,106,27},{140,107,24},{128,95,6},{128,96,4},{140,102,16},
{118,80,1},{148,115,27},{149,116,24},{137,104,6},{137,105,4},{149,111,16},{127,89,1},{156,123,27},{157,124,24},{145,112,6},{145,113,4},{157,119,16},{135,97,1},{164,131,27},{165,132,24},{153,120,6},{153,121,4},{165,127,16},{143,105,1},{172,139,27},{173,140,24},{161,128,6},{161,129,4},{173,135,16},{151,113,1},{181,148,27},{182,149,24},{170,137,6},{170,138,4},{182,144,16},{160,122,1},{189,156,27},
{190,157,24},{178,145,6},{178,146,4},{190,152,16},{168,130,1},{197,164,27},{198,165,24},{186,153,6},{186,154,4},{198,160,16},{176,138,1},{205,172,27},{206,173,24},{194,161,6},{194,162,4},{206,168,16},{184,146,1},{214,181,27},{215,182,24},{203,170,6},{203,171,4},{215,177,16},{193,155,1},{222,189,27},{223,190,24},{211,178,6},{211,179,4},{223,185,16},{201,163,1},{230,197,27},{231,198,24},{219,186,6},
{219,187,4},{231,193,16},{209,171,1},{238,205,27},{239,206,24},{227,194,6},{227,195,4},{239,201,16},{217,179,1},{247,214,27},{248,215,24},{236,203,6},{236,204,4},{248,210,16},{226,188,1},{255,222,27},{255,222,24},{244,211,6},{244,212,4},{255,220,16},{234,196,1},{254,218,6},{255,242,28},{252,219,6},{252,220,4},{255,245,16},{242,204,1},{255,238,7},{255,239,4},{255,238,7},{255,239,4},{255,0,0},
{250,212,1},{28,0,47},{28,0,44},{9,0,15},{9,0,12},{29,9,48},{0,0,0},{53,0,111},{53,0,108},{17,0,15},{17,0,12},{37,7,32},{0,0,0},{66,3,111},{63,7,108},{25,0,11},{25,7,12},{65,5,96},{7,0,3},{51,0,27},{71,15,108},{47,0,27},{33,6,8},{73,13,96},{15,0,3},{61,4,27},{80,24,108},{61,4,27},{60,6,24},{62,0,16},{24,4,3},{69,12,27},
{88,32,108},{69,12,27},{68,14,24},{70,8,16},{32,2,2},{77,20,27},{96,40,108},{58,2,6},{58,2,4},{78,16,16},{60,0,6},{85,28,27},{104,48,108},{66,10,6},{66,10,4},{86,24,16},{68,8,6},{94,37,27},{113,57,108},{75,19,6},{75,19,4},{95,33,16},{77,17,6},{102,45,27},{121,65,108},{83,27,6},{83,27,4},{103,41,16},{65,3,1},{110,53,27},{129,73,108},{91,35,6},
{91,35,4},{111,49,16},{73,11,1},{118,61,27},{137,81,108},{99,43,6},{99,43,4},{119,57,16},{81,19,1},{127,70,27},{146,90,108},{108,52,6},{108,52,4},{128,66,16},{90,28,1},{135,78,27},{154,98,108},{116,60,6},{116,60,4},{136,74,16},{98,36,1},{143,86,27},{162,106,108},{124,68,6},{124,68,4},{144,82,16},{106,44,1},{151,94,27},{170,114,108},{132,76,6},{132,76,4},{152,90,16},
{114,52,1},{160,103,27},{179,123,108},{141,85,6},{141,85,4},{161,99,16},{123,61,1},{168,111,27},{187,131,108},{149,93,6},{149,93,4},{169,107,16},{131,69,1},{176,119,27},{195,139,108},{157,101,6},{157,101,4},{177,115,16},{139,77,1},{184,127,27},{203,147,108},{165,109,6},{165,109,4},{185,123,16},{147,85,1},{193,136,27},{212,156,108},{174,118,6},{174,118,4},{194,132,16},{156,94,1},{201,144,27},
{220,164,108},{182,126,6},{182,126,4},{202,140,16},{164,102,1},{209,152,27},{228,172,108},{190,134,6},{190,134,4},{210,148,16},{172,110,1},{217,160,27},{236,180,108},{198,142,6},{198,142,4},{218,156,16},{180,118,1},{226,169,27},{245,189,108},{207,151,6},{207,151,4},{227,165,16},{189,127,1},{234,177,27},{253,197,108},{215,159,6},{215,159,4},{235,173,16},{197,135,1},{242,185,27},{243,186,24},{223,167,6},
{223,167,4},{243,181,16},{205,143,1},{250,193,27},{251,194,24},{231,175,6},{231,175,4},{251,189,16},{213,151,1},{255,203,27},{255,206,24},{240,184,6},{240,184,4},{255,208,16},{222,160,1},{251,188,6},{255,230,28},{248,192,6},{248,192,4},{255,233,16},{230,168,1},{255,200,6},{255,202,4},{255,200,6},{255,202,4},{255,0,0},{238,176,1},{255,226,7},{255,227,4},{255,226,7},{255,227,4},{255,0,0},
{246,184,1},{41,0,47},{41,0,44},{13,0,15},{13,0,12},{42,13,48},{0,0,0},{71,0,111},{71,0,108},{21,0,15},{21,0,12},{50,7,32},{0,0,0},{86,1,111},{85,2,108},{29,1,15},{29,3,12},{87,0,96},{3,0,3},{96,6,111},{93,10,108},{37,0,11},{37,11,12},{95,8,96},{11,0,3},{71,0,27},{102,19,108},{68,0,27},{46,7,8},{104,17,96},{20,0,3},{82,0,27},
{110,27,108},{81,0,27},{80,2,24},{112,25,96},{28,0,3},{91,7,27},{118,35,108},{91,7,27},{88,10,24},{91,2,16},{36,7,3},{99,15,27},{126,43,108},{99,15,27},{96,18,24},{99,10,16},{44,1,2},{108,24,27},{135,52,108},{108,24,27},{79,0,4},{108,19,16},{53,10,2},{116,32,27},{143,60,108},{87,5,6},{87,7,4},{116,27,16},{90,3,6},{124,40,27},{151,68,108},{95,13,6},
{95,15,4},{124,35,16},{98,11,6},{132,48,27},{159,76,108},{103,21,6},{103,23,4},{132,43,16},{106,19,6},{141,57,27},{168,85,108},{112,30,6},{112,32,4},{141,52,16},{115,28,6},{149,65,27},{176,93,108},{120,38,6},{120,40,4},{149,60,16},{94,5,1},{157,73,27},{184,101,108},{128,46,6},{128,48,4},{157,68,16},{102,13,1},{165,81,27},{192,109,108},{136,54,6},{136,56,4},{165,76,16},
{110,21,1},{174,90,27},{201,118,108},{145,63,6},{145,65,4},{174,85,16},{119,30,1},{182,98,27},{209,126,108},{153,71,6},{153,73,4},{182,93,16},{127,38,1},{190,106,27},{217,134,108},{161,79,6},{161,81,4},{190,101,16},{135,46,1},{198,114,27},{225,142,108},{169,87,6},{169,89,4},{198,109,16},{143,54,1},{207,123,27},{234,151,108},{178,96,6},{178,98,4},{207,118,16},{152,63,1},{215,131,27},
{242,159,108},{186,104,6},{186,106,4},{215,126,16},{160,71,1},{223,139,27},{250,167,108},{194,112,6},{194,114,4},{223,134,16},{168,79,1},{231,147,27},{231,147,24},{202,120,6},{202,122,4},{231,142,16},{176,87,1},{240,156,27},{240,156,24},{211,129,6},{211,131,4},{240,151,16},{185,96,1},{248,164,27},{248,164,24},{219,137,6},{219,139,4},{248,159,16},{193,104,1},{254,173,27},{255,174,24},{227,145,6},
{227,147,4},{255,169,16},{201,112,1},{255,182,27},{255,187,24},{235,153,6},{235,155,4},{255,193,16},{209,120,1},{249,159,6},{255,218,28},{244,162,6},{244,164,4},{255,220,16},{218,129,1},{254,169,6},{254,168,4},{252,170,6},{252,172,4},{255,245,16},{226,137,1},{255,184,6},{255,190,4},{255,184,6},{255,190,4},{255,0,0},{234,145,1},{255,213,7},{255,214,4},{255,213,7},{255,214,4},{255,0,0},
{242,153,1},{59,0,47},{59,0,44},{18,0,15},{18,0,12},{60,18,48},{0,0,0},{69,0,47},{68,2,44},{26,0,15},{26,0,12},{68,5,32},{0,0,0},{111,0,111},{111,0,108},{34,0,15},{34,0,12},{76,13,32},{0,0,0},{123,3,111},{122,5,108},{42,3,15},{42,6,12},{125,1,96},{6,0,3},{135,8,111},{131,14,108},{50,0,11},{51,15,12},{134,10,96},{15,0,3},{95,0,27},
{139,22,108},{59,2,11},{59,5,8},{142,18,96},{23,0,3},{105,0,27},{147,30,108},{99,0,27},{67,13,8},{150,26,96},{31,0,3},{115,0,27},{155,38,108},{113,1,27},{110,4,24},{158,34,96},{39,0,3},{125,7,27},{164,47,108},{125,7,27},{119,13,24},{167,43,96},{48,6,3},{133,15,27},{172,55,108},{133,15,27},{127,21,24},{134,5,16},{56,14,3},{141,23,27},{180,63,108},{141,23,27},
{135,29,24},{142,13,16},{64,1,2},{149,31,27},{188,71,108},{149,31,27},{143,37,24},{150,21,16},{72,9,2},{158,40,27},{197,80,108},{118,1,6},{117,6,4},{159,30,16},{120,0,6},{166,48,27},{205,88,108},{126,9,6},{125,14,4},{167,38,16},{130,6,6},{174,56,27},{213,96,108},{134,17,6},{133,22,4},{175,46,16},{138,14,6},{182,64,27},{221,104,108},{142,25,6},{141,30,4},{183,54,16},
{146,22,6},{191,73,27},{230,113,108},{151,34,6},{150,39,4},{192,63,16},{155,31,6},{199,81,27},{238,121,108},{159,42,6},{158,47,4},{200,71,16},{163,39,6},{207,89,27},{246,129,108},{167,50,6},{166,55,4},{208,79,16},{130,1,1},{215,97,27},{254,137,108},{175,58,6},{174,63,4},{216,87,16},{138,9,1},{224,106,27},{224,106,24},{184,67,6},{183,72,4},{225,96,16},{147,18,1},{232,114,27},
{232,114,24},{192,75,6},{191,80,4},{233,104,16},{155,26,1},{240,122,27},{240,122,24},{200,83,6},{199,88,4},{241,112,16},{163,34,1},{248,130,27},{248,130,24},{208,91,6},{207,96,4},{249,120,16},{171,42,1},{255,140,27},{254,142,24},{217,100,6},{216,105,4},{255,135,16},{180,51,1},{255,149,27},{255,156,24},{225,108,6},{224,113,4},{255,159,16},{188,59,1},{255,159,27},{253,196,28},{233,116,6},
{232,121,4},{255,184,16},{196,67,1},{247,120,6},{255,205,28},{241,124,6},{240,129,4},{255,208,16},{204,75,1},{252,132,6},{252,132,4},{250,133,6},{249,138,4},{255,236,16},{213,84,1},{255,144,6},{255,150,4},{255,144,6},{255,150,4},{255,0,0},{221,92,1},{254,185,7},{255,175,4},{253,187,7},{255,175,4},{255,0,0},{229,100,1},{255,196,7},{255,199,4},{255,196,7},{255,199,4},{255,0,0},
{237,108,1},{80,0,47},{80,0,44},{24,0,15},{24,0,12},{80,24,48},{0,0,0},{88,1,47},{88,1,44},{32,0,15},{32,0,12},{88,4,32},{0,0,0},{138,0,111},{138,0,108},{40,0,15},{40,0,12},{96,12,32},{0,0,0},{153,0,111},{153,0,108},{48,0,15},{48,0,12},{104,20,32},{0,0,0},{166,4,111},{164,8,108},{57,4,15},{57,9,12},{167,3,96},{9,0,3},{176,9,111},
{172,16,108},{65,0,11},{65,17,12},{175,11,96},{17,0,3},{189,13,111},{180,24,108},{73,0,11},{73,1,8},{183,19,96},{25,0,3},{129,0,27},{188,32,108},{83,2,11},{81,9,8},{191,27,96},{33,0,3},{141,0,27},{197,41,108},{132,0,27},{90,18,8},{200,36,96},{42,0,3},{151,0,27},{205,49,108},{148,0,27},{145,3,24},{208,44,96},{50,0,3},{160,3,27},{213,57,108},{160,3,27},
{153,11,24},{216,52,96},{58,2,3},{168,11,27},{221,65,108},{168,11,27},{161,19,24},{170,0,16},{66,10,3},{177,20,27},{230,74,108},{177,20,27},{170,28,24},{179,7,16},{75,19,3},{185,28,27},{238,82,108},{185,28,27},{178,36,24},{187,15,16},{83,0,2},{193,36,27},{246,90,108},{193,36,27},{186,44,24},{195,23,16},{91,7,2},{201,44,27},{254,98,108},{201,44,27},{147,0,4},{203,31,16},
{99,15,2},{210,53,27},{211,54,24},{157,1,6},{156,9,4},{212,40,16},{108,24,2},{218,61,27},{219,62,24},{165,9,6},{164,17,4},{220,48,16},{170,6,6},{226,69,27},{227,70,24},{173,17,6},{172,25,4},{228,56,16},{178,14,6},{234,77,27},{235,78,24},{181,25,6},{180,33,4},{236,64,16},{186,22,6},{243,86,27},{244,87,24},{190,34,6},{189,42,4},{245,73,16},{195,31,6},{251,94,27},
{252,95,24},{198,42,6},{197,50,4},{253,81,16},{203,39,6},{255,104,27},{255,107,24},{206,50,6},{205,58,4},{255,102,16},{211,47,6},{255,113,27},{255,123,24},{214,58,6},{213,66,4},{255,126,16},{219,55,6},{255,126,27},{253,172,28},{223,67,6},{222,75,4},{255,153,16},{174,2,1},{242,66,6},{254,182,28},{231,75,6},{230,83,4},{255,178,16},{182,10,1},{246,79,6},{255,190,28},{239,83,6},
{238,91,4},{255,202,16},{190,18,1},{251,89,6},{251,88,4},{247,91,6},{246,99,4},{255,227,16},{198,26,1},{255,102,6},{255,108,4},{255,102,6},{255,108,4},{255,0,0},{207,35,1},{255,117,6},{255,132,4},{255,117,6},{255,132,4},{255,0,0},{215,43,1},{254,167,7},{255,156,4},{254,167,7},{255,156,4},{255,0,0},{223,51,1},{255,175,7},{255,181,4},{255,175,7},{255,181,4},{255,0,0},
{231,59,1},{105,0,47},{105,0,44},{33,0,15},{33,0,12},{106,33,48},{0,0,0},{115,0,47},{114,2,44},{41,0,15},{41,0,12},{114,5,32},{0,0,0},{123,2,47},{124,3,44},{49,0,15},{49,0,12},{122,13,32},{0,0,0},{190,0,111},{190,0,108},{57,0,15},{57,0,12},{130,21,32},{0,0,0},{205,0,111},{205,0,108},{66,0,15},{66,0,12},{139,30,32},{0,0,0},{217,4,111},
{215,7,108},{74,4,15},{74,8,12},{218,3,96},{8,0,3},{229,8,111},{223,15,108},{80,0,11},{82,16,12},{226,11,96},{16,0,3},{237,13,111},{231,23,108},{89,0,11},{90,24,12},{234,19,96},{24,0,3},{251,17,111},{240,32,108},{99,0,11},{99,1,8},{243,28,96},{33,0,3},{169,0,27},{248,40,108},{108,2,11},{107,9,8},{251,36,96},{41,0,3},{181,0,27},{254,49,108},{166,0,27},
{115,17,8},{188,79,32},{49,0,3},{190,0,27},{255,59,108},{181,0,27},{123,25,8},{196,87,32},{57,0,3},{203,0,27},{203,0,24},{197,0,27},{196,2,24},{205,96,32},{66,0,3},{213,1,27},{212,3,24},{211,2,27},{204,10,24},{213,104,32},{74,1,3},{221,9,27},{220,11,24},{219,10,27},{212,18,24},{221,0,16},{82,9,3},{229,17,27},{228,19,24},{227,18,27},{220,26,24},{229,5,16},
{90,17,3},{238,26,27},{237,28,24},{236,27,27},{229,35,24},{238,14,16},{99,26,3},{246,34,27},{245,36,24},{244,35,27},{237,43,24},{246,22,16},{107,34,3},{254,42,27},{253,44,24},{252,43,27},{245,51,24},{254,30,16},{115,6,2},{255,52,27},{255,58,24},{255,52,27},{253,59,24},{255,53,16},{123,14,2},{255,64,27},{255,74,24},{196,0,6},{198,100,8},{255,80,16},{132,23,2},{255,74,27},
{255,89,24},{206,1,6},{206,4,4},{255,105,16},{140,31,2},{255,86,27},{253,147,28},{215,7,6},{214,12,4},{255,129,16},{219,4,6},{255,95,27},{255,155,28},{223,15,6},{222,20,4},{255,153,16},{227,12,6},{242,17,6},{255,166,28},{232,24,6},{231,29,4},{255,181,16},{236,21,6},{247,26,6},{255,175,28},{240,32,6},{239,37,4},{255,205,16},{244,29,6},{251,38,6},{251,37,4},{248,40,6},
{247,45,4},{255,230,16},{252,37,6},{255,50,6},{255,53,4},{255,50,6},{255,53,4},{255,0,0},{189,80,2},{255,65,6},{255,80,4},{255,65,6},{255,80,4},{255,0,0},{198,89,2},{252,131,7},{255,105,4},{252,131,7},{255,105,4},{255,0,0},{206,97,2},{254,139,7},{255,129,4},{253,141,7},{255,129,4},{255,0,0},{214,105,2},{255,150,7},{255,153,4},{255,150,7},{255,153,4},{255,0,0},
{222,0,1},{181,0,47},{178,0,44},{47,0,15},{47,0,12},{183,47,48},{0,0,0},{190,0,47},{190,0,44},{55,0,15},{55,0,12},{191,55,48},{0,0,0},{199,0,47},{199,0,44},{63,0,15},{63,0,12},{199,63,48},{0,0,0},{207,0,47},{207,1,44},{71,0,15},{71,0,12},{207,4,32},{0,0,0},{216,1,47},{218,3,44},{80,0,15},{80,0,12},{216,13,32},{0,0,0},{225,1,47},
{227,6,44},{88,0,15},{88,0,12},{224,21,32},{0,0,0},{233,2,47},{235,11,44},{96,1,15},{96,2,12},{232,29,32},{2,0,3},{241,4,47},{243,19,44},{104,5,15},{104,10,12},{240,37,32},{10,0,3},{251,5,47},{252,28,44},{113,9,15},{113,19,12},{249,46,32},{19,0,3},{255,9,47},{255,37,44},{117,0,11},{121,27,12},{255,55,32},{27,0,3},{255,9,47},{255,46,44},{126,0,11},
{129,35,12},{255,67,32},{35,0,3},{248,0,27},{255,55,44},{137,0,11},{137,43,12},{255,79,32},{43,0,3},{250,0,27},{255,64,44},{147,2,11},{146,6,8},{255,93,32},{52,0,3},{251,0,27},{255,73,44},{155,4,11},{154,14,8},{255,104,32},{60,0,3},{253,0,27},{248,0,24},{233,0,27},{162,22,8},{255,116,32},{68,0,3},{254,0,27},{254,0,24},{248,0,27},{170,30,8},{255,128,32},
{76,0,3},{255,1,27},{255,7,24},{255,1,27},{179,39,8},{255,22,16},{85,0,3},{255,2,27},{255,22,24},{255,7,27},{187,47,8},{255,47,16},{93,0,3},{255,4,27},{251,100,28},{182,0,7},{195,55,8},{255,71,16},{101,0,3},{255,4,27},{253,108,28},{191,0,7},{203,63,8},{255,95,16},{109,0,3},{255,7,27},{255,118,28},{200,0,7},{212,72,8},{255,123,16},{118,0,3},{246,0,7},
{255,129,28},{209,0,7},{220,80,8},{255,147,16},{126,0,3},{246,0,7},{255,138,28},{218,0,7},{228,88,8},{255,172,16},{134,0,3},{249,3,7},{245,91,8},{228,3,7},{236,96,8},{255,196,16},{142,6,3},{251,14,7},{250,102,8},{237,12,7},{245,105,8},{255,223,16},{151,15,3},{253,22,7},{254,112,8},{245,20,7},{253,113,8},{255,248,16},{159,23,3},{253,31,7},{255,124,8},{249,28,7},
{255,124,8},{255,0,0},{167,31,3},{254,39,7},{255,10,4},{252,37,7},{255,10,4},{255,0,0},{175,39,3},{255,48,7},{255,38,4},{254,48,7},{255,38,4},{255,0,0},{184,48,3},{255,56,7},{255,62,4},{255,56,7},{255,62,4},{255,0,0},{192,56,3},{255,65,7},{255,86,4},{255,65,7},{255,86,4},{255,0,0},{200,64,3},{255,74,7},{255,111,4},{255,77,7},{255,111,4},{255,0,0},
{208,5,2},
/**** ended inlining basisu_transcoder_tables_bc7_m5_alpha.inc ****/
	};
	
	static inline uint32_t set_block_bits(uint8_t* pBytes, uint32_t val, uint32_t num_bits, uint32_t cur_ofs)
	{
		assert(num_bits < 32);
		assert(val < (1ULL << num_bits));

		uint32_t mask = static_cast<uint32_t>((1ULL << num_bits) - 1);

		while (num_bits)
		{
			const uint32_t n = basisu::minimum<uint32_t>(8 - (cur_ofs & 7), num_bits);

			pBytes[cur_ofs >> 3] &= ~static_cast<uint8_t>(mask << (cur_ofs & 7));
			pBytes[cur_ofs >> 3] |= static_cast<uint8_t>(val << (cur_ofs & 7));

			val >>= n;
			mask >>= n;

			num_bits -= n;
			cur_ofs += n;
		}

		return cur_ofs;
	}

	struct bc7_mode_5
	{
		union
		{
			struct
			{
				uint64_t m_mode : 6;
				uint64_t m_rot : 2;

				uint64_t m_r0 : 7;
				uint64_t m_r1 : 7;
				uint64_t m_g0 : 7;
				uint64_t m_g1 : 7;
				uint64_t m_b0 : 7;
				uint64_t m_b1 : 7;
				uint64_t m_a0 : 8;
				uint64_t m_a1_0 : 6;

			} m_lo;

			uint64_t m_lo_bits;
		};

		union
		{
			struct
			{
				uint64_t m_a1_1 : 2;

				// bit 2
				uint64_t m_c00 : 1;
				uint64_t m_c10 : 2;
				uint64_t m_c20 : 2;
				uint64_t m_c30 : 2;

				uint64_t m_c01 : 2;
				uint64_t m_c11 : 2;
				uint64_t m_c21 : 2;
				uint64_t m_c31 : 2;

				uint64_t m_c02 : 2;
				uint64_t m_c12 : 2;
				uint64_t m_c22 : 2;
				uint64_t m_c32 : 2;

				uint64_t m_c03 : 2;
				uint64_t m_c13 : 2;
				uint64_t m_c23 : 2;
				uint64_t m_c33 : 2;

				// bit 33
				uint64_t m_a00 : 1;
				uint64_t m_a10 : 2;
				uint64_t m_a20 : 2;
				uint64_t m_a30 : 2;

				uint64_t m_a01 : 2;
				uint64_t m_a11 : 2;
				uint64_t m_a21 : 2;
				uint64_t m_a31 : 2;

				uint64_t m_a02 : 2;
				uint64_t m_a12 : 2;
				uint64_t m_a22 : 2;
				uint64_t m_a32 : 2;

				uint64_t m_a03 : 2;
				uint64_t m_a13 : 2;
				uint64_t m_a23 : 2;
				uint64_t m_a33 : 2;

			} m_hi;

			uint64_t m_hi_bits;
		};
	};

#if BASISD_WRITE_NEW_BC7_MODE5_TABLES
	static void create_etc1_to_bc7_m5_color_conversion_table()
	{
		FILE* pFile = nullptr;
		fopen_s(&pFile, "basisu_transcoder_tables_bc7_m5_color.inc", "w");

		uint32_t n = 0;

		for (int inten = 0; inten < 8; inten++)
		{
			for (uint32_t g = 0; g < 32; g++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(g, g, g, 255), false), inten);

				for (uint32_t sr = 0; sr < NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES; sr++)
				{
					const uint32_t low_selector = g_etc1_to_bc7_m5_selector_ranges[sr].m_low;
					const uint32_t high_selector = g_etc1_to_bc7_m5_selector_ranges[sr].m_high;

					for (uint32_t m = 0; m < NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS; m++)
					{
						uint32_t best_lo = 0;
						uint32_t best_hi = 0;
						uint64_t best_err = UINT64_MAX;

						for (uint32_t hi = 0; hi <= 127; hi++)
						{
							for (uint32_t lo = 0; lo <= 127; lo++)
							{
								uint32_t colors[4];

								colors[0] = (lo << 1) | (lo >> 6);
								colors[3] = (hi << 1) | (hi >> 6);

								colors[1] = (colors[0] * (64 - 21) + colors[3] * 21 + 32) / 64;
								colors[2] = (colors[0] * (64 - 43) + colors[3] * 43 + 32) / 64;

								uint64_t total_err = 0;

								for (uint32_t s = low_selector; s <= high_selector; s++)
								{
									int err = block_colors[s].g - colors[g_etc1_to_bc7_m5_selector_mappings[m][s]];

									int err_scale = 1;
									// Special case when the intensity table is 7, low_selector is 0, and high_selector is 3. In this extreme case, it's likely the encoder is trying to strongly favor 
									// the low/high selectors which are clamping to either 0 or 255.
									if (((inten == 7) && (low_selector == 0) && (high_selector == 3)) && ((s == 0) || (s == 3)))
										err_scale = 5;

									total_err += (err * err) * err_scale;
								}

								if (total_err < best_err)
								{
									best_err = total_err;
									best_lo = lo;
									best_hi = hi;
								}
							}
						}

						best_err = basisu::minimum<uint32_t>(best_err, 0xFFFF);

						fprintf(pFile, "{%u,%u,%u},", best_lo, best_hi, (uint32_t)best_err);
						n++;
						if ((n & 31) == 31)
							fprintf(pFile, "\n");
					} // m
				} // sr
			} // g
		} // inten

		fclose(pFile);
	}

	static void create_etc1_to_bc7_m5_alpha_conversion_table()
	{
		FILE* pFile = nullptr;
		fopen_s(&pFile, "basisu_transcoder_tables_bc7_m5_alpha.inc", "w");

		uint32_t n = 0;

		for (int inten = 0; inten < 8; inten++)
		{
			for (uint32_t g = 0; g < 32; g++)
			{
				color32 block_colors[4];
				decoder_etc_block::get_diff_subblock_colors(block_colors, decoder_etc_block::pack_color5(color32(g, g, g, 255), false), inten);

				for (uint32_t sr = 0; sr < NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES; sr++)
				{
					const uint32_t low_selector = g_etc1_to_bc7_m5a_selector_ranges[sr].m_low;
					const uint32_t high_selector = g_etc1_to_bc7_m5a_selector_ranges[sr].m_high;

					uint32_t best_lo = 0;
					uint32_t best_hi = 0;
					uint64_t best_err = UINT64_MAX;
					uint32_t best_output_selectors = 0;

					for (uint32_t hi = 0; hi <= 255; hi++)
					{
						for (uint32_t lo = 0; lo <= 255; lo++)
						{
							uint32_t colors[4];

							colors[0] = lo;
							colors[3] = hi;

							colors[1] = (colors[0] * (64 - 21) + colors[3] * 21 + 32) / 64;
							colors[2] = (colors[0] * (64 - 43) + colors[3] * 43 + 32) / 64;

							uint64_t total_err = 0;
							uint32_t output_selectors = 0;

							for (uint32_t s = low_selector; s <= high_selector; s++)
							{
								int best_mapping_err = INT_MAX;
								int best_k = 0;
								for (int k = 0; k < 4; k++)
								{
									int mapping_err = block_colors[s].g - colors[k];
									mapping_err *= mapping_err;

									// Special case when the intensity table is 7, low_selector is 0, and high_selector is 3. In this extreme case, it's likely the encoder is trying to strongly favor 
									// the low/high selectors which are clamping to either 0 or 255.
									if (((inten == 7) && (low_selector == 0) && (high_selector == 3)) && ((s == 0) || (s == 3)))
										mapping_err *= 5;

									if (mapping_err < best_mapping_err)
									{
										best_mapping_err = mapping_err;
										best_k = k;
									}
								} // k
								
								total_err += best_mapping_err;
								output_selectors |= (best_k << (s * 2));
							} // s

							if (total_err < best_err)
							{
								best_err = total_err;
								best_lo = lo;
								best_hi = hi;
								best_output_selectors = output_selectors;
							}

						} // lo
					} // hi
										
					fprintf(pFile, "{%u,%u,%u},", best_lo, best_hi, best_output_selectors);
					n++;
					if ((n & 31) == 31)
						fprintf(pFile, "\n");

				} // sr
			} // g
		} // inten

		fclose(pFile);
	}
#endif // BASISD_WRITE_NEW_BC7_MODE5_TABLES

	struct bc7_m5_match_entry
	{
		uint8_t m_hi;
		uint8_t m_lo;
	};

	static bc7_m5_match_entry g_bc7_m5_equals_1[256] =
	{
		{0,0},{1,0},{3,0},{4,0},{6,0},{7,0},{9,0},{10,0},{12,0},{13,0},{15,0},{16,0},{18,0},{20,0},{21,0},{23,0},
		{24,0},{26,0},{27,0},{29,0},{30,0},{32,0},{33,0},{35,0},{36,0},{38,0},{39,0},{41,0},{42,0},{44,0},{45,0},{47,0},
		{48,0},{50,0},{52,0},{53,0},{55,0},{56,0},{58,0},{59,0},{61,0},{62,0},{64,0},{65,0},{66,0},{68,0},{69,0},{71,0},
		{72,0},{74,0},{75,0},{77,0},{78,0},{80,0},{82,0},{83,0},{85,0},{86,0},{88,0},{89,0},{91,0},{92,0},{94,0},{95,0},
		{97,0},{98,0},{100,0},{101,0},{103,0},{104,0},{106,0},{107,0},{109,0},{110,0},{112,0},{114,0},{115,0},{117,0},{118,0},{120,0},
		{121,0},{123,0},{124,0},{126,0},{127,0},{127,1},{126,2},{126,3},{127,3},{127,4},{126,5},{126,6},{127,6},{127,7},{126,8},{126,9},
		{127,9},{127,10},{126,11},{126,12},{127,12},{127,13},{126,14},{125,15},{127,15},{126,16},{126,17},{127,17},{127,18},{126,19},{126,20},{127,20},
		{127,21},{126,22},{126,23},{127,23},{127,24},{126,25},{126,26},{127,26},{127,27},{126,28},{126,29},{127,29},{127,30},{126,31},{126,32},{127,32},
		{127,33},{126,34},{126,35},{127,35},{127,36},{126,37},{126,38},{127,38},{127,39},{126,40},{126,41},{127,41},{127,42},{126,43},{126,44},{127,44},
		{127,45},{126,46},{125,47},{127,47},{126,48},{126,49},{127,49},{127,50},{126,51},{126,52},{127,52},{127,53},{126,54},{126,55},{127,55},{127,56},
		{126,57},{126,58},{127,58},{127,59},{126,60},{126,61},{127,61},{127,62},{126,63},{125,64},{126,64},{126,65},{127,65},{127,66},{126,67},{126,68},
		{127,68},{127,69},{126,70},{126,71},{127,71},{127,72},{126,73},{126,74},{127,74},{127,75},{126,76},{125,77},{127,77},{126,78},{126,79},{127,79},
		{127,80},{126,81},{126,82},{127,82},{127,83},{126,84},{126,85},{127,85},{127,86},{126,87},{126,88},{127,88},{127,89},{126,90},{126,91},{127,91},
		{127,92},{126,93},{126,94},{127,94},{127,95},{126,96},{126,97},{127,97},{127,98},{126,99},{126,100},{127,100},{127,101},{126,102},{126,103},{127,103},
		{127,104},{126,105},{126,106},{127,106},{127,107},{126,108},{125,109},{127,109},{126,110},{126,111},{127,111},{127,112},{126,113},{126,114},{127,114},{127,115},
		{126,116},{126,117},{127,117},{127,118},{126,119},{126,120},{127,120},{127,121},{126,122},{126,123},{127,123},{127,124},{126,125},{126,126},{127,126},{127,127}
	};
	
	static void transcoder_init_bc7_mode5()
	{
#if 0
		// This is a little too much work to do at init time, so precompute it.
		for (int i = 0; i < 256; i++)
		{
			int lowest_e = 256;
			for (int lo = 0; lo < 128; lo++)
			{
				for (int hi = 0; hi < 128; hi++)
				{
					const int lo_e = (lo << 1) | (lo >> 6);
					const int hi_e = (hi << 1) | (hi >> 6);

					// Selector 1
					int v = (lo_e * (64 - 21) + hi_e * 21 + 32) >> 6;
					int e = abs(v - i);

					if (e < lowest_e)
					{
						g_bc7_m5_equals_1[i].m_hi = static_cast<uint8_t>(hi);
						g_bc7_m5_equals_1[i].m_lo = static_cast<uint8_t>(lo);

						lowest_e = e;
					}

				} // hi
								
			} // lo
			
			printf("{%u,%u},", g_bc7_m5_equals_1[i].m_hi, g_bc7_m5_equals_1[i].m_lo);
			if ((i & 15) == 15) printf("\n");
		}
#endif

		for (uint32_t i = 0; i < NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES; i++)
		{
			uint32_t l = g_etc1_to_bc7_m5_selector_ranges[i].m_low;
			uint32_t h = g_etc1_to_bc7_m5_selector_ranges[i].m_high;
			g_etc1_to_bc7_m5_selector_range_index[l][h] = i;
		}

		for (uint32_t i = 0; i < NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES; i++)
		{
			uint32_t l = g_etc1_to_bc7_m5a_selector_ranges[i].m_low;
			uint32_t h = g_etc1_to_bc7_m5a_selector_ranges[i].m_high;
			g_etc1_to_bc7_m5a_selector_range_index[l][h] = i;
		}
	}

	static void convert_etc1s_to_bc7_m5_color(void* pDst, const endpoint* pEndpoints, const selector* pSelector)
	{
		bc7_mode_5* pDst_block = static_cast<bc7_mode_5*>(pDst);
				
		// First ensure the block is cleared to all 0's
		static_cast<uint64_t*>(pDst)[0] = 0;
		static_cast<uint64_t*>(pDst)[1] = 0;

		// Set alpha to 255
		pDst_block->m_lo.m_mode = 1 << 5;
		pDst_block->m_lo.m_a0 = 255;
		pDst_block->m_lo.m_a1_0 = 63;
		pDst_block->m_hi.m_a1_1 = 3;

		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const uint32_t base_color_r = pEndpoints->m_color5.r;
		const uint32_t base_color_g = pEndpoints->m_color5.g;
		const uint32_t base_color_b = pEndpoints->m_color5.b;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (pSelector->m_num_unique_selectors == 1)
		{
			// Solid color block - use precomputed tables and set selectors to 1.
			uint32_t r, g, b;
			decoder_etc_block::get_block_color5(pEndpoints->m_color5, inten_table, low_selector, r, g, b);

			pDst_block->m_lo.m_r0 = g_bc7_m5_equals_1[r].m_lo;
			pDst_block->m_lo.m_g0 = g_bc7_m5_equals_1[g].m_lo;
			pDst_block->m_lo.m_b0 = g_bc7_m5_equals_1[b].m_lo;

			pDst_block->m_lo.m_r1 = g_bc7_m5_equals_1[r].m_hi;
			pDst_block->m_lo.m_g1 = g_bc7_m5_equals_1[g].m_hi;
			pDst_block->m_lo.m_b1 = g_bc7_m5_equals_1[b].m_hi;

			set_block_bits((uint8_t*)pDst, 0x2aaaaaab, 31, 66);
			return;
		}
		else if (pSelector->m_num_unique_selectors == 2)
		{
			// Only one or two unique selectors, so just switch to block truncation coding (BTC) to avoid quality issues on extreme blocks.
			color32 block_colors[4];

			decoder_etc_block::get_block_colors5(block_colors, color32(base_color_r, base_color_g, base_color_b, 255), inten_table);

			const uint32_t r0 = block_colors[low_selector].r;
			const uint32_t g0 = block_colors[low_selector].g;
			const uint32_t b0 = block_colors[low_selector].b;

			const uint32_t r1 = block_colors[high_selector].r;
			const uint32_t g1 = block_colors[high_selector].g;
			const uint32_t b1 = block_colors[high_selector].b;

			pDst_block->m_lo.m_r0 = r0 >> 1;
			pDst_block->m_lo.m_g0 = g0 >> 1;
			pDst_block->m_lo.m_b0 = b0 >> 1;

			pDst_block->m_lo.m_r1 = r1 >> 1;
			pDst_block->m_lo.m_g1 = g1 >> 1;
			pDst_block->m_lo.m_b1 = b1 >> 1;

			uint32_t output_low_selector = 0, output_bit_offset = 0, output_bits = 0;

			for (uint32_t y = 0; y < 4; y++)
			{
				for (uint32_t x = 0; x < 4; x++)
				{
					uint32_t s = pSelector->get_selector(x, y);
					uint32_t os = (s == low_selector) ? output_low_selector : (3 ^ output_low_selector);

					uint32_t num_bits = 2;

					if ((x | y) == 0)
					{
						if (os & 2)
						{
							pDst_block->m_lo.m_r0 = r1 >> 1;
							pDst_block->m_lo.m_g0 = g1 >> 1;
							pDst_block->m_lo.m_b0 = b1 >> 1;

							pDst_block->m_lo.m_r1 = r0 >> 1;
							pDst_block->m_lo.m_g1 = g0 >> 1;
							pDst_block->m_lo.m_b1 = b0 >> 1;

							output_low_selector = 3;
							os = 0;
						}

						num_bits = 1;
					}

					output_bits |= (os << output_bit_offset);
					output_bit_offset += num_bits;
				}
			}

			set_block_bits((uint8_t*)pDst, output_bits, 31, 66);
			return;
		}

		const uint32_t selector_range_table = g_etc1_to_bc7_m5_selector_range_index[low_selector][high_selector];

		//[32][8][RANGES][MAPPING]
		const etc1_to_bc7_m5_solution* pTable_r = &g_etc1_to_bc7_m5_color[(inten_table * 32 + base_color_r) * (NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS];
		const etc1_to_bc7_m5_solution* pTable_g = &g_etc1_to_bc7_m5_color[(inten_table * 32 + base_color_g) * (NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS];
		const etc1_to_bc7_m5_solution* pTable_b = &g_etc1_to_bc7_m5_color[(inten_table * 32 + base_color_b) * (NUM_ETC1_TO_BC7_M5_SELECTOR_RANGES * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS) + selector_range_table * NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS];

		uint32_t best_err = UINT_MAX;
		uint32_t best_mapping = 0;

		assert(NUM_ETC1_TO_BC7_M5_SELECTOR_MAPPINGS == 10);
#define DO_ITER(m) { uint32_t total_err = pTable_r[m].m_err + pTable_g[m].m_err + pTable_b[m].m_err; if (total_err < best_err) { best_err = total_err; best_mapping = m; } }
		DO_ITER(0); DO_ITER(1); DO_ITER(2); DO_ITER(3); DO_ITER(4);
		DO_ITER(5); DO_ITER(6); DO_ITER(7); DO_ITER(8); DO_ITER(9);
#undef DO_ITER

		const uint8_t* pSelectors_xlat = &g_etc1_to_bc7_m5_selector_mappings[best_mapping][0];

		uint32_t s_inv = 0;
		if (pSelectors_xlat[pSelector->get_selector(0, 0)] & 2)
		{
			pDst_block->m_lo.m_r0 = pTable_r[best_mapping].m_hi;
			pDst_block->m_lo.m_g0 = pTable_g[best_mapping].m_hi;
			pDst_block->m_lo.m_b0 = pTable_b[best_mapping].m_hi;

			pDst_block->m_lo.m_r1 = pTable_r[best_mapping].m_lo;
			pDst_block->m_lo.m_g1 = pTable_g[best_mapping].m_lo;
			pDst_block->m_lo.m_b1 = pTable_b[best_mapping].m_lo;
			
			s_inv = 3;
		}
		else
		{
			pDst_block->m_lo.m_r0 = pTable_r[best_mapping].m_lo;
			pDst_block->m_lo.m_g0 = pTable_g[best_mapping].m_lo;
			pDst_block->m_lo.m_b0 = pTable_b[best_mapping].m_lo;

			pDst_block->m_lo.m_r1 = pTable_r[best_mapping].m_hi;
			pDst_block->m_lo.m_g1 = pTable_g[best_mapping].m_hi;
			pDst_block->m_lo.m_b1 = pTable_b[best_mapping].m_hi;
		}

		uint32_t output_bits = 0, output_bit_ofs = 0;

		for (uint32_t y = 0; y < 4; y++)
		{
			for (uint32_t x = 0; x < 4; x++)
			{
				const uint32_t s = pSelector->get_selector(x, y);
				
				const uint32_t os = pSelectors_xlat[s] ^ s_inv;

				output_bits |= (os << output_bit_ofs);

				output_bit_ofs += (((x | y) == 0) ? 1 : 2);
			}
		}

		set_block_bits((uint8_t*)pDst, output_bits, 31, 66);
	}

	static void convert_etc1s_to_bc7_m5_alpha(void* pDst, const endpoint* pEndpoints, const selector* pSelector)
	{
		bc7_mode_5* pDst_block = static_cast<bc7_mode_5*>(pDst);

		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const uint32_t base_color_r = pEndpoints->m_color5.r;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (pSelector->m_num_unique_selectors == 1)
		{
			uint32_t r;
			decoder_etc_block::get_block_color5_r(pEndpoints->m_color5, inten_table, low_selector, r);

			pDst_block->m_lo.m_a0 = r;
			pDst_block->m_lo.m_a1_0 = r & 63;
			pDst_block->m_hi.m_a1_1 = r >> 6;
					  
			return;
		}
		else if (pSelector->m_num_unique_selectors == 2)
		{
			// Only one or two unique selectors, so just switch to block truncation coding (BTC) to avoid quality issues on extreme blocks.
			int block_colors[4];

			decoder_etc_block::get_block_colors5_g(block_colors, pEndpoints->m_color5, inten_table);

			pDst_block->m_lo.m_a0 = block_colors[low_selector];
			pDst_block->m_lo.m_a1_0 = block_colors[high_selector] & 63;
			pDst_block->m_hi.m_a1_1 = block_colors[high_selector] >> 6;

			uint32_t output_low_selector = 0, output_bit_offset = 0, output_bits = 0;

			for (uint32_t y = 0; y < 4; y++)
			{
				for (uint32_t x = 0; x < 4; x++)
				{
					const uint32_t s = pSelector->get_selector(x, y);
					uint32_t os = (s == low_selector) ? output_low_selector : (3 ^ output_low_selector);

					uint32_t num_bits = 2;

					if ((x | y) == 0)
					{
						if (os & 2)
						{
							pDst_block->m_lo.m_a0 = block_colors[high_selector];
							pDst_block->m_lo.m_a1_0 = block_colors[low_selector] & 63;
							pDst_block->m_hi.m_a1_1 = block_colors[low_selector] >> 6;

							output_low_selector = 3;
							os = 0;
						}

						num_bits = 1;
					}

					output_bits |= (os << output_bit_offset);
					output_bit_offset += num_bits;
				}
			}

			set_block_bits((uint8_t*)pDst, output_bits, 31, 97);
			return;
		}

		const uint32_t selector_range_table = g_etc1_to_bc7_m5a_selector_range_index[low_selector][high_selector];
						
		const etc1_g_to_bc7_m5a_conversion* pTable = &g_etc1_g_to_bc7_m5a[inten_table * (32 * NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES) + base_color_r * NUM_ETC1_TO_BC7_M5A_SELECTOR_RANGES + selector_range_table];

		pDst_block->m_lo.m_a0 = pTable->m_lo;
		pDst_block->m_lo.m_a1_0 = pTable->m_hi & 63;
		pDst_block->m_hi.m_a1_1 = pTable->m_hi >> 6;

		uint32_t output_bit_offset = 0, output_bits = 0, selector_trans = pTable->m_trans;

		for (uint32_t y = 0; y < 4; y++)
		{
			for (uint32_t x = 0; x < 4; x++)
			{
				const uint32_t s = pSelector->get_selector(x, y);
				uint32_t os = (selector_trans >> (s * 2)) & 3;

				uint32_t num_bits = 2;

				if ((x | y) == 0)
				{
					if (os & 2)
					{
						pDst_block->m_lo.m_a0 = pTable->m_hi;
						pDst_block->m_lo.m_a1_0 = pTable->m_lo & 63;
						pDst_block->m_hi.m_a1_1 = pTable->m_lo >> 6;

						selector_trans ^= 0xFF;
						os ^= 3;
					}

					num_bits = 1;
				}

				output_bits |= (os << output_bit_offset);
				output_bit_offset += num_bits;
			}
		}

		set_block_bits((uint8_t*)pDst, output_bits, 31, 97);
	}
#endif // BASISD_SUPPORT_BC7_MODE5

#if BASISD_SUPPORT_ETC2_EAC_A8 || BASISD_SUPPORT_UASTC
	static const uint8_t g_etc2_eac_a8_sel4[6] = { 0x92, 0x49, 0x24, 0x92, 0x49, 0x24 };
#endif

#if BASISD_SUPPORT_ETC2_EAC_A8
	static void convert_etc1s_to_etc2_eac_a8(eac_block* pDst_block, const endpoint* pEndpoints, const selector* pSelector)
	{
		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const color32& base_color = pEndpoints->m_color5;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (low_selector == high_selector)
		{
			uint32_t r;
			decoder_etc_block::get_block_color5_r(base_color, inten_table, low_selector, r);

			// Constant alpha block
			// Select table 13, use selector 4 (0), set multiplier to 1 and base color g
			pDst_block->m_base = r;
			pDst_block->m_table = 13;
			pDst_block->m_multiplier = 1;

			// selectors are all 4's
			memcpy(pDst_block->m_selectors, g_etc2_eac_a8_sel4, sizeof(g_etc2_eac_a8_sel4));

			return;
		}

		uint32_t selector_range_table = 0;
		for (selector_range_table = 0; selector_range_table < NUM_ETC2_EAC_SELECTOR_RANGES; selector_range_table++)
			if ((low_selector == s_etc2_eac_selector_ranges[selector_range_table].m_low) && (high_selector == s_etc2_eac_selector_ranges[selector_range_table].m_high))
				break;
		if (selector_range_table >= NUM_ETC2_EAC_SELECTOR_RANGES)
			selector_range_table = 0;

		const etc1_g_to_eac_conversion* pTable_entry = &s_etc1_g_to_etc2_a8[base_color.r + inten_table * 32][selector_range_table];

		pDst_block->m_base = pTable_entry->m_base;
		pDst_block->m_table = pTable_entry->m_table_mul >> 4;
		pDst_block->m_multiplier = pTable_entry->m_table_mul & 15;

		uint64_t selector_bits = 0;

		for (uint32_t y = 0; y < 4; y++)
		{
			for (uint32_t x = 0; x < 4; x++)
			{
				uint32_t s = pSelector->get_selector(x, y);

				uint32_t ds = (pTable_entry->m_trans >> (s * 3)) & 7;

				const uint32_t dst_ofs = 45 - (y + x * 4) * 3;
				selector_bits |= (static_cast<uint64_t>(ds) << dst_ofs);
			}
		}

		pDst_block->set_selector_bits(selector_bits);
	}
#endif // BASISD_SUPPORT_ETC2_EAC_A8

#if BASISD_SUPPORT_ETC2_EAC_RG11
	static const etc1_g_to_eac_conversion s_etc1_g_to_etc2_r11[32 * 8][NUM_ETC2_EAC_SELECTOR_RANGES] =
	{
		{{0,1,3328},{0,1,3328},{0,16,457},{0,16,456}},
		{{0,226,3936},{0,226,3936},{0,17,424},{8,0,472}},
		{{6,178,4012},{6,178,4008},{0,146,501},{16,0,472}},
		{{14,178,4012},{14,178,4008},{8,146,501},{24,0,472}},
		{{23,178,4012},{23,178,4008},{17,146,501},{33,0,472}},
		{{31,178,4012},{31,178,4008},{25,146,501},{41,0,472}},
		{{39,178,4012},{39,178,4008},{33,146,501},{49,0,472}},
		{{47,178,4012},{47,178,4008},{41,146,501},{27,228,496}},
		{{56,178,4012},{56,178,4008},{50,146,501},{36,228,496}},
		{{64,178,4012},{64,178,4008},{58,146,501},{44,228,496}},
		{{72,178,4012},{72,178,4008},{66,146,501},{52,228,496}},
		{{80,178,4012},{80,178,4008},{74,146,501},{60,228,496}},
		{{89,178,4012},{89,178,4008},{83,146,501},{69,228,496}},
		{{97,178,4012},{97,178,4008},{91,146,501},{77,228,496}},
		{{105,178,4012},{105,178,4008},{99,146,501},{85,228,496}},
		{{113,178,4012},{113,178,4008},{107,146,501},{93,228,496}},
		{{122,178,4012},{122,178,4008},{116,146,501},{102,228,496}},
		{{130,178,4012},{130,178,4008},{124,146,501},{110,228,496}},
		{{138,178,4012},{138,178,4008},{132,146,501},{118,228,496}},
		{{146,178,4012},{146,178,4008},{140,146,501},{126,228,496}},
		{{155,178,4012},{155,178,4008},{149,146,501},{135,228,496}},
		{{163,178,4012},{163,178,4008},{157,146,501},{143,228,496}},
		{{171,178,4012},{171,178,4008},{165,146,501},{151,228,496}},
		{{179,178,4012},{179,178,4008},{173,146,501},{159,228,496}},
		{{188,178,4012},{188,178,4008},{182,146,501},{168,228,496}},
		{{196,178,4012},{196,178,4008},{190,146,501},{176,228,496}},
		{{204,178,4012},{204,178,4008},{198,146,501},{184,228,496}},
		{{212,178,4012},{212,178,4008},{206,146,501},{192,228,496}},
		{{221,178,4012},{221,178,4008},{215,146,501},{201,228,496}},
		{{229,178,4012},{229,178,4008},{223,146,501},{209,228,496}},
		{{235,66,4012},{221,100,4008},{231,146,501},{217,228,496}},
		{{211,102,4085},{254,32,4040},{211,102,501},{254,32,456}},
		{{0,2,3328},{0,2,3328},{0,1,320},{0,1,320}},
		{{7,162,3905},{7,162,3904},{0,17,480},{0,17,480}},
		{{15,162,3906},{15,162,3904},{1,117,352},{1,117,352}},
		{{23,162,3906},{23,162,3904},{5,34,500},{4,53,424}},
		{{32,162,3906},{32,162,3904},{14,34,500},{3,69,424}},
		{{40,162,3906},{40,162,3904},{22,34,500},{1,133,496}},
		{{48,162,3906},{48,162,3904},{30,34,500},{4,85,496}},
		{{56,162,3906},{56,162,3904},{38,34,500},{12,85,496}},
		{{65,162,3906},{65,162,3904},{47,34,500},{1,106,424}},
		{{73,162,3906},{73,162,3904},{55,34,500},{9,106,424}},
		{{81,162,3906},{81,162,3904},{63,34,500},{7,234,496}},
		{{89,162,3906},{89,162,3904},{71,34,500},{15,234,496}},
		{{98,162,3906},{98,162,3904},{80,34,500},{24,234,496}},
		{{106,162,3906},{106,162,3904},{88,34,500},{32,234,496}},
		{{114,162,3906},{114,162,3904},{96,34,500},{40,234,496}},
		{{122,162,3906},{122,162,3904},{104,34,500},{48,234,496}},
		{{131,162,3906},{131,162,3904},{113,34,500},{57,234,496}},
		{{139,162,3906},{139,162,3904},{121,34,500},{65,234,496}},
		{{147,162,3906},{147,162,3904},{129,34,500},{73,234,496}},
		{{155,162,3906},{155,162,3904},{137,34,500},{81,234,496}},
		{{164,162,3906},{164,162,3904},{146,34,500},{90,234,496}},
		{{172,162,3906},{172,162,3904},{154,34,500},{98,234,496}},
		{{180,162,3906},{180,162,3904},{162,34,500},{106,234,496}},
		{{188,162,3906},{188,162,3904},{170,34,500},{114,234,496}},
		{{197,162,3906},{197,162,3904},{179,34,500},{123,234,496}},
		{{205,162,3906},{205,162,3904},{187,34,500},{131,234,496}},
		{{213,162,3906},{213,162,3904},{195,34,500},{139,234,496}},
		{{221,162,3906},{221,162,3904},{203,34,500},{147,234,496}},
		{{230,162,3906},{230,162,3904},{212,34,500},{156,234,496}},
		{{238,162,3906},{174,106,4008},{220,34,500},{164,234,496}},
		{{240,178,4001},{182,106,4008},{228,34,500},{172,234,496}},
		{{166,108,4085},{115,31,4080},{166,108,501},{115,31,496}},
		{{1,68,3328},{1,68,3328},{0,1,384},{0,1,384}},
		{{1,51,3968},{1,51,3968},{0,2,384},{0,2,384}},
		{{21,18,3851},{21,18,3848},{1,50,488},{1,50,488}},
		{{26,195,3851},{29,18,3848},{0,67,488},{0,67,488}},
		{{35,195,3851},{38,18,3848},{12,115,488},{0,3,496}},
		{{43,195,3851},{46,18,3848},{20,115,488},{2,6,424}},
		{{51,195,3851},{54,18,3848},{36,66,482},{4,22,424}},
		{{59,195,3851},{62,18,3848},{44,66,482},{3,73,424}},
		{{68,195,3851},{71,18,3848},{53,66,482},{3,22,496}},
		{{76,195,3851},{79,18,3848},{61,66,482},{2,137,496}},
		{{84,195,3851},{87,18,3848},{69,66,482},{1,89,496}},
		{{92,195,3851},{95,18,3848},{77,66,482},{9,89,496}},
		{{101,195,3851},{104,18,3848},{86,66,482},{18,89,496}},
		{{109,195,3851},{112,18,3848},{94,66,482},{26,89,496}},
		{{117,195,3851},{120,18,3848},{102,66,482},{34,89,496}},
		{{125,195,3851},{128,18,3848},{110,66,482},{42,89,496}},
		{{134,195,3851},{137,18,3848},{119,66,482},{51,89,496}},
		{{141,195,3907},{145,18,3848},{127,66,482},{59,89,496}},
		{{149,195,3907},{153,18,3848},{135,66,482},{67,89,496}},
		{{157,195,3907},{161,18,3848},{143,66,482},{75,89,496}},
		{{166,195,3907},{170,18,3848},{152,66,482},{84,89,496}},
		{{174,195,3907},{178,18,3848},{160,66,482},{92,89,496}},
		{{182,195,3907},{186,18,3848},{168,66,482},{100,89,496}},
		{{190,195,3907},{194,18,3848},{176,66,482},{108,89,496}},
		{{199,195,3907},{203,18,3848},{185,66,482},{117,89,496}},
		{{207,195,3907},{211,18,3848},{193,66,482},{125,89,496}},
		{{215,195,3907},{219,18,3848},{201,66,482},{133,89,496}},
		{{223,195,3907},{227,18,3848},{209,66,482},{141,89,496}},
		{{232,195,3907},{168,89,4008},{218,66,482},{150,89,496}},
		{{236,18,3907},{176,89,4008},{226,66,482},{158,89,496}},
		{{158,90,4085},{103,31,4080},{158,90,501},{103,31,496}},
		{{166,90,4085},{111,31,4080},{166,90,501},{111,31,496}},
		{{0,70,3328},{0,70,3328},{0,17,448},{0,17,448}},
		{{0,117,3904},{0,117,3904},{0,35,384},{0,35,384}},
		{{13,165,3905},{13,165,3904},{2,211,480},{2,211,480}},
		{{21,165,3906},{21,165,3904},{1,51,488},{1,51,488}},
		{{30,165,3906},{30,165,3904},{7,61,352},{7,61,352}},
		{{38,165,3906},{38,165,3904},{2,125,352},{2,125,352}},
		{{46,165,3906},{46,165,3904},{1,37,500},{10,125,352}},
		{{54,165,3906},{54,165,3904},{9,37,500},{5,61,424}},
		{{63,165,3906},{63,165,3904},{18,37,500},{1,189,424}},
		{{71,165,3906},{71,165,3904},{26,37,500},{9,189,424}},
		{{79,165,3906},{79,165,3904},{34,37,500},{4,77,424}},
		{{87,165,3906},{87,165,3904},{42,37,500},{12,77,424}},
		{{96,165,3906},{96,165,3904},{51,37,500},{8,93,424}},
		{{104,165,3906},{104,165,3904},{59,37,500},{3,141,496}},
		{{112,165,3906},{112,165,3904},{68,37,500},{11,141,496}},
		{{120,165,3906},{120,165,3904},{76,37,500},{6,93,496}},
		{{129,165,3906},{129,165,3904},{85,37,500},{15,93,496}},
		{{70,254,4012},{137,165,3904},{93,37,500},{23,93,496}},
		{{145,165,3906},{145,165,3904},{101,37,500},{31,93,496}},
		{{86,254,4012},{153,165,3904},{109,37,500},{39,93,496}},
		{{163,165,3906},{162,165,3904},{118,37,500},{48,93,496}},
		{{171,165,3906},{170,165,3904},{126,37,500},{56,93,496}},
		{{179,165,3906},{178,165,3904},{134,37,500},{64,93,496}},
		{{187,165,3906},{187,165,3904},{142,37,500},{72,93,496}},
		{{196,165,3906},{196,165,3904},{151,37,500},{81,93,496}},
		{{204,165,3906},{204,165,3904},{159,37,500},{89,93,496}},
		{{212,165,3906},{136,77,4008},{167,37,500},{97,93,496}},
		{{220,165,3906},{131,93,4008},{175,37,500},{105,93,496}},
		{{214,181,4001},{140,93,4008},{184,37,500},{114,93,496}},
		{{222,181,4001},{148,93,4008},{192,37,500},{122,93,496}},
		{{115,95,4085},{99,31,4080},{115,95,501},{99,31,496}},
		{{123,95,4085},{107,31,4080},{123,95,501},{107,31,496}},
		{{0,102,3840},{0,102,3840},{0,18,384},{0,18,384}},
		{{5,167,3904},{5,167,3904},{0,13,256},{0,13,256}},
		{{4,54,3968},{4,54,3968},{1,67,448},{1,67,448}},
		{{30,198,3850},{30,198,3848},{0,3,480},{0,3,480}},
		{{39,198,3850},{39,198,3848},{3,52,488},{3,52,488}},
		{{47,198,3851},{47,198,3848},{3,4,488},{3,4,488}},
		{{55,198,3851},{55,198,3848},{1,70,488},{1,70,488}},
		{{53,167,3906},{63,198,3848},{3,22,488},{3,22,488}},
		{{62,167,3906},{72,198,3848},{24,118,488},{0,6,496}},
		{{70,167,3906},{80,198,3848},{32,118,488},{2,89,488}},
		{{78,167,3906},{88,198,3848},{40,118,488},{1,73,496}},
		{{86,167,3906},{96,198,3848},{48,118,488},{0,28,424}},
		{{95,167,3906},{105,198,3848},{57,118,488},{9,28,424}},
		{{103,167,3906},{113,198,3848},{65,118,488},{5,108,496}},
		{{111,167,3906},{121,198,3848},{73,118,488},{13,108,496}},
		{{119,167,3906},{129,198,3848},{81,118,488},{21,108,496}},
		{{128,167,3906},{138,198,3848},{90,118,488},{6,28,496}},
		{{136,167,3906},{146,198,3848},{98,118,488},{14,28,496}},
		{{145,167,3906},{154,198,3848},{106,118,488},{22,28,496}},
		{{153,167,3906},{162,198,3848},{114,118,488},{30,28,496}},
		{{162,167,3906},{171,198,3848},{123,118,488},{39,28,496}},
		{{170,167,3906},{179,198,3848},{131,118,488},{47,28,496}},
		{{178,167,3906},{187,198,3848},{139,118,488},{55,28,496}},
		{{186,167,3906},{195,198,3848},{147,118,488},{63,28,496}},
		{{194,167,3906},{120,12,4008},{156,118,488},{72,28,496}},
		{{206,198,3907},{116,28,4008},{164,118,488},{80,28,496}},
		{{214,198,3907},{124,28,4008},{172,118,488},{88,28,496}},
		{{222,198,3395},{132,28,4008},{180,118,488},{96,28,496}},
		{{207,134,4001},{141,28,4008},{189,118,488},{105,28,496}},
		{{95,30,4085},{86,31,4080},{95,30,501},{86,31,496}},
		{{103,30,4085},{94,31,4080},{103,30,501},{94,31,496}},
		{{111,30,4085},{102,31,4080},{111,30,501},{102,31,496}},
		{{0,104,3840},{0,104,3840},{0,18,448},{0,18,448}},
		{{4,39,3904},{4,39,3904},{0,4,384},{0,4,384}},
		{{0,56,3968},{0,56,3968},{0,84,448},{0,84,448}},
		{{6,110,3328},{6,110,3328},{0,20,448},{0,20,448}},
		{{41,200,3850},{41,200,3848},{1,4,480},{1,4,480}},
		{{49,200,3850},{49,200,3848},{1,8,416},{1,8,416}},
		{{57,200,3851},{57,200,3848},{1,38,488},{1,38,488}},
		{{65,200,3851},{65,200,3848},{1,120,488},{1,120,488}},
		{{74,200,3851},{74,200,3848},{2,72,488},{2,72,488}},
		{{68,6,3907},{82,200,3848},{2,24,488},{2,24,488}},
		{{77,6,3907},{90,200,3848},{26,120,488},{10,24,488}},
		{{97,63,3330},{98,200,3848},{34,120,488},{2,8,496}},
		{{106,63,3330},{107,200,3848},{43,120,488},{3,92,488}},
		{{114,63,3330},{115,200,3848},{51,120,488},{11,92,488}},
		{{122,63,3330},{123,200,3848},{59,120,488},{7,76,496}},
		{{130,63,3330},{131,200,3848},{67,120,488},{15,76,496}},
		{{139,63,3330},{140,200,3848},{76,120,488},{24,76,496}},
		{{147,63,3330},{148,200,3848},{84,120,488},{32,76,496}},
		{{155,63,3330},{156,200,3848},{92,120,488},{40,76,496}},
		{{164,63,3330},{164,200,3848},{100,120,488},{48,76,496}},
		{{173,63,3330},{173,200,3848},{109,120,488},{57,76,496}},
		{{184,6,3851},{181,200,3848},{117,120,488},{65,76,496}},
		{{192,6,3851},{133,28,3936},{125,120,488},{73,76,496}},
		{{189,200,3907},{141,28,3936},{133,120,488},{81,76,496}},
		{{198,200,3907},{138,108,4000},{142,120,488},{90,76,496}},
		{{206,200,3907},{146,108,4000},{150,120,488},{98,76,496}},
		{{214,200,3395},{154,108,4000},{158,120,488},{106,76,496}},
		{{190,136,4001},{162,108,4000},{166,120,488},{114,76,496}},
		{{123,30,4076},{87,15,4080},{123,30,492},{87,15,496}},
		{{117,110,4084},{80,31,4080},{117,110,500},{80,31,496}},
		{{125,110,4084},{88,31,4080},{125,110,500},{88,31,496}},
		{{133,110,4084},{96,31,4080},{133,110,500},{96,31,496}},
		{{9,56,3904},{9,56,3904},{0,67,448},{0,67,448}},
		{{1,8,3904},{1,8,3904},{1,84,448},{1,84,448}},
		{{1,124,3904},{1,124,3904},{0,39,384},{0,39,384}},
		{{9,124,3904},{9,124,3904},{1,4,448},{1,4,448}},
		{{6,76,3904},{6,76,3904},{0,70,448},{0,70,448}},
		{{62,6,3859},{62,6,3856},{2,38,480},{2,38,480}},
		{{70,6,3859},{70,6,3856},{5,43,416},{5,43,416}},
		{{78,6,3859},{78,6,3856},{2,11,416},{2,11,416}},
		{{87,6,3859},{87,6,3856},{0,171,488},{0,171,488}},
		{{67,8,3906},{95,6,3856},{8,171,488},{8,171,488}},
		{{75,8,3907},{103,6,3856},{5,123,488},{5,123,488}},
		{{83,8,3907},{111,6,3856},{2,75,488},{2,75,488}},
		{{92,8,3907},{120,6,3856},{0,27,488},{0,27,488}},
		{{100,8,3907},{128,6,3856},{8,27,488},{8,27,488}},
		{{120,106,3843},{136,6,3856},{99,6,387},{16,27,488}},
		{{128,106,3843},{144,6,3856},{107,6,387},{2,11,496}},
		{{137,106,3843},{153,6,3856},{117,6,387},{11,11,496}},
		{{145,106,3843},{161,6,3856},{125,6,387},{19,11,496}},
		{{163,8,3851},{137,43,3904},{133,6,387},{27,11,496}},
		{{171,8,3851},{145,43,3904},{141,6,387},{35,11,496}},
		{{180,8,3851},{110,11,4000},{150,6,387},{44,11,496}},
		{{188,8,3851},{118,11,4000},{158,6,387},{52,11,496}},
		{{172,72,3907},{126,11,4000},{166,6,387},{60,11,496}},
		{{174,6,3971},{134,11,4000},{174,6,387},{68,11,496}},
		{{183,6,3971},{143,11,4000},{183,6,387},{77,11,496}},
		{{191,6,3971},{151,11,4000},{191,6,387},{85,11,496}},
		{{199,6,3971},{159,11,4000},{199,6,387},{93,11,496}},
		{{92,12,4084},{69,15,4080},{92,12,500},{69,15,496}},
		{{101,12,4084},{78,15,4080},{101,12,500},{78,15,496}},
		{{110,12,4084},{86,15,4080},{110,12,500},{86,15,496}},
		{{118,12,4084},{79,31,4080},{118,12,500},{79,31,496}},
		{{126,12,4084},{87,31,4080},{126,12,500},{87,31,496}},
		{{71,8,3602},{71,8,3600},{2,21,384},{2,21,384}},
		{{79,8,3611},{79,8,3608},{0,69,448},{0,69,448}},
		{{87,8,3611},{87,8,3608},{0,23,384},{0,23,384}},
		{{95,8,3611},{95,8,3608},{1,5,448},{1,5,448}},
		{{104,8,3611},{104,8,3608},{0,88,448},{0,88,448}},
		{{112,8,3611},{112,8,3608},{0,72,448},{0,72,448}},
		{{120,8,3611},{121,8,3608},{36,21,458},{36,21,456}},
		{{133,47,3091},{129,8,3608},{44,21,458},{44,21,456}},
		{{142,47,3091},{138,8,3608},{53,21,459},{53,21,456}},
		{{98,12,3850},{98,12,3848},{61,21,459},{61,21,456}},
		{{106,12,3850},{106,12,3848},{10,92,480},{69,21,456}},
		{{114,12,3851},{114,12,3848},{18,92,480},{77,21,456}},
		{{123,12,3851},{123,12,3848},{3,44,488},{86,21,456}},
		{{95,12,3906},{95,12,3904},{11,44,488},{94,21,456}},
		{{103,12,3906},{103,12,3904},{19,44,488},{102,21,456}},
		{{111,12,3907},{111,12,3904},{27,44,489},{110,21,456}},
		{{120,12,3907},{120,12,3904},{36,44,489},{119,21,456}},
		{{128,12,3907},{128,12,3904},{44,44,489},{127,21,456}},
		{{136,12,3907},{136,12,3904},{52,44,489},{135,21,456}},
		{{144,12,3907},{144,12,3904},{60,44,490},{144,21,456}},
		{{153,12,3907},{153,12,3904},{69,44,490},{153,21,456}},
		{{161,12,3395},{149,188,3968},{77,44,490},{161,21,456}},
		{{169,12,3395},{199,21,3928},{85,44,490},{169,21,456}},
		{{113,95,4001},{202,69,3992},{125,8,483},{177,21,456}},
		{{122,95,4001},{201,21,3984},{134,8,483},{186,21,456}},
		{{143,8,4067},{209,21,3984},{142,8,483},{194,21,456}},
		{{151,8,4067},{47,15,4080},{151,8,483},{47,15,496}},
		{{159,8,4067},{55,15,4080},{159,8,483},{55,15,496}},
		{{168,8,4067},{64,15,4080},{168,8,483},{64,15,496}},
		{{160,40,4075},{72,15,4080},{160,40,491},{72,15,496}},
		{{168,40,4075},{80,15,4080},{168,40,491},{80,15,496}},
		{{144,8,4082},{88,15,4080},{144,8,498},{88,15,496}},
	};

	static void convert_etc1s_to_etc2_eac_r11(eac_block* pDst_block, const endpoint* pEndpoints, const selector* pSelector)
	{
		const uint32_t low_selector = pSelector->m_lo_selector;
		const uint32_t high_selector = pSelector->m_hi_selector;

		const color32& base_color = pEndpoints->m_color5;
		const uint32_t inten_table = pEndpoints->m_inten5;

		if (low_selector == high_selector)
		{
			uint32_t r;
			decoder_etc_block::get_block_color5_r(base_color, inten_table, low_selector, r);

			// Constant alpha block
			// Select table 13, use selector 4 (0), set multiplier to 1 and base color r
			pDst_block->m_base = r;
			pDst_block->m_table = 13;
			pDst_block->m_multiplier = 1;

			// selectors are all 4's
			static const uint8_t s_etc2_eac_r11_sel4[6] = { 0x92, 0x49, 0x24, 0x92, 0x49, 0x24 };
			memcpy(pDst_block->m_selectors, s_etc2_eac_r11_sel4, sizeof(s_etc2_eac_r11_sel4));

			return;
		}

		uint32_t selector_range_table = 0;
		for (selector_range_table = 0; selector_range_table < NUM_ETC2_EAC_SELECTOR_RANGES; selector_range_table++)
			if ((low_selector == s_etc2_eac_selector_ranges[selector_range_table].m_low) && (high_selector == s_etc2_eac_selector_ranges[selector_range_table].m_high))
				break;
		if (selector_range_table >= NUM_ETC2_EAC_SELECTOR_RANGES)
			selector_range_table = 0;

		const etc1_g_to_eac_conversion* pTable_entry = &s_etc1_g_to_etc2_r11[base_color.r + inten_table * 32][selector_range_table];

		pDst_block->m_base = pTable_entry->m_base;
		pDst_block->m_table = pTable_entry->m_table_mul >> 4;
		pDst_block->m_multiplier = pTable_entry->m_table_mul & 15;

		uint64_t selector_bits = 0;

		for (uint32_t y = 0; y < 4; y++)
		{
			for (uint32_t x = 0; x < 4; x++)
			{
				uint32_t s = pSelector->get_selector(x, y);

				uint32_t ds = (pTable_entry->m_trans >> (s * 3)) & 7;

				const uint32_t dst_ofs = 45 - (y + x * 4) * 3;
				selector_bits |= (static_cast<uint64_t>(ds) << dst_ofs);
			}
		}

		pDst_block->set_selector_bits(selector_bits);
	}
#endif // BASISD_SUPPORT_ETC2_EAC_RG11

// ASTC
	struct etc1_to_astc_solution
	{
		uint8_t m_lo;
		uint8_t m_hi;
		uint16_t m_err;
	};

#if BASISD_SUPPORT_ASTC
	static dxt_selector_range g_etc1_to_astc_selector_ranges[] =
	{
		{ 0, 3 },

		{ 1, 3 },
		{ 0, 2 },

		{ 1, 2 },

		{ 2, 3 },
		{ 0, 1 },
	};

	const uint32_t NUM_ETC1_TO_ASTC_SELECTOR_RANGES = sizeof(g_etc1_to_astc_selector_ranges) / sizeof(g_etc1_to_astc_selector_ranges[0]);

	static uint32_t g_etc1_to_astc_selector_range_index[4][4];

	const uint32_t NUM_ETC1_TO_ASTC_SELECTOR_MAPPINGS = 10;
	static const uint8_t g_etc1_to_astc_selector_mappings[NUM_ETC1_TO_ASTC_SELECTOR_MAPPINGS][4] =
	{
		{ 0, 0, 1, 1 },
		{ 0, 0, 1, 2 },
		{ 0, 0, 1, 3 },
		{ 0, 0, 2, 3 },
		{ 0, 1, 1, 1 },
		{ 0, 1, 2, 2 },
		{ 0, 1, 2, 3 },
		{ 0, 2, 3, 3 },
		{ 1, 2, 2, 2 },
		{ 1, 2, 3, 3 },
	};

	static const etc1_to_astc_solution g_etc1_to_astc[32 * 8 * NUM_ETC1_TO_ASTC_SELECTOR_MAPPINGS * NUM_ETC1_TO_ASTC_SELECTOR_RANGES] = {
/**** start inlining basisu_transcoder_tables_astc.inc ****/
{0,2,18},{0,32,2},{0,16,10},{0,16,10},{0,32,35},{0,16,27},{0,16,11},{0,16,27},{0,16,36},{0,16,28},{0,2,18},{0,32,2},{0,16,10},{0,16,10},{16,0,35},{0,16,27},{0,16,11},{0,16,27},{32,0,35},{0,16,27},{0,16,1},{0,16,1},{0,16,1},{0,16,1},{0,16,2},{0,16,2},{0,16,2},{0,0,4},{0,0,4},{0,0,4},{0,16,1},
{0,16,1},{0,16,1},{0,16,1},{0,16,2},{0,16,2},{0,16,2},{0,0,4},{16,0,2},{0,0,4},{0,2,18},{0,32,2},{0,16,10},{0,16,10},{0,2,18},{2,0,18},{0,16,10},{0,16,18},{2,0,18},{0,16,18},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{16,4,44},{16,18,27},{16,2,30},
{16,2,30},{0,4,52},{0,18,20},{0,2,1},{0,32,27},{0,2,77},{0,32,36},{16,4,19},{16,18,2},{16,2,5},{16,2,5},{32,32,51},{0,18,20},{0,2,1},{0,32,27},{32,32,51},{0,32,27},{16,18,26},{16,18,26},{16,18,26},{16,32,27},{0,34,8},{0,2,1},{0,2,1},{0,32,2},{0,32,19},{0,32,11},{16,18,1},{16,18,1},{16,18,1},{16,32,2},{16,2,8},
{0,2,1},{0,2,1},{0,32,2},{34,0,8},{0,32,2},{16,4,18},{16,18,1},{16,2,4},{0,2,0},{16,4,18},{4,16,18},{0,2,0},{0,32,26},{4,16,18},{0,32,26},{16,0,26},{16,0,26},{16,0,26},{16,0,26},{0,18,0},{0,18,0},{0,18,0},{0,32,1},{0,16,10},{0,16,10},{32,36,36},{32,4,20},{32,34,31},{32,18,28},{16,6,60},{32,34,31},{32,18,19},
{16,18,28},{0,34,52},{0,18,22},{2,4,22},{2,34,6},{2,18,14},{2,18,14},{18,2,51},{32,34,22},{32,18,10},{0,18,18},{38,0,51},{0,18,18},{32,4,19},{32,4,19},{32,4,19},{32,18,19},{32,34,17},{32,18,10},{32,18,10},{32,2,13},{16,18,14},{16,2,8},{2,18,5},{2,18,5},{2,18,5},{2,18,5},{2,2,8},{32,18,1},{32,18,1},{32,2,4},{2,2,8},
{32,2,4},{4,0,18},{32,4,2},{2,18,10},{0,34,9},{4,0,18},{8,0,18},{0,34,9},{0,18,18},{8,0,18},{0,18,18},{32,0,18},{32,0,18},{32,0,18},{32,0,18},{32,18,9},{32,18,9},{32,18,9},{32,2,9},{16,2,4},{16,2,4},{18,6,44},{18,20,26},{18,4,30},{18,4,30},{2,6,52},{2,20,21},{2,4,1},{2,34,27},{16,20,52},{32,34,27},{18,6,19},
{18,20,1},{18,4,5},{18,4,5},{20,16,51},{2,20,21},{2,4,1},{2,34,27},{16,20,51},{2,34,27},{18,20,26},{18,20,26},{18,20,26},{18,34,27},{2,36,8},{2,4,1},{2,4,1},{2,34,2},{32,4,11},{32,34,2},{18,20,1},{18,20,1},{18,20,1},{18,34,2},{18,4,8},{2,4,1},{2,4,1},{2,34,2},{36,2,8},{2,34,2},{36,0,18},{18,20,0},{18,4,4},
{2,4,0},{36,0,18},{6,18,18},{2,4,0},{0,34,26},{6,18,18},{0,34,26},{18,0,26},{18,0,26},{18,0,26},{18,0,26},{2,4,1},{2,4,1},{2,4,1},{2,34,1},{32,34,1},{32,34,1},{34,8,40},{34,6,21},{34,36,33},{34,36,28},{34,22,55},{34,36,25},{34,36,13},{34,20,31},{32,6,55},{18,20,20},{4,6,21},{4,36,5},{4,36,5},{4,20,11},{22,0,51},
{34,36,21},{34,36,9},{2,20,19},{32,6,51},{2,20,19},{34,6,21},{34,6,21},{34,6,21},{34,20,21},{34,6,14},{34,20,6},{34,20,6},{34,4,14},{18,20,11},{18,20,11},{4,36,1},{4,36,1},{4,36,1},{4,20,2},{2,24,8},{34,20,2},{34,20,2},{4,4,10},{28,0,8},{4,4,10},{22,16,18},{34,6,1},{4,36,4},{18,36,4},{22,16,18},{44,16,18},{18,36,4},
{0,20,18},{44,16,18},{0,20,18},{34,0,20},{34,0,20},{34,0,20},{34,0,20},{34,20,5},{34,20,5},{34,20,5},{34,4,5},{18,20,2},{18,20,2},{20,8,46},{20,22,27},{20,6,31},{20,6,28},{4,24,52},{4,22,19},{4,6,7},{4,6,36},{18,22,56},{34,6,31},{20,8,21},{20,22,2},{20,6,6},{20,6,3},{36,6,51},{4,22,18},{4,6,6},{18,6,26},{31,0,51},
{18,6,26},{20,22,26},{20,22,26},{20,22,26},{20,6,27},{4,38,9},{4,6,6},{4,6,6},{4,36,1},{34,6,9},{4,36,4},{20,22,1},{20,22,1},{20,22,1},{20,6,2},{22,2,8},{20,36,4},{20,36,4},{4,36,0},{42,2,8},{4,36,0},{22,4,18},{20,22,1},{36,6,2},{4,6,2},{22,4,18},{15,0,18},{4,6,2},{0,6,26},{15,0,18},{0,6,26},{20,0,26},
{20,0,26},{20,0,26},{20,0,26},{4,22,1},{4,22,1},{4,22,1},{4,36,1},{4,36,4},{4,36,4},{36,40,38},{36,8,22},{36,38,33},{36,38,28},{36,24,55},{36,38,25},{36,38,13},{20,22,28},{34,8,55},{4,22,23},{6,8,21},{6,38,5},{6,38,5},{6,22,11},{24,2,51},{36,38,21},{20,38,9},{4,22,19},{13,0,51},{4,22,19},{36,8,21},{36,8,21},{36,8,21},
{36,22,21},{36,8,14},{36,22,6},{36,22,6},{36,6,14},{4,38,10},{4,22,14},{6,38,1},{6,38,1},{6,38,1},{6,22,2},{8,2,8},{36,22,2},{36,22,2},{6,6,10},{30,2,8},{6,6,10},{10,0,18},{36,8,2},{6,38,4},{4,38,5},{10,0,18},{4,8,18},{4,38,5},{0,22,18},{4,8,18},{0,22,18},{36,0,20},{36,0,20},{36,0,20},{36,0,20},{36,22,5},
{36,22,5},{36,22,5},{36,6,5},{4,22,5},{4,22,5},{22,10,46},{22,24,26},{22,8,30},{22,8,30},{6,26,52},{6,24,21},{6,8,5},{6,38,37},{20,24,53},{6,38,40},{22,10,21},{22,24,1},{22,8,5},{22,8,5},{16,11,51},{6,24,20},{6,8,4},{6,38,36},{25,0,51},{6,38,36},{22,24,26},{22,24,26},{22,24,26},{22,8,30},{6,40,9},{6,8,5},{6,8,5},
{6,38,1},{36,8,11},{6,38,4},{22,24,1},{22,24,1},{22,24,1},{22,8,5},{24,4,8},{6,8,4},{6,8,4},{6,38,0},{11,0,8},{6,38,0},{42,0,18},{22,24,0},{22,8,4},{6,8,0},{42,0,18},{9,0,18},{6,8,0},{0,8,36},{9,0,18},{0,8,36},{22,0,26},{22,0,26},{22,0,26},{22,0,26},{6,24,1},{6,24,1},{6,24,1},{6,38,1},{6,38,4},
{6,38,4},{38,12,44},{38,10,30},{38,40,46},{38,40,34},{38,26,53},{38,40,21},{38,40,9},{38,24,25},{6,10,57},{22,24,22},{8,10,21},{8,40,5},{8,40,5},{8,24,17},{28,0,51},{38,40,20},{38,40,8},{6,24,21},{37,0,51},{6,24,21},{38,26,26},{38,26,26},{38,26,26},{38,24,30},{38,10,9},{38,40,5},{38,40,5},{38,24,9},{22,24,11},{22,24,6},{8,40,1},
{8,40,1},{8,40,1},{8,24,1},{38,10,8},{8,24,4},{8,24,4},{6,24,5},{23,0,8},{6,24,5},{16,7,18},{38,10,4},{8,40,4},{6,40,5},{16,7,18},{21,0,18},{6,40,5},{0,24,20},{21,0,18},{0,24,20},{38,0,26},{38,0,26},{38,0,26},{38,0,26},{38,40,1},{38,40,1},{38,40,1},{38,8,2},{22,24,2},{22,24,2},{24,12,38},{24,26,22},{24,10,30},
{24,10,22},{8,28,52},{8,26,19},{8,10,6},{8,40,37},{38,26,55},{38,10,30},{24,12,22},{24,26,6},{24,10,14},{24,10,6},{42,6,51},{8,26,18},{24,10,3},{22,10,26},{19,0,51},{22,10,26},{24,26,21},{24,26,21},{24,26,21},{24,10,21},{8,42,9},{8,10,5},{8,10,5},{8,40,1},{38,10,8},{8,40,4},{24,26,5},{24,26,5},{24,26,5},{24,10,5},{28,2,8},
{24,10,2},{24,10,2},{8,40,0},{42,8,8},{8,40,0},{26,8,18},{24,26,2},{40,10,2},{8,10,1},{26,8,18},{15,6,18},{8,10,1},{0,10,26},{15,6,18},{0,10,26},{24,0,20},{24,0,20},{24,0,20},{24,0,20},{8,26,1},{8,26,1},{8,26,1},{8,40,1},{38,10,4},{38,10,4},{40,14,40},{40,12,21},{40,42,33},{40,42,33},{40,28,55},{40,42,23},{40,42,18},
{40,26,33},{8,12,57},{24,26,22},{10,12,21},{10,42,5},{10,42,5},{10,26,17},{30,2,51},{40,42,19},{24,42,14},{8,26,21},{35,2,51},{8,26,21},{40,12,21},{40,12,21},{40,12,21},{40,26,21},{40,12,14},{40,26,6},{40,26,6},{40,10,14},{8,42,10},{24,26,6},{10,42,1},{10,42,1},{10,42,1},{10,26,1},{16,3,8},{40,26,2},{40,26,2},{8,26,5},{17,0,8},
{8,26,5},{16,1,18},{40,12,1},{10,42,4},{8,42,5},{16,1,18},{19,2,18},{8,42,5},{0,26,20},{19,2,18},{0,26,20},{40,0,20},{40,0,20},{40,0,20},{40,0,20},{40,26,5},{40,26,5},{40,26,5},{40,10,5},{24,26,2},{24,26,2},{26,14,38},{26,28,22},{26,12,30},{26,12,22},{10,30,52},{10,28,21},{10,12,6},{10,42,37},{24,28,53},{40,12,31},{26,14,22},
{26,28,6},{26,12,14},{26,12,6},{47,2,51},{10,28,20},{26,12,3},{24,12,26},{17,2,51},{24,12,26},{26,28,21},{26,28,21},{26,28,21},{26,12,21},{10,44,9},{10,12,5},{10,12,5},{10,42,1},{40,12,9},{40,42,2},{26,28,5},{26,28,5},{26,28,5},{26,12,5},{30,4,8},{26,12,2},{26,12,2},{10,42,0},{15,8,8},{10,42,0},{15,0,18},{26,28,2},{42,12,5},
{10,12,1},{15,0,18},{5,4,18},{10,12,1},{0,12,26},{5,4,18},{0,12,26},{26,0,20},{26,0,20},{26,0,20},{26,0,20},{10,28,1},{10,28,1},{10,28,1},{10,42,1},{40,42,2},{40,42,2},{42,47,46},{42,14,31},{12,44,37},{42,44,31},{42,46,52},{42,44,21},{42,44,6},{42,28,25},{10,14,51},{26,28,20},{12,30,18},{12,14,2},{12,44,1},{12,44,10},{29,0,51},
{42,44,20},{42,44,5},{26,28,20},{37,6,51},{26,28,20},{42,30,26},{42,30,26},{42,30,26},{42,28,30},{42,14,9},{42,44,5},{42,44,5},{42,28,9},{10,44,9},{26,28,4},{12,44,0},{12,44,0},{12,44,0},{12,28,0},{45,0,8},{12,28,4},{12,28,4},{26,28,4},{27,8,8},{26,28,4},{45,2,18},{12,14,2},{12,44,1},{26,44,1},{45,2,18},{25,8,18},{26,44,1},
{0,28,20},{25,8,18},{0,28,20},{42,0,26},{42,0,26},{42,0,26},{42,0,26},{42,44,1},{42,44,1},{42,44,1},{42,12,5},{26,28,0},{26,28,0},{28,47,38},{28,30,22},{28,14,33},{28,14,25},{12,31,55},{12,30,23},{12,14,18},{12,14,33},{42,30,55},{42,14,21},{28,47,22},{28,30,6},{44,14,14},{28,14,9},{46,10,51},{12,30,19},{28,14,6},{26,14,21},{23,8,51},
{26,14,21},{28,30,21},{28,30,21},{28,30,21},{28,14,21},{12,46,14},{12,14,14},{12,14,14},{12,44,6},{42,14,10},{12,44,6},{28,30,5},{28,30,5},{28,30,5},{28,14,5},{29,2,8},{28,14,2},{28,14,2},{12,44,2},{35,6,8},{12,44,2},{8,7,18},{28,30,2},{44,14,5},{12,14,5},{8,7,18},{33,6,18},{12,14,5},{0,14,20},{33,6,18},{0,14,20},{28,0,20},
{28,0,20},{28,0,20},{28,0,20},{12,30,5},{12,30,5},{12,30,5},{12,44,5},{42,14,1},{42,14,1},{44,15,46},{44,47,27},{14,46,37},{44,46,31},{44,31,53},{44,46,21},{44,46,6},{44,30,25},{42,47,55},{28,30,22},{14,31,20},{14,47,8},{14,46,1},{14,46,10},{38,3,51},{44,46,20},{44,46,5},{12,30,21},{35,8,51},{12,30,21},{44,31,26},{44,31,26},{44,31,26},
{44,30,30},{44,47,11},{44,46,5},{44,46,5},{44,30,9},{12,46,9},{28,30,6},{14,46,0},{14,46,0},{14,46,0},{14,30,0},{43,2,8},{14,30,4},{14,30,4},{12,30,5},{21,8,8},{12,30,5},{41,0,18},{44,47,1},{14,46,1},{28,46,1},{41,0,18},{19,8,18},{28,46,1},{0,30,20},{19,8,18},{0,30,20},{44,0,26},{44,0,26},{44,0,26},{44,0,26},{44,46,1},
{44,46,1},{44,46,1},{44,14,5},{28,30,2},{28,30,2},{30,45,38},{30,31,21},{30,47,24},{30,47,24},{14,29,55},{14,31,22},{14,47,8},{14,46,55},{28,31,53},{44,47,41},{30,45,22},{30,31,5},{30,47,8},{30,47,8},{41,2,51},{14,31,18},{14,47,4},{12,47,37},{17,8,51},{12,47,37},{30,31,20},{30,31,20},{30,31,20},{30,47,24},{14,15,14},{14,47,8},{14,47,8},
{14,46,6},{44,47,9},{14,46,6},{30,31,4},{30,31,4},{30,31,4},{30,47,8},{38,1,8},{14,47,4},{14,47,4},{14,46,2},{43,28,8},{14,46,2},{9,0,18},{30,31,1},{46,47,1},{14,47,0},{9,0,18},{0,9,18},{14,47,0},{0,47,36},{0,9,18},{0,47,36},{30,0,20},{30,0,20},{30,0,20},{30,0,20},{14,31,4},{14,31,4},{14,31,4},{14,46,5},{14,46,5},
{14,46,5},{46,43,54},{46,45,41},{47,15,55},{46,15,44},{46,13,51},{46,15,20},{46,15,8},{46,31,24},{14,45,56},{30,31,21},{47,29,21},{47,15,6},{47,15,6},{47,31,17},{23,0,51},{46,15,20},{46,15,8},{30,31,20},{18,9,51},{30,31,20},{46,13,37},{46,13,37},{46,13,37},{46,15,40},{46,45,8},{46,15,4},{46,15,4},{46,31,8},{14,15,14},{30,31,5},{47,15,2},
{47,15,2},{47,15,2},{47,31,1},{39,0,8},{46,15,4},{46,15,4},{30,31,4},{9,28,8},{30,31,4},{39,2,18},{47,15,5},{47,15,5},{30,15,4},{39,2,18},{34,9,18},{30,15,4},{0,31,20},{34,9,18},{0,31,20},{46,0,36},{46,0,36},{46,0,36},{46,0,36},{46,15,0},{46,15,0},{46,15,0},{46,47,1},{30,31,1},{30,31,1},{31,43,38},{31,29,22},{31,45,25},
{31,45,22},{47,43,53},{47,29,19},{47,45,6},{47,15,37},{46,29,52},{46,45,27},{31,43,22},{31,29,6},{31,45,9},{31,45,6},{9,6,51},{47,29,18},{47,45,5},{30,45,26},{32,7,51},{30,45,26},{31,13,21},{31,13,21},{31,13,21},{31,45,21},{47,13,9},{47,45,5},{47,45,5},{47,15,1},{46,45,10},{46,15,8},{31,13,5},{31,13,5},{31,13,5},{31,45,5},{23,2,8},
{31,15,4},{31,15,4},{47,15,0},{20,9,8},{47,15,0},{47,11,18},{31,29,2},{15,45,5},{47,45,1},{47,11,18},{22,9,18},{47,45,1},{0,45,26},{22,9,18},{0,45,26},{31,0,20},{31,0,20},{31,0,20},{31,0,20},{47,29,1},{47,29,1},{47,29,1},{47,15,1},{46,45,1},{46,45,1},{15,11,38},{15,43,21},{15,13,33},{15,13,33},{15,27,55},{15,13,23},{15,13,18},
{15,29,33},{47,13,59},{31,29,22},{45,27,21},{45,13,6},{45,13,6},{45,29,17},{21,2,51},{15,13,19},{15,13,14},{31,29,21},{24,9,51},{31,29,21},{15,27,21},{15,27,21},{15,27,21},{15,29,21},{15,13,14},{15,29,6},{15,29,6},{15,45,14},{47,13,10},{31,29,6},{45,13,2},{45,13,2},{45,13,2},{45,29,1},{37,2,8},{15,29,2},{15,29,2},{31,29,5},{34,7,8},
{31,29,5},{35,0,18},{15,43,1},{45,13,5},{31,13,5},{35,0,18},{36,7,18},{31,13,5},{0,29,20},{36,7,18},{0,29,20},{15,0,20},{15,0,20},{15,0,20},{15,0,20},{15,13,5},{15,13,5},{15,13,5},{15,45,5},{31,29,2},{31,29,2},{29,41,38},{29,27,20},{29,43,25},{29,43,25},{45,41,53},{45,27,21},{45,43,6},{45,13,37},{31,27,53},{15,43,31},{29,41,22},
{29,27,4},{29,43,9},{29,43,9},{35,2,51},{45,27,20},{45,43,5},{31,43,26},{34,5,51},{31,43,26},{29,27,20},{29,27,20},{29,27,20},{29,13,24},{45,11,9},{45,43,5},{45,43,5},{45,13,1},{15,43,9},{15,13,2},{29,27,4},{29,27,4},{29,27,4},{29,13,8},{44,1,8},{29,13,4},{29,13,4},{45,13,0},{26,9,8},{45,13,0},{3,0,18},{29,27,0},{13,43,5},
{45,43,1},{3,0,18},{0,3,18},{45,43,1},{0,43,26},{0,3,18},{0,43,26},{29,0,20},{29,0,20},{29,0,20},{29,0,20},{45,27,1},{45,27,1},{45,27,1},{45,13,1},{15,13,2},{15,13,2},{13,9,46},{13,41,31},{43,11,37},{13,11,31},{13,25,53},{13,11,21},{13,11,6},{13,27,30},{29,11,56},{29,27,22},{43,25,18},{43,41,2},{43,11,1},{43,27,16},{17,0,51},
{13,11,20},{13,11,5},{29,27,21},{18,3,51},{29,27,21},{13,25,26},{13,25,26},{13,25,26},{13,27,27},{13,41,9},{13,27,3},{13,27,3},{13,27,14},{45,11,9},{29,27,6},{43,11,0},{43,11,0},{43,11,0},{43,27,0},{31,5,8},{13,27,2},{13,27,2},{29,27,5},{14,9,8},{29,27,5},{31,3,18},{43,41,2},{43,11,1},{29,11,1},{31,3,18},{34,3,18},{29,11,1},
{0,27,20},{34,3,18},{0,27,20},{13,0,26},{13,0,26},{13,0,26},{13,0,26},{13,11,1},{13,11,1},{13,11,1},{13,43,5},{29,27,2},{29,27,2},{27,39,38},{27,25,22},{27,41,33},{27,41,25},{43,23,55},{43,25,23},{43,41,18},{43,41,33},{13,25,55},{13,41,21},{27,39,22},{27,25,6},{11,41,14},{27,41,9},{27,9,51},{43,25,19},{27,41,6},{13,41,21},{14,7,51},
{13,41,21},{27,9,21},{27,9,21},{27,9,21},{27,41,21},{43,9,14},{43,25,14},{43,25,14},{43,11,5},{13,41,10},{43,11,5},{27,9,5},{27,9,5},{27,9,5},{27,41,5},{17,2,8},{27,41,2},{27,41,2},{43,11,1},{16,1,8},{43,11,1},{43,7,18},{27,25,2},{11,41,5},{27,41,5},{43,7,18},{18,1,18},{27,41,5},{0,41,20},{18,1,18},{0,41,20},{27,0,20},
{27,0,20},{27,0,20},{27,0,20},{43,9,5},{43,9,5},{43,9,5},{43,11,4},{13,41,1},{13,41,1},{11,7,46},{11,39,30},{41,9,37},{11,9,31},{11,23,53},{11,9,21},{11,9,6},{11,25,30},{13,39,55},{27,25,22},{41,23,18},{41,9,4},{41,9,1},{41,25,16},{17,6,51},{11,9,20},{11,9,5},{27,25,21},{20,1,51},{27,25,21},{11,23,26},{11,23,26},{11,23,26},
{11,25,27},{11,39,9},{11,25,3},{11,25,3},{11,25,14},{43,9,9},{27,25,6},{41,9,0},{41,9,0},{41,9,0},{41,25,0},{29,3,8},{11,25,2},{11,25,2},{27,25,5},{43,9,8},{27,25,5},{29,1,18},{41,9,4},{41,9,1},{27,9,1},{29,1,18},{36,1,18},{27,9,1},{0,25,20},{36,1,18},{0,25,20},{11,0,26},{11,0,26},{11,0,26},{11,0,26},{11,9,1},
{11,9,1},{11,9,1},{11,41,2},{27,25,2},{27,25,2},{25,37,38},{25,23,22},{25,39,25},{25,39,25},{41,21,55},{41,23,23},{41,39,9},{41,39,46},{27,23,53},{11,39,30},{25,37,22},{25,23,6},{25,39,9},{25,39,9},{27,3,51},{41,23,19},{41,39,5},{27,39,26},{39,9,51},{27,39,26},{25,7,21},{25,7,21},{25,7,21},{25,9,24},{41,7,14},{41,39,8},{41,39,8},
{41,9,5},{11,39,8},{41,9,5},{25,7,5},{25,7,5},{25,7,5},{25,9,8},{39,11,8},{25,9,4},{25,9,4},{41,9,1},{22,1,8},{41,9,1},{41,5,18},{25,23,2},{9,39,2},{41,39,1},{41,5,18},{23,9,18},{41,39,1},{0,39,26},{23,9,18},{0,39,26},{25,0,20},{25,0,20},{25,0,20},{25,0,20},{41,7,5},{41,7,5},{41,7,5},{41,9,4},{41,9,4},
{41,9,4},{39,21,54},{39,7,40},{39,7,37},{9,7,41},{9,5,51},{9,7,20},{9,7,5},{9,23,30},{41,37,51},{25,23,26},{39,21,18},{39,7,4},{39,7,1},{39,7,17},{11,1,51},{9,7,20},{9,7,5},{25,23,26},{5,9,51},{25,23,26},{39,7,36},{39,7,36},{39,7,36},{39,23,36},{9,21,10},{9,7,4},{9,7,4},{9,23,5},{41,7,12},{25,23,1},{39,7,0},
{39,7,0},{39,7,0},{39,23,0},{25,5,8},{9,7,4},{9,7,4},{25,23,1},{10,1,8},{25,23,1},{25,3,18},{39,7,4},{39,7,1},{25,7,1},{25,3,18},{12,1,18},{25,7,1},{0,23,26},{12,1,18},{0,23,26},{9,0,36},{9,0,36},{9,0,36},{9,0,36},{9,7,0},{9,7,0},{9,7,0},{9,23,4},{25,23,0},{25,23,0},{23,19,38},{23,5,23},{23,21,28},
{23,37,23},{39,19,55},{39,21,25},{39,37,13},{39,37,33},{9,21,53},{9,37,22},{7,21,27},{23,5,14},{7,37,14},{23,37,14},{23,5,51},{39,21,21},{23,37,6},{9,37,21},{14,1,51},{9,37,21},{23,5,19},{23,5,19},{23,5,19},{23,37,19},{39,5,14},{39,21,9},{39,21,9},{39,7,5},{9,37,11},{39,7,5},{23,5,10},{23,5,10},{23,5,10},{7,7,10},{9,3,8},
{23,37,2},{23,37,2},{39,7,1},{31,3,8},{39,7,1},{37,7,18},{23,5,5},{7,37,5},{23,37,5},{37,7,18},{30,1,18},{23,37,5},{0,37,20},{30,1,18},{0,37,20},{23,0,18},{23,0,18},{23,0,18},{23,0,18},{39,5,5},{39,5,5},{39,5,5},{39,7,4},{9,37,2},{9,37,2},{7,33,44},{7,35,31},{7,5,36},{7,5,31},{7,19,53},{7,5,22},{7,5,7},
{7,21,31},{39,35,57},{23,21,27},{37,19,20},{37,5,4},{37,5,1},{37,5,17},{23,17,51},{7,5,21},{7,5,6},{23,21,26},{45,17,51},{23,21,26},{7,19,26},{7,19,26},{7,19,26},{7,5,30},{7,35,9},{7,5,6},{7,5,6},{7,21,6},{39,5,9},{23,21,2},{37,5,0},{37,5,0},{37,5,0},{37,21,0},{23,3,8},{37,21,4},{37,21,4},{23,21,1},{43,3,8},
{23,21,1},{23,1,18},{37,5,4},{37,5,1},{23,5,1},{23,1,18},{33,7,18},{23,5,1},{0,21,26},{33,7,18},{0,21,26},{7,0,26},{7,0,26},{7,0,26},{7,0,26},{7,5,2},{7,5,2},{7,5,2},{7,37,2},{23,21,1},{23,21,1},{21,17,38},{21,19,20},{21,35,31},{21,35,23},{37,17,55},{37,19,22},{37,35,13},{37,35,33},{23,19,56},{7,35,21},{5,19,27},
{21,19,11},{5,35,14},{21,35,14},{21,3,51},{37,19,18},{21,35,6},{7,35,21},{43,1,51},{7,35,21},{21,3,19},{21,3,19},{21,3,19},{21,35,19},{37,3,14},{37,35,9},{37,35,9},{37,5,5},{7,35,10},{37,5,5},{21,3,10},{21,3,10},{21,3,10},{5,5,10},{3,25,8},{21,35,2},{21,35,2},{37,5,1},{29,1,8},{37,5,1},{35,5,18},{21,19,2},{5,35,5},
{21,35,5},{35,5,18},{27,1,18},{21,35,5},{0,35,20},{27,1,18},{0,35,20},{21,0,18},{21,0,18},{21,0,18},{21,0,18},{37,19,4},{37,19,4},{37,19,4},{37,5,4},{7,35,1},{7,35,1},{35,17,44},{35,33,27},{35,3,27},{35,3,35},{5,1,51},{5,3,26},{5,3,1},{5,19,30},{37,33,51},{21,19,26},{35,17,19},{35,33,2},{35,3,2},{35,3,10},{5,1,51},
{5,3,26},{5,3,1},{21,19,26},{9,1,51},{21,19,26},{35,3,27},{35,3,27},{35,3,27},{35,19,27},{5,17,10},{5,3,1},{5,3,1},{5,19,5},{37,3,12},{21,19,1},{35,3,2},{35,3,2},{35,3,2},{35,19,2},{19,5,8},{5,3,1},{5,3,1},{21,19,1},{37,3,8},{21,19,1},{19,3,18},{35,33,1},{35,3,1},{5,3,1},{19,3,18},{39,1,18},{5,3,1},
{0,19,26},{39,1,18},{0,19,26},{35,0,26},{35,0,26},{35,0,26},{35,0,26},{5,3,0},{5,3,0},{5,3,0},{5,19,4},{21,19,0},{21,19,0},{19,1,54},{19,1,22},{19,17,28},{19,33,27},{19,1,61},{35,17,30},{19,33,19},{35,33,31},{5,17,52},{5,33,20},{3,1,24},{3,17,8},{3,33,13},{3,33,17},{17,5,51},{35,17,21},{19,33,10},{5,33,19},{5,17,51},
{5,33,19},{19,1,18},{19,1,18},{19,1,18},{19,33,18},{19,17,19},{19,33,10},{19,33,10},{19,3,14},{5,33,11},{35,3,6},{3,33,4},{3,33,4},{3,33,4},{3,3,8},{3,3,8},{19,33,1},{19,33,1},{19,3,5},{3,3,8},{19,3,5},{33,33,18},{19,1,4},{3,33,9},{19,33,9},{33,33,18},{33,33,18},{19,33,9},{0,33,18},{33,33,18},{0,33,18},{19,0,18},
{19,0,18},{19,0,18},{19,0,18},{35,1,9},{35,1,9},{35,1,9},{19,3,10},{35,3,2},{35,3,2},{33,1,76},{33,1,36},{33,1,27},{33,1,35},{33,1,84},{3,1,26},{3,1,1},{3,17,30},{19,1,56},{19,17,27},{17,1,43},{33,1,11},{33,1,2},{33,1,10},{1,3,51},{3,1,26},{3,1,1},{19,17,26},{3,1,51},{19,17,26},{33,1,27},{33,1,27},{33,1,27},
{33,17,26},{3,1,16},{3,1,1},{3,1,1},{3,17,5},{35,1,12},{19,17,2},{33,1,2},{33,1,2},{33,1,2},{33,17,1},{17,3,8},{3,1,1},{3,1,1},{19,17,1},{35,1,8},{19,17,1},{17,1,18},{17,1,10},{33,1,1},{19,1,0},{17,1,18},{33,1,18},{19,1,0},{0,17,26},{33,1,18},{0,17,26},{33,0,26},{33,0,26},{33,0,26},{33,0,26},{3,1,0},
{3,1,0},{3,1,0},{3,17,4},{19,17,1},{19,17,1},{17,1,36},{17,1,28},{17,1,27},{17,1,19},{17,1,28},{17,1,12},{17,1,11},{17,1,10},{33,1,20},{33,1,2},{1,1,4},{1,1,4},{1,1,4},{1,1,4},{1,17,3},{17,1,3},{17,1,2},{17,1,1},{17,1,3},{17,1,1},{17,1,27},{17,1,27},{17,1,27},{17,1,19},{17,1,19},{17,1,11},{17,1,11},
{17,1,10},{33,1,11},{33,1,2},{1,1,4},{1,1,4},{1,1,4},{1,1,4},{1,17,2},{17,1,2},{17,1,2},{17,1,1},{17,1,2},{17,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{17,0,18},{17,0,18},{17,0,18},{17,0,18},{17,1,10},{17,1,10},{17,1,10},{17,1,10},{33,1,2},
{33,1,2},{0,4,74},{0,18,10},{0,2,1},{0,2,26},{0,34,154},{0,2,99},{0,32,49},{0,32,121},{0,32,162},{0,32,130},{0,4,74},{0,18,10},{0,2,1},{0,2,26},{32,0,153},{0,2,99},{0,32,49},{0,32,121},{0,32,153},{0,32,121},{0,2,0},{0,2,0},{0,2,0},{0,16,4},{0,32,13},{0,16,5},{0,16,5},{0,16,9},{0,16,14},{0,16,10},{0,2,0},
{0,2,0},{0,2,0},{0,16,4},{16,0,13},{0,16,5},{0,16,5},{0,16,9},{32,0,13},{0,16,9},{32,32,72},{0,18,10},{0,2,1},{0,2,26},{32,32,72},{32,32,72},{0,2,26},{0,32,72},{32,32,72},{0,32,72},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,38,81},{0,20,10},{16,34,34},
{0,18,26},{0,36,243},{0,34,99},{0,18,26},{0,2,139},{0,18,279},{0,2,164},{16,6,76},{16,4,8},{16,34,9},{16,18,24},{18,0,243},{0,34,99},{0,18,26},{0,2,139},{36,0,243},{0,2,139},{0,20,10},{0,20,10},{0,20,10},{0,18,10},{0,34,50},{0,18,10},{0,18,10},{0,32,20},{0,32,61},{0,32,29},{16,4,4},{16,4,4},{16,4,4},{16,2,5},{16,2,50},
{0,18,10},{0,18,10},{0,32,20},{34,0,50},{0,32,20},{18,2,72},{0,20,1},{16,34,5},{0,18,17},{18,2,72},{38,0,72},{0,18,17},{0,18,80},{38,0,72},{0,18,80},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,32,0},{0,32,0},{0,32,0},{0,16,0},{0,16,1},{0,16,1},{16,24,135},{16,6,66},{16,20,122},{16,4,66},{0,8,244},{0,20,81},{0,4,2},
{0,34,121},{0,4,344},{0,34,185},{32,8,74},{32,36,1},{32,4,17},{32,4,17},{16,38,243},{0,20,81},{0,4,2},{0,34,121},{38,16,243},{0,34,121},{16,22,61},{16,22,61},{16,22,61},{16,4,65},{0,6,52},{0,4,1},{0,4,1},{0,18,9},{0,18,94},{0,18,45},{32,36,0},{32,36,0},{32,36,0},{32,34,0},{2,2,50},{0,4,1},{0,4,1},{0,18,9},{2,2,50},
{0,18,9},{20,16,72},{32,36,1},{2,4,1},{0,4,1},{20,16,72},{16,20,72},{0,4,1},{0,34,72},{16,20,72},{0,34,72},{16,0,61},{16,0,61},{16,0,61},{16,0,61},{0,4,1},{0,4,1},{0,4,1},{0,2,1},{0,32,25},{0,32,25},{32,26,152},{2,22,91},{2,36,119},{32,20,89},{16,10,247},{16,6,78},{16,36,13},{16,4,110},{0,36,293},{0,4,103},{18,8,76},
{18,6,8},{18,36,9},{2,20,19},{20,2,243},{16,6,74},{32,20,9},{0,4,94},{42,0,243},{0,4,94},{32,8,81},{32,8,81},{32,8,81},{32,20,80},{16,8,54},{16,36,9},{16,36,9},{16,34,8},{0,4,77},{0,34,6},{18,6,4},{18,6,4},{18,6,4},{18,4,5},{18,4,50},{32,20,0},{32,20,0},{16,34,4},{36,2,50},{16,34,4},{18,8,72},{2,22,1},{18,36,5},
{0,36,4},{18,8,72},{44,0,72},{0,36,4},{0,4,90},{44,0,72},{0,4,90},{32,0,80},{32,0,80},{32,0,80},{32,0,80},{16,6,4},{16,6,4},{16,6,4},{16,34,4},{0,34,2},{0,34,2},{18,42,146},{18,8,83},{18,22,139},{18,6,79},{2,26,243},{2,38,78},{2,6,9},{2,36,110},{0,38,252},{0,36,79},{34,10,73},{34,8,6},{4,6,21},{34,6,14},{38,0,243},
{2,38,78},{18,6,6},{0,36,75},{30,0,243},{0,36,75},{18,24,75},{18,24,75},{18,24,75},{18,6,75},{2,24,50},{2,6,5},{2,6,5},{2,20,4},{0,6,53},{16,20,2},{34,8,2},{34,8,2},{34,8,2},{34,36,2},{2,24,50},{18,6,2},{18,6,2},{16,20,1},{28,0,50},{16,20,1},{36,6,72},{34,8,5},{4,6,5},{2,6,5},{36,6,72},{31,0,72},{2,6,5},
{0,36,74},{31,0,72},{0,36,74},{18,0,74},{18,0,74},{18,0,74},{18,0,74},{2,22,0},{2,22,0},{2,22,0},{2,4,4},{16,20,2},{16,20,2},{4,12,154},{4,24,85},{4,38,106},{34,38,95},{18,12,253},{18,8,90},{34,38,15},{18,6,122},{0,24,244},{16,22,93},{20,10,76},{20,8,9},{20,38,6},{20,22,24},{24,0,243},{18,8,81},{34,38,6},{0,22,80},{32,8,243},
{0,22,80},{4,40,80},{4,40,80},{4,40,80},{4,22,80},{34,24,59},{34,22,10},{34,22,10},{18,36,19},{32,22,53},{32,6,5},{20,8,5},{20,8,5},{20,8,5},{20,6,8},{22,2,50},{34,22,1},{34,22,1},{32,6,4},{42,2,50},{32,6,4},{24,2,72},{20,8,5},{20,38,2},{18,38,1},{24,2,72},{13,0,72},{18,38,1},{0,22,80},{13,0,72},{0,22,80},{4,0,80},
{4,0,80},{4,0,80},{4,0,80},{34,22,9},{34,22,9},{34,22,9},{18,36,10},{32,6,1},{32,6,1},{20,28,146},{20,10,77},{20,8,122},{20,8,77},{4,28,243},{4,40,78},{4,8,5},{4,38,110},{32,40,244},{2,38,79},{36,12,73},{36,40,2},{6,8,18},{36,8,18},{38,6,243},{34,40,73},{4,8,5},{2,38,75},{27,0,243},{2,38,75},{20,26,73},{20,26,73},{20,26,73},
{20,8,76},{4,26,50},{4,8,4},{4,8,4},{4,22,4},{2,8,50},{18,22,2},{36,40,1},{36,40,1},{36,40,1},{36,38,2},{8,2,50},{4,8,4},{4,8,4},{18,22,1},{30,2,50},{18,22,1},{16,11,72},{36,40,1},{6,8,2},{4,8,1},{16,11,72},{25,0,72},{4,8,1},{0,38,74},{25,0,72},{0,38,74},{20,0,72},{20,0,72},{20,0,72},{20,0,72},{4,24,0},
{4,24,0},{4,24,0},{4,6,4},{18,22,2},{18,22,2},{6,14,154},{6,26,85},{6,40,109},{6,24,97},{20,14,248},{20,10,78},{20,40,13},{20,8,110},{2,26,244},{34,8,91},{22,12,76},{22,10,9},{22,40,9},{22,24,29},{26,2,243},{20,10,74},{20,40,9},{2,24,80},{38,8,243},{2,24,80},{6,42,80},{6,42,80},{6,42,80},{6,24,81},{20,12,54},{20,40,9},{20,40,9},
{20,38,13},{34,24,53},{34,8,10},{22,10,5},{22,10,5},{22,10,5},{22,8,5},{24,4,50},{36,24,1},{36,24,1},{34,8,9},{11,0,50},{34,8,9},{26,4,72},{22,10,5},{22,40,5},{4,40,5},{26,4,72},{7,0,72},{4,40,5},{0,24,80},{7,0,72},{0,24,80},{6,0,80},{6,0,80},{6,0,80},{6,0,80},{20,10,4},{20,10,4},{20,10,4},{20,38,4},{34,8,1},
{34,8,1},{22,46,146},{22,12,83},{38,26,122},{22,10,79},{6,30,245},{6,42,74},{6,10,10},{6,40,111},{4,42,247},{20,40,79},{38,14,76},{38,12,5},{8,10,21},{38,10,17},{44,0,243},{6,42,73},{6,10,9},{4,40,74},{26,8,243},{4,40,74},{22,28,75},{22,28,75},{22,28,75},{22,10,75},{6,28,51},{6,10,6},{6,10,6},{6,24,6},{4,10,53},{20,24,6},{38,12,4},
{38,12,4},{38,12,4},{38,40,5},{38,10,50},{22,40,4},{22,40,4},{36,24,1},{23,0,50},{36,24,1},{42,6,72},{38,12,1},{8,10,5},{6,10,5},{42,6,72},{19,0,72},{6,10,5},{0,40,74},{19,0,72},{0,40,74},{22,0,74},{22,0,74},{22,0,74},{22,0,74},{6,26,2},{6,26,2},{6,26,2},{6,8,5},{20,24,5},{20,24,5},{8,46,154},{8,28,85},{8,42,106},
{8,42,97},{38,46,247},{38,12,82},{38,42,10},{22,26,119},{4,28,247},{36,26,83},{24,14,75},{24,28,6},{24,42,6},{24,26,30},{30,0,243},{38,12,78},{38,42,6},{4,26,75},{33,0,243},{4,26,75},{8,44,80},{8,44,80},{8,44,80},{8,26,80},{38,28,56},{38,26,9},{38,26,9},{38,40,14},{20,42,54},{6,10,9},{24,12,1},{24,12,1},{24,12,1},{24,10,5},{28,2,50},
{38,26,5},{38,26,5},{6,10,5},{42,8,50},{6,10,5},{30,2,72},{8,28,5},{24,42,5},{22,42,1},{30,2,72},{35,2,72},{22,42,1},{0,26,74},{35,2,72},{0,26,74},{8,0,80},{8,0,80},{8,0,80},{8,0,80},{38,26,5},{38,26,5},{38,26,5},{38,40,5},{6,40,4},{6,40,4},{24,15,146},{24,14,83},{24,28,126},{24,12,79},{8,31,243},{8,44,78},{8,12,9},
{8,42,110},{36,44,245},{22,42,79},{40,46,75},{40,14,6},{10,12,21},{40,12,14},{47,0,243},{8,44,78},{8,12,9},{6,42,74},{27,6,243},{6,42,74},{24,30,75},{24,30,75},{24,30,75},{24,12,75},{8,14,52},{8,12,5},{8,12,5},{8,26,5},{6,12,53},{22,26,6},{40,14,2},{40,14,2},{40,14,2},{40,42,1},{16,3,50},{24,42,4},{24,42,4},{38,26,1},{17,0,50},
{38,26,1},{47,2,72},{40,14,5},{10,12,5},{8,12,5},{47,2,72},{17,2,72},{8,12,5},{0,42,74},{17,2,72},{0,42,74},{24,0,74},{24,0,74},{24,0,74},{24,0,74},{8,28,0},{8,28,0},{8,28,0},{8,10,1},{22,26,5},{22,26,5},{10,15,152},{10,30,85},{10,44,106},{40,44,95},{40,31,252},{24,14,90},{40,44,15},{24,12,122},{6,30,247},{38,28,89},{26,47,73},
{26,30,6},{26,44,6},{26,28,21},{34,3,243},{8,30,76},{40,44,6},{6,28,80},{35,4,243},{6,28,80},{10,46,80},{10,46,80},{10,46,80},{10,28,81},{40,30,59},{40,28,10},{40,28,10},{24,42,18},{22,44,51},{8,12,9},{26,14,1},{26,14,1},{26,14,1},{26,12,5},{30,4,50},{40,28,1},{40,28,1},{8,12,5},{15,8,50},{8,12,5},{15,2,72},{10,30,5},{26,44,5},
{24,44,1},{15,2,72},{33,4,72},{24,44,1},{0,28,80},{33,4,72},{0,28,80},{10,0,80},{10,0,80},{10,0,80},{10,0,80},{40,28,9},{40,28,9},{40,28,9},{24,42,9},{8,12,5},{8,12,5},{26,13,152},{26,47,89},{42,30,122},{26,14,89},{10,29,245},{10,46,74},{10,30,15},{10,44,106},{8,46,247},{24,44,81},{42,45,76},{42,47,9},{12,30,18},{42,14,17},{4,1,243},
{10,46,73},{26,14,10},{8,44,80},{1,4,243},{8,44,80},{26,15,80},{26,15,80},{26,15,80},{26,14,80},{10,31,51},{10,30,6},{10,30,6},{10,28,6},{24,14,54},{24,28,6},{42,47,5},{42,47,5},{42,47,5},{42,44,5},{45,0,50},{26,14,1},{26,14,1},{40,28,1},{27,8,50},{40,28,1},{46,10,72},{12,46,5},{12,30,9},{26,14,9},{46,10,72},{23,8,72},{26,14,9},
{0,44,80},{23,8,72},{0,44,80},{26,0,80},{26,0,80},{26,0,80},{26,0,80},{10,30,2},{10,30,2},{10,30,2},{10,12,5},{24,44,1},{24,44,1},{12,13,148},{12,31,79},{12,46,110},{12,46,83},{42,29,248},{42,47,85},{42,46,5},{26,30,126},{8,31,247},{40,30,83},{28,29,75},{28,47,5},{28,46,5},{28,46,26},{27,0,243},{26,31,80},{42,46,1},{8,30,75},{37,8,243},
{8,30,75},{12,15,74},{12,15,74},{12,15,74},{12,30,74},{42,15,56},{42,46,5},{42,46,5},{26,14,21},{24,46,56},{10,14,2},{28,47,1},{28,47,1},{28,47,1},{28,14,5},{29,2,50},{42,46,1},{42,46,1},{10,14,1},{35,6,50},{10,14,1},{38,3,72},{28,47,4},{44,46,1},{26,46,1},{38,3,72},{35,8,72},{26,46,1},{0,30,74},{35,8,72},{0,30,74},{12,0,74},
{12,0,74},{12,0,74},{12,0,74},{42,46,4},{42,46,4},{42,46,4},{42,44,5},{10,14,1},{10,14,1},{28,11,146},{28,45,84},{44,47,121},{28,47,79},{12,27,244},{12,15,79},{12,47,6},{12,46,106},{40,15,244},{26,46,81},{44,43,76},{44,15,6},{14,47,11},{44,47,14},{28,13,243},{42,15,76},{12,47,5},{10,46,80},{27,12,243},{10,46,80},{28,13,75},{28,13,75},{28,13,75},
{28,47,78},{12,29,51},{12,47,5},{12,47,5},{12,30,6},{10,47,51},{26,30,6},{44,15,5},{44,15,5},{44,15,5},{44,46,5},{43,2,50},{12,47,4},{12,47,4},{42,30,1},{21,8,50},{42,30,1},{41,2,72},{44,15,2},{14,47,2},{12,47,1},{41,2,72},{17,8,72},{12,47,1},{0,46,80},{17,8,72},{0,46,80},{28,0,74},{28,0,74},{28,0,74},{28,0,74},{12,31,1},
{12,31,1},{12,31,1},{12,14,5},{26,46,1},{26,46,1},{14,11,148},{14,29,79},{14,15,114},{14,31,90},{44,27,248},{28,45,84},{28,15,18},{28,47,115},{10,29,247},{42,47,91},{30,27,75},{30,45,5},{30,15,6},{30,31,21},{40,3,243},{28,45,75},{28,15,9},{10,31,80},{16,9,243},{10,31,80},{14,13,74},{14,13,74},{14,13,74},{14,31,74},{44,13,56},{44,31,6},{44,31,6},
{44,46,21},{42,31,52},{42,47,10},{30,45,1},{30,45,1},{30,45,1},{30,47,2},{38,1,50},{44,31,2},{44,31,2},{42,47,9},{43,28,50},{42,47,9},{29,12,72},{30,45,4},{30,15,5},{12,15,5},{29,12,72},{26,13,72},{12,15,5},{0,31,80},{26,13,72},{0,31,80},{14,0,74},{14,0,74},{14,0,74},{14,0,74},{44,15,5},{44,15,5},{44,15,5},{44,46,5},{42,47,1},
{42,47,1},{30,9,154},{46,43,91},{46,29,115},{30,45,85},{14,9,247},{14,13,77},{14,29,18},{14,15,114},{12,13,247},{28,15,79},{46,25,83},{46,43,10},{47,45,21},{46,45,17},{10,1,243},{14,13,73},{30,45,6},{12,15,74},{4,9,243},{12,15,74},{30,11,80},{30,11,80},{30,11,80},{30,45,81},{14,11,56},{14,29,9},{14,29,9},{14,31,6},{12,45,56},{44,31,5},{46,43,9},
{46,43,9},{46,43,9},{46,15,10},{39,0,50},{30,45,2},{30,45,2},{44,31,1},{9,28,50},{44,31,1},{9,6,72},{46,43,1},{47,45,5},{14,45,5},{9,6,72},{32,7,72},{14,45,5},{0,15,74},{32,7,72},{0,15,74},{30,0,80},{30,0,80},{30,0,80},{30,0,80},{14,13,5},{14,13,5},{14,13,5},{14,31,5},{44,31,4},{44,31,4},{47,9,154},{47,27,81},{47,13,106},
{47,13,97},{46,9,244},{46,43,79},{46,13,6},{46,45,121},{12,27,248},{44,29,84},{31,25,75},{31,27,6},{31,13,6},{31,29,30},{21,0,243},{46,43,78},{46,13,5},{12,29,75},{18,7,243},{12,29,75},{47,11,80},{47,11,80},{47,11,80},{47,29,80},{46,11,51},{46,13,5},{46,13,5},{46,15,11},{28,13,54},{14,45,6},{31,43,1},{31,43,1},{31,43,1},{31,29,5},{23,2,50},
{46,13,4},{46,13,4},{14,45,5},{20,9,50},{14,45,5},{21,2,72},{47,27,1},{15,13,5},{30,13,1},{21,2,72},{24,9,72},{30,13,1},{0,29,74},{24,9,72},{0,29,74},{47,0,80},{47,0,80},{47,0,80},{47,0,80},{46,13,1},{46,13,1},{46,13,1},{46,15,2},{14,45,2},{14,45,2},{31,7,146},{31,41,83},{31,27,126},{31,43,79},{47,23,243},{47,11,78},{47,43,5},
{47,13,110},{14,11,247},{30,13,79},{15,9,75},{15,11,2},{15,27,21},{15,43,14},{14,3,243},{47,11,78},{47,43,5},{14,13,74},{32,5,243},{14,13,74},{31,9,75},{31,9,75},{31,9,75},{31,43,75},{47,25,52},{47,43,1},{47,43,1},{47,29,5},{14,43,56},{46,29,5},{15,11,1},{15,11,1},{15,11,1},{15,13,1},{37,2,50},{47,43,1},{47,43,1},{46,29,1},{34,7,50},
{46,29,1},{35,2,72},{15,11,1},{45,43,5},{47,43,4},{35,2,72},{34,5,72},{47,43,4},{0,13,74},{34,5,72},{0,13,74},{31,0,74},{31,0,74},{31,0,74},{31,0,74},{47,27,1},{47,27,1},{47,27,1},{47,45,1},{46,29,4},{46,29,4},{45,7,152},{45,25,81},{45,11,106},{45,11,97},{15,23,253},{31,25,90},{31,11,15},{31,43,122},{14,25,247},{46,27,89},{29,39,73},
{29,25,6},{29,11,6},{29,27,26},{46,3,243},{47,25,76},{31,11,6},{14,27,80},{16,3,243},{14,27,80},{45,9,80},{45,9,80},{45,9,80},{45,27,81},{15,25,59},{15,27,10},{15,27,10},{31,13,18},{30,11,54},{46,43,9},{29,41,1},{29,41,1},{29,41,1},{29,43,2},{44,1,50},{15,27,1},{15,27,1},{46,43,5},{26,9,50},{46,43,5},{46,1,72},{45,25,1},{13,11,5},
{31,11,2},{46,1,72},{26,7,72},{31,11,2},{0,27,80},{26,7,72},{0,27,80},{45,0,80},{45,0,80},{45,0,80},{45,0,80},{15,27,9},{15,27,9},{15,27,9},{31,13,9},{47,13,5},{47,13,5},{29,5,154},{29,39,89},{13,25,122},{29,41,89},{45,21,245},{45,9,74},{45,41,15},{45,11,106},{47,9,247},{31,11,85},{13,37,76},{13,9,9},{43,25,18},{13,41,17},{1,0,243},
{45,9,73},{29,41,10},{47,11,80},{0,1,243},{47,11,80},{29,7,80},{29,7,80},{29,7,80},{29,41,80},{45,23,51},{45,41,6},{45,41,6},{45,27,6},{31,41,54},{31,27,6},{13,9,5},{13,9,5},{13,9,5},{13,11,5},{31,5,50},{29,41,1},{29,41,1},{15,27,1},{14,9,50},{15,27,1},{27,9,72},{13,9,5},{43,25,9},{29,41,9},{27,9,72},{14,7,72},{29,41,9},
{0,11,80},{14,7,72},{0,11,80},{29,0,80},{29,0,80},{29,0,80},{29,0,80},{45,25,1},{45,25,1},{45,25,1},{45,27,5},{31,11,5},{31,11,5},{43,5,148},{43,23,79},{43,9,110},{43,9,83},{13,21,248},{13,39,85},{13,9,9},{29,25,126},{47,23,248},{15,25,83},{27,21,75},{27,23,6},{27,9,5},{27,9,26},{15,1,243},{29,23,74},{13,9,5},{31,25,75},{9,11,243},
{31,25,75},{43,7,74},{43,7,74},{43,7,74},{43,25,74},{13,7,56},{13,9,9},{13,9,9},{13,11,21},{15,9,56},{15,41,6},{27,39,1},{27,39,1},{27,39,1},{27,25,5},{17,2,50},{43,25,4},{43,25,4},{15,41,2},{16,1,50},{15,41,2},{17,6,72},{27,23,5},{11,9,1},{29,9,0},{17,6,72},{20,1,72},{29,9,0},{0,25,74},{20,1,72},{0,25,74},{43,0,74},
{43,0,74},{43,0,74},{43,0,74},{13,9,5},{13,9,5},{13,9,5},{13,11,5},{15,41,5},{15,41,5},{27,3,146},{27,37,83},{27,23,119},{27,39,79},{43,19,244},{43,7,79},{43,39,10},{43,9,106},{15,7,245},{29,9,85},{11,35,76},{11,7,9},{41,39,14},{11,39,17},{27,5,243},{13,7,78},{27,39,9},{45,9,80},{6,1,243},{45,9,80},{27,5,75},{27,5,75},{27,5,75},
{27,39,75},{43,21,51},{43,39,6},{43,39,6},{43,25,6},{45,39,53},{29,25,6},{11,7,5},{11,7,5},{11,7,5},{11,9,5},{29,3,50},{27,39,5},{27,39,5},{13,25,1},{43,9,50},{13,25,1},{27,3,72},{41,7,4},{41,39,5},{27,39,5},{27,3,72},{39,9,72},{27,39,5},{0,9,80},{39,9,72},{0,9,80},{27,0,74},{27,0,74},{27,0,74},{27,0,74},{43,23,1},
{43,23,1},{43,23,1},{43,25,5},{29,9,5},{29,9,5},{41,3,148},{41,21,79},{41,7,111},{41,7,91},{11,19,248},{11,37,85},{11,7,10},{27,39,122},{45,21,248},{13,23,83},{25,19,75},{25,21,6},{25,7,6},{25,23,30},{17,10,243},{27,21,81},{11,7,6},{29,23,75},{24,1,243},{29,23,75},{41,5,74},{41,5,74},{41,5,74},{41,23,74},{11,5,56},{11,7,9},{11,7,9},
{11,9,21},{29,7,54},{13,39,5},{25,37,1},{25,37,1},{25,37,1},{25,39,2},{39,11,50},{41,23,4},{41,23,4},{13,39,4},{22,1,50},{13,39,4},{39,7,72},{25,21,5},{9,7,5},{27,7,2},{39,7,72},{26,1,72},{27,7,2},{0,23,74},{26,1,72},{0,23,74},{41,0,74},{41,0,74},{41,0,74},{41,0,74},{11,7,5},{11,7,5},{11,7,5},{11,9,5},{13,39,1},
{13,39,1},{25,1,154},{9,35,91},{9,21,110},{25,37,89},{41,1,247},{41,5,77},{41,21,13},{41,7,109},{43,5,247},{27,7,85},{39,19,81},{9,35,10},{39,21,13},{9,37,22},{41,1,243},{41,5,73},{41,21,9},{43,7,80},{47,3,243},{43,7,80},{25,3,80},{25,3,80},{25,3,80},{25,37,80},{41,3,56},{41,21,9},{41,21,9},{41,23,9},{27,37,54},{11,23,9},{39,21,9},
{39,21,9},{39,21,9},{9,7,10},{25,5,50},{25,37,1},{25,37,1},{11,23,5},{10,1,50},{11,23,5},{23,5,72},{9,35,1},{39,21,4},{11,21,4},{23,5,72},{14,1,72},{11,21,4},{0,7,80},{14,1,72},{0,7,80},{25,0,80},{25,0,80},{25,0,80},{25,0,80},{41,5,5},{41,5,5},{41,5,5},{41,23,5},{27,7,5},{27,7,5},{39,1,148},{39,3,79},{39,5,110},
{39,5,83},{9,1,244},{9,35,79},{9,5,5},{9,21,122},{43,19,248},{11,21,77},{23,17,72},{23,19,2},{23,5,4},{23,5,25},{9,1,243},{25,19,74},{9,5,4},{27,21,73},{46,1,243},{27,21,73},{39,3,75},{39,3,75},{39,3,75},{39,21,75},{9,3,51},{9,5,5},{9,5,5},{9,7,18},{11,5,53},{41,37,2},{23,19,1},{23,19,1},{23,19,1},{23,21,1},{9,3,50},
{9,5,4},{9,5,4},{41,37,1},{31,3,50},{41,37,1},{23,17,72},{23,19,2},{7,5,4},{25,5,0},{23,17,72},{45,17,72},{25,5,0},{0,21,72},{45,17,72},{0,21,72},{39,0,74},{39,0,74},{39,0,74},{39,0,74},{9,5,1},{9,5,1},{9,5,1},{9,7,2},{41,37,1},{41,37,1},{7,1,184},{23,17,93},{7,19,122},{23,35,89},{23,1,260},{39,3,74},{39,35,15},
{39,5,106},{41,3,247},{25,5,85},{37,17,81},{7,33,5},{37,19,19},{7,35,17},{19,9,243},{39,3,73},{23,35,10},{41,5,80},{45,1,243},{41,5,80},{23,1,80},{23,1,80},{23,1,80},{23,35,80},{39,17,51},{39,35,6},{39,35,6},{39,21,6},{25,35,54},{9,21,9},{7,33,4},{7,33,4},{7,33,4},{7,5,5},{23,3,50},{23,35,1},{23,35,1},{9,21,5},{43,3,50},
{9,21,5},{21,3,72},{7,33,1},{37,19,10},{23,35,9},{21,3,72},{43,1,72},{23,35,9},{0,5,80},{43,1,72},{0,5,80},{23,0,80},{23,0,80},{23,0,80},{23,0,80},{39,19,1},{39,19,1},{39,19,1},{39,21,2},{25,5,5},{25,5,5},{21,1,234},{37,1,79},{37,3,110},{37,3,90},{37,1,300},{7,33,82},{7,3,9},{23,19,139},{41,17,248},{9,19,83},{21,1,90},
{21,17,2},{21,3,4},{21,3,25},{21,17,243},{23,17,74},{7,3,5},{25,19,75},{17,21,243},{25,19,75},{37,1,75},{37,1,75},{37,1,75},{37,19,74},{7,1,56},{7,19,6},{7,19,6},{7,5,21},{9,3,53},{9,35,6},{21,17,1},{21,17,1},{21,17,1},{21,19,4},{3,25,50},{7,19,2},{7,19,2},{9,35,2},{29,1,50},{9,35,2},{17,39,72},{21,17,2},{5,3,4},
{23,3,0},{17,39,72},{39,17,72},{23,3,0},{0,19,74},{39,17,72},{0,19,74},{37,0,74},{37,0,74},{37,0,74},{37,0,74},{7,3,5},{7,3,5},{7,3,5},{7,5,5},{9,35,5},{9,35,5},{5,1,290},{5,1,103},{5,17,110},{21,33,106},{5,1,345},{37,1,78},{37,17,13},{37,3,119},{39,1,248},{23,3,91},{35,1,126},{35,1,6},{35,17,8},{5,33,29},{35,1,243},
{37,1,74},{37,17,9},{9,33,81},{1,35,243},{9,33,81},{5,1,94},{5,1,94},{5,1,94},{21,33,90},{21,1,61},{37,17,9},{37,17,9},{37,19,9},{23,33,53},{7,19,8},{35,17,4},{35,17,4},{35,17,4},{35,3,4},{19,5,50},{21,33,0},{21,33,0},{7,19,4},{37,3,50},{7,19,4},{17,5,72},{35,1,2},{35,17,4},{7,17,4},{17,5,72},{5,17,72},{7,17,4},
{0,33,80},{5,17,72},{0,33,80},{5,0,90},{5,0,90},{5,0,90},{5,0,90},{37,1,4},{37,1,4},{37,1,4},{37,19,5},{23,3,1},{23,3,1},{19,1,349},{35,1,185},{35,1,121},{35,1,81},{19,1,398},{5,1,102},{5,1,2},{21,17,122},{37,1,270},{7,17,66},{3,1,126},{19,1,45},{19,1,9},{19,1,25},{33,33,221},{35,1,82},{5,1,1},{23,17,61},{33,33,221},
{23,17,61},{35,1,121},{35,1,121},{35,1,121},{35,17,73},{35,1,94},{5,1,2},{5,1,2},{5,33,17},{7,1,53},{37,33,1},{19,1,9},{19,1,9},{19,1,9},{19,17,1},{3,3,50},{5,1,1},{5,1,1},{37,33,0},{3,3,50},{37,33,0},{1,3,61},{33,1,25},{3,1,1},{5,1,1},{1,3,61},{3,1,61},{5,1,1},{0,17,61},{3,1,61},{0,17,61},{35,0,72},
{35,0,72},{35,0,72},{35,0,72},{5,1,1},{5,1,1},{5,1,1},{5,3,1},{37,33,1},{37,33,1},{3,1,239},{3,1,164},{3,1,139},{19,1,89},{3,1,239},{19,1,62},{19,1,26},{35,17,34},{35,1,163},{21,1,10},{33,1,69},{33,1,29},{33,1,20},{33,1,4},{1,3,93},{3,1,38},{19,1,10},{21,1,10},{3,1,93},{21,1,10},{3,1,139},{3,1,139},{3,1,139},
{19,1,89},{19,1,138},{19,1,26},{19,1,26},{35,17,9},{5,1,74},{5,17,8},{33,1,20},{33,1,20},{33,1,20},{33,1,4},{17,3,50},{19,1,10},{19,1,10},{5,17,4},{35,1,50},{5,17,4},{1,17,5},{17,1,1},{17,1,0},{33,1,0},{1,17,5},{17,1,5},{33,1,0},{0,1,9},{17,1,5},{0,1,9},{19,0,80},{19,0,80},{19,0,80},{19,0,80},{19,1,17},
{19,1,17},{19,1,17},{35,17,5},{21,1,1},{21,1,1},{17,1,162},{33,1,130},{33,1,121},{33,1,81},{33,1,138},{33,1,58},{33,1,49},{3,1,1},{19,1,82},{19,1,10},{17,1,18},{17,1,10},{17,1,9},{17,1,1},{1,17,18},{17,1,6},{17,1,5},{3,1,0},{17,1,18},{3,1,0},{33,1,121},{33,1,121},{33,1,121},{33,1,81},{33,1,89},{33,1,49},{33,1,49},
{3,1,1},{19,1,46},{19,1,10},{17,1,9},{17,1,9},{17,1,9},{17,1,1},{17,1,13},{17,1,5},{17,1,5},{3,1,0},{33,1,13},{3,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{33,0,72},{33,0,72},{33,0,72},{33,0,72},{3,1,26},{3,1,26},{3,1,26},{3,1,1},{19,1,10},
{19,1,10},{0,38,200},{0,20,25},{0,34,5},{0,34,85},{0,20,442},{0,34,266},{0,18,125},{0,2,318},{0,18,482},{0,2,343},{0,38,200},{0,20,25},{0,34,5},{0,34,85},{16,4,441},{0,34,266},{0,18,125},{0,2,318},{4,16,441},{0,2,318},{0,4,1},{0,4,1},{0,4,1},{0,2,1},{0,2,41},{0,32,13},{0,32,13},{0,16,25},{0,16,46},{0,16,26},{0,4,1},
{0,4,1},{0,4,1},{0,2,1},{0,2,41},{0,32,13},{0,32,13},{0,16,25},{2,0,41},{0,16,25},{18,2,200},{0,20,25},{0,34,5},{0,34,85},{18,2,200},{38,0,200},{0,34,85},{0,18,208},{38,0,200},{0,18,208},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,10,202},{0,22,1},{0,20,26},
{0,4,41},{0,22,689},{0,20,352},{0,4,141},{0,18,468},{0,34,750},{0,18,504},{0,10,202},{0,22,1},{0,20,26},{0,4,41},{34,0,686},{0,20,352},{0,4,141},{0,18,468},{0,34,686},{0,18,468},{0,22,0},{0,22,0},{0,22,0},{0,34,1},{0,34,145},{0,18,45},{0,18,45},{0,32,85},{0,32,158},{0,32,94},{0,22,0},{0,22,0},{0,22,0},{0,34,1},{16,2,145},
{0,18,45},{0,18,45},{0,32,85},{34,0,145},{0,32,85},{20,16,200},{0,22,1},{16,20,5},{0,4,41},{20,16,200},{16,20,200},{0,4,41},{0,34,200},{16,20,200},{0,34,200},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{16,12,229},{16,8,30},{16,6,74},{16,36,58},{0,10,723},{0,6,282},{0,36,69},
{0,4,414},{0,20,868},{0,34,513},{16,12,204},{16,8,5},{32,6,29},{16,36,33},{4,2,723},{0,6,282},{0,36,69},{0,4,414},{10,0,723},{0,4,414},{16,8,30},{16,8,30},{16,8,30},{16,4,33},{0,6,164},{0,4,25},{0,4,25},{0,18,65},{0,18,206},{0,18,101},{16,8,5},{16,8,5},{16,8,5},{16,4,8},{2,2,162},{0,4,25},{0,4,25},{0,18,65},{2,2,162},
{0,18,65},{18,8,200},{16,8,1},{2,36,5},{0,36,20},{18,8,200},{44,0,200},{0,36,20},{0,4,218},{44,0,200},{0,4,218},{16,0,29},{16,0,29},{16,0,29},{16,0,29},{0,18,1},{0,18,1},{0,18,1},{0,32,0},{0,32,9},{0,32,9},{16,30,327},{32,40,139},{32,22,202},{16,6,151},{0,28,724},{0,8,236},{0,22,14},{0,20,350},{0,6,956},{0,20,494},{2,12,203},
{2,24,2},{2,22,27},{2,6,42},{36,2,723},{0,8,236},{0,22,14},{0,20,350},{20,4,723},{0,20,350},{16,42,125},{16,42,125},{16,42,125},{16,6,126},{0,24,164},{0,6,1},{0,6,1},{0,4,37},{0,4,280},{0,34,109},{2,24,1},{2,24,1},{2,24,1},{2,36,2},{18,4,162},{0,6,1},{0,6,1},{0,4,37},{36,2,162},{0,4,37},{8,0,200},{2,24,1},{18,22,5},
{0,22,10},{8,0,200},{47,0,200},{0,22,10},{0,36,200},{47,0,200},{0,36,200},{16,0,125},{16,0,125},{16,0,125},{16,0,125},{0,6,0},{0,6,0},{0,6,0},{0,18,1},{0,18,37},{0,18,37},{2,46,410},{2,26,218},{2,8,317},{2,38,226},{16,30,725},{16,40,217},{16,38,15},{0,6,312},{0,38,988},{0,6,417},{34,28,209},{18,26,13},{34,8,27},{18,38,34},{8,0,723},
{0,40,203},{16,38,14},{0,6,296},{47,0,723},{0,6,296},{2,12,208},{2,12,208},{2,12,208},{2,22,209},{16,42,163},{16,38,6},{16,38,6},{16,20,26},{0,6,285},{0,20,81},{18,26,9},{18,26,9},{18,26,9},{34,6,10},{2,24,162},{32,22,5},{32,22,5},{0,20,17},{28,0,162},{0,20,17},{24,2,200},{18,26,4},{4,8,9},{0,8,10},{24,2,200},{13,0,200},{0,8,10},
{0,22,208},{13,0,200},{0,22,208},{2,0,208},{2,0,208},{2,0,208},{2,0,208},{16,8,1},{16,8,1},{16,8,1},{16,4,2},{0,4,45},{0,4,45},{18,15,402},{18,12,222},{34,24,331},{18,24,218},{32,47,725},{32,26,212},{32,24,5},{32,38,324},{0,24,884},{0,38,300},{4,30,200},{4,26,5},{4,24,36},{4,8,41},{40,0,723},{16,26,203},{32,24,4},{0,38,251},{46,2,723},
{0,38,251},{18,14,203},{18,14,203},{18,14,203},{18,8,202},{32,28,163},{32,24,5},{32,24,5},{32,6,21},{0,38,219},{0,6,13},{4,26,1},{4,26,1},{4,26,1},{4,38,0},{22,2,162},{2,8,1},{2,8,1},{0,6,4},{42,2,162},{0,6,4},{16,11,200},{34,42,4},{20,24,4},{32,24,0},{16,11,200},{25,0,200},{32,24,0},{0,38,202},{25,0,200},{0,38,202},{18,0,202},
{18,0,202},{18,0,202},{18,0,202},{32,40,1},{32,40,1},{32,40,1},{32,36,1},{0,6,9},{0,6,9},{4,15,408},{4,28,221},{4,10,317},{4,40,221},{18,31,725},{18,42,217},{18,40,6},{2,8,315},{0,26,788},{0,8,228},{20,47,204},{20,12,6},{36,10,27},{20,40,33},{10,2,723},{32,12,203},{18,40,5},{0,8,227},{45,2,723},{0,8,227},{4,14,208},{4,14,208},{4,14,208},
{4,24,209},{18,44,163},{18,40,2},{18,40,2},{18,22,26},{0,24,179},{0,38,3},{20,12,5},{20,12,5},{20,12,5},{20,24,5},{8,2,162},{18,40,1},{18,40,1},{0,38,2},{30,2,162},{0,38,2},{26,4,200},{20,12,2},{6,40,5},{18,40,4},{26,4,200},{7,0,200},{18,40,4},{0,24,208},{7,0,200},{0,24,208},{4,0,208},{4,0,208},{4,0,208},{4,0,208},{18,10,1},
{18,10,1},{18,10,1},{18,6,2},{0,38,2},{0,38,2},{20,13,402},{20,14,220},{36,26,331},{20,26,216},{34,45,725},{34,12,213},{34,26,5},{34,24,337},{0,12,740},{16,40,216},{6,31,202},{6,28,1},{6,26,36},{6,10,41},{42,2,723},{18,28,200},{34,26,4},{0,40,209},{44,6,723},{0,40,209},{20,46,200},{20,46,200},{20,46,200},{20,10,201},{34,30,163},{34,26,5},{34,26,5},
{34,8,18},{0,26,164},{32,8,2},{6,28,0},{6,28,0},{6,28,0},{6,40,1},{24,4,162},{4,10,1},{4,10,1},{2,8,2},{11,0,162},{2,8,2},{14,0,200},{6,28,1},{22,26,4},{34,26,0},{14,0,200},{8,10,200},{34,26,0},{0,40,200},{8,10,200},{0,40,200},{20,0,200},{20,0,200},{20,0,200},{20,0,200},{34,42,1},{34,42,1},{34,42,1},{34,38,1},{32,8,1},
{32,8,1},{6,13,404},{6,30,219},{6,12,327},{6,42,215},{20,29,725},{20,44,217},{20,12,9},{4,10,321},{0,14,723},{32,26,222},{38,15,206},{38,14,8},{38,12,24},{22,42,34},{14,0,723},{4,14,203},{20,12,8},{0,26,203},{8,10,723},{0,26,203},{6,47,202},{6,47,202},{6,47,202},{6,26,206},{20,46,163},{20,12,5},{20,12,5},{20,24,30},{32,42,164},{2,40,6},{38,44,4},
{38,44,4},{38,44,4},{38,10,5},{38,10,162},{20,12,4},{20,12,4},{2,40,2},{23,0,162},{2,40,2},{30,2,200},{38,14,4},{8,12,4},{4,12,4},{30,2,200},{35,2,200},{4,12,4},{0,26,202},{35,2,200},{0,26,202},{6,0,202},{6,0,202},{6,0,202},{6,0,202},{20,12,1},{20,12,1},{20,12,1},{20,8,2},{2,10,4},{2,10,4},{22,11,402},{22,47,227},{38,44,321},
{22,28,218},{36,27,727},{36,30,215},{36,28,13},{36,42,327},{32,30,725},{18,42,212},{8,29,200},{8,30,4},{24,28,29},{8,12,45},{0,1,723},{20,30,203},{6,28,5},{32,42,202},{1,0,723},{32,42,202},{22,15,202},{22,15,202},{22,15,202},{22,12,202},{36,15,168},{36,28,13},{36,28,13},{36,10,17},{2,28,163},{34,10,9},{8,30,0},{8,30,0},{8,30,0},{8,42,0},{28,2,162},
{6,12,5},{6,12,5},{4,10,4},{42,8,162},{4,10,4},{47,2,200},{8,30,4},{24,28,4},{36,28,0},{47,2,200},{17,2,200},{36,28,0},{0,42,202},{17,2,200},{0,42,202},{22,0,202},{22,0,202},{22,0,202},{22,0,202},{36,14,5},{36,14,5},{36,14,5},{36,40,4},{34,26,2},{34,26,2},{8,11,408},{8,31,221},{8,14,312},{8,44,226},{22,27,724},{22,46,217},{22,44,12},
{6,12,321},{2,47,728},{34,28,234},{40,29,209},{24,47,10},{40,14,22},{24,44,34},{18,3,723},{36,47,203},{22,44,11},{32,28,208},{45,8,723},{32,28,208},{8,15,209},{8,15,209},{8,15,209},{8,28,209},{22,15,165},{22,44,3},{22,44,3},{22,26,30},{34,44,164},{4,42,6},{40,30,10},{40,30,10},{40,30,10},{40,12,10},{16,3,162},{22,44,2},{22,44,2},{4,42,2},{17,0,162},
{4,42,2},{15,2,200},{24,47,1},{10,14,4},{6,14,5},{15,2,200},{33,4,200},{6,14,5},{0,28,208},{33,4,200},{0,28,208},{8,0,208},{8,0,208},{8,0,208},{8,0,208},{22,14,1},{22,14,1},{22,14,1},{22,10,2},{4,12,4},{4,12,4},{24,9,402},{24,45,227},{40,46,324},{24,30,218},{38,25,727},{38,47,213},{38,30,13},{38,28,325},{34,31,727},{20,44,218},{10,27,202},
{10,31,1},{26,30,29},{10,14,45},{4,3,723},{22,31,200},{8,30,8},{34,44,202},{3,4,723},{34,44,202},{24,13,202},{24,13,202},{24,13,202},{24,14,202},{38,13,168},{38,30,13},{38,30,13},{38,12,17},{4,30,163},{36,12,9},{10,31,0},{10,31,0},{10,31,0},{10,44,0},{30,4,162},{8,14,4},{8,14,4},{6,12,4},{15,8,162},{6,12,4},{43,0,200},{10,31,1},{26,30,4},
{38,30,0},{43,0,200},{19,6,200},{38,30,0},{0,44,202},{19,6,200},{0,44,202},{24,0,202},{24,0,202},{24,0,202},{24,0,202},{38,46,5},{38,46,5},{38,46,5},{38,42,4},{20,28,4},{20,28,4},{10,9,404},{10,29,219},{26,47,330},{10,46,222},{24,9,723},{24,15,209},{24,47,18},{8,14,324},{4,45,723},{36,30,222},{42,11,206},{42,15,8},{42,47,22},{26,46,41},{43,0,723},
{8,15,202},{24,47,18},{4,30,202},{19,6,723},{4,30,202},{10,43,202},{10,43,202},{10,43,202},{10,30,206},{24,13,162},{24,46,8},{24,46,8},{24,44,25},{20,47,168},{22,44,4},{42,15,4},{42,15,4},{42,15,4},{42,14,8},{45,0,162},{40,46,4},{40,46,4},{22,44,0},{27,8,162},{22,44,0},{38,3,200},{42,15,4},{12,47,9},{8,47,10},{38,3,200},{35,8,200},{8,47,10},
{0,30,202},{35,8,200},{0,30,202},{10,0,202},{10,0,202},{10,0,202},{10,0,202},{24,47,0},{24,47,0},{24,47,0},{24,12,4},{22,44,4},{22,44,4},{26,7,408},{42,13,236},{42,15,332},{26,31,224},{40,23,727},{40,45,213},{40,31,8},{40,46,312},{36,29,725},{22,46,218},{12,25,201},{12,13,6},{28,31,29},{12,47,42},{11,0,723},{24,29,203},{40,31,4},{6,46,209},{5,8,723},
{6,46,209},{26,41,208},{26,41,208},{26,41,208},{26,47,208},{40,11,168},{40,31,8},{40,31,8},{40,14,22},{6,31,163},{38,30,10},{12,13,2},{12,13,2},{12,13,2},{12,46,2},{29,2,162},{10,47,2},{10,47,2},{22,30,9},{35,6,162},{22,30,9},{41,2,200},{42,13,4},{44,31,1},{40,31,0},{41,2,200},{17,8,200},{40,31,0},{0,46,208},{17,8,200},{0,46,208},{26,0,208},
{26,0,208},{26,0,208},{26,0,208},{40,15,4},{40,15,4},{40,15,4},{40,44,5},{38,30,1},{38,30,1},{12,7,404},{12,27,212},{12,45,332},{12,15,215},{26,23,724},{26,13,216},{26,15,8},{26,47,318},{6,43,723},{8,47,227},{44,9,206},{44,13,8},{44,45,22},{28,15,38},{24,3,723},{10,13,202},{26,15,8},{6,31,209},{3,24,723},{6,31,209},{12,41,202},{12,41,202},{12,41,202},
{12,31,203},{26,11,162},{26,15,4},{26,15,4},{26,46,25},{38,15,162},{24,46,4},{44,13,4},{44,13,4},{44,13,4},{44,47,5},{43,2,162},{26,15,4},{26,15,4},{24,46,0},{21,8,162},{24,46,0},{29,12,200},{28,43,2},{14,15,8},{26,15,4},{29,12,200},{26,13,200},{26,15,4},{0,31,208},{26,13,200},{0,31,208},{12,0,202},{12,0,202},{12,0,202},{12,0,202},{26,45,0},
{26,45,0},{26,45,0},{26,14,4},{24,46,4},{24,46,4},{28,5,402},{28,41,222},{44,13,332},{28,29,218},{42,21,727},{42,43,213},{42,29,8},{42,31,340},{38,27,724},{24,15,217},{14,23,201},{14,27,3},{30,29,29},{14,45,42},{9,2,723},{26,27,201},{42,29,4},{22,15,201},{2,9,723},{22,15,201},{28,9,202},{28,9,202},{28,9,202},{28,45,203},{42,9,168},{42,29,8},{42,29,8},
{42,47,21},{8,29,163},{40,47,2},{14,11,2},{14,11,2},{14,11,2},{14,15,2},{38,1,162},{12,45,2},{12,45,2},{10,47,2},{43,28,162},{10,47,2},{37,0,200},{14,27,2},{46,29,1},{42,29,0},{37,0,200},{36,9,200},{42,29,0},{0,15,200},{36,9,200},{0,15,200},{28,0,202},{28,0,202},{28,0,202},{28,0,202},{42,13,4},{42,13,4},{42,13,4},{42,46,5},{40,47,1},
{40,47,1},{14,35,400},{14,25,217},{30,43,340},{14,13,213},{28,5,723},{28,11,209},{28,43,8},{12,45,332},{8,41,724},{40,29,222},{46,7,201},{46,41,2},{46,43,21},{30,13,41},{37,0,723},{12,41,203},{28,43,8},{8,29,202},{36,9,723},{8,29,202},{14,23,201},{14,23,201},{14,23,201},{14,13,204},{28,9,162},{28,43,4},{28,43,4},{28,31,29},{40,13,164},{26,15,3},{46,11,2},
{46,11,2},{46,11,2},{46,29,5},{39,0,162},{44,13,2},{44,13,2},{10,15,2},{9,28,162},{10,15,2},{21,2,200},{46,41,1},{47,43,5},{12,43,4},{21,2,200},{24,9,200},{12,43,4},{0,29,202},{24,9,200},{0,29,202},{14,0,200},{14,0,200},{14,0,200},{14,0,200},{28,43,0},{28,43,0},{28,43,0},{28,47,1},{26,15,2},{26,15,2},{30,3,410},{46,9,227},{46,27,318},
{30,27,224},{14,5,728},{44,25,215},{14,27,8},{44,13,332},{40,25,725},{26,13,212},{47,21,200},{47,25,4},{47,27,25},{47,43,52},{5,0,723},{28,25,203},{14,27,4},{40,13,202},{0,5,723},{40,13,202},{30,7,209},{30,7,209},{30,7,209},{30,43,208},{14,39,166},{14,27,8},{14,27,8},{44,45,22},{10,27,163},{12,45,8},{47,25,0},{47,25,0},{47,25,0},{47,13,0},{23,2,162},
{14,27,4},{14,27,4},{12,45,4},{20,9,162},{12,45,4},{35,2,200},{47,25,4},{15,27,4},{44,27,0},{35,2,200},{34,5,200},{44,27,0},{0,13,202},{34,5,200},{0,13,202},{30,0,208},{30,0,208},{30,0,208},{30,0,208},{14,27,4},{14,27,4},{14,27,4},{14,15,8},{42,29,2},{42,29,2},{47,3,408},{47,23,218},{47,41,312},{47,11,226},{30,3,723},{30,9,209},{30,41,8},
{14,43,332},{10,39,723},{12,43,236},{15,21,209},{31,39,10},{15,41,22},{31,11,43},{30,3,723},{44,39,204},{30,41,8},{40,27,208},{42,9,723},{40,27,208},{47,7,209},{47,7,209},{47,7,209},{47,27,208},{30,7,162},{30,41,4},{30,41,4},{30,29,29},{42,11,164},{12,13,6},{31,23,9},{31,23,9},{31,23,9},{31,27,10},{37,2,162},{46,11,2},{46,11,2},{12,13,2},{34,7,162},
{12,13,2},{46,1,200},{31,39,1},{45,41,5},{14,41,4},{46,1,200},{26,7,200},{14,41,4},{0,27,208},{26,7,200},{0,27,208},{47,0,208},{47,0,208},{47,0,208},{47,0,208},{30,41,0},{30,41,0},{30,41,0},{30,45,1},{12,43,4},{12,43,4},{31,1,402},{31,37,222},{15,9,324},{31,25,218},{46,17,733},{46,23,215},{46,25,18},{46,27,330},{42,23,725},{28,11,219},{45,19,202},
{45,23,4},{45,25,25},{45,41,52},{3,2,723},{30,23,203},{47,25,8},{42,11,202},{2,3,723},{42,11,202},{31,5,202},{31,5,202},{31,5,202},{31,41,202},{46,5,173},{46,25,18},{46,25,18},{46,43,22},{12,25,163},{14,43,8},{45,23,0},{45,23,0},{45,23,0},{45,11,0},{44,1,162},{47,41,4},{47,41,4},{14,43,4},{26,9,162},{14,43,4},{31,1,200},{45,23,4},{13,25,4},
{46,25,0},{31,1,200},{32,1,200},{46,25,0},{0,11,202},{32,1,200},{0,11,202},{31,0,202},{31,0,202},{31,0,202},{31,0,202},{46,9,10},{46,9,10},{46,9,10},{46,13,9},{28,27,4},{28,27,4},{45,1,404},{45,21,218},{29,39,325},{45,9,222},{31,1,723},{31,7,209},{31,39,13},{47,41,324},{12,37,725},{44,25,227},{13,3,206},{13,37,9},{13,39,17},{29,9,48},{31,1,723},
{47,7,208},{31,39,13},{12,25,202},{32,1,723},{12,25,202},{45,35,202},{45,35,202},{45,35,202},{45,25,203},{31,5,162},{31,9,8},{31,9,8},{31,27,29},{14,9,166},{30,11,1},{13,7,4},{13,7,4},{13,7,4},{13,25,8},{31,5,162},{15,9,4},{15,9,4},{30,11,0},{14,9,162},{30,11,0},{17,6,200},{29,21,4},{43,39,4},{46,39,5},{17,6,200},{20,1,200},{46,39,5},
{0,25,202},{20,1,200},{0,25,202},{45,0,202},{45,0,202},{45,0,202},{45,0,202},{31,39,0},{31,39,0},{31,39,0},{31,27,4},{30,11,1},{30,11,1},{13,1,440},{29,35,234},{13,7,321},{29,23,218},{45,1,732},{15,21,215},{45,23,12},{15,9,312},{44,21,731},{30,9,221},{43,17,201},{43,5,6},{27,23,30},{43,39,46},{45,1,723},{31,21,203},{45,23,3},{14,9,209},{27,9,723},
{14,9,209},{29,33,208},{29,33,208},{29,33,208},{29,39,208},{15,3,168},{45,23,11},{45,23,11},{15,41,22},{14,23,168},{46,25,10},{43,5,2},{43,5,2},{43,5,2},{43,9,2},{17,2,162},{45,23,2},{45,23,2},{30,25,10},{16,1,162},{30,25,10},{27,3,200},{13,5,4},{11,23,2},{15,23,1},{27,3,200},{39,9,200},{15,23,1},{0,9,208},{39,9,200},{0,9,208},{29,0,208},
{29,0,208},{29,0,208},{29,0,208},{15,7,5},{15,7,5},{15,7,5},{15,11,4},{46,25,1},{46,25,1},{43,1,500},{43,19,212},{43,37,327},{43,7,220},{13,1,760},{29,5,209},{29,37,13},{45,39,321},{14,35,724},{46,23,227},{11,1,206},{11,35,9},{11,37,17},{27,7,43},{33,10,723},{15,35,204},{29,37,13},{14,23,202},{38,1,723},{14,23,202},{43,33,202},{43,33,202},{43,33,202},
{43,23,203},{29,3,162},{29,7,5},{29,7,5},{29,25,29},{46,7,165},{31,9,4},{11,5,4},{11,5,4},{11,5,4},{11,39,5},{29,3,162},{13,7,5},{13,7,5},{31,9,0},{43,9,162},{31,9,0},{39,7,200},{27,35,2},{41,37,4},{15,37,5},{39,7,200},{26,1,200},{15,37,5},{0,23,202},{26,1,200},{0,23,202},{43,0,202},{43,0,202},{43,0,202},{43,0,202},{29,37,0},
{29,37,0},{29,37,0},{29,25,4},{31,9,4},{31,9,4},{11,1,530},{27,33,222},{11,5,321},{27,21,219},{27,1,812},{13,19,215},{13,21,9},{13,7,327},{46,19,724},{31,7,219},{25,1,225},{41,3,6},{25,21,30},{41,37,46},{41,3,723},{29,19,203},{13,21,5},{46,7,202},{21,9,723},{46,7,202},{27,1,203},{27,1,203},{27,1,203},{27,37,202},{13,1,168},{13,21,8},{13,21,8},
{13,39,24},{47,21,163},{15,39,8},{41,3,2},{41,3,2},{41,3,2},{41,7,1},{39,11,162},{13,21,4},{13,21,4},{45,39,4},{22,1,162},{45,39,4},{25,1,200},{11,3,4},{9,21,2},{13,21,1},{25,1,200},{33,9,200},{13,21,1},{0,7,202},{33,9,200},{0,7,202},{27,0,202},{27,0,202},{27,0,202},{27,0,202},{13,5,4},{13,5,4},{13,5,4},{13,9,4},{15,39,4},
{15,39,4},{25,1,634},{41,17,216},{25,35,337},{41,5,220},{41,1,863},{27,3,211},{27,35,5},{27,37,331},{47,33,725},{15,21,220},{9,1,251},{9,33,2},{9,35,18},{25,5,48},{25,1,723},{43,33,203},{27,35,5},{47,21,200},{33,9,723},{47,21,200},{41,1,209},{41,1,209},{41,1,209},{41,5,204},{27,1,164},{27,35,4},{27,35,4},{27,7,36},{15,5,164},{29,7,1},{9,3,2},
{9,3,2},{9,3,2},{9,21,2},{25,5,162},{11,5,1},{11,5,1},{29,7,0},{10,1,162},{29,7,0},{23,17,200},{9,33,1},{39,35,1},{43,35,1},{23,17,200},{45,17,200},{43,35,1},{0,21,200},{45,17,200},{0,21,200},{41,0,200},{41,0,200},{41,0,200},{41,0,200},{27,35,0},{27,35,0},{27,35,0},{27,23,4},{29,7,1},{29,7,1},{39,1,724},{9,1,228},{9,3,315},
{25,19,218},{9,1,932},{11,17,213},{41,19,6},{11,5,317},{15,17,725},{29,5,221},{23,1,288},{39,1,3},{23,19,26},{39,19,51},{39,1,723},{27,17,203},{41,19,2},{15,5,208},{31,1,723},{15,5,208},{9,1,227},{9,1,227},{9,1,227},{25,35,209},{41,1,174},{41,19,5},{41,19,5},{11,37,27},{45,19,163},{13,21,6},{39,1,2},{39,1,2},{39,1,2},{39,5,2},{9,3,162},
{41,19,1},{41,19,1},{13,21,5},{31,3,162},{13,21,5},{21,3,200},{39,1,2},{7,19,2},{11,19,1},{21,3,200},{43,1,200},{11,19,1},{0,5,208},{43,1,200},{0,5,208},{25,0,208},{25,0,208},{25,0,208},{25,0,208},{41,19,4},{41,19,4},{41,19,4},{41,7,5},{13,21,2},{13,21,2},{23,1,864},{39,1,300},{39,33,324},{39,3,222},{39,1,1020},{25,1,211},{25,33,5},
{25,35,331},{29,1,732},{13,19,222},{37,1,347},{7,1,13},{7,33,21},{23,3,48},{1,13,723},{25,1,211},{25,33,5},{15,19,203},{13,1,723},{15,19,203},{39,1,251},{39,1,251},{39,1,251},{39,19,203},{9,1,195},{25,33,4},{25,33,4},{25,5,36},{13,3,164},{27,5,5},{7,1,4},{7,1,4},{7,1,4},{7,35,8},{23,3,162},{9,3,1},{9,3,1},{27,5,1},{43,3,162},
{27,5,1},{17,39,200},{7,1,9},{37,33,1},{41,33,1},{17,39,200},{39,17,200},{41,33,1},{0,19,202},{39,17,200},{0,19,202},{39,0,202},{39,0,202},{39,0,202},{39,0,202},{25,33,0},{25,33,0},{25,33,0},{25,21,4},{43,35,4},{43,35,4},{37,1,1012},{7,1,417},{7,1,312},{23,17,218},{7,1,1144},{39,1,268},{39,17,15},{9,3,317},{27,1,804},{27,3,218},{5,1,398},
{21,1,81},{21,17,26},{37,17,51},{35,3,723},{23,1,254},{39,17,6},{13,3,208},{25,1,723},{13,3,208},{7,1,296},{7,1,296},{7,1,296},{23,33,209},{39,1,243},{39,17,14},{39,17,14},{9,35,27},{43,17,163},{27,19,13},{21,1,17},{21,1,17},{21,1,17},{37,3,1},{3,25,162},{39,17,5},{39,17,5},{27,19,9},{29,1,162},{27,19,9},{19,1,200},{5,1,45},{5,17,2},
{9,17,1},{19,1,200},{37,1,200},{9,17,1},{0,3,208},{37,1,200},{0,3,208},{23,0,208},{23,0,208},{23,0,208},{23,0,208},{9,1,10},{9,1,10},{9,1,10},{9,5,9},{27,19,4},{27,19,4},{5,1,919},{21,1,494},{21,1,350},{37,1,201},{21,1,1014},{7,1,251},{23,1,14},{23,33,202},{9,1,721},{41,33,139},{19,1,341},{35,1,109},{5,1,37},{5,1,26},{3,3,546},
{37,1,213},{7,1,1},{43,17,125},{3,3,546},{43,17,125},{21,1,350},{21,1,350},{21,1,350},{37,1,201},{37,1,312},{23,1,14},{23,1,14},{23,3,27},{41,1,168},{25,3,2},{5,1,37},{5,1,37},{5,1,37},{5,17,5},{19,5,162},{7,1,1},{7,1,1},{25,3,1},{37,3,162},{25,3,1},{33,1,113},{19,1,37},{19,1,1},{7,1,0},{33,1,113},{35,1,113},{7,1,0},
{0,17,125},{35,1,113},{0,17,125},{37,0,200},{37,0,200},{37,0,200},{37,0,200},{23,1,10},{23,1,10},{23,1,10},{23,19,5},{25,3,1},{25,3,1},{35,1,773},{35,1,513},{5,1,414},{5,1,227},{35,1,818},{21,1,218},{37,1,69},{7,17,74},{23,1,534},{9,17,30},{3,1,190},{19,1,101},{19,1,65},{35,1,5},{33,33,333},{35,1,114},{5,1,25},{9,17,30},{33,33,333},
{9,17,30},{5,1,414},{5,1,414},{5,1,414},{5,1,227},{21,1,373},{37,1,69},{37,1,69},{7,33,29},{9,1,216},{9,17,5},{19,1,65},{19,1,65},{19,1,65},{35,1,5},{3,3,162},{5,1,25},{5,1,25},{9,17,5},{3,3,162},{9,17,5},{17,1,25},{33,1,9},{33,1,0},{19,1,1},{17,1,25},{33,1,25},{19,1,1},{0,17,29},{33,1,25},{0,17,29},{5,0,218},
{5,0,218},{5,0,218},{5,0,218},{37,1,20},{37,1,20},{37,1,20},{37,3,5},{9,17,1},{9,17,1},{3,1,642},{19,1,504},{19,1,468},{35,1,264},{19,1,657},{35,1,229},{5,1,141},{21,1,26},{21,1,457},{23,1,1},{33,1,134},{33,1,94},{33,1,85},{3,1,26},{1,19,193},{19,1,81},{19,1,45},{23,1,0},{19,1,193},{23,1,0},{19,1,468},{19,1,468},{19,1,468},
{35,1,264},{35,1,425},{5,1,141},{5,1,141},{21,1,26},{7,1,254},{23,1,1},{33,1,85},{33,1,85},{33,1,85},{3,1,26},{17,3,145},{19,1,45},{19,1,45},{23,1,0},{35,1,145},{23,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{35,0,200},{35,0,200},{35,0,200},{35,0,200},{5,1,41},
{5,1,41},{5,1,41},{21,17,5},{23,1,1},{23,1,1},{3,1,418},{3,1,343},{3,1,318},{3,1,243},{3,1,370},{19,1,161},{19,1,125},{35,1,5},{35,1,250},{21,1,25},{17,1,34},{17,1,26},{17,1,25},{33,1,13},{17,1,54},{33,1,22},{33,1,13},{5,1,1},{33,1,54},{5,1,1},{3,1,318},{3,1,318},{3,1,318},{3,1,243},{3,1,270},{19,1,125},{19,1,125},
{35,1,5},{5,1,165},{21,1,25},{17,1,25},{17,1,25},{17,1,25},{33,1,13},{1,3,41},{33,1,13},{33,1,13},{5,1,1},{3,1,41},{5,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{19,0,208},{19,0,208},{19,0,208},{19,0,208},{35,1,85},{35,1,85},{35,1,85},{35,1,5},{21,1,25},
{21,1,25},{0,26,421},{0,38,45},{0,36,2},{0,20,160},{0,38,926},{0,20,577},{0,4,264},{0,18,701},{0,34,989},{0,18,737},{0,26,421},{0,38,45},{0,36,2},{0,20,160},{34,0,925},{0,20,577},{0,4,264},{0,18,701},{0,34,925},{0,18,701},{0,20,1},{0,20,1},{0,20,1},{0,18,1},{0,18,85},{0,2,34},{0,2,34},{0,32,53},{0,32,94},{0,32,62},{0,20,1},
{0,20,1},{0,20,1},{0,18,1},{32,0,85},{0,2,34},{0,2,34},{0,32,53},{18,0,85},{0,32,53},{34,4,421},{0,38,45},{0,36,2},{0,20,160},{34,4,421},{26,0,421},{0,20,160},{0,34,421},{26,0,421},{0,34,421},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,44,421},{0,24,13},{0,22,32},
{0,36,113},{0,24,1262},{0,6,670},{0,20,304},{0,4,886},{0,4,1382},{0,34,953},{0,44,421},{0,24,13},{0,22,32},{0,36,113},{18,4,1261},{0,6,670},{0,20,304},{0,4,886},{36,2,1261},{0,4,886},{0,8,0},{0,8,0},{0,8,0},{0,4,0},{0,4,221},{0,34,73},{0,34,73},{0,2,125},{0,2,246},{0,2,150},{0,8,0},{0,8,0},{0,8,0},{0,4,0},{2,0,221},
{0,34,73},{0,34,73},{0,2,125},{4,0,221},{0,2,125},{22,0,421},{0,24,13},{16,6,8},{0,36,113},{22,0,421},{44,0,421},{0,36,113},{0,20,433},{44,0,421},{0,20,433},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,31,430},{0,26,10},{16,38,79},{0,22,74},{0,42,1514},{0,38,717},{0,6,253},
{0,20,965},{0,36,1713},{0,4,1109},{16,46,425},{16,10,9},{16,38,54},{0,22,74},{20,2,1514},{0,38,717},{0,6,253},{0,20,965},{42,0,1514},{0,20,965},{0,26,10},{0,26,10},{0,26,10},{0,36,9},{0,6,340},{0,20,97},{0,20,97},{0,18,185},{0,18,382},{0,18,221},{16,10,5},{16,10,5},{16,10,5},{16,36,5},{2,2,338},{0,20,97},{0,20,97},{0,18,185},{2,2,338},
{0,18,185},{8,0,421},{0,26,1},{2,38,2},{0,22,65},{8,0,421},{31,0,421},{0,22,65},{0,36,421},{31,0,421},{0,36,421},{0,0,9},{0,0,9},{0,0,9},{0,0,9},{0,32,0},{0,32,0},{0,32,0},{0,16,0},{0,16,1},{0,16,1},{16,45,482},{16,12,66},{32,24,186},{16,8,126},{0,14,1517},{0,24,605},{0,38,117},{0,6,886},{0,22,1815},{0,36,1085},{32,31,421},
{32,42,1},{2,8,51},{32,38,85},{22,16,1514},{0,24,605},{0,38,117},{0,6,886},{44,16,1514},{0,6,886},{16,28,61},{16,28,61},{16,28,61},{16,22,62},{0,24,340},{0,22,37},{0,22,37},{0,4,125},{0,4,456},{0,34,213},{32,42,0},{32,42,0},{32,42,0},{32,6,1},{18,4,338},{0,22,37},{0,22,37},{0,4,125},{36,2,338},{0,4,125},{40,0,421},{32,42,1},{18,8,5},
{0,8,37},{40,0,421},{13,0,421},{0,8,37},{0,22,433},{13,0,421},{0,22,433},{16,0,61},{16,0,61},{16,0,61},{16,0,61},{0,4,1},{0,4,1},{0,4,1},{0,2,1},{0,32,25},{0,32,25},{32,43,623},{32,44,219},{32,10,382},{32,24,243},{0,15,1517},{0,26,497},{0,24,35},{0,22,761},{0,8,2003},{0,22,1050},{18,15,422},{18,28,6},{18,40,51},{2,24,77},{24,0,1514},
{0,26,497},{0,24,35},{0,22,761},{32,8,1514},{0,22,761},{32,30,202},{32,30,202},{32,30,202},{32,8,203},{0,28,338},{0,8,10},{0,8,10},{0,20,89},{0,6,565},{0,20,233},{18,12,1},{18,12,1},{18,12,1},{18,38,1},{2,24,338},{0,8,10},{0,8,10},{0,20,89},{28,0,338},{0,20,89},{40,4,421},{2,28,5},{4,40,1},{0,24,26},{40,4,421},{25,0,421},{0,24,26},
{0,38,425},{25,0,421},{0,38,425},{32,0,202},{32,0,202},{32,0,202},{32,0,202},{0,8,1},{0,8,1},{0,8,1},{0,4,1},{0,34,68},{0,34,68},{2,11,821},{2,30,429},{2,26,634},{2,10,429},{0,27,1514},{0,28,446},{0,10,8},{0,8,670},{0,40,2187},{0,38,1083},{34,29,422},{34,14,6},{4,26,50},{34,10,86},{38,6,1514},{0,28,446},{0,10,8},{0,8,670},{27,0,1514},
{0,8,670},{2,31,401},{2,31,401},{2,31,401},{2,24,404},{0,46,340},{0,10,4},{0,10,4},{0,6,52},{0,22,677},{0,6,277},{34,14,2},{34,14,2},{34,14,2},{34,8,5},{22,2,338},{0,10,4},{0,10,4},{0,6,52},{42,2,338},{0,6,52},{28,0,421},{34,14,5},{36,10,5},{0,10,4},{28,0,421},{7,0,421},{0,10,4},{0,24,425},{7,0,421},{0,24,425},{2,0,400},
{2,0,400},{2,0,400},{2,0,400},{0,26,0},{0,26,0},{0,26,0},{0,36,1},{0,4,149},{0,4,149},{18,9,842},{18,47,462},{34,12,661},{18,26,450},{16,25,1521},{16,14,441},{16,42,15},{16,24,653},{0,26,2030},{0,24,882},{20,13,425},{20,30,9},{20,42,54},{4,26,77},{26,2,1514},{0,14,422},{16,42,11},{0,24,626},{38,8,1514},{0,24,626},{18,45,421},{18,45,421},{18,45,421},
{18,10,422},{16,15,344},{16,42,14},{16,42,14},{16,38,38},{0,24,602},{0,38,162},{20,14,4},{20,14,4},{20,14,4},{20,40,5},{8,2,338},{32,26,2},{32,26,2},{0,38,18},{30,2,338},{0,38,18},{14,0,421},{4,30,5},{6,42,1},{0,42,1},{14,0,421},{19,0,421},{0,42,1},{0,40,421},{19,0,421},{0,40,421},{18,0,421},{18,0,421},{18,0,421},{18,0,421},{16,12,5},
{16,12,5},{16,12,5},{16,6,8},{0,6,80},{0,6,80},{4,9,854},{4,31,458},{4,28,674},{34,12,453},{2,25,1515},{32,30,445},{2,12,9},{32,10,657},{0,28,1850},{0,10,666},{36,27,422},{36,46,6},{6,28,48},{36,12,86},{16,7,1514},{16,46,421},{2,12,8},{0,10,545},{21,0,1514},{0,10,545},{34,13,433},{34,13,433},{34,13,433},{34,42,433},{2,15,339},{2,12,5},{2,12,5},
{32,8,50},{0,10,477},{0,8,62},{36,46,2},{36,46,2},{36,46,2},{36,10,5},{24,4,338},{2,12,4},{2,12,4},{0,24,9},{11,0,338},{0,24,9},{0,1,421},{36,46,5},{38,12,5},{2,12,4},{0,1,421},{1,0,421},{2,12,4},{0,26,425},{1,0,421},{0,26,425},{4,0,433},{4,0,433},{4,0,433},{4,0,433},{2,28,1},{2,28,1},{2,28,1},{2,38,2},{0,8,37},
{0,8,37},{20,7,842},{20,45,455},{36,14,655},{20,28,450},{34,9,1523},{18,47,450},{34,44,13},{18,42,659},{0,14,1731},{0,42,542},{22,11,422},{22,47,5},{22,44,50},{6,28,86},{30,0,1514},{2,47,421},{34,44,4},{0,42,506},{33,0,1514},{0,42,506},{20,43,421},{20,43,421},{20,43,421},{20,12,425},{18,13,347},{34,28,11},{34,28,11},{18,40,43},{0,12,389},{0,10,9},{22,47,1},
{22,47,1},{22,47,1},{22,42,1},{38,10,338},{34,28,2},{34,28,2},{0,10,5},{23,0,338},{0,10,5},{2,1,421},{22,47,4},{8,44,0},{18,44,0},{2,1,421},{1,2,421},{18,44,0},{0,42,425},{1,2,421},{0,42,425},{20,0,421},{20,0,421},{20,0,421},{20,0,421},{18,30,9},{18,30,9},{18,30,9},{34,8,10},{0,10,5},{0,10,5},{6,37,846},{6,29,454},{6,46,685},
{6,14,459},{4,7,1517},{4,31,446},{4,14,19},{34,12,655},{0,47,1635},{0,12,478},{38,9,425},{38,15,6},{8,30,41},{38,14,77},{47,0,1514},{18,15,422},{20,14,11},{0,12,477},{27,6,1514},{0,12,477},{6,27,426},{6,27,426},{6,27,426},{6,44,426},{4,13,340},{4,14,10},{4,14,10},{4,26,50},{0,44,355},{16,26,6},{38,15,5},{38,15,5},{38,15,5},{38,28,5},{28,2,338},
{20,14,2},{20,14,2},{32,26,1},{42,8,338},{32,26,1},{29,0,421},{38,15,2},{40,14,10},{18,30,9},{29,0,421},{33,4,421},{18,30,9},{0,28,425},{33,4,421},{0,28,425},{6,0,425},{6,0,425},{6,0,425},{6,0,425},{4,30,1},{4,30,1},{4,30,1},{4,40,0},{16,26,5},{16,26,5},{22,5,842},{22,43,455},{38,47,679},{22,30,455},{20,21,1521},{20,15,438},{36,46,13},
{20,28,653},{0,15,1553},{16,44,454},{24,9,422},{24,45,5},{24,46,50},{8,30,77},{34,3,1514},{4,45,422},{36,46,4},{0,44,434},{35,4,1514},{0,44,434},{22,41,421},{22,41,421},{22,41,421},{22,14,422},{20,11,342},{36,30,11},{36,30,11},{20,42,38},{0,46,340},{32,12,5},{24,45,1},{24,45,1},{24,45,1},{24,44,1},{16,3,338},{36,30,2},{36,30,2},{32,12,4},{17,0,338},
{32,12,4},{43,0,421},{24,45,4},{10,46,0},{4,46,1},{43,0,421},{19,6,421},{4,46,1},{0,44,425},{19,6,421},{0,44,425},{22,0,421},{22,0,421},{22,0,421},{22,0,421},{20,31,5},{20,31,5},{20,31,5},{20,26,5},{32,12,1},{32,12,1},{8,5,854},{8,27,458},{24,31,670},{38,47,462},{6,5,1517},{6,29,446},{6,47,6},{36,14,655},{0,29,1530},{32,30,465},{40,23,422},
{40,13,2},{10,31,48},{40,46,90},{45,2,1514},{20,13,425},{6,47,6},{0,30,434},{25,8,1514},{0,30,434},{8,25,434},{8,25,434},{8,25,434},{8,30,437},{6,11,340},{6,47,5},{6,47,5},{6,12,52},{16,47,342},{18,28,1},{40,13,1},{40,13,1},{40,13,1},{40,14,5},{30,4,338},{22,46,4},{22,46,4},{18,28,1},{15,8,338},{18,28,1},{11,0,421},{40,13,1},{26,47,5},
{6,47,2},{11,0,421},{5,8,421},{6,47,2},{0,30,425},{5,8,421},{0,30,425},{8,0,433},{8,0,433},{8,0,433},{8,0,433},{6,31,0},{6,31,0},{6,31,0},{6,42,0},{18,28,1},{18,28,1},{24,33,846},{24,41,465},{40,15,658},{24,31,454},{38,5,1518},{22,13,446},{38,15,14},{22,46,667},{0,27,1518},{18,46,453},{26,7,421},{26,27,2},{42,15,53},{10,31,86},{27,0,1514},
{6,43,425},{38,15,10},{16,46,434},{37,8,1514},{16,46,434},{24,23,426},{24,23,426},{24,23,426},{24,47,426},{38,25,344},{38,31,9},{38,31,9},{22,44,41},{18,31,341},{4,14,2},{26,27,1},{26,27,1},{26,27,1},{26,46,1},{45,0,338},{8,31,5},{8,31,5},{4,14,1},{27,8,338},{4,14,1},{8,1,421},{26,27,2},{12,15,1},{6,15,2},{8,1,421},{1,8,421},{6,15,2},
{0,46,433},{1,8,421},{0,46,433},{24,0,425},{24,0,425},{24,0,425},{24,0,425},{38,15,4},{38,15,4},{38,15,4},{38,12,8},{4,14,1},{4,14,1},{10,33,846},{10,25,454},{26,29,658},{10,45,454},{8,3,1517},{8,27,446},{8,45,8},{38,47,650},{32,11,1518},{4,47,462},{42,5,425},{42,11,9},{12,29,43},{42,15,89},{28,13,1514},{22,11,422},{8,45,8},{16,31,426},{27,12,1514},
{16,31,426},{10,23,425},{10,23,425},{10,23,425},{10,15,429},{8,9,340},{8,45,4},{8,45,4},{8,30,50},{34,45,341},{20,30,6},{42,11,5},{42,11,5},{42,11,5},{42,31,8},{29,2,338},{8,45,4},{8,45,4},{36,30,1},{35,6,338},{36,30,1},{23,0,421},{12,11,4},{44,45,5},{8,45,4},{23,0,421},{18,9,421},{8,45,4},{0,31,425},{18,9,421},{0,31,425},{10,0,425},
{10,0,425},{10,0,425},{10,0,425},{8,13,1},{8,13,1},{8,13,1},{8,44,0},{20,30,5},{20,30,5},{26,1,846},{26,39,470},{42,13,658},{26,29,454},{40,3,1523},{24,11,446},{24,13,20},{24,31,658},{2,25,1521},{20,15,458},{28,5,422},{28,25,6},{28,13,51},{12,29,86},{40,3,1514},{8,41,421},{24,13,11},{18,15,421},{16,9,1514},{18,15,421},{26,21,426},{26,21,426},{26,21,426},
{26,45,426},{24,7,347},{40,29,11},{40,29,11},{24,46,41},{20,29,342},{36,47,10},{28,41,1},{28,41,1},{28,41,1},{28,15,2},{43,2,338},{40,29,2},{40,29,2},{36,47,9},{21,8,338},{36,47,9},{37,0,421},{12,25,5},{14,13,1},{8,13,2},{37,0,421},{32,7,421},{8,13,2},{0,15,421},{32,7,421},{0,15,421},{26,0,425},{26,0,425},{26,0,425},{26,0,425},{24,27,9},
{24,27,9},{24,27,9},{40,14,10},{36,47,1},{36,47,1},{12,1,850},{12,23,454},{12,27,666},{12,43,454},{10,1,1517},{10,25,446},{10,43,8},{40,45,666},{34,9,1515},{6,45,462},{44,3,425},{44,9,9},{14,27,50},{44,13,89},{39,2,1514},{24,9,422},{10,43,8},{18,29,426},{34,9,1514},{18,29,426},{12,21,425},{12,21,425},{12,21,425},{12,13,429},{10,7,340},{10,43,4},{10,43,4},
{10,47,50},{36,43,341},{22,31,1},{44,9,5},{44,9,5},{44,9,5},{44,29,8},{38,1,338},{26,13,4},{26,13,4},{22,31,1},{43,28,338},{22,31,1},{5,0,421},{14,9,4},{46,43,5},{10,43,4},{5,0,421},{0,5,421},{10,43,4},{0,29,425},{0,5,421},{0,29,425},{12,0,425},{12,0,425},{12,0,425},{12,0,425},{10,27,0},{10,27,0},{10,27,0},{10,46,0},{22,31,1},
{22,31,1},{44,1,890},{44,7,462},{44,41,666},{28,27,454},{42,1,1518},{26,39,446},{42,11,8},{26,13,666},{4,23,1523},{22,13,454},{30,33,421},{30,23,1},{46,11,50},{30,27,89},{21,0,1514},{40,23,425},{42,11,4},{20,13,425},{18,7,1514},{20,13,425},{28,19,426},{28,19,426},{28,19,426},{28,27,429},{42,21,344},{42,11,8},{42,11,8},{26,15,50},{6,11,341},{8,45,9},{30,23,1},
{30,23,1},{30,23,1},{30,13,0},{39,0,338},{42,11,4},{42,11,4},{8,45,5},{9,28,338},{8,45,5},{14,1,421},{30,23,1},{47,11,0},{26,11,0},{14,1,421},{4,5,421},{26,11,0},{0,13,425},{4,5,421},{0,13,425},{28,0,425},{28,0,425},{28,0,425},{28,0,425},{42,11,4},{42,11,4},{42,11,4},{42,47,5},{8,15,4},{8,15,4},{30,1,950},{14,21,458},{30,25,658},
{14,41,446},{28,1,1542},{12,23,438},{12,25,20},{12,43,658},{36,7,1518},{38,27,470},{46,1,430},{46,37,10},{47,25,41},{46,41,82},{14,3,1514},{26,7,422},{28,41,11},{20,27,426},{32,5,1514},{20,27,426},{14,19,421},{14,19,421},{14,19,421},{14,11,422},{12,35,339},{12,25,11},{12,25,11},{12,29,51},{38,41,338},{24,29,6},{46,37,9},{46,37,9},{46,37,9},{46,27,10},{23,2,338},
{28,41,2},{28,41,2},{40,29,1},{20,9,338},{40,29,1},{17,0,421},{46,37,1},{31,25,10},{26,25,9},{17,0,421},{18,3,421},{26,25,9},{0,27,425},{18,3,421},{0,27,425},{14,0,421},{14,0,421},{14,0,421},{14,0,421},{12,9,2},{12,9,2},{12,9,2},{12,15,1},{24,13,5},{24,13,5},{46,1,1010},{46,5,462},{46,39,650},{30,25,454},{14,1,1575},{28,7,446},{44,9,8},
{28,27,658},{6,21,1523},{24,11,454},{31,1,422},{31,21,6},{31,9,50},{47,25,77},{46,3,1514},{42,21,425},{44,9,4},{22,11,425},{16,3,1514},{22,11,425},{30,17,426},{30,17,426},{30,17,426},{30,25,429},{44,19,344},{44,9,8},{44,9,8},{28,13,43},{8,9,340},{10,43,9},{31,37,1},{31,37,1},{31,37,1},{31,11,2},{37,2,338},{44,9,4},{44,9,4},{10,43,5},{34,7,338},
{10,43,5},{31,1,421},{31,21,5},{45,9,0},{12,9,1},{31,1,421},{32,1,421},{12,9,1},{0,11,425},{32,1,421},{0,11,425},{30,0,425},{30,0,425},{30,0,425},{30,0,425},{44,9,4},{44,9,4},{44,9,4},{44,45,5},{10,13,4},{10,13,4},{31,1,1098},{47,19,453},{47,23,667},{47,39,473},{46,1,1641},{14,21,438},{14,39,14},{14,41,658},{38,5,1518},{40,25,465},{45,1,437},
{15,5,2},{45,23,41},{15,39,86},{31,3,1514},{28,5,422},{30,39,9},{22,25,426},{34,3,1514},{22,25,426},{47,17,434},{47,17,434},{47,17,434},{47,9,437},{14,33,339},{14,39,10},{14,39,10},{14,43,53},{40,39,341},{26,27,2},{15,5,1},{15,5,1},{15,5,1},{15,25,5},{44,1,338},{30,9,5},{30,9,5},{26,27,1},{26,9,338},{26,27,1},{45,1,421},{15,5,1},{13,39,8},
{14,39,4},{45,1,421},{20,1,421},{14,39,4},{0,25,425},{20,1,421},{0,25,425},{47,0,433},{47,0,433},{47,0,433},{47,0,433},{14,7,2},{14,7,2},{14,7,2},{14,13,1},{26,27,2},{26,27,2},{45,1,1202},{31,33,465},{15,37,655},{31,7,461},{47,1,1725},{30,35,446},{46,7,6},{30,25,670},{24,19,1518},{26,9,458},{13,1,469},{29,19,1},{13,7,52},{29,23,85},{15,1,1514},
{14,35,426},{46,7,5},{24,9,434},{9,11,1514},{24,9,434},{31,1,434},{31,1,434},{31,1,434},{31,39,426},{46,17,339},{46,7,6},{46,7,6},{30,11,48},{10,7,341},{12,41,2},{29,19,1},{29,19,1},{29,19,1},{29,9,0},{31,5,338},{47,23,4},{47,23,4},{12,41,1},{14,9,338},{12,41,1},{43,1,421},{29,19,1},{43,7,0},{30,7,0},{43,1,421},{8,1,421},{30,7,0},
{0,9,433},{8,1,421},{0,9,433},{31,0,425},{31,0,425},{31,0,425},{31,0,425},{46,7,2},{46,7,2},{46,7,2},{46,27,5},{12,41,1},{12,41,1},{29,1,1346},{45,17,454},{29,21,653},{45,37,459},{45,1,1818},{47,19,446},{47,37,13},{46,39,679},{10,3,1518},{42,23,455},{27,1,526},{13,33,5},{43,21,38},{13,7,93},{27,5,1514},{30,3,425},{31,37,11},{40,23,421},{6,1,1514},
{40,23,421},{45,1,434},{45,1,434},{45,1,434},{45,7,429},{47,1,340},{47,37,4},{47,37,4},{47,25,50},{42,37,338},{44,25,5},{13,33,4},{13,33,4},{13,33,4},{13,23,5},{17,2,338},{31,37,2},{31,37,2},{44,25,1},{16,1,338},{44,25,1},{11,1,421},{13,33,1},{27,21,5},{30,21,5},{11,1,421},{26,1,421},{30,21,5},{0,23,421},{26,1,421},{0,23,421},{45,0,425},
{45,0,425},{45,0,425},{45,0,425},{47,5,1},{47,5,1},{47,5,1},{47,11,0},{44,25,4},{44,25,4},{43,1,1502},{13,1,478},{13,35,655},{29,5,461},{29,1,1926},{31,3,446},{15,5,19},{47,7,685},{26,17,1518},{28,7,454},{11,1,569},{27,17,6},{27,5,50},{43,21,97},{17,10,1514},{47,33,425},{15,5,10},{26,7,426},{24,1,1514},{26,7,426},{13,1,477},{13,1,477},{13,1,477},
{29,37,426},{15,1,355},{15,21,11},{15,21,11},{31,9,41},{12,5,341},{14,39,6},{27,33,1},{27,33,1},{27,33,1},{27,7,2},{29,3,338},{15,21,2},{15,21,2},{14,39,5},{43,9,338},{14,39,5},{25,1,421},{43,1,5},{41,5,0},{31,5,1},{25,1,421},{14,1,421},{31,5,1},{0,7,425},{14,1,421},{0,7,425},{29,0,425},{29,0,425},{29,0,425},{29,0,425},{31,19,9},
{31,19,9},{31,19,9},{31,25,10},{14,39,2},{14,39,2},{11,1,1634},{43,1,542},{43,19,659},{43,35,459},{43,1,2070},{45,17,446},{45,35,13},{15,37,655},{12,1,1518},{44,21,455},{25,1,646},{11,1,9},{41,19,43},{11,5,93},{25,3,1514},{31,1,422},{29,35,11},{42,21,421},{12,1,1514},{42,21,421},{43,1,506},{43,1,506},{43,1,506},{43,5,429},{29,1,381},{45,35,4},{45,35,4},
{45,23,50},{44,35,338},{46,23,5},{11,1,5},{11,1,5},{11,1,5},{11,21,5},{39,11,338},{29,35,2},{29,35,2},{46,23,1},{22,1,338},{46,23,1},{39,1,421},{11,1,5},{25,19,10},{31,19,9},{39,1,421},{31,1,421},{31,19,9},{0,21,421},{31,1,421},{0,21,421},{43,0,425},{43,0,425},{43,0,425},{43,0,425},{45,19,0},{45,19,0},{45,19,0},{45,9,0},{46,23,4},
{46,23,4},{25,1,1874},{11,1,666},{11,33,657},{27,3,451},{11,1,2201},{13,1,454},{13,3,9},{29,5,674},{14,1,1557},{30,5,458},{39,1,722},{9,1,62},{9,33,50},{25,19,85},{9,1,1514},{29,1,446},{13,3,5},{44,5,433},{46,1,1514},{44,5,433},{11,1,545},{11,1,545},{11,1,545},{27,19,429},{43,1,437},{13,3,8},{13,3,8},{29,7,48},{30,19,341},{47,37,6},{25,1,9},
{25,1,9},{25,1,9},{25,5,0},{25,5,338},{13,3,4},{13,3,4},{47,37,2},{10,1,338},{47,37,2},{37,1,421},{9,1,37},{39,3,2},{29,3,1},{37,1,421},{43,1,421},{29,3,1},{0,5,433},{43,1,421},{0,5,433},{27,0,425},{27,0,425},{27,0,425},{27,0,425},{13,3,4},{13,3,4},{13,3,4},{13,39,5},{46,21,5},{46,21,5},{9,1,2030},{25,1,882},{25,17,653},
{41,33,453},{9,1,2382},{27,1,546},{43,17,15},{13,35,661},{47,1,1653},{46,19,462},{7,1,837},{39,1,162},{39,17,38},{9,33,89},{19,9,1514},{27,1,521},{43,17,14},{44,19,421},{45,1,1514},{44,19,421},{25,1,626},{25,1,626},{25,1,626},{41,3,422},{11,1,488},{43,17,11},{43,17,11},{43,21,54},{46,33,340},{31,21,9},{39,1,18},{39,1,18},{39,1,18},{9,19,10},{9,3,338},
{27,33,2},{27,33,2},{15,21,4},{31,3,338},{15,21,4},{5,1,421},{7,1,80},{7,17,8},{13,17,5},{5,1,421},{9,1,421},{13,17,5},{0,19,421},{9,1,421},{0,19,421},{41,0,421},{41,0,421},{41,0,421},{41,0,421},{43,1,1},{43,1,1},{43,1,1},{43,7,1},{31,5,5},{31,5,5},{23,1,2201},{39,1,1083},{9,1,670},{25,1,450},{39,1,2443},{11,1,684},{11,1,8},
{27,3,634},{29,1,1751},{31,3,429},{21,1,890},{7,1,277},{7,1,52},{23,17,74},{21,17,1459},{25,1,603},{11,1,4},{30,3,401},{17,21,1459},{30,3,401},{9,1,670},{9,1,670},{9,1,670},{25,17,429},{25,1,579},{11,1,8},{11,1,8},{27,5,50},{47,1,340},{15,35,6},{7,1,52},{7,1,52},{7,1,52},{23,3,0},{23,3,338},{11,1,4},{11,1,4},{15,35,2},{43,3,338},
{15,35,2},{19,1,392},{5,1,149},{37,1,1},{27,1,0},{19,1,392},{37,1,392},{27,1,0},{0,3,400},{37,1,392},{0,3,400},{25,0,425},{25,0,425},{25,0,425},{25,0,425},{11,1,4},{11,1,4},{11,1,4},{11,37,5},{15,35,5},{15,35,5},{7,1,1901},{23,1,1050},{23,1,761},{39,1,426},{7,1,2093},{9,1,551},{25,1,35},{11,33,382},{13,1,1407},{45,33,219},{5,1,638},
{21,1,233},{21,1,89},{7,1,29},{17,39,1064},{23,1,398},{9,1,10},{31,33,202},{39,17,1064},{31,33,202},{23,1,761},{23,1,761},{23,1,761},{39,1,426},{9,1,638},{25,1,35},{25,1,35},{41,19,51},{45,1,355},{29,19,6},{21,1,89},{21,1,89},{21,1,89},{7,17,5},{3,25,338},{9,1,10},{9,1,10},{13,19,1},{29,1,338},{13,19,1},{3,1,200},{35,1,68},{5,1,1},
{9,1,1},{3,1,200},{5,1,200},{9,1,1},{0,33,202},{5,1,200},{0,33,202},{39,0,425},{39,0,425},{39,0,425},{39,0,425},{25,1,26},{25,1,26},{25,1,26},{41,5,1},{29,3,5},{29,3,5},{21,1,1606},{37,1,1085},{7,1,886},{7,1,461},{21,1,1749},{23,1,470},{39,1,117},{25,33,186},{11,1,1166},{13,17,66},{19,1,461},{35,1,213},{5,1,125},{21,1,2},{3,3,722},
{7,1,266},{23,1,37},{29,17,61},{3,3,722},{29,17,61},{7,1,886},{7,1,886},{7,1,886},{7,1,461},{23,1,770},{39,1,117},{39,1,117},{9,3,51},{43,1,426},{43,33,1},{5,1,125},{5,1,125},{5,1,125},{21,1,2},{19,5,338},{23,1,37},{23,1,37},{43,33,0},{37,3,338},{43,33,0},{1,3,61},{33,1,25},{3,1,1},{5,1,1},{1,3,61},{3,1,61},{5,1,1},
{0,17,61},{3,1,61},{0,17,61},{23,0,433},{23,0,433},{23,0,433},{23,0,433},{9,1,37},{9,1,37},{9,1,37},{9,19,5},{43,33,1},{43,33,1},{5,1,1450},{5,1,1109},{21,1,965},{37,1,542},{5,1,1505},{7,1,478},{7,1,253},{39,17,79},{9,1,1054},{27,1,10},{3,1,318},{19,1,221},{19,1,185},{35,1,29},{33,33,509},{35,1,210},{21,1,97},{27,1,10},{33,33,509},
{27,1,10},{21,1,965},{21,1,965},{21,1,965},{37,1,542},{21,1,900},{7,1,253},{7,1,253},{39,17,54},{41,1,549},{11,17,9},{19,1,185},{19,1,185},{19,1,185},{35,1,29},{3,3,338},{21,1,97},{21,1,97},{11,17,5},{3,3,338},{11,17,5},{1,17,5},{17,1,1},{17,1,0},{33,1,0},{1,17,5},{17,1,5},{33,1,0},{0,1,9},{17,1,5},{0,1,9},{37,0,421},
{37,0,421},{37,0,421},{37,0,421},{23,1,65},{23,1,65},{23,1,65},{39,3,2},{27,1,1},{27,1,1},{35,1,1213},{35,1,953},{5,1,886},{5,1,545},{35,1,1186},{21,1,448},{21,1,304},{23,1,32},{23,1,790},{25,1,13},{33,1,198},{3,1,150},{3,1,125},{19,1,36},{33,1,294},{19,1,121},{35,1,73},{9,1,0},{1,33,294},{9,1,0},{5,1,886},{5,1,886},{5,1,886},
{5,1,545},{5,1,765},{21,1,304},{21,1,304},{23,1,32},{39,1,486},{25,1,13},{3,1,125},{3,1,125},{3,1,125},{19,1,36},{3,1,221},{35,1,73},{35,1,73},{9,1,0},{5,1,221},{9,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{21,0,433},{21,0,433},{21,0,433},{21,0,433},{37,1,113},
{37,1,113},{37,1,113},{7,17,8},{25,1,13},{25,1,13},{3,1,885},{19,1,737},{19,1,701},{35,1,485},{19,1,834},{5,1,364},{5,1,264},{37,1,2},{7,1,554},{39,1,45},{17,1,82},{33,1,62},{33,1,53},{33,1,13},{1,3,114},{33,1,54},{3,1,34},{21,1,1},{3,1,114},{21,1,1},{19,1,701},{19,1,701},{19,1,701},{35,1,485},{35,1,574},{5,1,264},{5,1,264},
{37,1,2},{7,1,329},{39,1,45},{33,1,53},{33,1,53},{33,1,53},{33,1,13},{33,1,85},{3,1,34},{3,1,34},{21,1,1},{19,1,85},{21,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{35,0,421},{35,0,421},{35,0,421},{35,0,421},{21,1,160},{21,1,160},{21,1,160},{37,1,2},{39,1,45},
{39,1,45},{0,30,882},{0,10,100},{0,38,2},{0,6,340},{0,10,1896},{0,22,1189},{0,6,565},{0,4,1421},{0,20,2043},{0,4,1521},{0,30,882},{0,10,100},{0,38,2},{0,6,340},{4,2,1896},{0,22,1189},{0,6,565},{0,4,1421},{10,0,1896},{0,4,1421},{0,22,1},{0,22,1},{0,22,1},{0,34,0},{0,4,164},{0,18,52},{0,18,52},{0,32,98},{0,32,179},{0,32,107},{0,22,1},
{0,22,1},{0,22,1},{0,34,0},{32,16,162},{0,18,52},{0,18,52},{0,32,98},{16,32,162},{0,32,98},{38,0,882},{0,10,100},{0,38,2},{0,6,340},{38,0,882},{30,0,882},{0,6,340},{0,20,884},{30,0,882},{0,20,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,15,884},{0,12,34},{0,24,13},
{0,38,250},{0,12,2355},{0,8,1355},{0,22,585},{0,20,1669},{0,36,2567},{0,20,1813},{0,15,884},{0,12,34},{0,24,13},{0,38,250},{0,12,2355},{0,8,1355},{0,22,585},{0,20,1669},{12,0,2355},{0,20,1669},{0,10,0},{0,10,0},{0,10,0},{0,20,1},{0,20,340},{0,4,125},{0,4,125},{0,2,200},{0,18,376},{0,2,225},{0,10,0},{0,10,0},{0,10,0},{0,20,1},{16,4,338},
{0,4,125},{0,4,125},{0,2,200},{4,16,338},{0,2,200},{24,0,882},{0,12,34},{16,24,4},{0,38,250},{24,0,882},{32,8,882},{0,38,250},{0,6,884},{32,8,882},{0,6,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,27,882},{0,14,9},{0,40,81},{0,8,202},{0,14,2899},{0,24,1539},{0,8,643},
{0,6,2004},{0,6,3189},{0,36,2207},{0,27,882},{0,14,9},{16,40,61},{0,8,202},{6,2,2899},{0,24,1539},{0,8,643},{0,6,2004},{14,0,2899},{0,6,2004},{0,28,1},{0,28,1},{0,28,1},{0,6,4},{0,22,578},{0,36,221},{0,36,221},{0,18,365},{0,18,632},{0,18,401},{0,28,1},{0,28,1},{0,28,1},{0,6,4},{32,4,578},{0,36,221},{0,36,221},{0,18,365},{22,0,578},
{0,18,365},{38,6,882},{0,14,9},{2,40,1},{0,8,202},{38,6,882},{27,0,882},{0,8,202},{0,22,884},{27,0,882},{0,22,884},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{16,25,910},{16,46,31},{16,26,151},{0,40,181},{0,31,3048},{0,26,1416},{0,40,506},{0,22,1944},{0,38,3484},{0,6,2233},{16,25,885},
{16,46,6},{32,26,78},{16,40,165},{36,6,3048},{0,26,1416},{0,40,506},{0,22,1944},{31,0,3048},{0,22,1944},{16,30,26},{16,30,26},{16,30,26},{16,38,27},{0,24,650},{0,22,157},{0,22,157},{0,4,325},{0,4,766},{0,4,425},{16,30,1},{16,30,1},{16,30,1},{16,38,2},{18,4,648},{0,22,157},{0,22,157},{0,4,325},{36,2,648},{0,4,325},{26,2,882},{0,46,5},{18,26,4},
{0,40,145},{26,2,882},{38,8,882},{0,40,145},{0,8,890},{38,8,882},{0,8,890},{16,0,26},{16,0,26},{16,0,26},{16,0,26},{0,18,0},{0,18,0},{0,18,0},{0,32,1},{0,16,10},{0,16,10},{16,7,1009},{16,31,134},{32,12,297},{16,26,238},{0,43,3048},{0,28,1224},{0,26,267},{0,8,1764},{0,24,3685},{0,38,2157},{2,9,885},{2,47,5},{18,12,77},{32,26,165},{6,8,3048},
{0,28,1224},{0,26,267},{0,8,1764},{43,0,3048},{0,8,1764},{16,45,125},{16,45,125},{16,45,125},{16,40,126},{0,28,648},{0,24,80},{0,24,80},{0,36,260},{0,6,875},{0,20,413},{2,46,1},{2,46,1},{2,46,1},{2,24,2},{2,24,648},{0,24,80},{0,24,80},{0,36,260},{28,0,648},{0,36,260},{44,0,882},{2,47,4},{4,42,1},{0,26,98},{44,0,882},{26,8,882},{0,26,98},
{0,24,890},{26,8,882},{0,24,890},{16,0,125},{16,0,125},{16,0,125},{16,0,125},{0,6,0},{0,6,0},{0,6,0},{0,18,1},{0,18,37},{0,18,37},{32,5,1147},{32,45,282},{2,44,523},{32,12,381},{0,9,3051},{0,14,1110},{0,12,116},{0,24,1658},{0,26,3859},{0,8,2173},{18,23,886},{18,15,6},{34,28,69},{18,42,166},{40,4,3048},{0,14,1110},{0,12,116},{0,24,1658},{22,8,3048},
{0,24,1658},{32,13,265},{32,13,265},{32,13,265},{32,10,266},{0,46,650},{0,26,26},{0,26,26},{0,22,194},{0,22,987},{0,6,427},{18,15,5},{18,15,5},{18,15,5},{18,40,5},{22,2,648},{0,26,26},{0,26,26},{0,22,194},{42,2,648},{0,22,194},{30,0,882},{18,15,2},{20,28,4},{0,12,52},{30,0,882},{33,0,882},{0,12,52},{0,10,884},{33,0,882},{0,10,884},{32,0,265},
{32,0,265},{32,0,265},{32,0,265},{0,24,0},{0,24,0},{0,24,0},{0,4,4},{0,34,89},{0,34,89},{2,3,1365},{2,13,510},{18,14,813},{2,28,566},{0,37,3048},{0,46,1013},{0,44,42},{0,10,1509},{0,12,4057},{0,10,2185},{4,7,885},{4,45,10},{20,44,86},{34,28,173},{28,0,3048},{0,46,1013},{0,44,42},{0,10,1509},{37,0,3048},{0,10,1509},{2,11,482},{2,11,482},{2,11,482},
{2,42,481},{0,29,650},{0,12,4},{0,12,4},{0,38,128},{0,8,1161},{0,22,483},{4,15,2},{4,15,2},{4,15,2},{4,26,2},{8,2,648},{0,12,4},{0,12,4},{0,38,128},{30,2,648},{0,38,128},{47,0,882},{34,45,2},{6,44,1},{0,44,26},{47,0,882},{27,6,882},{0,44,26},{0,26,890},{27,6,882},{0,26,890},{2,0,481},{2,0,481},{2,0,481},{2,0,481},{0,12,0},
{0,12,0},{0,12,0},{0,6,1},{0,20,180},{0,20,180},{18,17,1647},{18,27,810},{34,46,1178},{2,14,837},{0,19,3048},{0,15,933},{0,14,21},{0,26,1443},{0,28,4329},{0,26,2227},{20,21,885},{20,29,5},{36,30,69},{20,44,182},{42,6,3048},{0,15,933},{0,14,21},{0,26,1443},{19,0,3048},{0,26,1443},{18,25,765},{18,25,765},{18,25,765},{18,28,766},{0,11,648},{0,14,5},{0,14,5},
{0,24,89},{0,10,1342},{0,8,557},{20,29,1},{20,29,1},{20,29,1},{20,42,2},{24,4,648},{16,44,2},{16,44,2},{0,24,89},{11,0,648},{0,24,89},{34,3,882},{4,13,4},{22,30,4},{0,14,17},{34,3,882},{35,4,882},{0,14,17},{0,12,884},{35,4,882},{0,12,884},{18,0,765},{18,0,765},{18,0,765},{18,0,765},{0,30,0},{0,30,0},{0,30,0},{0,38,0},{0,6,274},
{0,6,274},{34,1,1782},{34,41,940},{4,31,1356},{34,46,950},{16,1,3052},{16,29,910},{16,46,22},{0,12,1335},{0,46,4231},{0,12,1924},{6,5,882},{6,43,5},{22,47,86},{36,30,174},{32,3,3048},{0,13,900},{32,46,13},{0,12,1299},{39,4,3048},{0,12,1299},{34,23,882},{34,23,882},{34,23,882},{34,14,883},{16,9,651},{16,46,18},{16,46,18},{16,10,74},{0,42,1256},{0,40,434},{6,13,0},
{6,13,0},{6,13,0},{6,28,1},{38,10,648},{2,30,2},{2,30,2},{0,10,45},{23,0,648},{0,10,45},{4,1,882},{36,27,4},{8,46,4},{0,46,5},{4,1,882},{1,4,882},{0,46,5},{0,28,890},{1,4,882},{0,28,890},{34,0,882},{34,0,882},{34,0,882},{34,0,882},{16,31,1},{16,31,1},{16,31,1},{16,24,1},{0,8,250},{0,8,250},{20,1,1814},{20,25,945},{36,15,1363},
{4,47,945},{2,1,3052},{32,43,916},{2,47,23},{16,44,1337},{0,31,3975},{0,28,1612},{22,19,886},{22,11,6},{38,31,77},{22,46,185},{2,1,3048},{0,27,883},{2,47,19},{0,44,1188},{1,2,3048},{0,44,1188},{20,7,901},{20,7,901},{20,7,901},{20,30,900},{2,9,654},{2,47,14},{2,47,14},{32,26,70},{0,28,1059},{0,26,236},{22,27,4},{22,27,4},{22,27,4},{22,14,8},{28,2,648},
{18,46,5},{18,46,5},{0,42,36},{42,8,648},{0,42,36},{27,0,882},{22,11,2},{24,31,4},{0,31,5},{27,0,882},{37,8,882},{0,31,5},{0,14,882},{37,8,882},{0,14,882},{4,0,900},{4,0,900},{4,0,900},{4,0,900},{2,15,5},{2,15,5},{2,15,5},{2,40,4},{0,40,146},{0,40,146},{6,1,1864},{36,39,947},{6,29,1356},{36,31,943},{34,1,3073},{18,27,910},{18,15,21},
{2,14,1335},{0,45,3751},{0,14,1379},{8,3,885},{8,41,5},{24,15,85},{38,31,182},{29,0,3048},{32,11,884},{34,15,16},{0,30,1146},{37,6,3048},{0,30,1146},{36,21,882},{36,21,882},{36,21,882},{36,47,886},{18,7,649},{18,15,21},{18,15,21},{18,12,74},{0,14,945},{0,12,91},{8,11,2},{8,11,2},{8,11,2},{8,30,2},{16,3,648},{34,31,1},{34,31,1},{0,12,10},{17,0,648},
{0,12,10},{28,13,882},{38,25,4},{10,15,1},{2,15,1},{28,13,882},{27,12,882},{2,15,1},{0,30,890},{27,12,882},{0,30,890},{36,0,882},{36,0,882},{36,0,882},{36,0,882},{18,29,1},{18,29,1},{18,29,1},{18,26,1},{0,42,68},{0,42,68},{22,1,1944},{22,23,945},{38,13,1363},{6,45,945},{20,1,3115},{34,41,916},{4,45,18},{18,46,1337},{0,13,3580},{0,46,1153},{24,17,886},
{24,9,6},{40,29,69},{24,15,168},{46,10,3048},{2,25,885},{4,45,14},{0,46,1053},{23,8,3048},{0,46,1053},{22,5,901},{22,5,901},{22,5,901},{6,31,901},{4,7,654},{4,45,9},{4,45,9},{34,28,70},{0,47,825},{0,44,25},{24,25,4},{24,25,4},{24,25,4},{24,46,8},{30,4,648},{20,15,0},{20,15,0},{0,44,0},{15,8,648},{0,44,0},{40,3,882},{24,9,2},{26,29,4},
{2,29,5},{40,3,882},{16,9,882},{2,29,5},{0,47,890},{16,9,882},{0,47,890},{6,0,900},{6,0,900},{6,0,900},{6,0,900},{4,13,5},{4,13,5},{4,13,5},{4,42,4},{0,28,18},{0,28,18},{8,1,2056},{38,21,956},{8,43,1348},{38,29,949},{6,1,3156},{20,25,910},{20,13,22},{4,47,1318},{0,11,3384},{0,31,1015},{10,1,882},{10,39,5},{26,13,75},{40,29,174},{27,2,3048},
{34,9,885},{36,13,10},{0,31,990},{39,10,3048},{0,31,990},{38,3,885},{38,3,885},{38,3,885},{38,45,885},{20,5,649},{20,13,21},{20,13,21},{20,14,66},{0,15,729},{0,30,5},{10,9,0},{10,9,0},{10,9,0},{10,31,1},{45,0,648},{6,29,2},{6,29,2},{16,30,4},{27,8,648},{16,30,4},{10,1,882},{40,23,1},{12,13,1},{4,13,1},{10,1,882},{4,9,882},{4,13,1},
{0,31,890},{4,9,882},{0,31,890},{38,0,884},{38,0,884},{38,0,884},{38,0,884},{20,27,1},{20,27,1},{20,27,1},{20,28,1},{0,30,1},{0,30,1},{40,1,2134},{24,21,943},{40,11,1348},{8,43,952},{38,1,3240},{36,39,914},{6,27,22},{36,15,1330},{0,25,3244},{0,15,951},{42,1,891},{26,7,10},{42,27,77},{26,13,171},{8,1,3048},{4,23,886},{6,27,21},{0,15,950},{1,8,3048},
{0,15,950},{24,3,890},{24,3,890},{24,3,890},{24,29,890},{6,35,649},{6,43,10},{6,43,10},{36,30,77},{0,13,675},{32,46,6},{26,7,10},{26,7,10},{26,7,10},{26,15,10},{29,2,648},{22,13,2},{22,13,2},{2,46,2},{35,6,648},{2,46,2},{21,0,882},{26,7,1},{28,27,4},{4,27,5},{21,0,882},{18,7,882},{4,27,5},{0,45,884},{18,7,882},{0,45,884},{24,0,890},
{24,0,890},{24,0,890},{24,0,890},{6,11,1},{6,11,1},{6,11,1},{6,44,2},{16,47,2},{16,47,2},{26,1,2252},{40,35,940},{10,41,1348},{40,27,954},{24,1,3321},{22,23,910},{22,11,22},{6,45,1318},{0,39,3156},{0,29,940},{28,1,920},{12,37,5},{28,41,77},{12,27,180},{23,0,3048},{36,7,884},{38,11,10},{0,29,915},{18,9,3048},{0,29,915},{40,17,882},{40,17,882},{40,17,882},
{40,43,883},{22,3,649},{38,27,20},{38,27,20},{22,47,86},{0,27,656},{2,31,9},{12,7,0},{12,7,0},{12,7,0},{12,29,1},{43,2,648},{38,27,4},{38,27,4},{34,47,4},{21,8,648},{34,47,4},{14,3,882},{42,21,1},{14,11,1},{6,11,1},{14,3,882},{32,5,882},{6,11,1},{0,29,890},{32,5,882},{0,29,890},{40,0,882},{40,0,882},{40,0,882},{40,0,882},{22,25,1},
{22,25,1},{22,25,1},{22,30,2},{2,31,0},{2,31,0},{42,1,2404},{26,19,935},{42,9,1348},{10,41,952},{40,1,3409},{38,37,914},{8,41,30},{38,29,1354},{0,37,3087},{2,13,951},{14,1,954},{28,5,6},{44,25,77},{28,11,185},{9,6,3048},{6,21,886},{24,41,21},{0,43,900},{32,7,3048},{0,43,900},{26,1,891},{26,1,891},{26,1,891},{26,27,891},{8,3,654},{8,41,14},{8,41,14},
{38,31,77},{16,41,651},{34,15,3},{28,5,5},{28,5,5},{28,5,5},{28,13,5},{38,1,648},{24,11,2},{24,11,2},{4,15,2},{43,28,648},{4,15,2},{46,3,882},{28,5,2},{30,25,4},{6,25,5},{46,3,882},{16,3,882},{6,25,5},{0,43,884},{16,3,882},{0,43,884},{26,0,890},{26,0,890},{26,0,890},{26,0,890},{8,9,5},{8,9,5},{8,9,5},{8,46,4},{34,15,2},
{34,15,2},{28,1,2612},{12,3,951},{28,39,1354},{42,9,952},{26,1,3544},{24,21,910},{40,9,30},{8,43,1348},{0,35,3060},{18,27,935},{30,1,1005},{14,35,3},{30,39,77},{44,25,185},{41,12,3048},{38,35,890},{40,9,14},{0,27,891},{16,5,3048},{0,27,891},{42,1,900},{42,1,900},{42,1,900},{42,41,885},{24,1,654},{40,25,21},{40,25,21},{24,45,77},{2,9,651},{4,29,6},{14,5,2},
{14,5,2},{14,5,2},{14,27,2},{39,0,648},{10,25,2},{10,25,2},{4,29,5},{9,28,648},{4,29,5},{1,0,882},{44,19,2},{47,9,4},{8,9,5},{1,0,882},{0,1,882},{8,9,5},{0,27,890},{0,1,882},{0,27,890},{42,0,884},{42,0,884},{42,0,884},{42,0,884},{24,7,5},{24,7,5},{24,7,5},{24,31,4},{4,29,2},{4,29,2},{14,1,2774},{28,1,940},{44,7,1318},
{12,39,966},{12,1,3700},{40,35,916},{10,23,22},{40,11,1348},{16,19,3051},{34,41,940},{47,1,1061},{30,3,9},{46,23,86},{30,9,190},{14,1,3048},{8,19,886},{26,39,20},{16,41,882},{4,5,3048},{16,41,882},{28,1,915},{28,1,915},{28,1,915},{28,25,890},{10,1,657},{10,39,10},{10,39,10},{40,29,77},{18,39,654},{36,13,5},{46,35,4},{46,35,4},{46,35,4},{46,11,4},{23,2,648},
{42,9,4},{42,9,4},{6,13,0},{20,9,648},{6,13,0},{15,1,882},{30,3,0},{31,23,2},{24,23,1},{15,1,882},{9,11,882},{24,23,1},{0,41,882},{9,11,882},{0,41,882},{28,0,890},{28,0,890},{28,0,890},{28,0,890},{10,7,1},{10,7,1},{10,7,1},{10,15,1},{20,43,1},{20,43,1},{46,1,3014},{14,1,951},{14,37,1330},{44,23,958},{14,1,3865},{26,19,913},{26,7,22},
{10,41,1348},{32,33,3049},{20,25,943},{15,1,1154},{47,33,6},{31,37,77},{46,23,185},{17,0,3048},{40,3,885},{42,7,10},{2,25,890},{18,3,3048},{2,25,890},{14,1,950},{14,1,950},{14,1,950},{44,39,885},{42,1,672},{26,7,21},{26,7,21},{26,43,77},{34,7,652},{6,27,10},{47,3,2},{47,3,2},{47,3,2},{47,25,2},{37,2,648},{12,23,2},{12,23,2},{6,27,10},{34,7,648},
{6,27,10},{27,5,882},{46,17,2},{45,7,2},{10,7,1},{27,5,882},{6,1,882},{10,7,1},{0,25,890},{6,1,882},{0,25,890},{44,0,884},{44,0,884},{44,0,884},{44,0,884},{26,5,5},{26,5,5},{26,5,5},{26,29,4},{6,27,1},{6,27,1},{47,1,3214},{30,1,1015},{46,5,1318},{14,37,966},{30,1,4009},{42,33,916},{12,21,22},{42,9,1348},{18,17,3051},{20,39,956},{29,1,1240},
{31,1,5},{15,21,66},{31,7,185},{27,9,3048},{10,17,886},{12,21,21},{2,39,885},{14,7,3048},{2,39,885},{30,1,990},{30,1,990},{30,1,990},{30,23,890},{28,1,715},{12,37,10},{12,37,10},{12,27,75},{20,37,651},{38,11,5},{31,17,4},{31,17,4},{31,17,4},{31,9,8},{44,1,648},{28,7,2},{28,7,2},{8,11,0},{26,9,648},{8,11,0},{17,10,882},{31,1,1},{29,21,1},
{26,21,1},{17,10,882},{24,1,882},{26,21,1},{0,39,884},{24,1,882},{0,39,884},{30,0,890},{30,0,890},{30,0,890},{30,0,890},{12,5,1},{12,5,1},{12,5,1},{12,13,1},{22,41,1},{22,41,1},{15,1,3526},{47,1,1153},{47,19,1337},{46,5,958},{47,1,4231},{28,17,910},{44,5,18},{12,39,1363},{20,1,3067},{22,23,945},{43,1,1380},{45,1,25},{29,35,70},{45,21,185},{13,3,3048},
{12,1,888},{44,5,9},{18,7,901},{44,5,3048},{18,7,901},{47,1,1053},{47,1,1053},{47,1,1053},{46,37,890},{14,1,762},{44,5,14},{44,5,14},{28,41,69},{6,5,650},{8,25,6},{45,1,0},{45,1,0},{45,1,0},{45,23,0},{31,5,648},{14,21,0},{14,21,0},{24,25,4},{14,9,648},{24,25,4},{41,1,882},{29,1,18},{43,5,4},{12,5,5},{41,1,882},{47,3,882},{12,5,5},
{0,7,900},{47,3,882},{0,7,900},{46,0,890},{46,0,890},{46,0,890},{46,0,890},{28,3,5},{28,3,5},{28,3,5},{28,27,4},{8,25,2},{8,25,2},{29,1,3764},{15,1,1379},{15,3,1335},{47,35,950},{15,1,4477},{14,1,925},{14,19,21},{28,7,1356},{22,1,3145},{38,37,947},{11,1,1485},{13,1,91},{13,19,74},{29,35,178},{43,1,3048},{14,1,925},{14,19,21},{20,37,882},{8,1,3048},
{20,37,882},{31,1,1146},{31,1,1146},{31,1,1146},{31,21,890},{46,1,841},{14,35,16},{14,35,16},{14,25,85},{38,35,649},{40,9,5},{13,1,10},{13,1,10},{13,1,10},{13,7,9},{17,2,648},{30,35,1},{30,35,1},{10,9,2},{16,1,648},{10,9,2},{9,1,882},{43,1,68},{27,19,1},{28,19,1},{9,1,882},{46,1,882},{28,19,1},{0,37,882},{46,1,882},{0,37,882},{31,0,890},
{31,0,890},{31,0,890},{31,0,890},{14,3,1},{14,3,1},{14,3,1},{14,11,1},{40,9,4},{40,9,4},{13,1,4076},{29,1,1612},{45,17,1337},{15,3,950},{29,1,4684},{46,1,1035},{46,3,23},{14,37,1363},{24,1,3256},{24,21,945},{41,1,1650},{27,1,236},{27,33,70},{13,19,174},{11,1,3048},{47,1,1013},{46,3,14},{20,5,901},{5,9,3048},{20,5,901},{45,1,1188},{45,1,1188},{45,1,1188},
{15,35,883},{47,1,910},{46,3,19},{46,3,19},{30,39,77},{8,3,651},{10,23,6},{43,1,36},{43,1,36},{43,1,36},{43,21,1},{29,3,648},{47,19,5},{47,19,5},{26,23,4},{43,9,648},{26,23,4},{19,9,882},{41,1,146},{41,3,4},{14,3,5},{19,9,882},{45,1,882},{14,3,5},{0,5,900},{45,1,882},{0,5,900},{15,0,882},{15,0,882},{15,0,882},{15,0,882},{30,1,5},
{30,1,5},{30,1,5},{30,25,4},{10,23,2},{10,23,2},{11,1,4374},{13,1,1924},{13,1,1335},{45,33,961},{13,1,4972},{47,1,1225},{47,17,22},{30,5,1356},{26,1,3460},{40,35,940},{9,1,1755},{41,1,434},{11,17,74},{27,3,185},{23,5,3048},{15,1,1146},{47,17,18},{22,35,882},{14,1,3048},{22,35,882},{13,1,1299},{13,1,1299},{13,1,1299},{29,19,890},{15,1,1017},{47,33,13},{47,33,13},
{46,23,86},{40,33,652},{42,7,5},{11,1,45},{11,1,45},{11,1,45},{27,5,5},{39,11,648},{31,3,2},{31,3,2},{12,7,0},{22,1,648},{12,7,0},{21,17,882},{9,1,250},{25,17,1},{30,17,1},{21,17,882},{17,21,882},{30,17,1},{0,35,882},{17,21,882},{0,35,882},{29,0,890},{29,0,890},{29,0,890},{29,0,890},{47,1,5},{47,1,5},{47,1,5},{47,9,4},{26,37,4},
{26,37,4},{41,1,4427},{27,1,2227},{27,1,1443},{13,17,925},{11,1,4878},{45,1,1331},{15,1,21},{47,35,1178},{44,1,3438},{26,19,810},{23,1,1746},{9,1,557},{25,1,89},{11,17,142},{23,17,2814},{13,1,1125},{15,1,5},{24,19,765},{45,17,2814},{24,19,765},{27,1,1443},{27,1,1443},{27,1,1443},{13,33,884},{29,1,1132},{15,1,21},{15,1,21},{31,37,69},{10,1,648},{28,21,5},{25,1,89},
{25,1,89},{25,1,89},{41,19,1},{25,5,648},{45,17,2},{45,17,2},{28,21,1},{10,1,648},{28,21,1},{19,3,761},{7,1,274},{39,1,0},{31,1,0},{19,3,761},{39,1,761},{31,1,0},{0,19,765},{39,1,761},{0,19,765},{13,0,884},{13,0,884},{13,0,884},{13,0,884},{15,1,17},{15,1,17},{15,1,17},{31,23,4},{12,5,4},{12,5,4},{9,1,3933},{11,1,2185},{11,1,1509},
{27,1,899},{41,1,4346},{13,1,1109},{45,1,42},{15,19,813},{30,1,2958},{12,3,510},{7,1,1386},{23,1,483},{39,1,128},{25,17,59},{3,25,2249},{11,1,870},{13,1,4},{10,3,482},{29,1,2249},{10,3,482},{11,1,1509},{11,1,1509},{11,1,1509},{27,17,890},{43,1,1275},{45,1,42},{45,1,42},{45,21,86},{12,1,675},{44,5,10},{39,1,128},{39,1,128},{39,1,128},{9,3,5},{9,3,648},
{13,1,4},{13,1,4},{14,5,2},{31,3,648},{14,5,2},{3,3,481},{21,1,180},{7,1,1},{13,1,0},{3,3,481},{7,1,481},{13,1,0},{0,3,481},{7,1,481},{0,3,481},{27,0,890},{27,0,890},{27,0,890},{27,0,890},{45,1,26},{45,1,26},{45,1,26},{45,7,1},{44,35,2},{44,35,2},{39,1,3541},{9,1,2173},{25,1,1658},{11,1,893},{9,1,3801},{27,1,1035},{13,1,116},
{45,3,523},{47,1,2574},{44,33,282},{21,1,1070},{7,1,427},{23,1,194},{39,1,16},{21,17,1769},{25,1,673},{27,1,26},{12,33,265},{17,21,1769},{12,33,265},{25,1,1658},{25,1,1658},{25,1,1658},{11,1,893},{11,1,1386},{13,1,116},{13,1,116},{29,35,69},{14,1,734},{14,19,6},{23,1,194},{23,1,194},{23,1,194},{39,17,1},{23,3,648},{27,1,26},{27,1,26},{14,19,5},{43,3,648},
{14,19,5},{33,3,265},{35,1,89},{5,1,4},{25,1,0},{33,3,265},{21,1,265},{25,1,0},{0,33,265},{21,1,265},{0,33,265},{11,0,884},{11,0,884},{11,0,884},{11,0,884},{13,1,52},{13,1,52},{13,1,52},{29,21,4},{14,19,2},{14,19,2},{23,1,3267},{39,1,2157},{9,1,1764},{25,1,954},{39,1,3397},{11,1,950},{27,1,267},{13,33,297},{29,1,2313},{30,17,134},{5,1,838},
{21,1,413},{37,1,260},{7,1,9},{17,39,1374},{23,1,518},{25,1,80},{44,17,125},{39,17,1374},{44,17,125},{9,1,1764},{9,1,1764},{9,1,1764},{25,1,954},{25,1,1569},{27,1,267},{27,1,267},{13,19,77},{47,1,846},{46,3,5},{37,1,260},{37,1,260},{37,1,260},{7,1,9},{3,25,648},{25,1,80},{25,1,80},{47,3,1},{29,1,648},{47,3,1},{33,1,113},{19,1,37},{19,1,1},
{7,1,0},{33,1,113},{35,1,113},{7,1,0},{0,17,125},{35,1,113},{0,17,125},{25,0,890},{25,0,890},{25,0,890},{25,0,890},{27,1,98},{27,1,98},{27,1,98},{43,5,1},{46,3,4},{46,3,4},{7,1,3032},{7,1,2233},{23,1,1944},{39,1,1083},{7,1,3096},{9,1,1028},{41,1,506},{27,17,151},{13,1,2068},{47,17,31},{35,1,693},{5,1,425},{5,1,325},{21,1,52},{3,3,1032},
{7,1,406},{23,1,157},{31,17,26},{3,3,1032},{31,17,26},{23,1,1944},{23,1,1944},{23,1,1944},{39,1,1083},{39,1,1723},{41,1,506},{41,1,506},{27,33,78},{45,1,1034},{47,17,6},{5,1,325},{5,1,325},{5,1,325},{21,1,52},{19,5,648},{23,1,157},{23,1,157},{31,17,1},{37,3,648},{31,17,1},{17,1,18},{17,1,10},{33,1,1},{19,1,0},{17,1,18},{33,1,18},{19,1,0},
{0,17,26},{33,1,18},{0,17,26},{9,0,890},{9,0,890},{9,0,890},{9,0,890},{41,1,145},{41,1,145},{41,1,145},{27,19,4},{47,1,5},{47,1,5},{21,1,2710},{37,1,2207},{7,1,2004},{23,1,1173},{21,1,2775},{39,1,1036},{9,1,643},{41,1,81},{11,1,1900},{15,1,9},{3,1,524},{19,1,401},{19,1,365},{5,1,104},{33,33,771},{5,1,369},{37,1,221},{29,1,1},{33,33,771},
{29,1,1},{7,1,2004},{7,1,2004},{7,1,2004},{23,1,1173},{23,1,1784},{9,1,643},{9,1,643},{41,17,61},{13,1,1091},{15,1,9},{19,1,365},{19,1,365},{19,1,365},{5,1,104},{33,5,578},{37,1,221},{37,1,221},{29,1,1},{23,1,578},{29,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{23,0,884},
{23,0,884},{23,0,884},{23,0,884},{9,1,202},{9,1,202},{9,1,202},{41,3,1},{15,1,9},{15,1,9},{5,1,2214},{21,1,1813},{21,1,1669},{7,1,1109},{21,1,2151},{23,1,874},{23,1,585},{25,1,13},{25,1,1508},{13,1,34},{3,1,300},{3,1,225},{3,1,200},{19,1,61},{33,17,451},{35,1,192},{5,1,125},{11,1,0},{17,33,451},{11,1,0},{21,1,1669},{21,1,1669},{21,1,1669},
{7,1,1109},{7,1,1460},{23,1,585},{23,1,585},{25,1,13},{11,1,872},{13,1,34},{3,1,200},{3,1,200},{3,1,200},{19,1,61},{17,5,338},{5,1,125},{5,1,125},{11,1,0},{5,17,338},{11,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{7,0,884},{7,0,884},{7,0,884},{7,0,884},{39,1,250},
{39,1,250},{39,1,250},{25,17,4},{13,1,34},{13,1,34},{35,1,1818},{5,1,1521},{5,1,1421},{21,1,1028},{5,1,1675},{7,1,790},{7,1,565},{39,1,2},{23,1,1155},{11,1,100},{33,1,147},{33,1,107},{33,1,98},{3,1,29},{1,19,216},{19,1,88},{19,1,52},{23,1,1},{19,1,216},{23,1,1},{5,1,1421},{5,1,1421},{5,1,1421},{21,1,1028},{21,1,1158},{7,1,565},{7,1,565},
{39,1,2},{25,1,723},{11,1,100},{33,1,98},{33,1,98},{33,1,98},{3,1,29},{33,17,162},{19,1,52},{19,1,52},{23,1,1},{17,33,162},{23,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{21,0,884},{21,0,884},{21,0,884},{21,0,884},{7,1,340},{7,1,340},{7,1,340},{39,1,2},{11,1,100},
{11,1,100},{0,13,1568},{0,14,185},{0,10,5},{0,8,586},{0,44,3371},{0,24,2147},{0,8,1027},{0,36,2571},{0,6,3617},{0,20,2729},{0,13,1568},{0,14,185},{0,10,5},{0,8,586},{18,8,3371},{0,24,2147},{0,8,1027},{0,36,2571},{44,0,3371},{0,36,2571},{0,24,1},{0,24,1},{0,24,1},{0,20,1},{0,20,288},{0,34,100},{0,34,100},{0,2,164},{0,2,321},{0,2,189},{0,24,1},
{0,24,1},{0,24,1},{0,20,1},{32,2,288},{0,34,100},{0,34,100},{0,2,164},{20,0,288},{0,2,164},{24,2,1568},{0,14,185},{0,10,5},{0,8,586},{24,2,1568},{13,0,1568},{0,8,586},{0,22,1576},{13,0,1568},{0,22,1576},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,25,1568},{0,46,89},{0,26,20},
{0,40,505},{0,46,3968},{0,10,2316},{0,24,1078},{0,6,2880},{0,22,4305},{0,6,3105},{0,25,1568},{0,46,89},{0,26,20},{0,40,505},{18,10,3968},{0,10,2316},{0,24,1078},{0,6,2880},{46,0,3968},{0,6,2880},{0,12,1},{0,12,1},{0,12,1},{0,6,0},{0,6,514},{0,20,193},{0,20,193},{0,18,317},{0,18,556},{0,18,353},{0,12,1},{0,12,1},{0,12,1},{0,6,0},{2,2,512},
{0,20,193},{0,20,193},{0,18,317},{2,2,512},{0,18,317},{16,11,1568},{0,46,89},{16,26,5},{0,40,505},{16,11,1568},{25,0,1568},{0,40,505},{0,38,1570},{25,0,1568},{0,38,1570},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,7,1568},{0,15,37},{16,12,76},{0,10,421},{0,15,4652},{0,42,2540},{0,10,1097},
{0,38,3251},{0,8,5108},{0,22,3545},{0,7,1568},{0,15,37},{16,12,51},{0,10,421},{38,2,4651},{0,42,2540},{0,10,1097},{0,38,3251},{16,8,4651},{0,38,3251},{0,46,0},{0,46,0},{0,46,0},{0,38,1},{0,8,802},{0,6,289},{0,6,289},{0,34,493},{0,34,872},{0,34,557},{0,46,0},{0,46,0},{0,46,0},{0,38,1},{4,0,802},{0,6,289},{0,6,289},{0,34,493},{8,0,802},
{0,34,493},{26,4,1568},{0,15,37},{32,12,2},{0,10,421},{26,4,1568},{7,0,1568},{0,10,421},{0,24,1576},{7,0,1568},{0,24,1576},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,19,1570},{0,29,8},{16,44,166},{0,12,338},{0,29,5420},{0,28,2755},{0,26,1208},{0,8,3659},{0,24,5988},{0,38,4028},{0,19,1570},
{0,29,8},{16,44,141},{0,12,338},{40,0,5419},{0,28,2755},{0,26,1208},{0,8,3659},{46,2,5419},{0,8,3659},{0,45,0},{0,45,0},{0,45,0},{0,24,1},{0,24,1154},{0,22,433},{0,22,433},{0,4,697},{0,4,1270},{0,4,797},{0,45,0},{0,45,0},{0,45,0},{0,24,1},{18,4,1152},{0,22,433},{0,22,433},{0,4,697},{36,2,1152},{0,4,697},{14,0,1568},{0,29,8},{18,28,1},
{0,12,338},{14,0,1568},{8,10,1568},{0,12,338},{0,40,1568},{8,10,1568},{0,40,1568},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{16,17,1609},{16,43,45},{32,30,282},{0,28,325},{0,25,5419},{0,14,2514},{0,12,804},{0,24,3462},{0,10,6191},{0,8,3965},{32,3,1574},{32,13,17},{2,14,130},{16,28,312},{16,11,5419},
{0,14,2514},{0,12,804},{0,24,3462},{25,0,5419},{0,24,3462},{16,43,41},{16,43,41},{16,43,41},{16,10,42},{0,28,1152},{0,40,292},{0,40,292},{0,36,596},{0,6,1379},{0,20,761},{32,29,4},{32,29,4},{32,29,4},{32,10,5},{2,24,1152},{0,40,292},{0,40,292},{0,36,596},{28,0,1152},{0,36,596},{30,2,1568},{0,27,4},{4,14,1},{0,28,244},{30,2,1568},{35,2,1568},{0,28,244},
{0,26,1570},{35,2,1568},{0,26,1570},{16,0,41},{16,0,41},{16,0,41},{16,0,41},{0,4,1},{0,4,1},{0,4,1},{0,2,1},{0,32,13},{0,32,13},{32,1,1733},{32,11,159},{32,46,455},{16,14,377},{0,7,5419},{0,47,2294},{0,44,542},{0,10,3225},{0,12,6401},{0,10,3901},{2,17,1569},{2,27,6},{18,46,125},{32,14,296},{26,4,5419},{0,47,2294},{0,44,542},{0,10,3225},{7,0,5419},
{0,10,3225},{32,11,158},{32,11,158},{32,11,158},{32,42,157},{0,46,1154},{0,26,170},{0,26,170},{0,22,482},{0,22,1491},{0,6,739},{2,27,2},{2,27,2},{2,27,2},{2,26,5},{22,2,1152},{0,26,170},{0,26,170},{0,22,482},{42,2,1152},{0,22,482},{47,2,1568},{32,11,2},{20,30,5},{0,14,185},{47,2,1568},{17,2,1568},{0,14,185},{0,42,1570},{17,2,1568},{0,42,1570},{32,0,157},
{32,0,157},{32,0,157},{32,0,157},{0,22,0},{0,22,0},{0,22,0},{0,34,1},{0,18,45},{0,18,45},{2,1,2003},{32,9,330},{2,31,710},{32,30,517},{0,35,5420},{0,15,2081},{0,30,345},{0,42,3099},{0,28,6641},{0,10,3885},{34,1,1574},{34,11,17},{4,47,134},{18,30,312},{14,0,5419},{0,15,2081},{0,30,345},{0,42,3099},{8,10,5419},{0,42,3099},{32,23,317},{32,23,317},{32,23,317},
{32,28,317},{0,29,1154},{0,28,80},{0,28,80},{0,8,388},{0,8,1665},{0,38,753},{34,27,4},{34,27,4},{34,27,4},{34,12,5},{8,2,1152},{0,28,80},{0,28,80},{0,8,388},{30,2,1152},{0,8,388},{15,2,1568},{18,41,4},{36,47,5},{0,46,125},{15,2,1568},{33,4,1568},{0,46,125},{0,28,1576},{33,4,1568},{0,28,1576},{32,0,317},{32,0,317},{32,0,317},{32,0,317},{0,40,0},
{0,40,0},{0,40,0},{0,20,0},{0,34,113},{0,34,113},{18,1,2395},{2,23,570},{18,45,1046},{2,47,725},{0,1,5419},{0,13,1947},{0,47,185},{0,12,2880},{0,14,6964},{0,12,3841},{20,1,1593},{4,25,6},{20,15,131},{34,46,326},{0,1,5419},{0,13,1947},{0,47,185},{0,12,2880},{1,0,5419},{0,12,2880},{2,21,546},{2,21,546},{2,21,546},{2,14,545},{0,11,1152},{0,30,37},{0,30,37},
{0,24,317},{0,10,1846},{0,8,797},{4,25,2},{4,25,2},{4,25,2},{4,28,2},{24,4,1152},{0,30,37},{0,30,37},{0,24,317},{11,0,1152},{0,24,317},{43,0,1568},{34,9,1},{22,31,1},{0,47,85},{43,0,1568},{19,6,1568},{0,47,85},{0,44,1570},{19,6,1568},{0,44,1570},{2,0,545},{2,0,545},{2,0,545},{2,0,545},{0,28,0},{0,28,0},{0,28,0},{0,6,1},{0,20,208},
{0,20,208},{4,1,3030},{18,37,927},{34,29,1474},{18,15,1042},{16,1,5540},{0,27,1787},{0,15,57},{0,44,2668},{0,46,7299},{0,28,3836},{6,1,1612},{36,39,10},{6,29,136},{20,31,312},{47,2,5419},{0,27,1787},{0,15,57},{0,44,2668},{17,2,5419},{0,44,2668},{18,19,883},{18,19,883},{18,19,883},{18,46,882},{0,23,1152},{0,31,5},{0,31,5},{0,10,225},{0,26,2064},{0,10,901},{36,9,2},
{36,9,2},{36,9,2},{36,14,1},{38,10,1152},{0,31,5},{0,31,5},{0,10,225},{23,0,1152},{0,10,225},{38,3,1568},{20,39,4},{8,45,5},{0,15,41},{38,3,1568},{35,8,1568},{0,15,41},{0,30,1570},{35,8,1568},{0,30,1570},{18,0,882},{18,0,882},{18,0,882},{18,0,882},{0,47,1},{0,47,1},{0,47,1},{0,8,1},{0,6,325},{0,6,325},{20,1,3734},{34,5,1332},{4,43,1958},
{18,45,1395},{2,1,5808},{0,25,1676},{0,29,20},{0,30,2566},{0,15,7631},{0,44,3925},{22,1,1656},{6,7,9},{22,13,132},{6,45,309},{15,2,5419},{0,25,1676},{0,29,20},{0,30,2566},{33,4,5419},{0,30,2566},{34,33,1258},{34,33,1258},{34,33,1258},{34,47,1259},{0,5,1154},{0,45,5},{0,45,5},{0,42,180},{0,12,2275},{0,10,981},{6,23,4},{6,23,4},{6,23,4},{6,46,5},{28,2,1152},
{16,15,1},{16,15,1},{0,42,180},{42,8,1152},{0,42,180},{41,2,1568},{36,7,2},{24,29,5},{0,29,20},{41,2,1568},{17,8,1568},{0,29,20},{0,46,1576},{17,8,1568},{0,46,1576},{34,0,1258},{34,0,1258},{34,0,1258},{34,0,1258},{0,29,0},{0,29,0},{0,29,0},{0,40,1},{0,38,482},{0,38,482},{36,1,4356},{4,19,1676},{20,27,2370},{34,13,1683},{18,1,6121},{0,39,1616},{16,43,36},
{0,46,2397},{0,45,7815},{0,46,3837},{8,1,1715},{38,37,10},{8,43,131},{22,29,312},{43,0,5419},{0,39,1612},{16,43,27},{0,46,2393},{19,6,5419},{0,46,2393},{4,1,1577},{4,1,1577},{4,1,1577},{4,15,1576},{0,17,1156},{16,13,22},{16,13,22},{0,12,134},{0,44,2441},{0,12,971},{38,23,1},{38,23,1},{38,23,1},{38,47,2},{16,3,1152},{32,29,1},{32,29,1},{0,12,130},{17,0,1152},
{0,12,130},{29,12,1568},{22,37,4},{10,43,5},{0,43,10},{29,12,1568},{26,13,1568},{0,43,10},{0,31,1576},{26,13,1568},{0,31,1576},{4,0,1576},{4,0,1576},{4,0,1576},{4,0,1576},{0,41,4},{0,41,4},{0,41,4},{0,26,4},{0,24,562},{0,24,562},{22,1,4616},{20,3,1677},{6,41,2378},{20,43,1681},{4,1,6311},{16,7,1620},{32,27,20},{0,31,2365},{0,43,7444},{0,47,3383},{40,1,1766},
{8,5,11},{24,11,132},{8,13,314},{11,0,5419},{0,37,1577},{32,27,19},{0,31,2265},{5,8,5419},{0,31,2265},{20,1,1593},{20,1,1593},{20,1,1593},{20,29,1569},{32,1,1155},{32,27,19},{32,27,19},{16,44,125},{0,46,2150},{0,28,659},{8,21,2},{8,21,2},{8,21,2},{8,31,2},{30,4,1152},{18,13,1},{18,13,1},{0,44,72},{15,8,1152},{0,44,72},{37,0,1568},{38,5,2},{26,27,2},
{0,27,2},{37,0,1568},{36,9,1568},{0,27,2},{0,15,1568},{36,9,1568},{0,15,1568},{20,0,1568},{20,0,1568},{20,0,1568},{20,0,1568},{32,41,1},{32,41,1},{32,41,1},{32,42,2},{0,26,388},{0,26,388},{8,1,4936},{6,17,1676},{22,9,2363},{36,11,1689},{36,1,6476},{2,37,1620},{18,41,24},{32,15,2370},{0,41,7036},{0,15,2859},{26,1,1851},{40,35,11},{10,25,136},{24,27,321},{41,2,5419},
{16,5,1571},{18,41,20},{0,15,2130},{17,8,5419},{0,15,2130},{6,1,1619},{6,1,1619},{6,1,1619},{6,13,1571},{18,1,1158},{18,11,21},{18,11,21},{2,30,125},{0,31,1905},{0,14,425},{40,5,2},{40,5,2},{40,5,2},{40,45,2},{45,0,1152},{4,27,5},{4,27,5},{0,30,41},{27,8,1152},{0,30,41},{21,2,1568},{24,19,2},{12,41,1},{16,41,1},{21,2,1568},{24,9,1568},{16,41,1},
{0,29,1570},{24,9,1568},{0,29,1570},{6,0,1570},{6,0,1570},{6,0,1570},{6,0,1570},{18,25,4},{18,25,4},{18,25,4},{18,28,5},{0,28,232},{0,28,232},{24,1,5154},{38,1,1680},{8,39,2378},{22,41,1683},{22,1,6708},{18,5,1628},{34,25,24},{2,29,2363},{0,9,6740},{0,45,2553},{12,1,1964},{10,3,8},{26,9,139},{40,41,309},{29,12,5419},{2,35,1572},{4,25,21},{0,29,2027},{26,13,5419},
{0,29,2027},{38,1,1664},{38,1,1664},{38,1,1664},{22,27,1570},{4,1,1185},{34,25,20},{34,25,20},{34,46,130},{0,45,1721},{0,46,218},{10,19,4},{10,19,4},{10,19,4},{10,13,8},{29,2,1152},{20,11,5},{20,11,5},{0,47,25},{35,6,1152},{0,47,25},{35,2,1568},{10,3,4},{28,25,5},{18,25,4},{35,2,1568},{34,5,1568},{18,25,4},{0,13,1570},{34,5,1568},{0,13,1570},{22,0,1570},
{22,0,1570},{22,0,1570},{22,0,1570},{34,23,1},{34,23,1},{34,23,1},{34,14,4},{0,14,149},{0,14,149},{26,1,5444},{8,1,1724},{24,23,2378},{38,9,1689},{8,1,6964},{4,35,1620},{20,39,36},{34,13,2370},{0,23,6513},{0,13,2244},{44,1,2099},{42,33,11},{12,23,136},{26,25,321},{37,0,5419},{18,3,1571},{20,39,27},{0,43,1924},{36,9,5419},{0,43,1924},{8,1,1720},{8,1,1720},{8,1,1720},
{8,11,1577},{36,1,1224},{20,9,19},{20,9,19},{4,47,122},{0,43,1548},{0,31,90},{42,19,1},{42,19,1},{42,19,1},{42,43,2},{43,2,1152},{6,25,5},{6,25,5},{0,31,9},{21,8,1152},{0,31,9},{46,1,1568},{26,17,2},{14,39,2},{18,39,1},{46,1,1568},{26,7,1568},{18,39,1},{0,27,1576},{26,7,1568},{0,27,1576},{8,0,1576},{8,0,1576},{8,0,1576},{8,0,1576},{20,23,9},
{20,23,9},{20,23,9},{4,30,10},{0,47,73},{0,47,73},{42,1,5700},{40,1,1798},{40,37,2375},{24,39,1683},{40,1,7153},{20,3,1626},{36,23,24},{4,27,2386},{0,21,6243},{0,27,1980},{30,1,2210},{12,1,8},{28,7,132},{42,39,323},{5,0,5419},{4,33,1572},{6,23,21},{0,27,1836},{0,5,5419},{0,27,1836},{40,1,1762},{40,1,1762},{40,1,1762},{24,25,1570},{6,1,1275},{36,23,20},{36,23,20},
{36,15,125},{0,41,1395},{0,45,25},{12,17,4},{12,17,4},{12,17,4},{12,11,8},{38,1,1152},{22,9,4},{22,9,4},{0,45,0},{43,28,1152},{0,45,0},{31,1,1568},{42,1,2},{30,23,5},{4,23,5},{31,1,1568},{32,1,1568},{4,23,5},{0,11,1570},{32,1,1568},{0,11,1570},{24,0,1570},{24,0,1570},{24,0,1570},{24,0,1570},{36,21,1},{36,21,1},{36,21,1},{36,46,4},{0,15,25},
{0,15,25},{28,1,6116},{26,1,1980},{26,5,2386},{40,7,1689},{26,1,7408},{6,17,1620},{22,37,24},{36,41,2375},{0,35,6044},{0,41,1798},{47,1,2385},{44,1,25},{14,37,125},{28,23,315},{35,2,5419},{20,1,1569},{22,37,20},{0,41,1762},{34,5,5419},{0,41,1762},{26,1,1836},{26,1,1836},{26,1,1836},{10,9,1574},{38,1,1363},{22,7,21},{22,7,21},{6,29,132},{0,9,1284},{0,13,8},{44,1,0},
{44,1,0},{44,1,0},{44,41,0},{39,0,1152},{8,23,4},{8,23,4},{16,13,4},{9,28,1152},{16,13,4},{17,6,1568},{14,1,25},{47,37,4},{20,37,1},{17,6,1568},{20,1,1568},{20,37,1},{0,25,1570},{20,1,1568},{0,25,1570},{10,0,1570},{10,0,1570},{10,0,1570},{10,0,1570},{22,5,5},{22,5,5},{22,5,5},{22,31,5},{0,43,2},{0,43,2},{14,1,6434},{12,1,2244},{12,35,2370},
{26,21,1685},{12,1,7724},{38,1,1620},{38,21,36},{22,25,2378},{0,3,5839},{0,9,1724},{31,1,2546},{30,1,90},{46,5,122},{14,7,322},{46,1,5419},{22,1,1602},{8,21,19},{0,9,1720},{26,7,5419},{0,9,1720},{42,1,1924},{42,1,1924},{42,1,1924},{26,23,1577},{24,1,1414},{38,21,27},{38,21,27},{22,13,136},{0,7,1218},{32,43,11},{30,1,9},{30,1,9},{30,1,9},{14,9,10},{23,2,1152},
{24,7,5},{24,7,5},{18,43,1},{20,9,1152},{18,43,1},{27,3,1568},{46,1,73},{31,5,10},{36,5,9},{27,3,1568},{39,9,1568},{36,5,9},{0,9,1576},{39,9,1568},{0,9,1576},{26,0,1576},{26,0,1576},{26,0,1576},{26,0,1576},{38,19,1},{38,19,1},{38,19,1},{38,15,2},{16,27,2},{16,27,2},{30,1,6786},{44,1,2553},{28,3,2363},{42,5,1689},{28,1,8052},{24,1,1671},{24,35,24},
{38,9,2378},{0,1,5715},{0,39,1680},{45,1,2675},{47,1,218},{47,35,130},{30,21,315},{31,1,5419},{24,1,1667},{24,35,20},{0,39,1664},{32,1,5419},{0,39,1664},{28,1,2027},{28,1,2027},{28,1,2027},{12,7,1571},{10,1,1521},{24,5,21},{24,5,21},{8,27,139},{0,21,1169},{2,11,8},{46,1,25},{46,1,25},{46,1,25},{46,39,1},{37,2,1152},{10,21,5},{10,21,5},{18,11,4},{34,7,1152},
{18,11,4},{39,7,1568},{15,1,149},{15,35,4},{22,35,1},{39,7,1568},{26,1,1568},{22,35,1},{0,23,1570},{26,1,1568},{0,23,1570},{12,0,1570},{12,0,1570},{12,0,1570},{12,0,1570},{24,19,4},{24,19,4},{24,19,4},{24,29,5},{2,11,4},{2,11,4},{47,1,7186},{14,1,2859},{14,33,2370},{28,35,1683},{30,1,8313},{40,1,1819},{40,19,24},{8,23,2363},{16,1,5820},{16,7,1676},{13,1,2892},
{15,1,425},{31,3,125},{47,35,321},{45,1,5419},{26,1,1796},{10,19,21},{0,7,1619},{27,9,5419},{0,7,1619},{14,1,2130},{14,1,2130},{14,1,2130},{28,21,1570},{42,1,1608},{40,19,20},{40,19,20},{24,11,136},{0,19,1155},{34,41,11},{31,1,41},{31,1,41},{31,1,41},{47,7,5},{44,1,1152},{26,5,5},{26,5,5},{4,41,2},{26,9,1152},{4,41,2},{25,1,1568},{29,1,232},{29,19,5},
{24,19,4},{25,1,1568},{33,9,1568},{24,19,4},{0,7,1570},{33,9,1568},{0,7,1570},{28,0,1570},{28,0,1570},{28,0,1570},{28,0,1570},{40,17,1},{40,17,1},{40,17,1},{40,13,1},{18,25,2},{18,25,2},{15,1,7706},{46,1,3383},{30,1,2365},{14,3,1685},{47,1,8695},{42,1,2092},{26,33,20},{40,7,2378},{4,1,6099},{2,21,1677},{27,1,3152},{29,1,659},{45,17,125},{31,19,315},{27,3,5419},
{44,1,1993},{26,33,19},{0,21,1593},{39,9,5419},{0,21,1593},{30,1,2265},{30,1,2265},{30,1,2265},{14,5,1572},{28,1,1764},{26,33,19},{26,33,19},{10,25,132},{32,3,1155},{4,9,11},{45,1,72},{45,1,72},{45,1,72},{15,37,1},{31,5,1152},{12,19,1},{12,19,1},{20,9,2},{14,9,1152},{20,9,2},{23,17,1568},{27,1,388},{43,33,2},{40,33,1},{23,17,1568},{45,17,1568},{40,33,1},
{0,21,1568},{45,17,1568},{0,21,1568},{14,0,1568},{14,0,1568},{14,0,1568},{14,0,1568},{26,1,2},{26,1,2},{26,1,2},{26,27,2},{4,39,2},{4,39,2},{45,1,8016},{47,1,3837},{47,1,2397},{30,17,1685},{31,1,9093},{28,1,2484},{42,17,36},{26,21,2370},{36,1,6379},{18,5,1676},{11,1,3345},{13,1,971},{13,1,134},{15,33,326},{39,7,5419},{30,1,2185},{12,17,22},{0,5,1577},{26,1,5419},
{0,5,1577},{47,1,2393},{47,1,2393},{47,1,2393},{30,19,1576},{14,1,1890},{42,17,27},{42,17,27},{42,9,131},{2,17,1152},{36,39,10},{13,1,130},{13,1,130},{13,1,130},{45,5,5},{17,2,1152},{28,33,1},{28,33,1},{22,39,1},{16,1,1152},{22,39,1},{21,3,1568},{25,1,562},{27,1,4},{40,1,4},{21,3,1568},{43,1,1568},{40,1,4},{0,5,1576},{43,1,1568},{0,5,1576},{30,0,1576},
{30,0,1576},{30,0,1576},{30,0,1576},{42,1,10},{42,1,10},{42,1,10},{42,11,5},{36,23,4},{36,23,4},{13,1,7700},{45,1,3925},{31,1,2566},{47,17,1637},{45,1,8460},{14,1,2285},{28,1,20},{42,5,1958},{38,1,5932},{4,35,1332},{9,1,2987},{11,1,981},{43,1,180},{29,17,219},{9,3,4803},{47,1,1925},{44,1,5},{32,35,1258},{31,3,4803},{32,35,1258},{31,1,2566},{31,1,2566},{31,1,2566},
{47,3,1577},{30,1,2054},{28,1,20},{28,1,20},{12,23,132},{34,1,1155},{6,7,9},{43,1,180},{43,1,180},{43,1,180},{13,35,1},{29,3,1152},{14,17,1},{14,17,1},{22,7,4},{43,9,1152},{22,7,4},{21,1,1250},{39,1,482},{41,1,1},{28,1,0},{21,1,1250},{41,1,1250},{28,1,0},{0,35,1258},{41,1,1250},{0,35,1258},{47,0,1576},{47,0,1576},{47,0,1576},{47,0,1576},{28,1,20},
{28,1,20},{28,1,20},{28,25,5},{6,37,2},{6,37,2},{43,1,7164},{29,1,3836},{45,1,2668},{31,1,1579},{13,1,7780},{30,1,2041},{14,1,57},{28,35,1474},{24,1,5308},{36,19,927},{9,1,2475},{11,1,901},{11,1,225},{43,17,110},{37,7,4056},{15,1,1590},{30,1,5},{18,19,883},{30,1,4056},{18,19,883},{45,1,2668},{45,1,2668},{45,1,2668},{31,17,1570},{47,1,2214},{14,1,57},{14,1,57},
{28,7,136},{36,1,1186},{38,37,10},{11,1,225},{11,1,225},{11,1,225},{43,3,5},{39,11,1152},{30,1,5},{30,1,5},{8,37,2},{22,1,1152},{8,37,2},{5,1,882},{7,1,325},{9,1,1},{46,1,1},{5,1,882},{9,1,882},{46,1,1},{0,19,882},{9,1,882},{0,19,882},{31,0,1570},{31,0,1570},{31,0,1570},{31,0,1570},{14,1,41},{14,1,41},{14,1,41},{44,9,5},{38,21,4},
{38,21,4},{11,1,6493},{13,1,3841},{13,1,2880},{45,1,1574},{43,1,7071},{47,1,1822},{46,1,185},{44,19,1046},{26,1,4761},{22,3,570},{23,1,2034},{9,1,797},{25,1,317},{11,17,34},{23,17,3318},{13,1,1221},{31,1,37},{20,3,546},{45,17,3318},{20,3,546},{13,1,2880},{13,1,2880},{13,1,2880},{45,1,1574},{45,1,2443},{46,1,185},{46,1,185},{14,21,131},{38,1,1275},{24,5,6},{25,1,317},
{25,1,317},{25,1,317},{11,33,1},{25,5,1152},{31,1,37},{31,1,37},{24,5,2},{10,1,1152},{24,5,2},{3,3,545},{21,1,208},{7,1,1},{29,1,0},{3,3,545},{23,1,545},{29,1,0},{0,3,545},{23,1,545},{0,3,545},{45,0,1570},{45,0,1570},{45,0,1570},{45,0,1570},{46,1,85},{46,1,85},{46,1,85},{30,23,1},{8,35,1},{8,35,1},{41,1,6095},{11,1,3885},{43,1,3099},
{13,1,1636},{11,1,6422},{45,1,1767},{31,1,345},{30,3,710},{44,1,4358},{8,33,330},{7,1,1698},{39,1,753},{9,1,388},{41,1,9},{3,25,2753},{27,1,1018},{29,1,80},{22,33,317},{29,1,2753},{22,33,317},{43,1,3099},{43,1,3099},{43,1,3099},{13,1,1636},{29,1,2628},{31,1,345},{31,1,345},{46,5,134},{40,1,1395},{10,35,17},{9,1,388},{9,1,388},{9,1,388},{41,1,9},{9,3,1152},
{29,1,80},{29,1,80},{26,35,4},{31,3,1152},{26,35,4},{33,3,313},{35,1,113},{21,1,0},{41,1,0},{33,3,313},{21,1,313},{41,1,0},{0,33,317},{21,1,313},{0,33,317},{29,0,1576},{29,0,1576},{29,0,1576},{29,0,1576},{47,1,125},{47,1,125},{47,1,125},{46,37,5},{40,19,4},{40,19,4},{9,1,5661},{11,1,3901},{11,1,3225},{43,1,1739},{41,1,5978},{29,1,1685},{45,1,542},
{47,33,455},{14,1,3994},{10,33,159},{21,1,1418},{7,1,739},{23,1,482},{9,1,16},{21,17,2273},{11,1,878},{27,1,170},{10,33,158},{17,21,2273},{10,33,158},{11,1,3225},{11,1,3225},{11,1,3225},{43,1,1739},{43,1,2875},{45,1,542},{45,1,542},{47,19,125},{42,1,1584},{26,3,6},{23,1,482},{23,1,482},{23,1,482},{9,1,16},{23,3,1152},{27,1,170},{27,1,170},{26,3,2},{43,3,1152},
{26,3,2},{17,3,145},{19,1,45},{35,1,1},{23,1,0},{17,3,145},{35,1,145},{23,1,0},{0,33,157},{35,1,145},{0,33,157},{43,0,1570},{43,0,1570},{43,0,1570},{43,0,1570},{15,1,185},{15,1,185},{15,1,185},{31,21,5},{10,33,2},{10,33,2},{39,1,5341},{9,1,3965},{25,1,3462},{11,1,1889},{9,1,5505},{13,1,1765},{13,1,804},{31,33,282},{47,1,3750},{42,17,45},{5,1,1210},
{21,1,761},{37,1,596},{23,1,73},{17,39,1878},{9,1,781},{41,1,292},{42,17,41},{39,17,1878},{42,17,41},{25,1,3462},{25,1,3462},{25,1,3462},{11,1,1889},{11,1,3058},{13,1,804},{13,1,804},{15,3,130},{44,1,1798},{12,33,17},{37,1,596},{37,1,596},{37,1,596},{23,1,73},{3,25,1152},{41,1,292},{41,1,292},{28,33,4},{29,1,1152},{28,33,4},{1,3,41},{33,1,13},{3,1,1},
{5,1,1},{1,3,41},{3,1,41},{5,1,1},{0,17,41},{3,1,41},{0,17,41},{27,0,1570},{27,0,1570},{27,0,1570},{27,0,1570},{29,1,244},{29,1,244},{29,1,244},{15,5,1},{26,1,4},{26,1,4},{23,1,5128},{39,1,4028},{9,1,3659},{25,1,2169},{39,1,5148},{11,1,1917},{27,1,1208},{45,17,166},{29,1,3628},{28,1,8},{35,1,1089},{5,1,797},{5,1,697},{37,1,221},{3,3,1536},
{7,1,706},{23,1,433},{44,1,0},{3,3,1536},{44,1,0},{9,1,3659},{9,1,3659},{9,1,3659},{25,1,2169},{25,1,3366},{27,1,1208},{27,1,1208},{45,17,141},{47,1,2105},{28,1,8},{5,1,697},{5,1,697},{5,1,697},{37,1,221},{19,5,1152},{23,1,433},{23,1,433},{44,1,0},{37,3,1152},{44,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},
{0,1,0},{1,1,0},{0,1,0},{41,0,1568},{41,0,1568},{41,0,1568},{41,0,1568},{13,1,338},{13,1,338},{13,1,338},{29,19,1},{28,1,8},{28,1,8},{7,1,4416},{23,1,3545},{39,1,3251},{9,1,2027},{23,1,4372},{41,1,1771},{11,1,1097},{13,17,76},{29,1,2956},{14,1,37},{19,1,753},{35,1,557},{35,1,493},{21,1,148},{17,5,1067},{21,1,513},{7,1,289},{47,1,0},{5,17,1067},
{47,1,0},{39,1,3251},{39,1,3251},{39,1,3251},{9,1,2027},{9,1,2819},{11,1,1097},{11,1,1097},{13,17,51},{47,1,1769},{14,1,37},{35,1,493},{35,1,493},{35,1,493},{21,1,148},{5,1,802},{7,1,289},{7,1,289},{47,1,0},{9,1,802},{47,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{25,0,1576},
{25,0,1576},{25,0,1576},{25,0,1576},{11,1,421},{11,1,421},{11,1,421},{13,33,2},{14,1,37},{14,1,37},{21,1,3786},{7,1,3105},{7,1,2880},{23,1,1929},{7,1,3648},{9,1,1532},{25,1,1078},{27,1,20},{13,1,2452},{47,1,89},{3,1,456},{19,1,353},{19,1,317},{35,1,89},{33,33,683},{5,1,321},{21,1,193},{13,1,1},{33,33,683},{13,1,1},{7,1,2880},{7,1,2880},{7,1,2880},
{23,1,1929},{39,1,2411},{25,1,1078},{25,1,1078},{27,1,20},{29,1,1451},{47,1,89},{19,1,317},{19,1,317},{19,1,317},{35,1,89},{3,3,512},{21,1,193},{21,1,193},{13,1,1},{3,3,512},{13,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{39,0,1570},{39,0,1570},{39,0,1570},{39,0,1570},{41,1,505},
{41,1,505},{41,1,505},{27,17,5},{47,1,89},{47,1,89},{21,1,3210},{21,1,2729},{37,1,2571},{7,1,1825},{21,1,3015},{39,1,1388},{9,1,1027},{11,1,5},{11,1,2032},{15,1,185},{3,1,264},{3,1,189},{3,1,164},{19,1,45},{17,3,384},{35,1,164},{35,1,100},{25,1,1},{35,1,384},{25,1,1},{37,1,2571},{37,1,2571},{37,1,2571},{7,1,1825},{7,1,2112},{9,1,1027},{9,1,1027},
{11,1,5},{13,1,1235},{15,1,185},{3,1,164},{3,1,164},{3,1,164},{19,1,45},{33,3,288},{35,1,100},{35,1,100},{25,1,1},{21,1,288},{25,1,1},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{1,1,0},{0,1,0},{1,1,0},{0,1,0},{23,0,1576},{23,0,1576},{23,0,1576},{23,0,1576},{9,1,586},{9,1,586},{9,1,586},{11,1,5},{15,1,185},
{15,1,185},{0,37,2665},{0,45,274},{0,44,8},{0,26,1025},{0,15,5885},{0,12,3666},{0,10,1742},{0,38,4406},{0,8,6359},{0,22,4730},{0,37,2665},{0,45,274},{0,44,8},{0,26,1025},{22,4,5885},{0,12,3666},{0,10,1742},{0,38,4406},{15,0,5885},{0,38,4406},{0,28,0},{0,28,0},{0,28,0},{0,6,1},{0,22,545},{0,20,208},{0,20,208},{0,18,340},{0,18,593},{0,18,376},{0,28,0},
{0,28,0},{0,28,0},{0,6,1},{2,2,545},{0,20,208},{0,20,208},{0,18,340},{22,0,545},{0,18,340},{28,0,2665},{0,45,274},{0,44,8},{0,26,1025},{28,0,2665},{37,0,2665},{0,26,1025},{0,24,2665},{37,0,2665},{0,24,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,3,2665},{0,43,169},{0,14,17},
{0,12,865},{0,13,6669},{0,28,3898},{0,42,1825},{0,8,4826},{0,24,7263},{0,38,5231},{0,3,2665},{0,43,169},{0,14,17},{0,12,865},{24,2,6669},{0,28,3898},{0,42,1825},{0,8,4826},{13,0,6669},{0,8,4826},{0,46,1},{0,46,1},{0,46,1},{0,8,4},{0,8,841},{0,6,306},{0,6,306},{0,34,520},{0,34,917},{0,34,584},{0,46,1},{0,46,1},{0,46,1},{0,8,4},{4,0,841},
{0,6,306},{0,6,306},{0,34,520},{8,0,841},{0,34,520},{44,2,2665},{0,43,169},{16,14,2},{0,12,865},{44,2,2665},{3,0,2665},{0,12,865},{0,40,2669},{3,0,2665},{0,40,2669},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,1,2669},{0,41,85},{16,46,79},{0,44,725},{0,11,7538},{0,14,4214},{0,12,1842},
{0,8,5354},{0,10,8241},{0,8,5795},{0,1,2669},{0,41,85},{16,46,54},{0,44,725},{24,4,7538},{0,14,4214},{0,12,1842},{0,8,5354},{11,0,7538},{0,8,5354},{0,29,1},{0,29,1},{0,29,1},{0,24,4},{0,40,1201},{0,22,458},{0,22,458},{0,4,730},{0,4,1325},{0,4,830},{0,29,1},{0,29,1},{0,29,1},{0,24,4},{20,0,1201},{0,22,458},{0,22,458},{0,4,730},{40,0,1201},
{0,4,730},{30,2,2665},{0,41,85},{32,46,5},{0,44,725},{30,2,2665},{35,2,2665},{0,44,725},{0,26,2665},{35,2,2665},{0,26,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{16,1,2799},{0,9,29},{16,31,167},{0,14,650},{0,9,8493},{0,46,4485},{0,44,1934},{0,40,5878},{0,26,9333},{0,40,6503},{16,1,2774},
{0,9,29},{16,31,142},{0,14,650},{42,0,8493},{0,46,4485},{0,44,1934},{0,40,5878},{9,0,8493},{0,40,5878},{0,11,0},{0,11,0},{0,11,0},{0,26,1},{0,26,1625},{0,24,629},{0,24,629},{0,20,986},{0,20,1793},{0,20,1130},{0,11,0},{0,11,0},{0,11,0},{0,26,1},{36,0,1625},{0,24,629},{0,24,629},{0,20,986},{26,0,1625},{0,20,986},{2,1,2665},{0,9,29},{18,47,1},
{0,14,650},{2,1,2665},{1,2,2665},{0,14,650},{0,42,2669},{1,2,2665},{0,42,2669},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{32,1,3171},{0,7,4},{16,15,315},{0,46,514},{0,7,9669},{0,31,4865},{0,46,2114},{0,10,6505},{0,12,10663},{0,10,7181},{32,1,3050},{0,7,4},{32,15,274},{0,46,514},{26,4,9669},
{0,31,4865},{0,46,2114},{0,10,6505},{7,0,9669},{0,10,6505},{0,23,0},{0,23,0},{0,23,0},{0,12,1},{0,28,2178},{0,10,820},{0,10,820},{0,6,1348},{0,6,2405},{0,20,1553},{0,23,0},{0,23,0},{0,23,0},{0,12,1},{2,24,2178},{0,10,820},{0,10,820},{0,6,1348},{28,0,2178},{0,6,1348},{29,0,2665},{0,7,4},{4,31,9},{0,46,514},{29,0,2665},{17,4,2665},{0,46,514},
{0,28,2665},{17,4,2665},{0,28,2665},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{0,0,0},{2,1,3529},{16,21,35},{32,29,444},{0,31,474},{0,19,9670},{0,45,4529},{0,46,1634},{0,42,6265},{0,28,10919},{0,26,7149},{18,1,3146},{16,21,10},{2,29,285},{0,31,474},{14,0,9669},{0,45,4529},{0,46,1634},{0,42,6265},{8,10,9669},
{0,42,6265},{16,21,34},{16,21,34},{16,21,34},{16,44,34},{0,46,2180},{0,12,610},{0,12,610},{0,22,1184},{0,22,2517},{0,22,1473},{16,21,9},{16,21,9},{16,21,9},{16,44,9},{22,2,2178},{0,12,610},{0,12,610},{0,22,1184},{42,2,2178},{0,22,1184},{13,2,2665},{16,21,1},{20,45,0},{0,31,410},{13,2,2665},{3,6,2665},{0,31,410},{0,14,2677},{3,6,2665},{0,14,2677},{16,0,34},
{16,0,34},{16,0,34},{16,0,34},{0,34,0},{0,34,0},{0,34,0},{0,32,1},{0,32,10},{0,32,10},{18,1,3971},{32,35,150},{2,43,644},{16,45,534},{0,1,9669},{0,43,4214},{0,31,1226},{0,28,5954},{0,14,11246},{0,12,6951},{4,1,3285},{2,5,4},{34,13,274},{32,15,483},{0,1,9669},{0,43,4214},{0,31,1226},{0,28,5954},{1,0,9669},{0,28,5954},{16,3,146},{16,3,146},{16,3,146},
{16,30,147},{0,29,2180},{0,28,458},{0,28,458},{0,8,1018},{0,8,2691},{0,38,1419},{2,21,0},{2,21,0},{2,21,0},{2,14,1},{8,2,2178},{0,28,458},{0,28,458},{0,8,1018},{30,2,2178},{0,8,1018},{27,2,2665},{2,5,4},{6,29,9},{0,15,338},{27,2,2665},{35,8,2665},{0,15,338},{0,30,2665},{35,8,2665},{0,30,2665},{16,0,146},{16,0,146},{16,0,146},{16,0,146},{0,6,1},
{0,6,1},{0,6,1},{0,18,4},{0,18,40},{0,18,40},{34,1,4603},{32,3,307},{2,11,925},{32,29,667},{16,1,9779},{0,41,3905},{0,45,913},{0,44,5653},{0,46,11530},{0,44,6878},{36,1,3390},{18,19,11},{4,27,269},{2,29,474},{18,3,9669},{0,41,3905},{0,45,913},{0,44,5653},{45,8,9669},{0,44,5653},{32,1,291},{32,1,291},{32,1,291},{32,47,291},{0,11,2178},{0,46,305},{0,46,305},
{0,40,925},{0,10,2872},{0,8,1427},{34,5,10},{34,5,10},{34,5,10},{18,46,9},{24,4,2178},{0,46,305},{0,46,305},{0,40,925},{11,0,2178},{0,40,925},{8,1,2665},{18,19,2},{22,43,0},{0,29,265},{8,1,2665},{1,8,2665},{0,29,265},{0,46,2677},{1,8,2665},{0,46,2677},{32,0,290},{32,0,290},{32,0,290},{32,0,290},{0,24,1},{0,24,1},{0,24,1},{0,20,1},{0,34,100},
{0,34,100},{20,1,5538},{2,17,582},{34,41,1298},{2,43,882},{2,1,10086},{0,9,3618},{0,43,581},{0,30,5418},{0,15,11905},{0,14,6895},{22,1,3586},{4,3,3},{36,11,273},{34,43,478},{15,2,9669},{0,9,3618},{0,43,581},{0,30,5418},{33,4,9669},{0,30,5418},{2,1,570},{2,1,570},{2,1,570},{2,15,549},{0,23,2178},{0,15,185},{0,15,185},{0,10,765},{0,26,3090},{0,10,1441},{4,3,2},
{4,3,2},{4,3,2},{4,47,2},{38,10,2178},{0,15,185},{0,15,185},{0,10,765},{23,0,2178},{0,10,765},{23,0,2665},{4,3,2},{8,11,8},{0,43,181},{23,0,2665},{18,9,2665},{0,43,181},{0,31,2665},{18,9,2665},{0,31,2665},{2,0,545},{2,0,545},{2,0,545},{2,0,545},{0,28,0},{0